/*
 * BMCconfig.c
 *
 * This tool configures the BMC with LAN parameters.
 *
 * Copyright (c) 2001-2003 Intel Corporation.
 *
 * This software program is available to you under the following license.
 *
 *			Intel BSD + Patent License
 *
 * Recipient has requested a license and Intel Corporation ("Intel") is willing
 * to grant a license for the software entitled "panicsel" (the "Software")
 * being provided by Intel Corporation.  The following definitions apply to
 * this License: "Licensed Patents" means patent claims licensable by Intel
 * Corporation which are necessarily infringed by the use or sale of the
 * Software alone or when combined with the operating system referred to below.
 * "Recipient" means the party to whom Intel delivers this Software. "Licensee"
 * means Recipient and those third parties that receive a license to any
 * operating system available under the GNU Public License version 2.0 or
 * later.
 *
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * The license is provided to Recipient and Recipient's Licensees under the
 * following terms:
 * Redistribution and use in source and binary forms of the Software, with or
 * without modification, are permitted provided that the following conditions
 * are met: Redistributions of source code of the Software may retain the above
 * copyright notice, this list of conditions and the following disclaimer.
 * Redistributions in binary form of the Software may reproduce the above
 * copyright notice, this list of conditions and the following disclaimer in
 * the documentation and/or other materials provided with the distribution.
 * Neither the name of Intel Corporation nor the names of its contributors
 * shall be used to endorse or promote products derived from this Software with
 * out specific prior written permission. Intel hereby grants Recipient and
 * Licensees a non-exclusive, worldwide, royalty-free patent license under
 * Licensed Patents to make, use, sell, offer to sell, import and otherwise
 * transfer the Software, if any, in source code and object code form. This
 * license shall include changes to the Software that are error corrections or
 * other minor changes to the Software that do not add functionality or
 * features when the Software is incorporated in any version of a operating
 * system that has been distributed under the GNU General Public License 2.0 or
 * later.  This patent license shall apply to the combination of the Software
 * and any operating system licensed under the GNU Public License version 2.0
 * or later if, at the time Intel provides the Software to Recipient, such
 * addition of the Software to the then publicly available versions of such
 * operating system available under the GNU Public License version 2.0 or later
 * (whether in gold, beta or alpha form) causes such combination to be covered
 * by the Licensed Patents. The patent license shall not apply to any other
 * combinations that include the Software.  No hardware per se is licensed
 * hereunder.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>
#include <sys/socket.h>    /* for Get_IPMac_Addr() */
#include <sys/ioctl.h>     /* for Get_IPMac_Addr() */
#include <net/if.h>        /* for Get_IPMac_Addr() */
#include "ipmi_ioctls.h"   /* for ipmi_cmd_va() */

#define uchar   unsigned char
#define ushort  unsigned short
#define uint    unsigned int

#define RTF_UP          0x0001   /* route usable */

// special IMB defines
#define IMB_STATUS         int
#define SELprintf          printf
#define MAX_BUFFER_SIZE        64
#define MAX_NO_OF_RETRIES	3
#define DEFAULT_TIMEOUT     (1000 * 1000)
#define SEL_TIMEOUT           400L

// IPMI Commands, see Table 38-8
#define GET_LAN_CONFIG          0x02
#define SET_LAN_CONFIG          0x01
#define GET_SER_CONFIG          0x11
#define SET_SER_CONFIG          0x10
#define GET_PEF_CONFIG          0x13
#define SET_PEF_CONFIG          0x12
#define GET_SEL_INFO            0x40
#define GET_SEL_ALLOCATION_INFO 0x41
#define GET_SEL_ENTRY           0x43
#define GET_SEL_TIME
#define GET_DEVICE_ID           0x81  /* special mask with 0x80 | 0x01 */
#define SET_CHANNEL_ACC 		0xC0  /* special mask with 0x80 | 0x40 */
#define SET_USER				0x47  /* set the user up command */
#define SET_USER_ACCESS			0x43  /* set the user access mode */
#define GET_CHANNEL_ACC	        0xC1  /* special mask with 0x80 | 0x41 */

#define ACTIVATE_SOL			0x01
#define SET_SOL_CONFIG			0x03
#define GET_SOL_CONFIG			0x04

// Other IPMI values
#define PUBLIC_BUS      0
#define PRIVATE_BUS  0x03
#define BMC_SA       0x20
#define BMC_LUN         0
#define PEF_LUN      0x10  // from cmdtool, not used
#define NETFN_APP    0x06
#define NETFN_STOR   0x0a
#define NETFN_TRANS  0x0c
#define NETFN_SEVT   0x04  // sensor/event
#define NETFN_SPEF   0x20  // from cmdtool, not used

#define NETFN_SOL    0x34

#define SOL_ENABLE_PARAM			0x01
#define SOL_AUTHENTICATION_PARAM 	0x02
#define SOL_ENABLE_FLAG				0x01
#define SOL_PRIVILEGE_LEVEL_USER    0x02
#define SOL_BAUD_RATE_PARAM			0x05
#define SOL_PREFERRED_BAUD_RATE		0x07

#define MAC_ADDRESS_LAN_PARAM		0x05

#define MAX_DEVICE_NAME_LENGTH   	6
#define NUM_DEVICES_TO_CHECK		2
#define MAC_LENGTH					6
#define DEFAULT_ETHERNET_DEVICE		1

// ACCESN_* entries from imbapi.h defined where?
#define ACCESS_OK    0

// pack structures so that they don't pad
#pragma pack(1)
// Request structure provided to SendTimedImbpRequest()
//
typedef struct
{
        uchar   cmdType;        // IMB command
        uchar   rsSa;           // command destination address
        uchar   busType;        // not used
        uchar   netFn;          // IMB command class (network function)
        uchar   rsLun;          // subsystem on destination
        uchar * data;           // command body
        int     dataLength;     // body size
} IMBPREQUESTDATA;

typedef struct
{
        short              record_id;
        uchar              record_type;
        int                timestamp;
        unsigned short     generator_id;
        uchar              evm_rev;         //event message revision
        uchar              sensor_type;
        uchar              sensor_number;
        uchar              event_trigger;
        uchar              event_data1;
        uchar              event_data2;
        uchar              event_data3;
} SEL_RECORD;

typedef struct
{            /* See IPMI Table 19-3 */
        uchar              data[30];
} LAN_RECORD;

extern unsigned int SendTimedImbpRequest (
        IMBPREQUESTDATA *reqPtr,         // request info and data
        int              timeOut,        // how long to wait, in mSec units
        uchar           *respDataPtr,    // where to put response data
        int             *respDataLen,    // how much response data there is
        uchar           *completionCode  // request status from dest controller
        );

/* Global variables */
char   screenPrint  = 0;   /* printf to screen */
char   freadonly = 0;	/* only read the data */
char   fdisable  = 0;	/* disable the LAN channel */
char   fIPMI     = 0;	     /* =0 use Intel IPMI, =1 use valinux IPMI */
char * progname  = "BMCconfig";
char * progver   = "1.0";
uchar  rgmyip[4]   = {0,0,0,0};
uchar  rggwyip[4]  = {0,0,0,0};
uchar  rgdestip[4] = {0,0,0,0};
uchar  rgsubnet[4] = {0,0,0,0};  /* default subnet, i.e. 255.255.255.0 */
uchar  rgmymac[6]   = {0,0,0,0,0,0};
uchar  rggwymac[6]  = {0,0,0,0,0,0};
uchar  rgdestmac[6] = {0,0,0,0,0,0};
char   passwordData[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; /* password data */
char   rgcommunity[19] = "public";  /* default community string*/

#define NLAN  27
struct
{
  int cmd;
  int sz;
  char desc[28];
} lanparams[NLAN] = {   /* see IPMI Table 19-4 */
 /*  0 */  0, 1, "Set in progress",
 /*  1 */  1, 1, "Auth type support",
 /*  2 */  2, 5, "Auth type enables",
 /*  3 */  3, 4, "IP address",
 /*  4 */  4, 1, "IP addr src",  /* (DHCP/Static) */
 /*  5 */  5, 6, "MAC addr",
 /*  6 */  6, 4, "Subnet mask",
 /*  7 */  7, 3, "IPv4 header",
 /*  8 */  8, 2, "Prim RMCP port ",
 /*  9 */  9, 2, "Sec RMCP port ",
 /* 10 */ 10, 1, "BMC grat ARP ",
 /* 11 */ 11, 1, "grat ARP interval",
 /* 12 */ 12, 4, "Def gateway IP",
 /* 13 */ 13, 6, "Def gateway MAC",
 /* 14 */ 14, 4, "Sec gateway IP",
 /* 15 */ 15, 6, "Sec gateway MAC",
 /* 16 */ 16,18, "Community string",
 /* 17 */ 17, 1, "Num dest",
 /* 18 */ 18, 5, "Dest type",
 /* 19 */ 19, 13, "Dest address",
 /* 20 */ 96, 28, "OEM Alert String",
 /* 21 */ 97,  1, "Alert Retry Algorithm",
 /* 22 */ 98,  3, "UTC Offset",
 /* 23 */ 192, 4, "DHCP Server IP",
 /* 24 */ 193, 6, "DHCP MAC Address",
 /* 25 */ 194, 1, "DHCP Enable",
 /* 26 */ 201, 2, "Channel Access Mode(Lan)"
};

#define NSER  22   /* max=32 */
struct
{
  int cmd;
  int sz;
  char desc[28];
} serparams[NSER] = {   /* see IPMI Table 20-4 */
 /*  0 */  0, 1, "Set in progress",
 /*  1 */  1, 1, "Auth type support",
 /*  2 */  2, 5, "Auth type enables",
 /*  3 */  3, 1, "Connection Mode",
 /*  4 */  4, 1, "Sess Inactiv Timeout",
 /*  5 */  5, 5, "Channel Callback",
 /*  6 */  6, 1, "Session Termination",
 /*  7 */  7, 2, "IPMI Msg Comm",
 /*  8 */  8, 2, "Mux Switch",
 /*  9 */  9, 2, "Modem Ring Time",
 /* 10 */ 10,17, "Modem Init String",
 /* 11 */ 11, 5, "Modem Escape Seq",
 /* 12 */ 12, 8, "Modem Hangup Seq",
 /* 13 */ 13, 8, "Modem Dial Command",
 /* 14 */ 14, 1, "Page Blackout Interval",
 /* 15 */ 15,18, "Community String",
 /* 16 */ 16, 1, "Num of Alert Dest",
 /* 17 */ 17, 5, "Destination Info",
 /* 18 */ 18, 1, "Call Retry Interval",
 /* 19 */ 19, 3, "Destination Comm Settings",
 /* 20 */ 29, 2, "Terminal Mode Config",
 /* 21 */ 201, 2,"Channel Access Mode (Ser)"
};

#define NCMDS   15
struct
{
 uchar cmdtyp;
 uchar sa;
 uchar bus;
 uchar netfn;
 uchar lun;
 uchar len;
} ipmi_cmds[NCMDS] = {
// GET_SEL_ENTRY,    BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 6,
 GET_SEL_INFO,     BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 0,
 GET_LAN_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 4,
 SET_LAN_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 4,
 GET_SER_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 4,
 SET_SER_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 4,
 GET_PEF_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 3,
 SET_PEF_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 3,
 GET_DEVICE_ID,    BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 0,
 SET_CHANNEL_ACC,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 3,
 GET_CHANNEL_ACC,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 3,
 SET_USER,         BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 3,
 SET_USER_ACCESS,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 4,

 ACTIVATE_SOL,     BMC_SA, PUBLIC_BUS, NETFN_SOL,   BMC_LUN, 0,
 SET_SOL_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SOL,	BMC_LUN, 0,
 GET_SOL_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SOL,   BMC_LUN, 0

 // 0x61 /*GetConfigData*/, BMC_SA, PUBLIC_BUS, 0x30,  BMC_LUN, 8
};


int ipmi_cmd_ia(uchar cmd, uchar netfn, uchar *pdata, uchar sdata, uchar *presp,
		int *sresp, uchar *pcc, char fdebugcmd)
{
	IMBPREQUESTDATA requestData;
	IMB_STATUS status;
	struct stat     stbuf;
	char *imbDev;
	int i;

	imbDev = "/dev/imb";

	for (i = 0; i < NCMDS; i++)
	{
		if ( (ipmi_cmds[i].cmdtyp == cmd) && (ipmi_cmds[i].netfn == netfn))
		{
			break;
		}
	}

	if (i >= NCMDS)
	{
		fprintf(stderr, "ipmi_cmd_ia: Unknown command %x\n",cmd);
		return(-1);
	}

	if (cmd >= 0x80)
	{
		cmd = cmd & 0x7f;  /* unmask it */
	}

	requestData.cmdType	= cmd;
	requestData.rsSa	= ipmi_cmds[i].sa;
	requestData.busType	= ipmi_cmds[i].bus;
	requestData.netFn	= ipmi_cmds[i].netfn;
	requestData.rsLun	= ipmi_cmds[i].lun;
	requestData.dataLength = sdata;
	requestData.data       = pdata;

	if (fdebugcmd)
	{
		uchar * pc;
		int sz;

		sz = sizeof(IMBPREQUESTDATA);
		pc = (uchar *)&requestData.cmdType;
		printf("\nIPMI_cmd_ia request \n(len=%d): ",sz);

		for (i = 0; i < sz; i++)
		{
			printf("%02x ",pc[i]);
		}

		printf("\n");

		sz = requestData.dataLength;
		pc = (uchar *)&requestData.data[0];
		printf("Requested data=%p, len=%d : ", pc, sz);

		for (i = 0; i < sz; i++)
		{
			printf("%02x ",pc[i]);
		}

		printf("\n");
	}

	if (stat(imbDev, &stbuf) != -1)
	{
		memset(presp, 0, *sresp);
		for ( i =0 ; i < MAX_NO_OF_RETRIES; i++)
		{
			if((status =SendTimedImbpRequest(&requestData, DEFAULT_TIMEOUT, presp, sresp, pcc)) == ACCESS_OK )
			{
				break;
			}
		}

		if (screenPrint)   // only gets here if error
		{
		       SELprintf("ipmi_cmd_ia: sendImbRequest status=%x, ccode=%x\n",
		                      (uint)status, *pcc);
		}
	}
	else
	{
		fprintf(stderr,"ipmi_cmd_ia: No IMB driver found (%s)\n",imbDev);
		return(-2);
	}

	if (fdebugcmd)
	{
		SELprintf("ipmi_cmd_ia: sendImbRequest status=%x, ccode=%x\n",
	          (uint)status, *pcc);

		if (status == ACCESS_OK)
		{
			uchar * pc;
			int sz;

			sz = *sresp;
			pc = (uchar *)presp;
			printf("ipmi_cmd_ia: response (len=%d): ",sz);

			for (i = 0; i < sz; i++)
			{
				printf("%02x ",pc[i]);
			}
			printf("\n");
		}
	}

	return(status);
} /* end ipmi_cmd_ia() */

int ipmi_cmd_va(uchar cmd, uchar netfn, uchar *pdata, uchar sdata, uchar *presp,
		int *sresp, uchar *pcc, char fdebugcmd)
{
	IPMI_XFER xfer;
	IPMI_XFER *xferp = &xfer;
	char *ipmiDev;
	int fd, rc, i;
	uchar cc;

	for (i = 0; i < NCMDS; i++)
	{
    	if (ipmi_cmds[i].cmdtyp == cmd)
    	{
    		break;
		}
	 }

	if (i >= NCMDS)
	{
		fprintf(stderr, "ipmi_cmd_va: Unknown command %x\n",cmd);
		return(-1);
	}

	if (cmd >= 0x80)
	{
		cmd = cmd & 0x7f;  /* unmask it */
	}

	ipmiDev = "/dev/ipmikcs";

	if ((fd = open(ipmiDev, O_RDWR))<0)
	{
	   	SELprintf("ipmi_cmd_va: cannot open %s\n", ipmiDev);
	    ipmiDev = "/dev/ipmi/kcs";  /* some use this name instead */
	}

	if ((fd = open(ipmiDev, O_RDWR))<0)
	{
		SELprintf("ipmi_cmd_va: cannot open %s\n", ipmiDev);
		return(-2);
	}

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, ipmi_cmds[i].lun);
	SET_REQUEST_NETFN(xferp, ipmi_cmds[i].netfn);
	SET_REQUEST_CMD(xferp, cmd);

	if (sdata > 0)
	{
		SET_REQUEST_DATA(xferp, (uchar *) pdata, sdata);
	}

	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
	{
       	printf("ipmi_cmd_va: failed (%m)\n");
	       return(-1);
    }

	GET_RESPONSE_CC(xferp, cc);
	*pcc = cc;

	if (cc != 00)
	{
		printf("ipmi_cmd_va: completionCode = %x\n",cc);
		return(-1);
	} else {
		GET_RESPONSE_DATA(xferp, presp);
	}

	if (fdebugcmd)
	{
		SELprintf("ipmi_cmd_va: ioctl status=%x, ccode=%x\n", (uint)rc, cc);
		if (rc == 0)
		{
			uchar * pc;
			int sz;

			sz = *sresp;
			pc = (uchar *)presp;

			printf("ipmi_cmd_va: response (len=%d): ",sz);

			for (i = 0; i < sz; i++)
			{
				printf("%02x ",pc[i]);
			}

			printf("\n");
		}
     }

	close(fd);
	return(rc);
} /* end ipmi_cmd_va() */

int ipmi_cmd(uchar cmd, uchar netfn, uchar *pdata, uchar sdata, uchar *presp,
		int *sresp, uchar *pcc, char fdebugcmd)
{
	int rc;

	if (fIPMI == 1)
	{
		rc = ipmi_cmd_va(cmd, netfn, pdata,sdata, presp,sresp, pcc, fdebugcmd);
	}
	else
	{
		rc = ipmi_cmd_ia(cmd, netfn, pdata,sdata, presp,sresp, pcc, fdebugcmd);
	}

	return(rc);
}

int GetDeviceID(LAN_RECORD *pLanRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMB_STATUS status;
	uchar inputData[24];
	uchar completionCode;

	if (pLanRecord == NULL)
	{
		return(-1);
	}

	status = ipmi_cmd_ia(GET_DEVICE_ID, NETFN_APP, inputData, 0, responseData,
						 &responseLength, &completionCode, screenPrint);

	if (status != ACCESS_OK)
	{
	 	status = ipmi_cmd_va(GET_DEVICE_ID, NETFN_APP, inputData, 0, responseData,
							 &responseLength, &completionCode, screenPrint);

		if (status == ACCESS_OK)
		{
			fIPMI = 1;
		}
	}

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("GetDeviceID: completion code=%x\n", completionCode);
		}
		else
		{
			// dont copy first byte (Parameter revision, usu 0x11)
			memcpy(pLanRecord,&responseData[0],responseLength);
			return(0);  // successful, done
		}
	}  /* endif */
	/* if we get here, we had an error */
	return(-1);
}  /*end GetDeviceID() */

int GetChanAcc(uchar chan, uchar parm, LAN_RECORD *pLanRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMB_STATUS status;
	uchar inputData[24];
	uchar completionCode;

	if (pLanRecord == NULL)
	{
		return(-1);
	}

	responseLength = 3;
	inputData[0] = chan;
	inputData[1] = parm;  /* 0x80 = active, 0x40 = non-volatile */

	status = ipmi_cmd(GET_CHANNEL_ACC, NETFN_APP,inputData, 2, responseData,
					  &responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("GetChanAcc: completion code=%x\n",
					  completionCode);
		}
		else
		{
			// dont copy first byte (Parameter revision, usu 0x11)
			memcpy(pLanRecord,&responseData[0],responseLength);
			return(0);  // successful, done
		}
	}  /* endif */
	/* if we get here, we had an error */
 	return(-1);
}  /*GetChanAcc()*/

int SetChanAcc(uchar chan, uchar parm, uchar val)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMB_STATUS status;
	uchar inputData[24];
	uchar completionCode;
	int foo = SET_CHANNEL_ACC;

	//val = val | 0x08;

	/* parm: 0x80 = active, 0x40 = set non-vol*/
	responseLength = 1;
	inputData[0] = chan;  /* channel */
	inputData[1] = (parm & 0xc0) | (val & 0x3F);

	if (chan == 7)
	{
		/* if not set to admin, network, other dpccli commands don't work */
		inputData[2] = (parm & 0xc0) | 0x04;
	} else
	{
		inputData[2] = (parm & 0xc0) | 0x04;  /* set Admin priv level */
	                  						  /* serial defaults to 0x02 = User priv level */
	}

	if ( screenPrint )
	{
		SELprintf("SetChanAccess = %x, inputData[0] = %x, inputData[1] = %x, inputData[2] = %x",
			  foo, inputData[0], inputData[1], inputData[2]);
	}

	status = ipmi_cmd(SET_CHANNEL_ACC, NETFN_APP, inputData, 3, responseData,
	 				  &responseLength, &completionCode, screenPrint);

	if ( screenPrint )
	{
		SELprintf(" status = %x\n", status);
	}

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("\n\n\nSetChanAcc: My NEW completion code=%x\n",
				completionCode);
		}
		else
		{
			return(0);  // successful, done
		}
	}  /* endif */
	/* if get here, error */
 	return(-1);
}  /*SetChanAcc()*/


int SetLanConfig(void)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMBPREQUESTDATA requestData;
	uchar inputData[24];
	uchar completionCode;

	IMB_STATUS status;

	// set user auth type
	inputData[0] = 0x07;   // LAN Channel
	inputData[1] = 0x02;   // Authentication Types
	inputData[2] = 0x17;   //
	inputData[3] = 0x17;   //
	inputData[4] = 0x06;   // only md2/md5 
	inputData[5] = 0x06;   // only md2/md5
	inputData[6] = 0x00;   //

	status = ipmi_cmd(SET_LAN_CONFIG, NETFN_TRANS, inputData,
	7, responseData, &responseLength, &completionCode, screenPrint);
	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("SetLanConfig: completion code=%x\n", completionCode);
			return -1;
		}
	}
	else
	{
		SELprintf( "SetLanConfig: SET_LAN_CONFIG, set authentication failed\n" );
		return -1;
	}

	// enable gratuitious arps
	inputData[0] = 0x07;   // LAN Channel
	inputData[1] = 0x0a;   // Gratuitous ARPs interval
	inputData[2] = 0x01;

	status = ipmi_cmd(SET_LAN_CONFIG, NETFN_TRANS, inputData,
	3, responseData, &responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("SetLanConfig: completion code=%x\n", completionCode);
			return -1;
		}
	}
	else
	{
		SELprintf( "SetLanConfig: SET_LAN_CONFIG, set ARP interval failed\n" );
		return -1;
	}

	// enable gratuitious arps
	inputData[0] = 0x07;   // LAN Channel
	inputData[1] = 0x0b;   // Gratuitous ARPs enable
	inputData[2] = 0x03;

	status = ipmi_cmd(SET_LAN_CONFIG, NETFN_TRANS, inputData,
	3, responseData, &responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("SetLanConfig: completion code=%x\n", completionCode);
			return -1;
		}
	}
	else
	{
		SELprintf( "SetLanConfig: SET_LAN_CONFIG, enable ARPs failed\n" );
		return -1;
	}

 	return 0;
}  /* end SetLanConfig() */

int SetUser ( )
{
	int ret;
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMB_STATUS status;
	uchar completionCode;
	char  inputData[24] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; /* inputdata */
	int foo;

	responseLength = 1;
	inputData[0] = 0x01;
	inputData[1] = 0x01;

	if ( screenPrint )
	{
		SELprintf("SETUSER - inputData[0] = %x, inputData[1] = %x, inputData[2] = %x",
			inputData[0], inputData[1], inputData[2]);
	}

	status = ipmi_cmd(SET_USER, NETFN_APP, inputData, 2, responseData, &responseLength,
					  &completionCode, screenPrint);

	if ( screenPrint )
	{
		SELprintf(" status = %x\n", status);
	}

	inputData[0] = 0x01;
	inputData[1] = 0x02;
	strcpy(&inputData[2],passwordData);

	if ( screenPrint )
	{
		SELprintf("SETUSER - inputData[0] = %x, inputData[1] = %x, inputData[2] = %x",
				  inputData[0], inputData[1], inputData[2]);
	}

	status = ipmi_cmd(SET_USER, NETFN_APP, inputData, 18, responseData, &responseLength,
	 				  &completionCode, screenPrint);

	if ( screenPrint )
	{
		SELprintf(" status = %x\n", status);
	}

	ret = SetChanAcc(7, 0x80, 0x02);   /* PEF off, always avail */
	ret = SetChanAcc(7, 0x40, 0x02);   /* PEF off, always avail */

	inputData[0] = 0x97; /* change user access for LAN */
	inputData[1] = 0x01; /* user id = 1 */
	inputData[2] = 0x04; /* user limit = admin */
	inputData[3] = 0x00; /* no session limit set */

	if ( screenPrint )
	{
		SELprintf("inputData[0] = %x, inputData[1] = %x, inputData[2] = %x",
			inputData[0], inputData[1], inputData[2]);
	}

	status = ipmi_cmd(SET_USER_ACCESS, NETFN_APP, inputData, 4, responseData,
	 	&responseLength, &completionCode, screenPrint);

	if ( screenPrint )
	{
		SELprintf(" status = %x\n", status);
	}

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("SetChanAcc: My NEW completion code=%x\n",
				completionCode);
		}
		else
		{
			return(0);  // successful, done
		}
	}  /* endif */

	/* if get here, error */

 	return(-1);
}  /*SetUser()*/

int GetLanEntry(uchar subfunc, LAN_RECORD *pLanRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMBPREQUESTDATA requestData;
	uchar inputData[24];
	IMB_STATUS status;
	uchar completionCode;
	int   i;
	uchar chan; uchar bset;

	if (pLanRecord == NULL)
	{
		if (screenPrint)
		{
		    printf("GetLanEntry: error, output buffer is NULL\n");
		}
		return (-1);
	}

	chan = 7;  /* LAN 1 */

	if (subfunc == 18 || subfunc == 19)
	{
		bset = 1;  /* dest id = 1 */
	}
	else
	{
		bset = 0;
	}

	inputData[0] = chan;  // flags, channel 3:0 (LAN 1)
	inputData[1] = subfunc;  // Param selector (3 = ip addr)
	inputData[2] = bset;  // Set selector
	inputData[3] = 0;  // Block selector

	status = ipmi_cmd(GET_LAN_CONFIG, NETFN_TRANS, inputData, 4, responseData,
	 				  &responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("GetLanEntry: completion code=%x\n",
				completionCode); // responseData[0]);
		}
		else
		{
			// dont copy first byte (Parameter revision, usu 0x11)
			memcpy(pLanRecord,&responseData[1],responseLength-1);
			//successful, done
			return(0);
		}
	}

	// we are here because completionCode is not COMPLETION_CODE_OK
	if (screenPrint)
	{
		SELprintf("GetLanEntry: sendImbRequest completion code=%x\n", completionCode);
	}
 	return -1;
}  /* end GetLanEntry() */

int SetLanEntry(uchar subfunc, LAN_RECORD *pLanRecord, int reqlen)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMBPREQUESTDATA requestData;
	uchar inputData[24];
	IMB_STATUS status;
	struct stat stbuf;
	uchar completionCode;
	char *imbDev;
	int   i;

	if (pLanRecord == NULL)
	{
	   if (screenPrint)
	   {
	       printf("SetLanEntry: error, input buffer is NULL\n");
	   }
 	   return (-1);
	}

	inputData[0] = 7;        // flags, channel 3:0 (LAN 1)
	inputData[1] = subfunc;  // Param selector (3 = ip addr)
	memcpy(&inputData[2],pLanRecord,reqlen);

	status = ipmi_cmd(SET_LAN_CONFIG, NETFN_TRANS, inputData, reqlen+2, responseData,
					  &responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("SetLanEntry: completion code=%x\n", completionCode); // responseData[0]);
		}
		else
		{
			//successful, done
			return(0);
		}
	}

	// we are here because completionCode is not COMPLETION_CODE_OK
	if (screenPrint)
	{
		SELprintf("SetLanEntry: sendImbRequest completion code=%d\n", completionCode);
	}
 	return -1;
}  /* end SetLanEntry() */

static int ReadSELinfo()
{
	IMBPREQUESTDATA requestData;
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength;
	uchar completionCode;
	uchar inputData[6];
	IMB_STATUS status;
	struct stat stbuf;
	char *imbDev;
	int   i;

	//build IMB request data
	requestData.cmdType       = GET_SEL_INFO;
	requestData.rsSa          = BMC_SA;     // 0x20
	requestData.rsLun         = BMC_LUN;
	requestData.netFn         = NETFN_STOR;
	requestData.busType       = PUBLIC_BUS;
	requestData.data          = inputData;
	requestData.dataLength    = 0;
	responseLength            = MAX_BUFFER_SIZE;

	status = ipmi_cmd(GET_SEL_INFO, NETFN_STOR, inputData, 0, responseData,
		&responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
	 	SELprintf("Code %d SEL Ver %d Support %d\n",
	 		completionCode, responseData[0], responseData[13]);

		return(0);  //successful, done
	}
	else
	{
		return(1);
	}

}  /*end ReadSELinfo()*/

uchar htoi(uchar *inhex)
{
	// char rghex[16] = "0123456789ABCDEF";
	uchar val;
	uchar c;

	c = inhex[0] & 0x5f;  /* force cap */

	if (c > '9')
	{
		c += 9;  /* c >= 'A' */
	}

	val = (c & 0x0f) << 4;
	c = inhex[1] & 0x5f;  /* force cap */

	if (c > '9')
	{
		c += 9;  /* c >= 'A' */
	}

	val += (c & 0x0f);
	return(val);
}

void atomac(uchar *array, char *instr)
{
	int i,j,n;
	char *pi;

	j = 0;
	pi = instr;
	n = strlen(instr);

	for (i = 0; i <= n; i++)
	{
		if (instr[i] == ':')
	   	{
			array[j++] = htoi(pi);
			pi = &instr[i+1];
		}
		else
		if (instr[i] == 0)
		{
			array[j++] = htoi(pi);
		}
	}
}  /*end atomac()*/

void atoip(uchar *array,char *instr)
{
	int i,j,n;
	char *pi;

	j = 0;
	pi = instr;
	n = strlen(instr);

	for (i = 0; i <= n; i++)
	{
		if (instr[i] == '.')
		{
			instr[i] = 0;
			array[j++] = atoi(pi);
			pi = &instr[i+1];
		}
		else
		if (instr[i] == 0)
		{
			array[j++] = atoi(pi);
		}
	}

	if (screenPrint)
	{
		printf("atoip: %d %d %d %d\n", array[0],array[1],array[2],array[3]);
 	}
}  /*end atoip()*/

///////////////////////////////////////////////////////////////////////////////
// Purpose:
//		Finds the ethernet device from which BMC sends/receives messages.
//
//	Returns
//		Device number for the found device, else -1 on error.
//
//	Assumptions
//		GetLanEntry assumes BMC is using channel 7.
//
int getBmcEthernetDevice( )
{
    LAN_RECORD LanRecord;
    int nDeviceNumberToReturn = -1;
    int ret;
    int nCurDevice;
    char szDeviceName[ MAX_DEVICE_NAME_LENGTH+ + 1 ];
    uchar bmcMacAddress[ MAC_LENGTH ];

	int skfd;
	struct ifreq ifr;

    ret = GetLanEntry( MAC_ADDRESS_LAN_PARAM, &LanRecord );
    if ( ret < 0 )
    {
		printf( "getBmcEthernetDevice: GetLanEntry failed\n" );
		return nDeviceNumberToReturn;
	}

	if(screenPrint)
	{
		printf("BMC MAC %x:%x:%x:%x:%x:%x\n",
			LanRecord.data[ 0 ],
			LanRecord.data[ 1 ],
			LanRecord.data[ 2 ],
			LanRecord.data[ 3 ],
			LanRecord.data[ 4 ],
			LanRecord.data[ 5 ] );
	}

	if ( ( skfd = socket(AF_INET, SOCK_DGRAM, 0 ) ) < 0)
	{
		if ( screenPrint )
		{
			perror("socket");
			return nDeviceNumberToReturn;
		}
	}

	memcpy( bmcMacAddress, LanRecord.data, MAC_LENGTH );

	for ( 	nCurDevice = 0 ;
			( nCurDevice < NUM_DEVICES_TO_CHECK ) &&
			( nDeviceNumberToReturn == -1 );
			nCurDevice++ )
	{
		sprintf( szDeviceName, "eth%d", nCurDevice );

		strcpy(ifr.ifr_name, szDeviceName );
		if (ioctl(skfd, SIOCGIFHWADDR, &ifr) > 0)
		{
			if ( screenPrint )
			{
				printf( "getBmcEthernetDevice: Could not get MAC address for device %d\n" );
			}
		}
		else
		{
			if ( !memcmp( ifr.ifr_hwaddr.sa_data, bmcMacAddress, MAC_LENGTH ) )
			{
				nDeviceNumberToReturn = nCurDevice;
			}
		}
	}

	if ( screenPrint )
	{
		printf( "getBmcEthernetDevice: Returning Device %d\n",
			nDeviceNumberToReturn );
	}

	return nDeviceNumberToReturn;
}


int Get_IPMac_Addr()
{
	char rtfile[] = "/proc/net/route";
	char arpfile[] = "/proc/net/arp";
	char snmpfile[] = "/etc/snmp/snmpd.conf";
	FILE *fprt;
	FILE *fparp;
	int fd, skfd;
	struct ifreq ifr;
	extern int errno;
	uchar *pc;
	int rc = 0;
	int i,j;
	int nBmcEthernetDevice;
	uchar bnetadr[4];
	uchar bgateadr[4];
	char gate_addr[128];
	char buff[1024];
	char alertname[60];
	char szDeviceName[ 10 ];

	nBmcEthernetDevice = getBmcEthernetDevice( );
	if ( nBmcEthernetDevice < 0 )
	{
		if ( screenPrint )
		{
			printf( "Could not find ethernet device.  Using default\n" );
		}

		// app currently assumes channel 7, which might not be true, so
		// we'll use a default if this fails
		nBmcEthernetDevice = DEFAULT_ETHERNET_DEVICE;
	}

	sprintf( szDeviceName, "eth%d", nBmcEthernetDevice );

	/* Get the IP address and associated MAC address specified. */
	/* Local for ethX; Default Gateway; Alert Destination */

	/* Create a channel to the NET kernel. */
	if ((skfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
	{
		perror("socket");
		exit(1);
	}

	strcpy(ifr.ifr_name, szDeviceName);
	if (ioctl(skfd, SIOCGIFHWADDR, &ifr) > 0)
	{
		printf("ioctl(SIOCGIFHWADDR) error, errno = %d\n",errno);
	}
	else
	{
		if (rgmymac[0] == 0)   /* if not user-specified */
		{
			memcpy(rgmymac, ifr.ifr_hwaddr.sa_data, 8);
		}
	}

	fd = skfd;   // get_socket_for_af(AF_INET);

	if (fd >= 0)
	{
		strcpy(ifr.ifr_name, szDeviceName);
		ifr.ifr_addr.sa_family = AF_INET;

		if (ioctl(fd, SIOCGIFADDR, &ifr) < 0)
		{
			printf("ioctl(SIOCGIFADDR) error, errno = %d\n",errno);
		}
		else
		{
			pc = &ifr.ifr_addr.sa_data[2];
			if (screenPrint)
			{
				printf("addr = %d.%d.%d.%d \n", pc[0],pc[1],pc[2],pc[3]);
			}

			if (rgmyip[0] == 0)   /* if not user-specified */
			{
			 	memcpy(rgmyip, pc, 4);
			}

			strcpy(ifr.ifr_name, szDeviceName);

			if (ioctl(fd, SIOCGIFNETMASK, &ifr) < 0)
			{
				printf("ioctl(SIOCGIFNETMASK) error, errno = %d\n",errno);
			}
			else
			{              // sizeof(struct sockaddr)
				pc = &ifr.ifr_netmask.sa_data[2];

				if (screenPrint)
				{
			  		printf("subnet = %d.%d.%d.%d \n", pc[0],pc[1],pc[2],pc[3]);
				}

				if (rgsubnet[0] == 0)   /* if not user-specified */
				{
			     	memcpy(rgsubnet, pc, 4);
				}
			}
		} /* end if(ioctl...) else...*/
	} /* end if(fd>=0) */

	/* Get the default gateway IP */
	/* cat /proc/net/route and save Gwy if Dest == 0 and Gateway != 0 */
	fprt = fopen(rtfile,"r");

	if (fprt == NULL)
	{
		fprintf(stderr,"Cannot open %s, errno = %d\n",rtfile,errno);
	}
	else
	{
		char rtfmt[]  = "%16s %128s %128s %X %d %d %d %128s %d %d %d\n";
		int num, iflags, refcnt, use, metric, mss, window, irtt;
		char iface[16], flags[64];
		char mask_addr[128], net_addr[128];
		ulong *pnet;
		ulong *pgate;

		pnet = (long *)&bnetadr[0];
		pgate = (long *)&bgateadr[0];

		while (fgets(buff, 1023, fprt))
		{
			num = sscanf(buff, rtfmt, iface, net_addr, gate_addr, &iflags, &refcnt,
				   		&use, &metric, mask_addr, &mss, &window, &irtt);

			if (num < 10 || !(iflags & RTF_UP))
			{
				continue;
			}

			j = 6;

			for (i = 0; i < 4; i ++)
			{
				bnetadr[i] = htoi(&net_addr[j]);
				bgateadr[i] = htoi(&gate_addr[j]);
				j -= 2;
			}

			if ((*pnet == 0) && (*pgate != 0))
			{  /* found default gateway */
		    	if (screenPrint)
		    	{
		       		printf("default gateway: %s, %d.%d.%d.%d\n",gate_addr,
		                bgateadr[0], bgateadr[1], bgateadr[2], bgateadr[3]);
				}

	 	    	if (rggwyip[0] == 0)   /* if not user-specified */
				{
		     	  	memcpy(rggwyip,bgateadr,4);
				}
			}  /* end if ((*pnet == 0)... */
		} /*end while*/

		fclose(fprt);
	}  /* end if (fprt == NULL... */

	/* Get the default gateway MAC */
	/* scan the arp cache for a match with gateway */

	if (rggwyip[0] != 0)
	{
		char arping_cmd[80];
		sprintf(gate_addr,"%d.%d.%d.%d", rggwyip[0], rggwyip[1], rggwyip[2], rggwyip[3]);

	 	/* make sure the gateway is in the arp cache */
	 	sprintf(arping_cmd,"arping -c 1 %s >/dev/null \n",gate_addr);

	    if (screenPrint)
	    {
	    	printf(arping_cmd);
		}

	 	system(arping_cmd);

	 	fparp = fopen(arpfile,"r");

	 	if (fparp == NULL)
	 	{
	   		fprintf(stderr,"Cannot open %s, errno = %d\n",rtfile,errno);
	 	}
	 	else
	 	{
	   		char arpfmt[] = "%128s %16s %16s %128s %16s %16s\n";
	   		char iface[16], hw_type[16], rgflags[16], rgmask[16];
	   		char hw_addr[128], net_addr[128];
	   		int num;

	   		while (fgets(buff, 1023, fparp))
	   		{
	     		num = sscanf(buff, arpfmt, net_addr, hw_type, rgflags, hw_addr, rgmask,iface);

	        	if (num < 6)
	        	{
	        		continue;
				}

				if (screenPrint)
				{
	   				printf("arp: %s %s\n",net_addr,hw_addr);
				}

				if (strcmp(gate_addr,net_addr) == 0)  /* found it */
				{
	   				if (screenPrint)
					{
	      				printf("gateway mac: %s\n",hw_addr);
					}

 	   				if (rggwymac[0] == 0)   /* if not user-specified */
					{
			    		atomac(rggwymac,hw_addr);
					}
		   		} /* end if (strcmp(gate_....  */
			} /* end while  */
			fclose(fparp);
		}
	}  /*endif have a gateway ip */

	/* Get the Alert Destination IP */
	/* By default, attempt to obtain this from /etc/snmp/snmpd.conf. */
	/* cat /etc/snmp/snmpd.conf | grep trapsink | cut -f2 -d' ' */
	alertname[0] = 0;  /* default to null string */

	fprt = fopen(snmpfile,"r");

	if (fprt == NULL)
	{
		fprintf(stderr,"Cannot open %s, errno = %d\n",snmpfile,errno);
	}
	else
	{
		// char snmpfmt[] = "%20s %60s\n";
		// char *keywd, *value;
		int num, i;

		while (fgets(buff, 1023, fprt))
		{
			if (strncmp(buff,"trapsink",8) == 0)  /* found it */
			{
				num = strspn(&buff[8]," \t");
				i = 8 + num;
				if (buff[i] == '`') { continue; }

				num = strcspn(&buff[i]," \t\r\n");
				buff[i+num] = 0;
				strcpy(alertname,&buff[i]);
				break;
			}
		}
		fclose(fprt);
	}


	if (screenPrint)
	{
		printf("snmp alertname: %s\n",alertname);
	}

	/* Get the Alert Destination MAC */
	/* Check SSU's algorithm to make sure this is right. */
	if (alertname[0] != 0)
	{
		char alertfile[] = "/tmp/alert.arping";
		char arping_cmd[80];
		char *pb, *pm, *px;
		int num, i;

		if (strncmp(alertname,"localhost",9) == 0)
		{
			if (rgdestip[0] == 0)    /* if not user-specified */
			{
				memcpy(rgdestip,rgmyip,4);
			}

			if (rgdestmac[0] == 0)   /* if not user-specified */
			{
				memcpy(rgdestmac,rgmymac,6);
			}
			else
			{
			    /* make sure the gateway is in the arp cache */
				sprintf(arping_cmd,"arping -c 2 %s |grep reply |tail -1 >%s\n",
						alertname,alertfile);

				if (screenPrint)
				{
					printf(arping_cmd);
				}

				system(arping_cmd);

				fparp = fopen(alertfile,"r");
				if (fparp == NULL)
				{
					fprintf(stderr,"Cannot open %s, errno = %d\n",rtfile,errno);
			    }
			    else
			    {
					while (fgets(buff, 1023, fparp))
					{
						/* should only run through loop once */
						num = strcspn(buff," \t");    /* skip 1st word ("Unicast") */
						i = strspn(&buff[num]," \t");
						pb = &buff[num+i];

						if (strncmp(pb,"reply",5) == 0)
						{  /* valid output */

					    	/* Find the ip address */
							pb += 6 + 5;         /* skip "reply from " */
							num = strcspn(pb," \t");
							pb[num] = 0;

							if (screenPrint)
							{
								printf("Alert ip=%s\n",pb);
							}

							if (rgdestip[0] == 0)   /* if not user-specified */
							{
								atoip(rgdestip,pb);
							}

							/* Now find the mac address */
							pm = strchr(&pb[num+1],'[');
							if (pm == NULL)
							{
								pm = &pb[num+2];  /* just in case */
							}

							pm++;
							px = strchr(pm,']');

							if (px == NULL)
							{
								px = pm + 17;    /* just in case */
							}

							px[0] = 0;
							if (screenPrint)
							{
								printf("Alert mac=%s\n",pm);
							}

							if (rgdestmac[0] == 0)   /* if not user-specified */
							{
								atomac(rgdestmac,pm);
								break;
							}
						}
					} /*end while*/

					fclose(fparp);
				}  /*end else file opened*/
			}
		} /*endif not local */
	}  /*endif have alertname*/


	printf("%s \t\tip=%d.%d.%d.%d mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
		   szDeviceName, rgmyip[0], rgmyip[1], rgmyip[2], rgmyip[3], rgmymac[0],
		   rgmymac[1], rgmymac[2], rgmymac[3], rgmymac[4], rgmymac[5]);

	return(rc);
} /* end Get_IPMac_Addr */

void ShowChanAcc(uchar bchan)
{
    LAN_RECORD LanRecord;
    int ret;
    uchar access;
    char *pstr;

    if (bchan == 1) pstr = "ser";
    else if (bchan == 7) pstr = "lan";
    else pstr = "";
    ret = GetChanAcc(bchan, 0x40, &LanRecord);
    access = LanRecord.data[0];
    printf("GetChanAcc(%s), ret = %d, new value = %02x\n",
		pstr,ret,access);
    if (access & 0x20) printf("   PEF Alerts Disabled\n"); /*0*/
    else printf("   PEF Alerts Enabled\n");   /*1*/
    switch (access & 0x03) {
	     case 0: printf("   Access = Disabled\n");     break;
	     case 1: printf("   Access = Pre-Boot\n");     break;
	     case 2: printf("   Access = Always Avail\n"); break;
	     case 3: printf("   Access = Shared\n");       break;
	     }
}

/////////////////////////////////////////////////////////////////

int SetupSerialOverLan( void )
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	IMB_STATUS status;
	uchar requestData[24];
	uchar completionCode;
	int foo = GET_SOL_CONFIG;

	memset( requestData, 0, MAX_BUFFER_SIZE );  /* zero-fill */

	requestData[ 0 ] = 0x00;
	requestData[ 1 ] = SOL_ENABLE_PARAM;
	requestData[ 2 ] = SOL_ENABLE_FLAG;

	status = ipmi_cmd( SET_SOL_CONFIG, NETFN_SOL, requestData, 3, responseData,
			&responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		switch( completionCode )
		{
			case 0x00: /* success */
			break;

			case 0xC1: /* unsupported command */
				if ( screenPrint )
				{
					SELprintf( "\nSetupSerialOverLan: SOL not available on this platform" );
					return 0;
				}

				return 0;
			default: /* other error */
				SELprintf(	"\nSetupSerialOverLan: SOL_ENABLE_PARAM code=%x",
					completionCode );
				return -1;
			break;
		}
	}
	else
	{
		SELprintf( "SET_SOL_CONFIG, enable SOL failed\n" );
	 	return -1;
 	}

	requestData[ 0 ] = 0x00;
	requestData[ 1 ] = SOL_AUTHENTICATION_PARAM;
	requestData[ 2 ] = 0x00; // block
	requestData[ 3 ] = 0x00; // selector

	status = ipmi_cmd( GET_SOL_CONFIG, NETFN_SOL, requestData, 4, responseData,
			&responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("\n\n\nSetupSerialOverLan: GET_SOL_AUTHENTICATION_PARAM code=%x\n",
				completionCode);

			return -1;
		}
	}
	else
	{
		SELprintf( "SET_SOL_CONFIG, get SOL authentication failed\n" );
	 	return -1;
 	}

	requestData[ 0 ] = 0x00;
	requestData[ 1 ] = SOL_AUTHENTICATION_PARAM;
	requestData[ 2 ] = SOL_PRIVILEGE_LEVEL_USER | ( responseData[ 1 ] & 0x80 );

	status = ipmi_cmd( SET_SOL_CONFIG, NETFN_SOL, requestData, 3, responseData,
			&responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("\n\n\nSetupSerialOverLan: SET_SOL_AUTHENTICATION_PARAM code=%x\n",
				completionCode);

			return -1;
		}
	}
	else
	{
		SELprintf( "SET_SOL_CONFIG, set SOL authentication failed\n" );
	 	return -1;
 	}

	requestData[ 0 ] = 0x00;
	requestData[ 1 ] = SOL_BAUD_RATE_PARAM;
	requestData[ 2 ] = SOL_PREFERRED_BAUD_RATE;

	status = ipmi_cmd( SET_SOL_CONFIG, NETFN_SOL, requestData, 3, responseData,
			&responseLength, &completionCode, screenPrint);

	if (status == ACCESS_OK)
	{
		if( completionCode )
		{
			SELprintf("\n\n\nSetupSerialOverLan: SET SOL_PREFFERED_BAUD code=%x\n",
				completionCode);

			return -1;
		}
	}
	else
	{
		SELprintf( "SET_SOL_CONFIG, set SOL BAUD failed\n" );
	 	return -1;
 	}

	return 0;
}  /* SetupSerialOverLan */


main(int argc, char **argv)
{
	int ret;
	LAN_RECORD LanRecord;
	int i, idx, j;
	char c;
	uchar bchan;


   	//command parser definitions
   	while ( (c = getopt( argc, argv,"rwd:I:M:S:G:H:A:B:P:C:?")) != EOF )
	{

	   	switch(c)
		{
	        case 'r': freadonly = 1;  break;
	        case 'd': fdisable = 1;   break;

	        case 'w':
				screenPrint = 1;
				printf("\nDebug printing enabled\n");
				break;

	        case 'I': atoip (rgmyip,   optarg); break;	/* My IP Address (eth0) */
			case 'M': atomac(rgmymac,  optarg);	break;	/* My MAC Address */
	        case 'S': atoip (rgsubnet, optarg);	break;	/* Subnet IP Address */
	        case 'G': atoip (rggwyip,  optarg);	break;  /* Gateway IP Address */
	        case 'H': atomac(rggwymac, optarg);	break;	/* Gateway MAC Address */
	        case 'A': atoip (rgdestip, optarg);	break;	/* Alert Dest IP Address */
	        case 'B': atomac(rgdestmac,optarg);	break;	/* Alert Dest MAC Address */

	        case 'C':      /* Community String */
				if (strlen(optarg) > 18) optarg[18] = 0;
				strcpy(rgcommunity,optarg);
				if(screenPrint)
				{
					printf("\nCommunity String = %s\n", rgcommunity);
				}
				break;

	        case 'P':      /* Password */
				if (strlen(optarg) > 16) 
				{
					optarg[16] = 0;
					fprintf(stderr, "bmcconfig: The specified password is too long. The password was truncated to 16 characters.\n");
				}
				strcpy(passwordData,optarg);
				if(screenPrint)
				{
					printf("\nPassword = %s\n\n",passwordData);
				}
				break;

		  	default:
	        	printf("Usage: %s [-dlrsx]\n", progname);
	        	printf("         \t [-I ipadr -M macadr -S subnet]\n");
	        	printf("         \t [-G gwyip -H gwymac ]\n");
	        	printf("         \t [-A alertip -B alertmac -C community -P password]\n");
	        	exit(1);
	    }
	}

	if(screenPrint)
	{
		printf("%s ver %s starting ...\n",progname,progver);
	}

	ret = GetDeviceID(&LanRecord);

	if (ret == 0)
	{
		uchar ipmi_maj, ipmi_min;

		ipmi_maj = LanRecord.data[4] & 0x0f;
		ipmi_min = LanRecord.data[4] >> 4;

		if(screenPrint)
		{
			printf("-- BMC version %x.%x, IPMI version %d.%d \n", LanRecord.data[2],  LanRecord.data[3], ipmi_maj, ipmi_min);
		}

		if (ipmi_maj >= 1 && ipmi_min >= 5)
		{
			;  /* >= IPMI 1.5 is ok */
		}
		else
		{
			exit(-1);
		}
	}

	ret = ReadSELinfo();

	if (ret == 0)
	{
		if (!freadonly)
		{
			uchar access;

			/* Disable LAN channel and PEf in volatile and non-volatile memory */
			if (fdisable)
			{
				ret = SetChanAcc(7, 0x80, 0x20);   /* PEF off, disabled */
				ret = SetChanAcc(7, 0x40, 0x20);   /* PEF off, disabled */
			} else {
				ret = SetUser();
				ret = SetLanConfig();
			}

			/* Get the values to use from Linux eth0, etc. */
			ret = Get_IPMac_Addr();

			if (rgmyip[0] != 0)
		 	{
				memcpy(&LanRecord,rgmyip,4);
				ret = SetLanEntry(3, &LanRecord, 4);
				LanRecord.data[0] = 0x01;    /* static IP address source */
				ret = SetLanEntry(4, &LanRecord, 1);
				memcpy(&LanRecord,rgmymac,6);
				ret = SetLanEntry(5, &LanRecord, 6);
				memcpy(&LanRecord,rgsubnet,4);
				ret = SetLanEntry(6, &LanRecord, 4);
			} else 	{      /* error, don't continue */
				printf("Missing IP Address, can't continue\n");
				exit(1);
			}

		 	if (rggwyip[0] != 0)
			{
				printf("gateway \tip=%d.%d.%d.%d mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
						rggwyip[0], rggwyip[1], rggwyip[2], rggwyip[3],
						rggwymac[0], rggwymac[1], rggwymac[2], rggwymac[3],
						rggwymac[4], rggwymac[5]);

				/* Set the Default Gateway IP & MAC */
				memcpy(&LanRecord,rggwyip,4);
				ret = SetLanEntry(12, &LanRecord, 4);
				memcpy(&LanRecord,rggwymac,6);
				ret = SetLanEntry(13, &LanRecord, 6);
			}


			/* community string and alert IP adderss */
			if(screenPrint)
			{
            	printf("alert dest \tip=%d.%d.%d.%d mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
				rgdestip[0], rgdestip[1], rgdestip[2], rgdestip[3],
				rgdestmac[0], rgdestmac[1], rgdestmac[2], rgdestmac[3],
				rgdestmac[4], rgdestmac[5]);
			}

		    /* Only need the SNMP community if there is an Alert Destination */
			memset(&LanRecord.data[0], 0, 18);  /* zero-fill */
			strcpy(&LanRecord.data[0],rgcommunity);
			ret = SetLanEntry(16, &LanRecord, 18);
			if(screenPrint)
			{
            	printf("SetLanEntry(16), ret = %d\n",ret);
			}

	  	  	/* Set Alert Destination Type */
	 	   	LanRecord.data[0] = 0x01;     /* dest id = 1 */
			LanRecord.data[1] = 0x80;     /* dest type = PET & ack */
			LanRecord.data[2] = 0x01;
			LanRecord.data[3] = 0x03;
			// LanRecord.data[4] = 0x69;
			ret = SetLanEntry(18, &LanRecord, 4);
			if(screenPrint)
			{
            	printf("SetLanEntry(18), ret = %d\n",ret);
			}

	    	/* Set the Alert Destination IP & MAC */
	    	LanRecord.data[0] = 0x01;     /* dest id = 1 */
	    	LanRecord.data[1] = 0x00;
	    	LanRecord.data[2] = 0x00;
	    	memcpy(&LanRecord.data[3],rgdestip,4);
	    	memcpy(&LanRecord.data[7],rgdestmac,6);
            ret = SetLanEntry(19, &LanRecord, 13);
			if(screenPrint)
			{
            	printf("SetLanEntry(19), ret = %d\n", ret);
			}
		}  /*endif*/
	}  /* if ret == ok */


	ret = SetupSerialOverLan( );
	if ( screenPrint )
	{
		printf( "SetupSerialOverLan( ), ret = %d\n", ret );
	}


}  /* end main()*/

/* end pefconfig.c */
