
{Lists all DOS partitions on all hard disks, along with volume label and
  serial number; proof-of-concept test program}
program BootRec;

const
{Tab character}
  chTab         = #9;
{Default number of bytes per sector}
  DefBytePerSec = $0200;
{Boot record and boot sector signature}
  BootSectorSign= $AA55;
{Number of partition entries per boot record}
  PartPerBootRec= 4;
{Volume label size}
  VolumeLabelSize= 11;
{File system name size}
  FileSysNameSize= 8;
{DOS-compatible partition types}
  ptEmpty       = $00;
  ptFAT12       = $01;
  ptFAT16Max32MB= $04;
  ptExtended    = $05;
  ptFAT16       = $06;
  ptFAT32       = $0B;
  ptFAT32LBA    = $0C;
  ptFAT16LBA    = $0E;
  ptExtendedLBA = $0F;
{Hexadecimal digits}
  HexaNum       : string[16] = '0123456789ABCDEF';
{Empty volume label}
  EmptyVolLabel : string[7] = 'NO NAME';

type
{64-bit long long integer}
  Longlongint   = record
    LoLong,
    HiLong: Longint;
  end;
{Pointer to byte}
  PByte = ^Byte;
{Pointer to string}
  PString = ^string;
{Disk sector}
  TSector       = array [0..DefBytePerSec - 1] of Byte;
{BIOS parameter block}
  BIOSParBlock  = record
    bbJumpToExec: array [0..2] of Byte;
    bbOEMNameVer: array [0..7] of Byte;
    bbBytePerSector: Word;
    bbSectorPerClus: Byte;
    bbResSectorNum: Word;
    bbFATNum: Byte;
    bbRootDirLen: Word;
    bbSectorNum: Word;
    bbMediaID: Byte;
    bbSectorPerFAT: Word;
    bbSecPerTrack: Word;
    bbHeadNum: Word;
    bbHiddenSecNum: Longint;
    bbExtSectorNum: Longint;
  end;
{Extended BIOS parameter block}
  ExtBIOSParBlock= record
    xbBIOSParBlock: BIOSParBlock;
    xbExtSecParFAT: Longint;
    xbExtFlags: Word;
    xbFileSysVer: Word;
    xbRootDirClus: Longint;
    xbFSInfoSec: Word;
    xbBakBootSec: Word;
    xbReserved: array [0..5] of Word;
  end;
{File system information structure}
  FileSysInfo   = record
    fiPhysDriveNum: Byte;
    fiReserved: Byte;
    fiSignature: Byte;
    fiSerialNum: Longint;
    fiVolumeLabel: array [0..VolumeLabelSize - 1] of Byte;
    fiFileSystem: array [0..FileSysNameSize - 1] of Byte;
  end;
{Boot sector}
  BootSector    = record
    bsBIOSParBlock: BIOSParBlock;
    bsFileSysInfo: FileSysInfo;
  end;
  PBootSector = ^BootSector;
{Extended boot sector}
  ExtBootSector = record
    xsBIOSParBlock: ExtBIOSParBlock;
    xsFileSysInfo: FileSysInfo;
  end;
  PExtBootSector = ^ExtBootSector;
{Partition table entry}
  PartitionEntry= record
    peActive: Byte;
    peStartHead: Byte;
    peStartCylSec: Word;
    peType: Byte;
    peEndHead: Byte;
    peEndCylSec: Word;
    peStartRelSec: Longint;
    peSize: Longint;
  end;
{Boot record}
  BootRecord= record
    brCode: array [0..DefBytePerSec - 1 - SizeOf(Word) - PartPerBootRec * SizeOf(PartitionEntry)] of Byte;
    brPartTable: array [0..PartPerBootRec - 1] of PartitionEntry;
    brSignature: Word;
  end;
  PBootRecord   = ^BootRecord;
{Data structure for extended disk read/write functions}
  ExtDiskFuncPack= record
    xfSize: Byte;
    xfReserved: Byte;
    xfSectorNum: Word;
    xfDataBuffer: Longint;
    xfLBASector: Longlongint;
  end;
{Extended disk parameter block}
  ExtDiskParam= record
    xpSize: Word;
    xpFlags: Word;
    xpCylPerDisk: Longint;
    xpHeadPerCyl: Longint;
    xpSecPerHead: Longint;
    xpTotalSecNum: Longlongint;
    xpBytePerSec: Word;
  end;
{Sector position}
  SectorPos  = record
    case Byte of
      0: (Cylinder,
          Head,
          Sector: Word);
      1: (LBASector: Longint);
  end;
  PSectorPos = ^SectorPos;

var
  ExtDiskFunc: Boolean;
  PartNum,
  Drive: Byte;
  SerialNumber,
  CylPerDisk,
  HeadPerCyl,
  SecPerHead: Longint;
  SecPos,
  ExtStartSecPos: SectorPos;
  DiskFuncPack: ExtDiskFuncPack;
  DiskParam: ExtDiskParam;
  BootRecordBuf,
  BootSectorBuf: TSector;
  PartName: string[6];
  VolumeLabel: string[11];
  OutputFile: file;

(* DEBUG
{Get disk geometry
  Input : DL: drive number (80h=first hard disk; 81h=second hard disk; ...
  Output: AL: error code; 0, if no error}
function GetDiskGeo(Drive: Byte): Byte; assembler;
asm
    xor di, di;
    mov es, di;
    mov dl, Drive;
    mov ah, $08;
    int $13;
    mov al, ah;
    jc @1;
    mov ax, cx;
    rol al, 2;
    and al, $03;
    xchg al, ah;
    inc ax;
    mov word ptr CylPerDisk[0], ax;
    xor ax, ax;
    mov al, dh;
    inc ax;
    mov word ptr HeadPerCyl[0], ax;
    and cx, $003F;
    mov word ptr SecPerHead[0], cx;
    xor ax, ax;
    mov word ptr CylPerDisk[2], ax;
    mov word ptr HeadPerCyl[2], ax;
    mov word ptr SecPerHead[2], ax;
    xor al, al;
    clc;
@1:
end;

{Get extended disk geometry
  Input : DL: drive number (80h=first hard disk; 81h=second hard disk; ...
  Output: AL: error code; 0, if no error occurred}
function ExtGetDiskGeo(Drive: Byte): Byte; assembler;
asm
    mov si, Offset(DiskParam);
    mov [si].ExtDiskParam.xpSize, Type(DiskParam);
    mov dl, Drive;
    mov ah, $48;
    int $13;
    mov al, ah;
    jc @1;
    mov ax, word ptr [si].ExtDiskParam.xpCylPerDisk[0]
    mov word ptr CylPerDisk[0], ax;
    mov ax, word ptr [si].ExtDiskParam.xpCylPerDisk[2]
    mov word ptr CylPerDisk[2], ax;
    mov ax, word ptr [si].ExtDiskParam.xpHeadPerCyl[0]
    mov word ptr HeadPerCyl[0], ax;
    mov ax, word ptr [si].ExtDiskParam.xpHeadPerCyl[2]
    mov word ptr HeadPerCyl[2], ax;
    mov ax, word ptr [si].ExtDiskParam.xpSecPerHead[0]
    mov word ptr SecPerHead[0], ax;
    mov ax, word ptr [si].ExtDiskParam.xpSecPerHead[2]
    mov word ptr SecPerHead[2], ax;
    xor al, al;
    clc;
@1:
end;

{Write sector contents into a file
  Input : Name: file name
          Buffer: buffer holding sector contents}
procedure WriteSectorIntoFile(const Name: string; var Buffer: TSector);
begin
  Assign(OutputFile, Name);
  Rewrite(OutputFile, 1);
  BlockWrite(OutputFile, Buffer, DefBytePerSec);
  Close(OutputFile);
end;

{Write boot sector contents into a file
  Input : PartNum: partition number
          Buffer: buffer holding sector contents}
procedure WriteBootSectorIntoFile(PartNum: Byte; var Buffer: TSector);
begin
  WriteSectorIntoFile(PartName + '.bin', Buffer);
end;
   DEBUG *)

{Determine if disk supports extended INT13 functions
  Input : DL: drive number (80h=first hard disk; 81h=second hard disk; ...
  Output: AL: True, if extended INT13 functions supported; False otherwise}
function IsExtDiskFunc(Drive: Byte): Boolean; assembler;
asm
    mov dl, Drive;
    mov bx, $55AA;
    mov ah, $41;
    int $13;
    mov al, False;
    jc @1;
    cmp bx, $AA55;
    jne @1;
    test cl, 1;
    je @1;
    inc al;
@1:
end;

{Read sector from disk in CHS mode
  Input : Drive: drive number (80h=first hard disk; 81h=second hard disk; ...
          Cylinder, Head. Sector: cylinder, head and sector number; sector
                                  position in CHS notation
          Buffer: buffer to hold sector contents
  Output: error code; 0, if no error occurred}
function ReadSector(Drive: Byte; Cylinder, Head, Sector: Word; var Buffer: TSector): Byte; assembler;
asm
    mov cx, Cylinder;
    xchg cl, ch;
    ror cl, 2;
    mov ax, Sector;
    or cl, al;
    mov ax, Head;
    mov dh, al;
    les bx, Buffer;
    mov dl, Drive;
    mov ax, $0201;
    int $13;
    mov al, ah;
end;

{Read sector from disk in LBA mode
  Input : Drive: drive number (80h=first hard disk; 81h=second hard disk; ...
          LBASector: LBA sector number; sector position in LBA notation
          Buffer: buffer to hold sector contents
  Output: error code; 0, if no error occurred}
function ExtReadSector(Drive: Byte; LBASector: Longint; var Buffer: TSector): Byte; assembler;
asm
    mov DiskFuncPack.xfSize, Type(ExtDiskFuncPack);
    mov DiskFuncPack.xfReserved, 0;
    mov DiskFuncPack.xfSectorNum, 1;
    les bx, Buffer;
    mov word ptr DiskFuncPack.xfDataBuffer[0], bx;
    mov word ptr DiskFuncPack.xfDataBuffer[2], es;
    mov ax, word ptr LBASector[0];
    mov word ptr DiskFuncPack.xfLBASector[0], ax;
    mov ax, word ptr LBASector[2];
    mov word ptr DiskFuncPack.xfLBASector[2], ax;
    xor ax, ax;
    mov word ptr DiskFuncPack.xfLBASector[4], ax;
    mov word ptr DiskFuncPack.xfLBASector[6], ax;
    mov dl, Drive;
    mov si, Offset(DiskFuncPack);
    push ds;
    pop es;
    mov ah, $42;
    int $13;
    mov al, ah;
end;

{Read sector from disk in either CHS or LBA mode
  Input : Buffer: buffer to hold sector contents
  Output: error code; 0, if no error occurred}
function ReadSector2(const SecPos: SectorPos; var Buffer: TSector): Byte;
begin
  if ExtDiskFunc then ReadSector2 := ExtReadSector(Drive, SecPos.LBASector, Buffer) else
    ReadSector2 := ReadSector(Drive, SecPos.Cylinder, SecPos.Head, SecPos.Sector, Buffer);
end;

{Trim all trailing spaces from a string
  Input : Str: the original string
  Output: the trimmed string}
function RightTrim(Str: string): string;
begin
  while (Str <> '') and (Str[Length(Str)] = ' ') do Dec(Str[0]);
  RightTrim := Str;
end;

{Fetch a zero-terminated string
  Input : Str: pointer to the first character of the string
          MaxLen: maximum length
  Output: the fetched string}
function GetStr(Str: PByte; MaxLen: Byte): string; assembler;
asm
    push ds;
    mov cl, MaxLen;
    xor ch, ch;
    xor bx, bx;
    lds si, Str;
    les di, @Result;
    push di;
    inc di;
    jcxz @1;
@2: lodsb;
    or al, al;
    je @1;
    stosb;
    inc bx;
    loop @2;
@1: pop di;
    mov ss:[di], bl;
    pop ds;
end;

{Convert a long integer into a hexadecimal string
  Input : D: the long integer
          L: number of digits to put into the string
  Output: the hexadecimal string}
function HexaStr(D: Longint; L: Byte): string;
var
  I             : Byte;
  S             : string;
begin
  S := '';
  for I := L - 1 downto 0 do S := S + HexaNum[(D shr (I * 4) and $0F) + 1];
  HexaStr := S;
end;

{Process boot sector
  Input : SecPos: sector position
          MBR: when True, the master boot record is to be processed}
procedure ProcessBootSector(const SecPos: SectorPos; MBR: Boolean);

{Process boot sector, main procedure
  Input : FollowExt: when True, extended partitions are processed; otherwise
                     normal partitions}
procedure ProcessBootSectorInternal(FollowExt: Boolean);
var
  PartType,
  PartEntryNum: Byte;
  CylSec,
  Head: Word;
  NewSecPos: SectorPos;

{Convert relative sector position to absolute sector position
  Input : BaseSecPos: absolute sector position of base sector}
procedure FixSectorPos(BaseSecPos: PSectorPos);
begin
  if ExtDiskFunc then
  begin
    Inc(NewSecPos.LBASector, BaseSecPos^.LBASector);
  end
  else
  begin
    Inc(NewSecPos.Cylinder, BaseSecPos^.Cylinder);
    Inc(NewSecPos.Head, BaseSecPos^.Head);
    Inc(NewSecPos.Sector, BaseSecPos^.Sector);
  end;
end;

begin
  for PartEntryNum := 0 to PartPerBootRec - 1 do
  begin
    if ExtDiskFunc then
    begin
      NewSecPos.LBASector := PBootRecord(@BootRecordBuf)^.brPartTable[PartEntryNum].peStartRelSec;
    end
    else
    begin
      CylSec := PBootRecord(@BootRecordBuf)^.brPartTable[PartEntryNum].peStartCylSec;
      Head := PBootRecord(@BootRecordBuf)^.brPartTable[PartEntryNum].peStartHead;
      asm
        mov ax, CylSec;
        rol al, 2;
        and al, $03;
        xchg al, ah;
        mov word ptr NewSecPos.Cylinder[0], ax;
        mov ax, CylSec;
        and ax, $003F;
        mov word ptr NewSecPos.Sector[0], ax;
        mov ax, Head;
        mov word ptr NewSecPos.Head[0], ax;
        xor ax, ax;
        mov word ptr NewSecPos.Cylinder[2], ax;
        mov word ptr NewSecPos.Head[2], ax;
        mov word ptr NewSecPos.Sector[2], ax;
      end;
    end;
    PartType := PBootRecord(@BootRecordBuf)^.brPartTable[PartEntryNum].peType;
    Str(PartNum, PartName);
    PartName := 'hd' + Chr(Drive - ($80 - Ord('a'))) + PartName;
    case PartType of
      ptEmpty: ;
      ptExtended, ptExtendedLBA:
      begin
        if FollowExt then
        begin
          if MBR then ExtStartSecPos := NewSecPos else FixSectorPos(@ExtStartSecPos);
          ProcessBootSector(NewSecPos, False);
          Break;
        end;
      end;
    else
      if not FollowExt then
      begin
        if PartType in [ptFAT12, ptFAT16Max32MB, ptFAT16, ptFAT16LBA, ptFAT32, ptFAT32LBA] then
        begin
          FixSectorPos(@SecPos);
          if (ReadSector2(NewSecPos, BootSectorBuf) = 0) and (PBootRecord(@BootSectorBuf)^.brSignature = BootSectorSign) then
          begin
(* DEBUG
            WriteBootSectorIntoFile(PartNum, BootSectorBuf);
   DEBUG *)
            if PartType in [ptFAT32, ptFAT32LBA] then
            begin
              SerialNumber := PExtBootSector(@BootSectorBuf)^.xsFileSysInfo.fiSerialNum;
              VolumeLabel := RightTrim(GetStr(@PExtBootSector(@BootSectorBuf)^.xsFileSysInfo.fiVolumeLabel, VolumeLabelSize));
            end
            else
            begin
              SerialNumber := PBootSector(@BootSectorBuf)^.bsFileSysInfo.fiSerialNum;
              VolumeLabel := RightTrim(GetStr(@PBootSector(@BootSectorBuf)^.bsFileSysInfo.fiVolumeLabel, VolumeLabelSize));
            end;
            if VolumeLabel = EmptyVolLabel then VolumeLabel := '';
            WriteLn(PartName, chTab, VolumeLabel, chTab, HexaStr(SerialNumber, 8));
          end;
        end;
        if not MBR then Inc(PartNum);
      end;
    end;
    if MBR and not FollowExt then Inc(PartNum);
  end;
end;

begin
  if (ReadSector2(SecPos, BootRecordBuf) = 0) and (PBootRecord(@BootRecordBuf)^.brSignature = BootSectorSign) then
  begin
(* DEBUG
    if MBR then WriteBootSectorIntoFile(0, BootRecordBuf);
   DEBUG *)
    ProcessBootSectorInternal(False);
    ProcessBootSectorInternal(True);
  end;
end;

begin
  for Drive := $80 to $87 do
  begin
    ExtDiskFunc := IsExtDiskFunc(Drive);
(* DEBUG
    GetDiskGeo(Drive);
   DEBUG *)
    if ExtDiskFunc then
    begin
      SecPos.LBASector := 0;
    end
    else
    begin
      SecPos.Cylinder := 0;
      SecPos.Head := 0;
      SecPos.Sector := 1;
    end;
    PartNum := 1;
    ProcessBootSector(SecPos, True);
  end;
end.
