;
;   1541EMU, the Commodore 1541 disk drive emulator
;   Copyright (C) 2001-2002 Ville Muikkula
;   Copyright (C) Karsten Scheibler
;
;  This file was formerly part of ec64, the Commodore 64 emulator for Linux.
;  Written by
;   Karsten Scheibler <karsten.scheibler@bigfoot.de>
;  Modifications by
;   Ville Muikkula <1541@surfeu.fi>
;
;   This program is free software; you can redistribute it and/or modify
;   it under the terms of the GNU General Public License version 2 as
;   published by the Free Software Foundation.
;
;   This program is distributed in the hope that it will be useful,
;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;   GNU General Public License for more details.
;
;   You should have received a copy of the GNU General Public License
;   along with this program; if not, write to the Free Software
;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;****************************************************************************
;****************************************************************************
;*
;* core.n
;*
;****************************************************************************
;****************************************************************************


%define CORE_N
%include "core.i"
%include "link.i"
%include "timing.i"
%include "print.i"
%include "main.i"
%include "drive.i"
%include "status.i"

%define MODULID                 CORE_MODULID
%define C1541_CYCLES            CORE_C1541_CYCLES
%define CYCLE_DIFFERENCE1       CORE_CYCLE_DIFFERENCE1
%define CYCLE_DIFFERENCE2       CORE_CYCLE_DIFFERENCE2
%define CYCLE_COUNTER           CORE_CYCLE_COUNTER
%define CYCLE_COUNTER_BASE      CORE_CYCLE_COUNTER_BASE
%define CYCLES_COUNTED          CORE_CYCLES_COUNTED
%define CYCLES_LEFT             CORE_CYCLES_LEFT

%define STACK_POINTER_EXTERN    CORE_STACK_POINTER_EXTERN
%define STACK_POINTER           CORE_STACK_POINTER
%define STACK                   CORE_STACK

%define INFORMATION_MODULID     CORE_INFORMATION_MODULID
%define INFORMATION_TYPE        CORE_INFORMATION_TYPE
%define INFORMATION_NUMBER      CORE_INFORMATION_NUMBER

%define LOG_BUFFER              CORE_LOG_BUFFER
%define LOG_OFFSET              CORE_LOG_OFFSET
%define LOG_SIZE                CORE_LOG_SIZE

%define C1541                   CORE_C1541



;****************************************************************************
;* some data ****************************************************************
;****************************************************************************
section .data
disk_change_phase:      dd      0
phase_advance_moment:   dd      0FFFFFFFFh, 0FFFFFFFFh
PROCESSING_DISK_CHANGE: db      0
_CURRENT_GCR_IMAGE:     dd      0

_bamdir: incbin "bamdir.bin"

section .bss
tcore__1:                       resb    tcore_size

tc1541__1:                      equ     C1541



;****************************************************************************
;* core_enter ***************************************************************
;****************************************************************************
section .text
core_enter:
                        mov     dword [STACK_POINTER_EXTERN], esp
                        mov     dword esp, [STACK_POINTER]
                        pop     dword ebp
                        pop     dword edi
                        pop     dword esi
                        pop     dword edx
                        pop     dword ecx
                        pop     dword ebx
                        pop     dword eax
                        ret



;****************************************************************************
;* core_leave ***************************************************************
;****************************************************************************
section .text
core_leave:
                        push    dword eax
                        push    dword ebx
                        push    dword ecx
                        push    dword edx
                        push    dword esi
                        push    dword edi
                        push    dword ebp
                        mov     dword [STACK_POINTER], esp
                        mov     dword esp, [STACK_POINTER_EXTERN]
                        ret



;****************************************************************************
;* core_reset_hard **********************************************************
;****************************************************************************
section .text
_core_reset_hard:
                        ;-----------------------------------------
                        ;initialize the local core stack
                        ;ASSUMPTION: predecrementing stack pointer
                        ;-----------------------------------------

                        mov     dword [STACK_POINTER_EXTERN], esp
                        lea     dword esp, [STACK + CORE_STACK_SIZE]

                        ;--------------------------
                        ;reset the emulated modules
                        ;--------------------------

                        lea     dword esi, [C1541]
                        C1541LINK_CALL_RESET_HARD
                        C1541MEMORY_CALL_RESET_HARD
                        CPU65XX_CALL_RESET_HARD
                        VIA6522_CALL_RESET_HARD  tvia6522__c1541_1
                        mov byte al, 0FEh
                        mov     dword edi, (tc1541__1 + tvia6522__c1541_1)
                        call    via6522_signal_pa
                        VIA6522_CALL_RESET_HARD  tvia6522__c1541_2
                        DISC_CALL_RESET_HARD

                        xor     dword eax, eax
                        call    core_leave
                        jmp     _core_cycle_loop



;****************************************************************************
;* core_initialize **********************************************************
;****************************************************************************
section .text
_core_initialize:
                        ;-----------------------------
                        ;reset the core data structure
                        ;-----------------------------

                        xor     dword eax, eax
                        mov     dword ecx, tcore_size
                        lea     dword edi, [MODULID]
                        cld
                        rep stosb

                        ;-----------------------------------------
                        ;initialize the local core stack
                        ;ASSUMPTION: predecrementing stack pointer
                        ;-----------------------------------------

                        mov     dword [STACK_POINTER_EXTERN], esp
                        lea     dword esp, [STACK + CORE_STACK_SIZE]

                        ;--------------------------------
                        ;initialize the emulation modules
                        ;--------------------------------

                        C1541LINK_CALL_INITIALIZE
                        C1541MEMORY_CALL_INITIALIZE
                        CPU65XX_CALL_INITIALIZE
                        VIA6522_CALL_INITIALIZE
                        DISC_CALL_INITIALIZE

                        ;-------------------------------
                        ;initialize the module instances
                        ;-------------------------------

                        lea     dword esi, [C1541]
                        C1541LINK_CALL_INSTANCE_INITIALIZE  CORE_MODULID_C1541LINK
                        C1541MEMORY_CALL_INSTANCE_INITIALIZE  CORE_MODULID_C1541MEMORY

                        CPU65XX_CALL_INSTANCE_INITIALIZE  CORE_MODULID_C1541CPU, _core_get_c1541_cycles, c1541memory_read_opcode, c1541memory_read_stack, c1541memory_read_data, c1541memory_write_stack, c1541memory_write_data

                        VIA6522_CALL_INSTANCE_INITIALIZE  tvia6522__c1541_1, CORE_MODULID_C1541VIA1, C1541LINK_VIA1_COUNTER_OFFSET, c1541link_via1_irq, c1541link_via1_pb, c1541link_via1_cb2, c1541link_via1_pa, c1541link_via1_ca2, c1541link_via1_ca1
                        VIA6522_CALL_INSTANCE_INITIALIZE  tvia6522__c1541_2, CORE_MODULID_C1541VIA2, C1541LINK_VIA2_COUNTER_OFFSET, c1541link_via2_irq, c1541link_via2_pb, c1541link_via2_cb2, c1541link_via2_pa, c1541link_via2_ca2, c1541link_via2_ca1
                        DISC_CALL_INSTANCE_INITIALIZE  CORE_MODULID_C1541DISC

                        movzx dword eax, byte [_disk_number]
                        mul byte [_disk_image_t_size]
                        add dword eax, _diskette
                        mov byte al, [eax+4]         ; read_only
                        mov byte [_Write_Protect_Sense], al

                        ;--------------------
                        ;return to the caller
                        ;--------------------

                        xor     dword eax, eax
                        call    core_leave
                        mov     dword esp, [STACK_POINTER_EXTERN]
                        jmp     _core_reset_hard



;****************************************************************************
;* core_exit ****************************************************************
;****************************************************************************
section .text
core_exit:
                        mov     dword esp, [STACK_POINTER_EXTERN]
                        popad
                        popfd
                        ret



;****************************************************************************
;* core_new_frame ***********************************************************
;****************************************************************************
section .text
core_new_frame:
                        xor     dword eax, eax
                        jmp     core_leave



;****************************************************************************
;* core_emulate *************************************************************
;****************************************************************************
section .text
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;TODO: throw an exception if the core isn't initialized
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
core_emulate:           equ     core_enter



;****************************************************************************
;* core_cycle_loop **********************************************************
;****************************************************************************
section .text
_core_cycle_loop:
                        pushfd
                        pushad
                        mov     dword [STACK_POINTER_EXTERN], esp
                        ;----------------------------
                        ;initialize the clockcounters
                        ;----------------------------
                        mov     dword [CORE_C1541_CYCLES], 0
                        mov     dword [CORE_C1541_CYCLES + 4], 0

                        mov byte [_Key_Code], 0
                        mov dword [disk_change_phase], .phase1
                        call _init_timing

                        call status_vc1541_IO_tuning

.run_c1541_cycle:
;----------------------
;one c1541 clock cycle
;----------------------
                        lea dword esi, [C1541]
                        CPU65XX_CALL_CYCLE
                        C1541LINK_CALL_DECREMENT_COUNTER
                        DISC_CALL_CYCLE

                        test byte [PROCESSING_DISK_CHANGE], 1
                        jz .no_change
                        call _core_get_c1541_cycles
                        cmp dword edx, [phase_advance_moment+4]
                        jne near .continue
                        cmp dword eax, [phase_advance_moment]
                        jne near .continue
                        add dword [phase_advance_moment], DISC_CHANGE_DELAY
                        adc dword [phase_advance_moment+4], 0
                        call dword [disk_change_phase]
                        jmp .continue

.no_change              cmp byte [_Key_Code],0
                        je .continue
                        call .handle_keypress

.continue               call limit_speed
                        add dword [CORE_C1541_CYCLES], 1
                        adc dword [CORE_C1541_CYCLES + 4], 0
                        jmp .run_c1541_cycle



;****************************************************************************

.handle_keypress

            mov byte al, MAX_DISKS
            inc byte al
            cmp byte [_Key_Code], al
            ja NEAR .no_diskchg_key
            cmp byte [_Key_Code], 2
            jb NEAR .no_diskchg_key

            call status_vc1541_disk

            mov byte al, [_Key_Code]
            sub byte al, 2
            mov byte [_disk_number], al
            call _core_get_c1541_cycles
            add dword eax, DISC_CHANGE_DELAY
            adc dword edx, 0
            mov dword [phase_advance_moment], eax
            mov dword [phase_advance_moment+4], edx
            mov dword [disk_change_phase], .phase1
            mov byte [PROCESSING_DISK_CHANGE], 1
            jmp .return

.no_diskchg_key:
            cmp byte [_Key_Code], 1   ; ESC
            je .esc
            cmp byte [_Key_Code], 31h ; 'N'
            je .newimage
            cmp byte [_Key_Code], 18h ; 'O'
            je .openimage
            cmp byte [_Key_Code], 33h ; ','
            jne .notadvance
            dec dword [_IO_tuning]
            call status_vc1541_IO_tuning
            jmp .return
.notadvance cmp byte [_Key_Code], 34h ; '.'
            jne .notdelay
            inc dword [_IO_tuning]
            call status_vc1541_IO_tuning
            jmp .return
.notdelay   cmp byte [_Key_Code], 13h ; 'R'
            je .reset

.return     mov byte [_Key_Code], 0
            ret
.esc
.newimage
.openimage  jmp core_exit

.reset      lea   dword esi, [C1541]
            C1541LINK_CALL_RESET_HARD
            C1541MEMORY_CALL_RESET_HARD
            CPU65XX_CALL_RESET_HARD
            VIA6522_CALL_RESET_HARD  tvia6522__c1541_1
            mov byte al, 0FEh
            mov dword edi, (tc1541__1 + tvia6522__c1541_1)
            call  via6522_signal_pa
            VIA6522_CALL_RESET_HARD  tvia6522__c1541_2
            DISC_CALL_RESET_HARD
            jmp .return

;****************************************************************************

.phase1     mov byte [_Door_Open], 1                ; open the drive door
            mov dword [disk_change_phase], .phase2
            ret
            
.phase2     cmp dword [_CURRENT_GCR_IMAGE], 0       ; pull the disk partially
            je .nodisk1                             ; out (if there is one in
            mov byte [_Write_Protect_Sense], 1      ; the drive)
.nodisk1    mov dword [disk_change_phase], .phase3
            ret
            
.phase3     mov byte [_Write_Protect_Sense], 0      ; pull the disk completely
            mov dword [disk_change_phase], .phase4  ; out of the drive
            ret

.phase4     movzx dword eax, byte [_disk_number]    ; insert a new disk
            mul byte [_disk_image_t_size]           ; partially into to the
            add dword eax, _diskette+8   ; *gcr     ; drive (if there is one
            mov dword eax, [eax]                    ; to be inserted)
            mov dword [_CURRENT_GCR_IMAGE], eax
            cmp dword eax, 0
            je .nodisk2
            mov byte [_Write_Protect_Sense], 1
.nodisk2    mov dword [disk_change_phase], .phase5
            ret
            
.phase5     cmp dword [_CURRENT_GCR_IMAGE], 0       ; insert the new disk
            je .nodisk3                             ; completely into the
            movzx dword eax, byte [_disk_number]    ; drive (if there is one
            mul byte [_disk_image_t_size]           ; to be inserted)
            add dword eax, _diskette
            mov byte al, [eax+4]       ; read_only
            mov byte [_Write_Protect_Sense], al
.nodisk3    mov dword [disk_change_phase], .phase6
            ret

.phase6     cmp dword [_CURRENT_GCR_IMAGE], 0       ; close the drive door
            je .nodisk4                             ; (if there is a disk
            mov byte [_Door_Open], 0                ; in the drive)
.nodisk4    mov dword [disk_change_phase], .phase1
            mov byte [PROCESSING_DISK_CHANGE], 0
            call status_vc1541_disk
            ret


;****************************************************************************
;* core_exception ***********************************************************
;****************************************************************************
;* eax=>  modulid
;* ebx=>  type
;* ecx=>  number
;****************************************************************************
section .text
core_exception:
                        mov     dword [INFORMATION_MODULID], eax
                        mov     dword [INFORMATION_TYPE], ebx
                        mov     dword [INFORMATION_NUMBER], ecx
;                       mov     dword eax, -CORE_ERROR_EXCEPTION

                        PRINT_TEXT .cb_exception, eax

                        mov byte [_Key_Code], 1
                        call    core_exit
                        mov     dword esp, [STACK_POINTER_EXTERN]
                        jmp     _core_initialize

.cb_exception: db 10, 13, "An exception occurred in module ", STRING_INSERT_HEX32, 10, 13, '$'

;****************************************************************************
;* core_assertion ***********************************************************
;****************************************************************************
;* eax=>  modulid
;* ebx=>  type
;* ecx=>  number
;****************************************************************************
section .text
core_assertion:
                        mov     dword [INFORMATION_MODULID], eax
                        mov     dword [INFORMATION_TYPE], ebx
                        mov     dword [INFORMATION_NUMBER], ecx
                        mov     dword eax, -CORE_ERROR_ASSERTION
                        call    core_leave
                        mov     dword esp, [STACK_POINTER_EXTERN]
                        jmp     _core_initialize



;****************************************************************************
;* core_trap ****************************************************************
;****************************************************************************
;* eax=>  modulid
;* ebx=>  type
;* ecx=>  number
;****************************************************************************
section .text
core_trap:
                        mov     dword [INFORMATION_MODULID], eax
                        mov     dword [INFORMATION_TYPE], ebx
                        mov     dword [INFORMATION_NUMBER], ecx
                        mov     dword eax, -CORE_ERROR_TRAP
                        jmp     core_leave



;****************************************************************************
;* core_log *****************************************************************
;****************************************************************************
;* eax=>  modul id
;* ebx=>  breakpoint type
;* ecx=>  breakpoint number
;* edx=>  data !! QUICK HACK FOR CPU !!
;****************************************************************************
section .text
core_log:
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;TODO: this is a QUICK HACK for saving cpu log
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                        push    dword eax
                        push    dword ebx
                        push    dword ecx
                        push    dword edx
                        push    dword esi
                        push    dword edi
                        mov     dword esi, [LOG_BUFFER]
                        mov     dword edi, [LOG_OFFSET]
                        test    dword esi, esi
                        jz      .no_log
                        mov     dword edi, [edi]
                        call    .write
                        mov     dword eax, ebx
                        call    .write
                        mov     dword eax, ecx
                        call    .write
                        mov     dword eax, edx
                        call    .write
;                       call    core_get_c64_cycles
                        call    .write
                        mov     dword eax, edx
                        call    .write
                        call    _core_get_c1541_cycles
                        call    .write
                        mov     dword eax, edx
                        mov     dword ebx, [LOG_OFFSET]
                        call    .write
                        mov     dword [ebx], edi
.no_log:                pop     dword edi
                        pop     dword esi
                        pop     dword edx
                        pop     dword ecx
                        pop     dword ebx
                        pop     dword eax
                        ret

.write:                 mov     dword [esi + edi], eax
                        add     dword edi, 4
                        cmp     dword edi, [LOG_SIZE]
                        jb      .write_end
                        mov     dword eax, -CORE_ERROR_BPLOG_FULL
                        push    dword ebx
                        mov     dword ebx, [LOG_OFFSET]
                        mov     dword [ebx], edi
                        pop     dword ebx
                        call    core_leave
                        xor     dword edi, edi
.write_end:             ret




;****************************************************************************
;* core_get_c1541_cycles ****************************************************
;****************************************************************************
;* <=eax  low 32 Bits of cycle counter
;* <=edx  high 32 Bits of cycle counter
;****************************************************************************
section .text
_core_get_c1541_cycles:
                        mov     dword eax, [CORE_C1541_CYCLES]
                        mov     dword edx, [CORE_C1541_CYCLES + 4]
                        ret



%ifndef CORE_ENABLE_BP
;****************************************************************************
;* core_bp_reject ***********************************************************
;****************************************************************************
section .text
core_bp_reject:
                        mov     dword eax, -CORE_ERROR_NOT_IMPLEMENTED
                        ret

core_bp_entries:        equ     core_bp_reject
core_bp_get:            equ     core_bp_reject
core_bp_add:            equ     core_bp_reject
core_bp_delete:         equ     core_bp_reject



%else
;****************************************************************************
;* MODULID_SELECTOR *********************************************************
;****************************************************************************
%macro MODULID_SELECTOR 1
                        lea     dword esi, [C64]
                        mov     dword edx, cpu65xx_%1
                        cmp     dword eax, CORE_MODULID_C64CPU
                        je      %%call
                        lea     dword esi, [C1541]
                        mov     dword edx, cpu65xx_%1
                        cmp     dword eax, CORE_MODULID_C1541CPU
                        je      %%call
                        mov     dword eax, -CORE_ERROR_NOT_IMPLEMENTED
                        ret
%%call:                 mov     dword eax, ebx
                        mov     dword ebx, ecx
                        jmp     dword edx
%endmacro



;****************************************************************************
;* core_bp_entries **********************************************************
;****************************************************************************
;* eax=>  modul id
;* ebx=>  breakpoint type
;* ecx=>  pointer to dword variable
;****************************************************************************
section .text
core_bp_entries:
                        MODULID_SELECTOR  bp_entries



;****************************************************************************
;* core_bp_get **************************************************************
;****************************************************************************
;* eax=>  modul id
;* ebx=>  breakpoint type
;* ecx=>  pointer to data structure
;****************************************************************************
section .text
core_bp_get:
                        MODULID_SELECTOR  bp_get



;****************************************************************************
;* core_bp_add **************************************************************
;****************************************************************************
;* eax=>  modul id
;* ebx=>  breakpoint type
;* ecx=>  pointer to data structure
;****************************************************************************
section .text
core_bp_add:
                        MODULID_SELECTOR  bp_add



;****************************************************************************
;* core_bp_delete ***********************************************************
;****************************************************************************
;* eax=>  modul id
;* ebx=>  breakpoint type
;* ecx=>  breakpoint number
;****************************************************************************
section .text
core_bp_delete:
                        MODULID_SELECTOR  bp_delete
%endif



;****************************************************************************
;* core_get_information *****************************************************
;****************************************************************************
;* eax=>  pointer to tcoreinfo structure
;* <=eax  return status
;****************************************************************************
section .text
core_get_information:
                        mov     dword ebx, [INFORMATION_MODULID]
                        mov     dword ecx, [INFORMATION_TYPE]
                        mov     dword edx, [INFORMATION_NUMBER]
                        mov     dword [eax + d__coreinfo_modulid], ebx
                        mov     dword [eax + d__coreinfo_type], ecx
                        mov     dword [eax + d__coreinfo_number], edx
                        xor     dword eax, eax
                        ret



;****************************************************************************
;* core_set_log_options *****************************************************
;****************************************************************************
;* eax=>  pointer to log buffer
;* ebx=>  point to log offset variable
;* ecx=>  log size
;* <=eax  return status
;****************************************************************************
section .text
core_set_log_options:
                        mov     dword [LOG_BUFFER], eax
                        mov     dword [LOG_OFFSET], ebx
                        and     byte  cl, 0fch
                        mov     dword [LOG_SIZE], ecx
                        xor     dword eax, eax
                        ret

