;
;   1541EMU, the Commodore 1541 disk drive emulator
;   Copyright (C) 2001-2002 Ville Muikkula
;   Copyright (C) Karsten Scheibler
;
;  This file was formerly part of ec64, the Commodore 64 emulator for Linux.
;  Written by
;   Karsten Scheibler <karsten.scheibler@bigfoot.de>
;  Modifications by
;   Ville Muikkula <1541@surfeu.fi>
;
;   This program is free software; you can redistribute it and/or modify
;   it under the terms of the GNU General Public License version 2 as
;   published by the Free Software Foundation.
;
;   This program is distributed in the hope that it will be useful,
;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;   GNU General Public License for more details.
;
;   You should have received a copy of the GNU General Public License
;   along with this program; if not, write to the Free Software
;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;****************************************************************************
;****************************************************************************
;*
;* cpu65xx.n
;*
;****************************************************************************
;****************************************************************************


%define CPU65XX_N
%include "cpu65xx.i"
%include "core.i"


;****************************************************************************
;****************************************************************************
;*
;* PART 1: assign's & defines's
;*
;****************************************************************************
;****************************************************************************





%assign MAX_BP_READ_OPCODE      CPU65XX_MAX_BP_READ_OPCODE
%assign MAX_BP_READ_STACK       CPU65XX_MAX_BP_READ_STACK
%assign MAX_BP_READ_DATA        CPU65XX_MAX_BP_READ_DATA
%assign MAX_BP_WRITE_STACK      CPU65XX_MAX_BP_WRITE_STACK
%assign MAX_BP_WRITE_DATA       CPU65XX_MAX_BP_WRITE_DATA



%define MODULID                 CPU65XX_MODULID
%define GET_CYCLES              CPU65XX_GET_CYCLES
%define BASE                    CPU65XX_BASE
%define MEMORY_FUNCTIONS        CPU65XX_MEMORY_FUNCTIONS
%ifdef CPU65XX_ENABLE_BP
%define MEMORY_FUNCTIONS_NO_BP  CPU65XX_MEMORY_FUNCTIONS_NO_BP
%define BP_OFFSETS              CPU65XX_BP_OFFSETS
%define BP_READ_OPCODE          CPU65XX_BP_READ_OPCODE
%define BP_READ_STACK           CPU65XX_BP_READ_STACK
%define BP_READ_DATA            CPU65XX_BP_READ_DATA
%define BP_WRITE_STACK          CPU65XX_BP_WRITE_STACK
%define BP_WRITE_DATA           CPU65XX_BP_WRITE_DATA
%endif



%define STATE                   CPU65XX_STATE
%define EXECUTE_CYCLE           CPU65XX_EXECUTE_CYCLE
%define OPTIONAL_ADDRESS        CPU65XX_OPTIONAL_ADDRESS
%define IRQ_TIMESTAMP           CPU65XX_IRQ_TIMESTAMP
%define NMI_TIMESTAMP           CPU65XX_NMI_TIMESTAMP
%define DMA_TIMESTAMP           CPU65XX_DMA_TIMESTAMP
%define PROGRAM_COUNTER         CPU65XX_PROGRAM_COUNTER
%define STACK_POINTER           CPU65XX_STACK_POINTER
%define ZEROPAGE_ADDRESS        CPU65XX_ZEROPAGE_ADDRESS
%define EFFECTIVE_ADDRESS       CPU65XX_EFFECTIVE_ADDRESS
%define OPCODE                  CPU65XX_OPCODE
%define DATA_BYTE               CPU65XX_DATA_BYTE
%define ALU_BYTE                CPU65XX_ALU_BYTE
%define ACCUMULATOR             CPU65XX_ACCUMULATOR
%define INDEX_X                 CPU65XX_INDEX_X
%define INDEX_Y                 CPU65XX_INDEX_Y
%define FLAG_REGISTER           CPU65XX_FLAG_REGISTER
%define READ_OPCODE             CPU65XX_READ_OPCODE
%define READ_OPCODE_NO_DMA      CPU65XX_READ_OPCODE_NO_DMA
%define READ_STACK              CPU65XX_READ_STACK
%define READ_STACK_NO_DMA       CPU65XX_READ_STACK_NO_DMA
%define READ_DATA               CPU65XX_READ_DATA
%define READ_DATA_NO_DMA        CPU65XX_READ_DATA_NO_DMA
%define WRITE_STACK             CPU65XX_WRITE_STACK
%define WRITE_DATA              CPU65XX_WRITE_DATA
%ifdef CPU65XX_ENABLE_BP
%define READ_OPCODE_NO_BP       CPU65XX_READ_OPCODE_NO_BP
%define READ_STACK_NO_BP        CPU65XX_READ_STACK_NO_BP
%define READ_DATA_NO_BP         CPU65XX_READ_DATA_NO_BP
%define WRITE_STACK_NO_BP       CPU65XX_WRITE_STACK_NO_BP
%define WRITE_DATA_NO_BP        CPU65XX_WRITE_DATA_NO_BP
%define BP_READ_OPCODE_OFFSET   CPU65XX_BP_READ_OPCODE_OFFSET
%define BP_READ_STACK_OFFSET    CPU65XX_BP_READ_STACK_OFFSET
%define BP_READ_DATA_OFFSET     CPU65XX_BP_READ_DATA_OFFSET
%define BP_WRITE_STACK_OFFSET   CPU65XX_BP_WRITE_STACK_OFFSET
%define BP_WRITE_DATA_OFFSET    CPU65XX_BP_WRITE_DATA_OFFSET
%endif



%assign STATE_ILLEGAL_OPCODE            CPU65XX_STATE_ILLEGAL_OPCODE
%assign STATE_ILLEGAL_IRQ               CPU65XX_STATE_ILLEGAL_IRQ
%assign STATE_SIGNAL_IRQ1               CPU65XX_STATE_SIGNAL_IRQ1
%assign STATE_SIGNAL_IRQ2               CPU65XX_STATE_SIGNAL_IRQ2
%assign STATE_SIGNAL_IRQ3               CPU65XX_STATE_SIGNAL_IRQ3
%assign STATE_SIGNAL_IRQ4               CPU65XX_STATE_SIGNAL_IRQ4
%assign STATE_SIGNAL_IRQ                CPU65XX_STATE_SIGNAL_IRQ
%assign STATE_SIGNAL_NMI                CPU65XX_STATE_SIGNAL_NMI
%assign STATE_SIGNAL_IRQ_OR_NMI         CPU65XX_STATE_SIGNAL_IRQ_OR_NMI
%assign STATE_SIGNAL_DMA                CPU65XX_STATE_SIGNAL_DMA
%assign STATE_PROCESSING_IRQ            CPU65XX_STATE_PROCESSING_IRQ
%assign STATE_PROCESSING_NMI            CPU65XX_STATE_PROCESSING_NMI
%assign STATE_PROCESSING_RESET          CPU65XX_STATE_PROCESSING_RESET



%assign FLAG_N                  CPU65XX_FLAG_N
%assign FLAG_N_BIT              CPU65XX_FLAG_N_BIT
%assign FLAG_V                  CPU65XX_FLAG_V
%assign FLAG_V_BIT              CPU65XX_FLAG_V_BIT
%assign FLAG_R                  CPU65XX_FLAG_R
%assign FLAG_R_BIT              CPU65XX_FLAG_R_BIT
%assign FLAG_B                  CPU65XX_FLAG_B
%assign FLAG_B_BIT              CPU65XX_FLAG_B_BIT
%assign FLAG_D                  CPU65XX_FLAG_D
%assign FLAG_D_BIT              CPU65XX_FLAG_D_BIT
%assign FLAG_I                  CPU65XX_FLAG_I
%assign FLAG_I_BIT              CPU65XX_FLAG_I_BIT
%assign FLAG_Z                  CPU65XX_FLAG_Z
%assign FLAG_Z_BIT              CPU65XX_FLAG_Z_BIT
%assign FLAG_C                  CPU65XX_FLAG_C
%assign FLAG_C_BIT              CPU65XX_FLAG_C_BIT





;****************************************************************************
;****************************************************************************
;*
;* PART 2: Opcode decoding tables
;*
;****************************************************************************
;****************************************************************************





section .text
                        align   4
ap__opcodes:
%ifdef CPU65XX_DISABLE_ILLEGAL_OPCODES
                        ;---------
                        ;$00 - $0f
                        ;---------

                        dd      opcode_brk,             opcode_illegal
                        dd      address_x_indirect,     opcode_ora.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_ora.read
                        dd      address_zeropage,       opcode_asl.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_php,             opcode_illegal
                        dd      opcode_ora.immediate,   opcode_illegal
                        dd      opcode_asl.accumulator, opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_ora.read
                        dd      address_absolute,       opcode_asl.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$10 - $1f
                        ;---------

                        dd      opcode_bpl,             opcode_illegal
                        dd      address_indirect_y,     opcode_ora.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_ora.read
                        dd      address_zeropage_x,     opcode_asl.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_clc,             opcode_illegal
                        dd      address_absolute_y,     opcode_ora.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_ora.read
                        dd      address_absolute_x_write,       opcode_asl.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$20 - $2f
                        ;---------

                        dd      opcode_jsr,             opcode_illegal
                        dd      address_x_indirect,     opcode_and.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_bit.read
                        dd      address_zeropage,       opcode_and.read
                        dd      address_zeropage,       opcode_rol.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_plp,             opcode_illegal
                        dd      opcode_and.immediate,   opcode_illegal
                        dd      opcode_rol.accumulator, opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_bit.read
                        dd      address_absolute,       opcode_and.read
                        dd      address_absolute,       opcode_rol.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$30 - $3f
                        ;---------

                        dd      opcode_bmi,             opcode_illegal
                        dd      address_indirect_y,     opcode_and.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_and.read
                        dd      address_zeropage_x,     opcode_rol.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_sec,             opcode_illegal
                        dd      address_absolute_y,     opcode_and.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_and.read
                        dd      address_absolute_x_write,       opcode_rol.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$40 - $4f
                        ;---------

                        dd      opcode_rti,             opcode_illegal
                        dd      address_x_indirect,     opcode_eor.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_eor.read
                        dd      address_zeropage,       opcode_lsr.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_pha,             opcode_illegal
                        dd      opcode_eor.immediate,   opcode_illegal
                        dd      opcode_lsr.accumulator, opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_jmp,             opcode_illegal
                        dd      address_absolute,       opcode_eor.read
                        dd      address_absolute,       opcode_lsr.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$50 - $5f
                        ;---------

                        dd      opcode_bvc,             opcode_illegal
                        dd      address_indirect_y,     opcode_eor.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_eor.read
                        dd      address_zeropage_x,     opcode_lsr.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_cli,             opcode_illegal
                        dd      address_absolute_y,     opcode_eor.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_eor.read
                        dd      address_absolute_x_write,       opcode_lsr.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$60 - $6f
                        ;---------

                        dd      opcode_rts,             opcode_illegal
                        dd      address_x_indirect,     opcode_adc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_adc.read
                        dd      address_zeropage,       opcode_ror.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_pla,             opcode_illegal
                        dd      opcode_adc.immediate,   opcode_illegal
                        dd      opcode_ror.accumulator, opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_jmp_indirect,    opcode_illegal
                        dd      address_absolute,       opcode_adc.read
                        dd      address_absolute,       opcode_ror.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$70 - $7f
                        ;---------

                        dd      opcode_bvs,             opcode_illegal
                        dd      address_indirect_y,     opcode_adc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_adc.read
                        dd      address_zeropage_x,     opcode_ror.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_sei,             opcode_illegal
                        dd      address_absolute_y,     opcode_adc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_adc.read
                        dd      address_absolute_x_write,       opcode_ror.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$80 - $8f
                        ;---------

                        dd      opcode_illegal,         opcode_illegal
                        dd      address_x_indirect,     opcode_sta
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_sty
                        dd      address_zeropage,       opcode_sta
                        dd      address_zeropage,       opcode_stx
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_dey,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_txa,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_sty
                        dd      address_absolute,       opcode_sta
                        dd      address_absolute,       opcode_stx
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$90 - $9f
                        ;---------

                        dd      opcode_bcc,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_sta
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_sty
                        dd      address_zeropage_x,     opcode_sta
                        dd      address_zeropage_y,     opcode_stx
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_tya,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_sta
                        dd      opcode_txs,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x_write,       opcode_sta
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$a0 - $af
                        ;---------

                        dd      opcode_ldy.immediate,   opcode_illegal
                        dd      address_x_indirect,     opcode_lda.read
                        dd      opcode_ldx.immediate,   opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_ldy.read
                        dd      address_zeropage,       opcode_lda.read
                        dd      address_zeropage,       opcode_ldx.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_tay,             opcode_illegal
                        dd      opcode_lda.immediate,   opcode_illegal
                        dd      opcode_tax,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_ldy.read
                        dd      address_absolute,       opcode_lda.read
                        dd      address_absolute,       opcode_ldx.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$b0 - $bf
                        ;---------

                        dd      opcode_bcs,             opcode_illegal
                        dd      address_indirect_y,     opcode_lda.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_ldy.read
                        dd      address_zeropage_x,     opcode_lda.read
                        dd      address_zeropage_y,     opcode_ldx.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_clv,             opcode_illegal
                        dd      address_absolute_y,     opcode_lda.read
                        dd      opcode_tsx,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_ldy.read
                        dd      address_absolute_x,     opcode_lda.read
                        dd      address_absolute_y,     opcode_ldx.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$c0 - $cf
                        ;---------

                        dd      opcode_cpy.immediate,   opcode_illegal
                        dd      address_x_indirect,     opcode_cmp.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_cpy.read
                        dd      address_zeropage,       opcode_cmp.read
                        dd      address_zeropage,       opcode_dec.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_iny,             opcode_illegal
                        dd      opcode_cmp.immediate,   opcode_illegal
                        dd      opcode_dex,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_cpy.read
                        dd      address_absolute,       opcode_cmp.read
                        dd      address_absolute,       opcode_dec.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$d0 - $df
                        ;---------

                        dd      opcode_bne,             opcode_illegal
                        dd      address_indirect_y,     opcode_cmp.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_cmp.read
                        dd      address_zeropage_x,     opcode_dec.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_cld,             opcode_illegal
                        dd      address_absolute_y,     opcode_cmp.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_cmp.read
                        dd      address_absolute_x_write,       opcode_dec.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$e0 - $ef
                        ;---------

                        dd      opcode_cpx.immediate,   opcode_illegal
                        dd      address_x_indirect,     opcode_sbc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_cpx.read
                        dd      address_zeropage,       opcode_sbc.read
                        dd      address_zeropage,       opcode_inc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_inx,             opcode_illegal
                        dd      opcode_sbc.immediate,   opcode_illegal
                        dd      opcode_nop,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_cpx.read
                        dd      address_absolute,       opcode_sbc.read
                        dd      address_absolute,       opcode_inc.read
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$f0 - $ff
                        ;---------

                        dd      opcode_beq,             opcode_illegal
                        dd      address_indirect_y,     opcode_sbc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_sbc.read
                        dd      address_zeropage_x,     opcode_inc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_sed,             opcode_illegal
                        dd      address_absolute_y,     opcode_sbc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_sbc.read
                        dd      address_absolute_x_write,       opcode_inc.read
                        dd      opcode_illegal,         opcode_illegal



%else
                        ;---------
                        ;$00 - $0f
                        ;---------

                        dd      opcode_brk,             opcode_illegal
                        dd      address_x_indirect,     opcode_ora.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_x_indirect,     opcode_aso.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_ora.read
                        dd      address_zeropage,       opcode_asl.read
                        dd      address_zeropage,       opcode_aso.read
                        dd      opcode_php,             opcode_illegal
                        dd      opcode_ora.immediate,   opcode_illegal
                        dd      opcode_asl.accumulator, opcode_illegal
                        dd      opcode_anc.immediate,   opcode_illegal
                        dd      address_absolute,       opcode_top
                        dd      address_absolute,       opcode_ora.read
                        dd      address_absolute,       opcode_asl.read
                        dd      address_absolute,       opcode_aso.read

                        ;---------
                        ;$10 - $1f
                        ;---------

                        dd      opcode_bpl,             opcode_illegal
                        dd      address_indirect_y,     opcode_ora.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_aso.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_ora.read
                        dd      address_zeropage_x,     opcode_asl.read
                        dd      address_zeropage_x,     opcode_aso.read
                        dd      opcode_clc,             opcode_illegal
                        dd      address_absolute_y,     opcode_ora.read
                        dd      opcode_nop,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_aso.read
                        dd      address_absolute_x,     opcode_top
                        dd      address_absolute_x,     opcode_ora.read
                        dd      address_absolute_x_write,       opcode_asl.read
                        dd      address_absolute_x_write,       opcode_aso.read

                        ;---------
                        ;$20 - $2f
                        ;---------

                        dd      opcode_jsr,             opcode_illegal
                        dd      address_x_indirect,     opcode_and.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_x_indirect,     opcode_rla.read
                        dd      address_zeropage,       opcode_bit.read
                        dd      address_zeropage,       opcode_and.read
                        dd      address_zeropage,       opcode_rol.read
                        dd      address_zeropage,       opcode_rla.read
                        dd      opcode_plp,             opcode_illegal
                        dd      opcode_and.immediate,   opcode_illegal
                        dd      opcode_rol.accumulator, opcode_illegal
                        dd      opcode_anc.immediate,   opcode_illegal
                        dd      address_absolute,       opcode_bit.read
                        dd      address_absolute,       opcode_and.read
                        dd      address_absolute,       opcode_rol.read
                        dd      address_absolute,       opcode_rla.read

                        ;---------
                        ;$30 - $3f
                        ;---------

                        dd      opcode_bmi,             opcode_illegal
                        dd      address_indirect_y,     opcode_and.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_rla.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_and.read
                        dd      address_zeropage_x,     opcode_rol.read
                        dd      address_zeropage_x,     opcode_rla.read
                        dd      opcode_sec,             opcode_illegal
                        dd      address_absolute_y,     opcode_and.read
                        dd      opcode_nop,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_rla.read
                        dd      address_absolute_x,     opcode_top
                        dd      address_absolute_x,     opcode_and.read
                        dd      address_absolute_x_write,       opcode_rol.read
                        dd      address_absolute_x_write,       opcode_rla.read

                        ;---------
                        ;$40 - $4f
                        ;---------

                        dd      opcode_rti,             opcode_illegal
                        dd      address_x_indirect,     opcode_eor.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_x_indirect,     opcode_lse.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_eor.read
                        dd      address_zeropage,       opcode_lsr.read
                        dd      address_zeropage,       opcode_lse.read
                        dd      opcode_pha,             opcode_illegal
                        dd      opcode_eor.immediate,   opcode_illegal
                        dd      opcode_lsr.accumulator, opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_jmp,             opcode_illegal
                        dd      address_absolute,       opcode_eor.read
                        dd      address_absolute,       opcode_lsr.read
                        dd      address_absolute,       opcode_lse.read

                        ;---------
                        ;$50 - $5f
                        ;---------

                        dd      opcode_bvc,             opcode_illegal
                        dd      address_indirect_y,     opcode_eor.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_lse.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_eor.read
                        dd      address_zeropage_x,     opcode_lsr.read
                        dd      address_zeropage_x,     opcode_lse.read
                        dd      opcode_cli,             opcode_illegal
                        dd      address_absolute_y,     opcode_eor.read
                        dd      opcode_nop,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_lse.read
                        dd      address_absolute_x,     opcode_top
                        dd      address_absolute_x,     opcode_eor.read
                        dd      address_absolute_x_write,       opcode_lsr.read
                        dd      address_absolute_x_write,       opcode_lse.read

                        ;---------
                        ;$60 - $6f
                        ;---------

                        dd      opcode_rts,             opcode_illegal
                        dd      address_x_indirect,     opcode_adc.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_x_indirect,     opcode_rra.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage,       opcode_adc.read
                        dd      address_zeropage,       opcode_ror.read
                        dd      address_zeropage,       opcode_rra.read
                        dd      opcode_pla,             opcode_illegal
                        dd      opcode_adc.immediate,   opcode_illegal
                        dd      opcode_ror.accumulator, opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_jmp_indirect,    opcode_illegal
                        dd      address_absolute,       opcode_adc.read
                        dd      address_absolute,       opcode_ror.read
                        dd      address_absolute,       opcode_rra.read

                        ;---------
                        ;$70 - $7f
                        ;---------

                        dd      opcode_bvs,             opcode_illegal
                        dd      address_indirect_y,     opcode_adc.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_rra.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_adc.read
                        dd      address_zeropage_x,     opcode_ror.read
                        dd      address_zeropage_x,     opcode_rra.read
                        dd      opcode_sei,             opcode_illegal
                        dd      address_absolute_y,     opcode_adc.read
                        dd      opcode_nop,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_rra.read
                        dd      address_absolute_x,     opcode_top
                        dd      address_absolute_x,     opcode_adc.read
                        dd      address_absolute_x_write,       opcode_ror.read
                        dd      address_absolute_x_write,       opcode_rra.read

                        ;---------
                        ;$80 - $8f
                        ;---------

                        dd      opcode_illegal,         opcode_illegal
                        dd      address_x_indirect,     opcode_sta
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_x_indirect,     opcode_axs
                        dd      address_zeropage,       opcode_sty
                        dd      address_zeropage,       opcode_sta
                        dd      address_zeropage,       opcode_stx
                        dd      address_zeropage,       opcode_axs
                        dd      opcode_dey,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_txa,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_sty
                        dd      address_absolute,       opcode_sta
                        dd      address_absolute,       opcode_stx
                        dd      address_absolute,       opcode_axs

                        ;---------
                        ;$90 - $9f
                        ;---------

                        dd      opcode_bcc,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_sta
                        dd      opcode_hlt,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_sty
                        dd      address_zeropage_x,     opcode_sta
                        dd      address_zeropage_y,     opcode_stx
                        dd      address_zeropage_y,     opcode_axs
                        dd      opcode_tya,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_sta
                        dd      opcode_txs,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x_write,       opcode_sta
                        dd      opcode_illegal,         opcode_illegal
                        dd      opcode_illegal,         opcode_illegal

                        ;---------
                        ;$a0 - $af
                        ;---------

                        dd      opcode_ldy.immediate,   opcode_illegal
                        dd      address_x_indirect,     opcode_lda.read
                        dd      opcode_ldx.immediate,   opcode_illegal
                        dd      address_x_indirect,     opcode_lax.read
                        dd      address_zeropage,       opcode_ldy.read
                        dd      address_zeropage,       opcode_lda.read
                        dd      address_zeropage,       opcode_ldx.read
                        dd      address_zeropage,       opcode_lax.read
                        dd      opcode_tay,             opcode_illegal
                        dd      opcode_lda.immediate,   opcode_illegal
                        dd      opcode_tax,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_ldy.read
                        dd      address_absolute,       opcode_lda.read
                        dd      address_absolute,       opcode_ldx.read
                        dd      address_absolute,       opcode_lax.read

                        ;---------
                        ;$b0 - $bf
                        ;---------

                        dd      opcode_bcs,             opcode_illegal
                        dd      address_indirect_y,     opcode_lda.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_indirect_y,     opcode_lax.read
                        dd      address_zeropage_x,     opcode_ldy.read
                        dd      address_zeropage_x,     opcode_lda.read
                        dd      address_zeropage_y,     opcode_ldx.read
                        dd      address_zeropage_y,     opcode_lax.read
                        dd      opcode_clv,             opcode_illegal
                        dd      address_absolute_y,     opcode_lda.read
                        dd      opcode_tsx,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute_x,     opcode_ldy.read
                        dd      address_absolute_x,     opcode_lda.read
                        dd      address_absolute_y,     opcode_ldx.read
                        dd      address_absolute_y,     opcode_lax.read

                        ;---------
                        ;$c0 - $cf
                        ;---------

                        dd      opcode_cpy.immediate,   opcode_illegal
                        dd      address_x_indirect,     opcode_cmp.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_x_indirect,     opcode_dcm.read
                        dd      address_zeropage,       opcode_cpy.read
                        dd      address_zeropage,       opcode_cmp.read
                        dd      address_zeropage,       opcode_dec.read
                        dd      address_zeropage,       opcode_dcm.read
                        dd      opcode_iny,             opcode_illegal
                        dd      opcode_cmp.immediate,   opcode_illegal
                        dd      opcode_dex,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_cpy.read
                        dd      address_absolute,       opcode_cmp.read
                        dd      address_absolute,       opcode_dec.read
                        dd      address_absolute,       opcode_dcm.read

                        ;---------
                        ;$d0 - $df
                        ;---------

                        dd      opcode_bne,             opcode_illegal
                        dd      address_indirect_y,     opcode_cmp.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_dcm.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_cmp.read
                        dd      address_zeropage_x,     opcode_dec.read
                        dd      address_zeropage_x,     opcode_dcm.read
                        dd      opcode_cld,             opcode_illegal
                        dd      address_absolute_y,     opcode_cmp.read
                        dd      opcode_nop,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_dcm.read
                        dd      address_absolute_x,     opcode_top
                        dd      address_absolute_x,     opcode_cmp.read
                        dd      address_absolute_x_write,       opcode_dec.read
                        dd      address_absolute_x_write,       opcode_dcm.read

                        ;---------
                        ;$e0 - $ef
                        ;---------

                        dd      opcode_cpx.immediate,   opcode_illegal
                        dd      address_x_indirect,     opcode_sbc.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_x_indirect,     opcode_ins.read
                        dd      address_zeropage,       opcode_cpx.read
                        dd      address_zeropage,       opcode_sbc.read
                        dd      address_zeropage,       opcode_inc.read
                        dd      address_zeropage,       opcode_ins.read
                        dd      opcode_inx,             opcode_illegal
                        dd      opcode_sbc.immediate,   opcode_illegal
                        dd      opcode_nop,             opcode_illegal
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_absolute,       opcode_cpx.read
                        dd      address_absolute,       opcode_sbc.read
                        dd      address_absolute,       opcode_inc.read
                        dd      address_absolute,       opcode_ins.read

                        ;---------
                        ;$f0 - $ff
                        ;---------

                        dd      opcode_beq,             opcode_illegal
                        dd      address_indirect_y,     opcode_sbc.read
                        dd      opcode_hlt,             opcode_illegal
                        dd      address_indirect_y_write,       opcode_ins.read
                        dd      opcode_illegal,         opcode_illegal
                        dd      address_zeropage_x,     opcode_sbc.read
                        dd      address_zeropage_x,     opcode_inc.read
                        dd      address_zeropage_x,     opcode_ins.read
                        dd      opcode_sed,             opcode_illegal
                        dd      address_absolute_y,     opcode_sbc.read
                        dd      opcode_nop,             opcode_illegal
                        dd      address_absolute_y_write,       opcode_ins.read
                        dd      address_absolute_x,     opcode_top
                        dd      address_absolute_x,     opcode_sbc.read
                        dd      address_absolute_x_write,       opcode_inc.read
                        dd      address_absolute_x_write,       opcode_ins.read
%endif



%ifdef CPU65XX_ENABLE_SNAPSHOT
cpu65xx_cycle_table:
%include "cpu65xx.l"
%endif





;****************************************************************************
;****************************************************************************
;*
;* PART 3: Initialisation & Management Routines
;*
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* cpu65xx_initialize *******************************************************
;****************************************************************************
section .text
cpu65xx_initialize:
                        ret



;****************************************************************************
;* cpu65xx_exit *************************************************************
;****************************************************************************
section .text
cpu65xx_exit:
                        ret



;****************************************************************************
;* cpu65xx_instance_initialize **********************************************
;****************************************************************************
section .text
cpu65xx_instance_initialize:
                        INSTANCE_INITIALIZE  tcpu65xx_size



;****************************************************************************
;* cpu65xx_instance_exit ****************************************************
;****************************************************************************
section .text
cpu65xx_instance_exit:
                        ret



;****************************************************************************
;* cpu65xx_reset_hard *******************************************************
;****************************************************************************
section .text
cpu65xx_reset_hard:
                        push    dword eax
                        push    dword ecx
                        push    dword edi
                        xor     dword eax, eax
                        mov     dword ecx, CPUBASE_SIZE
                        lea     dword edi, [BASE]
                        cld
                        rep stosb
                        pop     dword edi
                        pop     dword ecx
                        pop     dword eax
                        jmp     cpu65xx_reset_soft



;****************************************************************************
;* cpu65xx_reset_soft *******************************************************
;****************************************************************************
section .text
cpu65xx_reset_soft:
                        push    dword eax
                        push    dword ebx
                        mov     byte  [FLAG_REGISTER], (FLAG_R | FLAG_B | FLAG_I)
%ifdef CPU65XX_USE_RESET_ADDRESS
                        mov     dword eax, CPU65XX_USE_RESET_ADDRESS
                        xor     dword ebx, ebx
                        mov     dword [PROGRAM_COUNTER], eax
%else
                        mov     dword ebx, 00000fffch
                        call    dword [READ_OPCODE]
                        inc     dword ebx
                        mov     byte  [PROGRAM_COUNTER], al
                        call    dword [READ_OPCODE]
                        xor     dword ebx, ebx
                        mov     byte  [PROGRAM_COUNTER + 1], al
%endif
                        mov     dword [ZEROPAGE_ADDRESS], ebx
                        xor     dword eax, eax
                        mov     dword [EFFECTIVE_ADDRESS], ebx
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;TODO: what about STATE_PROCESSING_RESET ?
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;                       or      dword eax, STATE_PROCESSING_RESET
                        inc     byte  bh
                        mov     dword [STATE], eax
                        mov     dword ecx, get_opcode
                        mov     dword [STACK_POINTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        pop     dword ebx
                        pop     dword eax
                        ret





;****************************************************************************
;****************************************************************************
;*
;* PART ?: Breakpoint Management Routines
;*
;****************************************************************************
;****************************************************************************





%ifndef CPU65XX_ENABLE_BP
;****************************************************************************
;****************************************************************************
;* PART ?.1: Dummy Breakpoint Routines (breakpoints are disabled)
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* cpu65xx_bp_reject ********************************************************
;****************************************************************************
section .text
cpu65xx_bp_reject:
                        xor     dword eax, -CORE_ERROR_NOT_IMPLEMENTED
                        ret

cpu64xx_bp_entries:     equ     cpu65xx_bp_reject
cpu65xx_bp_get:         equ     cpu65xx_bp_reject
cpu65xx_bp_add:         equ     cpu65xx_bp_reject
cpu65xx_bp_delete:      equ     cpu65xx_bp_reject





%else
;****************************************************************************
;****************************************************************************
;* PART ?.2: Real Breakpoint Routines
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* BREAKPOINT_HANDLER *******************************************************
;****************************************************************************
%macro BREAKPOINT_HANDLER 1
%ifidn %1, READ_OPCODE
%define __BP_XXX                BP_READ_OPCODE
%define __CORE_BPTYPE_CPU_XXX   CORE_BPTYPE_CPU_READ_OPCODE
%define __BP_XXX_OFFSET         BP_READ_OPCODE_OFFSET
%define __XXX_NO_BP             READ_OPCODE_NO_BP
%define __READ
%else
%ifidn %1, READ_STACK
%define __BP_XXX                BP_READ_STACK
%define __CORE_BPTYPE_CPU_XXX   CORE_BPTYPE_CPU_READ_STACK
%define __BP_XXX_OFFSET         BP_READ_STACK_OFFSET
%define __XXX_NO_BP             READ_STACK_NO_BP
%define __READ
%else
%ifidn %1, READ_DATA
%define __BP_XXX                BP_READ_DATA
%define __CORE_BPTYPE_CPU_XXX   CORE_BPTYPE_CPU_READ_DATA
%define __BP_XXX_OFFSET         BP_READ_DATA_OFFSET
%define __XXX_NO_BP             READ_DATA_NO_BP
%define __READ
%else
%ifidn %1, WRITE_STACK
%define __BP_XXX                BP_WRITE_STACK
%define __CORE_BPTYPE_CPU_XXX   CORE_BPTYPE_CPU_WRITE_STACK
%define __BP_XXX_OFFSET         BP_WRITE_STACK_OFFSET
%define __XXX_NO_BP             WRITE_STACK_NO_BP
%else
%ifidn %1, WRITE_DATA
%define __BP_XXX                BP_WRITE_DATA
%define __CORE_BPTYPE_CPU_XXX   CORE_BPTYPE_CPU_WRITE_DATA
%define __BP_XXX_OFFSET         BP_WRITE_DATA_OFFSET
%define __XXX_NO_BP             WRITE_DATA_NO_BP
%else
%error "BREAKPOINT_HANDLER: unknown parameter"
%endif
%endif
%endif
%endif
%endif

%ifdef __READ
                        call    [__XXX_NO_BP]
%endif
                        push    dword eax
                        push    dword ecx
                        xor     dword ecx, ecx
                        test    byte  [__BP_XXX_OFFSET], NOT8
                        jnz     .compare
                        mov     dword eax, [MODULID]
                        xor     dword ebx, ebx
                        mov     dword ecx, __LINE__
                        jmp     core_assertion
.check_bp:              cmp     word  bx, [__BP_XXX + CPUBP_SIZE * ecx + CPUBP_LOW_LIMIT]
                        jb      .no_match
                        cmp     word  bx, [__BP_XXX + CPUBP_SIZE * ecx + CPUBP_HIGH_LIMIT]
                        ja      .no_match
                        and     byte  al, [__BP_XXX + CPUBP_SIZE * ecx + CPUBP_BIT_MASK]
                        cmp     byte  al, [__BP_XXX + CPUBP_SIZE * ecx + CPUBP_VALUE]
                        jne     .no_match
                        mov     word  ax, [__BP_XXX + CPUBP_SIZE * ecx + CPUBP_FLAGS]
                        test    dword eax, CORE_BPFLAG_TRAP
                        jz      .skip_trap
                        push    dword eax
                        push    dword ebx
                        mov     dword eax, [MODULID]
                        mov     dword ebx, __CORE_BPTYPE_CPU_XXX
                        call    core_trap
                        pop     dword ebx
                        pop     dword eax
.skip_trap:             test    dword eax, CORE_BPFLAG_LOG
                        jz      .no_match
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;TODO: using edx this way to get the read/write value is a QUICK HACK
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                        push    dword edx
                        mov     dword edx, [esp + 8]
                        xor     byte  dh, dh
                        shl     dword edx, 16
                        mov     word  dx, bx
;                       push    dword eax
                        push    dword ebx
                        mov     dword eax, [MODULID]
                        mov     dword ebx, __CORE_BPTYPE_CPU_XXX
                        call    core_log
                        pop     dword ebx
;                       pop     dword eax
                        pop     dword edx
.no_match:              inc     dword ecx
.compare:               cmp     byte  cl, [__BP_XXX_OFFSET]
                        mov     dword eax, [esp + 4]
                        jb      .check_bp
                        pop     dword ecx
                        pop     dword eax
%ifdef __READ
                        ret
%else
                        jmp     [__XXX_NO_BP]
%endif

%undef __BP_XXX
%undef __CORE_BPTYPE_CPU_XXX
%undef __BP_XXX_OFFSET
%undef __XXX_NO_BP
%undef __READ
%endmacro



;****************************************************************************
;* breakpoint_get_type ******************************************************
;****************************************************************************
;* eax=>  bptype
;* <=eax  return status
;* <=edx  max offset for bptype
;* <=edi  according index to bptype
;* <=ebp  address of the breakpoint data
;****************************************************************************
section .text
breakpoint_get_type:
                        push    dword ebx
                        xor     dword edi, edi
                        dec     dword edi
.loop:                  inc     dword edi
                        lea     dword ebp, [MODULID]
                        mov     dword ebx, [.ad__bptypes + 8 * edi]
                        add     dword ebp, [.ad__bptypes + 8 * edi + 4]
                        test    dword ebx, ebx
                        je      .error
                        cmp     dword eax, ebx
                        jne     .loop
                        xor     dword eax, eax
                        xor     dword edx, edx
                        pop     dword ebx
                        mov     byte  dl, [.ab__max_offsets + edi]
                        ret
.error:                 mov     dword eax, -CORE_ERROR_BAD_BPTYPE
                        pop     dword ebx
                        ret

                        align   4
.ad__bptypes:           dd      CORE_BPTYPE_CPU_READ_OPCODE, atcpubp__cpu65xx_bp_read_opcode
                        dd      CORE_BPTYPE_CPU_READ_STACK, atcpubp__cpu65xx_bp_read_stack
                        dd      CORE_BPTYPE_CPU_READ_DATA, atcpubp__cpu65xx_bp_read_data
                        dd      CORE_BPTYPE_CPU_WRITE_STACK, atcpubp__cpu65xx_bp_write_stack
                        dd      CORE_BPTYPE_CPU_WRITE_DATA, atcpubp__cpu65xx_bp_write_data
                        dd      0, 0

.ab__max_offsets        db      MAX_BP_READ_OPCODE
                        db      MAX_BP_READ_STACK
                        db      MAX_BP_READ_DATA
                        db      MAX_BP_WRITE_STACK
                        db      MAX_BP_WRITE_DATA



;****************************************************************************
;* breakpoint_read_opcode ***************************************************
;****************************************************************************
;* ebx=>  address
;****************************************************************************
section .text
breakpoint_read_opcode:
                        BREAKPOINT_HANDLER  READ_OPCODE



;****************************************************************************
;* breakpoint_read_stack ****************************************************
;****************************************************************************
;* ebx=>  address
;****************************************************************************
section .text
breakpoint_read_stack:
                        BREAKPOINT_HANDLER  READ_STACK



;****************************************************************************
;* breakpoint_read_data *****************************************************
;****************************************************************************
;* ebx=>  address
;****************************************************************************
section .text
breakpoint_read_data:
                        BREAKPOINT_HANDLER  READ_DATA



;****************************************************************************
;* breakpoint_write_stack ***************************************************
;****************************************************************************
;* al==>  byte to write
;* ebx=>  address
;****************************************************************************
section .text
breakpoint_write_stack:
                        BREAKPOINT_HANDLER  WRITE_STACK



;****************************************************************************
;* breakpoint_write_data ****************************************************
;****************************************************************************
;* al==>  byte to write
;* ebx=>  address
;****************************************************************************
section .text
breakpoint_write_data:
                        BREAKPOINT_HANDLER  WRITE_DATA



;****************************************************************************
;* cpu65xx_bp_entries *******************************************************
;****************************************************************************
;* eax=>  breakpoint type
;* ebx=>  pointer to entries variable
;* <=eax  return status
;****************************************************************************
section .text
cpu65xx_bp_entries:
                        push    dword ecx
                        push    dword edx
                        push    dword edi
                        push    dword ebp
                        call    breakpoint_get_type
                        test    dword eax, eax
                        js      .end
                        xor     dword ecx, ecx
                        mov     byte  cl, [BP_OFFSETS + edi]
                        mov     dword [ebx], ecx
.end:                   pop     dword ebp
                        pop     dword edi
                        pop     dword edx
                        pop     dword ecx
                        ret



;****************************************************************************
;* cpu65xx_bp_get ***********************************************************
;****************************************************************************
;* eax=>  breakpoint type
;* ebx=>  pointer to tbpcpu data structure
;* <=eax  return status
;****************************************************************************
section .text
cpu65xx_bp_get:
                        push    dword ecx
                        push    dword edx
                        push    dword edi
                        push    dword ebp
                        call    breakpoint_get_type
                        test    dword eax, eax
                        js      .end
                        mov     dword ecx, [ebx + d__bpcpu_number]
                        mov     dword eax, -CORE_ERROR_BAD_VALUE
                        cmp     dword ecx, 000000100h
                        jae     .end
                        lea     dword ebp, [ebp + CPUBP_SIZE * ecx]
                        cmp     byte  cl, [BP_OFFSETS + edi]
                        jae     .end
                        xor     dword ecx, ecx
                        xor     dword edx, edx
                        mov     word  cx, [ebp + CPUBP_FLAGS]
                        mov     word  dx, [ebp + CPUBP_LOW_LIMIT]
                        mov     dword [ebx + d__bpcpu_flags], ecx
                        mov     dword [ebx + d__bpcpu_low_limit], edx
                        mov     word  cx, [ebp + CPUBP_HIGH_LIMIT]
                        mov     byte  al, [ebp + CPUBP_BIT_MASK]
                        mov     byte  dl, [ebp + CPUBP_VALUE]
                        mov     dword [ebx + d__bpcpu_high_limit], ecx
                        mov     byte  [ebx + b__bpcpu_bit_mask], al
                        mov     byte  [ebx + b__bpcpu_value], dl
                        xor     dword eax, eax
.end:                   pop     dword ebp
                        pop     dword edi
                        pop     dword edx
                        pop     dword ecx
                        ret



;****************************************************************************
;* cpu65xx_bp_add ***********************************************************
;****************************************************************************
;* eax=>  breakpoint type
;* ebx=>  pointer to tbpcpu data structure
;****************************************************************************
section .text
cpu65xx_bp_add:

                        push    dword ecx
                        push    dword edx
                        push    dword edi
                        push    dword ebp
                        call    breakpoint_get_type
                        xor     dword ecx, ecx
                        test    dword eax, eax
                        js      .end
                        mov     byte  cl, [BP_OFFSETS + edi]
                        mov     dword eax, -CORE_ERROR_BPTABLE_FULL
                        cmp     byte  cl, dl
                        jae     .end
                        lea     dword ebp, [ebp + CPUBP_SIZE * ecx]
                        mov     dword edx, [ebx + d__bpcpu_flags]
                        mov     dword eax, -CORE_ERROR_BAD_VALUE
                        cmp     dword edx, 000010000h
                        jae     .end
                        mov     word  [ebp + CPUBP_FLAGS], dx
                        mov     dword edx, [ebx + d__bpcpu_low_limit]
                        cmp     dword edx, 000010000h
                        jae     .end
                        mov     word  [ebp + CPUBP_LOW_LIMIT], dx
                        mov     dword edx, [ebx + d__bpcpu_high_limit]
                        cmp     dword edx, 000010000h
                        jae     .end
                        mov     word  [ebp + CPUBP_HIGH_LIMIT], dx
                        mov     byte  al, [ebx + b__bpcpu_bit_mask]
                        mov     byte  dl, [ebx + b__bpcpu_value]
                        mov     byte  [ebp + CPUBP_BIT_MASK], al
                        mov     byte  [ebp + CPUBP_VALUE], dl
                        mov     dword [ebx + d__bpcpu_number], ecx
                        test    dword ecx, ecx
                        jnz     .no_address_change
                        mov     dword edx, [.ap__breakpoint_functions + 4 * edi]
                        test    dword [STATE], STATE_SIGNAL_DMA
                        jnz     .dma
                        mov     dword [MEMORY_FUNCTIONS + CPUMEM_SIZE * edi + CPUMEM_DMA], edx
.dma:                   mov     dword [MEMORY_FUNCTIONS + CPUMEM_SIZE * edi + CPUMEM_NO_DMA], edx
.no_address_change:     inc     dword ecx
                        xor     dword eax, eax
                        mov     byte  [BP_OFFSETS + edi], cl
.end:                   pop     dword ebp
                        pop     dword edi
                        pop     dword edx
                        pop     dword ecx
                        ret

                                align   4
.ap__breakpoint_functions:      dd      breakpoint_read_opcode
                                dd      breakpoint_read_stack
                                dd      breakpoint_read_data
                                dd      breakpoint_write_stack
                                dd      breakpoint_write_data



;****************************************************************************
;* cpu65xx_bp_delete ********************************************************
;****************************************************************************
;* eax=>  breakpoint type
;* ebx=>  breakpoint number
;****************************************************************************
section .text
cpu65xx_bp_delete:
                        push    dword ecx
                        push    dword edx
                        push    dword edi
                        push    dword ebp
                        call    breakpoint_get_type
                        test    dword eax, eax
                        js      .end
                        xor     dword ecx, ecx
                        mov     dword eax, -CORE_ERROR_BAD_VALUE
                        mov     byte  ch, [BP_OFFSETS + edi]
                        cmp     dword ebx, 000000100h
                        jae     .end
                        cmp     byte  bl, ch
                        jae     .end
                        sub     byte  ch, bl
                        cmp     byte  ch, 1
                        je      .no_move
                        shr     dword ecx, 8
                        shl     dword ecx, 1    ;implies that CPUBP_SIZE = 8
                        push    dword esi
                        push    dword edi
                        lea     dword edi, [ebp + CPUBP_SIZE * ebx]
                        lea     dword esi, [edi + CPUBP_SIZE]
                        cld
                        rep movsd
                        pop     dword edi
                        pop     dword esi
.no_move:               dec     byte  [BP_OFFSETS + edi]
                        jnz     .no_address_change
                        mov     dword eax, [MEMORY_FUNCTIONS_NO_BP + 4 * edi]
                        test    dword [STATE], STATE_SIGNAL_DMA
                        jnz     .dma
                        mov     dword [MEMORY_FUNCTIONS + CPUMEM_SIZE * edi + CPUMEM_DMA], eax
.dma:                   mov     dword [MEMORY_FUNCTIONS + CPUMEM_SIZE * edi + CPUMEM_NO_DMA], eax
.no_address_change:     xor     dword eax, eax
.end:                   pop     dword ebp
                        pop     dword edi
                        pop     dword edx
                        pop     dword ecx
                        ret
%endif





;****************************************************************************
;****************************************************************************
;*
;* PART 4: Handlers for external Signals
;*
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* cpu65xx_signal_set_irq ***************************************************
;****************************************************************************
;* al==>  IRQ Number
;****************************************************************************
section .text
cpu65xx_signal_set_irq:
                        push    dword ebx
                        push    dword ecx
                        push    dword edx
                        mov     byte  cl, al
                        mov     dword ebx, [STATE]
                        mov     dword edx, (STATE_SIGNAL_IRQ1 / 2)
                        test    dword ebx, STATE_SIGNAL_IRQ
                        jnz     .no_new_timestamp
                        push    dword eax
                        push    dword edx
                        call    dword [GET_CYCLES]
                        mov     dword [IRQ_TIMESTAMP], eax
                        mov     dword [IRQ_TIMESTAMP + 4], edx
                        pop     dword edx
                        pop     dword eax
.no_new_timestamp:      shl     dword edx, cl
                        and     dword edx, (STATE_SIGNAL_IRQ4 | STATE_SIGNAL_IRQ3 | STATE_SIGNAL_IRQ2 | STATE_SIGNAL_IRQ1)
                        jnz     .result_ok
                        or      dword ebx, STATE_ILLEGAL_IRQ
                        mov     dword eax, [MODULID]
                        mov     dword [STATE], ebx
                        xor     dword ebx, ebx
                        mov     dword ecx, __LINE__
                        jmp     core_exception
.result_ok:             or      dword edx, (STATE_SIGNAL_IRQ | STATE_SIGNAL_IRQ_OR_NMI)
                        or      dword ebx, edx
                        pop     dword edx
                        mov     dword [STATE], ebx
                        pop     dword ecx
                        pop     dword ebx
                        ret



;****************************************************************************
;* cpu65xx_signal_clear_irq *************************************************
;****************************************************************************
;* al==>  IRQ Number
;****************************************************************************
section .text
cpu65xx_signal_clear_irq:
                        push    dword ebx
                        push    dword ecx
                        push    dword edx
                        mov     byte  cl, al
                        mov     dword edx, (STATE_SIGNAL_IRQ1 / 2)
                        mov     dword ebx, [STATE]
                        shl     dword edx, cl
                        and     dword edx, (STATE_SIGNAL_IRQ4 | STATE_SIGNAL_IRQ3 | STATE_SIGNAL_IRQ2 | STATE_SIGNAL_IRQ1)
                        jnz     .result_ok
                        or      dword ebx, STATE_ILLEGAL_IRQ
                        mov     dword eax, [MODULID]
                        mov     dword [STATE], ebx
                        xor     dword ebx, ebx
                        mov     dword ecx, __LINE__
                        jmp     core_exception
.result_ok:             not     dword edx
                        and     dword ebx, edx
                        test    dword ebx, (STATE_SIGNAL_IRQ4 | STATE_SIGNAL_IRQ3 | STATE_SIGNAL_IRQ2 | STATE_SIGNAL_IRQ1)
                        jnz     .end
                        and     dword ebx, (NOT32 - STATE_SIGNAL_IRQ)
                        test    dword ebx, STATE_SIGNAL_NMI
                        jnz     .end
                        and     dword ebx, (NOT32 - STATE_SIGNAL_IRQ_OR_NMI)
.end:                   pop     dword edx
                        mov     dword [STATE], ebx
                        pop     dword ecx
                        pop     dword ebx
                        ret



;****************************************************************************
;* cpu65xx_signal_set_nmi ***************************************************
;****************************************************************************
section .text
cpu65xx_signal_set_nmi:
                        push    dword eax
                        push    dword edx
                        or      dword [STATE], (STATE_SIGNAL_NMI | STATE_SIGNAL_IRQ_OR_NMI)
                        call    dword [GET_CYCLES]
                        mov     dword [NMI_TIMESTAMP], eax
                        mov     dword [NMI_TIMESTAMP + 4], edx
                        pop     dword edx
                        pop     dword eax
                        ret



;****************************************************************************
;* cpu65xx_signal_set_dma ***************************************************
;****************************************************************************
section .text
cpu65xx_signal_set_dma:
                        test    dword [STATE], STATE_SIGNAL_DMA
                        jnz     .end
                        push    dword eax
                        push    dword ebx
                        push    dword ecx
                        push    dword edx
                        mov     dword eax, [READ_OPCODE]
                        mov     dword ebx, [READ_STACK]
                        mov     dword ecx, [READ_DATA]
                        mov     dword [READ_OPCODE_NO_DMA], eax
                        mov     dword [READ_STACK_NO_DMA], ebx
                        mov     dword [READ_DATA_NO_DMA], ecx
                        mov     dword eax, .ignore_read
                        mov     dword ebx, [STATE]
                        mov     dword [READ_OPCODE], eax
                        or      dword ebx, STATE_SIGNAL_DMA
                        mov     dword [READ_STACK], eax
                        mov     dword [STATE], ebx
                        mov     dword [READ_DATA], eax
                        call    dword [GET_CYCLES]
                        mov     dword [DMA_TIMESTAMP], eax
                        mov     dword [DMA_TIMESTAMP + 4], edx
                        pop     dword edx
                        pop     dword ecx
                        pop     dword ebx
                        pop     dword eax
.end:                   ret

.ignore_read:           pop     dword eax
                        ret



;****************************************************************************
;* cpu65xx_signal_clear_dma *************************************************
;****************************************************************************
section .text
cpu65xx_signal_clear_dma:
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;TODO: influence the irq, nmi timestamp ??
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                        test    dword [STATE], STATE_SIGNAL_DMA
                        jz      .end
                        push    dword eax
                        push    dword ebx
                        push    dword ecx
                        mov     dword eax, [READ_OPCODE_NO_DMA]
                        mov     dword ebx, [READ_STACK_NO_DMA]
                        mov     dword ecx, [READ_DATA_NO_DMA]
                        mov     dword [READ_OPCODE], eax
                        mov     dword [READ_STACK], ebx
                        mov     dword eax, [STATE]
                        mov     dword [READ_DATA], ecx
                        and     dword eax, (NOT32 - STATE_SIGNAL_DMA)
                        pop     dword ecx
                        mov     dword [STATE], eax
                        pop     dword ebx
                        pop     dword eax
.end:                   ret



;****************************************************************************
;* cpu65xx_signal_set_overflow **********************************************
;****************************************************************************
section .text
cpu65xx_signal_set_overflow:
                        or      byte  [FLAG_REGISTER], FLAG_V
                        ret





;****************************************************************************
;****************************************************************************
;*
;* PART 5: Emulation Code
;*
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* CYCLE_LABEL **************************************************************
;****************************************************************************
%macro CYCLE_LABEL 1
%1:
%endmacro





;****************************************************************************
;****************************************************************************
;* PART 5.1: Addressing Modes
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* CALL_READ_OPCODE *********************************************************
;****************************************************************************
%macro CALL_READ_OPCODE 0
                        call    dword [READ_OPCODE]
%ifdef CPU65XX_FUTURE_PLANS
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;TODO: but this makes trouble if the VIC switches DMA on
;      this has only advantages in speculative execution mode
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                        mov     dword ecx, [ADDRESS]
                        cmp     dword ebx, [LOW_LIMIT]
                        jae     .low_limit_ok
                        call    dword [READ_RECALCULATE]
.low_limit_ok:          cmp     dword ebx, [HIGH_LIMIT]
                        jb      .high_limit_ok
                        call    dword [READ_RECALCULATE]
.high_limit_ok:         mov     dword al, [ecx + ebx]
%endif
%endmacro



;****************************************************************************
;* CALL_READ_STACK **********************************************************
;****************************************************************************
%macro CALL_READ_STACK 0
                        call    dword [READ_STACK]
%endmacro



;****************************************************************************
;* CALL_READ_DATA ***********************************************************
;****************************************************************************
%macro CALL_READ_DATA 0
                        call    dword [READ_DATA]
%endmacro



;****************************************************************************
;* CALL_WRITE_STACK *********************************************************
;****************************************************************************
%macro CALL_WRITE_STACK 0
                        call    dword [WRITE_STACK]
%endmacro



;****************************************************************************
;* CALL_WRITE_DATA **********************************************************
;****************************************************************************
%macro CALL_WRITE_DATA 0
                        call    dword [WRITE_DATA]
%endmacro



;****************************************************************************
;* CHECK_FOR_DMA ************************************************************
;****************************************************************************
%macro CHECK_FOR_DMA 0
;                       test    dword [STATE], STATE_SIGNAL_DMA
;                       jz      %%no_dma
;                       ret
;%%no_dma:
%endmacro



;****************************************************************************
;* address ******************************************************************
;****************************************************************************
section .text
address:
CYCLE_LABEL .delay
                        CHECK_FOR_DMA
                        mov     dword eax, [OPTIONAL_ADDRESS]
                        mov     dword [EXECUTE_CYCLE], eax
                        ret

CYCLE_LABEL .high_byte
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        mov     dword ecx, [OPTIONAL_ADDRESS]
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .high_byte_increase
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     byte  al
                        inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        mov     dword ecx, .delay
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .high_byte_delay
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        mov     dword ecx, .delay
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* address_zeropage *********************************************************
;****************************************************************************
section .text
address_zeropage:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        xor     dword ecx, ecx
                        inc     word  bx
                        mov     byte  cl, al
                        mov     dword edx, [OPTIONAL_ADDRESS]
                        mov     dword [EFFECTIVE_ADDRESS], ecx
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], edx
                        ret



;****************************************************************************
;* address_zeropage_x *******************************************************
;****************************************************************************
section .text
address_zeropage_x:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        xor     dword ecx, ecx
                        add     byte  al, [INDEX_X]
                        inc     word  bx
                        mov     byte  cl, al
                        mov     dword edx, address.delay
                        mov     dword [EFFECTIVE_ADDRESS], ecx
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], edx
                        ret



;****************************************************************************
;* address_zeropage_y *******************************************************
;****************************************************************************
section .text
address_zeropage_y:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        xor     dword ecx, ecx
                        add     byte  al, [INDEX_Y]
                        inc     word  bx
                        mov     byte  cl, al
                        mov     dword edx, address.delay
                        mov     dword [EFFECTIVE_ADDRESS], ecx
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], edx
                        ret



;****************************************************************************
;* address_absolute *********************************************************
;****************************************************************************
section .text
address_absolute:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword ecx, address.high_byte
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* address_absolute_x *******************************************************
;****************************************************************************
section .text
address_absolute_x:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        add     byte  al, [INDEX_X]
                        mov     dword ecx, address.high_byte
                        jnc     .cycle1_1
                        mov     dword ecx, address.high_byte_increase
.cycle1_1:              inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* address_absolute_y *******************************************************
;****************************************************************************
section .text
address_absolute_y:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        add     byte  al, [INDEX_Y]
                        mov     dword ecx, address.high_byte
                        jnc     .cycle1_1
                        mov     dword ecx, address.high_byte_increase
.cycle1_1:              inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* address_x_indirect *******************************************************
;****************************************************************************
section .text
address_x_indirect:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        add     byte  al, [INDEX_X]
                        inc     word  bx
                        mov     dword ecx, .cycle2
                        mov     byte  [ZEROPAGE_ADDRESS], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle2
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle3
                        ret

CYCLE_LABEL .cycle3
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        CALL_READ_DATA
                        mov     dword ecx, .cycle4
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle4
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        mov     dword ecx, [OPTIONAL_ADDRESS]
                        inc     byte  bl
                        CALL_READ_DATA
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* address_indirect_y *******************************************************
;****************************************************************************
section .text
address_indirect_y:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [ZEROPAGE_ADDRESS], al
                        mov     dword ecx, .cycle2
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        CALL_READ_DATA
                        add     byte  al, [INDEX_Y]
                        mov     dword ecx, .cycle3_1
                        jnc     .cycle2_1
                        mov     dword ecx, .cycle3_2
.cycle2_1:              mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle3_1
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        mov     dword ecx, [OPTIONAL_ADDRESS]
                        inc     byte  bl
                        CALL_READ_DATA
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        ret

CYCLE_LABEL .cycle3_2
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        mov     dword ecx, address.delay
                        inc     byte  bl
                        CALL_READ_DATA
                        inc     byte  al
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        ret



;****************************************************************************
;* address_absolute_x_write *************************************************
;****************************************************************************
section .text
address_absolute_x_write:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        add     byte  al, [INDEX_X]
                        mov     dword ecx, address.high_byte_delay
                        jnc     .cycle1_1
                        mov     dword ecx, address.high_byte_increase
.cycle1_1:              inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* address_absolute_y_write *************************************************
;****************************************************************************
section .text
address_absolute_y_write:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        add     byte  al, [INDEX_Y]
                        mov     dword ecx, address.high_byte_delay
                        jnc     .cycle1_1
                        mov     dword ecx, address.high_byte_increase
.cycle1_1:              inc     word  bx
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* address_indirect_y_write *************************************************
;****************************************************************************
section .text
address_indirect_y_write:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [ZEROPAGE_ADDRESS], al
                        mov     dword ecx, .cycle2
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        CALL_READ_DATA
                        add     byte  al, [INDEX_Y]
                        mov     dword ecx, .cycle3_1
                        jnc     .cycle2_1
                        mov     dword ecx, .cycle3_2
.cycle2_1:              mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle3_1
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        mov     dword ecx, address.delay
                        inc     byte  bl
                        CALL_READ_DATA
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        ret

;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;use address_indirect_y.cycle3_2 ?
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
CYCLE_LABEL .cycle3_2
                        mov     dword ebx, [ZEROPAGE_ADDRESS]
                        mov     dword ecx, address.delay
                        inc     byte  bl
                        CALL_READ_DATA
                        inc     byte  al
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        ret



;****************************************************************************
;* address_rmw **************************************************************
;****************************************************************************
section .text
address_rmw:
CYCLE_LABEL .delay
                        mov     byte  al, [DATA_BYTE]
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        mov     dword ecx, .write
                        CALL_WRITE_DATA
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .write
                        mov     byte  al, [ALU_BYTE]
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        mov     dword ecx, get_opcode
                        CALL_WRITE_DATA
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret





;****************************************************************************
;****************************************************************************
;* PART 5.2: Opcodes
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* opcode_adc ***************************************************************
;****************************************************************************
section .text
opcode_adc:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  ah, [ACCUMULATOR]
                        mov     dword [PROGRAM_COUNTER], ebx
                        test    byte  [FLAG_REGISTER], FLAG_D
                        jz      near  .binary
.decimal:               mov     dword ecx, eax
                        mov     dword edx, eax
                        and     dword ecx, 000000f0fh
                        and     dword edx, 00000f0f0h
                        test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_1
                        inc     byte  ch
                        inc     byte  ah
.carry_clear_1:         add     byte  cl, ch
                        add     byte  al, ah
                        setz    byte  ah
                        cmp     byte  cl, 00ah
                        jb      .no_add_1
                        add     byte  cl, 006h
.no_add_1:              or      byte  dl, 00fh
                        cmp     byte  cl, 010h
                        cmc
                        adc     byte  dl, dh
                        setc    byte  dh
                        sets    byte  bh
                        seto    byte  bl
                        cmp     dword edx, 0000000a0h
                        jb      .no_add_2
                        add     dword edx, 000000060h
.no_add_2:              and     byte  dl, 0f0h
                        and     byte  cl, 00fh
                        test    byte  dh, NOT8
                        setnz   byte  al
                        shl     byte  ah, FLAG_Z_BIT
                        or      byte  cl, dl
                        shl     byte  bl, FLAG_V_BIT
                        mov     byte  dl, [FLAG_REGISTER]
                        shl     byte  bh, FLAG_N_BIT
                        or      byte  al, ah
                        mov     byte  [ACCUMULATOR], cl
                        or      byte  bl, bh
                        and     byte  dl, (NOT8 - FLAG_N - FLAG_V - FLAG_Z - FLAG_C)
                        or      byte  al, bl
                        mov     dword ecx, get_opcode
                        or      byte  dl, al
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [FLAG_REGISTER], dl
                        ret

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  ah, [ACCUMULATOR]
                        test    byte  [FLAG_REGISTER], FLAG_D
                        jnz     near  .decimal
.binary:                test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_2
                        stc
.carry_clear_2:         adc     byte  ah, al
                        setc    byte  cl
                        jmp     opcode_sbc.common



;****************************************************************************
;* opcode_sbc ***************************************************************
;****************************************************************************
section .text
opcode_sbc:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  ah, [ACCUMULATOR]
                        mov     dword [PROGRAM_COUNTER], ebx
                        test    byte  [FLAG_REGISTER], FLAG_D
                        jz      .binary
.decimal:               mov     dword ecx, eax
                        mov     dword edx, eax
                        and     dword ecx, 000000f0fh
                        and     dword edx, 00000f0f0h
                        inc     byte  cl
                        test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_1
                        dec     byte  cl
                        stc
.carry_clear_1:         cmc
                        sbb     byte  ah, al
                        setnc   byte  al
                        setz    byte  ah
                        seto    byte  bl
                        sets    byte  bh
                        sub     byte  ch, cl
                        jnc     .no_sub_1
                        sub     byte  ch, 006h
                        stc
.no_sub_1:              sbb     byte  dh, dl
                        jnc     .no_sub_2
                        sub     byte  dh, 060h
.no_sub_2:              and     byte  ch, 00fh
                        and     byte  dh, 0f0h
                        shl     byte  ah, FLAG_Z_BIT
                        or      byte  ch, dh
                        shl     byte  bl, FLAG_V_BIT
                        mov     byte  dl, [FLAG_REGISTER]
                        shl     byte  bh, FLAG_N_BIT
                        or      byte  al, ah
                        mov     byte  [ACCUMULATOR], ch
                        or      byte  bl, bh
                        and     byte  dl, (NOT8 - FLAG_N - FLAG_V - FLAG_Z - FLAG_C)
                        or      byte  al, bl
                        mov     dword ecx, get_opcode
                        or      byte  dl, al
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [FLAG_REGISTER], dl
                        ret

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  ah, [ACCUMULATOR]
                        test    byte  [FLAG_REGISTER], FLAG_D
                        jnz     .decimal
.binary:                test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_2
                        stc
.carry_clear_2:         cmc
                        sbb     byte  ah, al
                        setnc   byte  cl

.common:                setz    byte  ch
                        seto    byte  dl
                        sets    byte  dh
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_V_BIT
                        mov     byte  [ACCUMULATOR], ah
                        shl     byte  dh, FLAG_N_BIT
                        or      byte  cl, ch
                        or      byte  dl, dh
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_V - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_cmp ***************************************************************
;****************************************************************************
section .text
opcode_cmp:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  ah, [ACCUMULATOR]
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_cpy.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  ah, [ACCUMULATOR]
                        jmp     opcode_cpy.common



;****************************************************************************
;* opcode_cpx ***************************************************************
;****************************************************************************
section .text
opcode_cpx:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  ah, [INDEX_X]
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_cpy.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  ah, [INDEX_X]
                        jmp     opcode_cpy.common



;****************************************************************************
;* opcode_cpy ***************************************************************
;****************************************************************************
section .text
opcode_cpy:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  ah, [INDEX_Y]
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_cpy.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  ah, [INDEX_Y]

.common:                sub     byte  ah, al
                        setnc   byte  cl
                        setz    byte  ch
                        sets    byte  dl
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        or      byte  cl, ch
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_and ***************************************************************
;****************************************************************************
section .text
opcode_and:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        mov     byte  ch, [ACCUMULATOR]
                        CALL_READ_OPCODE
                        inc     word  bx
                        and     byte  ch, al
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_ora.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        mov     byte  ch, [ACCUMULATOR]
                        CALL_READ_DATA
                        and     byte  ch, al
                        jmp     opcode_ora.common



;****************************************************************************
;* opcode_eor ***************************************************************
;****************************************************************************
section .text
opcode_eor:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        mov     byte  ch, [ACCUMULATOR]
                        CALL_READ_OPCODE
                        inc     word  bx
                        xor     byte  ch, al
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_ora.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        mov     byte  ch, [ACCUMULATOR]
                        CALL_READ_DATA
                        xor     byte  ch, al
                        jmp     opcode_ora.common



;****************************************************************************
;* opcode_ora ***************************************************************
;****************************************************************************
section .text
opcode_ora:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        mov     byte  ch, [ACCUMULATOR]
                        CALL_READ_OPCODE
                        inc     word  bx
                        or      byte  ch, al
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_ora.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        mov     byte  ch, [ACCUMULATOR]
                        CALL_READ_DATA
                        or      byte  ch, al

.common:                setz    byte  cl
                        sets    byte  dl
                        shl     byte  cl, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        mov     byte  [ACCUMULATOR], ch
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_asl ***************************************************************
;****************************************************************************
section .text
opcode_asl:
CYCLE_LABEL .accumulator
                        CHECK_FOR_DMA
                        mov     byte  al, [ACCUMULATOR]
                        shl     byte  al, 1
                        jmp     opcode_ror.accumulator_common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        shl     byte  al, 1
                        jmp     opcode_ror.read_common



;****************************************************************************
;* opcode_lsr ***************************************************************
;****************************************************************************
section .text
opcode_lsr:
CYCLE_LABEL .accumulator
                        CHECK_FOR_DMA
                        mov     byte  al, [ACCUMULATOR]
                        shr     byte  al, 1
                        jmp     opcode_ror.accumulator_common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        shr     byte  al, 1
                        jmp     opcode_ror.read_common



;****************************************************************************
;* opcode_rol ***************************************************************
;****************************************************************************
section .text
opcode_rol:
CYCLE_LABEL .accumulator
                        CHECK_FOR_DMA
                        mov     byte  al, [ACCUMULATOR]
                        test    byte  [FLAG_REGISTER], CPU65XX_FLAG_C
                        jz      .carry_clear_1
                        stc
.carry_clear_1:         rcl     byte  al, 1
                        jmp     opcode_ror.accumulator_common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        test    byte  [FLAG_REGISTER], CPU65XX_FLAG_C
                        jz      .carry_clear_2
                        stc
.carry_clear_2:         rcl     byte  al, 1
                        jmp     opcode_ror.read_common



;****************************************************************************
;* opcode_ror ***************************************************************
;****************************************************************************
section .text
opcode_ror:
CYCLE_LABEL .accumulator
                        CHECK_FOR_DMA
                        mov     byte  al, [ACCUMULATOR]
                        test    byte  [FLAG_REGISTER], CPU65XX_FLAG_C
                        jz      .carry_clear_1
                        stc
.carry_clear_1:         rcr     byte  al, 1

.accumulator_common:    setc    byte  cl
                        test    byte  al, al
                        setz    byte  ch
                        sets    byte  dl
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        or      byte  cl, ch
                        mov     byte  [ACCUMULATOR], al
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        test    byte  [FLAG_REGISTER], CPU65XX_FLAG_C
                        jz      .carry_clear_2
                        stc
.carry_clear_2:         rcr     byte  al, 1

.read_common:           setc    byte  cl
                        test    byte  al, al
                        setz    byte  ch
                        sets    byte  dl
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        or      byte  cl, ch
                        mov     byte  [ALU_BYTE], al
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, address_rmw
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_dec ***************************************************************
;****************************************************************************
section .text
opcode_dec:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        dec     byte  al
                        jmp     opcode_inc.common


;****************************************************************************
;* opcode_inc ***************************************************************
;****************************************************************************
section .text
opcode_inc:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        inc     byte  al
.common:                setz    byte  cl
                        sets    byte  dl
                        shl     byte  cl, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        mov     byte  [ALU_BYTE], al
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z)
                        or      byte  cl, dl
                        mov     dword eax, address_rmw
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_dex ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_dex
                        CHECK_FOR_DMA
                        dec     byte  [INDEX_X]
                        jmp     opcode_iny.common



;****************************************************************************
;* opcode_dey ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_dey
                        CHECK_FOR_DMA
                        dec     byte  [INDEX_Y]
                        jmp     opcode_iny.common



;****************************************************************************
;* opcode_inx ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_inx
                        CHECK_FOR_DMA
                        inc     byte  [INDEX_X]
                        jmp     opcode_iny.common


;****************************************************************************
;* opcode_iny ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_iny
                        CHECK_FOR_DMA
                        inc     byte  [INDEX_Y]
.common:                setz    byte  cl
                        sets    byte  dl
                        shl     byte  cl, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_bcc ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_bcc
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_C
                        mov     dword [PROGRAM_COUNTER], ebx
                        jz      near  opcode_bvs.common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_bcs ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_bcs
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_C
                        mov     dword [PROGRAM_COUNTER], ebx
                        jnz     near  opcode_bvs.common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_beq ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_beq
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_Z
                        mov     dword [PROGRAM_COUNTER], ebx
                        jnz     near  opcode_bvs.common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_bne ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_bne
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_Z
                        mov     dword [PROGRAM_COUNTER], ebx
                        jz      opcode_bvs.common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_bmi ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_bmi
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_N
                        mov     dword [PROGRAM_COUNTER], ebx
                        jnz     opcode_bvs.common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_bpl ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_bpl
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_N
                        mov     dword [PROGRAM_COUNTER], ebx
                        jz      opcode_bvs.common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_bvc ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_bvc
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_V
                        mov     dword [PROGRAM_COUNTER], ebx
                        jz      opcode_bvs.common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_bvs ***************************************************************
;****************************************************************************
section .text
opcode_bvs:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        test    byte  [FLAG_REGISTER], FLAG_V
                        mov     dword [PROGRAM_COUNTER], ebx
                        jnz     .common
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret

.common:                cbw
                        mov     word  [EFFECTIVE_ADDRESS], ax
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        CHECK_FOR_DMA
                        mov     dword eax, [EFFECTIVE_ADDRESS]
                        mov     dword ebx, [PROGRAM_COUNTER]
                        add     word  ax, bx
                        mov     dword [PROGRAM_COUNTER], eax
                        cmp     byte  ah, bh
                        je      .cycle3_1
                        mov     dword [EXECUTE_CYCLE], .cycle3
                        ret

CYCLE_LABEL .cycle3
                        CHECK_FOR_DMA
.cycle3_1:              mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_clc ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_clc
                        CHECK_FOR_DMA
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, get_opcode
                        and     byte  al, (NOT8 - FLAG_C)
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     byte  [FLAG_REGISTER], al
                        ret



;****************************************************************************
;* opcode_cld ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_cld
                        CHECK_FOR_DMA
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, get_opcode
                        and     byte  al, (NOT8 - FLAG_D)
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     byte  [FLAG_REGISTER], al
                        ret



;****************************************************************************
;* opcode_cli ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_cli
                        CHECK_FOR_DMA
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, get_opcode
                        test    byte  al, FLAG_I
                        jz      .flag_cleared
                        and     byte  al, (NOT8 - FLAG_I)
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     byte  [FLAG_REGISTER], al
                        call    dword [GET_CYCLES]
                        mov     dword [IRQ_TIMESTAMP], eax
                        mov     dword [IRQ_TIMESTAMP + 4], edx
                        ret
.flag_cleared:          mov     dword [EXECUTE_CYCLE], ebx
                        ret



;****************************************************************************
;* opcode_clv ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_clv
                        CHECK_FOR_DMA
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, get_opcode
                        and     byte  al, (NOT8 - FLAG_V)
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     byte  [FLAG_REGISTER], al
                        ret



;****************************************************************************
;* opcode_sec ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_sec
                        CHECK_FOR_DMA
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, get_opcode
                        or      byte  al, FLAG_C
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     byte  [FLAG_REGISTER], al
                        ret



;****************************************************************************
;* opcode_sed ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_sed
                        CHECK_FOR_DMA
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, get_opcode
                        or      byte  al, FLAG_D
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     byte  [FLAG_REGISTER], al
                        ret



;****************************************************************************
;* opcode_sei ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_sei
                        CHECK_FOR_DMA
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, get_opcode
                        or      byte  al, FLAG_I
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     byte  [FLAG_REGISTER], al
                        ret


;****************************************************************************
;* opcode_tax ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_tax
                        CHECK_FOR_DMA
                        mov     byte  al, [ACCUMULATOR]
                        mov     byte  [INDEX_X], al
                        jmp     opcode_tya.common



;****************************************************************************
;* opcode_tay ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_tay
                        CHECK_FOR_DMA
                        mov     byte  al, [ACCUMULATOR]
                        mov     byte  [INDEX_Y], al
                        jmp     opcode_tya.common



;****************************************************************************
;* opcode_tsx ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_tsx
                        CHECK_FOR_DMA
                        mov     byte  al, [STACK_POINTER]
                        mov     byte  [INDEX_X], al
                        jmp     opcode_tya.common



;****************************************************************************
;* opcode_txa ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_txa
                        CHECK_FOR_DMA
                        mov     byte  al, [INDEX_X]
                        mov     byte  [ACCUMULATOR], al
                        jmp     opcode_tya.common



;****************************************************************************
;* opcode_txs ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_txs
                        CHECK_FOR_DMA
                        mov     byte  al, [INDEX_X]
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        mov     byte  [STACK_POINTER], al
                        ret


;****************************************************************************
;* opcode_tya ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_tya
                        CHECK_FOR_DMA
                        mov     byte  al, [INDEX_Y]
                        mov     byte  [ACCUMULATOR], al
.common:                test    byte  al, al
                        setz    byte  cl
                        sets    byte  dl
                        shl     byte  cl, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_lda ***************************************************************
;****************************************************************************
section .text
opcode_lda:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [ACCUMULATOR], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_ldy.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [ACCUMULATOR], al
                        jmp     opcode_ldy.common



;****************************************************************************
;* opcode_ldx ***************************************************************
;****************************************************************************
section .text
opcode_ldx:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [INDEX_X], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_ldy.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [INDEX_X], al
                        jmp     opcode_ldy.common



;****************************************************************************
;* opcode_ldy ***************************************************************
;****************************************************************************
section .text
opcode_ldy:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     byte  [INDEX_Y], al
                        mov     dword [PROGRAM_COUNTER], ebx
                        jmp     opcode_ldy.common

CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [INDEX_Y], al

.common:                test    byte  al, al
                        setz    byte  cl
                        sets    byte  dl
                        shl     byte  cl, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_sta ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_sta
                        mov     byte  al, [ACCUMULATOR]
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_WRITE_DATA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_stx ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_stx
                        mov     byte  al, [INDEX_X]
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_WRITE_DATA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_sty ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_sty
                        mov     byte  al, [INDEX_Y]
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_WRITE_DATA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_jmp ***************************************************************
;****************************************************************************
section .text
opcode_jmp:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        mov     dword ecx, .cycle2
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [PROGRAM_COUNTER]
                        xor     dword ecx, ecx
                        inc     word  bx
                        CALL_READ_OPCODE
                        mov     byte  ch, al
                        mov     dword edx, get_opcode
                        mov     byte  cl, [EFFECTIVE_ADDRESS]
                        mov     dword [EXECUTE_CYCLE], edx
                        mov     dword [PROGRAM_COUNTER], ecx
                        ret



;****************************************************************************
;* opcode_jmp_indirect ******************************************************
;****************************************************************************
section .text
opcode_jmp_indirect:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        mov     dword ecx, .cycle2
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [PROGRAM_COUNTER]
                        mov     dword ecx, .cycle3
                        inc     word bx
                        CALL_READ_OPCODE
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle3
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     dword ecx, .cycle4
                        mov     byte  [PROGRAM_COUNTER], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle4
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        mov     dword ecx, get_opcode
                        inc     byte  bl  ; no page crossing
                        CALL_READ_DATA
                        mov     byte  [PROGRAM_COUNTER + 1], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret



;****************************************************************************
;* opcode_jsr ***************************************************************
;****************************************************************************
section .text
opcode_jsr:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        inc     word  bx
                        mov     dword ecx, .cycle2
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     byte  [EFFECTIVE_ADDRESS], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        mov     dword ecx, .cycle3
                        mov     byte  [EFFECTIVE_ADDRESS + 1], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle3
                        mov     byte  al, [PROGRAM_COUNTER + 1]
                        mov     dword ebx, [STACK_POINTER]
                        mov     dword ecx, .cycle4
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle4
                        mov     byte  al, [PROGRAM_COUNTER]
                        mov     dword ebx, [STACK_POINTER]
                        mov     dword ecx, .cycle5
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle5
                        CHECK_FOR_DMA
                        mov     dword eax, [EFFECTIVE_ADDRESS]
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        mov     dword [PROGRAM_COUNTER], eax
                        ret



;****************************************************************************
;* opcode_bit ***************************************************************
;****************************************************************************
section .text
opcode_bit:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  bl, [FLAG_REGISTER]
                        and     byte  bl, (3fh - FLAG_Z)
                        test    byte  [ACCUMULATOR], al
                        jnz     .bit1
                        or      byte  bl, FLAG_Z
.bit1:                  and     byte  al, 0c0h
                        mov     dword ecx, get_opcode
                        or      byte  bl, al
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_brk ***************************************************************
;****************************************************************************
section .text
opcode_brk:
CYCLE_LABEL .cycle1
                        CHECK_FOR_DMA
                        inc     word  [PROGRAM_COUNTER]
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [STACK_POINTER]
                        mov     byte  al, [PROGRAM_COUNTER + 1]
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], .cycle3
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle3
                        mov     dword ebx, [STACK_POINTER]
                        mov     byte  al, [PROGRAM_COUNTER]
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], .cycle4
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle4
                        mov     dword ebx, [STACK_POINTER]
                        mov     byte  al, [FLAG_REGISTER]
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], .cycle5
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle5
                        mov     dword ebx, 00000fffeh
                        CALL_READ_OPCODE
                        mov     byte  [PROGRAM_COUNTER], al
                        mov     dword [EXECUTE_CYCLE], .cycle6
                        ret

CYCLE_LABEL .cycle6
                        mov     dword ebx, 00000ffffh
                        CALL_READ_OPCODE
                        mov     byte  bl, [FLAG_REGISTER]
                        mov     byte  [PROGRAM_COUNTER + 1], al
                        or      byte  bl, (FLAG_B | FLAG_I)
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_nop ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_nop
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_pha ***************************************************************
;****************************************************************************
section .text
opcode_pha:
CYCLE_LABEL .cycle1
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [STACK_POINTER]
                        mov     byte  al, [ACCUMULATOR]
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        mov     dword [STACK_POINTER], ebx
                        ret



;****************************************************************************
;* opcode_php ***************************************************************
;****************************************************************************
section .text
opcode_php:
CYCLE_LABEL .cycle1
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [STACK_POINTER]
                        mov     byte  al, [FLAG_REGISTER]
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        mov     dword [STACK_POINTER], ebx
                        ret



;****************************************************************************
;* opcode_pla ***************************************************************
;****************************************************************************
section .text
opcode_pla:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [STACK_POINTER]
                        inc     byte  bl
                        CALL_READ_STACK
                        mov     byte  [ACCUMULATOR], al
                        mov     dword [STACK_POINTER], ebx
                        test    byte  al, al
                        setz    byte  cl
                        sets    byte  dl
                        shl     byte  cl, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z)
                        or      byte  cl, dl
                        mov     dword eax, .cycle2
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret

CYCLE_LABEL .cycle2
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle3
                        ret

CYCLE_LABEL .cycle3
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_plp ***************************************************************
;****************************************************************************
section .text
opcode_plp:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [STACK_POINTER]
                        inc     byte  bl
                        CALL_READ_STACK
                        or      byte  al, (FLAG_R | FLAG_B)
                        mov     byte  [FLAG_REGISTER], al
                        mov     dword [STACK_POINTER], ebx
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle3
                        ret

CYCLE_LABEL .cycle3
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret




;****************************************************************************
;* opcode_rti ***************************************************************
;****************************************************************************
section .text
opcode_rti:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [STACK_POINTER]
                        inc     byte  bl
                        CALL_READ_STACK
                        or      byte  al, (FLAG_R | FLAG_B)
                        mov     byte  [FLAG_REGISTER], al
                        mov     dword [STACK_POINTER], ebx
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [STACK_POINTER]
                        inc     byte  bl
                        CALL_READ_STACK
                        mov     byte  [PROGRAM_COUNTER], al
                        mov     dword [STACK_POINTER], ebx
                        mov     dword [EXECUTE_CYCLE], .cycle3
                        ret

CYCLE_LABEL .cycle3
                        mov     dword ebx, [STACK_POINTER]
                        inc     byte  bl
                        CALL_READ_STACK
                        mov     byte  [PROGRAM_COUNTER + 1], al
                        mov     dword [STACK_POINTER], ebx
                        mov     dword [EXECUTE_CYCLE], .cycle4
                        ret

CYCLE_LABEL .cycle4
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle5
                        ret

CYCLE_LABEL .cycle5
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_rts ***************************************************************
;****************************************************************************
section .text
opcode_rts:
CYCLE_LABEL .cycle1
                        mov     dword ebx, [STACK_POINTER]
                        inc     byte  bl
                        CALL_READ_STACK
                        mov     byte  [PROGRAM_COUNTER], al
                        mov     dword [STACK_POINTER], ebx
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        mov     dword ebx, [STACK_POINTER]
                        inc     byte  bl
                        CALL_READ_STACK
                        mov     byte  [PROGRAM_COUNTER + 1], al
                        mov     dword [STACK_POINTER], ebx
                        mov     dword [EXECUTE_CYCLE], .cycle3
                        ret

CYCLE_LABEL .cycle3
                        CHECK_FOR_DMA
                        mov     dword eax, [PROGRAM_COUNTER]
                        mov     dword ebx, .cycle4
                        inc     word  ax
                        mov     dword [EXECUTE_CYCLE], ebx
                        mov     dword [PROGRAM_COUNTER], eax
                        ret

CYCLE_LABEL .cycle4
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle5
                        ret

CYCLE_LABEL .cycle5
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



%ifndef CPU65XX_DISABLE_ILLEGAL_OPCODES
;****************************************************************************
;* opcode_hlt ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_hlt
                        mov     dword eax, [MODULID]
                        mov     dword ebx, 00bad0000h
                        mov     dword ecx, __LINE__
                        mov     byte  bl, [OPCODE]
                        jmp     core_exception



;****************************************************************************
;* opcode_aso ***************************************************************
;****************************************************************************
section .text
opcode_aso:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        shl     byte  al, 1
                        mov     byte  ah, [ACCUMULATOR]
                        setc    byte  cl
                        or      byte  ah, al
                        jmp     opcode_lse.read_common



;****************************************************************************
;* opcode_rla ***************************************************************
;****************************************************************************
section .text
opcode_rla:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        test    byte  [FLAG_REGISTER], FLAG_C
                        mov     byte  ah, [ACCUMULATOR]
                        jz      .carry_clear
                        stc
.carry_clear:           rcl     byte  al, 1
                        setc    byte  cl
                        and     byte  ah, al
                        jmp     opcode_lse.read_common



;****************************************************************************
;* opcode_lse ***************************************************************
;****************************************************************************
section .text
opcode_lse:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        mov     byte  ah, [ACCUMULATOR]
                        shr     byte  al, 1
                        setc    byte  cl
                        xor     byte  ah, al
.read_common:           setz    byte  ch
                        sets    byte  dl
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        or      byte  cl, ch
                        mov     byte  [ALU_BYTE], al
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z - FLAG_C)
                        mov     byte  [ACCUMULATOR], ah
                        or      byte  cl, dl
                        mov     dword eax, address_rmw
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_rra ***************************************************************
;****************************************************************************
section .text
opcode_rra:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  bl, [FLAG_REGISTER]
                        mov     byte  [DATA_BYTE], al
                        test    byte  bl, FLAG_C
                        mov     byte  ah, [ACCUMULATOR]
                        jz      .carry_clear
                        stc
.carry_clear:           rcr     byte  al, 1
                        setc    byte  cl
                        and     byte  bl, (NOT8 - FLAG_C)
                        or      byte  bl, cl
                        mov     byte  [ALU_BYTE], al
                        test    byte  bl, FLAG_D
                        mov     byte  [FLAG_REGISTER], bl
                        jz      near  .binary
.decimal:               mov     dword ecx, eax
                        mov     dword edx, eax
                        and     dword ecx, 000000f0fh
                        and     dword edx, 00000f0f0h
                        test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_1
                        inc     byte  ch
                        inc     byte  ah
.carry_clear_1:         add     byte  cl, ch
                        add     byte  al, ah
                        setz    byte  ah
                        cmp     byte  cl, 00ah
                        jb      .no_add_1
                        add     byte  cl, 006h
.no_add_1:              or      byte  dl, 00fh
                        cmp     byte  cl, 010h
                        cmc
                        adc     byte  dl, dh
                        setc    byte  dh
                        sets    byte  bh
                        seto    byte  bl
                        cmp     dword edx, 0000000a0h
                        jb      .no_add_2
                        add     dword edx, 000000060h
.no_add_2:              and     byte  dl, 0f0h
                        and     byte  cl, 00fh
                        test    byte  dh, NOT8
                        setnz   byte  al
                        shl     byte  ah, FLAG_Z_BIT
                        or      byte  cl, dl
                        shl     byte  bl, FLAG_V_BIT
                        mov     byte  dl, [FLAG_REGISTER]
                        shl     byte  bh, FLAG_N_BIT
                        or      byte  al, ah
                        mov     byte  [ACCUMULATOR], cl
                        or      byte  bl, bh
                        and     byte  dl, (NOT8 - FLAG_N - FLAG_V - FLAG_Z - FLAG_C)
                        or      byte  al, bl
                        mov     dword ecx, address_rmw
                        or      byte  dl, al
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [FLAG_REGISTER], dl
                        ret

.binary:                test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_2
                        stc
.carry_clear_2:         adc     byte  ah, al
                        setc    byte  cl
                        setz    byte  ch
                        seto    byte  dl
                        sets    byte  dh
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_V_BIT
                        mov     byte  [ACCUMULATOR], ah
                        shl     byte  dh, FLAG_N_BIT
                        or      byte  cl, ch
                        or      byte  dl, dh
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_V - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, address_rmw
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_axs ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_axs
                        mov     byte  al, [ACCUMULATOR]
                        and     byte  al, [INDEX_X]
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_WRITE_DATA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_lax ***************************************************************
;****************************************************************************
section .text
opcode_lax:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [ACCUMULATOR], al
                        mov     byte  [INDEX_X], al
                        test    byte  al, al
                        setz    byte  cl
                        sets    byte  dl
                        shl     byte  cl, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_dcm ***************************************************************
;****************************************************************************
section .text
opcode_dcm:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        mov     byte  ah, [ACCUMULATOR]
                        dec     byte  al
                        sub     byte  ah, al
                        setnc   byte  cl
                        setz    byte  ch
                        sets    byte  dl
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_N_BIT
                        or      byte  cl, ch
                        mov     byte  [ALU_BYTE], al
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, address_rmw
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret


;****************************************************************************
;* opcode_ins ***************************************************************
;****************************************************************************
section .text
opcode_ins:
CYCLE_LABEL .read
                        mov     dword ebx, [EFFECTIVE_ADDRESS]
                        CALL_READ_DATA
                        mov     byte  [DATA_BYTE], al
                        mov     byte  ah, [ACCUMULATOR]
                        inc     byte  al
                        mov     byte  [ALU_BYTE], al
                        test    byte  [FLAG_REGISTER], FLAG_D
                        jz      .binary
.decimal:               mov     dword ecx, eax
                        mov     dword edx, eax
                        and     dword ecx, 000000f0fh
                        and     dword edx, 00000f0f0h
                        inc     byte  cl
                        test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_1
                        dec     byte  cl
                        stc
.carry_clear_1:         cmc
                        sbb     byte  ah, al
                        setnc   byte  al
                        setz    byte  ah
                        seto    byte  bl
                        sets    byte  bh
                        sub     byte  ch, cl
                        jnc     .no_sub_1
                        sub     byte  ch, 006h
                        stc
.no_sub_1:              sbb     byte  dh, dl
                        jnc     .no_sub_2
                        sub     byte  dh, 060h
.no_sub_2:              and     byte  ch, 00fh
                        and     byte  dh, 0f0h
                        shl     byte  ah, FLAG_Z_BIT
                        or      byte  ch, dh
                        shl     byte  bl, FLAG_V_BIT
                        mov     byte  dl, [FLAG_REGISTER]
                        shl     byte  bh, FLAG_N_BIT
                        or      byte  al, ah
                        mov     byte  [ACCUMULATOR], ch
                        or      byte  bl, bh
                        and     byte  dl, (NOT8 - FLAG_N - FLAG_V - FLAG_Z - FLAG_C)
                        or      byte  al, bl
                        mov     dword ecx, address_rmw
                        or      byte  dl, al
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     byte  [FLAG_REGISTER], dl
                        ret

.binary:                test    byte  [FLAG_REGISTER], FLAG_C
                        jz      .carry_clear_2
                        stc
.carry_clear_2:         cmc
                        sbb     byte  ah, al
                        setnc   byte  cl
                        setz    byte  ch
                        seto    byte  dl
                        sets    byte  dh
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        shl     byte  dl, FLAG_V_BIT
                        mov     byte  [ACCUMULATOR], ah
                        shl     byte  dh, FLAG_N_BIT
                        or      byte  cl, ch
                        or      byte  dl, dh
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_V - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, address_rmw
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret



;****************************************************************************
;* opcode_top ***************************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_top
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        ret



;****************************************************************************
;* opcode_anc ***************************************************************
;****************************************************************************
section .text
opcode_anc:
CYCLE_LABEL .immediate
                        mov     dword ebx, [PROGRAM_COUNTER]
                        mov     byte  ch, [ACCUMULATOR]
                        CALL_READ_OPCODE
                        inc     word  bx
                        and     byte  ch, al
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     byte  [ACCUMULATOR], ch
                        sets    byte  cl
                        setz    byte  ch
                        shl     byte  ch, FLAG_Z_BIT
                        mov     byte  dl, cl
                        shl     byte  dl, FLAG_N_BIT
                        mov     byte  bl, [FLAG_REGISTER]
                        or      byte  cl, ch
                        and     byte  bl, (NOT8 - FLAG_N - FLAG_Z - FLAG_C)
                        or      byte  cl, dl
                        mov     dword eax, get_opcode
                        or      byte  bl, cl
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     byte  [FLAG_REGISTER], bl
                        ret
%endif



;****************************************************************************
;* opcode_illegal ***********************************************************
;****************************************************************************
section .text
CYCLE_LABEL opcode_illegal
                        or      dword [STATE], CPU65XX_STATE_ILLEGAL_OPCODE
                        mov     dword eax, [MODULID]
                        xor     dword ebx, ebx
                        mov     dword ecx, __LINE__
                        dec     dword ebx
                        mov     byte  bl, [OPCODE]
                        jmp     core_exception



;****************************************************************************
;* get_opcode ***************************************************************
;****************************************************************************
;* fetch a byte from memory and interpret it as opcode
;****************************************************************************
section .text
CYCLE_LABEL get_opcode
                        test    dword [STATE], STATE_SIGNAL_IRQ_OR_NMI
                        jz      .no_irq_or_nmi

                        mov     dword eax, [STATE]
                        test    byte  [FLAG_REGISTER], FLAG_I
                        jnz     .no_irq
                        test    dword eax, STATE_SIGNAL_IRQ
                        jz      .no_irq
                        call    dword [GET_CYCLES]
                        sub     dword eax, [IRQ_TIMESTAMP]
                        sbb     dword edx, [IRQ_TIMESTAMP + 4]
                        cmp     dword eax, 2
                        jbe     .no_irq
                        or      dword [STATE], STATE_PROCESSING_IRQ
                        jmp     process_irq

.no_irq:                test    dword eax, STATE_SIGNAL_NMI
                        jz      .no_irq_or_nmi
                        call    dword [GET_CYCLES]
                        sub     dword eax, [NMI_TIMESTAMP]
                        sbb     dword edx, [NMI_TIMESTAMP + 4]
                        cmp     dword eax, 2
                        jbe     .no_irq_or_nmi
                        or      dword [STATE], STATE_PROCESSING_NMI
                        jmp     process_nmi

.no_irq_or_nmi:         mov     dword ebx, [PROGRAM_COUNTER]
                        CALL_READ_OPCODE
                        xor     dword ecx, ecx
                        mov     byte  [OPCODE], al
                        inc     word  bx
                        mov     byte  cl, al
                        mov     dword [PROGRAM_COUNTER], ebx
                        mov     dword eax, [ap__opcodes + 8 * ecx]
                        mov     dword ebx, [ap__opcodes + 8 * ecx + 4]
                        mov     dword [EXECUTE_CYCLE], eax
                        mov     dword [OPTIONAL_ADDRESS], ebx
                        ret



;****************************************************************************
;* process_irq **************************************************************
;****************************************************************************
process_irq:
CYCLE_LABEL .cycle0
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle1
                        ret

CYCLE_LABEL .cycle1
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        mov     byte  al, [PROGRAM_COUNTER + 1]
                        mov     dword ebx, [STACK_POINTER]
                        mov     dword ecx, .cycle3
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle3
                        mov     byte  al, [PROGRAM_COUNTER]
                        mov     dword ebx, [STACK_POINTER]
                        mov     dword ecx, .cycle4
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle4
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, [STACK_POINTER]
                        and     byte  al, (NOT8 - FLAG_B)
                        mov     dword ecx, .cycle5
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle5
                        mov     dword ebx, 00000fffeh
                        CALL_READ_OPCODE
                        mov     dword ecx, .cycle6
                        mov     byte  [PROGRAM_COUNTER], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle6
                        mov     dword ebx, 00000ffffh
                        CALL_READ_OPCODE
                        mov     byte  bl, [FLAG_REGISTER]
                        mov     byte  [PROGRAM_COUNTER + 1], al
                        or      byte  bl, FLAG_I
                        mov     dword eax, [STATE]
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        and     dword eax, (NOT32 - STATE_PROCESSING_IRQ)
                        mov     byte  [FLAG_REGISTER], bl
                        mov     dword [STATE], eax
                        ret



;****************************************************************************
;* process_nmi **************************************************************
;****************************************************************************
process_nmi:
CYCLE_LABEL .cycle0
                        CHECK_FOR_DMA
                        mov     dword eax, [STATE]
                        mov     dword ecx, .cycle1
                        test    dword eax, STATE_SIGNAL_IRQ
                        jnz     .cycle0_end
                        and     dword eax, (NOT32 - STATE_SIGNAL_IRQ_OR_NMI)
.cycle0_end:            and     dword eax, (NOT32 - STATE_SIGNAL_NMI)
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STATE], eax
                        ret

CYCLE_LABEL .cycle1
                        CHECK_FOR_DMA
                        mov     dword [EXECUTE_CYCLE], .cycle2
                        ret

CYCLE_LABEL .cycle2
                        mov     byte  al, [PROGRAM_COUNTER + 1]
                        mov     dword ebx, [STACK_POINTER]
                        mov     dword ecx, .cycle3
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle3
                        mov     byte  al, [PROGRAM_COUNTER]
                        mov     dword ebx, [STACK_POINTER]
                        mov     dword ecx, .cycle4
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle4
                        mov     byte  al, [FLAG_REGISTER]
                        mov     dword ebx, [STACK_POINTER]
                        and     byte  al, (NOT8 - FLAG_B)
                        mov     dword ecx, .cycle5
                        CALL_WRITE_STACK
                        dec     byte  bl
                        mov     dword [EXECUTE_CYCLE], ecx
                        mov     dword [STACK_POINTER], ebx
                        ret

CYCLE_LABEL .cycle5
                        mov     dword ebx, 00000fffah
                        CALL_READ_OPCODE
                        mov     dword ecx, .cycle6
                        mov     byte  [PROGRAM_COUNTER], al
                        mov     dword [EXECUTE_CYCLE], ecx
                        ret

CYCLE_LABEL .cycle6
                        mov     dword ebx, 00000fffbh
                        CALL_READ_OPCODE
                        mov     byte  bl, [FLAG_REGISTER]
                        mov     byte  [PROGRAM_COUNTER + 1], al
                        or      byte  bl, FLAG_I
                        mov     dword eax, [STATE]
                        mov     dword [EXECUTE_CYCLE], get_opcode
                        and     dword eax, (NOT32 - STATE_PROCESSING_NMI)
                        mov     byte  [FLAG_REGISTER], bl
                        mov     dword [STATE], eax
                        ret
