/*
    1541EMU, the Commodore 1541 disk drive emulator
    Copyright (C) 2001-2002 Ville Muikkula
    Copyright (C) 1998-2000 Andreas Boose
 *
 * diskimage.c - Common low-level disk image access.
 *
 * This file was formerly part of VICE, the Versatile Commodore Emulator.
 * Written by
 *  Andreas Boose <boose@linux.rz.fh-hannover.de>
 * Modifications by
 *  Ville Muikkula <1541@surfeu.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include <stdlib.h>
#include <io.h>
#include <string.h>
#include <fnmatch.h>

#include "diskimg.h"
#include "core.h"
#include "gcr.h"
#include "utils.h"
#include "drive.h"
#include "tui.h"

/* Defines for probing the disk image type.  */
#define IS_D64_LEN(x) ((x) == D64_FILE_SIZE_35 || (x) == D64_FILE_SIZE_35E || \
                       (x) == D64_FILE_SIZE_40 || (x) == D64_FILE_SIZE_40E)


/*-----------------------------------------------------------------------*/
/* Disk constants.  */

static unsigned int speed_map_1541[42] = {
                           3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
                           3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 1, 1,
                           1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                           0, 0, 0 };

unsigned int disk_image_speed_map_1541(unsigned int track)
{
    return speed_map_1541[track-1];
}


/*-----------------------------------------------------------------------*/
/* Disk image probing.  */

static int disk_image_check_for_d64(disk_image_t *image)
{
    unsigned int blk = 0;
    size_t len;
    BYTE block[256];

    if (!(IS_D64_LEN(util_file_length(image->fd))))
        return 0;

    image->type = DISK_IMAGE_TYPE_D64;
    image->tracks = NUM_TRACKS_1541;

    rewind(image->fd);

    while ((len = fread(block, 1, 256, image->fd)) == 256) {
        if (++blk > (EXT_BLOCKS_1541 + 3)) {
            tui_error("Disk image too large.");
            break;
        }
    }

    if (blk < NUM_BLOCKS_1541) {
        tui_error("Cannot read block %d.", blk);
        return 0;
    }

    switch (blk) {
      case NUM_BLOCKS_1541:
        image->tracks = NUM_TRACKS_1541;
        break;
      case NUM_BLOCKS_1541 + 2:
        if (len != 171) {
            tui_error("Cannot read block %d.", blk);
            return 0;
        }
        image->error_info = (BYTE *)xmalloc(MAX_BLOCKS_1541);
        memset(image->error_info, 0, MAX_BLOCKS_1541);
        if (fseek(image->fd, 256 * NUM_BLOCKS_1541, SEEK_SET) < 0)
            return 0;
        if (fread(image->error_info, 1, NUM_BLOCKS_1541, image->fd)
            < NUM_BLOCKS_1541)
            return 0;
        image->tracks = NUM_TRACKS_1541;
        break;
      case EXT_BLOCKS_1541:
        image->tracks = EXT_TRACKS_1541;
        break;
      case EXT_BLOCKS_1541 + 3:
        image->error_info = (BYTE *)xmalloc(MAX_BLOCKS_1541);
        memset(image->error_info, 0, MAX_BLOCKS_1541);
        if (fseek(image->fd, 256 * EXT_BLOCKS_1541, SEEK_SET) < 0)
            return 0;
        if (fread(image->error_info, 1, EXT_BLOCKS_1541, image->fd)
            < EXT_BLOCKS_1541)
            return 0;

        image->tracks = EXT_TRACKS_1541;
        break;
      default:
        return 0;
    }
    return 1;
}


/*-----------------------------------------------------------------------*/
/* GCR disk image probing and intial GCR buffer setup.  */

int disk_image_read_gcr_image(disk_image_t *image)
{
    unsigned int track, num_tracks;
    DWORD gcr_track_p[MAX_GCR_TRACKS];
    DWORD gcr_speed_p[MAX_GCR_TRACKS];

    num_tracks = image->tracks;

    fseek(image->fd, 12, SEEK_SET);
    if (read_dword(image->fd, gcr_track_p, num_tracks * 8) < 0) {
        tui_error("Could not read GCR disk image.");
        return -1;
    }

    fseek(image->fd, 12 + num_tracks * 8, SEEK_SET);
    if (read_dword(image->fd, gcr_speed_p, num_tracks * 8) < 0) {
        tui_error("Could not read GCR disk image.");
        return -1;
    }

    memset(image->gcr->data, 0x55, MAX_GCR_TRACKS * NUM_MAX_BYTES_TRACK);

    for (track = 0; track < MAX_TRACKS_1541; track++) {
        BYTE *track_data;// ,*zone_data;

        track_data = image->gcr->data + GCR_OFFSET(2*track+1, 0);
//        track_data = image->gcr->data + (2*track)*NUM_MAX_BYTES_TRACK;
//        zone_data = image->gcr->speed_zone;
//        zone_data = image->gcr->speed_zone + 2*track * NUM_MAX_BYTES_TRACK;
//        memset(zone_data, 0x00, NUM_MAX_BYTES_TRACK / 4);
//        image->gcr->track_size[track] = 6250;

        if (track <= num_tracks && gcr_track_p[track * 2] != 0) {
            BYTE len[2];
            long offset;
            size_t track_len;
            unsigned int zone_len;

            offset = gcr_track_p[track * 2];

            fseek(image->fd, offset, SEEK_SET);
            if (fread((char *)len, 2, 1, image->fd) < 1) {
                tui_error("Could not read GCR disk image.");
                return -1;
            }

            track_len = len[0] + len[1] * 256;

            if (track_len < 5000 || track_len > 7928) {
                tui_error("Track field length %i is not supported.",
                          (int)track_len);
                return -1;
            }

            image->gcr->track_size[track] = track_len;

            fseek(image->fd, offset + 2, SEEK_SET);
            if (fread((char *)track_data, track_len, 1, image->fd) < 1) {
                tui_error("Could not read GCR disk image.");
                return -1;
            }

            zone_len = (track_len + 3) / 4;

            if (gcr_speed_p[track * 2] > 3) {
                    tui_error("Varying speeds within a track are not supported.");
                    return -1;
            }
        }
    }
    return 0;
}

static int disk_image_check_for_gcr(disk_image_t *image)
{
    int trackfield;
    BYTE header[32];

    fseek(image->fd, 0, SEEK_SET);
    if (fread((BYTE *)header, sizeof (header), 1, image->fd) < 1) {
        tui_error("Cannot read image header.");
        return 0;
    }

    if (strncmp("GCR-1541", (char*)header, 8))
        return 0;

    if (header[8] != 0) {
        tui_error("Import GCR: Unknown GCR image version %i.",
                  (int)header[8]);
        return 0;
    }

    if (header[9] < NUM_TRACKS_1541 * 2 || header[9] > MAX_TRACKS_1541 * 2) {
        tui_error("Import GCR: Invalid number of tracks (%i).",
                  (int)header[9]);
        return 0;
    }

    trackfield = header[10] + header[11] * 256;
    if (trackfield != 7928) {
        tui_error("Import GCR: Invalid track field number %i.",
                  trackfield);
        return 0;
    }

    image->type = DISK_IMAGE_TYPE_G64;
    image->tracks = header[9] / 2;

    if (image->gcr != NULL) {
        if (disk_image_read_gcr_image(image) < 0)
            return 0;
    }
    return 1;
}


/*-----------------------------------------------------------------------*/
/* Open a disk image.  */

int disk_image_open(disk_image_t *image)
{
    int attr;

//    if ((fnmatch("*.D64", image->name, FNM_NOCASE) != 0)
//        && (fnmatch("*.G64", image->name, FNM_NOCASE) != 0))
    if (fnmatch("*.D64", image->name, FNM_NOCASE) != 0)
    {
//        tui_error("Only D64 and G64 disk image formats are supported.");
        tui_error("Only the D64 disk image format is supported.");
        return -1;
    }

    if (fnmatch("*.G64", image->name, FNM_NOCASE) == 0)
        image->read_only = 1;

    attr = _chmod(image->name, 0);
    if (image->read_only == 0 && attr!=-1)
        image->read_only = attr & 1;

    if (image->read_only) {
        image->fd = fopen(image->name, MODE_READ);
    } else  {
        image->fd = fopen(image->name, MODE_READ_WRITE);

        /* If we cannot open the image read/write, try to open it read only. */
        if (image->fd == NULL) {
            image->fd = fopen(image->name, MODE_READ);
            image->read_only = 1;
        }
    }

    image->error_info = NULL;

    if (image->fd == NULL) {
        tui_error("Cannot open file %s.", image->name);
        return -1;
    }

    if (util_file_length(image->fd) != D64_FILE_SIZE_35)
        image->read_only = 1;

    if (disk_image_check_for_d64(image))
        return 0;
//    if (disk_image_check_for_gcr(image))
//        return 0;

    fclose(image->fd);
//    tui_error("%s is not a valid D64 or G64 file.", image->name);
    tui_error("%s is not a valid D64 file.", image->name);
    return -1;

}


/*-----------------------------------------------------------------------*/
/* Close a disk image.  */

int disk_image_close(disk_image_t *image)
{
    if (image->fd == NULL)
        return -1;

    fclose(image->fd);

    free(image->name);
    image->name = NULL;
    if (image->error_info != NULL)
    {
        free(image->error_info);
        image->error_info = NULL;
    }
    return 0;
}


/*-----------------------------------------------------------------------*/
/* Create GCR disk image.  */

static int disk_image_create_gcr(disk_image_t *image)
{
    BYTE gcr_header[12], gcr_track[7930], *gcrptr;
    DWORD gcr_track_p[MAX_TRACKS_1541 * 2];
    DWORD gcr_speed_p[MAX_TRACKS_1541 * 2];
    int track, sector;

    strcpy((char *) gcr_header, "GCR-1541");

    gcr_header[8] = 0;
    gcr_header[9] = MAX_TRACKS_1541 * 2;
    gcr_header[10] = 7928 % 256;
    gcr_header[11] = 7928 / 256;

    if (fwrite((char *)gcr_header, sizeof(gcr_header), 1, image->fd) < 1) {
        tui_error("Cannot write GCR header.");
        return -1;
    }

    for (track = 0; track < MAX_TRACKS_1541; track++) {
        gcr_track_p[track * 2] = 12 + MAX_TRACKS_1541 * 16 + track * 7930;
        gcr_track_p[track * 2 + 1] = 0;
        gcr_speed_p[track * 2] = disk_image_speed_map_1541(track);
        gcr_speed_p[track * 2 + 1] = 0;
    }

    if (write_dword(image->fd, gcr_track_p, sizeof(gcr_track_p)) < 0) {
        tui_error("Cannot write track header.");
        return -1;
    }
    if (write_dword(image->fd, gcr_speed_p, sizeof(gcr_speed_p)) < 0) {
        tui_error("Cannot write speed header.");
        return -1;
    }
    for (track = 0; track < MAX_TRACKS_1541; track++) {
        int raw_track_size[4] = { 6250, 6667, 7143, 7692 };

        memset(&gcr_track[2], 0xff, 7928);
        gcr_track[0] = raw_track_size[disk_image_speed_map_1541(track)] % 256;
        gcr_track[1] = raw_track_size[disk_image_speed_map_1541(track)] / 256;
        gcrptr = &gcr_track[2];

        for (sector = 0;
        sector < disk_image_sector_per_track(DISK_IMAGE_TYPE_D64, track + 1);
        sector++)
        {
            BYTE chksum, rawdata[260];
            int i;

            memset(rawdata, 0, 260);
            rawdata[0] = 7;
            if ((track+1==BAM_TRACK_1541) && (sector<2))
            {
                memcpy(&rawdata[1], &bamdir[sector], 256);
            }
            chksum = rawdata[1];
            for (i = 1; i < 256; i++)
                chksum ^= rawdata[i + 1];
            rawdata[257] = chksum;

            gcr_convert_sector_to_GCR(rawdata, gcrptr, track + 1, sector,
                                      bamdir[ID_1541], bamdir[ID_1541+1], 0);
            gcrptr += 360;
        }
        if (fwrite((char *)gcr_track, sizeof(gcr_track), 1, image->fd) < 1 )
        {
            tui_error("Cannot write track data.");
            return -1;
        }
    }
    return 0;
}


/*-----------------------------------------------------------------------*/
/* Create a disk image.  */

int disk_image_create(const char *name, const char *diskname,
                      const char *diskid, unsigned int type)
{
    disk_image_t *image;
    long size = 0;
    char block[256];
    int i;

    switch(type)
    {
      case DISK_IMAGE_TYPE_D64:
        size = D64_FILE_SIZE_35;
        break;
      case DISK_IMAGE_TYPE_G64:
        break;
      default:
        tui_error("Wrong image type.  Cannot create disk image.");
        return -1;
    }

    image = (disk_image_t *)xmalloc(sizeof(disk_image_t));
    image->name = stralloc(name);

    image->fd = fopen(image->name, MODE_WRITE);

    if (image->fd == NULL)
    {
        tui_error("Cannot create disk image `%s'.", image->name);
        free(image->name);
        image->name = NULL;
        return -1;
    }

    memset(block, 0, 256);

    for (i=0; i<16; i++)
        bamdir[NAME_1541+i] = 0xA0;

    for (i=0; i<2; i++)
        bamdir[ID_1541+i] = 0xA0;

    for (i=0; i<strlen(diskname); i++)
        bamdir[NAME_1541+i] = diskname[i];

    for (i=0; i<strlen(diskid); i++)
        bamdir[ID_1541+i] = diskid[i];

    switch (type)
    {
      case DISK_IMAGE_TYPE_D64:
        for (i = 0; i < (size / 256); i++)
        {
            if (fwrite(block, 256, 1, image->fd) < 1)
            {
                tui_error("Cannot seek to end of disk image `%s'.",image->name);
                fclose(image->fd);
                free(image->name);
                image->name = NULL;
                free(image);
                return -1;
            }
        }

        image->type=type;
        image->read_only=0;
        disk_image_write_sector(image, &bamdir[0], BAM_TRACK_1541, 0);
        disk_image_write_sector(image, &bamdir[256], BAM_TRACK_1541, 1);
        break;
        
      case DISK_IMAGE_TYPE_G64:
        if (disk_image_create_gcr(image) < 0) {
            fclose(image->fd);
            free(image->name);
            image->name = NULL;
            free(image);
            return -1;
        }
        break;

    }

    fclose(image->fd);
    free(image->name);
    image->name = NULL;
    free(image);
    return 0;
}

/*-----------------------------------------------------------------------*/
/* Check for track out of bounds.  */

static char sector_map_d64[43] =
{
    0,
    21, 21, 21, 21, 21, 21, 21, 21, 21, 21, /*  1 - 10 */
    21, 21, 21, 21, 21, 21, 21, 19, 19, 19, /* 11 - 20 */
    19, 19, 19, 19, 18, 18, 18, 18, 18, 18, /* 21 - 30 */
    17, 17, 17, 17, 17,                     /* 31 - 35 */
    17, 17, 17, 17, 17, 17, 17              /* 36 - 42 */
};

int disk_image_sector_per_track(unsigned int format, unsigned int track)
{
    switch (format) {
      case DISK_IMAGE_TYPE_D64:
        if (track >= sizeof(sector_map_d64)) {
            tui_error("Track %i exceeds sector map.", track);
            return -1;
        }
        return sector_map_d64[track];
        break;
      default:
        tui_error("Unknown disk type %i.  Cannot calculate sectors per track.",
               format);
    }
    return -1;
}

int disk_image_check_sector(unsigned int format, unsigned int track,
                            unsigned int sector)
{
    unsigned int sectors = 0, i;

    if (track < 1 || sector < 0)
        return -1;

    switch (format) {
      case DISK_IMAGE_TYPE_D64:
        if (track > MAX_TRACKS_1541 || sector
            >= disk_image_sector_per_track(DISK_IMAGE_TYPE_D64, track))
            return -1;
        for (i = 1; i < track; i++)
            sectors += disk_image_sector_per_track(DISK_IMAGE_TYPE_D64, i);
        sectors += sector;
        break;
      default:
        return -1;
    }
    return (int)(sectors);
}


/*-----------------------------------------------------------------------*/
/* Read an entire GCR track from the disk image.  */

int disk_image_read_track(disk_image_t *image, unsigned int track,
                          BYTE *gcr_data, int *gcr_track_size)
{
    int track_len;
    BYTE len[2];
    DWORD gcr_track_p;
    long offset;

    if (image->fd == NULL) {
        tui_error("Attempt to read without disk image.");
        return -1;
    }

    fseek(image->fd, 12 + (track - 1) * 8, SEEK_SET);
    if (read_dword(image->fd, &gcr_track_p, 4) < 0) {
        tui_error("Could not read GCR disk image.");
        return -1;
    }

    memset(gcr_data, 0xff, NUM_MAX_BYTES_TRACK);
    *gcr_track_size = 6250;

    if (gcr_track_p != 0) {

        offset = gcr_track_p;

        fseek(image->fd, offset, SEEK_SET);
        if (fread((char *)len, 2, 1, image->fd) < 1) {
            tui_error("Could not read GCR disk image.");
            return -1;
        }

        track_len = len[0] + len[1] * 256;

        if (track_len < 5000 || track_len > 7928) {
            tui_error("Track field length %i is not supported.",
                    track_len);
            return -1;
        }

        *gcr_track_size = track_len;

        fseek(image->fd, offset + 2, SEEK_SET);
        if (fread((char *)gcr_data, track_len, 1, image->fd) < 1) {
            tui_error("Could not read GCR disk image.");
            return -1;
        }
    }
    return 0;
}


/*-----------------------------------------------------------------------*/
/* Read an sector from the disk image.  */

int disk_image_read_sector(disk_image_t *image, BYTE *buf, unsigned int track,
                           unsigned int sector)
{
    int sectors;
    long offset;

    if (image->fd == NULL) {
        tui_error("Attempt to read without disk image.");
        return 74;
    }

    switch (image->type) {
      case DISK_IMAGE_TYPE_D64:
        sectors = disk_image_check_sector(image->type, track, sector);

        if (sectors < 0) {
            tui_error("Track %i, Sector %i out of bounds.", track, sector);
            return 66;
        }

        offset = sectors << 8;

        fseek(image->fd, offset, SEEK_SET);

        if (fread((char *)buf, 256, 1, image->fd) < 1) {
            tui_error("Error reading T:%i S:%i from disk image.", track, sector);
            return -1;
        }

        if (image->error_info != NULL) {
            switch (image->error_info[sectors]) {
              case 0x0:
              case 0x1:
                return 0;
              case 0x2:
                return 20;
              case 0x3:
                return 21;
              case 0x4:
                return 22;
              case 0x5:
                return 23;
              case 0x7:
                return 25;
              case 0x8:
                return 26;
              case 0x9:
                return 27;
              case 0xA:
                return 28;
              case 0xB:
                return 29;
              case 0xF:
                return 74;
              case 0x10:
                return 24;
              default:
                return 0;
            }
        }
        break;
      case DISK_IMAGE_TYPE_G64:
        {
            BYTE gcr_data[NUM_MAX_BYTES_TRACK], *gcr_track_start_ptr;
            int gcr_track_size, gcr_current_track_size;

            if (track > image->tracks) {
                tui_error("Track %i out of bounds.  Cannot read GCR track.",
                    track);
                return -1;
            }
            if (image->gcr == NULL) {
                if (disk_image_read_track(image, track, gcr_data,
                    &gcr_track_size) < 0) {
                    tui_error("Cannot read track %i from GCR image.", track);
                    return -1;
                }
                gcr_track_start_ptr = gcr_data;
                gcr_current_track_size = gcr_track_size;
            } else {
                gcr_track_start_ptr = image->gcr->data
                                      + ((track - 1) * NUM_MAX_BYTES_TRACK);
                gcr_current_track_size = image->gcr->track_size[track - 1];
            }
            if (gcr_read_sector(gcr_track_start_ptr, gcr_current_track_size,
                buf, track, sector) < 0) {
                tui_error("Cannot find track: %i sector: %i within GCR image.",
                          track, sector);
                return -1;
            }
        }
        break;
      default:
        tui_error("Unknown disk image type %i.  Cannot read sector.",
                  image->type);
        return -1;
    }
    return 0;
}


/*-----------------------------------------------------------------------*/
/* Write an entire GCR track to the disk image.  */

int disk_image_write_track(disk_image_t *image, unsigned int track,
                           int gcr_track_size, BYTE *gcr_speed_zone,
                           BYTE *gcr_track_start_ptr)
{
    int gap, i;
    unsigned int num_tracks;
    BYTE len[2];
    DWORD gcr_track_p[MAX_TRACKS_1541 * 2];
    DWORD gcr_speed_p[MAX_TRACKS_1541 * 2];
    int offset;

    if (image->fd == NULL) {
        tui_error("Attempt to write without disk image.");
        return -1;
    }

    if (image->read_only != 0) {
        tui_error("Attempt to write to read-only disk image.");
        return -1;
    }

    num_tracks = image->tracks;

    fseek(image->fd, 12, SEEK_SET);
    if (read_dword(image->fd, gcr_track_p, num_tracks * 8) < 0) {
        tui_error("Could not read GCR disk image header.");
        return -1;
    }

    fseek(image->fd, 12 + num_tracks * 8, SEEK_SET);
    if (read_dword(image->fd, gcr_speed_p, num_tracks * 8) < 0) {
        tui_error("Could not read GCR disk image header.");
        return -1;
    }

    if (gcr_track_p[(track - 1) * 2] == 0) {
        offset = fseek(image->fd, 0, SEEK_END);
        if (offset < 0) {
            tui_error("Could not extend GCR disk image.");
            return -1;
        }
        gcr_track_p[(track - 1) * 2] = offset;
    }

    offset = gcr_track_p[(track - 1) * 2];

    len[0] = gcr_track_size % 256;
    len[1] = gcr_track_size / 256;

    if (fseek(image->fd, offset, SEEK_SET) < 0
        || fwrite((char *)len, 2, 1, image->fd) < 1) {
        tui_error("Could not write GCR disk image.");
        return -1;
    }

    /* Clear gap between the end of the actual track and the start of
       the next track.  */
    gap = NUM_MAX_BYTES_TRACK - gcr_track_size;
    if (gap > 0)
        memset(gcr_track_start_ptr + gcr_track_size, 0, gap);

    if (fseek(image->fd, offset + 2, SEEK_SET) < 0
        || fwrite((char *)gcr_track_start_ptr, NUM_MAX_BYTES_TRACK,
        1, image->fd) < 1) {
        tui_error("Could not write GCR disk image.");
        return -1;
    }

    if (gcr_speed_zone != NULL) {
        for (i = 0; (gcr_speed_zone[(track - 1) * NUM_MAX_BYTES_TRACK]
            == gcr_speed_zone[(track - 1) * NUM_MAX_BYTES_TRACK + i])
            && i < NUM_MAX_BYTES_TRACK; i++);

        if (i < gcr_track_size) {
            /* This will change soon.  */
            tui_error("Saving different speed zones is not supported yet.");
            return -1;
        }

        if (gcr_speed_p[(track - 1) * 2] >= 4) {
            /* This will change soon.  */
            tui_error("Adding new speed zones is not supported yet.");
            return -1;
        }

        offset = 12 + num_tracks * 8 + (track - 1) * 8;
        if (fseek(image->fd, offset, SEEK_SET) < 0
            || write_dword(image->fd, &gcr_speed_p[(track - 1) * 2], 4) < 0) {
            tui_error("Could not write GCR disk image.");
            return -1;
        }
    }

#if 0  /* We do not support writing different speeds yet.  */
    for (i = 0; i < (NUM_MAX_BYTES_TRACK / 4); i++)
        zone_len = (gcr_track_size + 3) / 4;
    zone_data = gcr_speed_zone + (track - 1) * NUM_MAX_BYTES_TRACK;

    if (gap > 0)
        memset(zone_data + gcr_track_size, 0, gap);

    for (i = 0; i < (NUM_MAX_BYTES_TRACK / 4); i++)
        comp_speed[i] = (zone_data[i * 4]
                         | (zone_data[i * 4 + 1] << 2)
                         | (zone_data[i * 4 + 2] << 4)
                         | (zone_data[i * 4 + 3] << 6));

    if (fseek(image->fd, offset, SEEK_SET) < 0
        || fwrite((char *)comp_speed, NUM_MAX_BYTES_TRACK / 4, 1
        image->fd) < 1) {
        tui_error("Could not write GCR disk image.");
        return;
    }
#endif
    return 0;
}


/*-----------------------------------------------------------------------*/
/* Write a sector to the disk image.  */

int disk_image_write_sector(disk_image_t *image, BYTE *buf, unsigned int track,
                            unsigned int sector)
{
    int sectors;
    long offset;

    if (image->fd == NULL) {
        tui_error("Attempt to write without disk image.");
        return -1;
    }

    if (image->read_only != 0) {
        tui_error("Attempt to write to read-only disk image.");
        return -1;
    }

    sectors = disk_image_check_sector(image->type, track, sector);

    switch (image->type) {
      case DISK_IMAGE_TYPE_D64:
        if (sectors < 0) {
            tui_error("Track: %i, Sector: %i out of bounds.",
                   track, sector);
            return -1;
        }
        offset = sectors << 8;

        fseek(image->fd, offset, SEEK_SET);

        if (fwrite((char *)buf, 256, 1, image->fd) < 1) {
            tui_error("Error writing T:%i S:%i to disk image.",
                   track, sector);
            return -1;
        }
        break;
      case DISK_IMAGE_TYPE_G64:
        {
            BYTE gcr_data[NUM_MAX_BYTES_TRACK];
            BYTE *gcr_track_start_ptr, *speed_zone;
            int gcr_track_size, gcr_current_track_size;

            if (track > image->tracks) {
                tui_error("Track %i out of bounds.  Cannot write GCR sector.",
                       track);
                return -1;
            }
            if (image->gcr == NULL) {
                if (disk_image_read_track(image, track, gcr_data,
                    &gcr_track_size) < 0) {
                    tui_error("Cannot read track %i from GCR image.", track);
                    return -1;
                }
                gcr_track_start_ptr = gcr_data;
                gcr_current_track_size = gcr_track_size;
                speed_zone = NULL;
            } else {
                gcr_track_start_ptr = image->gcr->data
                                      + ((track - 1) * NUM_MAX_BYTES_TRACK);
                gcr_current_track_size = image->gcr->track_size[track - 1];
                speed_zone = image->gcr->speed_zone;
            }
            if (gcr_write_sector(gcr_track_start_ptr,
                gcr_current_track_size, buf, track, sector) < 0) {
                tui_error("Could not find track %i sector %i in disk image.",
                       track, sector);
                return -1;
            }
            if (disk_image_write_track(image, track, gcr_current_track_size,
                speed_zone, gcr_track_start_ptr) < 0) {
                tui_error("Failed writing track %i to disk image.", track);
                return -1;
            }
        }
        break;
      default:
        tui_error("Unknown disk image.  Cannot write sector.");
        return -1;
    }
    return 0;
}

