/*
    1541EMU, the Commodore 1541 disk drive emulator
    Copyright (C) 2001-2002 Ville Muikkula
    Copyright (C) 1998-2000 Andreas Boose
    Copyright (C) 1996-1999 Ettore Perazzoli
    Copyright (C) 1997-2000 Daniel Sladic
    Copyright (C) 1996-1999 Andr Fachat
    Copyright (C) 1993-1994, 1997-1999 Teemu Rantanen
 *
 * drive.c - Hardware-level Commodore disk drive emulation.
 *
 * This file was formerly part of VICE, the Versatile Commodore Emulator.
 * Written by
 *  Andreas Boose <boose@linux.rz.fh-hannover.de>
 * Based on old code by
 *  Daniel Sladic <sladic@eecg.toronto.edu>
 *  Ettore Perazzoli <ettore@comm2000.it>
 *  Andr Fachat <fachat@physik.tu-chemnitz.de>
 *  Teemu Rantanen <tvr@cs.hut.fi>
 * Modifications by
 *  Ville Muikkula <1541@surfeu.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.

 * TODO:
        - more accurate emulation of disk rotation.
        - different speeds within one track.
*/

#include <string.h>

#include "diskimg.h"
#include "drive.h"
#include "gcr.h"
#include "md5sum.h"
#include "tui.h"

/* ------------------------------------------------------------------------- */

disk_image_t diskette[MAX_DISKS];

/* Disk ID.  */
BYTE diskID1, diskID2;

/* Prototypes of functions called by resource management.  */
static int drive_check_image_format(unsigned int format);

/* If nonzero, at least one vaild drive ROM has already been loaded.  */
int rom_loaded = 0;

/* If nonzero, we are far enough in init that we can load ROMs */
static int drive_rom_load_ok = 0;

/* ------------------------------------------------------------------------- */

/* RAM/ROM.  */
BYTE drive_rom1541[DRIVE_ROM1541_SIZE];

/* If nonzero, the ROM image has been loaded.  */
unsigned int rom1541_loaded = 0;

unsigned int drive_rom1541_size;

/* Number of bytes per track size.  */
static unsigned int raw_track_size[4] = { 6250, 6667, 7143, 7692 };

//static void drive_extend_disk_image(unsigned int dnr);
static int drive_load_rom_images(void);

/* ------------------------------------------------------------------------- */

/* Disk image handling. */

static void drive_read_image_d64_d71(disk_image_t *image)
{
    BYTE buffer[260], chksum;
    int i;
    unsigned int track, sector;

    buffer[258] = buffer[259] = 0;

    /* Since the D64/D71 format does not provide the actual track sizes or
       speed zones, we set them to standard values.  */

    for (track = 0; track < MAX_TRACKS_1541; track++)
    {
        image->gcr->track_size[track<<1] =
            raw_track_size[disk_image_speed_map_1541(track+1)];
        image->gcr->track_size[(track<<1)+1] =
            raw_track_size[disk_image_speed_map_1541(track+1)];
//        memset(drive.gcr->speed_zone, disk_image_speed_map_1541(track),
//            NUM_MAX_BYTES_TRACK);
    }

    memset(image->gcr->data, 0x55, MAX_GCR_TRACKS * NUM_MAX_BYTES_TRACK);
    for (track = 1; track <= image->tracks; track++) {
        BYTE *ptr;
        unsigned int max_sector = 0;

        ptr = image->gcr->data + GCR_OFFSET(2*track-1, 0);
        max_sector = disk_image_sector_per_track(image->type, track);

        for (sector = 0; sector < max_sector; sector++) {
            int rc;
            ptr = image->gcr->data + GCR_OFFSET(2*track-1, sector);

            rc = disk_image_read_sector(image, buffer + 1, track, sector);
            if (rc < 0) {
                tui_error("Cannot read T:%d S:%d from disk image.",
                          track, sector);
                          continue;
            }

            if (rc == 21) {
                ptr = image->gcr->data + GCR_OFFSET(2*track-1, 0);
                memset(ptr, 0x00, NUM_MAX_BYTES_TRACK);
                break;
            }

            buffer[0] = (rc == 22) ? 0xff : 0x07;

            chksum = buffer[1];
            for (i = 2; i < 257; i++)
                chksum ^= buffer[i];
            buffer[257] = (rc == 23) ? chksum ^ 0xff : chksum;
            gcr_convert_sector_to_GCR(buffer, ptr, track, sector,
                                      diskID1, diskID2, (BYTE)(rc));
        }
    }
}

static int setID(disk_image_t *image)
{
    BYTE buffer[256];
    int rc;

    rc = disk_image_read_sector(image, buffer, 18, 0);
    if (rc >= 0) {
        diskID1 = buffer[0xa2];
        diskID2 = buffer[0xa3];
    }

    return rc;
}

/* ------------------------------------------------------------------------- */

/* Initialize the hardware-level drive emulation (should be called at least
   once before anything else).  Return 0 on success, -1 on error.  */
int drive_init(void)
{
    if (rom_loaded)
        return 0;

    if (drive_load_rom_images() < 0) {
        return -1;
    }

    rom_loaded = 1;
    return 0;
}


int drive_load_1541(void)
{
    FILE *fp = NULL;
    char filename[256] = "";
    int rsize;

    unsigned char md5buffer[16];
    int fail;
    char buffer[3];
    char md5sum_text[33] = "";

    if (!drive_rom_load_ok)
        return 0;

    strcpy(filename, dir);
    strcat(filename, drive_rom_name);

    fail = md5_file (filename, 1, md5buffer);
    if (!fail)
    {
            size_t i;

            for (i=0; i<16; ++i)
            {
                    sprintf(buffer, "%02x", md5buffer[i]);
                    strncat (md5sum_text, buffer, 2);
            }

            printf("Drive ROM type is ");
            if (strncmp(md5sum_text, DRIVE_ROM1540_MD5SUM, 32)==0)
                    printf("1540.\n");
            else if (strncmp(md5sum_text, DRIVE_ROM1541_R1_MD5SUM, 32)==0)
                    printf("1541 r1.\n");
            else if (strncmp(md5sum_text, DRIVE_ROM1541_R2_MD5SUM, 32)==0)
                    printf("1541 r2.\n");
            else if (strncmp(md5sum_text, DRIVE_ROM1541_R3_MD5SUM, 32)==0)
                    printf("1541 r3.\n");
            else if (strncmp(md5sum_text, DRIVE_ROM1541_R5_MD5SUM, 32)==0)
                    printf("1541 r5.\n");
            else if (strncmp(md5sum_text, DRIVE_ROM1541C_R1_MD5SUM, 32)==0)
                    printf("1541C r1.\n");
            else if (strncmp(md5sum_text, DRIVE_ROM1541C_R2_MD5SUM, 32)==0)
                    printf("1541C r2.\n");
            else if (strncmp(md5sum_text, DRIVE_ROM1541_II_MD5SUM, 32)==0)
                    printf("1541-II.\n");
            else if (strncmp(md5sum_text, DRIVE_ROMJIFFYDOS50_MD5SUM, 32)==0)
                    printf("JiffyDOS 5.0.\n");
            else if (strncmp(md5sum_text, DRIVE_ROMJIFFYDOS50R_MD5SUM, 32)==0)
                    printf("JiffyDOS 5.0.\n");
            else
            {
                    printf("unknown!\n");
                    fail = 1;
            }
    }

    if (!fail)
    {
            /* Load the ROMs. */
            fp = fopen(filename, "rb");
            if (fp != NULL)
            {
                    rsize = fread(drive_rom1541, 1, DRIVE_ROM1541_SIZE, fp);
                    fclose(fp);
            }
            else
                    rsize = -1;

            if (rsize != DRIVE_ROM1541_SIZE)
            {
                    tui_error("1541 ROM image file %s not found.", filename);
                    drive_rom1541_size = 0;
                    return -1;
            }
            else
            {
                    rom1541_loaded = 1;
                    drive_rom1541_size = DRIVE_ROM1541_SIZE;
                    return 0;
            }
    }
    return -1;
}

static int drive_load_rom_images(void)
{
    drive_rom_load_ok = 1;

    drive_load_1541();

    if (!rom1541_loaded)
        return -1;

    return 0;
}


/* ------------------------------------------------------------------------- */
/* Check if the drive type matches the disk image type.  */
static int drive_check_image_format(unsigned int format)
{
    switch (format) {
      case DISK_IMAGE_TYPE_D64:
      case DISK_IMAGE_TYPE_G64:
        break;
      default:
        return -1;
    }
    return 0;
}

/* Attach a disk image to the true drive emulation. */
int drive_attach_image(disk_image_t *image)
{
    if (drive_check_image_format(image->type) < 0)
        return -1;

    if (image->type == DISK_IMAGE_TYPE_G64) {
        if (disk_image_read_gcr_image(image) < 0) {
            return -1;
        }
    } else {
        if (setID(image) >= 0) {
            drive_read_image_d64_d71(image);
            return 0;
        } else {
            return -1;
        }
    }
//    drive.GCR_image_loaded = 1;

    return 0;
}

/* Detach a disk image from the true drive emulation. */
int drive_detach_image(disk_image_t *image)
{
    if ((image->name != NULL) && (image->read_only != 1))
        drive_gcr_data_writeback(image);

    return 0;
}

/* ------------------------------------------------------------------------- */

/* Hack... otherwise you get internal compiler errors when optimizing on
    gcc2.7.2 on RISC OS */
static void gcr_data_writeback2(BYTE *buffer, BYTE *offset,
                                disk_image_t *image, unsigned int track,
                                unsigned int sector)
{
    int rc;

    gcr_convert_GCR_to_sector(buffer, offset,
                         (BYTE *)(image->gcr->data+GCR_OFFSET(2*track-1, 0)),
                         raw_track_size[disk_image_speed_map_1541(track)]);
    if (buffer[0] != 0x7) {
        tui_error("Could not find data block id of T:%d S:%d.", track, sector);
    } else {
        rc = disk_image_write_sector(image, buffer + 1, track, sector);
        if (rc < 0)
            tui_error("Could not update T:%d S:%d.", track, sector);
    }
}

void drive_gcr_data_writeback(disk_image_t *image)
{
    unsigned int track, sector, max_sector = 0;
    BYTE buffer[260], *offset;

    if (image->GCR_dirty != 1)
        return;

    if (!tui_ask_confirmation("Update %s? (Y/N)", image->name))
        return;

    if (image->type == DISK_IMAGE_TYPE_G64) {
        BYTE *gcr_track_start_ptr;
        unsigned int gcr_current_track_size;

        gcr_current_track_size = image->gcr->track_size[track - 1];

        gcr_track_start_ptr = image->gcr->data
                              + ((track - 1) * NUM_MAX_BYTES_TRACK);

        disk_image_write_track(image, track,
                               gcr_current_track_size,
                               image->gcr->speed_zone,
                               gcr_track_start_ptr);
        return;
    }

    if (image->type == DISK_IMAGE_TYPE_D64) {
    }

    for (track = 1; track < NUM_TRACKS_1541+1; track++)
    {
        max_sector = disk_image_sector_per_track(DISK_IMAGE_TYPE_D64, track);
        for (sector = 0; sector < max_sector; sector++)
        {
            offset = gcr_find_sector_header(track, sector,
                         (BYTE *)(image->gcr->data+GCR_OFFSET(2*track-1, 0)),
                         raw_track_size[disk_image_speed_map_1541(track)]);

            if (offset == NULL)
            {
                tui_error("Could not find header of T:%d S:%d.",track, sector);
            }
            else
            {
                offset = gcr_find_sector_data(offset,
                         (BYTE *)(image->gcr->data+GCR_OFFSET(2*track-1, 0)),
                         raw_track_size[disk_image_speed_map_1541(track)]);

                if (offset == NULL)
                {
                    tui_error("Could not find data sync of T:%d S:%d.",
                        track, sector);
                }
                else
                {
                    gcr_data_writeback2(buffer, offset, image, track, sector);
                }
            }
        }
    }
}
