/*
    1541EMU, the Commodore 1541 disk drive emulator
    Copyright (C) 2001-2002 Ville Muikkula <1541@surfeu.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as
    published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#define MAIN_C
#include "main.h"
#include "drive.h"
#include "keys.h"

int is_valid_filename_character(int c)
{
    if (((c >= '!') && (c <= ')') && (c != '"'))
        || (c == '-')
        || ((c >= '0') && (c <= '9'))
        || ((c >= '@') && (c <= 'Z'))
        || ((c >= '^') && (c <= '~') && (c != '|')))
        return 1;
    else
        return 0;
}

int is_valid_diskname_character(int c)
{
    if (((c>=' ') && (c<=']') && (c!='\\')) || ((c>='a') && (c<='z')))
        return 1;
    else
        return 0;
}


int new_diskimage_dialog(unsigned char *filename, unsigned char *diskname,
                         unsigned char *diskid)
{
    const unsigned char x = 29;
    const unsigned char y = 6;
    const unsigned char field_x[5] = { x+4, x+4, x+4, x+6, x+13 };
    const unsigned char field_y[5] = { y+3, y+6, y+9, y+11, y+11 };
    char field_data[5][17];
    unsigned char field_id = 0;
    int exit_dialog = 0;
    unsigned char temp_filename[13]="";
    struct find_t find;
    int key,i;
    unsigned char saved_screen[4000];

    for (i=0; i<5; i++)
        field_data[i][0] = '\0';

    ScreenRetrieve(saved_screen);
    textattr(LIGHTGRAY);
    ScreenClear();

    gotoxy(x,y);
    printf("Create a disk imageͻ");
    gotoxy(x,y+13);
    printf("ͼ");
    for (i=0; i<12; i++)
    {
        gotoxy(x,y+i+1);
        printf("");
        gotoxy(x+22,y+i+1);
        printf("");
    }

    gotoxy(field_x[field_id], field_y[field_id]-1);
    printf("Filename:");
    gotoxy(field_x[field_id]+8, wherey()+1);
    printf(".D64");
    field_id=1;
    gotoxy(field_x[field_id], field_y[field_id]-1);
    printf("Disk name:");
    field_id=2;
    gotoxy(field_x[field_id], field_y[field_id]-1);
    printf("Disk ID:");
    field_id=3;
    gotoxy(field_x[field_id], field_y[field_id]);
    printf("OK");
    field_id=4;
    gotoxy(field_x[field_id], field_y[field_id]);
    printf("Cancel");

    field_id=0;

    textattr(LIGHTGRAY<<4);
    gotoxy(x+4, y+3);
    cputs("        ");
    gotoxy(x+4, y+6);
    cputs("                ");
    gotoxy(x+4, y+9);
    cputs("  ");
    gotoxy(x+4, y+3);

    do
    {
        key = getkey();
        switch (key)
        {
            case K_Escape:
                exit_dialog=1;
                break;

            case K_Return:
                if (field_id==3)
                {
                    strncpy(temp_filename, field_data[0],9);
                    strcat(temp_filename,".D64");
                    if ((strlen(field_data[0])!=0)
                        && (_dos_findfirst(temp_filename,
                        (_A_NORMAL|_A_RDONLY|_A_HIDDEN|_A_SYSTEM
                        |_A_SUBDIR|_A_ARCH), &find)!=0))
                    {
                        strncpy(filename, field_data[0], 9);
                        strncpy(diskname, field_data[1], 17);
                        strncpy(diskid, field_data[2], 3);
                        exit_dialog=1;
                    }
                    else
                    {
                        if (strlen(field_data[0])!=0)
                            tui_error("That filename is already in use.");
                        field_id=0;
                        field_data[field_id][0]='\0';
                        gotoxy(field_x[field_id], field_y[field_id]);
                        printf("        ");
                        gotoxy(field_x[field_id], field_y[field_id]);
                        
                    }
                    break;
                }
                else if (field_id==4)
                {
                    exit_dialog=1;
                    break;
                }
                // Notice: No break instruction here! Enter behaves like Tab
                // when the cursor is not on the "OK" or "Cancel" button.

            case K_Tab:
                field_id=((field_id+1) % 5);
                gotoxy(field_x[field_id]+strlen(field_data[field_id]),
                       field_y[field_id]);
                break;

            case K_BackTab:
                if (field_id==0)
                    field_id=5;
                field_id=((field_id-1) % 5);
                gotoxy(field_x[field_id]+strlen(field_data[field_id]),
                       field_y[field_id]);
                break;

            case K_BackSpace:
                switch(field_id)
                {
                    case 0:
                    case 1:
                    case 2:
                        if (strlen(field_data[field_id])!=0)
                        {
                            field_data[field_id]
                                [strlen(field_data[field_id])-1] = '\0';
                            gotoxy(wherex()-1,wherey());
                            printf(" ");
                            gotoxy(wherex()-1,wherey());
                        }
                        break;
                }
                break;

            default:

                if (((field_id==0)
                    && is_valid_filename_character(key)
                    && (strlen(field_data[field_id])<8))

                    || ((field_id==1)
                    && is_valid_diskname_character(key)
                    && (strlen(field_data[field_id])<16))

                    || ((field_id==2)
                    && is_valid_diskname_character(key)
                    && (strlen(field_data[field_id])<2)))
                {
                    key=toupper(key);
                    printf("%c",key);
                    field_data[field_id]
                        [strlen(field_data[field_id])+1] = '\0';
                    field_data[field_id]
                        [strlen(field_data[field_id])] = (char) key;
                }
                break;
        }        
    } while (!exit_dialog);

    textattr(LIGHTGRAY);
    ScreenUpdate(saved_screen);
    gotoxy(1,25);

    if (key==K_Return)
        return 1;
    else
        return 0;
}


void install_ISR(int num, void *ISR, __dpmi_paddr *old_vector)
{
    __dpmi_paddr new_vector;

    __dpmi_get_protected_mode_interrupt_vector(num, old_vector);
    new_vector.offset32 = (unsigned long)ISR;
    new_vector.selector = _my_cs();
    __dpmi_set_protected_mode_interrupt_vector(num, &new_vector);
}

void remove_ISR(int num, __dpmi_paddr *old_vector)
{
    __dpmi_set_protected_mode_interrupt_vector(num, old_vector);
}


int Read_Configuration(void)
{
    int error = 0;

    CPU_MHz = Detect_CPU_Speed();

    Device_Number=GetPrivateProfileInt("1541EMU","Device_Number",8,filename);
    Global_Read_Only_Access=GetPrivateProfileInt("1541EMU",
        "Global_Read_Only_Access", 1, filename);

    IO_tuning=GetPrivateProfileInt("1541EMU", "IO_tuning", 0, filename);
    GetPrivateProfileString("1541EMU", "drive_rom_name", "1541V5.ROM",
        drive_rom_name, 13, filename);

    if (!Detect_Cable(&LPT_IO, &Cable_Type))
    {
        tui_error("Could not detect the cable on any LPT ports.");
        error = 1;
    }
    else if (!Detect_LPT_IRQ(&LPT_IRQ, &LPT_IRQ_Level_Triggered,
                             &LPT_IRQ_Active_Edge))
    {
        tui_error("Could not detect the LPT port IRQ.");
        error = 1;
    }
    else if (Device_Number<8||Device_Number>11)
    {
        tui_error("Invalid disk drive device number configuration!");
        error = 1;
    }
    else if (Global_Read_Only_Access>1)
    {
        tui_error("Invalid global disk image read only access configuration!");
        error = 1;
    }

    if (!error)
    {
        printf("Device number of the emulated Commodore disk drive is %d.\n",
            Device_Number);
            
        if (Global_Read_Only_Access)
            printf("Disk images are globally write-protected.\n");

        printf("Detected PC CPU speed is %u MHz.\n", CPU_MHz);
        printf("Detected type %d cable (0x%X, IRQ %d, ", Cable_Type, LPT_IO,
               LPT_IRQ);
        if (LPT_IRQ_Level_Triggered)
            printf("level, ");
        else
        {
            printf(LPT_IRQ_Active_Edge ? "rising" : "falling");
            printf(" edge, ");
        }
        printf("%f us).\n", (double) detect_IO_delay()/(65536*CPU_MHz));
    }
    return error;
}


int Load_Disk_Images(int ac, char *av[])
{
    int err = 0;
    int i;

    for (i=0; i<MAX_DISKS; i++)
    {
        diskette[i].name = NULL;
        diskette[i].gcr = NULL;
    }

    if (ac>1)
    {
        for (i=0; i<ac-1; i++)
        if (attach_disk_image(&diskette[i], av[i+1]) != 0) err |= 1;
    }
    return err;
}

/* This function reads date and time from the real-time clock
   and updates the DOS date and time accordingly. */
void Set_DOS_Time(void)
{
    union REGS r;
    struct date d;
    struct time t;

    r.h.ah = 0x04;          // Read Date from Real Time Clock
    int86(0x1A, &r, &r);

    d.da_year = BCDtoBIN(r.h.ch) * 100 + BCDtoBIN(r.h.cl);
    d.da_mon = BCDtoBIN(r.h.dh);
    d.da_day = BCDtoBIN(r.h.dl);

    r.h.ah = 0x02;          // Read Time from Real Time Clock
    int86(0x1A, &r, &r);

    t.ti_hour = BCDtoBIN(r.h.ch);
    t.ti_min = BCDtoBIN(r.h.cl);
    t.ti_sec = BCDtoBIN(r.h.dh);
    t.ti_hund = 0;

    if (t.ti_hour == 0)
    {
        r.h.ah = 0x04;      // Read Date from Real Time Clock
        int86(0x1A, &r, &r);

        d.da_year = BCDtoBIN(r.h.ch) * 100 + BCDtoBIN(r.h.cl);
        d.da_mon = BCDtoBIN(r.h.dh);
        d.da_day = BCDtoBIN(r.h.dl);
    }

    setdate(&d);
    settime(&t);
}

void New_disk_image(void)
{
    unsigned char filename[13] = "";
    unsigned char diskname[17] = "";
    unsigned char diskid[3] = "";


    if (new_diskimage_dialog(filename, diskname, diskid)==1)
    {
        detach_disk_image(&diskette[disk_number]);
        strcat(filename,".D64");
        if (disk_image_create(filename, diskname, diskid,
            DISK_IMAGE_TYPE_D64)==0)
        {
            if (attach_disk_image(&diskette[disk_number], filename)==0)
            {
                gotoxy(2,13+disk_number);
                printf("%c", diskette[disk_number].read_only ? 'r' : ' ');
                printf("%s", diskette[disk_number].name);
                clreol();
                CURRENT_GCR_IMAGE = (unsigned int) diskette[disk_number].gcr;
                Door_Open = 0;
                gotoxy(1,25);
            }
            else tui_error("Could not attach the disk image file.");
        }
        else tui_error("Could not create the disk image file.");
    }
    return;
}

void Choose_disk_image(void)
{
    char temppath1[PATH_MAX], temppath2[PATH_MAX];
    int duplicate_file;

    char *default_item = NULL, *directory = NULL;
    char *name, *file = NULL;
    int i;

    /* "*.d64;*.g64" */
    name = tui_file_selector("Attach a disk image", directory, "*.D64",
            default_item, &file);
    if (name != NULL)
    {
        duplicate_file=0;
        _truename(name, temppath1);
        for (i=0; i<MAX_DISKS; i++)
        {
            temppath2[0]= '\0';
            _truename(diskette[i].name, temppath2);
            if (strncmp(temppath1, temppath2, PATH_MAX)==0)
            {
                tui_error("File is already open.");
                duplicate_file |= 1;
            }
        }
        if (!duplicate_file)
        {
            detach_disk_image(&diskette[disk_number]);
            if (attach_disk_image(&diskette[disk_number], name)==0)
            {
                gotoxy(2,13+disk_number);
                printf("%c", diskette[disk_number].read_only ? 'r' : ' ');
                printf("%s", diskette[disk_number].name);
                clreol();
                CURRENT_GCR_IMAGE = (unsigned int) diskette[disk_number].gcr;
                Door_Open = 0;
                gotoxy(1,25);
            }
        }
    }
}


int main(int argc, char *argv[])
{
    __dpmi_paddr old_keyb_vector;
    __dpmi_paddr old_LPT_vector;

    unsigned char KEY_INT;
    unsigned char LPT_INT;

    unsigned char old_IMR_PIC1;
    unsigned char old_IMR_PIC2;
    unsigned char PIC_mask;

    __dpmi_version_ret dpmi_info;
    int i;

    char original_directory[PATH_MAX];


    getcwd(original_directory, PATH_MAX);

    disk_image_t_size=(unsigned int)&diskette[1] - (unsigned int)&diskette[0];

    ScreenAddr = _go32_info_block.linear_address_of_primary_screen;
    ScreenClear();
    gotoxy(1,3);
    printf("1541EMU 2002-07-21, Copyright (C) 2001-2002 Ville Muikkula <1541@surfeu.fi>\n");

    for (i=strlen(argv[0])-1; argv[0][i]!='/'; i--);
    strncpy(dir, argv[0], i+1);
    dir[i+1] = '\0';
    for (i=0; i<strlen(dir); i++)
        if (dir[i]=='/') dir[i]='\\';
    strcpy(filename, dir);
    strcat(filename, INI_FILE);
    strupr(filename);

    selector = _dos_ds;

    if (!CheckRDTSC())
    {
        tui_error("The CPU of your PC does not support the RDTSC opcode.");
    }
    else if ((_my_cs() & 3) != 0)
    {
        tui_error("CPL is not 0! Please restart your computer in real mode DOS.");
    }
    else if (Read_Configuration() != 0) {}
    else if (drive_init() < 0) {}
    else if (argc > MAX_DISKS+1)
    {
        tui_error("No more that %d disk images are currently supported.", MAX_DISKS);
    }
    else if (Load_Disk_Images(argc, argv) != 0) {}
    else
    {
        gotoxy(1,10);
        printf("ESC = Quit, R = Reset, 1-0 = Set slot, O = Open disk image, N = New disk image\n");
        printf("(,/.) = Adjust I/O tuning value:\n");
        for (i=0; i<80; i++) printf("-");
        for (i=0; i<MAX_DISKS; i++)
        {
            gotoxy(1, 13+i);
            if (i>=9)
                printf("%d", i-9);
            else
                printf("%d", i+1);
            if (diskette[i].name != NULL)
            {
                printf("%c", diskette[i].read_only ? 'r' : ' ');
                printf("%s", diskette[i].name);
            }
        }

        gotoxy(1, 13);
        textattr(LIGHTGRAY<<4);
        putch('1');
        textattr(LIGHTGRAY);

        disk_number = 0;

        gotoxy(1, 23);
        for (i=0; i<80; i++) printf("-");
        printf("                        LED:     Motor:     Track:  1.0");
        gotoxy(1,25);

        CURRENT_GCR_IMAGE = 0;
        if (diskette[0].gcr != NULL)
        {
            CURRENT_GCR_IMAGE = (unsigned int) diskette[0].gcr;
            Door_Open = 0;
        }

        __dpmi_get_version(&dpmi_info);
        KEY_INT = dpmi_info.master_pic+KEY_IRQ;
        if (LPT_IRQ <= 7)
            LPT_INT = dpmi_info.master_pic+LPT_IRQ;
        else
            LPT_INT = dpmi_info.slave_pic+LPT_IRQ-8;

        do
        {
            install_ISR(KEY_INT, (void *)keyb_ISR, &old_keyb_vector);
            install_ISR(LPT_INT, (void *)LPT_ISR, &old_LPT_vector);

            // Wait until PC floppy disk drive motors shut down
            while (_farpeekb(_dos_ds, 0x0043F) & 0x0F);

            /* Enable LPT port IRQ */
            outportb(LPT_IO+2, inportb(LPT_IO+2) | Enable_IRQ);

            old_IMR_PIC1 = inportb(PIC1+1);
            old_IMR_PIC2 = inportb(PIC2+1);
            PIC_mask = 1;
            if (LPT_IRQ <= 7)
            {
                PIC_mask = PIC_mask << LPT_IRQ;
                outportb(PIC1+1, ~(2|PIC_mask));
            }
            else
            {
                PIC_mask = PIC_mask << (LPT_IRQ-8);
                outportb(PIC1+1, ~(2|4));
                outportb(PIC2+1, ~PIC_mask);
            }

            core_initialize();
            core_reset_hard();
            core_cycle_loop();

            outportb(PIC1+1, old_IMR_PIC1);
            outportb(PIC2+1, old_IMR_PIC2);

            /* Disable LPT port IRQ */
            outportb(LPT_IO+2, inportb(LPT_IO+2) & ~Enable_IRQ);

            remove_ISR(KEY_INT, &old_keyb_vector);
            remove_ISR(LPT_INT, &old_LPT_vector);

            if (Key_Code==0x31)
            {
                if (Global_Read_Only_Access)
                    tui_error("Can't create file (global write protection is active).");
                else
                    New_disk_image();
            }
            if (Key_Code==0x18) Choose_disk_image();
        } while (Key_Code != 1);

        for (i=0; i<MAX_DISKS; i++) detach_disk_image(&diskette[i]);
    }

    /* Update DOS time and date from the real time clock. */
    Set_DOS_Time();

    if (!WritePrivateProfileInt("1541EMU", "IO_tuning", IO_tuning, filename))
        tui_error("Could not update %s.", filename);

    chdir(original_directory);
    return 0;
}

