;
;   1541EMU, the Commodore 1541 disk drive emulator
;   Copyright (C) 2001-2002 Ville Muikkula
;   Copyright (C) Karsten Scheibler
;
;  This file was formerly part of ec64, the Commodore 64 emulator for Linux.
;  Written by
;   Karsten Scheibler <karsten.scheibler@bigfoot.de>
;  Modifications by
;   Ville Muikkula <1541@surfeu.fi>
;
;   This program is free software; you can redistribute it and/or modify
;   it under the terms of the GNU General Public License version 2 as
;   published by the Free Software Foundation.
;
;   This program is distributed in the hope that it will be useful,
;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;   GNU General Public License for more details.
;
;   You should have received a copy of the GNU General Public License
;   along with this program; if not, write to the Free Software
;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;****************************************************************************
;****************************************************************************
;*
;* via6522.n
;*
;****************************************************************************
;****************************************************************************


%define VIA6522_N
%include "via6522.i"
%include "core.i"
%include "c1541lnk.i"


;****************************************************************************
;****************************************************************************
;*
;* PART 1: assign's & define's
;*
;****************************************************************************
;****************************************************************************


%assign PB              VIA6522_PB
%assign PA              VIA6522_PA
%assign DDRB            VIA6522_DDRB
%assign DDRA            VIA6522_DDRA
%assign T1C_LO          VIA6522_T1C_LO
%assign T1C_HI          VIA6522_T1C_HI
%assign T1L_LO          VIA6522_T1L_LO
%assign T1L_HI          VIA6522_T1L_HI
%assign T2C_LO          VIA6522_T2C_LO
%assign T2C_HI          VIA6522_T2C_HI
%assign SR              VIA6522_SR
%assign ACR             VIA6522_ACR
%assign PCR             VIA6522_PCR
%assign IFR             VIA6522_IFR
%assign IER             VIA6522_IER
%assign PA_NH           VIA6522_PA_NH   ;PA no handshake

%define MODULID         VIA6522_MODULID
%define STATE           VIA6522_STATE
%define LINK_IRQ        VIA6522_LINK_IRQ
%define LINK_PB         VIA6522_LINK_PB
%define LINK_CB2        VIA6522_LINK_CB2
%define LINK_PA         VIA6522_LINK_PA
%define LINK_CA1        VIA6522_LINK_CA1
%define LINK_CA2        VIA6522_LINK_CA2
%define COUNTER_OFFSET  VIA6522_COUNTER_OFFSET
%define TIMESTAMP_T1    VIA6522_TIMESTAMP_T1
%define COUNTER_T1      VIA6522_COUNTER_T1
%define TIMESTAMP_T2    VIA6522_TIMESTAMP_T2
%define COUNTER_T2      VIA6522_COUNTER_T2
%define LATCH_PB        VIA6522_LATCH_PB
%define SIGNAL_PB       VIA6522_SIGNAL_PB
%define LATCH_PA        VIA6522_LATCH_PA
%define SIGNAL_PA       VIA6522_SIGNAL_PA
%define LATCH_T2_LO     VIA6522_LATCH_T2_LO
%define IRQ_MASK        VIA6522_IRQ_MASK
%define REGISTERS       VIA6522_REGISTERS

%assign T1_STOPPED      VIA6522_T1_STOPPED
%assign T2_STOPPED      VIA6522_T2_STOPPED
%assign STATE_CB1       VIA6522_STATE_CB1
%assign STATE_CA1       VIA6522_STATE_CA1

%macro GET_CYCLES 0
                        call    _core_get_c1541_cycles
%endmacro


;****************************************************************************
;****************************************************************************
;*
;* PART 2: Initialisation & Management Routines
;*
;****************************************************************************
;****************************************************************************


;****************************************************************************
;* via6522_initialize *******************************************************
;****************************************************************************
section .text
via6522_initialize:
                        ret



;****************************************************************************
;* via6522_exit *************************************************************
;****************************************************************************
section .text
via6522_exit:
                        ret



;****************************************************************************
;* via6522_instance_initialize **********************************************
;****************************************************************************
section .text
via6522_instance_initialize:
                        INSTANCE_INITIALIZE  tvia6522_size



;****************************************************************************
;* via6522_instance_exit ****************************************************
;****************************************************************************
section .text
via6522_instance_exit:
                        ret





;****************************************************************************
;****************************************************************************
;*
;* PART 3: Signal Routines
;*
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* via6522_reset_hard *******************************************************
;****************************************************************************
section .text
via6522_reset_hard:
                        jmp     via6522_reset_soft



;****************************************************************************
;* via6522_reset_soft *******************************************************
;****************************************************************************
section .text
via6522_reset_soft:
                        push    dword eax
                        mov     dword [STATE], (T1_STOPPED | T2_STOPPED)
                        xor     dword eax, eax
                        mov     dword [COUNTER_T1], eax
                        mov     dword [COUNTER_T2], eax
                        mov     byte  [LATCH_PB], al
                        mov     byte  [SIGNAL_PB], al
                        mov     byte  [LATCH_PA], al
                        mov     byte  [SIGNAL_PA], al
                        mov     byte  [LATCH_T2_LO], al
                        mov     byte  [IRQ_MASK], al
                        mov     dword [REGISTERS], eax
                        mov     dword [REGISTERS + 4], eax
                        mov     dword [REGISTERS + 8], eax
                        mov     dword [REGISTERS + 12], eax
                        pop     dword eax
                        ret



;****************************************************************************
;* via6522_signal_pb ********************************************************
;****************************************************************************
;* al==>  PB state
;****************************************************************************
section .text
via6522_signal_pb:
                        push    dword eax
                        mov     byte  [SIGNAL_PB], al
                        mov     byte  ah, [REGISTERS + DDRB]
                        xor     byte  ah, NOT8
                        and     byte  al, ah
                        mov     byte  ah, [LATCH_PB]
                        and     byte  ah, [REGISTERS + DDRB]
                        or      byte  ah, al
                        mov     byte  [REGISTERS + PB], ah
                        pop     dword eax
                        ret



;****************************************************************************
;* via6522_signal_pa ********************************************************
;****************************************************************************
;* al==>  PA state
;****************************************************************************
section .text
via6522_signal_pa:
                        push    dword eax
                        mov     byte  [SIGNAL_PA], al
                        mov     byte  ah, [REGISTERS + DDRA]
                        xor     byte  ah, NOT8
                        and     byte  al, ah
                        mov     byte  ah, [LATCH_PA]
                        and     byte  ah, [REGISTERS + DDRA]
                        or      byte  ah, al
                        mov     byte  [REGISTERS + PA], ah
                        pop     dword eax
                        ret



;****************************************************************************
;* via6522_signal_cb1 *******************************************************
;****************************************************************************
;* al==>  CB1 state
;****************************************************************************
section .text
via6522_signal_cb1:
                        push    dword eax
                        push    dword ebx
                        push    dword ecx
                        mov     byte  bl, al
                        mov     dword eax, [STATE]
                        xor     byte  bh, bh
                        test    dword eax, STATE_CB1
                        jz      .no_xor_1
                        xor     byte  bl, 1
                        mov     byte  bh, 1
.no_xor_1:              test    byte  bl, 1
                        jz      .exit
                        test    byte  [REGISTERS + PCR], 010h
                        jnz     .no_xor_2
                        xor     byte  bh, 1
.no_xor_2:              cmp     byte  bh, 1
                        je      .exit
                        mov     byte  ch, 010h
                        call    irq_set
.exit:                  pop     dword ecx
                        pop     dword ebx
                        pop     dword eax
                        ret



;****************************************************************************
;* via6522_signal_cb2 *******************************************************
;****************************************************************************
section .text
via6522_signal_cb2:
                        ret



;****************************************************************************
;* via6522_signal_ca1 *******************************************************
;****************************************************************************
;* al==>  CA1 state
;****************************************************************************
section .text
via6522_signal_ca1:
                        push    dword eax
                        push    dword ebx
                        push    dword ecx
                        mov     byte  bl, al
                        mov     dword eax, [STATE]
                        xor     byte  bh, bh
                        test    dword eax, STATE_CA1
                        jz      .no_xor_1
                        xor     byte  bl, 1
                        mov     byte  bh, 1
.no_xor_1:              test    byte  bl, 1
                        jz      .exit
                        test    byte  [REGISTERS + PCR], 001h
                        jnz     .no_xor_2
                        xor     byte  bh, 1
.no_xor_2:              cmp     byte  bh, 1
                        je      .exit
                        mov     byte  ch, 002h
                        call    irq_set
.exit:                  pop     dword ecx
                        pop     dword ebx
                        pop     dword eax
                        ret



;****************************************************************************
;* via6522_signal_ca2 *******************************************************
;****************************************************************************
section .text
via6522_signal_ca2:
                        ret





;****************************************************************************
;****************************************************************************
;*
;* PART 4: Emulation Code
;*
;****************************************************************************
;****************************************************************************





;****************************************************************************
;* irq_clear ****************************************************************
;****************************************************************************
;* cl==>  IFR
;* ch==>  bit in IFR
;****************************************************************************
section .text
irq_clear:
                        test    byte  cl, ch
                        jz      .no_irq
                        xor     byte  ch, NOT8
                        and     byte  cl, ch
                        test    byte  cl, [IRQ_MASK]
                        jnz     .irq_left
                        and     byte  cl, 07fh
                        push    dword eax
                        xor     byte  al, al
                        call    dword [LINK_IRQ]
                        pop     dword eax
.irq_left:              mov     byte  [REGISTERS + IFR], cl
.no_irq:                ret



;****************************************************************************
;* irq_set ******************************************************************
;****************************************************************************
;* ch==>  bit in IFR
;****************************************************************************
section .text
irq_set:
                        mov     byte  cl, [REGISTERS + IFR]
                        or      byte  cl, ch
                        test    byte  [IRQ_MASK], ch
                        jz      .no_irq
                        push    dword eax
                        or      byte  cl, 080h
                        mov     byte  al, 001h
                        call    [LINK_IRQ]
                        pop     dword eax
.no_irq:                mov     byte  [REGISTERS + IFR], cl
                        ret



;****************************************************************************
;* via6522_read *************************************************************
;****************************************************************************
;* al==>  value
;* ebx=>  address
;****************************************************************************
section .text
via6522_read:
                        mov     dword ebp, ebx
                        push    dword ecx
                        and     dword ebp, 00000000fh
                        mov     byte  cl, [REGISTERS + IFR]
                        jmp     dword [.ap__registers + 4 * ebp]

                        align   4
.ap__registers:         dd      read_pb
                        dd      read_pa
                        dd      read_register
                        dd      read_register
                        dd      read_t1c_lo
                        dd      read_t1c_hi
                        dd      read_register
                        dd      read_register
                        dd      read_t2c_lo
                        dd      read_t2c_hi
                        dd      read_register
                        dd      read_register
                        dd      read_register
                        dd      read_register
                        dd      read_ier
                        dd      read_register

read_register:
                        mov     byte  al, [REGISTERS + ebp]
                        pop     dword ecx
                        ret

read_pb:
                        mov     byte  ch, 010h
                        call    irq_clear
                        clc
                        call dword [LINK_PB]
                        mov     byte  al, [REGISTERS + PB]
                        pop     dword ecx
                        ret

read_pa:
                        mov     byte  ch, 002h
                        call    irq_clear
                        mov     byte  al, [REGISTERS + PA]
                        pop     dword ecx
                        ret

read_t1c_lo:
                        mov     byte  ch, 040h
                        call    irq_clear
                        test    dword [STATE], T1_STOPPED
                        jnz     .stopped
                        call    read_timer_1
                        pop     dword ecx
                        ret
.stopped:               xor     byte  al, al
                        pop     dword ecx
                        ret

read_t1c_hi:
                        test    dword [STATE], T1_STOPPED
                        jnz     .stopped
                        call    read_timer_1
                        pop     dword ecx
                        mov     byte  al, ah
                        ret
.stopped:               xor     byte  al, al
                        pop     dword ecx
                        ret

read_timer_1:
                        push    dword ebx
                        push    dword edx
                        GET_CYCLES
                        mov     dword ebx, [COUNTER_T1]
                        sub     dword eax, [TIMESTAMP_T1]
                        sub     dword ebx, eax
                        pop     dword edx
                        mov     dword eax, ebx
                        pop     dword ebx
                        ret

read_t2c_lo:
                        test    dword [STATE], T2_STOPPED
                        jnz     .stopped
                        mov     byte  ch, 020h
                        call    irq_clear
                        call    read_timer_2
                        pop     dword ecx
                        ret
.stopped:               xor     byte  al, al
                        pop     dword ecx
                        ret

read_t2c_hi:
                        test    dword [STATE], T2_STOPPED
                        jnz     .stopped
                        call    read_timer_2
                        pop     dword ecx
                        mov     byte  al, ah
                        ret
.stopped:               xor     byte  al, al
                        pop     dword ecx
                        ret

read_timer_2:
                        push    dword ebx
                        push    dword edx
                        GET_CYCLES
                        mov     dword ebx, [COUNTER_T2]
                        sub     dword eax, [TIMESTAMP_T2]
                        sub     dword ebx, eax
                        pop     dword edx
                        mov     dword eax, ebx
                        pop     dword ebx
                        ret

read_ier:
                        mov     byte  al, [IRQ_MASK]
                        pop     dword ecx
                        or      byte  al, 080h
                        ret



;****************************************************************************
;* via6522_write ************************************************************
;****************************************************************************
;* ebx=>  address
;* <==al  value
;****************************************************************************
section .text
via6522_write:
                        mov     dword ebp, ebx
                        push    dword ecx
                        and     dword ebp, 00000000fh
                        mov     byte  cl, [REGISTERS + IFR]
                        jmp     dword [.ap__registers + 4 * ebp]

                        align   4
.ap__registers:         dd      write_pb
                        dd      write_pa
                        dd      write_ddrb
                        dd      write_ddra
                        dd      write_t1c_lo
                        dd      write_t1c_hi
                        dd      write_register
                        dd      write_register
                        dd      write_t2c_lo
                        dd      write_t2c_hi
                        dd      write_ignore
                        dd      write_register
                        dd      write_pcr
                        dd      write_ifr
                        dd      write_ier
                        dd      write_pa_nh

write_ignore:
                        pop     dword ecx
                        ret

write_register:
                        mov     byte  [REGISTERS + ebp], al
                        pop     dword ecx
                        ret

write_pb:
;!!!!!!!!!!!!!!!
;TODO: handshake
;!!!!!!!!!!!!!!!
                        mov     byte  [LATCH_PB], al
                        mov     byte  ch, 010h
                        mov     byte  ah, [REGISTERS + DDRB]
                        call    irq_clear
                        and     byte  al, ah
                        xor     byte  ah, NOT8
                        and     byte  ah, [SIGNAL_PB]
                        or      byte  al, ah
                        mov     byte  [REGISTERS + PB], al
                        stc
                        call    dword [LINK_PB]
                        pop     dword ecx
                        ret


write_pa:
;!!!!!!!!!!!!!!!
;TODO: handshake
;!!!!!!!!!!!!!!!
                        mov     byte  [LATCH_PA], al
                        mov     byte  ch, 002h
                        mov     byte  ah, [REGISTERS + DDRA]
                        call    irq_clear
                        and     byte  al, ah
                        xor     byte  ah, NOT8
                        and     byte  ah, [SIGNAL_PA]
                        or      byte  al, ah
                        mov     byte  [REGISTERS + PA], al
                        call    dword [LINK_PA]
                        pop     dword ecx
                        ret

write_ddrb:
                        mov     byte  [REGISTERS + DDRB], al
                        mov     byte  ah, [LATCH_PB]
                        and     byte  ah, al
                        xor     byte  al, NOT8
                        and     byte  al, [SIGNAL_PB]
                        or      byte  ah, al
                        mov     byte  [REGISTERS + PB], ah
                        stc
                        call    dword [LINK_PB]
                        pop     dword ecx
                        ret

write_ddra:
                        mov     byte  [REGISTERS + DDRA], al
                        mov     byte  ah, [LATCH_PA]
                        and     byte  ah, al
                        xor     byte  al, NOT8
                        and     byte  al, [SIGNAL_PA]
                        or      byte  ah, al
                        mov     byte  [REGISTERS + PA], ah
                        call    dword [LINK_PA]
                        pop     dword ecx
                        ret

write_t1c_lo:
                        mov     byte  [REGISTERS + T1L_LO], al
                        pop     dword ecx
                        ret

write_t1c_hi:
                        mov     byte  ch, 040h
                        call    irq_clear
                        xor     dword ecx, ecx
                        mov     byte  [REGISTERS + T1L_HI], al
                        push    dword ebx
                        mov     byte  cl, [REGISTERS + T1L_LO]
                        push    dword edx
                        mov     byte  ch, al
                        mov     dword ebx, process_timer_1
                        inc     dword ecx
                        mov     dword eax, [COUNTER_OFFSET]
                        mov     dword [COUNTER_T1], ecx
                        call    c1541link_via_counter_set
                        GET_CYCLES
                        mov     dword ebx, [STATE]
                        mov     dword [TIMESTAMP_T1], eax
                        and     dword ebx, (NOT32 - T1_STOPPED)
                        mov     dword [TIMESTAMP_T1 + 4], edx
                        mov     dword [STATE], ebx
                        pop     dword edx
                        pop     dword ebx
                        pop     dword ecx
                        ret

process_timer_1:
                        test    byte  [REGISTERS + ACR], 040h
                        jz      .one_shot
                        xor     dword ecx, ecx
                        mov     dword eax, [COUNTER_OFFSET]
                        mov     byte  cl, [REGISTERS + T1L_LO]
                        mov     dword ebx, process_timer_1
                        mov     byte  ch, [REGISTERS + T1L_HI]
                        mov     dword [COUNTER_T1], ecx
                        call    c1541link_via_counter_refresh
                        GET_CYCLES
                        mov     byte  ch, 040h
                        mov     dword [TIMESTAMP_T1], eax
                        call    irq_set
                        mov     dword [TIMESTAMP_T1 + 4], edx
                        ret
.one_shot:              mov     dword eax, [STATE]
                        mov     byte  ch, 040h
                        or      dword eax, T1_STOPPED
                        call    irq_set
                        mov     dword [STATE], eax
                        ret

write_t2c_lo:
                        mov     byte  [LATCH_T2_LO], al
                        pop     dword ecx
                        ret

write_t2c_hi:
                        mov     byte  ch,020h
                        call    irq_clear
                        xor     dword ecx, ecx
                        push    dword ebx
                        mov     byte  cl, [LATCH_T2_LO]
                        push    dword edx
                        mov     dword ebx, process_timer_2
                        mov     byte  ch, al
                        mov     dword eax, [COUNTER_OFFSET]
                        inc     dword ecx
                        inc     dword eax
                        mov     dword [COUNTER_T2], ecx
                        call    c1541link_via_counter_set
                        GET_CYCLES
                        mov     dword ebx, [STATE]
                        mov     dword [TIMESTAMP_T2], eax
                        and     dword ebx, (NOT32 - T2_STOPPED)
                        mov     dword [TIMESTAMP_T2 + 4], edx
                        mov     dword [STATE], ebx
                        pop     dword edx
                        pop     dword ebx
                        pop     dword ecx
                        ret

process_timer_2:
                        mov     dword eax, [STATE]
                        mov     byte  ch, 020h
                        or      dword eax, T2_STOPPED
                        call    irq_set
                        mov     dword [STATE], eax
                        ret

write_pcr:
                        mov   byte  [REGISTERS + PCR], al
         call  dword [LINK_CA1]
         mov   byte  ah, al
                        xor     dword ecx, ecx
                        and     byte  al, 00eh
                        cmp     byte  al, 00eh
                        jne     .ca2_low
                        mov     byte  cl, 1
.ca2_low:               and     byte  ah, 0e0h
                        cmp     byte  ah, 0e0h
                        jne     .cb2_low
                        mov     byte  ch, 1
.cb2_low:               mov     byte  al, cl
                        call    dword [LINK_CA2]
                        mov     byte  al, ch
                        call    dword [LINK_CB2]
                        pop     dword ecx
                        ret

write_ifr:
                        and     byte  al, 07fh
                        jz      .end
                        xor     byte  al, 07fh
                        and     byte  al, [REGISTERS + IFR]
                        mov     byte  [REGISTERS + IFR], al
                        test    byte  al, [IRQ_MASK]
                        jnz     .end
                        xor     byte  al, al
                        call    dword [LINK_IRQ]
.end:                   pop     dword ecx
                        ret

write_ier:
                        test    byte  al, 080h
                        jz      .disable
                        and     byte  al, 07fh
                        or      byte  al, [IRQ_MASK]
                        jmp     .common
.disable:               xor     byte  al, 07fh
                        and     byte  al, [IRQ_MASK]
.common:                mov     byte  [IRQ_MASK], al
                        test    byte  al, [REGISTERS + IFR]
                        jnz     .no_clear
;!!!!!!!!!!!!!!!!!!
;TODO: irq_clear ??
;!!!!!!!!!!!!!!!!!!
                        and     byte  [REGISTERS + IFR], 07fh
.no_clear:              pop     dword ecx
                        ret

write_pa_nh:
                        mov     byte  [LATCH_PA], al
                        mov     byte  ah, [REGISTERS + DDRA]
                        and     byte  al, ah
                        not     byte  ah
                        and     byte  ah, [SIGNAL_PA]
                        or      byte  al, ah
                        mov     byte  [REGISTERS + PA], al
                        call    dword [LINK_PA]
                        pop     dword ecx
                        ret
