/*
 * Copyright (C) 1998 Wolfgang Moser aka Womo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Parallel port autodetection routines, version 0.11
 *
 * Wolfgang Moser, 1998, November, 15, 21:47 GMT
 *   womo@mindless.com,
 *   http://www.gm.fh-koeln.de/~womo (up to September 1999)
 *
 *
 * Basic informations from JAN'S PARALLEL PORT FAQ, Latest update: 4/22/97
 *   Jan Axelson, jaxelson@lvr.com
 *   links at: http://www.lvr.com/parport.htm
 *
 * Basic implementations by Kovcs Balzs aka Joe Forster/STA
 *   sta@ludens.elte.hu, http://ludens.elte.hu/~sta
 * Check out for his Star Commander at http://ludens.elte.hu/~sta/sc.html,
 * the final solution to handle disk/tape images for C64 emulators.
 *
 *
 * For additional informations to printer port programming check:
 *   http://www.doc.ic.ac.uk/~ih/doc/par/index.html
 *   http://www.senet.com.au/~cpeacock/parallel.htm
 *   http://www.cs.unc.edu/~tracker/tech/epp.html
 *   http://www.paranoia.com/~filipg/HTML/LINK/PORTS/F_Parallel.html
 *
 * One of the best parallel port detection utilities is
 * PARALLEL, Version 1.4 from Parallel Technologies, Inc.
 *   http://www.lpt.com/
 * It includes tests for an automatic IRQ detection.
 * You can get the file para14.zip from: ftp://lpt.com/Parallel/
 *                                      http://www.fapo.com/useful.htm
 */

#include "lptdetct.h"

lptMode LPTmode(register int port){
	register lptMode ret=lptN_A;
		// check for valid portaddresses (LPT 1-6)
		// valid port addresses only at 0x200, 0x204, 0x208, ..., 0x3fc
	if((port&~0x1fc)!=0x200) return lptN_A;
													//	if(!(port&0x07)){
													// test for ECP/EPP only at 0/8-bases

//	ResetLPT(port);
//	ECP test doesn't touch any data registers, so no reset is needed
	do{
			// tests for an ECP
		if((ret=ECPdetect(port))!=lptN_A) break;

			// perform a reset to prevent printers from printing unusable stuff
		ResetLPT(port);
			// tests for an EPP
		if((ret=EPPdetect(port))!=lptN_A) break;
#if AdvancedEPPTests
			// tests for an EPP with different control words
		if((ret=AdvEPP(port))!=lptN_A) break;
#endif
			// tests for a SPP or PS/2
		if((ret=PPPdetect(port))!=lptN_A) break;
		}while(0);
	BIDIoutp(port);
	return ret;
	}

//----------------------------------------------------------------
// Port Detection Routines
//----------------------------------------------------------------

	// ECP port detection
lptMode ECPdetect(register int port){
	register lptMode	ret=lptN_A;
	unsigned char		ECR=inportb(port+0x402)&~0x07;
	do{
		outportb(port+0x402,0x34);
		outportb(port+2,0xC6);
		if(inportb(port+0x402)!=0x35){
			ECR=0xC4;							// There's no ECR,
			break;
			}
		outportb(port+0x402,0x35);
		outportb(port+0x402,0xd4);
		inportb(port+0x400);
		outportb(port+0x400,0xAA);
		if(inportb(port+0x400)!=0xAA) break;
		outportb(port+0x400,0x55);
		if(inportb(port+0x400)!=0x55) break;

		ret=lptECP;

		}while(0);
	outportb(port+0x402,0x35);
	outportb(port+0x402,ECR);
	return ret;
	}

	// EPP port detection without Control port initialisation
lptMode EPPdWOC(register int port){
	do{
		EPPclear(port);
		outportb(port+3,0xAA);
		EPPclear(port);
		if(inportb(port+3)!=0xAA) break;
		EPPclear(port);
		outportb(port+3,0x55);
		EPPclear(port);
		if(inportb(port+3)!=0x55) break;

		return lptEPP;

		}while(0);
	EPPclear(port);
	outportb(port+3,0x00);
	inportb(port+3);
	if(!(inportb(port+1)&0x01)) return lptN_A;
	EPPclear(port);
	if(inportb(port+1)&0x01) return lptN_A;

	return lptEPP;
	}

	// Parallel Printer Port detection (SPP or PS/2)
lptMode PPPdetect(register int port){
	BIDIoutp(port);

	outportb(port,0xAA);
	if(inportb(port)!=0xAA) return lptN_A;
	outportb(port,0x55);
	if(inportb(port)!=0x55) return lptN_A;

	BIDIinp(port);

	outportb(port,0xAA);
	if(inportb(port)!=0xAA) return lptPS2;
	outportb(port,0x55);
	if(inportb(port)!=0x55) return lptPS2;

	return lptSPP;
	}

//----------------------------------------------------------------
// Port Mode Resolving Routines
//----------------------------------------------------------------

#if AdvancedEPPTests
lptMode AdvEPP(register int port){
	unsigned char EPPctrl=0;

		// check all control words to free up the EPP
	do{
		EPPctrl|=0x04;	// don't do a reset
			// write the special Control word, for freeing up the EPP
		outportb(port+2,EPPctrl);
		if(EPPdWOC(port)!=lptN_A) return lptEPPc;
		EPPctrl++;
		}while(EPPctrl);
	return lptN_A;
	}

// Resolving the special Control-Word to enable an EPP
char *EPPcontrol(int port){
	register int ret;
	static char ctrlWord[8];
	unsigned int  done0[8], done1[8];
	unsigned char i,EPPctrl, OldCtrl,mask;

	for(i=0;i<8;i++) done0[i]=done1[i]=0;

	OldCtrl=inportb(port+2)&0x1f;
	EPPctrl=0;  // check all control words to free up the EPP
	do{
		EPPctrl|=0x04;	// don't do a reset
			// write the special Control word, for freeing up the EPP
		outportb(port+2,EPPctrl);
		if(EPPdWOC(port)!=lptN_A){
			// EPP is enabled with this control word
			// return EPPctrl;
			for(i=0,mask=0x80;i<8;i++,mask>>=1){
				if(EPPctrl&mask) done1[i]++;
				else             done0[i]++;
				}
			}
		EPPctrl++;
		}while(EPPctrl);
	outportb(port+2,OldCtrl);
	for(i=0;i<8;i++){
		if(done0[i]==done1[i]){
			if(!done0[i]) 	 ctrlWord[i]='!';	// Control-Word could not found
			else			  	 ctrlWord[i]='X';	// This Bit cares nobody
			}
		else if(!done0[i]) ctrlWord[i]='1'; // This Bit must be 1
		else if(!done1[i]) ctrlWord[i]='0';	// This Bit must be 0
		else               ctrlWord[i]='?';	// This Bit depends on other Bits
		}
	return ctrlWord;
	}
#else
char *EPPcontrol(int){
	return "XX0X0100";
	}
#endif

//----------------------------------------------------------------
// Some helper functions
//----------------------------------------------------------------

	// perform a parallel printer port reset
void ResetLPT(register int port){
	register int i;

		// since a port read/write command is delayed by ISA bus waitstates to
		// 1,6 s, we can use it for a simple system independent delay routine.

		// But it would be much better to program one of the system
		// timers to delay 16 micro seconds
	for(i=10;i>0;i--)	outportb(port+2,0xC0);
	BIDIoutp(port);
	}

