/*
 * Copyright (C) 1998 Wolfgang Moser aka Womo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Parallel port BIOS initialisation, version 0.11
 *
 * Wolfgang Moser, 1998, November, 15, 21:47 GMT
 *   womo@mindless.com,
 *   http://www.gm.fh-koeln.de/~womo (up to September 1999)
 *
 *
 * Basic informations from JAN'S PARALLEL PORT FAQ, Latest update: 4/22/97
 *   Jan Axelson, jaxelson@lvr.com
 *   links at: http://www.lvr.com/parport.htm
 *
 * Basic implementations by Kovcs Balzs aka Joe Forster/STA
 *   sta@ludens.elte.hu, http://ludens.elte.hu/~sta
 * Check out for his Star Commander at http://ludens.elte.hu/~sta/sc.html,
 * the final solution to handle disk/tape images for C64 emulators.
 *
 *
 * For additional informations to printer port programming check:
 *   http://www.doc.ic.ac.uk/~ih/doc/par/index.html
 *   http://www.senet.com.au/~cpeacock/parallel.htm
 *   http://www.cs.unc.edu/~tracker/tech/epp.html
 *   http://www.paranoia.com/~filipg/HTML/LINK/PORTS/F_Parallel.html
 *
 * One of the best parallel port detection utilities is
 * PARALLEL, Version 1.4 from Parallel Technologies, Inc.
 *   http://www.lpt.com/
 * It includes tests for an automatic IRQ detection.
 * You can get the file para14.zip from: ftp://lpt.com/Parallel/
 *                                      http://www.fapo.com/useful.htm
 */


#include "stdio.h"
#include "lptdetct.h"

static int swM=0;

int preprprt(int port){
	lptMode mode;
	const char *modes[6]={"N/A","SPP","PS/2","EPP","ECP","EPPc"};

	mode=LPTmode(port);
	printf("at 0x%03X, %s", port, modes[mode]);
	return mode;
	}

int prport(int port){
	const char *ecpM[9]={
		"no ECR found",
		"Standard Mode",
		"Byte Mode",
		"Parallel Port FIFO Mode",
		"ECP FIFO Mode",
		"EPP Mode",
		"Reserved",
		"FIFO Test Mode",
		"Configuration Mode"
		};

	register int ret;

	ret=1;
	switch (preprprt(port)){
		case lptN_A:
			ret=0;
			break;
		case lptEPPc:
			printf(",  EPP-Enable-Timeout-Bit-Detection-Control-Word:"
					 " %s", EPPcontrol(port));
			break;
		case lptECP:
			printf(",  ECP-Mode: %s", ecpM[ECPmode(port)]);
		}
	printf("\n");
	return ret;
	}

int arg2int(const char *arg){
	int i,sum;

	if(arg==NULL) return 0;
	for(i=0;arg[i]!='\0';i++);
	if(i<3) return 0;
	sum=0;
	if(arg[0]=='0' && (arg[1]=='x' || arg[1]=='X')){
		if(i<5) return 0;
		for(i=2;arg[i]!='\0';i++){
			sum<<=4;
			if     (arg[i]>='A' && arg[i]<='F') sum|=arg[i]-'A'+0x0a;
			else if(arg[i]>='a' && arg[i]<='f') sum|=arg[i]-'a'+0x0a;
			else if(arg[i]>='0' && arg[i]<='9') sum|=arg[i]-'0';
			else return 0;
			}
		}
	else{
		for(i=0;arg[i]!='\0';i++){
			sum*=10;
			if(arg[i]>='0' && arg[i]<='9') sum+=arg[i]-'0';
			else return 0;
			}
		}
	return sum;
	}

void port2BIOSlpt(int lpt, int port){
	int padr;
		// The BIOS table has space for 4 LPT-entries,
		// but uses only up to 3 in most cases
	if(lpt<1 || lpt>4) return;
	poke(0x40,0x06+(lpt<<1),port);
	}

void reinit(int port, int LPTNr){
	register int mode;

	printf("LPT%d ",LPTNr);
	mode=preprprt(port);
	if(mode>lptSPP && ((swM&1) || (mode==lptECP && swM&2))){
		printf(",  switching port to ");
		if(swM&1){
			outportb(port+2,0xE4);
			printf("Input");
			}
		if(mode==lptECP && (swM&2)){
			if(swM&1) printf(" and ");
			outportb(port+0x402,(inportb(0x402)&0x1F)|0x20);
			printf("Byte");
			}
		printf(" mode");
		}
	printf("\n");
	if(swM&4) port2BIOSlpt(LPTNr,port);
	}

int main(int argc, char** argv){
	int i,j,k,arg=1,n=0,port, chk=3;
	const int padrs[3]={0x3bc, 0x378, 0x278};
	int newpa[4];

	for(i=1;i<argc;i++){
		if(argv[i]!=NULL) switch(argv[i][0]){
			case '/':
			case '-':
				switch(argv[i][1]){
					case 'i':
					case 'I':		// init non SPP's to input mode
						arg++;
						swM|=1;
						continue;
					case 'b':
					case 'B':		// init ECP's to Byte mode
						arg++;
						swM|=2;
						continue;
					case 'r':
					case 'R':		// reinitialise the BIOS data area
						arg++;
						swM|=4;
						continue;
					case 'd':
					case 'D':		// don't check BIOS for port addresses
						arg++;
						chk&=~1;
						continue;
					case 's':
					case 'S':		// don't check standard port addresses
						arg++;
						chk&=~2;
						continue;
					}
			}
		if(!arg2int(argv[i])){
				// Help and exit
			printf("Checking for unrecognized LPT ports and inserting "
					 "them to the BIOS data area.\n\nLPTBIOS [Address [A"
					 "ddress [...]]] [/I] [/B]\n\nAddress - userdefined "
					 "port addresses to check for LPT ports,\n          "
					 "arguments can be in decimal or sedecimal (hex, pre"
					 "fix 0x) form.\n/I      - switch all bidirectional "
					 "ports into input mode (else output mode)\n/B      "
					 "- switch all ECP ports into Byte mode\n/D      - d"
					 "on't check BIOS for LPT ports\n/S      - don't che"
					 "ck standard port addresses\n/R      - perform rein"
					 "itialisation of the BIOS data area\n\nExample: LPT"
					 "DETCT /B 0x268 636 /I\n");
			return 1;
			}
		}

	if(chk&1){
			// let's do a BIOS printer port check
		printf("Checking BIOS for recognized LPT ports ...\n\n");
		for(i=1;i<5;i++){
			port=BIOSlpt2port(i);
			if(!port) break;
			printf("LPT%1d ",i);
			if(prport(port) && n<4){
				for(j=0;j<n;j++) if(newpa[j]==port) break;
				if(j>=n) newpa[n++]=port;
				}
			}
		}

	if(argc>arg){
		printf("\nChecking userdefined LPT port addresses ...\n\n");
		for(i=1;i<argc;i++){
			port=arg2int(argv[i]);
			if(port>=0x200 && port<0x400){
				printf("Port ");
				if(prport(port) && n<4){
					for(j=0;j<n;j++) if(newpa[j]==port) break;
					if(j>=n) newpa[n++]=port;
					}
				}
			}
		}

	if(chk&2){
		k=0;
		for(i=0;i<3;i++){
			port=padrs[i];
			if(port && LPTmode(port)!=lptN_A && n<4){
				for(j=0;j<n;j++) if(newpa[j]==port) break;
				if(j>=n){
					if(!k){
						k=1;
						if(chk&1) printf("\nFound unrecognized ports at standard port addresses!!!\n\n");
						else printf("\nChecking standard port addresses ...\n\n");
						}
					printf("Port ");
					prport(port);
					newpa[n++]=port;
					}
				}
			}
		}

	k=1;
   if(swM){
		if(swM&4) printf("\n\nReinitialising BIOS data area with:\n\n");
   	else 		 printf("\n\nSetting LPT ports to:\n\n");
	   if(chk){	// standard addresses and already BIOS-registered
				   // port addresses were scanned, so we have to sort
				   // the standard port addresses 0x3BC, 0x378 and
				   // 0x278 in that order to the beginning of the
				   // BIOS LPT port entry table
		   for(i=0;i<3;i++){
			   port=padrs[i];
			   for(j=0;j<n;j++) if(port==newpa[j]) break;
			   if(j<n){
				   newpa[j]=0;
				   reinit(port, k++);
				   }
			   }
		   }
	   for(i=0;i<n;i++){
		   port=newpa[i];
		   if(port!=0) reinit(port, k++);
		   }
	   if(swM&4) for(;k<=4;k++){
		   port2BIOSlpt(k,0);		// Rest lschen
//			port2BIOSlpt(k,2);  		// Win'95 style ???
   	   }
		}
	return 0;
	}
