{ SORTTEXT.PAS : Sort textfile

  Title   : SORTTEXT
  Version : 2.3
  Date    : Dec 01,1996
  Author  : J R Ferguson
  Language: Turbo Pascal v7.0
  Usage   : Refer procedure Help
  Remarks : In memory sort
  Download: Author's Internet site www.xs4all.nl/~ferguson
            To compile this sourcefile you will need units from the Pascal
            library JRFPAS that can be found on the same Internet site.
  E-mail  : j.r.ferguson@iname.com
}


program SORTTEXT;


uses DefLib, ArgLib, StpLib, ChrLib;


const
  MAXFNM    = 79;         { Max filespec length (integrated environment) }
  DFLMSG    = 'CON';      { Default message output destination }
  INPBUFSIZ = 16 * 1024;  { Input buffer size in bytes }
  OUTBUFSIZ = 16 * 1024;  { Output buffer length in bytes }
  MAXLINE   = 5000;       { Max number of lines in text }

  { Error codes and messages: }
  ERROK     = 0;
  ERRARG    = 1;
  ERRFNF    = 2;
  ERRCRE    = 3;
  ERRREA    = 4;
  ERRWRI    = 5;
  ERRMAX    = 6;
  ERRMEM    = 7;

  ERRMSG    : array[ERRFNF..ERRMEM] of StpTyp =
 ('File not found',
  'File creation error',
  'Read error',
  'Write error',
  'Too many lines',
  'Out of memory'
 );

type
  InpBufTyp = array[1..INPBUFSIZ] of char;
  OutBufTyp = array[1..OUTBUFSIZ] of char;
  StpPtr    = ^StpTyp;
  PtrArrInd = 0..MAXLINE;
  PtrArrTyp = array[1..MAXLINE] of StpPtr;

var
  ErrCod    : integer;
  InpFnm,
  OutFnm    : StpTyp;
  Msg,
  Inp,
  Out       : Text;
  InpBuf    : InpBufTyp;
  OutBuf    : OutBufTyp;
  InpOpn,
  OutOpn    : boolean;
  PtrArr    : PtrArrTyp;
  LineCnt   : PtrArrInd;
  OptHlp    : boolean;
  OptKeyBeg : StpInd;
  OptKeyLen : StpInd;
  OptIgn    : boolean;
  OptLex    : boolean;
  OptDes    : integer; { -1 = descending, +1 = ascending}


{--- General routines ---}


procedure Help;
  procedure wi(i: integer); begin write  (Msg,i) end;
  procedure wr(s: StpTyp);  begin write  (Msg,s) end;
  procedure wl(s: StpTyp);  begin writeln(Msg,s) end;
begin {Help}
wl('SORTTEXT v2.3 - Sort lines in textfile.');
wl('Usage  : SORTTEXT [<in] [>out] [/option[...] [...]]');
wl('Options:');
wr('    K[c1][,c2]  Define key first col c1 [1], last col c2 [');
  wi(MaxStp); wl('].');
wr(' or K[c1][,+c2]            first col c1 [1], nr of cols c2 [');
  wi(MaxStp); wl('].');
wl('    D  Sort in descending order.');
wl('    I  Ignore upper/lower case. Implied by /L.');
wr('    L  Sort in Lexical order (control,punctuation,digits,letters).');
  wl(' Implies /I.');
wl('    H  Send this help text to (redirected) output.');
end;


{$F+} function HeapFunc(Size: word): integer; {$F-}
{ Make New() and GetMem() return a nil pointer when the heap is full }
begin HeapFunc:= 1; end;


{--- Command line parsing routines ---}


function ReadUns(var arg: StpTyp): integer;
var n: integer; c: char;
begin
  n:= 0; c:= StpcRet(arg,1);
  while (ErrCod=ERROK) and IsDigit(c) do begin
    StpDel(arg,1,1);
    n:= 10 * n + (ord(c) - ord('0'));
    c:= StpcRet(arg,1);
  end;
  ReadUns:= n;
end;


procedure ReadKeyDesc(var arg: StpTyp);
var n: integer;
begin
  if IsDigit(StpcRet(arg,1)) then OptKeyBeg:= ReadUns(arg) mod MaxStp;
  if StpcRet(arg,1) = ',' then begin
    StpDel(arg,1,1);
    if StpcRet(arg,1) = '+' then begin
      StpDel(arg,1,1);
      if IsDigit(StpcRet(arg,1)) then OptKeyLen:= ReadUns(arg) mod MaxStp
      else ErrCod:= ERRARG;
    end
    else begin
      if IsDigit(StpcRet(arg,1)) then begin
        n:= (ReadUns(arg) mod MaxStp) - OptKeyBeg + 1;
        if n < 0 then ErrCod:= ERRARG else OptKeyLen:= n;
      end
      else ErrCod:= ERRARG;
    end;
  end;
end;


procedure ReadOpt(var arg: StpTyp);
var nextopt: boolean;
begin
  StpDel(arg,1,1);
  repeat
    if StpEmpty(arg) or (StpcRet(arg,1) = '/') then ErrCod:= ERRARG
    else begin
      nextopt:= false;
      while (ErrCod=ERROK) and not nextopt and not StpEmpty(arg) do
      case StpcGet(arg) of
        'I': OptIgn:= true;
        'L': OptLex:= true;
        'D': OptDes:= -1;
        'K': ReadKeyDesc(arg);
        'H': OptHlp:= true;
        '/': nextopt:= true;
        else ErrCod:= ERRARG;
      end;
    end;
  until (ErrCod <> ERROK) or not nextopt;
end;


procedure ReadArgs;
var i   : ArgInd;
    arg : StpTyp;
begin
  GetArgs; i:= 0;
  while (i < ArgC) and (ErrCod = ERROK) do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    case StpcRet(arg,1) of
      '/': ReadOpt(arg);
      '<': StpSub(InpFnm,arg,2,MAXFNM); {simulate MS-DOS in Integrated Env}
      '>': StpSub(OutFnm,arg,2,MAXFNM); {simulate MS-DOS in Integrated Env}
      else ErrCod:= ERRARG;
    end;
  end;
  if OptHlp then if ErrCod = ERROK then ErrCod:= ERRARG else OptHlp:= false;
end;


{--- Low-level I/O routines ---}


procedure OpenMsg;
begin
  if OptHlp then Assign(Msg,OutFnm) else Assign(Msg,DFLMSG);
  rewrite(Msg);
end;


procedure CloseMsg;
begin Close(Msg) end;


procedure OpenInp;
begin
  Assign(Inp,InpFnm); SetTextBuf(Inp,InpBuf);
  {$I-} reset(Inp); {$I+}
  if IOresult <> 0 then ErrCod:= ERRFNF else InpOpn:= true;
end;


procedure CloseInp;
begin
  {$I-} Close(Inp); {$I+}
  if IOresult = 0 then InpOpn:= false;
end;


procedure OpenOut;
begin
  Assign(Out,OutFnm); SetTextBuf(Out,OutBuf);
  {$I-} rewrite(Out); {$I+}
  if IOresult <> 0 then ErrCod:= ERRCRE else OutOpn:= true;
end;


procedure CloseOut;
begin
  {$I-} Close(Out); {$I+}
  if IOresult = 0 then OutOpn:= false else begin
    if ErrCod = ERROK then ErrCod:= ERRWRI;
  end;
end;



{--- Processing routines ---}


procedure ReadText;
var line: StpTyp;
begin
  LineCnt:= 0;
  while (ErrCod=ERROK) and (LineCnt < MAXLINE) and not eof(Inp) do begin
    Inc(LineCnt);
    {$I-} ReadLn(Inp,line); {$I+}
    if IOresult <> 0 then ErrCod:= ERRREA else begin
      GetMem(PtrArr[LineCnt], succ(Length(line)));
      if PtrArr[LineCnt]=nil then ErrCod:= ERRMEM else
        PtrArr[LineCnt]^:= line;
    end;
  end;
  if (ErrCod=ERROK) and (LineCnt = MAXLINE) and not eof(Inp) then
    ErrCod:= ERRMAX;
end;


procedure WriteText;
var i: PtrArrInd;
begin
  i:= 0;
  while (ErrCod = ERROK) and (i < LineCnt) do begin
    Inc(i);
    {$I-} Writeln(Out,PtrArr[i]^); {$I+}
    if IOresult <> 0 then ErrCod:= ERRWRI;
  end;
end;


procedure SortPointers;
{ Insertion sort with Binary search }
var i,l,r,m: PtrArrInd; p: StpPtr; RefKey: StpTyp;
    key: StpTyp;
begin
  i:= 1;
  while i < LineCnt do begin
    Inc(i);
    p:= PtrArr[i];
    StpSub(RefKey, p^, OptKeyBeg, OptKeyLen);
    if OptIgn then StpUpp(RefKey);
    l:= 1; r:= pred(i);
    while l <= r do begin
      m:= (l + r) shr 1;
      StpSub(key, PtrArr[m]^, OptKeyBeg, OptKeyLen);
      if OptLex then begin
        if OptDes * StpLexCmp(key,RefKey) > 0
           then r:= pred(m) else l:= succ(m);
      end
      else begin
        if OptIgn then StpUpp(key);
        if OptDes * StpCmp(key,RefKey) > 0
           then r:= pred(m) else l:= succ(m);
      end;
    end;
    if l < i then begin
      Move(PtrArr[l], PtrArr[succ(l)], (i-l) * SizeOf(StpPtr));
      PtrArr[l]:= p;
    end;
  end;
end;


{--- Main line ---}


procedure MainProcess;
var Line: StpTyp; Heap: Pointer;
begin
  Mark(Heap);
  ReadText;
  if ErrCod = ERROK then begin
    SortPointers;
    WriteText;
  end;
  Release(Heap);
end;


procedure MainInit;
begin
  ErrCod:= ERROK;
  HeapError:= @HeapFunc; {install function to catch Out-of-Heap condition}
  StpCreate(InpFnm); InpOpn:= false;
  StpCreate(OutFnm); OutOpn:= false;
  OptHlp:= false; OptKeyBeg:= 1; OptKeyLen:= MaxStp;
  OptIgn:= false; OptLex:= false; OptDes:= +1;
  ReadArgs;
  if ErrCod = ERROK then OpenInp;
  if ErrCod = ERROK then OpenOut;
end;


procedure MainTerm;
begin
  if InpOpn then CloseInp;
  if OutOpn then CloseOut;
  if ErrCod <> ERROK then begin
    OpenMsg;
    if ErrCod=ERRARG then Help
    else writeln(Msg,'SORTTEXT: ',ERRMSG[ErrCod]);
    CloseMsg;
  end;
end;


begin { Main program }
  MainInit;
  if ErrCod = ERROK then MainProcess;
  MainTerm;
end.
