/*
 * Copyright (C) 1998 Wolfgang Moser aka Womo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Parallel port BIOS initialisation, version 0.11
 *
 * Wolfgang Moser, 1998, November, 15, 21:47 GMT
 *   womo@mindless.com,
 *   http://www.gm.fh-koeln.de/~womo (up to September 1999)
 *
 *
 * Basic informations from JAN'S PARALLEL PORT FAQ, Latest update: 4/22/97
 *   Jan Axelson, jaxelson@lvr.com
 *   links at: http://www.lvr.com/parport.htm
 *
 * Basic implementations by Kovcs Balzs aka Joe Forster/STA
 *   sta@ludens.elte.hu, http://ludens.elte.hu/~sta
 * Check out for his Star Commander at http://ludens.elte.hu/~sta/sc.html,
 * the final solution to handle disk/tape images for C64 emulators.
 *
 *
 * For additional informations to printer port programming check:
 *   http://www.doc.ic.ac.uk/~ih/doc/par/index.html
 *   http://www.senet.com.au/~cpeacock/parallel.htm
 *   http://www.cs.unc.edu/~tracker/tech/epp.html
 *   http://www.paranoia.com/~filipg/HTML/LINK/PORTS/F_Parallel.html
 *
 * One of the best parallel port detection utilities is
 * PARALLEL, Version 1.4 from Parallel Technologies, Inc.
 *   http://www.lpt.com/
 * It includes tests for an automatic IRQ detection.
 * You can get the file para14.zip from: ftp://lpt.com/Parallel/
 *                                      http://www.fapo.com/useful.htm
 */

#ifndef __LPTdetect_H__
#define __LPTdetect_H__
#include <dos.h>

	// Set ExtendedTests to 1, if you want to try searching for unusual EPP's
	// But take notice that the functionality of this test is not guaranteed!
#define AdvancedEPPTests 1
// #define AdvancedEPPTests 1


enum lptMode{
	lptN_A	= 0,
	lptSPP	= 1,
	lptPS2	= 2,
	lptEPP	= 3,
	lptECP	= 4,
	lptEPPc	= 5,			// EPP with special control word to enable
	};

enum ecpMode{
	ecpNOECR   = 0,      // no ECP-Port found!!!

	ecpSTNDRD  = 1,
	ecpBYTE    = 2,
	ecpSTDFIFO = 3,
	ecpECPFIFO = 4,
	ecpEPP     = 5,
	ecpRESVRD  = 6,
	ecpFIFOTST = 7,
	ecpCONFIG  = 8
	};

//----------------------------------------------------------------
// Main Port Detection Routines
//----------------------------------------------------------------

inline int		BIOSlpt2port(register int	lpt);
lptMode        LPTmode     (register int	port);

//----------------------------------------------------------------
// Port Mode Resolving Routines
//----------------------------------------------------------------

inline ecpMode	ECPmode     (register int port);
lptMode			AdvEPP		(register int port);
char          *EPPcontrol  (register int port);

//----------------------------------------------------------------
// Port Detection Routines for each port type
//----------------------------------------------------------------

lptMode			ECPdetect	(register int port);
inline lptMode	EPPdetect	(register int port);
lptMode			EPPdWOC		(register int port);
lptMode			PPPdetect	(register int port);

//----------------------------------------------------------------
// Some helper functions
//----------------------------------------------------------------

inline void		BIDIoutp		(register int port);
inline void		BIDIinp		(register int port);
void				ResetLPT		(register int port);
inline void		EPPclear		(register int port);

//----------------------------------------------------------------
// Inline Implementations
//----------------------------------------------------------------

inline int BIOSlpt2port(register int lpt){
	int padr;
		// The BIOS table has space for 4 LPT-entries,
		// but uses only up to 3 in most cases
	if(lpt<1 || lpt>4) return 0;

	padr=peek(0x40,0x06+(lpt<<1));
	if(padr>=0x200 && padr<0x400) return padr;
	return 0;
	}

	// set the Bidirectional-Flag to input (reverse mode) and block
	// the DataStrobe, AddressStrobe and Write line manually, so
	// that the EPP can't send any automatic handshake signal
inline lptMode EPPdetect(register int port){
	outportb(port+2,0xEF);
	return EPPdWOC(port);
	}

inline ecpMode ECPmode(int port){
	if(LPTmode(port)!=lptECP) return ecpNOECR;
	return ecpMode(ecpSTNDRD + (inportb(port+0x402)>>5));
	}

	// set the Bidirectional-Flag to output (normal mode)
inline void BIDIoutp(register int port){
	outportb(port+2,0xC4);
	}

	// set the Bidirectional-Flag to input (reverse mode)
inline void BIDIinp(register int port){
	outportb(port+2,0xE4);
	}

	// clear the Timeout-Flag
inline void EPPclear(int port){
	register unsigned char val=inportb(port+1);
											// Reset Timeout-Flag by reading
	outportb(port+1,val|0x01);    // or by writing 0
	outportb(port+1,val&0xFE);    // or by writing 1 to it
	}

#endif
