(*
 * Copyright (C) 1998 Wolfgang Moser
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

(*
 * XCDetect, a Commodore/IBM-PC transfercable autodetection utility,
 * designed for the users of The Star Commander and other transfer
 * utilities using the X-cables X1541, XE1541, XP1541 and XH1541.
 *
 * Wolfgang Moser <womo@mindless.com>
 *   http://www.gm.fh-koeln.de/~womo (up to September 1999)
 *
 *
 * Basic implementations by Joe Forster/STA <http://sta.c64.org>
 *   His util, The Star Commander is the state of the art utility
 *   for transferring files between modern IBM PCs and floppies
 *   of the Commodore world.
 *
 *   Check out the Star Commander home page at:
 *         http://sta.c64.org/sc.html
 *
 *
 * To learn more about the transfer cables used by The Star Commander
 * and this util, check the X1541 cable page at:
 *
 *   http://sta.c64.org/x1541.html
 *)

(*
 * This is a mixture between different versions of the source code
 * of The Star Commander, (C) 1994-1998 by Joe Forster/STA.
 *
 * Most of the transfer routines, the PrintStr, HexaStr and DecStr
 * routines are taken from version 0.82.11 beta source, the DecStr
 * routine is a patched version of SepStr.
 *
 * The ComputeDelay, _Delay and LPTMode routines and all the timing
 * constants (delay values) are taken from version 0.82.34 beta source.
 *
 *)

unit SWBase;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

const
{Pin assignments on the LPT printer port}
  Atn           = $01;
  Clk           = $02;
  _Reset        = $04;
  Data          = $08;
  Bidirectional = $20;
  ByteMode      = $20;
{LPT port modes}
  pmNone        = 0;
  pmSPP         = 1;
  pmPS2         = 2;
  pmEPP         = 3;
  pmECP         = 4;

var
  More,
  EOI,
  ExtCable      : Boolean;
  CBMDevNum,
  FileSecAddr,
  Serial,
  InData,
  OutData,
  Status,
  CableMode     : Byte;
  ChkDevSpeed,
  LPTAddr,
  ParLPTAddr,
  DelayValue    : Word;

procedure ComputeDelay;
procedure _Delay;
procedure InitLPTMode;
procedure ResetLPTPort;
procedure ResetDrive;
procedure ParallelInput;
procedure ParallelOutput;
procedure Receive;
procedure Send;
function CheckDevice: Boolean;
procedure Talk;
procedure Listen;
procedure TalkSec;
procedure ListenSec;
procedure Untalk;
procedure Unlisten;
procedure Open(SecAddr: Byte; const Command: string);
procedure _Close;
function ReadCBMError(var Error: string; Check: Boolean): Boolean;
function LPTMode(Address: Word): Byte;
procedure PrintStr(const S: String);
function DecStr(D: Longint; L: Byte): string;
function HexaStr(D: Longint; L: Byte): string;

implementation

{Compute the number of ticks needed for the execution of the timer setup
  routine; this value is subtracted from the number of ticks put into the
  timer by the real delay routine simulated here}
procedure ComputeDelay; assembler;
asm
    cli;
    mov cx, 100;
    mov dx, -1;
    mov DelayValue, 0;
@4: mov al, $B0;
    out $43, al;
    in al, $61;
    and al, $FD;
    or al, 1;
    out $61, al;
    mov al, $FF;
    out $42, al;
    out $42, al;
    mov ax, 10;
    call far ptr @1;
    mov al, $80;
    out $43, al;
    call @2;
    in al, $42;
    mov ah, al;
    call @2;
    in al, $42;
    xchg al, ah;
    not ax;
    or ax, ax;
    je @3;
    cmp dx, ax;
    jbe @3;
    mov dx, ax;
@3: loop @4;
    mov DelayValue, dx;
    sti;
    jmp @5;
@1: sub ax, DelayValue;
    jae @6;
@6: push ax;
    pop ax;
    in al, $61;
    and al, $FD;
    or al, 1;
    out $61, al;
    mov ah, al;
    out $61, al;
    out $61, al;
    in al, $61;
    and al, $20;
    je @7;
@7: retf;
@2: nop;
    retn;
@5:
end;

{Synchronize data transfer between the PC and the external CBM drive
  Input : AX: number of cycles to wait}
procedure _Delay; assembler;
asm
    sub ax, DelayValue;
    ja @1;
    mov ax, 1;
@1: push ax;
    mov al, $B0;
    out $43, al;
    pop ax;
    out $42, al;
    mov al, ah;
    out $42, al;
    in al, $61;
    and al, $FD;
    or al, 1;
    out $61, al;
@2: in al, $61;
    and al, $20;
    je @2;
end;

{Set the CLK line of the LPT port to low}
procedure ClkLo; assembler;
asm
    mov dx, LPTAddr;
    mov al, Serial;
    or al, Clk;
    out dx, al;
    mov Serial, al;
end;

{Set the CLK line of the LPT port to high}
procedure ClkHi; assembler;
asm
    mov dx, LPTAddr;
    mov al, Serial;
    and al, not Clk;
    out dx, al;
    mov Serial, al;
end;

{Set the DATA line of the LPT port to low}
procedure DataLo; assembler;
asm
    mov dx, LPTAddr;
    mov al, Serial;
    or al, Data;
    out dx, al;
    mov Serial, al;
end;

{Set the DATA line of the LPT port to high}
procedure DataHi; assembler;
asm
    mov dx, LPTAddr;
    mov al, Serial;
    and al, not Data;
    out dx, al;
    mov Serial, al;
end;

{Set the ATN line of the LPT port to low}
procedure AtnLo; assembler;
asm
    mov dx, LPTAddr;
    mov al, Serial;
    or al, Atn;
    out dx, al;
    mov Serial, al;
end;

{Set the ATN line of the LPT port to high}
procedure AtnHi; assembler;
asm
    mov dx, LPTAddr;
    mov al, Serial;
    and al, not Atn;
    out dx, al;
    mov Serial, al;
end;

{Read the status of the CLK line of the LPT port
  Output: ZF: when 1, CLK is high, otherwise CLK is low}
procedure ReadClk; assembler;
asm
    mov dx, LPTAddr;
    cmp ExtCable, 0;
    je @2;
    dec dx;
@2: in al, dx;
@1: mov ah, al;
    in al, dx;
    cmp ah, al;
    jne @1;
    cmp ExtCable, 0;
    je @3;
    xor al, $30;
    shr al, 4;
    mov ah, al;
@3: and al, Clk;
end;

{Read the status of the DATA line of the LPT port
  Output: ZF: when 1, DATA is high, otherwise DATA is low}
procedure ReadData; assembler;
asm
    mov dx, LPTAddr;
    cmp ExtCable, 0;
    je @2;
    dec dx;
@2: in al, dx;
@1: mov ah, al;
    in al, dx;
    cmp ah, al;
    jne @1;
    cmp ExtCable, 0;
    je @3;
    xor al, $30;
    shr al, 4;
    mov ah, al;
@3: and al, Data;
end;

{Determine if the parallel port is in the interval of valid addresses
  Input : Addr: the port address
  Output: when True, a port address is valid}
function ValidLPTAddr(Addr: Word): Boolean; assembler;
asm
    mov al, False;
    mov dx, Addr;

    and dx, not $1FC
    cmp dx, $0202;
    jne @1;

    inc al;
@1:
end;

{$F+}

{Set the error status
  Input : AL: error code to set error status to}
procedure Error; assembler;
asm
    or Status, al;
    sti;
    call AtnHi;
    call ClkHi;
    call DataHi;
    xor al, al;
    stc;
end;

{Initialize the mode of the parallel ports; ECP ports must be set to Byte
  mode, and all kinds of ports must be set to input mode
  Input : DX: the address of the port to initialize}
procedure InitLPTMode; assembler;
asm
    push dx;
    push dx;
    call ValidLPTAddr;
    pop dx;
    or al, al;
    je @1;
    add dx, $0402;
    in al, dx;
    and al, $1F;
    or al, ByteMode;
    out dx, al;
    sub dx, $0400;
    mov al, $FF;
    out dx, al;
    add dx, 2;
    mov al, Serial;
    or al, Bidirectional;
    mov Serial, al;
    out dx, al;
@1:
end;

{Reset the parallel port}
procedure ResetLPTPort; assembler;
asm
    push LPTAddr;
    call ValidLPTAddr;
    or al, al;
    je @1;
    mov dx, LPTAddr;
    in al, dx;
    or al, _Reset;
    and al, $EF;
    mov Serial, al;
    call AtnHi;
    call ClkLo;
    call DataHi;
@1:
end;

{Reset all external CBM drives}
procedure ResetDrive; assembler;
asm
    push LPTAddr;
    call ValidLPTAddr;
    or al, al;
    je @2;
    mov dx, LPTAddr;
    add dx, 2;
    mov al, Serial;
    and al, not _Reset;
    out dx, al;
    mov cx, 100;
@1: mov ax, 1000;
    call _Delay;
    loop @1;
    mov al, Serial;
    out dx, al;
@2:
end;

{Switch the parallel LPT port to input}
procedure ParallelInput; assembler;
asm
    mov dx, ParLPTAddr;
    mov al, $FF;
    out dx, al;
    add dx, 2;
    mov al, Serial;
    or al, Bidirectional;
    mov Serial, al;
    out dx, al;
@1:
end;

{Switch the parallel LPT port to output}
procedure ParallelOutput; assembler;
asm
    mov dx, ParLPTAddr;
    add dx, 2;
    mov al, Serial;
    and al, not Bidirectional;
    mov Serial, al;
    out dx, al;
@1:
end;

{Receive a byte from the external CBM drive
  Output: AL: the byte received}
procedure Receive; assembler;
asm
    cli;
    call ClkHi;
@1: call ReadClk;
    jne @1;
    xor cl, cl;
@6: mov bx, 30;
    call DataHi;
@3: dec bx;
    je @2;
    mov ax, 20;
    call _Delay;
    call ReadClk;
    je @3;
    jne @4;
@2: or cl, cl;
    je @5;
    mov al, 2;
    jmp Error;
@5: call DataLo;
    call ClkHi;
    or Status, $40;
    mov ax, 60;
    call _Delay;
    inc cl;
    jmp @6;
@4: mov bx, 8;
@7: call ReadClk;
    jne @7;
    and ah, Data;
    stc;
    je @8;
    clc;
@8: rcr InData, 1;
@9: call ReadClk;
    je @9;
    dec bx;
    jne @7;
    call DataLo;
    test Status, $40;
    je @10;
    mov ax, 150;
    call _Delay;
    call ClkHi;
    call DataHi;
@10:mov al, InData;
    sti;
end;

{Send a byte to the external CBM drive}
procedure RealSend; assembler;
asm
    cli;
    call DataHi;
    mov ax, 25;
    call _Delay;
    call ReadData;
    je @1;
    call ClkHi;
    cmp EOI, False;
    je @2;
@3: call ReadData;
    jne @3;
@4: call ReadData;
    je @4;
@2: call ReadData;
    jne @2;
    call ClkLo;
    mov bx, 8;
@8: mov ax, 25;
    call _Delay;
    call ReadData;
    jne @5;
    mov ax, 100;
    call _Delay;
    shr OutData, 1;
    jc @6;
    call DataLo;
    jne @7;
@6: call DataHi;
@7: call ClkHi;
    mov ax, 25;
    call _Delay;
    call DataHi;
    call ClkLo;
    dec bx;
    jne @8;
    mov cx, 10;
@10:mov ax, 20;
    call _Delay;
    call ReadData;
    jne @9;
    loop @10;
@5: mov al, 3;
    jmp Error;
@1: mov al, $80;
    jmp Error;
@9: sti;
end;

{Buffer a byte to be sent to the external CBM drive and send the previous
  byte
  Input : AL: the byte to send}
procedure Send; assembler;
asm
    push ax;
    cmp More, False;
    jne @1;
    mov More, True;
    je @2;
@1: call RealSend;
@2: pop ax;
    mov OutData, al;
end;

{Determine if a device is present on the serial bus
  Output: when True, there is a device on the bus}
function CheckDevice: Boolean; assembler;
asm
    cli;
    push ax;
    push si;
    mov Status, 0;
    mov dx, LPTAddr;
    mov si, dx;
    mov bh, Data;
    cmp ExtCable, 0;
    je @1;
    shl bh, 4;
    dec si;
@1: mov di, 1;
    mov cx, ChkDevSpeed;
    call ResetLPTPort;
    mov al, Serial;
    and al, not (Atn + Clk + Data);
    mov Serial, al;
    mov bl, al;
@5: mov al, bl;
    out dx, al;
    call @6;
    mov al, bl;
    or al, Atn;
    out dx, al;
    xchg dx, si;
    in al, dx;
    and al, bh;
    xchg dx, si;
    je @3;
    call @6;
    xchg dx, si;
    in al, dx;
    and al, bh;
    xchg dx, si;
    je @4;
@3: shl di, 1;
    loop @5;
    mov al, $80;
    call Error;
@4: mov al, Serial;
    out dx, al;
    call @6;
    pop si;
    pop ax;
    sti;
    mov al, False;
    test Status, $80;
    jne @2;
    inc al;
    jmp @2;
@6: push cx;
    mov cx, di;
@7: mov ax, 1000;
    call _Delay;
    loop @7;
    pop cx;
    retn;
@2:
end;

{Send a command (single byte with ATN low) to the external CBM drive
  Input : AL: command byte to be sent}
procedure Command; assembler;
asm
    push ax;
    cmp More, False;
    je @1;
    mov EOI, True;
    call RealSend;
    mov More, False;
    mov EOI, False;
@1: pop ax;
    mov OutData, al;
    call DataHi;
    call AtnLo;
    cli;
    call ClkLo;
    call DataHi;
    mov ax, 500;
    call _Delay;
    jmp RealSend;
end;

{Send a TALK command to the external CBM drive}
procedure Talk; assembler;
asm
    mov al, CBMDevNum;
    or al, $40;
    jmp Command;
end;

{Send a LISTEN command to the external CBM drive}
procedure Listen; assembler;
asm
    mov al, CBMDevNum;
    or al, $20;
    jmp Command;
end;

{Send a TALK secondary address to the external CBM drive
  Input : AL: secondary address to be sent}
procedure TalkSec; assembler;
asm
    mov OutData, al;
    test Status, $80;
    jne @3;
    cli;
    call ClkLo;
    call DataHi;
    mov ax, 500;
    call _Delay;
    call RealSend;
    cli;
    call DataLo;
    call AtnHi;
    call ClkHi;
    xor cx, cx;
@1: call ReadClk;
    je @2;
    loop @1;
@2: mov ax, 500;
    call _Delay;
    sti;
@3:
end;

{Send a LISTEN secondary address to the external CBM drive
  Input : AL: secondary address to be sent}
procedure ListenSec; assembler;
asm
    mov OutData, al;
    test Status, $80;
    jne @1;
    cli;
    call ClkLo;
    call DataHi;
    mov ax, 500;
    call _Delay;
    call RealSend;
    call AtnHi;
@1:
end;

{Send an UNTALK command to the external CBM drive}
procedure Untalk; assembler;
asm
    cli;
    call ClkLo;
    call AtnLo;
    mov al, $5F;
    call Command;
    cli;
    call AtnHi;
    mov ax, 200;
    call _Delay;
    call ClkHi;
    call DataHi;
    sti;
end;

{Send an UNLISTEN command to the external CBM drive}
procedure Unlisten; assembler;
asm
    cli;
    mov al, $3F;
    call Command;
    cli;
    call AtnHi;
    mov ax, 200;
    call _Delay;
    call ClkHi;
    call DataHi;
    sti;
end;

{Open a file on a disk or send a disk command to the external CBM drive
  Input : SecAddr: secondary address of the file or $6F if disk command
          Command: the file name or disk command to be sent}
procedure Open(SecAddr: Byte; const Command: string); assembler;
asm
    test Status, $80;
    jne @1;
    les di, Command;
    cmp byte ptr es:[di], 0;
    je @1;
    mov Status, 0;
    mov More, False;
    mov EOI, False;
    call Listen;
    mov al, SecAddr;
    cmp al, $6F;
    je @3;
    mov FileSecAddr, al;
    or al, $F0;
@3: call ListenSec;
    test Status, $80;
    jne @1;
    xor bx, bx;
    les di, Command;
    mov cl, byte ptr es:[di];
    xor ch, ch;
@2: inc bx;
    push bx;
    push cx;
    mov al, byte ptr es:[di][bx];
    call Send;
    pop cx;
    pop bx;
    loop @2;
    call Unlisten;
@1:
end;

{Close a file on the disk in the external CBM drive}
procedure _Close; assembler;
asm
    call Listen;
    mov al, FileSecAddr;
    and al, $0F;
    or al, $E0;
    call ListenSec;
    call Unlisten;
end;

{$F-}
{Receive the disk error from the external CBM drive
  Input : Error: the string to contain the error message
          Check: when True, the serial bus status is checked and the
                 process of reading the error message is skipped if the
                 drive is off
  Output: when False, an error occured}
function ReadCBMError(var Error: string; Check: Boolean): Boolean;
var
  C             : Char;
begin
  Error := '';
  if not Check or (Status and $80 = 0) then
  begin
    asm
      mov Status, 0;
      mov More, False;
      mov EOI, False;
      call Talk;
      mov al, $6F;
      call TalkSec;
      test Status, $80;
      jne @1;
      les di, Error;
      xor bx, bx;
      cld;
  @2: push bx;
      push cx;
      call Receive;
      pop cx;
      pop bx;
      cmp Status, 0;
      jne @3;
      inc bl;
      mov es:[di][bx], al;
      cmp bl, 255;
      je @3;
      loop @2;
  @3: mov es:[di], bl;
      call Untalk;
  @1:
    end;
  end;
  if Status and $80 > 0 then
  begin
    Str(CBMDevNum, Error);
    Error := 'Drive ' + Error + ': not present';
  end;
  if Error[Length(Error)] = #13 then Dec(Error[0]);
  ReadCBMError := ((Length(Error) > 2) and (Error[1] = '0'));
end;

{Determine the mode of the current LPT port
  Input : Address: the address of the LPT port
  Output: LPT port mode}
function LPTMode(Address: Word): Byte; assembler;
asm
    push Address;
    call ValidLPTAddr;
    xor cl, cl;
    or al, al;
    je @1;
    mov dx, Address;
{   add dx, 2;       version 0.82.11 uses base+2 as port address}
    mov bx, dx;
    mov cl, pmECP;
    add dx, $0400;
    in al, dx;
    and al, $F8;
    mov ch, al;
    mov al, $34;
    out dx, al;
    mov dx, bx;
    mov al, $C6;
    out dx, al;
    add dx, $0400;
    in al, dx;
    cmp al, $35;
    jne @2;
    mov al, $D4;
    out dx, al;
    sub dx, 2;
    in al, dx;
    mov al, $AA;
    out dx, al;
    jmp @11;
@11:in al, dx;
    cmp al, $AA;
    jne @2;
    mov al, $55;
    out dx, al;
    jmp @12;
@12:in al, dx;
    cmp al, $55;
    jne @2;
    add dx, 2;
    mov al, $35;
    out dx, al;
    mov al, ch;
    out dx, al;
    jmp @1;
@2: mov dx, bx;
    add dx, $0400;
    mov al, ch;
    or al, _Reset;
    out dx, al;
    mov cl, pmEPP;
    mov al, $EF;
    call @5;
    jnc @1;
    xor al, al;
@18:push ax;
    or al, _Reset;
    call @5;
    pop ax;
    jnc @1;
    inc al;
    jne @18;
    mov cl, pmSPP;
    mov dx, bx;
    in al, dx;
    or al, Bidirectional;
    out dx, al;
    sub dx, 2;
    mov al, $AA;
    out dx, al;
    jmp @9;
@9: in al, dx;
    cmp al, $AA;
    jne @3;
    mov al, $55;
    out dx, al;
    jmp @10;
@10:in al, dx;
    cmp al, $55;
    je @13;
@3: mov cl, pmPS2;
@13:mov dx, bx;
    in al, dx;
    and al, not Bidirectional;
    out dx, al;
    sub dx, 2;
    mov al, $AA;
    out dx, al;
    jmp @14;
@14:in al, dx;
    cmp al, $AA;
    jne @4;
    mov al, $55;
    out dx, al;
    jmp @8;
@8: in al, dx;
    cmp al, $55;
    je @1;
@4: xor cl, cl;
    jmp @1;
@5: mov dx, bx;
    out dx, al;
    inc dx;
    call @7;
    mov al, $AA;
    out dx, al;
    call @7;
    in al, dx;
    cmp al, $AA;
    stc;
    jne @6;
    call @7;
    mov al, $55;
    out dx, al;
    call @7;
    in al, dx;
    cmp al, $55;
    stc;
    jne @6;
    clc;
@6: retn;
@7: push ax;
    push dx;
    mov dx, bx;
    dec dx;
    in al, dx;
    or al, $01;
    out dx, al;
    and al, $FE;
    out dx, al;
    pop dx;
    pop ax;
    retn;
@1: mov al, cl;
end;

procedure PrintStr(const S: String); assembler;
asm
    push ds;
    lds si, S;
    cld;
    lodsb;
    xor ah, ah;
    xchg ax, cx;
    mov ah, $40;
    mov bx, 1;
    mov dx, si;
    int $21;
    pop ds;
end;

{Convert a long integer into a decimal string with thousands separator
  Input : D: the long integer
          L: number of digits to put into the string
  Output: the decimal string}
function DecStr(D: Longint; L: Byte): string;
var
  S             : string;
begin
  Str(D:L,S);
  DecStr:=S;
end;

{Convert a long integer into a hexadecimal string
  Input : D: the long integer
          L: number of digits to put into the string
  Output: the hexadecimal string}
function HexaStr(D: Longint; L: Byte): string;
var
  I             : Byte;
  S             : string;
const
{Hexadecimal digits}
  HexaNum       : Array [0..15] of Char = '0123456789ABCDEF';
begin
  S := '';
  for I := L - 1 downto 0 do S := S + HexaNum[(D shr (I * 4) and $0F)];
  HexaStr := S;
end;

end.