(*
 * Copyright (C) 1994-2004 Joe Forster/STA
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

(*
 * XCDetect, a Commodore/IBM-PC transfercable autodetection utility,
 * designed for the users of The Star Commander and other transfer
 * utilities using the X-cables:
 *         X1541, XE1541, XM1541, XA1541, XP1541 and XH1541.
 *
 * Wolfgang Moser <cbm (at) d81 (dot) de>
 *   http://d81.de
 *
 *
 * Basic implementations by Joe Forster/STA <http://sta.c64.org>
 *   His util, The Star Commander is the state of the art utility
 *   for transferring files between modern IBM PCs and floppies
 *   of the Commodore world.
 *
 *   Check out the Star Commander home page at:
 *         http://sta.c64.org/sc.html
 *
 *
 * To learn more about the transfer cables used by The Star Commander
 * and this util, check the X1541 cable page at:
 *
 *   http://sta.c64.org/x1541.html
 *)

(*
 * This is a mixture between different versions of the source code
 * of The Star Commander, (C) 1994-2004 by Joe Forster/STA.
 *
 * Most of the transfer routines, the PrintStr, HexaStr and DecStr
 * routines are taken from version 0.83.17 beta source, the DecStr
 * routine is a patched version of SepStr.
 *
 * The ComputeDelay, _Delay and LPTMode routines and all the timing
 * constants (delay values) are taken from version 0.83.17 beta source.
 *
 *)

unit SWBase;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

const
{former configuration variables, now fixed}
  ECPAddrOffs   = $400;
{Execution priority levels}
  plLow         = 0;
  plHigh        = 1;
{Maximum value for a word}
  MaxWord       = $FFFF;
{Serial bus lines, converted to pin assignments on the LPT printer port; note
  the different DATA and RESET bits for the normal, extended and active
  cables}
  sbAtnNormal   = $01;
  sbClkNormal   = $02;
  sbDataNormal  = $08;
  sbDataSwapped = $04;
  sbResetNormal = $04;
  sbResetSwapped= $08;
  sbBidir       = $20;
{Parallel cables}
  pcNone        = 0;
  pcHybrid      = 1;
  pcParallel    = 2;
{Serial cables}
  scNone        = 0;
  scNormal      = 1;
  scExtended    = 2;
  scMultitask   = 3;
  scActive      = 4;
{LPT port modes}
  pmNone        = 0;
  pmSPP         = 1;
  pmPS2         = 2;
  pmEPP         = 3;
  pmECP         = 4;
  pmUnknown     = $80;
{Serial bus status codes}
  ssTimeout     = $02;
  ssInputTimeout= $03;
  ssErrorMask   = $3F;
  ssEOF         = $40;
  ssNoDevice    = $80;
{Secondary address of Commodore drive command channel}
  CmdSecAddr    = $6F;
{Maximum length of strings}
  MaxStrLen     = 254;
{PETSCII character constants}
  chReturn      = #$0D;
{Hexadecimal digits}
  HexaNum       : string[16] = '0123456789ABCDEF';

type
{Procedure without parameters}
  TProc         = procedure;

var
  EOI,
  InterruptsOff,
  More,
  Timeout       : Boolean;

  ClkDataBits,
  DataBit,
  ResetBit,
  InData,
  OutData,
  Serial,
  ParLPTCtrl,
  ParallelCable,
  SerialCable,
  CBMDevNum,
  FileSecAddr,
  Status        : Byte;

  ChkDevSpeed,
  CopyDelayValue,
  LPTAddr,
  ParLPTAddr    : Word;

  ReadLine,
  AtnLo,
  AtnHi,
  ClkLo,
  ClkHi,
  DataLo,
  DataHi        : TProc;


procedure ComputeDelay;
procedure Delay;
function ValidLPTAddr(Addr: Word): Boolean;
function GetLPTMode(Address: Word): Byte;
procedure ResetLPTPort;
procedure InterruptOff(Level: Byte);
procedure InterruptOn;

procedure InitLPTBits;
procedure InitLPTPorts;
procedure InitTransfer;
procedure ResetDrives;
function CheckDevice: Boolean;

procedure ParallelInput;
procedure ParallelOutput;
procedure Receive;
procedure Send;

procedure Talk;
procedure Listen;
procedure TalkSec;
procedure ListenSec;
procedure Untalk;
procedure Unlisten;

procedure OpenCBMChannel(SecAddr: Byte; const Command: string);
procedure CloseCBMChannel;
function ReadCBMError(var Error: string; Check: Boolean): Boolean;

procedure PrintStr(const S: string);
function DecStr(D: Longint; L: Byte): string;
function HexaStr(D: Longint; L: Byte): string;
function LeadingSpace(Num: Longint; DigitNum: Word): string;


implementation


{Compute the number of ticks needed for the execution of the timer setup
  routine; this value is subtracted from the number of ticks put into the
  timer by the real delay routine simulated here}
procedure ComputeDelay; assembler;
asm
    push plHigh;
    call InterruptOff;
    mov cx, 100;
    mov dx, MaxWord;
    mov CopyDelayValue, 0;
@4: mov al, $B0;
    out $43, al;
    in al, $61;
    and al, $FD;
    or al, 1;
    out $61, al;
    mov al, $FF;
    out $42, al;
    out $42, al;
    mov ax, 10;
    call far ptr @1;
    mov al, $80;
    out $43, al;
    call @2;
    in al, $42;
    mov ah, al;
    call @2;
    in al, $42;
    xchg al, ah;
    not ax;
    or ax, ax;
    je @3;
    cmp dx, ax;
    jbe @3;
    mov dx, ax;
@3: loop @4;
    mov CopyDelayValue, dx;
    call InterruptOn;
    jmp @5;
@1: sub ax, CopyDelayValue;
    jae @6;
@6: push ax;
    pop ax;
    in al, $61;
    and al, $FD;
    or al, 1;
    out $61, al;
    mov ah, al;
    out $61, al;
    out $61, al;
    in al, $61;
    and al, $20;
    je @7;
@7: retf;
@2: nop;
    retn;
@5:
end;

{Synchronize data transfer between the PC and the external CBM drive, by waiting
  a given amount of time
  Input : AX: number of PIT cycles to wait}
procedure Delay; assembler;
asm
    sub ax, CopyDelayValue;
    ja @1;
    mov ax, 1;
@1: push ax;
    mov al, $B0;
    out $43, al;
    pop ax;
    out $42, al;
    mov al, ah;
    out $42, al;
    in al, $61;
    and al, $FD;
    or al, 1;
    out $61, al;
@2: in al, $61;
    and al, $20;
    je @2;
end;

{Turn interrupts and task switching off
  Input : Level: priority level; switch off occurs only if high priority level}
procedure InterruptOff(Level: Byte); assembler;
asm
    cmp InterruptsOff, False;
    jne @1;
    mov al, Level;
    cmp al, plLow;
    je @1;
    mov InterruptsOff, True;
    cli;
@1:
end;

{Turn interrupts and task switching on}
procedure InterruptOn; assembler;
asm
    cmp InterruptsOff, False;
    je @1;
    mov InterruptsOff, False;
    sti;
@1:
end;

{Set the ATN line of the LPT port to low on a normal/extended cable}
procedure AtnLoNormal; far; assembler;
asm
    mov dx, LPTAddr;
    add dx, 2;
    mov al, sbAtnNormal;
    or al, Serial;
    out dx, al;
    mov Serial, al;
end;

{Set the ATN line of the LPT port to high on a normal/extended cable}
procedure AtnHiNormal; far; assembler;
asm
    mov dx, LPTAddr;
    add dx, 2;
    mov al, not sbAtnNormal;
    and al, Serial;
    out dx, al;
    mov Serial, al;
end;

{Set the CLK line of the LPT port to low on a normal/extended cable}
procedure ClkLoNormal; far; assembler;
asm
    mov dx, LPTAddr;
    add dx, 2;
    mov al, sbClkNormal;
    or al, Serial;
    out dx, al;
    mov Serial, al;
end;

{Set the CLK line of the LPT port to high on a normal/extended cable}
procedure ClkHiNormal; far; assembler;
asm
    mov dx, LPTAddr;
    add dx, 2;
    mov al, not sbClkNormal;
    and al, Serial;
    out dx, al;
    mov Serial, al;
end;

{Set the DATA line of the LPT port to low on a normal/extended/active cable}
procedure DataLoNormal; far; assembler;
asm
    mov dx, LPTAddr;
    add dx, 2;
    mov al, DataBit;
    or al, Serial;
    out dx, al;
    mov Serial, al;
end;

{Set the DATA line of the LPT port to high on a normal/extended/active cable}
procedure DataHiNormal; far; assembler;
asm
    mov dx, LPTAddr;
    add dx, 2;
    mov al, DataBit;
    not al;
    and al, Serial;
    out dx, al;
    mov Serial, al;
end;

{Read the status of the DATA and CLK lines of the normal cable
  Output: AL: the status of the lines
          CF: when not zero, a timeout has occured, the user pressed
              Escape or F10}
procedure ReadLineNormal; far; assembler;
asm
{
    call CheckTimeout;
    jc @2;
}
    mov dx, LPTAddr;
    add dx, 2;
    in al, dx;
@1: mov ah, al;
    in al, dx;
    cmp ah, al;
    jne @1;
    and al, (sbClkNormal + sbDataNormal);
    clc;
@2:
end;

{Read the status of the DATA and CLK lines of the extended cable
  Output: AL: the status of the lines
          CF: when not zero, a timeout has occured, the user pressed
              Escape or F10}
procedure ReadLineExtended; far; assembler;
asm
{
    call CheckTimeout;
    jc @2;
}
    mov dx, LPTAddr;
    inc dx;
    in al, dx;
@1: mov ah, al;
    in al, dx;
    cmp ah, al;
    jne @1;
    xor al, ((sbAtnNormal + sbClkNormal) shl 4);
    shr al, 4;
    mov ah, al;
    and al, (sbClkNormal + sbDataNormal);
    clc;
@2:
end;

{Read the status of the DATA and CLK lines of the multitask cable
  Output: AL: the status of the lines
          CF: when not zero, a timeout has occured, the user pressed
              Escape or F10}
procedure ReadLineMultitask; far; assembler;
asm
{
    call CheckTimeout;
    jc @2;
}
    mov dx, LPTAddr;
    inc dx;
    in al, dx;
@1: mov ah, al;
    in al, dx;
    cmp ah, al;
    jne @1;
    xor al, ((sbAtnNormal + sbClkNormal + sbDataSwapped) shl 4);
    shr al, 4;
    mov ah, al;
    and al, (sbClkNormal + sbDataSwapped);
    clc;
@2:
end;

{Read the status of the DATA and CLK lines of the active cable
  Output: AL: the status of the lines
          CF: when not zero, a timeout has occured, the user pressed
              Escape or F10}
procedure ReadLineActive; far; assembler;
asm
{
    call CheckTimeout;
    jc @2;
}
    mov dx, LPTAddr;
    inc dx;
    in al, dx;
@1: mov ah, al;
    in al, dx;
    cmp ah, al;
    jne @1;
    xor al, ((sbAtnNormal + sbClkNormal + sbDataSwapped) shl 4);
    shr al, 4;
    mov ah, al;
    and al, (sbClkNormal + sbDataSwapped);
    clc;
@2:
end;

{Set the error status
  Input : AL: error code to set error status to
  Output: AL: empty data byte
          CF: set to indicate error}
procedure Error; far; assembler;
asm
    or Status, al;
    call InterruptOn;
    call AtnHi;
    call ClkHi;
    call DataHi;
    xor al, al;
    stc;
end;

{Determine the default parallel port control byte for a serial cable}
procedure InitSerial;
begin
  case SerialCable of
    scMultitask: Serial := sbDataSwapped;
    scActive: Serial := (sbAtnNormal + sbClkNormal + sbResetSwapped);
  else
    Serial := sbResetNormal;
  end;
end;

{Initialize LPT port bit variables}
procedure InitLPTBits;
begin
  DataBit := sbDataNormal;
  ResetBit := sbResetNormal;
  case SerialCable of
    scMultitask:
    begin
      DataBit := sbDataSwapped;
      ResetBit := sbResetSwapped;
    end;
    scActive:
    begin
      DataBit := sbDataSwapped;
      ResetBit := sbResetSwapped;
    end;
  end;
  ClkDataBits := DataBit or sbClkNormal;
end;

{Reset the parallel port of a serial cable}
procedure ResetLPTPort; assembler;
asm
    push LPTAddr;
    call ValidLPTAddr;
    or al, al;
    je @1;
    mov dx, LPTAddr;
    add dx, 2;
    call InitSerial;
    mov al, Serial;
    out dx, al;
@1:
end;

{Reset all external Commodore drives on a serial cable}
procedure ResetDrives; assembler;
asm
    push LPTAddr;
    call ValidLPTAddr;
    or al, al;
    je @2;
    call InitSerial;
    mov dx, LPTAddr;
    add dx, 2;
    mov al, ResetBit;
    cmp SerialCable, scMultitask;
    jne @3;
    or al, Serial;
    jmp @4;
@3: not al;
    and al, Serial;
@4: out dx, al;
    mov cx, 100;
@1: mov ax, 1000;
    call Delay;
    loop @1;
    mov al, Serial;
    out dx, al;
@2:
end;

{Determine if the parallel port is in the interval of valid addresses
  Input : Addr: the port address
  Output: when True, the port address is valid}
function ValidLPTAddr(Addr: Word): Boolean; assembler;
asm
    mov al, False;
{
    cmp DisableLPTPorts, False;
    jne @1;
}
    mov dx, Addr;
    cmp dx, $0200;
    jb @1;
    cmp dx, $F800;
    ja @1;
    test dx, 3;
    jne @1;
    inc al;
@1:
end;

{Initialize both parallel ports if they - and the corresponding cable - are
  not defined as 'None'; ECP ports must be set to Byte mode, and all kinds
  of ports must be set to input mode}
procedure InitLPTPorts; assembler;
asm
    cmp SerialCable, scNone;
    je @1;
    mov dx, LPTAddr;
    call @2;
    jc @1;
    cmp ParallelCable, pcNone;
    je @1;
    mov dx, ParLPTAddr;
    cmp dx, LPTAddr;
    je @1;
    call @2;

       { decide between the Serial and ParLPTCtrl shadow
         registers, if and only if the both LPT port
         addresses are not identical }
    mov ParLPTCtrl, al;

    jmp @1;
@2: push dx;
    push dx;
    call ValidLPTAddr;
    pop dx;
    or al, al;
    stc;
    je @3;
    add dx, 2;
    in al, dx;
    and al, not sbBidir;
    out dx, al;
    sub dx, 2;
    mov al, $FF;
    out dx, al;
    add dx, ECPAddrOffs;
    add dx, 2;
    mov al, $34;
    out dx, al;
    sub dx, ECPAddrOffs;
    in al, dx;
    or al, sbBidir;
    out dx, al;
@3: clc;
    retn;
@1:
end;

{Initialize the variables related to data transfer from or to the external
  Commodore drive}
{procedure InitTransfer(Warnings: Boolean);}
procedure InitTransfer;
const
  ReadLines: array [scNormal..scActive] of TProc = (ReadLineNormal, ReadLineExtended, ReadLineMultitask, ReadLineActive);
begin
  (*
  if DelayValue = 0 then ComputeDelay else CopyDelayValue := DelayValue;
  LPTAddr := LPTAddresses[LPTNum];
  ParLPTAddr := LPTAddresses[ParLPTNum];
  LPTMode := LPTModes[LPTNum];
  ParLPTMode := LPTModes[ParLPTNum];
  *)

  ComputeDelay;

  InitLPTBits;
  AtnLo := AtnLoNormal;
  AtnHi := AtnHiNormal;
  ClkLo := ClkLoNormal;
  ClkHi := ClkHiNormal;
  DataLo := DataLoNormal;
  DataHi := DataHiNormal;
  case SerialCable of
    scMultitask:
    begin
      DataLo := DataHiNormal;
      DataHi := DataLoNormal;
    end;
    scActive:
    begin
      AtnLo := AtnHiNormal;
      AtnHi := AtnLoNormal;
      ClkLo := ClkHiNormal;
      ClkHi := ClkLoNormal;
    end;
  end;

  ReadLine := ReadLines[SerialCable];
end;

{Switch the parallel port of the hybrid/parallel cable to input}
procedure ParallelInput; assembler;
asm
{
    mov al, CopyCableMode;
    cmp al, cmHybrid;
    je @1;
    cmp al, cmParallel;
    jne @2;
}
@1: mov dx, ParLPTAddr;
    mov al, $FF;
    out dx, al;
    cmp dx, LPTAddr;
    jne @3;
    add dx, 2;
    mov al, Serial;
    or al, sbBidir;
    mov Serial, al;
    jmp @4

@3: add dx, 2;
    mov al, ParLPTCtrl;
    or al, sbBidir;
    mov ParLPTCtrl, al;
@4: out dx, al;
@2:

end;

{Switch the parallel port of the hybrid/parallel cable to output}
procedure ParallelOutput; assembler;
asm
{
    mov al, CopyCableMode;
    cmp al, cmHybrid;
    je @1;
    cmp al, cmParallel;
    jne @2;
}
@1: mov dx, ParLPTAddr;
    cmp dx, LPTAddr;
    jne @3;
    add dx, 2;
    mov al, Serial;
    and al, not sbBidir;
    mov Serial, al;
    jmp @4

@3: add dx, 2;
    mov al, ParLPTCtrl;
    and al, not sbBidir;
    mov ParLPTCtrl, al;
@4: out dx, al;
@2:
end;

{Receive a byte from the external CBM drive
  Output: AL: the byte received}
procedure Receive; assembler;
asm
    push plHigh;
    call InterruptOff;
    xor cl, cl;
    mov InData, cl;
    call ClkHi;
@1: call ReadLine;
    jc @11;
    test al, sbClkNormal;
    jne @1;
@6: mov bx, 20;
    call DataHi;
@3: dec bx;
    je @2;
    mov ax, 20;
    call Delay;
    call ReadLine;
    jc @11;
    test al, sbClkNormal;
    je @3;
    jmp @4;
@2: or cl, cl;
    je @5;
@11:mov al, 2;
    jmp Error;
@5: call DataLo;
    call ClkHi;
    or Status, ssEOF;
    mov ax, 60;
    call Delay;
    inc cl;
    jmp @6;
@4: mov bx, 8;
@7: call ReadLine;
    jc @11;
    test al, sbClkNormal;
    jne @7;
    test al, DataBit;
    stc;
    je @8;
    clc;
@8: rcr InData, 1;
@9: call ReadLine;
    jc @11;
    test al, sbClkNormal;
    je @9;
    dec bx;
    jne @7;
    call DataLo;
    test Status, ssEOF;
    je @10;
    mov ax, 150;
    call Delay;
    call ClkHi;
    call DataHi;
@10:mov al, InData;
    call InterruptOn;
end;

{Send a byte to the external CBM drive}
procedure SendReal; far; assembler;
asm
    push plHigh;
    call InterruptOff;
    call DataHi;
    mov ax, 25;
    call Delay;
    call ReadLine;
    jc @5;
    test al, DataBit;
    je @1;
    call ClkHi;
    cmp EOI, False;
    je @2;
@3: call ReadLine;
    jc @5;
    test al, DataBit;
    jne @3;
@4: call ReadLine;
    jc @5;
    test al, DataBit;
    je @4;
@2: call ReadLine;
    jc @5;
    test al, DataBit;
    jne @2;
    call ClkLo;
    mov bx, 8;
@8: mov ax, 25;
    call Delay;
    call ReadLine;
    jc @5;
    test al, DataBit;
    jne @5;
    mov ax, 100;
    call Delay;
    shr OutData, 1;
    jc @6;
    call DataLo;
    jne @7;
@6: call DataHi;
@7: call ClkHi;
    mov ax, 25;
    call Delay;
    call DataHi;
    call ClkLo;
    dec bx;
    jne @8;
    mov cx, 10;
@10:mov ax, 20;
    call Delay;
    call ReadLine;
    jc @5;
    test al, DataBit;
    jne @9;
    loop @10;
@5: mov al, 3;
    jmp Error;
@1: mov al, ssNoDevice;
    jmp Error;
@9: call InterruptOn;
end;

{Buffer a byte to be sent to the external CBM drive and send the previous
  byte
  Input : AL: the byte to send}
procedure Send; assembler;
asm
    cmp More, False;
    jne @1;
    mov More, True;
    jmp @2;
@1: push ax;
    call SendReal;
    pop ax;
@2: mov OutData, al;
end;

{Determine if a device is present on the serial bus
  Output: when True, there is a device on the bus}
function CheckDevice: Boolean; assembler;
asm
    push plHigh;
    call InterruptOff;
    push ax;
    push si;
    mov Status, 0;
{    mov cx, 20000; }
    mov cx, ChkDevSpeed;
    call ResetLPTPort;
@5: call AtnLo;
    mov ax, 100;
    call Delay;
    call ReadLine;
    push ax;
    call AtnHi;
    pop ax;
    test al, DataBit;
    je @3;
    mov ax, 100;
    call Delay;
    call ReadLine;
    test al, DataBit;
    je @4;
@3: loop @5;
    mov al, ssNoDevice;
    call Error;
@4: call AtnHi;
    call ClkHi;
    call DataHi;
    pop si;
    pop ax;
    call InterruptOn;
    mov al, False;
    test Status, ssNoDevice;
    jne @2;
    inc al;
@2:
end;

{Send a command (single byte with ATN low) to the external CBM drive
  Input : AL: command byte to be sent}
procedure Command; assembler;
asm
{
    push ax;
    push True;
    call SetTimeout;
    pop ax;
}
    push ax;
    cmp More, False;
    je @1;
    mov EOI, True;
    call SendReal;
    mov More, False;
    mov EOI, False;
@1: pop ax;
    mov OutData, al;
    call DataHi;
    call AtnLo;
    push plHigh;
    call InterruptOff;
    call ClkLo;
    call DataHi;
    mov ax, 500;
    call Delay;
    call SendReal;
end;

{Send a TALK command to the external CBM drive}
procedure Talk; assembler;
asm
    mov al, CBMDevNum;
    or al, $40;
    call Command;
end;

{Send a LISTEN command to the external CBM drive}
procedure Listen; assembler;
asm
    mov al, CBMDevNum;
    or al, $20;
    call Command;
end;

{Send a TALK secondary address to the external CBM drive
  Input : AL: secondary address to be sent}
procedure TalkSec; assembler;
asm
    mov OutData, al;
    test Status, ssNoDevice;
    jne @3;
    push plHigh;
    call InterruptOff;
    call ClkLo;
    call DataHi;
    mov ax, 500;
    call Delay;
    call SendReal;
    push plHigh;
    call InterruptOff;
    call DataLo;
    call AtnHi;
    call ClkHi;
    xor cx, cx;
@1: call ReadLine;
    test al, sbClkNormal;
    je @2;
    loop @1;
@2: mov ax, 500;
    call Delay;
    call InterruptOn;
@3:
end;

{Send a LISTEN secondary address to the external CBM drive
  Input : AL: secondary address to be sent}
procedure ListenSec; assembler;
asm
    mov OutData, al;
    test Status, ssNoDevice;
    jne @1;
    push plHigh;
    call InterruptOff;
    call ClkLo;
    call DataHi;
    mov ax, 500;
    call Delay;
    call SendReal;
    call AtnHi;
@1:
end;

{Send an UNTALK command to the external CBM drive}
procedure Untalk; assembler;
asm
    push plHigh;
    call InterruptOff;
    call ClkLo;
    call AtnLo;
    mov al, $5F;
    call Command;
    push plHigh;
    call InterruptOff;
    call AtnHi;
    mov ax, 200;
    call Delay;
    call ClkHi;
    call DataHi;
    call InterruptOn;
end;

{Send an UNLISTEN command to the external CBM drive}
procedure Unlisten; assembler;
asm
    push plHigh;
    call InterruptOff;
    mov al, $3F;
    call Command;
    push plHigh;
    call InterruptOff;
    call AtnHi;
    mov ax, 200;
    call Delay;
    call ClkHi;
    call DataHi;
    call InterruptOn;
end;

{Open a file on a disk or send a disk command to the external CBM drive
  Input : SecAddr: secondary address of the file or $6F if disk command
          Command: the file name or disk command to be sent
          Timeout: when True, timeouts are handled}
{ procedure OpenCBMChannel(SecAddr: Byte; const Command: string; Timeout: Boolean); }
procedure OpenCBMChannel(SecAddr: Byte; const Command: string);
begin
{  SetTimeout(Timeout); }
  asm
    test Status, ssNoDevice;
    jne @1;
    les di, Command;
    cmp byte ptr es:[di], 0;
    je @1;
    mov Status, 0;
    mov More, False;
    mov EOI, False;
    call Listen;
    mov al, SecAddr;
    cmp al, CmdSecAddr;
    je @3;
    mov FileSecAddr, al;
    or al, $F0;
@3: call ListenSec;
    test Status, ssNoDevice;
    jne @1;
    xor bx, bx;
    les di, Command;
    mov cl, es:[di];
    xor ch, ch;
@2: inc bx;
    push bx;
    push cx;
    mov al, es:[di][bx];
    call Send;
    pop cx;
    pop bx;
    loop @2;
    call Unlisten;
@1:
  end;
{  SetTimeout(False); }
end;

{Close a file on the disk in the external CBM drive}
procedure CloseCBMChannel;
begin
{  SetTimeout(True); }
  asm
    call Listen;
    mov al, FileSecAddr;
    and al, $0F;
    or al, $E0;
    call ListenSec;
    call Unlisten;
  end;
{  SetTimeout(False); }
end;

{Receive the disk error from the external CBM drive
  Input : Error: the string to contain the error message
          Check: when True, the serial bus status is checked and the
                 process of reading the error message is skipped if the
                 drive is off or a timeout occured
          Off: when True, the turbo program running in the external CBM
               drive is shut down before reading the status
          Timeout: when True, timeouts are handled
  Output: when False, an error occured}
{function ReadCBMError(var Error: string; Check, Off, Timeout: Boolean): Boolean;}
function ReadCBMError(var Error: string; Check: Boolean): Boolean;
var
  C             : Char;
begin
  asm
    mov ax, 10000;
    call Delay;
  end;
{  if Off and (CopyTransferMode <> tmNormal) then TurboOff; }
  Error := '';
  if not Check or (Status and ssNoDevice = 0) then
  begin
{    SetTimeout(Timeout); }
    asm
      mov Status, 0;
      mov More, False;
      mov EOI, False;
      call Talk;
      mov al, CmdSecAddr;
      call TalkSec;
      test Status, ssNoDevice;
      jne @1;
      les di, Error;
      xor bx, bx;
      cld;
  @2: push bx;
      push cx;
      call Receive;
      pop cx;
      pop bx;
      cmp Status, 0;
      jne @3;
      inc bl;
      mov es:[di][bx], al;
      cmp bl, MaxStrLen;
      je @3;
      loop @2;
  @3: mov es:[di], bl;
      call Untalk;
  @1:
    end;
{    SetTimeout(False); }
  end;
  if Status and ssTimeout > 0 then
  begin
    Error := 'Timeout detected';
  end
  else
  begin
    if Status and ssNoDevice > 0 then Error := 'Drive ' + LeadingSpace(CBMDevNum, 0) + ': not present';
  end;
  if (Error <> '') and (Error[Length(Error)] = chReturn) then Dec(Error[0]);
  ReadCBMError := ((Error = '') or ((Length(Error) > 2) and (Error[1] = '0')));
end;

{Determine the mode of an LPT port
  Input : Address: the address of the LPT port
  Output: LPT port mode}
function GetLPTMode(Address: Word): Byte; assembler;
asm
    push Address;
    call ValidLPTAddr;
    xor cl, cl;
    or al, al;
    je @20;
    mov dx, Address;
    add dx, 2;
    mov bx, dx;
    mov cl, pmECP;
    add dx, ECPAddrOffs;
    in al, dx;
    and al, $F8;
    mov ch, al;
    mov al, $34;
    out dx, al;
    mov dx, bx;
    mov al, $C6;
    cmp SerialCable, scExtended;
    jbe @17;
    mov al, $CA;
@17:out dx, al;
    add dx, ECPAddrOffs;
    in al, dx;
    cmp al, $35;
    jne @2;
    mov al, $D4;
    out dx, al;
    sub dx, 2;
    in al, dx;
    mov al, $AA;
    out dx, al;
    jmp @11;
@11:in al, dx;
    cmp al, $AA;
    jne @2;
    mov al, $55;
    out dx, al;
    jmp @12;
@12:in al, dx;
    cmp al, $55;
    jne @2;
    add dx, 2;
    mov al, $35;
    out dx, al;
    mov al, ch;
    out dx, al;
    jmp @1;
@2: mov dx, bx;
    add dx, ECPAddrOffs;
    mov al, ResetBit;
    cmp SerialCable, scExtended;
    jbe @18;
    or al, ch;
    jmp @19;
@18:not al;
    and al, ch;
@19:out dx, al;
    mov cl, pmEPP;
    mov al, $EF;
    call @5;
    jnc @1;
    xor al, al;
@15:push ax;
    test al, ResetBit;
    je @16;
    call @5;
    pop ax;
    jnc @1;
@16:inc al;
    jne @15;
    mov cl, pmSPP;
    mov dx, bx;
    in al, dx;
    or al, sbBidir;
    out dx, al;
    sub dx, 2;
    mov al, $AA;
    out dx, al;
    jmp @9;
@9: in al, dx;
    cmp al, $AA;
    jne @3;
    mov al, $55;
    out dx, al;
    jmp @10;
@10:in al, dx;
    cmp al, $55;
    je @13;
@3: mov cl, pmPS2;
@13:mov dx, bx;
    in al, dx;
    and al, not sbBidir;
    out dx, al;
    sub dx, 2;
    mov al, $AA;
    out dx, al;
    jmp @14;
@14:in al, dx;
    cmp al, $AA;
    jne @4;
    mov al, $55;
    out dx, al;
    jmp @8;
@8: in al, dx;
    cmp al, $55;
    je @1;
@4: xor cl, cl;
    jmp @1;
@5: mov dx, bx;
    out dx, al;
    inc dx;
    call @7;
    mov al, $AA;
    out dx, al;
    call @7;
    in al, dx;
    cmp al, $AA;
    stc;
    jne @6;
    call @7;
    mov al, $55;
    out dx, al;
    call @7;
    in al, dx;
    cmp al, $55;
    stc;
    jne @6;
    clc;
@6: retn;
@7: push ax;
    push dx;
    mov dx, bx;
    dec dx;
    in al, dx;
    or al, $01;
    out dx, al;
    and al, $FE;
    out dx, al;
    pop dx;
    pop ax;
    retn;
@1: mov dx, bx;
    mov al, ResetBit;
    out dx, al;
@20:mov al, cl;
end;

procedure PrintStr(const S: String); assembler;
asm
    push ds;
    lds si, S;
    cld;
    lodsb;
    xor ah, ah;
    xchg ax, cx;
    mov ah, $40;
    mov bx, 1;
    mov dx, si;
    int $21;
    pop ds;
end;

{Convert a long integer into a decimal string with thousands separator
  Input : D: the long integer
          L: number of digits to put into the string
  Output: the decimal string}
function DecStr(D: Longint; L: Byte): string;
var
  S             : string;
begin
  Str(D:L,S);
  DecStr:=S;
end;

{Convert a long integer into a hexadecimal string
  Input : D: the long integer
          L: number of digits to put into the string
  Output: the hexadecimal string}
function HexaStr(D: Longint; L: Byte): string;
var
  I             : Byte;
  S             : string;
begin
  S := '';
  for I := L - 1 downto 0 do S := S + HexaNum[(D shr (I * 4) and $0F) + 1];
  HexaStr := S;
end;

{Convert a long integer into a decimal string with leading blanks
  Input : Num: the long integer to be converted
          DigitNum: number of digits to put into the string
  Output: the decimal string}
function LeadingSpace(Num: Longint; DigitNum: Word): string;
var
  S             : string;
begin
  Str(Num:DigitNum, S);
  LeadingSpace := S;
end;

end.
