
{*************************************************}
{                 Joe Forster/STA                 }
{                                                 }
{                   CLIPBRD.PAS                   }
{                                                 }
{        The Star Commander clipboard unit        }
{*************************************************}

unit Clipbrd;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O+,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Dialogs, DOS,
  Base2, Constant, ExtFiles, LowLevel;

const
{Clipboard modes}
  cbDisk        = 0;
  cbEMS         = 1;
  cbXMS         = 2;
  cbWindows     = 3;
{Clipboard file name}
  ClipFilename  = 'SC.CLP';
{Windows clipboard format used for OEM text}
  WinClipFormat = $07;
{Granularity of the Windows clipboard in bytes}
  WinClipGrain  = $20;

procedure FarInc(var P: Pointer; Add: Longint);
procedure DegranulateClipboard(Data: Pointer; var Len: Longint);
procedure InitClipboard;
procedure DoneClipboard;
procedure ClearClipboard;
function GetClipboardSize: Longint;
function GetClipboard(Data: Pointer; MaxLen: Longint): Boolean;
function PutClipboard(Data: Pointer; Len: Longint): Boolean;

var
  ClipMode,
  ClipGrain,
  ClipGrainMask : Byte;
  ClipName      : string;
  ClipFile      : ExtFile;

implementation

procedure FarInc(var P: Pointer; Add: Longint); assembler;
asm
    mov ax, word ptr Add[0];
    mov dx, word ptr Add[2];
    push ax;
    mov cx, 4;
@1: shr dx, 1;
    rcr ax, 1;
    loop @1;
    mov dx, ax;
    pop ax;
    and ax, $000F;
    les di, P;
    add es:[di][0], ax;
    add es:[di][2], dx;
end;

{Find out the actual length of the data read from the clipboard, by locating
  the first zero byte in the last grain
  Input : Data: pointer to the memory area holding the clipboard data
          Len: the length of the clipboard data}
procedure DegranulateClipboard(Data: Pointer; var Len: Longint); assembler;
asm
    cmp ClipGrain, 1;
    jbe @2;
    les di, Len;
    mov ax, word ptr es:[di][0];
    mov dx, word ptr es:[di][2];
    mov cx, ax;
    sub ax, 1;
    sbb dx, 0;
    and al, ClipGrainMask;
    sub cx, ax;
    les di, Data;
    add di, ax;
    adc dx, 0;
    push di;
    push cx;
    mov cx, 4;
@1: shr dx, 1;
    rcr di, 1;
    loop @1;
    mov dx, di;
    pop cx;
    pop di;
    and di, $000F;
    mov ax, es;
    add ax, dx;
    mov es, ax;
    mov bx, cx;
    xor al, al;
    cld;
    repne scasb;
    jne @3;
    inc cx;
@3: les di, Len;
    sub word ptr es:[di][0], cx;
    sbb word ptr es:[di][2], 0;
@2:
end;

{Initialize the clipboard}
procedure InitClipboard;
begin
  ClipMode := cbDisk;
  ClipGrain := 1;
  if not DisableWinClipboard then
  begin
    asm;
      mov ax, $1700;
      int $2F;
      cmp ax, $1700;
      je @1;
      mov ClipMode, cbWindows;
  @1:
    end;
  end;
  case ClipMode of
    cbDisk:
    begin
      ClipName := GetTempPath;
      ClipName := AddToPath(ClipName, ClipFileName, chDirSep);
    end;
    cbWindows: ClipGrain := WinClipGrain;
  end;
  ClipGrainMask := (ClipGrain - 1) xor $FF;
end;

{Dispose of the clipboard}
procedure DoneClipboard;
begin
  case ClipMode of
    cbDisk: ClearClipboard;
    cbWindows:
  end;
end;

{Clear the clipboard contents}
procedure ClearClipboard;
begin
  case ClipMode of
    cbDisk:
    begin
      InOutRes := 0;
      ClockOff;
      if FileExists(ClipName, False) then LongErase(ClipName);
      InOutRes := 0;
      ClockOn;
    end;
    cbWindows:
    begin
      asm
        mov ax, $1701;
        int $2F;
        mov ax, $1702;
        int $2F;
        mov ax, $1708;
        int $2F;
      end;
    end;
  end;
end;

{Get the length of data stored in the clipboard
  Output: the length of clipboard data}
function GetClipboardSize: Longint;
var
  E             : ExtSearchRec;
begin
  case ClipMode of
    cbDisk:
    begin
      InOutRes := 0;
      ClockOff;
      LongFindFirst(ClipName, AnyFile, E);
      GetClipboardSize := LonglongintToLongint(E.LongSize);
      LongFindClose(E);
      InOutRes := 0;
      ClockOn;
    end;
    cbWindows:
    begin
      asm
        mov ax, $1701;
        int $2F;
        mov ax, $1704;
        mov dx, WinClipFormat;
        int $2F;
        mov word ptr @Result[0], ax;
        mov word ptr @Result[2], dx;
        mov ax, $1708;
        int $2F;
      end;
    end;
  end;
end;

{Read data from the clipboard
  Input : Data: pointer to the memory area to load the clipboard into
          MaxLen: the maximum length of data to be read
  Output: when False, an error occured}
function GetClipboard(Data: Pointer; MaxLen: Longint): Boolean;
var
  O             : Boolean;
  W             : Word;
  L             : Longint;
begin
  O := False;
  case ClipMode of
    cbDisk:
    begin
      InOutRes := 0;
      ClockOff;
      if LongOpenFile(ClipName, ClipFile, fmReadOnly) = 0 then
      begin
        L := ExtFileSize(ClipFile);
        if (L > 0) and (L <= MaxLen) then
        begin
          while MaxLen > 0 do
          begin
            if MaxLen > TSmallBufSize then W := TSmallBufSize else W := MaxLen;
            Dec(MaxLen, W);
            ExtBlockRead(ClipFile, PSmallBuf(Data)^, W);
            FarInc(Data, W);
          end;
          ExtClose(ClipFile);
          O := (IOResult = 0);
        end;
      end;
      InOutRes := 0;
      ClockOn;
    end;
    cbWindows:
    begin
      asm
        mov ax, $1701;
        int $2F;
        mov ax, $1704;
        mov dx, WinClipFormat;
        int $2F;
        mov word ptr L[0], ax;
        mov word ptr L[2], dx;
      end;
      if (L > 0) and (L <= MaxLen) then
      begin
        asm
          mov ax, $1705;
          mov dx, WinClipFormat;
          les bx, Data;
          int $2F;
          mov O, al;
        end;
      end;
      asm;
        mov ax, $1708;
        int $2F;
      end;
    end;
  end;
  GetClipboard := O;
end;

{Write data into the clipboard
  Input : Data: pointer to the memory area to save into the clipboard
          Len: the length of data to be written
  Output: when False, an error occured}
function PutClipboard(Data: Pointer; Len: Longint): Boolean;
var
  O             : Boolean;
  W             : Word;
  P             : Pointer;
begin
  O := False;
  if Len = 0 then
  begin
    ClearClipboard;
    O := True;
  end
  else
  begin
    case ClipMode of
      cbDisk:
      begin
        InOutRes := 0;
        ClockOff;
        if LongOpenFile(ClipName, ClipFile, fmWriteOnly) = 0 then
        begin
          while Len > 0 do
          begin
            if Len > TSmallBufSize then W := TSmallBufSize else W := Len;
            Dec(Len, W);
            ExtBlockWrite(ClipFile, PSmallBuf(Data)^, W);
            FarInc(Data, W);
          end;
          ExtClose(ClipFile);
          O := (IOResult = 0);
        end;
        InOutRes := 0;
        ClockOn;
      end;
      cbWindows:
      begin
        P := Data;
        FarInc(P, Len);
        asm
          les di, P;
          mov al, byte ptr es:[di];
          push ax;
          mov byte ptr es:[di], 0;
          mov ax, $1701;
          int $2F;
          mov ax, $1703;
          mov dx, WinClipFormat;
          les bx, Data;
          mov cx, word ptr Len[0];
          mov si, word ptr Len[2];
          inc cx;
          jne @1;
          inc si;
      @1: int $2F;
          mov O, al;
          mov ax, $1708;
          int $2F;
          pop ax;
          les di, P;
          mov byte ptr es:[di], al;
        end;
      end;
    end;
  end;
  PutClipboard := O;
end;

begin
  DegranulateClipboardProc := DegranulateClipboard;
  GetClipboardSizeProc := GetClipboardSize;
  GetClipboardProc := GetClipboard;
  PutClipboardProc := PutClipboard;
end.
