
{*************************************************}
{                 Joe Forster/STA                 }
{                                                 }
{                    COMMON.PAS                   }
{                                                 }
{          The Star Commander common unit         }
{*************************************************}

unit Common;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Dialogs, Drivers, Objects, Views,
  Base1, Base2, ExtFiles, LowLevel, Help;

const
{Tab size}
  TabSize       : Word = 8;
{EOL modes}
  elCRLF        = 0;
  elCR          = 1;
  elLF          = 2;
  elUser        = 3;
  elFixLen      = 4;
  elNext        = 5;
{Text symbols}
  tsEndOfFile   = #4;
  tsEndOfLine   = #20;
  tsTab         = #26;
  tsSpace       = #250;
{Number of bytes checked through at the beginning of the file to
  autodetect line feed mode and Tab translation mode}
  CheckDataLen  = 4096;
{Maximum length of a line in the viewer/editor without being wrapped}
  MaxWrapLen    = 2048;
{End-of-line characters for different line feed modes}
  EOLChars      : array [0..4] of Char = (chCR, chCR, chLF, #0, #0);
{Screen code lowercase/uppercase to ASCII converter table (only for the
  lower 128 characters)}
  SCRtoASCLower : array [0..127] of Byte = (
    $40, $61, $62, $63, $64, $65, $66, $67, $68, $69, $6A, $6B, $6C, $6D, $6E, $6F,
    $70, $71, $72, $73, $74, $75, $76, $77, $78, $79, $7A, $5B, $A9, $5D, $18, $1B,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $C4, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $C5, $F9, $B3, $F9, $F9,
    $5F, $DD, $DC, $C4, $C4, $B3, $F9, $D9, $F9, $F9, $B3, $C3, $F9, $C0, $BF, $C4,
    $DA, $C1, $C2, $B4, $B3, $DD, $DE, $C4, $DF, $DC, $FB, $F9, $F9, $D9, $F9, $F9);
{Screen code uppercase/graphics to ASCII converter table (only for the
  lower 128 characters)}
  SCRtoASCUpper : array [0..127] of Byte = (
    $40, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $5B, $A9, $5D, $18, $1B,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $C4, $06, $B3, $C4, $C4, $C4, $C4, $B3, $B3, $BF, $C0, $D9, $C0, $5C, $2F, $DA,
    $BF, $07, $C4, $03, $B3, $DA, $58, $09, $05, $B3, $04, $C5, $F9, $B3, $E3, $F9,
    $5F, $DD, $DC, $C4, $C4, $B3, $F9, $D9, $F9, $F9, $B3, $C3, $F9, $C0, $BF, $C4,
    $DA, $C1, $C2, $B4, $B3, $DD, $DE, $C4, $DF, $DC, $FB, $F9, $F9, $D9, $F9, $F9);
{Screen code lowercase/uppercase to extended ASCII converter table (only
  for the lower 128 characters; used when the C64 character set is on)}
  SCRtoExtLower : array [0..127] of Byte = (
    $40, $61, $62, $63, $64, $65, $66, $67, $68, $69, $6A, $6B, $6C, $6D, $6E, $6F,
    $70, $71, $72, $73, $74, $75, $76, $77, $78, $79, $7A, $5B, $A9, $5D, $18, $1B,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $80, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $9B, $9C, $9D, $AA, $AB,
    $E0, $E1, $E2, $E3, $E4, $E5, $E6, $E7, $E8, $AC, $EA, $EB, $EC, $ED, $EE, $EF,
    $F0, $F1, $F2, $F3, $F4, $F5, $F6, $F7, $A0, $A1, $FB, $A3, $A4, $A5, $A6, $A7);
{Screen code uppercase/graphics to extended ASCII converter table (only
  for the lower 128 characters; used when the C64 character set is on)}
  SCRtoExtUpper : array [0..127] of Byte = (
    $40, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $5B, $A9, $5D, $18, $1B,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $80, $81, $82, $83, $84, $85, $86, $87, $88, $89, $8A, $8B, $8C, $8D, $8E, $8F,
    $90, $91, $92, $93, $94, $95, $96, $97, $98, $99, $9A, $9B, $9C, $9D, $9E, $9F,
    $E0, $E1, $E2, $E3, $E4, $E5, $E6, $E7, $E8, $E9, $EA, $EB, $EC, $ED, $EE, $EF,
    $F0, $F1, $F2, $F3, $F4, $F5, $F6, $F7, $A0, $A1, $A2, $A3, $A4, $A5, $A6, $A7);
{ASCII to screen code converter table}
  ASCtoSCR      : array [0..255] of Byte = (
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $00, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $5B, $20, $5D, $20, $20,
    $20, $01, $02, $03, $04, $05, $06, $07, $08, $09, $0A, $0B, $0C, $0D, $0E, $0F,
    $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $1A, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20);

type
  TTitle        = object(TFalseMenu)
    Name        : string;
    procedure Draw; virtual;
  end;
  PTitle        = ^TTitle;
  TTextInput    = object(TInputLine)
    CodeInput   : PInputLine;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;
  PTextInput    = ^TTextInput;
  TCodeInput    = object(TInputLine)
    TextInput   : PInputLine;
    constructor Init(var Bounds: TRect; ALen, AMaxLen: Integer);
    procedure HandleEvent(var Event: TEvent); virtual;
  end;
  PCodeInput    = ^TCodeInput;
{Character case converter function}
  TCharCaseFunc = function(Ch: Char): Char;
  PCharCaseFunc = ^TCharCaseFunc;

var
  CtrlShiftPressed,
  AltShiftPressed,
  Standard,
  StdMouse,
  StdMouseReverse,
  TitlePrinted,
  ImageReadOnly,
  ExtMode,
  CopyGEOSFormat,
  GEOSFormat    : Boolean;
  StdScreenCol,
  PanelMode,
  DiskType,
  MaxTrack,
  Track,
  Sector,
  SideTrack,
  SideSector,
  RecordLen,
  DirTrack,
  DirTrack2,
  DirSector,
  Number,
  EntryPos,
  ViewMode      : Byte;
  DirSep        : Char;
  NameEnd,
  DirPos,
  ReadDirPos,
  ImageSize     : Word;
  OrigSize,
  _Free,
  FileTime,
  ImagePos,
  HeaderPos,
  PrevSize      : Longint;
  PanelDir      : PDirBuffer;
  PanelNames    : PNameBuffer;
  UpCaseFunc,
  LoCaseFunc,
  InvCaseFunc   : PCharCaseFunc;
  Image         : ExtFile;
  _Label        : string[24];
  TapeName      : string[32];
  ReadName,
  FullName,
  ImageName,
  ImagePath,
  RealImagePath,
  ImageDir,
  LeftPath,
  RightPath     : string;
  BAM,
  BAM2,
  BAM3,
  DirBuffer,
  DataBuffer    : TBlock;

procedure SetStandardParams;
function TextToHex(const T: string; B, L: Boolean): string;
function HexToText(const T: string; B: Boolean): string;
procedure SetupCaseConversion;
procedure MakeFullName;
procedure CheckDiskType;
function SectorNum(T: Byte): Byte;
function DiskPos(T, S: Byte): Longint;
procedure ReadDiskBlock(T, S: Byte; Buffer: PBlock);
procedure WriteDiskBlock(T, S: Byte; Buffer: PBlock);
procedure ReadTapeBlock(S: Word; Buffer: PBlock);
procedure WriteTapeBlock(S: Word; Buffer: PBlock);
function GetBAMOffset(T: Byte): Word;
function GetNamePtr(Pos: Integer): PString;
function InsName(const Name: string): Integer;
function ReadCBMEntry(var Entry: TDirEntry): Boolean;
procedure Prepare;
function OpenImage(Write: Boolean): Integer;

implementation

uses
  App, DOS, Menus, Validate,
  Constant;

var
  OnlineHelp    : POnlineHelp;

procedure SetStandardParams; assembler;
asm
    push ds;
    push ds;
    pop es;
    mov ds, PrefixSeg;
    mov si, $0081;
@2: lodsb;
    or al, al;
    je @1;
    cmp al, chCR;
    jne @2;
    xor al, al;
    jmp @3;
@1: dec si;
    mov di, Offset(SochaSign);
    mov cx, 9;
    repe cmpsb;
    jne @3;
    mov al, True;
    mov ah, [si];
    mov bx, [si][3];
@3: pop ds;
    mov Standard, al;
    mov StdScreenCol, ah;
    mov StdMouseReverse, bl;
    mov StdMouse, bh;
end;

function TextToHex(const T: string; B, L: Boolean): string;
var
  I,
  J             : Integer;
  S             : string;
begin
  S := '';
  J := Length(T);
  if L and (J > 21) then J := 21;
  if B then
  begin
    for I := 1 to J do S := S + HexaStr(Ord(T[I]), 2) + stSpace;
  end
  else
  begin
    case ViewMode of
      vmASCII: for I := 1 to J do S := S + HexaStr(Ord(T[I]), 2) + stSpace;
      vmPETSCII: for I := 1 to J do S := S + HexaStr(ASCtoPET[Ord(T[I])], 2) + stSpace;
      vmScreen: for I := 1 to J do S := S + HexaStr(ASCtoSCR[Ord(T[I])], 2) + stSpace;
    end;
  end;
  TextToHex := S;
end;

function HexToText(const T: string; B: Boolean): string;
var
  I,
  J             : Integer;
  S             : string;
begin
  S := '';
  J := (Length(T) div 3);
  S[0] := Chr(J);
  if B then
  begin
    for I := 1 to J do S[I] := Chr(HexaEval(Copy(T, I * 3 - 2, 2), NumOk));
  end
  else
  begin
    case ViewMode of
      vmASCII: for I := 1 to J do S[I] := Chr(HexaEval(Copy(T, I * 3 - 2, 2), NumOk));
      vmPETSCII: for I := 1 to J do S[I] := Chr(PETtoASCLower[HexaEval(Copy(T, I * 3 - 2, 2), NumOk)]);
      vmScreen: for I := 1 to J do S[I] := Chr(SCRtoASCLower[HexaEval(Copy(T, I * 3 - 2, 2), NumOk)]);
    end;
  end;
  HexToText := S;
end;

procedure SetupCaseConversion;
begin
  case ViewMode of
    vmASCII:
    begin
      UpCaseFunc := @UpCase;
      LoCaseFunc := @LoCase;
      InvCaseFunc := @InvertCase;
    end;
    vmPETSCII:
    begin
      UpCaseFunc := @CBMUpCase;
      LoCaseFunc := @CBMLoCase;
      InvCaseFunc := @CBMInvertCase;
    end;
    vmScreen:
    begin
      UpCaseFunc := @SCRUpCase;
      LoCaseFunc := @SCRLoCase;
      InvCaseFunc := @SCRInvertCase;
    end;
  end;
end;

procedure MakeFullName;
var
  N,
  S,
  U             : string;
begin
  FullName := LongName(ImageName, True);
  if PanelMode <> pmDOS then
  begin
    FullName := MakeTypeStr(PanelMode) + ':' + FullName + chDirSep;
    U := AddToPath(ConvertCBMPath(ImagePath, False, False, hxNone, DirSep), stEmpty, chDirSep);
    if PanelMode = pmTAR then S := ReadName else S := MakeCBMName(ReadName, GEOSFormat);
    S := U + S;
    U := U + stAllFilesUnix;
    ImageDir := FullName + U;
    FullName := FullName + S;
  end;
end;

procedure CheckDiskType;
begin
  case DiskType and dtTypeMask of
    dt1541:
    begin
      DirTrack := 18;
      DirTrack2 := MaxByte;
      MaxTrack := Max1541Tracks;
    end;
    dt1541Ext:
    begin
      DirTrack := 18;
      DirTrack2 := MaxByte;
      MaxTrack := 41;
    end;
    dt1571:
    begin
      DirTrack := 18;
      DirTrack2 := 53;
      MaxTrack := 71;
    end;
    dt1581:
    begin
      DirTrack := 40;
      DirTrack2 := MaxByte;
      MaxTrack := 81;
    end;
  end;
end;

function SectorNum(T: Byte): Byte;
var
  B             : Byte;
begin
  B := DiskType and dtTypeMask;
  if B = dt1581 then
  begin
    SectorNum := 40;
  end
  else
  begin
    case T of
      1..17: SectorNum := 21;
      18..24: SectorNum := 19;
      25..30: SectorNum := 18;
      31..42: SectorNum := 17;
    end;
    if B = dt1571 then
    begin
      case T of
        Max1541Tracks..52: SectorNum := 21;
        53..59: SectorNum := 19;
        60..65: SectorNum := 18;
        66..70: SectorNum := 17;
      end;
    end;
  end;
end;

function DiskPos(T, S: Byte): Longint;
var
  I             : Integer;
  P             : Longint;
begin
  P := 0;
  for I := 1 to T - 1 do Inc(P, SectorNum(I));
  Inc(P, S);
  DiskPos := P;
end;

procedure ReadDiskBlock(T, S: Byte; Buffer: PBlock);
var
  B             : Byte;
  E             : string[30];
begin
  if ShowReadErrors and (DiskType and dtErrorInfo > 0) then
  begin
    ExtSeek(Image, (DiskPos(MaxTrack, 0) shl 8) + DiskPos(T, S));
    ExtBlockRead(Image, B, 1);
    if MakeErrorStr(B, T, S, E) then ErrorWin(stError, E, stEmpty, CurHelpCtx, sbSkip);
  end;
  ExtSeek(Image, DiskPos(T, S) shl 8);
  ExtBlockRead(Image, Buffer^, 256);
end;

procedure WriteDiskBlock(T, S: Byte; Buffer: PBlock);
var
  B             : Byte;
begin
  if DiskType and dtErrorInfo > 0 then
  begin
    ExtSeek(Image, (DiskPos(MaxTrack, 0) shl 8) + DiskPos(T, S));
    B := 1;
    ExtBlockWrite(Image, B, 1);
  end;
  ExtSeek(Image, DiskPos(T, S) shl 8);
  ExtBlockWrite(Image, Buffer^, 256);
end;

procedure ReadTapeBlock(S: Word; Buffer: PBlock);
begin
  ExtSeek(Image, S shl 5);
  ExtBlockRead(Image, Buffer^, 32);
end;

procedure WriteTapeBlock(S: Word; Buffer: PBlock);
begin
  ExtSeek(Image, S shl 5);
  ExtBlockWrite(Image, Buffer^, 32);
end;

function GetBAMOffset(T: Byte): Word;
var
  W             : Word;
begin
  W := T shl 2;
  if T >= Max1541Tracks then
  begin
    case DiskExtBAMMode of
      xbSpeedDOS: Inc(W, $30);
      xbDolphinDOS: Inc(W, $1C);
    end;
  end;
  GetBAMOffset := W;
end;

function GetNamePtr(Pos: Integer): PString;
begin
  GetNamePtr := PString(@PanelNames^[PanelDir^[Pos].Name]);
end;

function InsName(const Name: string): Integer;
begin
  if NameEnd + Length(Name) <= SizeOf(TNameBuffer) then
  begin
    InsName := NameEnd;
    Move(Name, PanelNames^[NameEnd], Length(Name) + 1);
    Inc(NameEnd, Length(Name) + 1);
  end
  else
  begin
    InsName := -1;
    NameEnd := SizeOf(TNameBuffer);
  end;
end;

function ReadCBMEntry(var Entry: TDirEntry): Boolean;
var
  B,
  C,
  F,
  O             : Boolean;
  M,
  P,
  Q,
  R             : Byte;
  W,
  X,
  Y             : Word;
  K,
  L             : Longint;
  E             : string[4];
  N,
  S             : string;

procedure ProcessArcPath;
begin
  if Copy(GetPath(S, DirSep), 1, Length(RealImagePath)) = RealImagePath then
  begin
    if S[Length(S)] = DirSep then
    begin
      Q := faPartition;
      Dec(S[0]);
    end;
    M := Length(RealImagePath) + 1;
    if M > 1 then Inc(M);
    N := Copy(S, M, MaxStrLen);
    if N <> '' then
    begin
      if GetPath(N, DirSep) <> '' then
      begin
        Q := faPartition;
        M := LeftPos(DirSep, N);
        if M = 0 then S := N else S := Copy(N, 1, M - 1);
        Entry.Time := 0;
      end;
      if Q = faPartition then L := 0;
      Entry.Attr := Q or faClosed;
      Entry.Size := L;
      Entry.Name := CutPath(S, DirSep);
      if Entry.Name <> '' then
      begin
        Inc(ImageSize);
        O := True;
      end;
    end;
  end;
end;

begin
  B := True;
  O := False;
  Entry.ExtAttr := 0;
  RecordLen := 0;
  case PanelMode of
    pmDisk:
    begin
      Inc(DirPos);
      Inc(Number);
      if Number = 8 then
      begin
        Number := 0;
        if (DirBuffer[0] = 0) then
        begin
          B := False;
        end
        else
        begin
          DirSector := DirBuffer[1];
          FillChar(BAM, 3 * 256, 0);
          ReadDiskBlock(DirTrack, DirSector, @DirBuffer);
        end;
      end;
      if B then
      begin
        EntryPos := Number shl 5;
        Entry.Attr := DirBuffer[EntryPos + 2];
        RecordLen := DirBuffer[EntryPos + 23];
        Entry.ExtAttr := DirBuffer[EntryPos + 24];
        if not GEOSSupport or not (Entry.ExtAttr in [1..GEOSTypeNum]) then Entry.ExtAttr := 0;
        if (Entry.ExtAttr > 0) and (RecordLen > 0) then Entry.ExtAttr := Entry.ExtAttr or xaGEOSVLIR;
        Entry.Name := '';
        for P := 0 to CBMNameLen - 1 do Entry.Name := Entry.Name + Chr(DirBuffer[EntryPos + P + 5]);
        Entry.Name := CutChar(Entry.Name, chShiftSpace);
        Entry.Size := BytesToLongint(DirBuffer[EntryPos + 30], DirBuffer[EntryPos + 31], 0, 0);
        Entry.DirPos := DirPos;
        Track := DirBuffer[EntryPos + 3];
        Sector := DirBuffer[EntryPos + 4];
        SideTrack := DirBuffer[EntryPos + 21];
        SideSector := DirBuffer[EntryPos + 22];
        Entry.Track := Track;
        Entry.Sector := Sector;
        Entry.SideTrack := SideTrack;
        Entry.SideSector := SideSector;
        if (Entry.Attr and faTypeMask = faPartition) and (DiskType and dtTypeMask = dt1581) then
        begin
          if (Entry.Size >= 120) and (Entry.Size mod 40 = 0) and (Entry.Sector = 0) then
          begin
            P := Entry.Track;
            Q := P + (Entry.Size div 40);
            if ((P < DirTrack) and (Q < DirTrack)) or ((P > DirTrack) and (Q > DirTrack)) then
              Entry.ExtAttr := xaDirectory;
          end;
        end;
      end;
    end;
    pmTape:
    begin
      Inc(DirPos);
      if DirPos > ImageSize + 1 then
      begin
        B := False;
        ImagePos := OrigSize;
        _Free := DirPos;
      end
      else
      begin
        ReadTapeBlock(DirPos, @DirBuffer);
      end;
      if B then
      begin
        Entry.Attr := DirBuffer[0];
        if Entry.Attr in [faTapeNormal, faTapeFrozen] then
        begin
          if Entry.Attr = faTapeNormal then Entry.Attr := DirBuffer[1] else Entry.Attr := faFrozen;
          if not (Entry.Attr in [faDeleted..faRelative, faFrozen]) then Entry.Attr := faProgram;
          Entry.Attr := Entry.Attr or faClosed;
          ImagePos := BytesToLongint(DirBuffer[8], DirBuffer[9], DirBuffer[10], DirBuffer[11]);
          Entry.Name := '';
          for P := 0 to CBMNameLen - 1 do Entry.Name := Entry.Name + Chr(DirBuffer[P + 16]);
          Entry.Name := CutChar(Entry.Name, ' ');
          Entry.Start := BytesToLongint(DirBuffer[2], DirBuffer[3], 0, 0);
        end
        else
        begin
          B := False;
          ImagePos := OrigSize;
          _Free := DirPos;
        end;
        Entry.Size := ImagePos;
        Entry.DirPos := DirPos;
      end;
    end;
    pmFile:
    begin
      Inc(DirPos);
      if DirPos = 0 then
      begin
        S := LowerCase(FileExt(ImageName));
        Entry.Attr := 0;
        for P := 1 to 4 do if S[1] = ShortCBMExt[P][1] then Entry.Attr := P or faClosed;
        Entry.Name := CutChar(_Label, #0);
        Entry.Start := BytesToLongint(DirBuffer[26], DirBuffer[27], 0, 0);
        Entry.Size := ImagePos - 26;
        RecordLen := DirBuffer[25];
      end
      else
      begin
        B := False;
      end;
    end;
    pmLynx, pmArkive:
    begin
      Inc(DirPos);
      if DirPos = ImageSize then
      begin
        B := False;
        Inc(ImagePos, PrevSize);
      end
      else
      begin
        ExtSeek(Image, HeaderPos);
        ExtBlockRead(Image, TempBuffer, 40);
        B := (IOResult = 0);
        if B then
        begin
          M := PanelMode;
          R := 0;
          C := True;
(* ?ASM? *)
          asm
            cmp M, pmArkive;
            je @7;
            xor si, si;
        @2: mov al, byte ptr TempBuffer[si];
            cmp al, chReturn;
            je @3;
            mov byte ptr S[si][1], al;
            inc si;
            cmp si, CBMNameLen + 1;
            jb @2;
            xor si, si;
        @3: mov ax, si;
            mov byte ptr S[0], al;
            or si, si;
            je @1;
            inc si;
            mov cx, 10;
            call ReadNum;
            jc @1;
            or dx, dx;
            jne @1;
            cmp bl, chReturn;
            jne @1;
            mov X, ax;
            inc si;
            mov ah, byte ptr TempBuffer[si];
            xor al, al;
            cmp ah, 'D';
            je @5;
            inc al;
            cmp ah, 'S';
            je @5;
            inc al;
            cmp ah, 'P';
            je @5;
            inc al;
            cmp ah, 'U';
            je @5;
            inc al;
            cmp ah, 'R';
            jne @1;
        @5: mov Q, al;
            inc si;
            cmp byte ptr TempBuffer[si], chReturn;
            jne @1;
            inc si;
            cmp al, faRelative;
            jne @9;
            mov cx, 10;
            call ReadNum;
            jc @1;
            cmp bl, chReturn;
            jne @1;
            or ah, ah;
            jne @1;
            or dx, dx;
            jne @1;
            mov R, al;
            inc si;
        @9: mov cx, 10;
            call ReadNum;
            jnc @6;
            mov C, False;
            jmp @4;
        @6: cmp bl, chReturn;
            jne @1;
            or ah, ah;
            jne @1;
            or dx, dx;
            jne @1;
            mov Y, ax;
            inc si;
        @8: mov W, si;
            jmp @4;
        @7: mov si, Offset(TempBuffer);
            cld;
            lodsb;
            mov Q, al;
            lodsb;
            xor ah, ah;
            mov Y, ax;
            push ss;
            pop es;
            lea di, S;
            mov al, CBMNameLen;
            stosb;
            mov cl, al;
            xor ch, ch;
            rep movsb;
            lodsb;
            mov R, al;
            add si, 8;
            lodsw;
            mov X, ax;
            mov si, 29;
            jmp @8;
        @1: mov B, False;
        @4:
          end;
        end;
        K := ImagePos + PrevSize;
        Inc(K, PadSize(K));
        if not C then
        begin
          if DirPos = ImageSize - 1 then
          begin
            if X = 0 then
            begin
              Y := 0;
            end
            else
            begin
              L := OrigSize - K - (Longint(X) - 1) * 254;
              if L > 254 then Y := 255 else Y := L mod 254 + 1;
            end;
          end
          else
          begin
            B := False;
          end;
        end;
        L := 0;
        if X <> 0 then L := Longint(X) * 254 - 255 + Y;
        if B then
        begin
          Inc(HeaderPos, W);
          Inc(ImagePos, PrevSize);
          Entry.Name := CutChar(S, chShiftSpace);
          if M = pmLynx then Q := Q or faClosed;
          Entry.Attr := Q;
          PrevSize := Longint(X) * 254;
          Entry.Size := L;
          Entry.DirPos := DirPos;
          RecordLen := R;
        end;
      end;
    end;
    pmTAR:
    begin
      repeat
        Inc(DirPos);
        ExtSeek(Image, HeaderPos);
        ExtBlockRead(Image, TempBuffer, 512);
        B := ((IOResult = 0) and (TempBuffer[0] <> 0));
        if B then
        begin
(* ?ASM? *)
          asm
            push False;
            call ComputeTARCheck;
            push ax;
            push dx;
            mov si, $0094;
            mov cx, 8;
            call ReadNum;
            pop cx;
            pop bx;
            jc @3;
            cmp ax, bx;
            jne @3;
            cmp dx, cx;
            jne @3;
            mov si, Offset(TempBuffer);
            push ss;
            pop es;
            lea di, S[1];
            cld;
            xor cl, cl;
        @6: lodsb;
            or al, al;
            je @5;
            stosb;
            inc cl;
            jmp @6;
        @5: mov byte ptr S[0], cl;
            mov si, $0080;
            mov cx, 8;
            call ReadNum;
            jc @3;
            mov word ptr L[0], ax;
            mov word ptr L[2], dx;
            jmp @4;
        @3: mov B, False;
        @4:
          end;
          if B then
          begin
            PrevSize := L and $FFFFFE00;
            if L and $000001FF > 0 then Inc(PrevSize, 512);
            ImagePos := HeaderPos + 512;
            HeaderPos := ImagePos + PrevSize;
            Q := 2;
            Entry.Size := L;
            Entry.DirPos := DirPos;
            if S <> '' then
            begin
              if (Length(S) > 4) and (S[Length(S) - 3] = '.') then
              begin
                E := LowerCase(Copy(S, Length(S) - 2, 3));
                F := False;
                M := 0;
                while not F and (M < 4) do
                begin
                  F := (E = ShortCBMExt[M]);
                  if not F then Inc(M);
                end;
                if F then
                begin
                  Dec(S[0], 4);
                  Q := M;
                end;
              end;
              ProcessArcPath;
            end;
          end;
        end;
      until not B or O;
    end;
  end;
  ReadCBMEntry := B;
end;

procedure Prepare;
var
  C             : Byte;
  P,
  N,
  S             : string;

procedure CheckImage;
begin
  ImageName := AddToPath(P, S, chDirSep);
  if OpenImage(False) = 0 then ExtClose(Image);
end;

begin
  S := ImageName;
  PanelMode := DetermineTypePrefix(S);
  case PanelMode of
    pmExt: PanelMode := pmDOS;
    pmDOS:
  else
    S := Copy(S, LeftPos(':', S) + 1, MaxStrLen);
    P := GetPath(S, chDirSep);
    ReadName := CutPath(S, chDirSep);
    S := CutPath(P, chDirSep);
    P := GetPath(P, chDirSep);
    ImagePath := '';
    CheckImage;
    ReadName := ReconvertCBMName(ReadName, CopyGEOSFormat, True, hxDollar);
    while not FileExists(P, True) do
    begin
      CheckImage;
      if (PanelMode = pmDisk) or GetPanelModeAttrib(PanelMode, paDirectories) then
      begin
        if PanelMode = pmDisk then S := ReconvertCBMName(S, CopyGEOSFormat, True, hxDollar);
        ImagePath := AddToPath(S, ImagePath, DirSep);
        if (ImagePath <> '') and (ImagePath[Length(ImagePath)] = DirSep) then
          Dec(ImagePath[0]);
      end;
      S := CutPath(P, chDirSep);
      P := GetPath(P, chDirSep);
      if (Length(P) > 3) and (P[Length(P)] = chDirSep) then Dec(P[0]);
    end;
    ImageName := AddToPath(P, S, chDirSep);
  end;
end;

function OpenImage(Write: Boolean): Integer;
var
  B,
  F             : Boolean;
  P,
  Q             : Byte;
  W,
  X             : Word;
  I             : Integer;
  L             : Longint;
  T             : string[CBMNameLen];
  S             : string;
  E             : TDirEntry;

procedure ResetDir;
begin
  DirBuffer[0] := DirTrack;
  DirBuffer[1] := 0;
  Number := 7;
end;

begin
  InOutRes := 0;
  CopyGEOSFormat := False;
  PrevSize := 0;
  DirPos := MaxWord;
  ImagePos := 0;
  _Label := '';
  TapeName := '';
  I := 0;
  RealImagePath := ImagePath;
  DirSep := chDirSep;
  if PanelMode in [pmDisk, pmTAR] then
  begin
    DirSep := '/';
    if RealImagePath <> '' then for P := 1 to Length(RealImagePath) do
      if RealImagePath[P] = chDirSep then RealImagePath[P] := '/';
  end;
  if Write then P := fmReadWrite else P := fmReadOnly;
  LongGetFAttr(ImageName, ImageAttr);
  ImageReadOnly := (ImageAttr and ReadOnly > 0);
  I := LongOpenFile(ImageName, Image, P);
  ExtGetFTime(Image, FileTime);
  OrigSize := ExtFileSize(Image);
  if I = 0 then
  begin
    case PanelMode of
      pmDisk:
      begin
        DiskType := GetDiskType(OrigSize);
        if DiskType <> dtInvalid then
        begin
          CheckDiskType;
          if (DiskType and dtTypeMask = dt1581) and (ImagePath <> '') then
          begin
            DirTrack := 40;
            S := RealImagePath;
            repeat
              ResetDir;
              P := LeftPos(DirSep, S);
              if P = 0 then P := Length(S) + 1;
              T := Copy(S, 1, P - 1);
              S := Copy(S, P + 1, MaxStrLen);
              DirSector := FirstDirSec(DiskType);
              F := False;
              while not F and (ReadCBMEntry(Entry)) do F := ((Entry.Name = T) and (Entry.Attr and faTypeMask = faPartition) and
                (Entry.ExtAttr and xaDirectory > 0));
              if F then DirTrack := Entry.Track;
            until (S = '') or not F;
          end;
          ResetDir;
          DirSector := 0;
          DirPos := MaxWord;
          case DiskType and dtTypeMask of
            dt1541, dt1541Ext: ReadDiskBlock(DirTrack, 0, @BAM);
            dt1571:
            begin
              ReadDiskBlock(DirTrack, 0, @BAM);
              ReadDiskBlock(DirTrack2, 0, @BAM2);
            end;
            dt1581:
            begin
              ReadDiskBlock(DirTrack, 0, @BAM);
              ReadDiskBlock(DirTrack, 1, @BAM2);
              ReadDiskBlock(DirTrack, 2, @BAM3);
            end;
          end;
          ImageSize := 0;
          Q := NameOffset(DiskType, DiskExtBAMMode);
          for P := 0 to CBMNameLen - 1 do _Label := _Label + Chr(BAM[P + Q]);
          _Label := _Label + ',';
          Inc(Q, DiskIDRelPos);
          for P := 0 to CBMBAMIDLen - 1 do _Label := _Label + Chr(BAM[P + Q]);
          for P := 1 to MaxTrack - 1 do if P <> DirTrack then Inc(_Free, BAM[GetBAMOffset(P)]);
          if GEOSSupport then
          begin
            S[0] := Chr(Length(GEOSSign));
            Move(BAM[GEOSSignPos], S[1], Length(GEOSSign));
            CopyGEOSFormat := (S = GEOSSign);
          end;
          DirSector := FirstDirSec(DiskType);
          ReadDiskBlock(DirTrack, DirSector, @DirBuffer);
          Number := MaxByte;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmTape:
      begin
        if OrigSize > 95 then
        begin
          DirPos := 0;
          ReadTapeBlock(DirPos, @DirBuffer);
          for P := 0 to 31 do TapeName := TapeName + Chr(DirBuffer[P]);
          Inc(DirPos);
          ReadTapeBlock(DirPos, @DirBuffer);
          ImageSize := BytesToLongint(DirBuffer[2], DirBuffer[3], 0, 0);
          for P := 0 to 23 do _Label := _Label + Chr(DirBuffer[P + 8]);
          _Label := CutChar(_Label, ' ');
        end
        else
        begin
          I := 255;
        end;
      end;
      pmFile:
      begin
        _Label := '';
        if OrigSize > 25 then
        begin
          ExtBlockRead(Image, DirBuffer, 28);
          for P := 0 to 7 do _Label := _Label + Chr(DirBuffer[P]);
        end;
        if _Label = PC64Sign then
        begin
          ImagePos := OrigSize;
          _Label := '';
          for P := 0 to CBMNameLen - 1 do _Label := _Label + Chr(DirBuffer[P + 8]);
          _Label := CutChar(_Label, #0);
        end
        else
        begin
          I := 255;
        end;
      end;
      pmLynx:
      begin
        if OrigSize >= 254 then
        begin
          ExtBlockRead(Image, TempBuffer, 128);
(* ?ASM? *)
          asm
            xor si, si;
        @3: cmp word ptr TempBuffer[si], 0;
            jne @1;
            cmp word ptr TempBuffer[si][2], $0D00;
            jne @1;
            add si, 4;
            jmp @2;
        @1: inc si;
            cmp si, 92;
            jb @3;
            xor si, si;
        @2: mov word ptr L[0], si;
            mov word ptr L[2], 0;
          end;
          ExtSeek(Image, L);
          ExtBlockRead(Image, TempBuffer, 128);
(* ?ASM? *)
          asm
            xor si, si;
            mov cx, 10;
            call ReadNum;
            jc @1;
            or dx, dx;
            jne @1;
            or ax, ax;
            je @1;
            mov W, ax;
            mov dx, si;
            xor bl, bl;
        @5: cmp byte ptr TempBuffer[si], chReturn;
            je @2;
            cmp word ptr TempBuffer[si], 'YL';
            jne @4;
            cmp word ptr TempBuffer[si][2], 'XN';
            jne @4;
            inc bl;
        @4: inc si;
            cmp si, 30;
            jb @5;
            jmp @1;
        @2: or bl, bl;
            je @1;
            inc si;
            xchg si, dx;
            mov cx, dx;
            sub cx, si;
            add si, Offset(TempBuffer);
            lea di, S;
            push ss;
            pop es;
            cld;
            mov al, cl;
            stosb;
            rep movsb;
            mov si, dx;
            mov cx, 10;
            call ReadNum;
            jc @1;
            or dx, dx;
            jne @1;
            or ax, ax;
            je @1;
            cmp bl, chReturn;
            jne @1;
            mov X, ax;
            inc si;
            add word ptr L[0], si;
            jmp @3;
        @1: mov I, 255;
        @3:
          end;
          if I = 0 then
          begin
            ImageSize := X;
            ImagePos := W * 254;
            HeaderPos := L;
            _Label := S;
          end;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmArkive:
      begin
        if OrigSize >= 254 then
        begin
          ExtBlockRead(Image, P, 1);
(* ?ASM? *)
          asm
            mov al, P;
            xor ah, ah;
            mov X, ax;
            mov cl, 29;
            mul cl;
            inc ax;
            mov cl, 254;
            div cl;
            or ah, ah;
            je @1;
            inc al;
        @1: xor ah, ah;
            mov W, ax;
          end;
          ImageSize := X;
          ImagePos := W * 254;
          HeaderPos := 1;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmTAR:
      begin
        if OrigSize >= 512 then
        begin
          ImageSize := 0;
          HeaderPos := 0;
          ImagePos := 0;
          PrevSize := 0;
        end
        else
        begin
          I := 255;
        end;
      end;
    end;
  end;
  if (I <> deOK) and not ExtIsClosed(Image) then ExtClose(Image);
  if I = deAccessDenied then I := deOK;
  GEOSFormat := CopyGEOSFormat;
  OpenImage := I;
end;

procedure TTitle.Draw;
var
  A             : Byte;
  B             : TDrawBuffer;
begin
  A := GetColor(1);
  MoveChar(B, ' ', A, Size.X);
  MoveCBMStr(@B, Name, A, False);
  MoveStr(B[ScreenWidth - 43], Text^, A);
  WriteBuf(0, 0, Size.X, 1, B);
end;

procedure TTextInput.HandleEvent(var Event: TEvent);
var
  I             : Integer;
  S             : string[64];
begin
  TInputLine.HandleEvent(Event);
  if ChangedData then
  begin
    S := TextToHex(Data^, False, True);
    CodeInput^.SetData(S);
  end;
  if ChangedPos then
  begin
    I := CurPos * 3;
    if I > 63 then I := 63;
    CodeInput^.CurPos := I;
    CodeInput^.SelStart := 0;
    CodeInput^.SelEnd := 0;
    CodeInput^.DrawView;
  end;
end;

constructor TCodeInput.Init(var Bounds: TRect; ALen, AMaxLen: Integer);
begin
  TInputLine.Init(Bounds, ALen, AMaxLen, stEmpty, drNone);
  MaxLen := (AMaxLen div 3) * 3;
  SetState(sfAlwaysIns, False);
end;

procedure TCodeInput.HandleEvent(var Event: TEvent);
var
  WasAppending  : Boolean;
  Delta,
  I,
  OldFirstPos,
  OldSelStart,
  OldSelEnd     : Integer;
  S,
  OldData       : string;

function MouseDelta: Integer;
var
  Mouse         : TPoint;
begin
  MakeLocal(Event.Where, Mouse);
  if Mouse.X <= 0 then MouseDelta := -1 else
  if Mouse.X >= Size.X - 1 then MouseDelta := 1 else
  MouseDelta := 0;
end;

function MousePos: Integer;
var
  Pos           : Integer;
  Mouse         : TPoint;
begin
  MakeLocal(Event.Where, Mouse);
  if Mouse.X < 0 then Mouse.X := 0;
  Pos := Mouse.X + FirstPos;
  if Pos < 0 then Pos := 0;
  if Pos > Length(Data^) then Pos := Length(Data^);
  MousePos := Pos;
end;

procedure SaveState;
begin
  if Validator <> nil then
  begin
    OldData := Data^;
    OldCurPos := CurPos;
    OldFirstPos := FirstPos;
    OldSelStart := SelStart;
    OldSelEnd := SelEnd;
    WasAppending := Length(Data^) = CurPos;
  end;
end;

procedure RestoreState;
begin
  if Validator <> nil then
  begin
    Data^ := OldData;
    CurPos := OldCurPos;
    FirstPos := OldFirstPos;
    SelStart := OldSelStart;
    SelEnd := OldSelEnd;
  end;
end;

function CheckValid(NoAutoFill: Boolean): Boolean;
var
  OldLen        : Integer;
  NewData       : string;
begin
  if Validator <> nil then
  begin
    CheckValid := False;
    OldLen := Length(Data^);
    NewData := Data^;
    if Validator^.IsValidInput(NewData, NoAutoFill) then
    begin
      if Length(NewData) > MaxLen then NewData[0] := Char(MaxLen);
      Data^ := NewData;
      if (CurPos >= OldLen) and (Length(Data^) > OldLen) then
        CurPos := Length(Data^);
      CheckValid := True;
    end
    else
    begin
      RestoreState;
    end;
  end
  else
  begin
    CheckValid := True;
  end;
end;

begin
  ChangedData := False;
  ChangedPos := False;
  if Event.What and evMouse > 0 then if Event.Buttons and mbLeftButton = 0 then ClearEvent(Event);
  TView.HandleEvent(Event);
  if State and sfSelected <> 0 then
  begin
    case Event.What of
      evMouseDown:
      begin
        Delta := MouseDelta;
        if not Event.Double then
        begin
          repeat
            if Event.What = evMouseAuto then Delta := MouseDelta;
            if MouseInView(Event.Where) then
            begin
              if MousePos mod 3 <> 2 then
              begin
                CurPos := MousePos;
                ChangedPos := True;
              end;
              CheckValid(True);
            end;
            DrawView;
          until not MouseEvent(Event, evMouseMove + evMouseAuto);
        end;
        ClearEvent(Event);
      end;
      evKeyDown:
      begin
        SaveState;
        case Event.KeyCode of
          kbEnter:
          begin
            Event.What := evCommand;
            Event.Command := cmOK;
            Exit;
          end;
          kbLeft, kbCtrlS:
          begin
            if CurPos > 0 then
            begin
              Dec(CurPos);
              if CurPos mod 3 = 2 then Dec(CurPos);
              ChangedPos := True;
              CheckValid(True);
              ClearEvent(Event);
            end;
          end;
          kbRight, kbCtrlD:
          begin
            if CurPos < Length(Data^) then
            begin
              Inc(CurPos);
              if CurPos mod 3 = 2 then Inc(CurPos);
              ChangedPos := True;
              CheckValid(True);
              ClearEvent(Event);
            end;
          end;
          kbCtrlLeft, kbCtrlA:
          begin
            if CurPos > 0 then CurPos := ((CurPos - 1) div 3) * 3;
            ChangedPos := True;
            CheckValid(True);
          end;
          kbCtrlRight, kbCtrlF:
          begin
            if CurPos < Length(Data^) then CurPos := (CurPos div 3) * 3 + 3;
            ChangedPos := True;
            CheckValid(True);
          end;
          kbHome, kbCtrlHome:
          begin
            CurPos := 0;
            ChangedPos := True;
            CheckValid(True);
          end;
          kbEnd, kbCtrlEnd:
          begin
            CurPos := Length(Data^);
            ChangedPos := True;
            CheckValid(True);
          end;
          kbBack, kbCtrlH,
          kbCtrlBack, kbCtrlW:
          begin
            if CurPos > 2 then
            begin
              CurPos := (CurPos div 3) * 3 - 3;
              Delete(Data^, CurPos + 1, 3);
              ChangedData := True;
              CheckValid(True);
            end;
          end;
          kbDel, kbCtrlG,
          kbCtrlT:
          begin
            if CurPos < Length(Data^) then
            begin
              CurPos := (CurPos div 3) * 3;
              SelStart := CurPos;
              SelEnd := CurPos + 3;
              ChangedData := True;
            end;
            CheckValid(True);
          end;
          kbCtrlY:
          begin
            Data^ := '';
            CurPos := 0;
            ChangedData := True;
            CheckValid(True);
          end;
          kbCtrlK:
          begin
            CurPos := (CurPos div 3) * 3;
            Delete(Data^, CurPos + 1, MaxStrLen);
            ChangedData := True;
            CheckValid(True);
          end;
          kbCtrlV, kbIns:
          begin
            InsMode := not InsMode;
            SetState(sfCursorIns, InsMode);
          end;
        else
          Event.CharCode := UpCase(Event.CharCode);
          case Event.CharCode of
            '0'..'9', 'A'..'F':
            begin
              if CheckValid(True) then
              begin
                if CurPos < MaxLen then
                begin
                  if FirstPos > CurPos then FirstPos := CurPos;
                  if (InsMode or (CurPos = Length(Data^))) and (CurPos mod 3 = 0) then
                  begin
                    if Length(Data^) >= MaxLen then Data^[0] := Chr(MaxLen - 3);
                    Insert('00 ', Data^, CurPos + 1);
                  end;
                  Data^[CurPos + 1] := Event.CharCode;
                  ChangedData := True;
                  if not CursorStill then
                  begin
                    Inc(CurPos);
                    if CurPos mod 3 = 2 then Inc(CurPos);
                  end;
                end
                else
                begin
                  GoSound := True;
                end;
                CheckValid(False);
              end;
            end;
          end;
        end;
      end;
    end;
    if ChangedData then
    begin
      ChangedPos := True;
      S := HexToText(Data^, False);
      TextInput^.SetData(S);
    end;
    if ChangedPos then
    begin
      I := 15;
      if Len < 15 then I := Len;
      if CurPos < FirstPos then if CurPos > I then FirstPos := CurPos - I else FirstPos := 0;
      I := CurPos - Len;
      if FirstPos < I then FirstPos := I;
      SelStart := 0;
      SelEnd := 0;
      DrawView;
      ClearEvent(Event);
      TextInput^.CurPos := CurPos div 3;
      TextInput^.SelStart := 0;
      TextInput^.SelEnd := 0;
      TextInput^.DrawView;
    end;
  end;
end;

end.
