
{*************************************************}
{                 Joe Forster/STA                 }
{                                                 }
{                   COMPDRV.PAS                   }
{                                                 }
{    The Star Commander drive routine compiler    }
{*************************************************}

program The_Star_Commander_Drive_Routine_Compiler;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}
{$M 16384, 0, 655360}

uses
  DOS;

const
{File open modes}
  fmReadOnly    = 0;
  fmWriteOnly   = 1;
  fmReadWrite   = 2;
{Disk types}
  dtNormal35    = 0;
  dtError35     = 1;
  dtNormal40    = 2;
  dtError40     = 3;
  dtInvalid     = 255;
{Length of the data file header}
  DataHeaderLen = 160;
{Size of copy buffer}
  TBufferSize   = 4096;
{Character constants}
  chShiftSpace  = #160;

type
{CBM-style block}
  TBlock        = array [0..255] of Byte;
  PBlock        = ^TBlock;
{Copy buffer}
  TBuffer       = array [0..TBufferSize - 1] of Byte;
  PBuffer       = ^TBuffer;
{Directory entry}
  TDirEntry     = record
    Name        : string[16];
    Attr        : Byte;
    Size,
    Time        : Longint;
    Start       : Word;
  end;

var
  ExtMode,
  ErrorInfo,
  _End,
  Error         : Boolean;
  DiskType,
  MaxTrack,
  EntryPos,
  Track,
  Sector,
  Number,
  DirSector,
  HeaderCount,
  Data          : Byte;
  ReadSize      : Word;
  Image,
  DataFile      : file;
  HeaderBuffer  : array [0..DataHeaderLen - 1] of Byte;
  CBMEntry      : TDirEntry;
  Buffer        : TBuffer;
  BAM,
  DataBuffer,
  DirBuffer     : TBlock;

{Determine the type of disk image (35 or 40 tracks, with or without error
  info) on basis of the file size
  Input : L: the size of the disk image
  Output: the type of the disk image}
function GetDiskType(L: Longint): Byte;
var
  B             : Byte;
begin
  B := dtInvalid;
  if L = 174848 then B := dtNormal35;
  if L = 174848 + 683 then B := dtError35;
  if L = 196608 then B := dtNormal40;
  if L = 196608 + 768 then B := dtError40;
  GetDiskType := B;
end;

{Check disk type: 35 or 40 tracks, with or without error info}
procedure CheckDiskType;
begin
  ExtMode := (DiskType >= dtNormal40);
  ErrorInfo := (DiskType in [dtError35, dtError40]);
  MaxTrack := 36;
  if ExtMode then MaxTrack := 41;
end;

{Cut the specified character off of the end of the file name
  Input : Name: the original file name
          C: the character to be cut off
  Output: the corrected file name}
function CutChar(Name: string; C: Char): string;
begin
  while (Length(Name) > 0) and (Name[Length(Name)] = C) do Dec(Name[0]);
  CutChar := Name;
end;

{Convert four bytes into a long integer
  Input : B1, B2, B3, B4: the four bytes of the long integer
  Output: the long integer}
function BytesToLongint(B1, B2, B3, B4: Byte): Longint; assembler;
asm
    mov al, B1;
    mov ah, B2;
    mov dl, B3;
    mov dh, B4;
end;

{Get the number of sectors on the specified track
  Input : Track: the number of the track
  Output: the number of sectors on the track}
function SectorNum(Track: Byte): Byte;
begin
  case Track of
    1..17: SectorNum := 21;
    18..24: SectorNum := 19;
    25..30: SectorNum := 18;
    31..42: SectorNum := 17;
  end;
end;

{Compute the offset of the specified block in a disk image
  Input : Track, Sector: the track and sector number of the block
  Output: the offset of the block divided by 256}
function DiskPos(Track, Sector: Byte): Longint;
var
  I             : Integer;
  P             : Longint;
begin
  P := 0;
  for I := 1 to Track - 1 do Inc(P, SectorNum(I));
  Inc(P, Sector);
  DiskPos := P;
end;

{Read the specified block from the disk image into the given buffer
  Input : T, S: track and sector number of the block
          Buffer: buffer to read the block into}
procedure ReadDiskBlock(T, S: Byte; Buffer: PBlock);
begin
  Seek(Image, DiskPos(T, S) shl 8);
  BlockRead(Image, Buffer^, 256);
end;

{Read a directory entry from the disk image
  Input : Entry: entry record to contain the directory entry
  Output: when False, an empty directory entry was read}
function ReadCBMEntry(var Entry: TDirEntry): Boolean;
var
  B             : Boolean;
  P             : Byte;
begin
  B := True;
  Inc(Number);
  if Number = 8 then
  begin
    Number := 0;
    if (DirBuffer[0] = 0) then
    begin
      B := False;
    end
    else
    begin
      DirSector := DirBuffer[1];
      ReadDiskBlock(18, DirSector, @DirBuffer);
    end;
  end;
  if B then
  begin
    EntryPos := Number shl 5;
    Entry.Attr := DirBuffer[EntryPos + 2];
    Track := DirBuffer[EntryPos + 3];
    Sector := DirBuffer[EntryPos + 4];
    Entry.Name := '';
    for P := 0 to 15 do Entry.Name := Entry.Name + Chr(DirBuffer[EntryPos + P + 5]);
    Entry.Name := CutChar(Entry.Name, chShiftSpace);
    Entry.Size := BytesToLongint(DirBuffer[EntryPos + 30], DirBuffer[EntryPos + 31], 0, 0);
  end;
  ReadCBMEntry := B;
end;

{Open the disk image file
  Input : Name: name of the disk image
  Output: when not 0, an error occured}
function OpenImage(const Name: string): Integer;
var
  P             : Byte;
  I             : Integer;
begin
  I := IOResult;
  FileMode := fmReadOnly;
  Assign(Image, Name);
  Reset(Image, 1);
  I := IOResult;
  if I = 0 then
  begin
    DiskType := GetDiskType(FileSize(Image));
    if DiskType = dtInvalid then
    begin
      I := 255;
    end
    else
    begin
      CheckDiskType;
      DirSector := 0;
      ReadDiskBlock(18, 0, @BAM);
      Inc(DirSector);
      ReadDiskBlock(18, DirSector, @DirBuffer);
      Number := $FF;
    end;
  end;
  OpenImage := I;
  FileMode := fmReadOnly;
end;

{Read a buffer of data from the input file
  Input : Buffer: buffer to contain the data read
          Length: length of the data read}
procedure ReadPart(Buffer: PBuffer; var Length: Word);
var
  M,
  Q             : Byte;
begin
  Error := False;
  _End := Error;
  Length := 0;
  while not Error and (Length < TBufferSize) and (Track > 0) do
  begin
    Error := ((Track > MaxTrack) or (Sector >= SectorNum(Track)));
    if not Error then
    begin
      ReadDiskBlock(Track, Sector, @DataBuffer);
      Track := DataBuffer[0];
      Sector := DataBuffer[1];
      if Track > 0 then
      begin
        M := 254;
      end
      else
      begin
        M := Sector - 1;
        _End := True;
      end;
      for Q := 0 to M - 1 do Buffer^[Length + Q] := DataBuffer[Q + 2];
      Inc(Length, M);
    end;
  end;
  Error := Error or (IOResult <> 0);
  _End := Error or _End;
  if Error then Length := 0;
end;

{Copy a file into the data file
  Input : ReadName: name of the file to be copied
          Important: when True, if the file is missing, the program is aborted;
                     otherwise empty data is copied}
procedure CopyFile(const ReadName: string; Important: Boolean);
var
  F             : Boolean;
  I,
  P             : Longint;
begin
  if ReadName <> '' then
  begin
    if OpenImage('DRIVEPRG.D64') = 0 then
    begin
      F := False;
      while not F and ReadCBMEntry(CBMEntry) do F := ((CBMEntry.Attr > 0) and (CBMEntry.Name = ReadName));
      if F then
      begin
        ReadPart(@Buffer, ReadSize);
        Close(Image);
        Dec(Buffer[1], $10);
        BlockWrite(DataFile, Buffer, ReadSize);
      end
      else
      begin
        Close(Image);
        if Important then Error := True;
      end;
    end
    else
    begin
      Error := True;
    end;
  end;
  if Error then
  begin
    Close(DataFile);
    Erase(DataFile);
    Halt;
  end
  else
  begin
    P := FilePos(DataFile);
    HeaderBuffer[HeaderCount] := Lo(P);
    HeaderBuffer[HeaderCount + 1] := Hi(P);
    HeaderCount := HeaderCount + 2;
  end;
end;

var
  I             : Byte;
  P             : Longint;
begin
  Assign(DataFile, 'DRIVEPRG.BIN');
  Rewrite(DataFile, 1);
  for I := 0 to DataHeaderLen - 1 do HeaderBuffer[I] := 0;
  BlockWrite(DataFile, HeaderBuffer, DataHeaderLen);
  HeaderCount := 0;
  P := FilePos(DataFile);
  HeaderBuffer[HeaderCount] := Lo(P);
  HeaderBuffer[HeaderCount + 1] := Hi(P);
  HeaderCount := HeaderCount + 2;
  CopyFile('BASE', True);
  CopyFile('TDDETECT', True);
  CopyFile('TDCHANGE', True);
  CopyFile('TDIRLOAD', True);
  CopyFile('WDIRLOAD', True);
  CopyFile('TDVALID', True);
  CopyFile('WDVALID', True);
  CopyFile('TDFORMAT', True);
  CopyFile('TDFORMV', True);
  CopyFile('ATTRIB', True);
  CopyFile('TFLOAD', True);
  CopyFile('WFLOAD', True);
  CopyFile('TFSAVE', True);
  CopyFile('WFSAVE', True);
  CopyFile('TFDELETE', True);
  CopyFile('TDEDIT', True);
  CopyFile('LABEL', True);
  CopyFile('TDLOAD', True);
  CopyFile('WDLOAD', True);
  CopyFile('TDSAVE', True);
  CopyFile('WDSAVE', True);
  CopyFile('WDVERIFY', True);
  CopyFile('UPLOAD', True);
  CopyFile('TSEND', True);
  CopyFile('ASEND', True);
  CopyFile('HSEND', True);
  CopyFile('PSEND', True);
  CopyFile('TRECEIVE', True);
  CopyFile('ARECEIVE', True);
  CopyFile('PRECEIVE', True);
  CopyFile('TRECVBLK', True);
  CopyFile('ARECVBLK', True);
  CopyFile('PRECVBLK', True);
  CopyFile('BASE81', False);
  CopyFile('TDDET81', False);
  CopyFile('TDCHG81', False);
  CopyFile('TDIRLD81', True);
  CopyFile('WDIRLD81', False);
  CopyFile('TDVAL81', False);
  CopyFile('WDVAL81', False);
  CopyFile('TDFORM81', False);
  CopyFile('TDFRMV81', False);
  CopyFile('ATTRIB81', False);
  CopyFile('TFLOAD81', False);
  CopyFile('WFLOAD81', False);
  CopyFile('TFSAVE81', False);
  CopyFile('WFSAVE81', False);
  CopyFile('TFDEL81', False);
  CopyFile('TDEDIT81', False);
  CopyFile('LABEL81', False);
  CopyFile('TDLOAD81', True);
  CopyFile('WDLOAD81', False);
  CopyFile('TDSAVE81', False);
  CopyFile('WDSAVE81', False);
  CopyFile('WDVRFY81', False);
  CopyFile('UPLOAD81', False);
  CopyFile('TSEND81', False);
  CopyFile('ASEND81', True);
  CopyFile('HSEND81', False);
  CopyFile('PSEND81', False);
  CopyFile('TRECV81', False);
  CopyFile('ARECV81', True);
  CopyFile('PRECV81', False);
  CopyFile('TRECVB81', False);
  CopyFile('ARECVB81', True);
  CopyFile('PRECVB81', False);
  Seek(DataFile, 0);
  BlockWrite(DataFile, HeaderBuffer, DataHeaderLen);
  Close(DataFile);
end.
