
{*************************************************}
{                 Joe Forster/STA                 }
{                                                 }
{                    CONFIG.PAS                   }
{                                                 }
{          The Star Commander Config unit         }
{*************************************************}

unit Config;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O+,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Dialogs, Drivers, Objects, Views,
  Base2, CfgMenus, Colors, Constant;

const
{Maximum length of panel paths in the configuration file}
  MaxCfgPathLen = MaxStrLen;
{Maximum length of program file extension string in configuration file}
  MaxCfgPrgExtLen= MaxPrgExtLen;
{Maximum length of drive names}
  MaxDriveNameLen= 32;
{Event command base for selecting drives}
  DriveCmdBase  = $FF00;

type
  TExecMenuProc = function (var Event: TEvent): Boolean;
  TSetupMenu    = object(TView)
    Items,
    CurItem     : PSetupItem;
    ExecMenuProc: TExecMenuProc;
    constructor Init(var Bounds: TRect; AItems, ACurItem: PSetupItem; AExecMenuProc: TExecMenuProc);
    destructor Done; virtual;
    procedure ExecProc(P: TMenuProc; MenuOff, MenuOn: Boolean; NextMenu, PrevMenu: Word; var Event: TEvent);
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Draw; virtual;
  end;
  PSetupMenu    = ^TSetupMenu;
  TDriveConfig  = record
    Name        : string[MaxDriveNameLen];
    ManualTimeouts: Boolean;
    DetectPortModes: Byte;
    DetectDiskChange,
    EndlessRetry,
    VerifyWrite,
    FormatBumpsHead,
    RetryHalftracks,
    RetryBumpsHead: Boolean;
    TransferMode,
    SerialCable,
    ParallelCable,
    LPTNum,
    ParLPTNum,
    ExternalDrive,
    ExtendedDiskMode,
    DiskExtBAMMode,
    DiskCopyMode,
    InvalidGCRCodeMode,
    CmdExecMode,
    HeadSpeed,
    RetryNum,
    SmartRetryNum: Byte;
    DriveInts   : array [0..DriveIntNum - 1] of Byte;
  end;
  PDriveConfig  = ^TDriveConfig;
  TPrevDriveConfig= record
    Name        : string[MaxDriveNameLen];
    ManualTimeouts,
    DetectPortModes,
    DetectDiskChange,
    EndlessRetry,
    VerifyWrite,
    FormatBumpsHead,
    RetryHalftracks,
    RetryBumpsHead: Boolean;
    TransferMode,
    SerialCable,
    ParallelCable,
    LPTNum,
    ParLPTNum,
    ExternalDrive,
    ExtendedDiskMode,
    DiskExtBAMMode,
    DiskCopyMode,
    CmdExecMode,
    HeadSpeed,
    RetryNum    : Byte;
    DriveInts   : array [0..DriveIntNum - 1] of Byte;
  end;
  PPrevDriveConfig  = ^TPrevDriveConfig;

var
  MemoryCfgOK,
  SetupChanged,
  LeftSel,
  LeftVis,
  RightVis,
  LeftOrigVis,
  RightOrigVis,
  LeftHiddenFiles,
  RightHiddenFiles: Boolean;
  ConfigStatus,
  OrigScreenWidth,
  OrigScreenHeight,
  OrigLeftPanWidth,
  OrigPanWinSize,
  OrigHalfPanWinSize,
  LeftMode,
  RightMode,
  LeftOrigMode,
  RightOrigMode,
  LeftSortMode,
  RightSortMode,
  LeftQuickView,
  RightQuickView,
  LeftColumnMode,
  RightColumnMode,
  LeftDOSNameColWidth,
  RightDOSNameColWidth,
  LeftCBMNameColWidth,
  RightCBMNameColWidth,
  LeftMiniStatus,
  RightMiniStatus,
  DriveNum,
  CurDriveNum   : Byte;
  SetupCmd,
  SetupHelpCtx  : Word;
  SetupMenu     : PSetupMenu;
  LeftPath,
  RightPath     : string;
  DriveConfigs  : array [0..MaxDrives - 1] of TDriveConfig;

procedure PutDrivePars(Drive: PDriveConfig);
function ReadConfigStr(Num: Byte; TrailingZeros: Boolean): string;
function NewSetupItem(const AHotStr, ATitle: string; ACommand, AHelpCtx: Word; PrevItem: PSetupItem): PSetupItem;
procedure PrintConfigError;
function GetName(const Title, Text: string; MaxLen: Integer): Boolean;
procedure ResetConfig;
procedure LoadConfig;
function SaveConfig(Dest: Byte): Boolean;
procedure MainConfig;
procedure DriveConfig;

implementation

uses
  App, DOS, Menus,
{$IFNDEF ExternalSetup}
  MiscFunc, Panel1,
{$ENDIF}
  Base1, ExtFiles, LowLevel;

{Destroy a list of setup menu items
  Input : P: the list of items to destroy}
procedure FreeSetupItem(P: PSetupItem);
begin
  if P <> nil then
  begin
    FreeSetupItem(P^.Next);
    Dispose(P);
  end;
end;

{Create a new item for the setup menu
  Input : AHotStr: hotkey for the item
          ATitle: the title of the item
          ACommand: the command generated by the item
          AHelpCtx: the help context associated with the item
          PrevItem: the previous item in the list
  Output: the new list of items}
function NewSetupItem(const AHotStr, ATitle: string; ACommand, AHelpCtx: Word; PrevItem: PSetupItem): PSetupItem;
var
  B             : Byte;
  I             : Integer;
  P             : PSetupItem;
  S             : string;
begin
  P := New(PSetupItem);
  P^.HelpCtx := AHelpCtx;
  P^.Command := ACommand;
  P^.Next := nil;
  S := AHotStr;
  while S[Length(S)] in WhiteSpace do Dec(S[0]);
  if Length(S) <= 1 then
  begin
    if Length(S) > 0 then P^.HotCode := Ord(UpCase(S[1]));
  end
  else
  begin
    if S[1] = 'F' then
    begin
      Val(Copy(S, 2, MaxStrLen), B, I);
      if B in [1..10] then P^.HotCode := kbF1 + (B - 1) shl 8;
    end;
  end;
  while Length(S) < 4 do
  begin
    Inc(S[0]);
    S[Length(S)] := ' ';
  end;
  P^.Title := S + ATitle;
  if PrevItem <> nil then PrevItem^.Next := P;
  NewSetupItem := P;
end;

{Display an error or warning message on the screen if there was something
  wrong with the configuration file}
procedure PrintConfigError;

{Display the actual message
  Input : SetupFileName: name of the setup file; empty for the main setup
                         file
          OKMask: bit mask for the setup file's "OK" bit
          PrevMask: bit mask for the setup file's "from previous release" bit}
procedure PrintError(const SetupFileName: string; OKMask, PrevMask: Byte);
begin
  if ConfigStatus and OKMask > 0 then
  begin
    if ConfigStatus and PrevMask > 0 then ErrorWin('Warning', 'You are using the ' + SetupFileName + 'setup file',
      'of the previous version.', hcNoContext, sbNone);
  end
  else
  begin
    ConfigStatus := ConfigStatus or OKMask;
    ErrorWin(stError, 'The ' + SetupFileName + 'setup file is corrupted.',
      'Falling back to default values.', hcNoContext, sbNone);
  end;
end;

begin
  PrintError(stEmpty, ssMainOK, ssMainPrev);
  if MainProgram then PrintError('drive ', ssDriveOK, ssDrivePrev);
end;

{Copy the drive parameters from the main configuration into a drive
  Input : Drive: the drive to copy the parameters into}
procedure GetDrivePars(Drive: PDriveConfig);
begin
  Drive^.TransferMode := TransferMode;
  Drive^.SerialCable := SerialCable;
  Drive^.ParallelCable := ParallelCable or (AsyncTransfer shl 6);
  Drive^.ManualTimeouts := ManualTimeouts;
  Drive^.LPTNum := LPTNum;
  Drive^.ParLPTNum := ParLPTNum;
  Drive^.DetectPortModes := DetectPortModes;
  Drive^.ExternalDrive := ExternalDrive;
  Drive^.ExtendedDiskMode := ExtendedDiskMode;
  Drive^.DiskExtBAMMode := DiskExtBAMMode;
  Drive^.DiskCopyMode := DiskCopyMode;
  Drive^.InvalidGCRCodeMode := InvalidGCRCodeMode;
  Drive^.DetectDiskChange := DetectDiskChange;
  Drive^.EndlessRetry := EndlessRetry;
  Drive^.VerifyWrite := VerifyWrite;
  Drive^.CmdExecMode := CmdExecMode;
  Drive^.HeadSpeed := HeadSpeed;
  Drive^.FormatBumpsHead := FormatBumpsHead;
  Drive^.RetryNum := RetryNum;
  Drive^.SmartRetryNum := SmartRetryNum;
  Drive^.RetryHalftracks := RetryHalftracks;
  Drive^.RetryBumpsHead := RetryBumpsHead;
  Move(DriveInts, Drive^.DriveInts, DriveIntNum);
end;

{Copy the drive parameters from a drive into the main configuration
  Input : Drive: the drive to copy the parameters from}
procedure PutDrivePars(Drive: PDriveConfig);
begin
  TransferMode := Drive^.TransferMode;
  SerialCable := Drive^.SerialCable;
  ParallelCable := Drive^.ParallelCable and pcCableMask;
  AsyncTransfer := (Drive^.ParallelCable and pcAsyncMask) shr 6;
  ManualTimeouts := Drive^.ManualTimeouts;
  LPTNum := Drive^.LPTNum;
  ParLPTNum := Drive^.ParLPTNum;
  DetectPortModes := Drive^.DetectPortModes;
  ExternalDrive := Drive^.ExternalDrive;
  ExtendedDiskMode := Drive^.ExtendedDiskMode;
  DiskExtBAMMode := Drive^.DiskExtBAMMode;
  DiskCopyMode := Drive^.DiskCopyMode;
  InvalidGCRCodeMode := Drive^.InvalidGCRCodeMode;
  DetectDiskChange := Drive^.DetectDiskChange;
  EndlessRetry := Drive^.EndlessRetry;
  VerifyWrite := Drive^.VerifyWrite;
  CmdExecMode := Drive^.CmdExecMode;
  HeadSpeed := Drive^.HeadSpeed;
  FormatBumpsHead := Drive^.FormatBumpsHead;
  RetryNum := Drive^.RetryNum;
  SmartRetryNum := Drive^.SmartRetryNum;
  RetryHalftracks := Drive^.RetryHalftracks;
  RetryBumpsHead := Drive^.RetryBumpsHead;
  Move(Drive^.DriveInts, DriveInts, DriveIntNum);
  InitTransfer(False);
end;

{Reset the parameters of a drive
  Input : Drive: the drive to reset
          DriveType: type of the drive to fetch default parameters for}
procedure ResetDrivePars(Drive: PDriveConfig; DriveType: Byte);
begin
  with Drive^ do
  begin
    TransferMode := tmNormal;
    SerialCable := scNone;
    ParallelCable := pcNone;
    ManualTimeouts := True;
    LPTNum := 0;
    ParLPTNum := 0;
    DetectPortModes := dpAll;
    ExternalDrive := DriveType;
    ExtendedDiskMode := xtNever;
    DiskExtBAMMode := xbSpeedDOS;
    DiskCopyMode := dcFull;
    InvalidGCRCodeMode := igNone;
    DetectDiskChange := False;
    EndlessRetry := False;
    VerifyWrite := True;
    CmdExecMode := cxNormal;
    HeadSpeed := $20;
    FormatBumpsHead := True;
    RetryNum := 5;
    SmartRetryNum := 0;
    RetryHalftracks := True;
    RetryBumpsHead := True;
    Move(DefDriveInts[ExtDriveTypes[DriveType]], DriveInts, DriveIntNum);
  end;
  InitTransfer(False);
end;

{Reset the settings to the default values}
procedure ResetConfig;
var
  B             : Byte;
begin
  for B := apBlackWhite to apMonochrome do CApplication[B] := CDefApplication[B];
  ScreenCol := apColor;
  ShowMenu := False;
  CursorFollowsFilename := True;
  ErrorSound := True;
  AutoSaveSetup := True;
  InsMovesDown := True;
  EscTogglesPanels := True;
  AltPopsMenu := True;
  AlternativeHotkeys := False;
  SnowCheck := True;
  VESASupport := True;
  MouseReverse := False;
  FastMouse := False;
  SaverDelay := 5;
  AutoMenus := False;
  PathPrompt := True;
  ShowKeyBar := True;
  FullScreen := True;
  ShowClock := True;
  LongNames := True;
  PreferLongNames := False;
  KeepLowerCase := False;
  DOSSizeBlocks := False;
  ConvertConfirm := True;
  DeleteConfirm := True;
  QuitConfirm := True;
  TransferConfirm := True;
  DiskEditConfirm := True;
  AutoUnselect := True;
  KeepNonStandardExt := False;
  PrgExt := 'prg';
  IntoFileImages := ifNever;
  ExtractFileImages := xfNever;
  TransferWarning := True;
  FileSizeWarning := True;
  CharStartMode := csIBMLower;
  EightColFont := False;
  MakeBackup := True;
  GEOSSupport := True;
  StartInfo := False;
  ShowReadErrors := True;
  DiskExtBAMMode := xbSpeedDOS;
  ImageExtBAMMode := xbSpeedDOS;
  KeepTime := False;
  CopyToDirTrack := False;
  ConvInvalidChars := ccNone;
  KeepUpperCase := False;
  WipeFiles := False;
  OrigPattern := False;
  Move(DefImageInts, ImageInts, (DiskTypeNum * 2));
  TransferMode := tmNormal;
  SerialCable := scNone;
  ParallelCable := pcNone;
  AsyncTransfer := atAuto;
  ManualTimeouts := True;
  DelayValue := 0;
  LPTNum := 0;
  ParLPTNum := 0;
  DetectPortModes := dpAll;
  ExternalDrive := xd1541;
  ExtendedDiskMode := xtNever;
  DiskExtBAMMode := xbSpeedDOS;
  DiskCopyMode := dcFull;
  InvalidGCRCodeMode := igNone;
  DetectDiskChange := False;
  EndlessRetry := False;
  VerifyWrite := True;
  CmdExecMode := cxNormal;
  HeadSpeed := $20;
  FormatBumpsHead := True;
  RetryNum := 5;
  SmartRetryNum := 0;
  RetryHalftracks := True;
  RetryBumpsHead := True;
  Move(DefDriveInts[ExtDriveTypes[ExternalDrive]], DriveInts, DriveIntNum);
  OrigScreenWidth := 80;
  OrigScreenHeight := 25;
  OrigLeftPanWidth := 40;
  OrigPanWinSize := 23;
  OrigHalfPanWinSize := 15;
  LeftSel := False;
  LeftVis := True;
  RightVis := True;
  LeftOrigVis := True;
  RightOrigVis := True;
  LeftQuickView := qvNone;
  RightQuickView := qvNone;
  LeftColumnMode := cmDOSBrief + cmCBMFull;
  RightColumnMode := cmDOSBrief + cmCBMFull;
  LeftDOSNameColWidth := DefDOSNameColWidth;
  RightDOSNameColWidth := DefDOSNameColWidth;
  LeftCBMNameColWidth := DefCBMNameColWidth;
  RightCBMNameColWidth := DefCBMNameColWidth;
  LeftMiniStatus := msDOSBrief + msCBMFull;
  RightMiniStatus := msDOSBrief + msCBMFull;
  LeftSortMode := psUnsorted + (psUnsorted shl 3);
  RightSortMode := psUnsorted + (psUnsorted shl 3);
  LeftHiddenFiles := True;
  RightHiddenFiles := True;
  LeftFileFilter := pfAll;
  RightFileFilter := pfAll;
  LeftPath := '';
  RightPath := '';
  DriveNum := 0;
  ECPAddrOffs := DefECPAddrOffs;
end;

{Read a byte from the setup file
  Output: the byte read}
function ReadConfig: Byte;
var
  B             : Byte;
begin
  B := GCRBuffer[ConfigCount];
  Inc(ConfigCount);
  ConfigXORSum := ConfigXORSum xor B;
  Dec(ConfigSUBSum, B);
  ReadConfig := B;
end;

{Read a word from the setup file
  Output: the word read}
function ReadConfigWord: Word;
var
  W             : Word;
begin
  W := ReadConfig;
  W := (ReadConfig shl 8) or W;
  ReadConfigWord := W;
end;

{Read a string from the setup file
  Input : Num: the number of characters to read
          TrailingZeros: when True, trailing zero bytes are left in the string
  Output: the string read}
function ReadConfigStr(Num: Byte; TrailingZeros: Boolean): string;
var
  I             : Integer;
  S             : string;
begin
  S := '';
  for I := 1 to Num do S := S + Chr(ReadConfig);
  if not TrailingZeros then S := CutChar(S, #0);
  ReadConfigStr := S;
end;

{Write a byte to the setup file
  Input : B: the byte to write}
procedure WriteConfig(B: Byte);
begin
  GCRBuffer[ConfigCount] := B;
  Inc(ConfigCount);
  ConfigXORSum := ConfigXORSum xor B;
  Dec(ConfigSUBSum, B);
end;

{Write a word to the setup file
  Input : B: the word to write}
procedure WriteConfigWord(W: Word);
begin
  WriteConfig(Lo(W));
  WriteConfig(Hi(W));
end;

{Write a string to the setup file
  Input : S: the string to write
          TrailingZeros: when True, the string is padded with zero bytes, if
                         too short
          N: the number of characters to write}
procedure WriteConfigStr(const S: string; Num: Byte);
var
  I             : Integer;
begin
  for I := 1 to Num do if I > Length(S) then WriteConfig(0) else WriteConfig(Ord(S[I]));
end;

{Check whether the configuration file checksums are correct
  Input : OK: when True, the file seems to be correct up to this phase
  Output: when True, the checksums are also correct}
function CheckConfigChecksum(OK: Boolean): Boolean;
var
  O             : Boolean;
begin
  O := OK;
  if not Resident and OK and ((ConfigXORSum <> GCRBuffer[ConfigCount]) or
    (ConfigSUBSum <> GCRBuffer[ConfigCount + 1])) then O := False;
  Inc(ConfigCount, 2);
  CheckConfigChecksum := O;
end;

{Reset config file index and checksums
  Input : Start: starting index; when below 0, the file index is not reset}
procedure ResetConfigCount(Start: Longint);
begin
  if Start >= 0 then ConfigCount := Start;
  ConfigXORSum := 0;
  ConfigSUBSum := 0;
end;

{Check config file signature
  Input : Sign: signature string
  Output: when True, the string matches}
function CheckSign(const Sign: string): Boolean;
var
  O             : Boolean;
  B             : Byte;
begin
  O := True;
  for B := 1 to Length(Sign) do
  begin
    if ReadConfig <> Ord(Sign[B]) then
    begin
      O := False;
      break;
    end;
  end;
  CheckSign := O;
end;

{Load the settings from the setup file or from the memory of the loader}
procedure LoadConfig;
var
  F,
  G,
  O,
  P,
  Q,
  R             : Boolean;
  B,
  C             : Byte;
  I,
  J             : Longint;
  D             : PDriveConfig;

begin
  SetHomePath;
  F := True;
  G := True;
  O := False;
  P := False;
  Q := False;
  R := False;
  ConfigStatus := ConfigStatus and not (ssMainOK + ssMainPrev + ssDriveOK + ssDrivePrev);
  ResetConfig;
  FillChar(GCRBuffer, SetupLen, 0);
  if MemoryCfgOK then
  begin
    Move(ShellBuffer^.Config, GCRBuffer, SetupLen);
    ResetConfigCount(0);
    O := True;
  end
  else
  begin
    if LongOpenFile(AddToPath(HomePath, ConfigFileName, chDirSep), ReadFile, fmReadOnly) = 0 then
    begin
      I := ExtFileSize(ReadFile);
      if I = Length(ConfigSign) + (4 * PaletteLen + DiskTypeNum * 2 +
        (MaxLPTPorts - MaxRealPorts) * SizeOf(Word) + DriveIntNum + MaxCfgPathLen * 2 + 117) then
      begin
        ExtBlockRead(ReadFile, GCRBuffer, I);
        ResetConfigCount(0);
        O := CheckSign(ConfigSign);
      end;
      if not O and (I = Length(PrevConfigSign) + (4 * PaletteLen + DiskTypeNum * 2 +
        (MaxLPTPorts - MaxRealPorts) * SizeOf(Word) + DriveIntNum + 275)) then
      begin
        ExtSeek(ReadFile, 0);
        ExtBlockRead(ReadFile, GCRBuffer, I);
        ResetConfigCount(0);
        O := CheckSign(PrevConfigSign);
        P := True;
      end;
      ExtClose(ReadFile);
    end
    else
    begin
      F := False;
    end;
  end;
  if O then
  begin
    ScreenCol := ReadConfig;
    SnowCheck := Boolean(ReadConfig);
    VESASupport := Boolean(ReadConfig);
    MouseReverse := Boolean(ReadConfig);
    FastMouse := Boolean(ReadConfig);
    ConvertConfirm := Boolean(ReadConfig);
    DeleteConfirm := Boolean(ReadConfig);
    QuitConfirm := Boolean(ReadConfig);
    TransferConfirm := Boolean(ReadConfig);
    DiskEditConfirm := Boolean(ReadConfig);
    TransferWarning := Boolean(ReadConfig);
    FileSizeWarning := Boolean(ReadConfig);
    InsMovesDown := Boolean(ReadConfig);
    AlternativeHotkeys := Boolean(ReadConfig);
    EscTogglesPanels := Boolean(ReadConfig);
    AltPopsMenu := Boolean(ReadConfig);
    CursorFollowsFilename := Boolean(ReadConfig);
    ErrorSound := Boolean(ReadConfig);
    AutoSaveSetup := Boolean(ReadConfig);
    AutoMenus := Boolean(ReadConfig);
    PathPrompt := Boolean(ReadConfig);
    ShowKeyBar := Boolean(ReadConfig);
    ShowClock := Boolean(ReadConfig);
    ShowMenu := Boolean(ReadConfig);
    MakeBackup := Boolean(ReadConfig);
    StartInfo := Boolean(ReadConfig);
    ShowReadErrors := Boolean(ReadConfig);
    AutoUnselect := Boolean(ReadConfig);
    DetectDiskChange := Boolean(ReadConfig);
    KeepTime := Boolean(ReadConfig);
    DOSSizeBlocks := Boolean(ReadConfig);
    GEOSSupport := Boolean(ReadConfig);
    ExtractFileImages := ReadConfig;
    IntoFileImages := ReadConfig;
    CopyToDirTrack := Boolean(ReadConfig);
    WipeFiles := Boolean(ReadConfig);
    OrigPattern := Boolean(ReadConfig);
    KeepNonStandardExt := Boolean(ReadConfig);
    ConvInvalidChars := ReadConfig;
    KeepUpperCase := Boolean(ReadConfig);
    for B := 0 to (DiskTypeNum * 2) - 1 do ImageInts[B] := ReadConfig;
    ExternalDrive := ReadConfig;
    for B := 0 to DriveIntNum - 1 do DriveInts[B] := ReadConfig;
    HeadSpeed := ReadConfig;
    TransferMode := ReadConfig;
    ParallelCable := ReadConfig;
    AsyncTransfer := (ParallelCable and pcAsyncMask) shr 6;
    ParallelCable := ParallelCable and pcCableMask;
    SerialCable := ReadConfig;
    DetectPortModes := ReadConfig;
    ManualTimeouts := Boolean(ReadConfig);
    VerifyWrite := Boolean(ReadConfig);
    CmdExecMode := ReadConfig;
    FormatBumpsHead := Boolean(ReadConfig);
    ExtendedDiskMode := ReadConfig;
    DiskExtBAMMode := ReadConfig;
    ImageExtBAMMode := ReadConfig;
    DiskCopyMode := ReadConfig;
    if not P then InvalidGCRCodeMode := ReadConfig;
    DelayValue := ReadConfig;
    RetryNum := ReadConfig;
    if not P then SmartRetryNum := ReadConfig;
    RetryHalftracks := Boolean(ReadConfig);
    RetryBumpsHead := Boolean(ReadConfig);
    EndlessRetry := Boolean(ReadConfig);
    CharStartMode := ReadConfig;
    EightColFont := Boolean(ReadConfig);
    LongNames := Boolean(ReadConfig);
    PreferLongNames := Boolean(ReadConfig);
    KeepLowerCase := Boolean(ReadConfig);
    LPTNum := ReadConfig;
    ParLPTNum := ReadConfig;
    for B := MaxRealPorts to MaxLPTPorts - 1 do LPTAddresses[B] := ReadConfigWord;
    SaverDelay := ReadConfig;
    OrigScreenWidth := ReadConfig;
    OrigScreenHeight := ReadConfig;
    OrigLeftPanWidth := ReadConfig;
    OrigPanWinSize := ReadConfig;
    OrigHalfPanWinSize := ReadConfig;
    LeftSel := Boolean(ReadConfig);
    LeftMode := ReadConfig;
    LeftOrigMode := ReadConfig;
    RightMode := ReadConfig;
    RightOrigMode := ReadConfig;
    LeftVis := Boolean(ReadConfig);
    RightVis := Boolean(ReadConfig);
    LeftOrigVis := Boolean(ReadConfig);
    RightOrigVis := Boolean(ReadConfig);
    LeftQuickView := ReadConfig;
    RightQuickView := ReadConfig;
    LeftColumnMode := ReadConfig;
    RightColumnMode := ReadConfig;
    LeftDOSNameColWidth := ReadConfig;
    RightDOSNameColWidth := ReadConfig;
    LeftCBMNameColWidth := ReadConfig;
    RightCBMNameColWidth := ReadConfig;
    LeftMiniStatus := ReadConfig;
    RightMiniStatus := ReadConfig;
    LeftSortMode := ReadConfig;
    RightSortMode := ReadConfig;
    LeftHiddenFiles := Boolean(ReadConfig);
    RightHiddenFiles := Boolean(ReadConfig);
    LeftFileFilter := ReadConfig;
    RightFileFilter := ReadConfig;
    PrgExt := ReadConfigStr(MaxCfgPrgExtLen, False);
    B := MaxCfgPathLen;
    if P then B := 80;
    LeftPath := ReadConfigStr(B, False);
    RightPath := ReadConfigStr(B, False);
    LoadPalettes(P);
    O := CheckConfigChecksum(O);
    if O then
    begin
      if MemoryCfgOK then
      begin
        ResetConfigCount(-1);
        Q := True;
      end
      else
      begin
        if MainProgram then
        begin
          if LongOpenFile(AddToPath(HomePath, DriveCfgFileName, chDirSep), ReadFile, fmReadOnly) = 0 then
          begin
            J := ExtFileSize(ReadFile);
            if (J > Length(DriveCfgSign) + 4) and (J <= Length(DriveCfgSign) + (4 + MaxDrives * SizeOf(TDriveConfig))) and
              ((J - (Length(DriveCfgSign) + 4)) mod (SizeOf(TDriveConfig) - 1) = 0) then
            begin
              ExtBlockRead(ReadFile, GCRBuffer[I], J);
              ResetConfigCount(I);
              Q := CheckSign(DriveCfgSign);
            end;
            if not Q and (J > Length(PrevDriveCfgSign) + 4) and (J <= Length(PrevDriveCfgSign) +
              (4 + MaxDrives * SizeOf(TPrevDriveConfig))) and
              ((J - (Length(PrevDriveCfgSign) + 4)) mod (SizeOf(TPrevDriveConfig) - 1) = 0) then
            begin
              ExtSeek(ReadFile, 0);
              ExtBlockRead(ReadFile, GCRBuffer[I], J);
              ResetConfigCount(I);
              Q := CheckSign(PrevDriveCfgSign);
              R := True;
            end;
            ExtClose(ReadFile);
          end
          else
          begin
            G := False;
          end;
        end;
      end;
      if Q then
      begin
        DriveNum := ReadConfig;
        if (DriveNum > 0) and (DriveNum <= MaxDrives) then
        begin
          CurDriveNum := ReadConfig;
          D := @DriveConfigs[0];
          for C := 0 to DriveNum - 1 do
          begin
            with D^ do
            begin
              Name := ReadConfigStr(MaxDriveNameLen, False);
              TransferMode := ReadConfig;
              SerialCable := ReadConfig;
              ParallelCable := ReadConfig;
              ManualTimeouts := Boolean(ReadConfig);
              LPTNum := ReadConfig;
              ParLPTNum := ReadConfig;
              DetectPortModes := ReadConfig;
              ExternalDrive := ReadConfig;
              if R then
              begin
                case Boolean(ReadConfig) of
                  False:
                  begin
                    ReadConfig;
                    ExtendedDiskMode := xtNever;
                  end;
                  True:
                  case Boolean(ReadConfig) of
                    False: ExtendedDiskMode := xtAlways;
                    True: ExtendedDiskMode := xtDetect;
                  end;
                end;
              end
              else
              begin
                ExtendedDiskMode := ReadConfig;
              end;
              if not R then DiskExtBAMMode := ReadConfig;
              DiskCopyMode := ReadConfig;
              if not R then InvalidGCRCodeMode := ReadConfig;
              DetectDiskChange := Boolean(ReadConfig);
              EndlessRetry := Boolean(ReadConfig);
              VerifyWrite := Boolean(ReadConfig);
              CmdExecMode := ReadConfig;
              HeadSpeed := ReadConfig;
              FormatBumpsHead := Boolean(ReadConfig);
              RetryNum := ReadConfig;
              if not R then SmartRetryNum := ReadConfig;
              RetryHalftracks := Boolean(ReadConfig);
              RetryBumpsHead := Boolean(ReadConfig);
              for B := 0 to DriveIntNum - 1 do DriveInts[B] := ReadConfig;
            end;
            Inc(D);
          end;
          Q := CheckConfigChecksum(Q);
        end;
      end;
    end;
  end;
  if not O then
  begin
    G := False;
    ResetConfig;
  end;
  if Q then
  begin
    if DriveNum > 0 then PutDrivePars(@DriveConfigs[CurDriveNum]);
  end
  else
  begin
    DriveNum := 0;
  end;
  if O or not F then ConfigStatus := ConfigStatus or ssMainOK;
  if P then ConfigStatus := ConfigStatus or ssMainPrev;
  if Q or not G then ConfigStatus := ConfigStatus or ssDriveOK;
  if R then ConfigStatus := ConfigStatus or ssDrivePrev;
end;

{Save the current checksums into the cofnioguration file buffer}
procedure SaveConfigChecksum;
begin
  GCRBuffer[ConfigCount] := ConfigXORSum;
  Inc(ConfigCount);
  GCRBuffer[ConfigCount] := ConfigSUBSum;
  Inc(ConfigCount);
end;

{Save the current configuration into the setup file or the DOS shell buffer
  Input : Dest: the destination to save the configuration into
  Output: when False, an error occured}
function SaveConfig(Dest: Byte): Boolean;
var
  B,
  C             : Byte;
  X,
  Y             : Word;
  I             : Integer;
  D             : PDriveConfig;

{If a panel is a disk/tape/file image or archive panel, save DOS mode instead
  Input : Mode: original mode of panel
  Output: fixed mode of panel}
function NoImagePanel(Mode: Byte): Byte;
begin
  if (Dest = sdFile) and GetPanelModeAttrib(Mode, (paImage + paArchive)) then NoImagePanel := pmDOS else NoImagePanel := Mode;
end;

begin
  SaveConfig := False;
  FillChar(GCRBuffer, SetupLen, 0);
  ConfigCount := 0;
  ConfigXORSum := 0;
  ConfigSUBSum := 0;
  if Dest = sdFile then for B := 1 to Length(ConfigSign) do WriteConfig(Ord(ConfigSign[B]));
  WriteConfig(ScreenCol);
  WriteConfig(Byte(SnowCheck));
  WriteConfig(Byte(VESASupport));
  WriteConfig(Byte(MouseReverse));
  WriteConfig(Byte(FastMouse));
  WriteConfig(Byte(ConvertConfirm));
  WriteConfig(Byte(DeleteConfirm));
  WriteConfig(Byte(QuitConfirm));
  WriteConfig(Byte(TransferConfirm));
  WriteConfig(Byte(DiskEditConfirm));
  WriteConfig(Byte(TransferWarning));
  WriteConfig(Byte(FileSizeWarning));
  WriteConfig(Byte(InsMovesDown));
  WriteConfig(Byte(AlternativeHotkeys));
  WriteConfig(Byte(EscTogglesPanels));
  WriteConfig(Byte(AltPopsMenu));
  WriteConfig(Byte(CursorFollowsFilename));
  WriteConfig(Byte(ErrorSound));
  WriteConfig(Byte(AutoSaveSetup));
  WriteConfig(Byte(AutoMenus));
  WriteConfig(Byte(PathPrompt));
  WriteConfig(Byte(ShowKeyBar));
  WriteConfig(Byte(ShowClock));
  WriteConfig(Byte(ShowMenu));
  WriteConfig(Byte(MakeBackup));
  WriteConfig(Byte(StartInfo));
  WriteConfig(Byte(ShowReadErrors));
  WriteConfig(Byte(AutoUnselect));
  WriteConfig(Byte(DetectDiskChange));
  WriteConfig(Byte(KeepTime));
  WriteConfig(Byte(DOSSizeBlocks));
  WriteConfig(Byte(GEOSSupport));
  WriteConfig(ExtractFileImages);
  WriteConfig(IntoFileImages);
  WriteConfig(Byte(CopyToDirTrack));
  WriteConfig(Byte(WipeFiles));
  WriteConfig(Byte(OrigPattern));
  WriteConfig(Byte(KeepNonStandardExt));
  WriteConfig(Byte(ConvInvalidChars));
  WriteConfig(Byte(KeepUpperCase));
  for B := 0 to (DiskTypeNum * 2) - 1 do WriteConfig(ImageInts[B]);
  WriteConfig(ExternalDrive);
  for B := 0 to DriveIntNum - 1 do WriteConfig(DriveInts[B]);
  WriteConfig(HeadSpeed);
  WriteConfig(TransferMode);
  WriteConfig(ParallelCable or (AsyncTransfer shl 6));
  WriteConfig(SerialCable);
  WriteConfig(Byte(DetectPortModes));
  WriteConfig(Byte(ManualTimeouts));
  WriteConfig(Byte(VerifyWrite));
  WriteConfig(CmdExecMode);
  WriteConfig(Byte(FormatBumpsHead));
  WriteConfig(ExtendedDiskMode);
  WriteConfig(DiskExtBAMMode);
  WriteConfig(ImageExtBAMMode);
  WriteConfig(DiskCopyMode);
  WriteConfig(InvalidGCRCodeMode);
  WriteConfig(DelayValue);
  WriteConfig(RetryNum);
  WriteConfig(SmartRetryNum);
  WriteConfig(Byte(RetryHalftracks));
  WriteConfig(Byte(RetryBumpsHead));
  WriteConfig(Byte(EndlessRetry));
  WriteConfig(CharSetMode);
  WriteConfig(Byte(EightColFont));
  WriteConfig(Byte(LongNames));
  WriteConfig(Byte(PreferLongNames));
  WriteConfig(Byte(KeepLowerCase));
  WriteConfig(LPTNum);
  WriteConfig(ParLPTNum);
  for B := MaxRealPorts to MaxLPTPorts - 1 do WriteConfigWord(LPTAddresses[B]);
  WriteConfig(SaverDelay);
  WriteConfig(OrigScreenWidth);
  WriteConfig(OrigScreenHeight);
  WriteConfig(OrigLeftPanWidth);
  WriteConfig(OrigPanWinSize);
  WriteConfig(OrigHalfPanWinSize);
  WriteConfig(Byte(LeftSel));
  WriteConfig(NoImagePanel(LeftMode));
  WriteConfig(NoImagePanel(LeftOrigMode));
  WriteConfig(NoImagePanel(RightMode));
  WriteConfig(NoImagePanel(RightOrigMode));
  WriteConfig(Byte(LeftVis));
  WriteConfig(Byte(RightVis));
  WriteConfig(Byte(LeftOrigVis));
  WriteConfig(Byte(RightOrigVis));
  WriteConfig(Byte(LeftQuickView));
  WriteConfig(Byte(RightQuickView));
  WriteConfig(LeftColumnMode);
  WriteConfig(RightColumnMode);
  WriteConfig(LeftDOSNameColWidth);
  WriteConfig(RightDOSNameColWidth);
  WriteConfig(LeftCBMNameColWidth);
  WriteConfig(RightCBMNameColWidth);
  WriteConfig(LeftMiniStatus);
  WriteConfig(RightMiniStatus);
  WriteConfig(LeftSortMode);
  WriteConfig(RightSortMode);
  WriteConfig(Byte(LeftHiddenFiles));
  WriteConfig(Byte(RightHiddenFiles));
  WriteConfig(LeftFileFilter);
  WriteConfig(RightFileFilter);
  WriteConfigStr(PrgExt, MaxCfgPrgExtLen);
  WriteConfigStr(LeftPath, MaxCfgPathLen);
  WriteConfigStr(RightPath, MaxCfgPathLen);
  for B := apBlackWhite to apMonochrome do WriteConfigStr(CApplication[B], PaletteLen);
  SaveConfigChecksum;
  X := ConfigCount;
  ConfigXORSum := 0;
  ConfigSUBSum := 0;
  Y := 0;
  if MainProgram then
  begin
    if Dest = sdFile then for B := 1 to Length(DriveCfgSign) do WriteConfig(Ord(DriveCfgSign[B]));
    WriteConfig(DriveNum);
    if DriveNum > 0 then
    begin
      WriteConfig(CurDriveNum);
      D := @DriveConfigs[0];
      for C := 0 to DriveNum - 1 do
      begin
        with D^ do
        begin
          WriteConfigStr(Name, MaxDriveNameLen);
          WriteConfig(TransferMode);
          WriteConfig(SerialCable);
          WriteConfig(ParallelCable);
          WriteConfig(Byte(ManualTimeouts));
          WriteConfig(LPTNum);
          WriteConfig(ParLPTNum);
          WriteConfig(Byte(DetectPortModes));
          WriteConfig(ExternalDrive);
          WriteConfig(ExtendedDiskMode);
          WriteConfig(DiskExtBAMMode);
          WriteConfig(DiskCopyMode);
          WriteConfig(InvalidGCRCodeMode);
          WriteConfig(Byte(DetectDiskChange));
          WriteConfig(Byte(EndlessRetry));
          WriteConfig(Byte(VerifyWrite));
          WriteConfig(CmdExecMode);
          WriteConfig(HeadSpeed);
          WriteConfig(Byte(FormatBumpsHead));
          WriteConfig(RetryNum);
          WriteConfig(SmartRetryNum);
          WriteConfig(Byte(RetryHalftracks));
          WriteConfig(Byte(RetryBumpsHead));
          for B := 0 to DriveIntNum - 1 do WriteConfig(DriveInts[B]);
        end;
        Inc(D);
      end;
      SaveConfigChecksum;
    end;
    Y := ConfigCount - X;
  end;
  SaveConfig := True;
  case Dest of
    sdFile:
    begin
      ClockOff;
      I := LongOpenFile(AddToPath(HomePath, ConfigFileName, chDirSep), WriteFile, fmWriteOnly);
      if I = 0 then
      begin
        ExtBlockWrite(WriteFile, GCRBuffer, X);
        ExtClose(WriteFile);
        I := IOResult;
        if I = 0 then
        begin
          if DriveNum > 0 then
          begin
            I := LongOpenFile(AddToPath(HomePath, DriveCfgFileName, chDirSep), WriteFile, fmWriteOnly);
            if I = 0 then
            begin
              ExtBlockWrite(WriteFile, GCRBuffer[X], Y);
              ExtClose(WriteFile);
              I := IOResult;
            end;
          end
          else
          begin
            LongErase(AddToPath(HomePath, DriveCfgFileName, chDirSep));
            I := IOResult;
            if I = deFileNotFound then I := 0;
          end;
        end;
      end;
      SaveConfig := (I = 0);
      ClockOn;
    end;
    sdShellBuffer:
    begin
      Move(GCRBuffer, ShellBuffer^.Config, SetupLen);
      ShellBuffer^.ComPath := HomePath;
      ShellBuffer^.ConfigOK := True;
      MemoryCfgOK := True;
    end;
  end;
end;

{Check whether the settings have been changed since the setup menu has been
  entered}
procedure CheckSetup;
var
  W             : Word;
begin
  SaveConfig(sdNone);
  SetupChanged := False;
  W := 0;
  while not SetupChanged and (W < ConfigCount) do
  begin
    if GCRBuffer[W] <> GCRBuffer[W + TempBufferSize - SetupLen] then SetupChanged := True;
    Inc(W);
  end;
end;

{Redraw the panels, after having changed an option that affects the panel
  display}
procedure RedrawPanels;
begin
{$IFNDEF ExternalSetup}
  Act^.DrawPanel;
  Inact^.DrawPanel;
{$ENDIF}
end;

{'Configuration' item in the main setup menu: set the configuration
  parameters}
function CfgConfig: Word; far;
begin
  CfgConfig := CfgMenus.CfgConfig;
{$IFNDEF ExternalSetup}
  ChangePanels;
{$ENDIF}
end;

{'Other options' item in the main setup menu: set additional configuration
  parameters}
function CfgOther: Word; far;
begin
  CfgOther := CfgMenus.CfgOther;
  RedrawPanels;
end;

{'Image options' item in the main setup menu: set image-related configuration
  parameters}
function CfgImage: Word; far;
begin
  CfgImage := CfgMenus.CfgImage;
  RedrawPanels;
end;

{'Transfer options' item in the main setup menu: set transfer-related
  configuration parameters}
function CfgTransfer: Word; far;
begin
  CfgTransfer := CfgMenus.CfgTransfer;
  CopyExtDisk := ExtendedDisk;
end;

{'Set default configuration' item in the main setup menu: reset the complete
  configuration to the defaults}
function CfgReset: Word; far;
begin
  CfgReset := CfgMenus.CfgReset;
{$IFNDEF ExternalSetup}
  ChangePanels;
  CommandLine^.DrawView;
{$ENDIF}
end;

{Exit the main config menu and ask for confirmation if some options have been
  changed}
function ExitConfig: Word; far;
var
  O             : Boolean;
  A,
  P             : PSetupItem;
  R             : TRect;
begin
  ChangeHelpCtx(hcCfgQuit);
  SetupCmd := cmNo;
  CheckSetup;
  if SetupChanged then SetupCmd := SureConfirm('Setup', 'Do you wish to keep the changes', 'you''ve made in the setup?',
    stEmpty, stEmpty, stYes, stEmpty, stEmpty, stNo, stEmpty, nil, CurHelpCtx, ayNone, True, DummyByte);
  RestoreHelpCtx;
  if SetupCmd = cmCancel then SetupDialog^.Show;
  ExitConfig := cmOK;
end;

{Execute the procedure responsible for displaying a given item in the main
  configuration menu
  Input : Event: event record to get command from and put new event into
  Output: when True, the command was recognized}
function MainConfigExecMenuProc(var Event: TEvent): Boolean; far;
begin
  MainConfigExecMenuProc := True;
  case Event.Command of
    cmCfgConfig: SetupMenu^.ExecProc(CfgConfig, False, False, cmCfgOther, cmCfgDrive, Event);
    cmCfgOther: SetupMenu^.ExecProc(CfgOther, False, False, cmCfgImage, cmCfgConfig, Event);
    cmCfgImage: SetupMenu^.ExecProc(CfgImage, False, False, cmCfgTransfer, cmCfgOther, Event);
    cmCfgTransfer: SetupMenu^.ExecProc(CfgTransfer, False, False, cmCfgDrive, cmCfgImage, Event);
    cmCfgDrive: SetupMenu^.ExecProc(CfgDrive, False, False, cmCfgConfig, cmCfgTransfer, Event);
    cmCfgPalette: SetupMenu^.ExecProc(CfgPalette, True, True, cmNone, cmNone, Event);
    cmCfgReset: SetupMenu^.ExecProc(CfgReset, True, True, cmNone, cmNone, Event);
    cmCfgCountry: SetupMenu^.ExecProc(CfgCountry, False, False, cmNone, cmNone, Event);
  else
    MainConfigExecMenuProc := False;
  end;
end;

{'Configuration' item in the 'Options' menu: display main configuration menu
  and execute its items}
procedure MainConfig;
var
  O             : Boolean;
  B             : Byte;
  A,
  P             : PSetupItem;
  R             : TRect;
begin
  ChangeHelpCtx(hcMainConfig2);
  O := EightColFont;
  B := SerialCable;
{$IFNDEF ExternalSetup}
  CommandLine^.HideCursor;
{$ENDIF}
  P := NewSetupItem('F2', 'Configuration', cmCfgConfig, hcCfgConfig, nil);
  A := P;
  P := NewSetupItem('F3', 'Other options', cmCfgOther, hcCfgOther, P);
  P := NewSetupItem('F4', 'Image options', cmCfgImage, hcCfgImage, P);
  P := NewSetupItem('F5', 'Transfer options', cmCfgTransfer, hcCfgTransfer, P);
  P := NewSetupItem('F6', 'Drive options', cmCfgDrive, hcCfgDrive, P);
  P := NewSetupItem('F7', 'Set palettes', cmCfgPalette, hcCfgPalette, P);
  P := NewSetupItem('F8', 'Set default configuration', cmCfgReset, hcCfgReset, P);
  P := NewSetupItem('F9', 'Country info', cmCfgCountry, hcCfgCountry, P);
  P := NewSetupItem('F10', 'Exit', cmCancel, hcCfgQuit, P);
  R.Assign(4, 2, 0, 11);
  SetupMenu := New(PSetupMenu, Init(R, A, A, MainConfigExecMenuProc));
  MakeWinBounds(R, SetupMenu^.Size.X, 9);
  SetupDialog := New(PDialog, Init(R, 'Setup', fxNormal, fyNormal, False));
  SetupDialog^.Insert(SetupMenu);
  SetupDialog^.Palette := wpHistory;
  KeyBar^.MakeFirst;
  SaveConfig(sdNone);
  Move(GCRBuffer, GCRBuffer[TempBufferSize - SetupLen], SetupLen);
  TempDialog := nil;
  Application^.ExecView(SetupDialog, True, True);
  Dispose(SetupDialog, Done);
  RestoreHelpCtx;
  if SetupCmd = cmSkip then
  begin
    Move(GCRBuffer[TempBufferSize - SetupLen], ShellBuffer^.Config, SetupLen);
    ShellBuffer^.ConfigOK := True;
    MemoryCfgOK := True;
    LoadConfig;
  end;
  if DriveNum > 0 then GetDrivePars(@DriveConfigs[CurDriveNum]);
  InitLongNames;
  InitSelectMode;
  if B <> SerialCable then ResetLPTPort;
  InitTransfer(True);
  if (EightColFont <> O) and (CharSetMode >= csCBMLower) then
  begin
    if O then SetVideoMode(ScreenMode);
    Application^.SetCharSet(CharSetMode, True, True);
  end;
  Application^.InitScreen;
{$IFNDEF ExternalSetup}
  ChangePanels;
{$ENDIF}
  if ShowClock then Clock^.Show else Clock^.Hide;
  if ShowKeyBar then KeyBar^.Show else KeyBar^.Hide;
{$IFNDEF ExternalSetup}
  if SetupChanged and (SetupCmd = cmOK) then
  begin
    if not AutomaticSaveSetup then ErrorWin(stEmpty, 'There was an error while', 'saving the setup file',
      hcNoContext, sbNone);
    ReadPanel(Act, False);
    ReadPanel(Inact, False);
  end
  else
  begin
    Act^.Loading := False;
    Inact^.Loading := False;
    Act^.DrawView;
    Inact^.DrawView;
  end;
  CommandLine^.ShowCursor;
  CommandLine^.DrawView;
{$ENDIF}
end;

{Get a name
  Input : Title: title for the dialog box
          Text: text to display above input line
          MaxLen: maximum length of string
  Output: when True, the user accepted the operation}
function GetName(const Title, Text: string; MaxLen: Integer): Boolean;
var
  C             : Word;
  D             : PDialog;
  I             : PInputLine;
  T             : string;
  R             : TRect;
begin
  T := Title;
  if T = '' then T := BoxTitle;
  MakeWinBounds(R, 66, 2);
  GetClock(True);
  GetMouse(True);
  D := New(PDialog, Init(R, T, fxNormal, fyNormal, True));
  R.Assign(5, 2, 64, 2);
  I := New(PInputLine, Init(R, 64, MaxLen, Text, drUp));
  I^.SetData(SourceName);
  D^.Insert(I);
  C := Application^.ExecView(D, True, True);
  if C = cmOK then
  begin
    I^.GetData(SourceName);
    if SourceName = '' then C := cmCancel;
  end;
  Dispose(D, Done);
  GetName := (C = cmOK);
  SetMouse;
  SetClock;
end;

{Reset a drive configuration
  Input : Title: the title for the dialog boxes
          Drive: the drive parameter structure to reset
  Output: when True, the user accepted the operation}
function ResetDrive(const Title: string; Drive: PDriveConfig): Boolean;
var
  B             : Byte;
  C             : Word;
begin
  ResetDrive := True;
  if DriveNum = 0 then
  begin
    C := cmOK;
  end
  else
  begin
    C := Confirm(Title, 'Do you wish to get the global drive parameters', 'or the default parameters for a drive type?',
      ' '+ColorChar+'G'+ColorChar+'lobal ', ' '+ColorChar+'D'+ColorChar+'rive default ', stEmpty,
      nil, CurHelpCtx, True, DummyByte);
  end;
  case C of
    cmOK: GetDrivePars(Drive);
    cmSkip:
    begin
      C := SelectMenuInputItem('Select drive type', xd1541, xd157xEmu, xd1541, 1, DriveTypeStr);
      RestoreHelpCtx;
      if C = cmOK then ResetDrivePars(Drive, HistoryItem) else ResetDrive := False;
    end;
  else
    ResetDrive := False;
  end;
end;

{Get the index of the current drive in the list}
function GetDriveIndex: Byte;
begin
  if DriveNum = 0 then GetDriveIndex := 0 else GetDriveIndex := HistoryItem - DriveCmdBase;
end;

{Get a name for a drive
  Input : Title: title for the dialog box
          Text: text to display above input line
  Output: when True, the drive name is OK}
function GetDriveName(const Title, Text: string): Boolean;
var
  O             : Boolean;
  B             : Byte;
begin
  O := False;
  if GetName(Title, Text, MaxDriveNameLen) then
  begin
    B := 0;
    O := True;
    while O and (B < DriveNum) do
    begin
      O := (SourceName <> DriveConfigs[B].Name);
      Inc(B);
    end;
    if not O then
    begin
      ErrorWin(stError, 'A drive named "' + SourceName + '"', 'already exists.', CurHelpCtx, sbNone);
      SourceName := '';
    end;
  end;
  GetDriveName := O;
end;

{Generate menu items for the drives
  Input : OldItems: when not nil, the old items have to be freed first
  Output: the menu item list}
function GenerateDriveItems(OldItems: PSetupItem): PSetupItem;
var
  W             : Word;
  I             : Integer;
  A,
  P             : PSetupItem;
  S,
  T             : string;
begin
  if OldItems <> nil then FreeSetupItem(OldItems);
  I := 0;
  W := DriveCmdBase;
  A := nil;
  P := nil;
  while I <= DriveNum do
  begin
    S := '';
    T := '';
    if I < DriveNum then
    begin
      S := DriveConfigs[I].Name;
      T := HexaStr(I + 1, 1);
    end;
    P := NewSetupItem(T, S, W, hcDriveConfig, P);
    if I = 0 then A := P;
    Inc(I);
    Inc(W);
  end;
  GenerateDriveItems := A;
end;

{Regenerate the setup menu with new items
  Input : Index: index of the drive item under the cursor}
procedure RegenerateSetupMenu(Index: Byte);
var
  X             : Integer;
  P             : PSetupItem;
  R             : TRect;
begin
  SetupMenu^.Items := GenerateDriveItems(SetupMenu^.Items);
  P := SetupMenu^.Items;
  while (P <> nil) and (Index > 0) do
  begin
    P := P^.Next;
    Dec(Index);
  end;
  SetupMenu^.CurItem := P;
  P := SetupMenu^.Items;
  X := 15;
  while P <> nil do
  begin
    if X < Length(P^.Title) then X := Length(P^.Title);
    P := P^.Next;
  end;
  Inc(X, 4);
  R.Assign(4, 2, X + 4, DriveNum + 3);
  SetupMenu^.SetBounds(R);
  MakeWinBounds(R, X, DriveNum + 1);
  Inc(R.B.X, R.A.X);
  Inc(R.B.Y, R.A.Y);
  SetupDialog^.ChangeBounds(R);
end;

{Create a new drive
  Input : ResetNewDrive: when True, parameters of the newly created drive are
                         reset}
procedure MakeNewDrive(ResetNewDrive: Boolean);
var
  F             : Boolean;
  B,
  C             : Integer;
  S             : string;
  D             : TDriveConfig;
begin
  B := GetDriveIndex;
  if (DriveNum < MaxDrives) and (ResetNewDrive or ((DriveNum > 0) and (B < DriveNum))) then
  begin
    SourceName := '';
    S := 'New drive';
    if not ResetNewDrive then S := 'Duplicate drive';
    if GetDriveName(S, 'Enter name for new drive') then
    begin
      if not ResetNewDrive or ResetDrive('New drive', @D) then
      begin
        if DriveNum > 0 then
        begin
          C := DriveNum - 1;
          while C >= B do
          begin
            DriveConfigs[C + 1] := DriveConfigs[C];
            Dec(C);
          end;
        end;
        if ResetNewDrive then DriveConfigs[B] := D;
        C := B;
        if not ResetNewDrive then Inc(C);
        DriveConfigs[C].Name := SourceName;
        if DriveNum = 0 then CurDriveNum := 0 else if CurDriveNum >= C then Inc(CurDriveNum);
        Inc(DriveNum);
        RegenerateSetupMenu(B);
      end;
    end;
    RedrawAllViews;
  end;
end;

{Insert a new drive
  Output: when cmOK, the user accepted the changes}
function DrvCfgNew: Word; far;
begin
  MakeNewDrive(True);
  DrvCfgNew := cmOK;
end;

{Edit the drive parameters
  Input : MenuProc: the procedure to execute
  Output: when cmOK, the user accepted the changes}
function DrvCfgEdit(MenuProc: TMenuProc): Word; far;
var
  B             : Byte;
  C             : Word;
  D             : TDriveConfig;
begin
  B := GetDriveIndex;
  if B < DriveNum then
  begin
    GetDrivePars(@D);
    PutDrivePars(@DriveConfigs[B]);
    C := MenuProc;
    if C <> cmCancel then GetDrivePars(@DriveConfigs[B]);
    PutDrivePars(@D);
  end;
  DrvCfgEdit := C;
end;

{Edit the transfer parameters of a drive
  Output: when cmOK, the user accepted the changes}
function DrvCfgTransfer: Word; far;
begin
  DrvCfgTransfer := DrvCfgEdit(CfgMenus.CfgTransfer);
end;

{Edit the drive parameters of a drive
  Output: when cmOK, the user accepted the changes}
function DrvCfgDrive: Word; far;
begin
  DrvCfgDrive := DrvCfgEdit(CfgMenus.CfgDrive);
end;

{Duplicate a drive
  Output: when cmOK, the user accepted the changes}
function DrvCfgDupe: Word; far;
begin
  MakeNewDrive(False);
  DrvCfgDupe := cmOK;
end;

{Rename a drive
  Output: when cmOK, the user accepted the changes}
function DrvCfgRename: Word; far;
var
  B             : Byte;
begin
  B := GetDriveIndex;
  if B < DriveNum then
  begin
    SourceName := DriveConfigs[B].Name;
    if GetDriveName('Rename drive', 'Enter new name for drive') then
    begin
      DriveConfigs[B].Name := SourceName;
      RegenerateSetupMenu(B);
    end;
    SetupMenu^.DrawView;
  end;
  DrvCfgRename := cmOK;
end;

{Swap a drive with the next one
  Output: when cmOK, the user accepted the changes}
function DrvCfgSwap: Word; far;
var
  B             : Byte;
  D             : TDriveConfig;
begin
  if DriveNum > 0 then
  begin
    B := GetDriveIndex;
    if B + 1 < DriveNum then
    begin
      D := DriveConfigs[B];
      DriveConfigs[B] := DriveConfigs[B + 1];
      DriveConfigs[B + 1] := D;
      if CurDriveNum = B then Inc(CurDriveNum) else
        if CurDriveNum = B + 1 then Dec(CurDriveNum);
      Inc(B);
      RegenerateSetupMenu(B);
      SetupMenu^.DrawView;
    end;
  end;
  DrvCfgSwap := cmOK;
end;

{Delete a drive
  Output: when True, the user accepted the changes}
function DrvCfgDelete: Word; far;
var
  B,
  C             : Integer;
  P             : PSetupItem;
  R             : TRect;
  D             : TDriveConfig;
begin
  if DriveNum > 0 then
  begin
    B := GetDriveIndex;
    if (B < DriveNum) and (Confirm('Delete drive', 'Do you wish to delete', 'drive "' + DriveConfigs[B].Name + '"?',
      stDelete, stEmpty, stEmpty, nil, CurHelpCtx, True, DummyByte) = cmOK) then
    begin
      C := B;
      while C < DriveNum do
      begin
        DriveConfigs[C] := DriveConfigs[C + 1];
        Inc(C);
      end;
      if (DriveNum = 0) or (CurDriveNum = B) then CurDriveNum := 0 else if CurDriveNum > B then Dec(CurDriveNum);
      Dec(DriveNum);
      if (B > 0) and (B >= DriveNum) then Dec(B);
      RegenerateSetupMenu(B);
    end;
    RedrawAllViews;
  end;
  DrvCfgDelete := cmOK;
end;

function DrvCfgReset: Word; far;
var
  B             : Byte;
begin
  if DriveNum > 0 then
  begin
    B := GetDriveIndex;
    if B < DriveNum then ResetDrive('Reset drive', @DriveConfigs[B]);
  end;
  DrvCfgReset := cmOK;
end;

{Execute the procedure responsible for displaying a given item in the drive
  configuration menu
  Input : Event: event record to get command from and put new event into
  Output: when True, the command was recognized}
function DriveConfigExecMenuProc(var Event: TEvent): Boolean; far;
var
  B             : Byte;
begin
  DriveConfigExecMenuProc := True;
  case Event.Command of
    cmDrvCfgNew: SetupMenu^.ExecProc(DrvCfgNew, False, False, cmNone, cmNone, Event);
    cmDrvCfgTransfer: SetupMenu^.ExecProc(DrvCfgTransfer, False, False, cmDrvCfgDrive, cmDrvCfgDrive, Event);
    cmDrvCfgDrive: SetupMenu^.ExecProc(DrvCfgDrive, False, False, cmDrvCfgTransfer, cmDrvCfgTransfer, Event);
    cmDrvCfgDupe: SetupMenu^.ExecProc(DrvCfgDupe, False, False, cmNone, cmNone, Event);
    cmDrvCfgRename: SetupMenu^.ExecProc(DrvCfgRename, False, False, cmNone, cmNone, Event);
    cmDrvCfgSwap: SetupMenu^.ExecProc(DrvCfgSwap, False, False, cmNone, cmNone, Event);
    cmDrvCfgDelete: SetupMenu^.ExecProc(DrvCfgDelete, False, False, cmNone, cmNone, Event);
    cmDrvCfgReset: SetupMenu^.ExecProc(DrvCfgReset, False, False, cmNone, cmNone, Event);
  else
    if Event.Command >= DriveCmdBase then
    begin
      B := GetDriveIndex;
      if B < DriveNum then
      begin
        CurDriveNum := B;
        PutDrivePars(@DriveConfigs[B]);
        Event.Command := cmOK;
      end
      else
      begin
        Event.What := evNothing;
      end;
    end
    else
    begin
      DriveConfigExecMenuProc := False;
    end;
  end;
end;

{'Drive setup' item in the 'Options' menu: display drive configuration menu
  and execute its items}
procedure DriveConfig;
var
  B,
  I             : Byte;
  W             : Word;
  A,
  C,
  P             : PSetupItem;
  R             : TRect;
begin
  ChangeHelpCtx(hcDriveConfig2);
  SetupHelpCtx := hcDriveConfig;
  B := SerialCable;
{$IFNDEF ExternalSetup}
  CommandLine^.HideCursor;
{$ENDIF}
  R.Assign(0, 0, 0, 0);
  SetupMenu := New(PSetupMenu, Init(R, nil, nil, DriveConfigExecMenuProc));
  R.Assign(0, 0, 0, 0);
  SetupDialog := New(PDialog, Init(R, 'Drive setup', fxNormal, fyNormal, False));
  SetupDialog^.Insert(SetupMenu);
  SetupDialog^.Palette := wpHistory;
  RegenerateSetupMenu(CurDriveNum);
  KeyBar^.MakeFirst;
  SaveConfig(sdNone);
  Move(GCRBuffer, GCRBuffer[TempBufferSize - SetupLen], SetupLen);
  TempDialog := nil;
  Application^.ExecView(SetupDialog, True, True);
  Dispose(SetupDialog, Done);
  RestoreHelpCtx;
  if SetupCmd = cmSkip then
  begin
    Move(GCRBuffer[TempBufferSize - SetupLen], ShellBuffer^.Config, SetupLen);
    ShellBuffer^.ConfigOK := True;
    MemoryCfgOK := True;
    LoadConfig;
  end;
  if B <> SerialCable then ResetLPTPort;
  InitTransfer(True);
{$IFNDEF ExternalSetup}
  if SetupChanged and (SetupCmd = cmOK) then
  begin
    if DriveNum > 0 then PutDrivePars(@DriveConfigs[CurDriveNum]);
    if not AutomaticSaveSetup then ErrorWin(stEmpty, 'There was an error while', 'saving the setup file',
      hcNoContext, sbNone);
    if Left^.Mode = pmExt then Left^.Read;
    if Right^.Mode = pmExt then Right^.Read;
  end
  else
  begin
    Act^.Loading := False;
    Inact^.Loading := False;
    Act^.DrawView;
    Inact^.DrawView;
  end;
  CommandLine^.ShowCursor;
  CommandLine^.DrawView;
{$ENDIF}
end;

{Initialize the setup menu}
constructor TSetupMenu.Init(var Bounds: TRect; AItems, ACurItem: PSetupItem; AExecMenuProc: TExecMenuProc);
var
  P             : PSetupItem;
begin
  P := AItems;
  while (P <> nil) do
  begin
    if Bounds.B.X < Length(P^.Title) then Bounds.B.X := Length(P^.Title);
    P := P^.Next;
  end;
  Inc(Bounds.B.X, 8);
  TView.Init(Bounds);
  Options := Options or ofSelectable;
  EventMask := EventMask or evBroadcast;
  Items := AItems;
  CurItem := ACurItem;
  ExecMenuProc := AExecMenuProc;
end;

{Deallocate the memory used by the setup menu and its items}
destructor TSetupMenu.Done;
begin
  TView.Done;
  FreeSetupItem(Items);
end;

{Execute setup menu function
  Input : P: procedure to be executed
          MenuOff: when True, the configuration menu is hidden before the
                   function is executed
          MenuOn: when True, the configuration menu is shown after the
                  function has been executed
          NextMenu, PrevMenu: event commands to execute next or previous
                              menu
          Event: event record to put new event into}
procedure TSetupMenu.ExecProc(P: TMenuProc; MenuOff, MenuOn: Boolean; NextMenu, PrevMenu: Word; var Event: TEvent);
var
  O             : Boolean;
  H             : Word;
begin
  H := CurHelpCtx;
  O := HelpCtxSet;
  if TempDialog = nil then KeyBar^.Update;
  if MenuOff then SetupDialog^.Hide;
  Event.Command := cmNone;
  case P of
    cmYes: Event.Command := NextMenu;
    cmNo: Event.Command := PrevMenu;
  end;
  if Event.Command = cmNone then ClearEvent(Event);
  CurHelpCtx := H;
  AppHelpCtx := H;
  HelpCtxSet := O;
  LastShiftState := MaxByte;
  if Event.What = evNothing then
  begin
    if MenuOn or not SetupDialog^.GetState(sfVisible) then SetupDialog^.Show;
    if TempDialog <> nil then
    begin
      Dispose(TempDialog, Done);
      TempDialog := nil;
    end;
    KeyBar^.Update;
  end
  else
  begin
    SetupDialog^.Hide;
  end;
end;

{Handle setup menu events}
procedure TSetupMenu.HandleEvent(var Event: TEvent);
var
  O             : Boolean;
  B             : Byte;
  C             : Char;
  W             : Word;
  P             : PSetupItem;
  T             : TPoint;
begin
  O := False;
  if Event.What and evKeyboard > 0 then
  begin
    B := 0;
    W := Event.KeyCode;
    case Event.KeyCode of
      kbEsc:
      begin
        Event.What := evCommand;
        if MainProgram then Event.Command := cmOK else Event.Command := cmCancel;
      end;
      kbEnter, kbCtrlEnter:
      begin
        P := Items;
        B := 1;
        while (P <> nil) and (P <> CurItem) do
        begin
          Inc(B);
          P := P^.Next;
        end;
        O := (P <> nil);
        if O then
        begin
          Event.What := evCommand;
          Event.Command := P^.Command;
        end
        else
        begin
          ClearEvent(Event);
        end;
      end;
      kbUp:
      begin
        if CurItem = Items then P := nil else P := CurItem;
        CurItem := Items;
        while (CurItem <> nil) and (CurItem^.Next <> P) do CurItem := CurItem^.Next;
        DrawView;
      end;
      kbDown:
      begin
        CurItem := CurItem^.Next;
        if CurItem = nil then CurItem := Items;
        DrawView;
      end;
      kbPgUp, kbHome:
      begin
        CurItem := Items;
        DrawView;
      end;
      kbPgDn, kbEnd:
      begin
        CurItem := Items;
        while CurItem^.Next <> nil do CurItem := CurItem^.Next;
        DrawView;
      end;
    else
      P := Items;
      if Event.CharCode >= ' ' then W := Ord(UpCase(Event.CharCode));
      while (P <> nil) and (P^.HotCode <> W) do P := P^.Next;
      if P <> nil then
      begin
        CurItem := P;
        Event.What := evCommand;
        Event.Command := P^.Command;
      end;
    end;
  end;
  if Event.What and evMouse > 0 then
  begin
    repeat
      if MouseInView(Event.Where) then
      begin
        O := Event.Double;
        MakeLocal(Event.Where, T);
        B := 1;
        P := Items;
        while (P <> nil) and (B <= T.Y) do
        begin
          Inc(B);
          P := P^.Next;
        end;
        if P <> nil then CurItem := P;
        DrawView;
      end;
    until not MouseEvent(Event, evMouseMove);
    if O and MouseInView(Event.Where) then
    begin
      Event.What := evCommand;
      Event.Command := P^.Command;
    end;
  end;
  if Event.What and evCommand > 0 then
  begin
    repeat
      O := True;
      if CurItem <> nil then HistoryItem := CurItem^.Command;
      case Event.Command of
        cmOK:
        begin
          O := False;
          CheckSetup;
          Event.Command := cmCancel;
          SetupCmd := cmOK;
        end;
        cmCancel:
        begin
          ExecProc(ExitConfig, True, False, cmNone, cmNone, Event);
          ClearEvent(Event);
          if SetupCmd <> cmCancel then
          begin
            O := False;
            Event.What := evCommand;
            Event.Command := cmCancel;
            Application^.PutEvent(Event);
          end;
        end;
      else
        O := ExecMenuProc(Event);
      end;
    until (Event.What <> evCommand) or not O;
  end;
end;

{Draw the setup menu}
procedure TSetupMenu.Draw;
var
  A,
  C             : Word;
  Y             : Integer;
  P             : PSetupItem;
  S             : string;
  B             : TDrawBuffer;
begin
  A := GetColor($0101);
  C := GetColor($0202);
  P := Items;
  for Y := 0 to Size.Y - 1 do
  begin
    MoveChar(B, ' ', A, Size.X);
    if P <> nil then
    begin
      if P = CurItem then MoveChar(B[5], ' ', C, Size.X - 6);
      MoveChar(B[Size.X - 1], ' ', A, 1);
      S := Copy(P^.Title, 1, 3);
      MoveStr(B[1], S, A);
      MoveChar(B[4], ' ', A, 1);
      S := Copy(P^.Title, 4, MaxStrLen);
      if P = CurItem then
      begin
        MoveStr(B[5], S, C);
        SetupHelpCtx := P^.HelpCtx;
      end
      else
      begin
        MoveStr(B[5], S, A);
      end;
      P := P^.Next;
    end;
    WriteBuf(0, Y, Size.X, 1, B);
  end;
end;

end.
