
{*************************************************}
{                 Joe Forster/STA                 }
{                                                 }
{                    CPDISK.PAS                   }
{                                                 }
{        The Star Commander Copy disk unit        }
{*************************************************}

unit Cpdisk;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O+,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Panel1;

var
  SkipAuto      : Boolean;
  AllDiskCopy   : Byte;

procedure DisplayTS(Panel: PPanel);
function GetTrackType(Panel: PPanel): Byte;
function OpenDiskWrite(FirstOpen: Boolean): Boolean;
procedure CloseDiskWrite(InitDisk: Boolean);
procedure ConvertTrack(Track, Src, Dest, SecNum: Byte; SelBlocks: Boolean);
function WriteTrack(SelBlocks: Boolean): Boolean;
procedure CopyDisk(Mode: Byte);

implementation

uses
  App, Dialogs, DOS, Drivers, Menus, Objects, Views,
  Base1, Base2, Constant, Disked, ExtFiles, FCopy, LowLevel, MiscFunc, Panel2, Script, XferLo;

{Initalize the disk copy progress indicator
  Input : D: dialog box containing the information
          Text1, Text2, Text3: additional data to be displayed in the box}
procedure MakeDisplayTS(var D: PDialog; const Text1, Text2, Text3: string);
var
  S             : string[10];
begin
  S := 'Copying';
  if AppendFile then S := 'Merging';
  D := CopyInfoWin(stEmpty, S + Text1, Text2, Text3, 'Track:      Sector:   ');
  Application^.Insert(D);
end;

{Create a drive status code out of the error code in the error message
  Input : S: the error message}
procedure MakeDriveStatus(const S: string);
var
  B             : Byte;
begin
  Val(Copy(S, Length(S) - 1, 2), Act^.Sector, NumOK);
  Val(Copy(S, 1, 2), B, NumOK);
  if NumOK = 0 then DriveStatus := ErrorCodeToStatus(B);
end;

{Update the disk copy progress indicator
  Input : Panel: the panel with the external drive}
procedure DisplayTS(Panel: PPanel);
var
  S             : string[2];
begin
  S := LeadingSpace(Panel^.Track, 2);
  CopyInd^.Text^[8] := S[1];
  CopyInd^.Text^[9] := S[2];
  S := LeadingSpace(Panel^.Sector, 2);
  CopyInd^.Text^[21] := S[1];
  CopyInd^.Text^[22] := S[2];
  CopyInd^.Draw;
end;

{Determine the track type for a panel mode
  Input : Panel: the panel
  Output: the track type}
function GetTrackType(Panel: PPanel): Byte;
begin
  GetTrackType := ttNormal;
  case Panel^.CopyMode of
    pmExt: if CopyTransferMode = tmWarp then GetTrackType := ttGCR;
    pmDisk: if Panel^.FCOPYBlock then GetTrackType := ttNormalF64;
    pmGCRDisk: GetTrackType := ttRaw;
    pmSixZip: GetTrackType := ttGCR;
  end;
end;

{Compute the checksum for a sector in the track buffer
  Input : Sector: the sector number
  Output: the checksum}
function GetSectorChecksum(Sector: Byte): Byte;
var
  C             : Byte;
  X,
  Y             : Word;
begin
  X := (Sector shl 8);
  Y := 256;
  C := 0;
  while (Y > 0) do
  begin
    C := C xor TempBuffer[X];
    Inc(X);
    Dec(Y);
  end;
  GetSectorChecksum := C;
end;

{Clear all data in track buffer
  Input : TrackType: type of the track data}
procedure ClearTrackData(TrackType: Byte);
var
  M,
  S             : Byte;
  J,
  K             : Word;
begin
  M := Act^.SectorNum(Act^.Track) - 1;
  case TrackType of
    ttNormal, ttNormalF64:
    begin
      for S := 0 to M do FillFormatPattern(@TempBuffer[S shl 8]);
      if TrackType = ttNormalF64 then
      begin
        J := Act^.DiskPos(Act^.Track, 0);
        K := Act^.CopyDiskSize;
        for S := 0 to M do
        begin
          InfoBuffer^[J + K] := HeaderSign;
          InfoBuffer^[J + (K shl 1)] := GetSectorChecksum(S);
          Inc(J);
        end;
      end;
    end;
    ttGCR:
    begin
      for S := 0 to M do
      begin
        CreateSectorHeader(Act^.Track, S, FirstHeaderPadding(Act^.CopyDiskType, Act^.ExtBAMMode));
(* ?ASM? *)
        asm
          mov al, S;
          mov ah, HeaderGCRSize;
          mul ah;
          mov si, Offset(UndoBuffer);
          mov di, Offset(HeaderBuffer);
          add di, ax;
          call GCREncodeHeader;
        end;
        FillFormatPattern(@UndoBuffer);
        asm
          mov si, Offset(UndoBuffer);
          mov al, S;
          xor ah, ah;
          mov di, 326;
          mul di;
          mov di, Offset(GCRBuffer);
          add di, ax;
          mov al, True;
          call GCREncodeSector;
        end;
      end;
    end;
  end;
end;

{Note sector or track error in the error info buffer
  Input : Sector: sector number of bad block
          WholeTrack: when True, the whole track is filled with the error}
procedure NoteError(Sector: Byte; WholeTrack: Boolean);
var
  I,
  J,
  M             : Integer;
begin
  J := Act^.DiskPos(Act^.Track, 0);
  M := Act^.SectorNum(Act^.Track);
  if WholeTrack then
  begin
    for I := 0 to M - 1 do
    begin
      InfoBuffer^[J] := DriveStatus;
      Inc(J);
    end;
    ClearTrackData(GetTrackType(Inact));
  end
  else
  begin
    if Sector < M then
    begin
      Inc(J, Sector);
      InfoBuffer^[J] := DriveStatus;
    end;
  end;
end;

{Open source disk or disk image
  Input : FirstOpen: when True, the source is opened for the first time
  Output: when False, an error occured}
function OpenDiskRead(FirstOpen: Boolean): Boolean;
var
  B,
  O             : Boolean;
  C             : Byte;
  W             : Word;
  L             : Longint;
  N             : string;
  X             : array [0..2] of Byte;
begin
  with Act^ do
  begin
    O := False;
    TurboOn := False;
    FileTime := 0;
    FCOPYBlock := False;
    case CopyMode of
      pmExt:
      begin
        CBMDevNum := CopyCBMDev;
        if CheckDevice then
        begin
          SendConfigData;
          case CopyTransferMode of
            tmNormal:
            begin
              if not FirstOpen then CloseCBMChannel(saData);
              OpenCBMChannel(saData, '#', True);
            end;
            tmTurbo:
            begin
              if FirstOpen then OpenCBMChannel(saCommand, 'I0', True);
              TurboOn := (SendDriveProg(deTurboDiskLoad, True) and ExecDriveProg(deTurboDiskLoad, stEmpty));
            end;
            tmWarp:
            begin
              if FirstOpen then OpenCBMChannel(saCommand, 'I0', True);
              TurboOn := (SendDriveProg(deWarpDiskLoad, True) and ExecDriveProg(deWarpDiskLoad, stEmpty));
            end;
          end;
        end;
        O := True;
        if not TurboOn then
        begin
          O := ReadCBMError(N, True, True, True);
          if not O then ErrorWin(stError, N, stEmpty, CurHelpCtx, sbNone);
        end;
      end;
      pmDisk:
      begin
        if FirstOpen then
        begin
          O := (LongOpenFile(AddToPath(CopyPath, CopyImageName, chDirSep), ReadFile, fmReadOnly) = 0);
          if O then
          begin
            CopyDiskType := GetDiskType(ExtFileSize(ReadFile));
            CheckDiskType;
            B := False;
            L := ReadTime;
            if CopyDiskType and dtErrorInfo = 0 then
            begin
              if (CopyDiskType and dtTypeMask in [dt1541, dt1541Ext]) and
                (LongOpenFile(AddToPath(CopyPath, MakeFileExt(CopyImageName, FCOPYExt), chDirSep),
                TempFile, fmReadOnly) = 0) then
              begin
                W := SureConfirm(stEmpty, 'An FCOPY-PC info file is present.',
                  TempFile.LongName, 'Do you wish to use the data in it?', stEmpty, ' '+ColorChar+'U'+ColorChar+'se ',
                  stSkip, stEmpty, stEmpty, stEmpty, nil, CurHelpCtx, ayAllNo, False, AllUseFCOPY);
                case W of
                  cmOK:
                  begin
                    if ExtFileSize(TempFile) = (CopyDiskSize * 3) + CBMHeaderIDLen then
                    begin
                      FCOPYBlock := True;
                      ExtBlockRead(TempFile, CopyHeaderID, CBMHeaderIDLen);
                      W := 0;
                      B := True;
                      while B and (W < CopyDiskSize) do
                      begin
                        ExtBlockRead(TempFile, X, 3);
                        case Chr(X[0]) of
                          '-', '+', '?': C := dsOK;
                          'H': C := ds27READ;
                          'S': C := ds21READ;
                          'I': C := ds22READ;
                          'C': C := ds23READ;
                          'L': C := ds24READ;
                          'F': C := ds25WRITE;
                          'P': C := ds26PROTECT;
                        else
                          B := False;
                        end;
                        if (X[1] <> SectorSign) and not (C in [ds20READ, ds21READ, ds27READ]) then
                          C := ds22READ;
                        InfoBuffer2^[W] := C;
                        InfoBuffer2^[W + CopyDiskSize] := X[1];
                        InfoBuffer2^[W + (CopyDiskSize shl 1)] := X[2];
                        Inc(W);
                      end;
                    end;
                    B := B and (IOResult = 0);
                    if not B then
                    begin
                      FCOPYBlock := False;
                      ErrorWin(stError, 'The FCOPY-PC info file is invalid.', TempFile.LongName, CurHelpCtx, sbNone);
                    end;
                  end;
                  cmCancel: O := False;
                end;
                ExtClose(TempFile);
                ReadTime := L;
              end;
            end
            else
            begin
              ExtSeek(ReadFile, CopyDiskSize shl 8);
              ExtBlockRead(ReadFile, InfoBuffer2^, CopyDiskSize);
              ExtSeek(ReadFile, 0);
              B := (IOResult = 0);
            end;
            if not B then FillChar(InfoBuffer2^, CopyDiskSize, dsEmpty);
          end;
        end;
        FileTime := ReadTime;
      end;
      pmGCRDisk:
      begin
        ErrorWin(stError, 'This disk type is currently not supported.', CopyFullName, CurHelpCtx, sbNone);
        O := False;
      end;
      pmDiskZip, pmSixZip: O := True;
    end;
  end;
  OpenDiskRead := O;
end;

{Close the source disk or disk image}
procedure CloseDiskRead;
begin
  with Act^ do
  begin
    case CopyMode of
      pmExt: if CopyTransferMode = tmNormal then CloseCBMChannel(saData) else TurboOff;
      pmDisk, pmGCRDisk..pmSixZip: ExtClose(ReadFile);
    end;
  end;
end;

{Open destination disk or disk image
  Input : FirstOpen: when True, the destination is opened for the first time
  Output: when False, an error occured}
function OpenDiskWrite(FirstOpen: Boolean): Boolean;
var
  B,
  O             : Boolean;
  M             : Byte;
  L             : Longint;
  N             : string;
begin
  with Inact^ do
  begin
    O := True;
    case CopyMode of
      pmExt:
      begin
        if CheckDevice then
        begin
          SendConfigData;
          case CopyTransferMode of
            tmNormal:
            begin
              if not FirstOpen then CloseCBMChannel(saData);
              OpenCBMChannel(saData, '#', True);
            end;
            tmTurbo:
            begin
              if FirstOpen then OpenCBMChannel(saCommand, 'I0', True);
              TurboOn := (SendDriveProg(deTurboDiskSave, True) and ExecDriveProg(deTurboDiskSave, stEmpty));
            end;
            tmWarp:
            begin
              if FirstOpen then OpenCBMChannel(saCommand, 'I0', True);
              TurboOn := (SendDriveProg(deWarpDiskSave, True) and (not VerifyWrite or
                SendDriveProg(deWarpDiskVerify, False)) and ExecDriveProg(deWarpDiskSave, stEmpty));
            end;
          end;
        end;
        if not TurboOn then
        begin
          O := ReadCBMError(N, True, True, True);
          if not O then ErrorWin(stError, N, stEmpty, CurHelpCtx, sbNone);
        end;
      end;
      pmDisk:
      begin
        if FirstOpen then
        begin
          M := fmWriteOnly;
          if AppendFile then M := fmReadWrite;
          O := (LongOpenFile(AddToPath(CopyPath, CopyImageName, chDirSep), WriteFile, M) = 0);
          if O then
          begin
            B := True;
            if AppendFile then
            begin
              CopyDiskType := GetDiskType(ExtFileSize(WriteFile));
              CheckDiskType;
              ExtGetFTime(WriteFile, ReadTime);
              if CopyDiskType and dtErrorInfo > 0 then
              begin
                ExtSeek(WriteFile, CopyDiskSize shl 8);
                ExtBlockRead(WriteFile, InfoBuffer^, CopyDiskSize);
                ExtSeek(WriteFile, 0);
                B := (IOResult <> 0);
              end;
            end;
            if B then FillChar(InfoBuffer^, MaxLocations, dsEmpty);
          end
          else
          begin
            ErrorWin(stError, 'Can''t open the file', CopyFullName, CurHelpCtx, sbNone);
          end;
        end;
      end;
      pmGCRDisk:
      begin
        ErrorWin(stError, 'This disk type is currently not supported.', CopyFullName, CurHelpCtx, sbNone);
        O := False;
      end;
      pmDiskZip, pmSixZip:
      begin
        FileTime := 0;
        O := True;
      end;
    end;
  end;
  OpenDiskWrite := O;
end;

{Close the destination disk or disk image
  Input : InitDisk: when True, the Commodore drive is initialized}
procedure CloseDiskWrite(InitDisk: Boolean);
var
  L             : Longint;
begin
  with Inact^ do
  begin
    InOutRes := 0;
    case CopyMode of
      pmExt:
      begin
        if CopyTransferMode = tmNormal then CloseCBMChannel(saData) else TurboOff;
        if InitDisk then OpenCBMChannel(saCommand, 'I0', True);
      end;
      pmDisk, pmGCRDisk..pmSixZip:
      begin
        if KeepTime then
        begin
          L := 0;
          if Other^.CopyMode <> pmExt then L := Other^.FileTime;
          if AppendFile and (FileTime < L) then FileTime := L;
          if L > 0 then ExtSetFTime(WriteFile, L);
        end;
        ExtClose(WriteFile);
      end;
    end;
  end;
end;

{Read a track of data from source disk or disk image
  Input : SelBlocks: when True, sectors to copy are selected from the BAM;
          otherwise the track map has to be filled prior to call
  Output: when False, an error occured}
function ReadTrack(SelBlocks: Boolean): Boolean;
var
  B,
  F,
  O,
  P,
  W,
  X             : Boolean;
  Q,
  R,
  S,
  T,
  U             : Byte;
  I,
  J,
  K,
  M             : Word;
  L             : Longint;
  Z             : PBuffer;
  N             : string;
  A             : array [0..TrackMapSize - 1] of Byte;

procedure MakeError(Error: Byte);
var
  S             : Byte;
begin
  for S := 0 to M - 1 do if TrackMap[S] = 0 then InfoBuffer^[J + S] := Error;
end;

begin
  with Act^ do
  begin
    Sector := 0;
    T := Track;
    O := True;
    M := SectorNum(T);
    J := DiskPos(T, 0);
    if AppendFile and (T < Other^.CopyMaxTrack) then
    begin
      L := ExtFilePos(WriteFile);
      ExtBlockRead(WriteFile, TempBuffer, M shl 8);
      ExtSeek(WriteFile, L);
    end
    else
    begin
      ClearTrackData(GetTrackType(Other));
    end;
    case CopyMode of
      pmExt:
      begin
        FillChar(GCRMap, TrackMapSize, gmEmpty);
        Sector := 0;
        W := (CopyTransferMode = tmWarp);
        RetryCount := 0;
        if W then RetryCount := RetryNum + 1;
        if not SelBlocks or ClearTrackMap(not W) then
        begin
          B := True;
          F := True;
          P := True;
          X := False;
          while F and O do
          begin
            S := Sector;
            case CopyTransferMode of
              tmNormal:
              begin
                DisplayTS(Act);
                OpenCBMChannel(saCommand, 'U1: ' + LeadingSpace(saData, 0) + ' 0 ' +
                  LeadingSpace(T, 0) + stSpace + LeadingSpace(S, 0), False);
                asm
                  mov al, saData;
                  call Talk;
                  mov ah, S;
                  xor al, al;
                  mov di, Offset(TempBuffer);
                  add di, ax;
                  mov si, 256;
              @1: call Receive;
                  mov [di], al;
                  cmp Status, 0;
                  jne @2;
                  inc di;
                  dec si;
                  jne @1;
              @2: call Untalk;
                end;
              end;
              tmTurbo:
              begin
                DisplayTS(Act);
                asm
                  push word ptr CopyPriorityMode;
                  call InterruptOff;
                  call ParallelOutput;
                  mov al, S;
                  call TSend;
                  mov al, T;
                  call TSend;
                  mov ax, 50;
                  call Delay;
                  mov ah, S;
                  xor al, al;
                  mov di, Offset(TempBuffer);
                  add di, ax;
                  mov si, 256;
                  call ParallelInput;
              @1: call TReceive;
                  mov [di], al;
                  cmp Status, 0;
                  jne @2;
                  inc di;
                  dec si;
                  jne @1;
              @2: call InterruptOn;
                end;
              end;
              tmWarp:
              begin
                if B then
                begin
                  if P then
                  begin
                    Sector := 0;
                    DisplayTS(Act);
                  end;
                  asm
                    push word ptr CopyPriorityMode;
                    call InterruptOff;
                    cmp P, False;
                    je @1;
                    call ParallelOutput;
                    mov Status, 0;
                    mov P, False;
                    mov al, T;
                    call TSend;
                    mov si, Offset(TrackMap);
                    mov di, M;
                @2: mov al, [si];
                    call TSend;
                    inc si;
                    dec di;
                    jne @2;
                    mov si, Offset(TrackMap);
                    add si, TrackMapSize;
                    mov al, [si];
                    call TSend;
                    mov ax, 50;
                    call Delay;
                    call ParallelInput;
                @1: push ds;
                    pop es;
                    call TReceive;
                    mov S, al;
                    cmp Status, 0;
                    jne @3;
                    mov al, S;
                    xor ah, ah;
                    mov di, Offset(GCRMap);
                    add di, ax;
                    inc byte ptr [di];
                    push ax;
                    mov si, 10;
                    mul si;
                    mov di, ax;
                    add di, Offset(HeaderBuffer);
                @5: call TReceive;
                    mov [di], al;
                    cmp Status, 0;
                    jne @3;
                    inc di;
                    dec si;
                    jne @5;
                    pop ax;
                    mov si, 326;
                    mul si;
                    mov di, ax;
                    add di, Offset(GCRBuffer);
                @4: call TReceive;
                    mov [di], al;
                    cmp Status, 0;
                    jne @3;
                    inc di;
                    dec si;
                    jne @4;
                @3: call InterruptOn;
                  end;
                  if Status = 0 then
                  begin
                    Sector := S;
                    DisplayTS(Act);
                  end
                  else
                  begin
                    TurboOn := False;
                  end;
                end
                else
                begin
                  GCRError := 0;
(* ?ASM? *)
                  asm
                    xor bx, bx;
                    mov si, Offset(GCRBuffer);
                    mov dx, Offset(HeaderBuffer);
                @1: mov S, bl;
                    push bx;
                    test byte ptr GCRMap[bx], gmCountMask;
                    jne @2;
                    add si, 326;
                    add dx, 10;
                    jmp @3;
                @2: push si;
                    mov si, dx;
                    mov di, Offset(TempBuffer[TempBufferSize - 256]);
                    call GCRDecodeHeader;
                    mov dx, si;
                    pop si;
                    push dx;
                    mov di, Offset(TempBuffer[TempBufferSize - 256]);
                    call GCRDecodeSector;
                    pop dx;
                @3: pop bx;
                    cmp GCRError, 0;
                    jne @4;
                    inc bx;
                    cmp bx, M;
                    jb @1;
                @4:
                  end;
                  Sector := S;
                  Status := GCRError;
                end;
              end;
            end;
            O := EndlessRetry or not CancelTransfer(True);
            if O then
            begin
              if CopyTransferMode = tmNormal then O := ReadCBMError(N, False, False, False) else
                O := (Status and not ssEOF = 0);
              if O then
              begin
                if B then
                begin
                  InfoBuffer^[J + S] := dsOK;
                  F := NextSector(not W);
                  if W and not F then
                  begin
                    F := True;
                    B := False;
                  end;
                end
                else
                begin
                  F := False;
                end;
              end
              else
              begin
                I := cmCancel;
                if (CopyTransferMode <> tmNormal) then
                begin
                  if not B then
                  begin
                    if (SmartRetryNum > 0) and (GCRMap[S] and gmCountMask > 0) then
                    begin
                      if (GCRMap[S] > 1) and not CompMem(GCRBuffer[S * 326], TempBuffer[S * 326], 326) then
                        GCRMap[S] := 1;
                      if GCRMap[S] >= SmartRetryNum then
                      begin
                        GCRMap[S] := gmSkipped;
                        I := cmNo;
                      end
                      else
                      begin
                        if GCRMap[S] = 1 then Move(GCRBuffer[S * 326], TempBuffer[S * 326], 326);
                      end;
                    end;
                    if not EndlessRetry then Dec(RetryCount);
                    N := ReadErrorStr(GCRError, T, S);
                  end
                  else
                  begin
                    O := ReadCBMError(N, False, True, True);
                  end;
                end;
                MakeDriveStatus(N);
                S := Sector;
                DisplayTS(Act);
                if I = cmCancel then
                begin
                  if ((EndlessRetry and not Escape) or (not EndlessRetry and not B)) and (RetryCount > 0) then
                  begin
                    I := cmOK;
                  end
                  else
                  begin
                    if SkipAuto then I := cmNo else I := DiskErrorWin(stError, N, stEmpty, CurHelpCtx, True, True, True);
                  end;
                end;
                if I = cmExtra then
                begin
                  SkipAuto := True;
                  I := cmNo;
                end;
                if (DriveStatus = ds21READ) and (I = cmNo) then I := cmYes;
                O := True;
                if W and (RetryCount = 0) then RetryCount := RetryNum + 1;
                case I of
                  cmOK:
                  begin
                    if not B then
                    begin
                      TrackMap[S] := 0;
                      Inc(TrackMap[TrackMapSize]);
                      B := True;
                    end;
                  end;
                  cmNo:
                  begin
                    NoteError(S, False);
                    if B then
                    begin
                      if W then
                      begin
                        TrackMap[S] := 1;
                        Dec(TrackMap[TrackMapSize]);
                        F := (TrackMap[TrackMapSize] > 0);
                        if not F then
                        begin
                          F := True;
                          B := False;
                        end;
                      end
                      else
                      begin
                        F := NextSector(True);
                      end;
                    end
                    else
                    begin
                      if GCRMap[S] and gmCountMask > 0 then GCRMap[S] := gmSkipped;
                    end;
                  end;
                  cmYes:
                  begin
                    NoteError(0, True);
                    for I := 0 to M - 1 do if GCRMap[I] and gmCountMask > 0 then GCRMap[I] := gmSkipped;
                    F := False;
                  end;
                  cmCancel: O := False;
                end;
                if O then
                begin
                  if not W or not TurboOn then O := OpenDiskRead(False);
                  P := True;
                end;
              end;
            end;
          end;
        end
        else
        begin
          if not (AppendFile and (T < Other^.CopyMaxTrack)) then
          begin
            ClearTrackData(GetTrackType(Other));
            for S := 0 to M - 1 do
            begin
              GCRMap[S] := 1;
              InfoBuffer^[J + S] := dsOK;
            end;
          end;
        end;
      end;
      pmDisk:
      begin
        I := J;
        DisplayTS(Act);
        X := ((Other^.CopyMode <> pmExt) and ((CopyDiskCopyMode <> dcFull) or AppendFile));
        Z := @TempBuffer;
        if X then Z := @GCRBuffer;
        ExtSeek(ReadFile, Longint(J) shl 8);
        ExtBlockRead(ReadFile, Z^, M shl 8);
        O := (IOResult = 0);
        J := I;
        K := CopyDiskSize;
        F := FCOPYBlock;
        if O then
        begin
          if X then
          begin
            if ClearTrackMap(False) then
            begin
              for S := 0 to M - 1 do
              begin
                if TrackMap[S] = 0 then
                begin
                  Move(GCRBuffer[S shl 8], TempBuffer[S shl 8], 256);
                  InfoBuffer^[J] := InfoBuffer2^[J];
                  if F then
                  begin
                    InfoBuffer^[J + K] := InfoBuffer2^[J + K];
                    InfoBuffer^[J + (K shl 1)] := InfoBuffer2^[J + (K shl 1)];
                  end;
                end;
                Inc(J);
              end;
            end;
          end
          else
          begin
            Move(InfoBuffer2^[J], InfoBuffer^[J], M);
            if F then
            begin
              Move(InfoBuffer2^[J + K], InfoBuffer^[J + K], M);
              Move(InfoBuffer2^[J + (K shl 1)], InfoBuffer^[J + (K shl 1)], M);
            end;
          end;
        end;
      end;
      pmDiskZip:
      begin
        if ClearTrackMap(False) then
        begin
          MakeError(ds20READ);
          if OpenZipFile(ReadFile, J, 0, fmReadOnly) = 0 then
          begin
            repeat
              B := ReadZipCodeBlock(ReadFile, @DataBuffer, False, zrUncompress, Q, S, I);
              if B and (Q = T) and (S < M) then
              begin
                if TrackMap[S] = 0 then
                begin
                  InfoBuffer^[J + S] := dsOK;
                  Move(DataBuffer, TempBuffer[S shl 8], 256);
                end;
              end;
            until not B or (Q > T);
            O := (IOResult = 0);
          end;
        end;
      end;
      pmSixZip:
      begin
        if ClearTrackMap(False) then
        begin
          FillChar(GCRMap, TrackMapSize, gmEmpty);
          FillChar(A, TrackMapSize, 0);
          Q := SixZipFileNum - 1;
          while T < SixZipTracks[Q] do Dec(Q);
          Q := SixZipTracks[Q];
          MakeError(ds20READ);
          O := (OpenZipFile(ReadFile, J, 0, fmReadOnly) = 0);
          if O then
          begin
            while Q < T do
            begin
              ExtBlockRead(ReadFile, DataBuffer, 256);
              if DataBuffer[$FF] <> 0 then
                ExtSeek(ReadFile, ExtFilePos(ReadFile) + 326 * DataBuffer[$FF]);
              Inc(Q);
            end;
            ExtBlockRead(ReadFile, DataBuffer, 256);
            K := DataBuffer[$FF];
            if K = 0 then
            begin
              MakeError(ds21READ);
              for S := 0 to M - 1 do GCRMap[S] := gmEmpty;
            end
            else
            begin
              if K > M then K := M;
              FillChar(HeaderBuffer, TrackMapSize * HeaderGCRSize, 0);
              Move(DataBuffer, HeaderBuffer, K * HeaderGCRSize);
              R := 0;
              for Q := 0 to K - 1 do
              begin
                ExtBlockRead(ReadFile, GCRBuffer[R * 326 + 256], 70);
                ExtBlockRead(ReadFile, GCRBuffer[R * 326], 256);
                GCRError := 0;
(* ?ASM? *)
                asm
                  mov al, R;
                  mov ah, HeaderGCRSize;
                  mul ah;
                  mov si, Offset(HeaderBuffer);
                  add si, ax;
                  mov di, Offset(UndoBuffer);
                  push di;
                  call GCRDecodeHeader;
                  pop di;
                  mov al, [di].TSecDecHeader.TrackNum;
                  mov U, al;
                  mov al, [di].TSecDecHeader.SectorNum;
                  mov S, al;
                  mov al, R;
                  xor ah, ah;
                  mov si, 326;
                  mul si;
                  mov si, Offset(GCRBuffer);
                  add si, ax;
                  mov di, Offset(UndoBuffer);
                  call GCRDecodeSector;
                end;
                if (U = T) and (S < M) then
                begin
                  GCRMap[S] := 1;
                  InfoBuffer^[J + S] := ErrorCodeToStatus(GCRError);
                end;
                A[R] := 1;
                Inc(R, SixZipInterleave);
                while (R >= K) do Dec(R, K);
                S := R;
                F := (A[R] <> 0);
                while F do
                begin
                  Inc(R);
                  if (R >= K) then Dec(R, K);
                  F := (A[R] <> 0) and (R <> S);
                end;
              end;
            end;
            O := (IOResult = 0);
          end;
        end;
      end;
    end;
  end;
  ReadTrack := O;
end;

{Convert track data from the source format to the destination format
  Input: Track: track number
         Src: source track type
         Dest: destination track type
         SecNum: number of sectors on track
         SelBlocks: when True, sectors to copy are selected from the BAM;
                    otherwise the track map has to be filled prior to call}
procedure ConvertTrack(Track, Src, Dest, SecNum: Byte; SelBlocks: Boolean);
var
  F             : Boolean;
  R,
  S,
  U             : Byte;
  J,
  K             : Word;
begin
  if Src <> Dest then
  begin
    J := Act^.DiskPos(Track, 0);
    K := Act^.CopyDiskSize;
    case Dest of
      ttNormal:
      begin
        case Src of
          ttNormalF64:
          begin
            if not SelBlocks or Act^.ClearTrackMap(False) then
            begin
              for S := 0 to SecNum - 1 do
              begin
                if TrackMap[S] = 0 then
                begin
                  if (GetSectorChecksum(S) <> InfoBuffer2^[J + (Act^.CopyDiskSize shl 1)]) and
                    not (InfoBuffer^[J] in [ds20READ, ds21READ, ds22READ, ds27READ]) then
                    InfoBuffer^[J] := ds23READ;
                end;
                Inc(J);
              end;
            end;
          end;
          ttGCR:
          begin
            ClearTrackData(ttNormal);
            for R := 0 to SecNum - 1 do
            begin
              if HeaderBuffer[R][0] <> 0 then
              begin
(* ?ASM? *)
                asm
                  mov al, R;
                  mov ah, HeaderGCRSize;
                  mul ah;
                  mov si, Offset(HeaderBuffer);
                  add si, ax;
                  mov di, Offset(TempBuffer[TempBufferSize - 256]);
                  push di;
                  call GCRDecodeHeader;
                  pop di;
                  mov al, [di].TSecDecHeader.TrackNum;
                  mov U, al;
                  mov al, [di].TSecDecHeader.SectorNum;
                  mov S, al;
                end;
                if (U = Track) and (S < SecNum) and (GCRMap[S] > gmEmpty) then
                begin
(* ?ASM? *)
                  asm
                    mov al, R;
                    xor ah, ah;
                    mov si, 326;
                    mul si;
                    mov si, Offset(GCRBuffer);
                    add si, ax;
                    mov al, S;
                    xor ah, ah;
                    shl ax, 8;
                    mov di, Offset(TempBuffer);
                    add di, ax;
                    call GCRDecodeSector;
                  end;
                end;
              end;
            end;
          end;
        end;
      end;
      ttGCR:
      begin
        ClearTrackData(ttGCR);
        case Src of
          ttNormal, ttNormalF64:
          begin
            F := (Src <> ttNormalF64);
            FillChar(DataBuffer, $60, 0);
            if not SelBlocks or Act^.ClearTrackMap(False) then
            begin
              R := 0;
              for S := 0 to SecNum - 1 do
              begin
                if TrackMap[S] = 0 then
                begin
                  GCRError := InfoBuffer^[J];
                  if GCRError = dsEmpty then GCRError := dsOK;
                  if not ((GCRError = ds20READ) and (Inact^.CopyMode = pmSixZip)) then
                  begin
                    CreateSectorHeader(Track, S, FirstHeaderPadding(Act^.CopyDiskType, Act^.ExtBAMMode));
                    if not F then
                    begin
                      GCRSign := InfoBuffer^[J + K];
                      GCRChecksum := InfoBuffer^[J + (K shl 1)];
                    end;
(* ?ASM? *)
                    asm
                      mov al, R;
                      mov ah, HeaderGCRSize;
                      mul ah;
                      mov si, Offset(UndoBuffer);
                      mov di, Offset(HeaderBuffer);
                      add di, ax;
                      call GCREncodeHeader;
                      mov al, R;
                      xor ah, ah;
                      push ax;
                      push ax;
                      shl ax, 8;
                      mov si, Offset(TempBuffer);
                      add si, ax;
                      pop ax;
                      mov di, 326;
                      mul di;
                      mov di, Offset(GCRBuffer);
                      add di, ax;
                      mov al, F;
                      call GCREncodeSector;
                      pop bx;
                      mov byte ptr DataBuffer[bx], al;
                      mov byte ptr DataBuffer[bx][$20], ah;
                      mov byte ptr DataBuffer[bx][$40], 1;
                    end;
                  end;
                end;
                Inc(R);
                Inc(J);
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;

{Write a track of data to destination disk or disk image
  Input : SelBlocks: when True, sectors to copy are selected from the BAM;
          otherwise the track map has to be filled prior to call
  Output: when False, an error occured}
function WriteTrack(SelBlocks: Boolean): Boolean;
var
  B,
  F,
  O,
  P,
  W             : Boolean;
  Q,
  R,
  S,
  T,
  X             : Byte;
  I,
  K,
  M             : Word;
  N             : string;
  A             : array [0..TrackMapSize - 1] of Byte;
begin
  with Inact^ do
  begin
    Track := Other^.Track;
    Sector := 0;
    T := Track;
    O := True;
    M := SectorNum(T);
    I := DiskPos(T, 0);
    FCOPYBlock := False;
    case CopyMode of
      pmExt:
      begin
        W := (CopyTransferMode = tmWarp);
        RetryCount := 0;
        if W then RetryCount := RetryNum + 1;
        B := True;
        if not SelBlocks or Other^.ClearTrackMap(not W) then
        begin
          if W then
          begin
            S := 0;
            while (S < M) and (TrackMap[S] > 0) do Inc(S);
            Sector := S;
            Q := TrackMap[TrackMapSize];
            TrackMap[S] := 1;
            Dec(TrackMap[TrackMapSize]);
            B := True;
          end
          else
          begin
            Sector := Other^.Sector;
          end;
          F := True;
          while F and O do
          begin
            if SelBlocks then DisplayTS(Inact);
            S := Sector;
            case CopyTransferMode of
              tmNormal:
              begin
                OpenCBMChannel(saCommand, 'B-P: ' + LeadingSpace(saData, 0) + ' 0', False);
                asm
                  mov al, saData;
                  call Listen;
                  mov ah, S;
                  xor al, al;
                  mov si, Offset(TempBuffer);
                  add si, ax;
                  mov di, 256;
              @1: mov al, [si];
                  call Send;
                  inc si;
                  dec di;
                  jne @1;
                  call Unlisten;
                end;
                OpenCBMChannel(saCommand, 'U2: ' + LeadingSpace(saData, 0) + ' 0 ' +
                  LeadingSpace(T, 0) + stSpace + LeadingSpace(S, 0), False);
              end;
              tmTurbo:
              begin
                asm
                  push word ptr CopyPriorityMode;
                  call InterruptOff;
                  call ParallelOutput;
                  mov al, S;
                  call TSend;
                  mov al, T;
                  call TSend;
                  mov ax, 50;
                  call Delay;
                  mov ah, S;
                  xor al, al;
                  mov si, Offset(TempBuffer);
                  add si, ax;
                  mov di, 256;
              @1: mov al, [si];
                  call TSend;
                  inc si;
                  dec di;
                  jne @1;
                  mov ax, 50;
                  call Delay;
                  call ParallelInput;
                  call InterruptOn;
                end;
              end;
              tmWarp:
              begin
                if B then
                begin
                  asm
                    push word ptr CopyPriorityMode;
                    call InterruptOff;
                    call ParallelOutput;
                    mov al, S;
                    call TSend;
                    mov al, T;
                    call TSend;
                    mov ax, 50;
                    call Delay;
                    mov di, 326;
                    mov al, S;
                    xor ah, ah;
                    mul di;
                    dec di;
                    mov si, ax;
                    add si, Offset(GCRBuffer);
                @1: mov al, [si];
                    call TSend;
                    inc si;
                    dec di;
                    jne @1;
                    mov ax, 50;
                    call Delay;
                    call ParallelInput;
                    call InterruptOn;
                  end;
                end
                else
                begin
                  asm
                    push word ptr CopyPriorityMode;
                    call InterruptOff;
                    call ParallelOutput;
                    mov al, MaxByte;
                    call TSend;
                    mov al, T;
                    call TSend;
                    mov di, M;
                    xor si, si;
                @1: mov al, byte ptr DataBuffer[si];
                    call TSend;
                    mov al, byte ptr DataBuffer[si][$20];
                    call TSend;
                    mov al, byte ptr DataBuffer[si][$40];
                    call TSend;
                    inc si;
                    dec di;
                    jne @1;
                    mov al, Q;
                    call TSend;
                    mov ax, 50;
                    call Delay;
                    call ParallelInput;
                    call InterruptOn;
                  end;
                end;
              end;
            end;
            O := not CancelTransfer(True);
            if O then
            begin
              if CopyTransferMode = tmNormal then
              begin
                O := ReadCBMError(N, False, False, False);
              end
              else
              begin
                TWait;
                O := (Status = 0);
              end;
              if O then
              begin
                if B then
                begin
                  F := NextSector(True);
                  if W and not F and VerifyWrite then
                  begin
                    F := True;
                    B := False;
                    Sector := 0;
                    if SelBlocks then DisplayTS(Inact);
                  end;
                end
                else
                begin
                  F := False;
                end;
              end
              else
              begin
                if CopyTransferMode <> tmNormal then O := ReadCBMError(N, False, True, True);
                if O then
                begin
                  I := cmOK;
                end
                else
                begin
                  if W and not B then Dec(RetryCount);
                  if W and not B and (RetryCount > 0) then I := cmOK else
                    I := DiskErrorWin(stError, N, stEmpty, CurHelpCtx, True, True, False);
                  if W and (RetryCount = 0) then RetryCount := RetryNum + 1;
                  P := False;
                  if W and (Length(N) > 9) then
                  begin
                    if Copy(N, 1, 2) = '25' then
                    begin
                      Val(Copy(N, Length(N) - 4, 2), S, NumOK);
                      if S = T then
                      begin
                        Val(Copy(N, Length(N) - 1, 2), S, NumOK);
                        P := ((NumOK = 0) and (S <= M));
                      end;
                    end;
                  end;
                  O := True;
                  case I of
                    cmOK:
                    begin
                      if W and not B and P then
                      begin
                        Sector := S;
                        TrackMap[S] := 0;
                        Inc(TrackMap[TrackMapSize]);
                        B := True;
                      end;
                    end;
                    cmNo:
                    begin
                      if W and not B and P then
                      begin
                        PWord(@DataBuffer[S shl 1])^ := 0;
                        Dec(Q);
                        F := (Q > 0);
                      end
                      else
                      begin
                        F := NextSector(True);
                      end;
                    end;
                    cmYes: F := False;
                    cmCancel:
                    begin
                      O := False;
                      EscPressed := True;
                    end;
                  end;
                  if O then O := OpenDiskWrite(False);
                end;
              end;
            end;
          end;
        end
        else
        begin
          O := not CancelTransfer(True);
        end;
      end;
      pmDisk:
      begin
        if SelBlocks then DisplayTS(Inact);
        if AppendFile and BothBAMsOK then
          for Q := 0 to M - 1 do if TrackMap2[Q] = 0 then AllocBlock(T, Q, Other^.IsBlockUsed(T, Q));
        ExtBlockWrite(WriteFile, TempBuffer, M shl 8);
        O := (IOResult = 0);
      end;
      pmDiskZip:
      begin
        for X := 0 to DiskZipFileNum - 1 do if ImagePos = DiskZipBlocks[X] then O := False;
        if not O then
        begin
          if ImagePos <> 0 then
          begin
            ExtTruncate(WriteFile);
            CloseDiskWrite(False);
          end;
          O := (OpenZipFile(WriteFile, I, 0, fmWriteOnly) = 0);
        end;
        if O then
        begin
          case T of
            1..17: X := 11;
            18..24: X := 10;
            25..40: X := 9;
          end;
          S := 0;
          Q := 0;
          while Q < M do
          begin
            WriteZipCodeBlock(WriteFile, @TempBuffer[S shl 8], False, T, S);
            if Q and 1 = 0 then Inc(S, X) else Dec(S, X - 1);
            Inc(Q);
            Inc(ImagePos);
          end;
        end;
      end;
      pmSixZip:
      begin
        FillChar(A, TrackMapSize, 0);
        for X := 0 to SixZipFileNum - 1 do if ImagePos = SixZipBlocks[X] then O := False;
        if not O then
        begin
          if ImagePos <> 0 then
          begin
            ExtTruncate(WriteFile);
            CloseDiskWrite(False);
          end;
          O := (OpenZipFile(WriteFile, I, 0, fmWriteOnly) = 0);
        end;
        if O then
        begin
          if T = Max1541Tracks then CopiedSize := ExtFilePos(WriteFile);
          S := 0;
          F := False;
          while not F and (S < M) do
          begin
            F := (InfoBuffer^[I + S] <> ds20READ);
            Inc(S);
          end;
          S := 0;
          while F and (S < M) do
          begin
            F := (InfoBuffer^[I + S] <> ds21READ);
            Inc(S);
          end;
          FillChar(DataBuffer, 256, 0);
          K := 0;
          if F then
          begin
            for S := 0 to M - 1 do
            begin
              if InfoBuffer^[I + S] <> ds20READ then
              begin
                Move(HeaderBuffer[K], DataBuffer[K * HeaderGCRSize], HeaderGCRSize);
                A[K] := 1;
                Inc(K);
              end;
            end;
          end;
          DataBuffer[$FF] := K;
          ExtBlockWrite(WriteFile, DataBuffer, 256);
          if F then
          begin
            R := 0;
            for Q := 0 to K - 1 do
            begin
              ExtBlockWrite(WriteFile, GCRBuffer[R * 326 + 256], 70);
              ExtBlockWrite(WriteFile, GCRBuffer[R * 326], 256);
              A[R] := 0;
              Inc(R, SixZipInterleave);
              while (R >= K) do Dec(R, K);
              S := R;
              F := (A[R] = 0);
              while F do
              begin
                Inc(R);
                if (R >= K) then Dec(R, K);
                F := (A[R] = 0) and (R <> S);
              end;
            end;
          end;
        end;
        Inc(ImagePos, M);
      end;
    end;
  end;
  WriteTrack := O;
end;

{Delete the destination disk if the disk copy was cancelled}
procedure DeleteDiskWrite;
var
  B             : Byte;
  S,
  T             : string;
begin
  case Inact^.CopyMode of
    pmDisk: LongErase(WriteFile.LongName);
    pmDiskZip, pmSixZip:
    begin
      case Inact^.CopyMode of
        pmDiskZip:
        begin
          B := DiskZipFileNum;
          if Inact^.CopyDiskType <> dt1541Ext then Dec(B);
        end;
        pmSixZip: B := SixZipFileNum;
      end;
      S := GetPath(WriteFile.LongName, chDirSep);
      T := CutPath(WriteFile.LongName, chDirSep);
      for T[1] := '1' to Chr(Ord('0') + B) do LongErase(AddToPath(S, T, chDirSep));
    end;
  end;
end;

{Create the full name for the destination disk or disk image
  Input : DiskType: type of the destination disk or disk image}
procedure MakeFullName(DiskType: Byte);
begin
  Inact^.CopyName := '';
  if Inact^.CopyMode = pmExt then
  begin
    Inact^.CopyImageName := '';
    Inact^.CopyFullName := DriveNumber(Inact, True);
  end
  else
  begin
    Inact^.CopyImageProtoName := Inact^.CopyImageName;
    Inact^.CorrectImageName(False);
    Inact^.CopyFullName := MakeTypeStr(Inact^.CopyMode) + ':' + AddToPath(Inact^.CopyPath, Inact^.CopyImageName, chDirSep);
  end;
end;

{Check the disk type of the source and the destination}
procedure CheckDiskType;
begin
  Act^.CheckDiskType;
  Inact^.CheckDiskType;
  MakeFullName(Inact^.CopyDiskType);
end;

{Clear the extra BAM, when merging an extended 1541 disk image into a normal
  one}
procedure ClearExtraBAM;
var
  T,
  S             : Byte;
begin
  for T := Max1541Tracks to Inact^.CopyMaxTrack do for S := 0 to Inact^.SectorNum(T) - 1 do Inact^.AllocBlock(T, S, False);
end;

{Duplicate the BAM of the source disk image into the sector selection area}
procedure DuplicateBAM;
var
  O             : Boolean;
  S,
  T             : Byte;
begin
  O := Act^.IsBAMValid;
  ManualSelect := New(PLocationSet);
  ManualSelect^.Init(not O);
  if O then
  begin
    for T := 1 to Act^.CopyMaxTrack - 1 do for S := 0 to Act^.SectorNum(T) - 1 do
      if Act^.IsBlockUsed(T, S) then ManualSelect^.Include(Act^.DiskPos(T, S));
    if CopyDiskCopyMode = dcSafeBAM then
    begin
      T := Act^.DirTrack;
      for S := 0 to Act^.SectorNum(T) - 1 do ManualSelect^.Include(Act^.DiskPos(T, S));
    end;
  end;
end;

{Copy a chunk of the BAM area from the source disk or disk image into the
  destination disk image
  Input : Start: the starting offset of the chunk
          Len: the length of the chunk}
procedure CopyBAMChunk(Start, Len: Word);
begin
  Move(Act^.BAM[Start], Inact^.BAM[Start], Len);
end;

{Determine if both the source and destination are 1541/1571 disks or disk
  images, except for the active being 1571 and the inactive being 1541
  Output: when True, both are of 1541/1571 type}
function Both1541or1571Disks: Boolean;
begin
  Both1541or1571Disks := (Act^.CopyDiskType and dtTypeMask in [dt1541, dt1541Ext, dt1571]) and
    (Inact^.CopyDiskType and dtTypeMask in [dt1541, dt1541Ext, dt1571]) and
    not ((Act^.CopyDiskType and dtTypeMask = dt1571) and
    (Inact^.CopyDiskType and dtTypeMask in [dt1541, dt1541Ext]));
end;

function ConfirmDiskCopy: Boolean;
var
  B,
  C             : Byte;
  I             : Integer;
  F,
  N,
  U,
  Z             : string;
begin
  B := Act^.Mode;
  C := Act^.CopyMode;
  U := Act^.Under;
  N := Act^.ImageName;
  Z := Act^.CopyImageName;
  F := Act^.GetNamePtr(CopyFileNum)^;
  I := Act^.ListNum;
  Act^.NewMode := Act^.CopyMode;
  Act^.ImageName := Z;
  Act^.RealImagePath := '';
  Act^.Rescan;
  Act^.Cur := -1;
  Act^.DrawPanel;
  CheckDiskType;
  ConfirmDiskCopy := (Confirm(stEmpty, 'Do you wish to copy this disk?', stEmpty, ' '+ColorChar+'C'+ColorChar+'opy ',
    stEmpty, stAll, nil, hcOnlyQuit, True, AllDiskCopy) = cmOK);
  Act^.Under := U;
  Act^.ImageName := N;
  Act^.ListNum := I;
  Act^.NewCur := 0;
  if B = Act^.Mode then
  begin
    Act^.ReselectFiles(False);
    Act^.SearchName;
  end
  else
  begin
    Act^.NewMode := B;
    Act^.Scan;
  end;
  Act^.DrawPanel;
  CopyFileNum := 0;
  while (CopyFileNum < Act^.Max) and (F <> Act^.GetNamePtr(CopyFileNum)^) do Inc(CopyFileNum);
  Act^.CopyImageName := Z;
  Act^.CopyMode := C;
end;

procedure UnselectAllVolumes;
var
  B             : Byte;
  C             : Char;
  I             : Word;
  P             : PString;
  S             : string;
begin
  if (Act^.CopyMode in [pmDiskZip, pmSixZip]) and (CopyFileMode = cfSelected) and (Act^.SelNum > 0) then
  begin
    case Act^.CopyMode of
      pmDiskZip: C := Chr(Ord('0') + DiskZipFileNum);
      pmSixZip: C := Chr(Ord('0') + SixZipFileNum);
    end;
    S := Copy(Act^.CopyImageName, 2, MaxStrLen);
    for I := 0 to Act^.Max do
    begin
      B := Act^.Dir[I].Status;
      if B and fsProcessMask = fsSelected then
      begin
        P := Act^.GetNamePtr(I);
        if (Copy(P^, 2, MaxStrLen) = S) and (P^[1] >= '1') and (P^[1] <= C) then
          Act^.Dir[I].Status := (B and fsStatusMask) or fsProcessed;
      end;
    end;
    Act^.DrawView;
  end;
end;

{'Copy disk' item in the 'Commands' menu: copy the contents of a disk into
  a disk image or copy a disk image onto a disk
  Input : Mode: selection mode for source disks}
procedure CopyDisk(Mode: Byte);
var
  F,
  H,
  K,
  O,
  Q,
  U,
  X,
  Y             : Boolean;
  OrigDiskCopyMode,
  AllTruncateExtraTracks,
  AllCreateErrorInfo,
  AllChangeBAMIDToSectorHeaderID,
  AllCopyFullDiskWhenBAMInvalid,
  B,
  C,
  P,
  S             : Byte;
  W             : Word;
  I             : Integer;
  L             : Longint;
  V             : PView;
  D             : PDialog;
  A1            : PSItem;
  A2            : PInputLine;
  R             : TRect;
  M,
  N,
  Z             : string;

{Check the BAM of the source prior to BAM disk copy and ask for confirmation
  about copying the whole disk if the BAM is invalid}
function CheckBAM: Boolean;
var
  S,
  T             : string;
begin
  CheckBAM := True;
  BothBAMsOK := Act^.IsBAMValid;
  if not BothBAMsOK then
  begin
    case CopyDiskCopyMode of
      dcBAM, dcSafeBAM:
      begin
        BothBAMsOK := True;
        CopyDiskCopyMode := dcFull;
        S := 'The following image file has an invalid BAM';
        T := Act^.CopyFullName;
        if Act^.CopyMode = pmExt then
        begin
          S := DriveNumber(Act, True);
          S[1] := UpCase(S[1]);
          T := 'has an invalid BAM.';
        end;
        CheckBAM := (SureConfirm(stEmpty, S, T, 'Do you wish to copy the full disk?', stEmpty,
          stYes, stEmpty, stEmpty, stEmpty, stNo, nil, CurHelpCtx, ayAllYes, True, AllCopyFullDiskWhenBAMInvalid) = cmOK);
      end;
      dcManualSelect: Act^.ClearBAM(True);
    end;
  end;
end;

{Generate the next destination disk image file name: increase the index in the
  file name and append the disk side letter, if specified
  Input : IncName: when True, the next file name is generated, otherwise only
                   the current one is corrected}
procedure NextFileName(IncName: Boolean);
var
  F,
  Q             : Boolean;
  E             : Char;
  D             : string[1];
  B             : string[8];
  A,
  G             : string;

{Swap the side letter at the end of the file name between 'a' and 'b'}
procedure SwapSideLetter;
begin
  D := E;
  if IncName then D[1] := Chr(Ord(D[1]) xor (Ord('B') xor Ord('A')));
  Dec(M[0]);
end;

begin
  Q := (CutPath(Z, chDirSep) <> '');
  if (C and 2 > 0) then
  begin
    Q := True;
    LongFSplit(Inact^.CopyImageName, N, M, G);
    D := '';
    if (C and 4 > 0) and (Length(M) > 0) then
    begin
      F := IncName;
      E := M[Length(M)];
      case UpCase(E) of
        'A':
        begin
          SwapSideLetter;
          F := False;
        end;
        'B': SwapSideLetter;
      else
        D := Chr(Ord('a') + Byte(IncName and Y));
        F := X and not Y;
      end;
      if X then
      begin
        IncName := F;
      end
      else
      begin
        D := '';
      end;
    end;
    S := 0;
    if (Length(M) > 0) and (M[Length(M)] in ['0'..'9']) then
    begin
      P := Length(M);
      while (P > 0) and (M[P] in ['0'..'9']) do Dec(P);
      S := Length(M) - P;
      if S > 8 then S := 8;
      Val(Copy(M, Length(M) - S + 1, MaxStrLen), L, NumOK);
      M[0] := Chr(P);
      if IncName then Inc(L);
      if L >= 100000000 then Dec(L, 100000000);
    end
    else
    begin
      if IncName then
      begin
        S := 1;
        L := 2;
      end;
    end;
    B := '';
    if S > 0 then B := LeadingZero(L, S);
    S := Length(M) + Length(B) + Length(D);
    if not LongFileNames and (S > 8) then
    begin
      while (S > 8) and (Length(B) > 0) and (B[1] = '0') do
      begin
        B := Copy(B, 2, MaxStrLen);
        Dec(S);
      end;
      while (M <> '') and (S > 8) do
      begin
        Dec(M[0]);
        Dec(S);
      end;
      while (S > 8) do
      begin
        Dec(B[0]);
        Dec(S);
      end;
    end;
    Inact^.CopyImageName := N + M + B + D + G;
  end;
  MakeFullName(Act^.CopyDiskType);
  if (K or Q) and (Inact^.CopyMode <> pmExt) then Z := Inact^.CopyFullName;
end;

{Determine if the current disk image resembles a single- or double-sided disk
  and save the previous value of this setting}
procedure CheckSideNumber;
begin
  Y := X;
  X := (Act^.CopyDiskType in [dt1541, dt1541Ext]);
end;

{Go to the next source disk image
  Output : when True, another disk image was found}
function NextDisk: Boolean;
var
  B             : Byte;
  O             : Boolean;
begin
  O := False;
  while not O and Act^.FindNextFile(False, True, False) do
  begin
    if Act^.CopyAttr and Directory = 0 then
    begin
      B := DetermineTypeName(Act^.CopyName);
      case B of
        pmDisk:
        begin
          Act^.CopyDiskType := GetDiskType(CopySize);
          O := (Act^.CopyDiskType <> dtInvalid);
          CheckSideNumber;
        end;
        pmGCRDisk, pmDiskZip, pmSixZip:
        begin
          Act^.CopyDiskType := dt1541;
          O := True;
          CheckSideNumber;
        end;
      end;
    end;
    if not O then Inc(CopyFileNum);
  end;
  if O then
  begin
    SourceName := Act^.CopyName;
    Act^.CopyMode := B;
  end;
  NextDisk := O;
end;

{Ask for confirmation about overwriting the destination file}
procedure ConfirmOverwrite;
var
  S,
  T             : string[10];
begin
  if (C and 2 > 0) and (C and 8 > 0) then
  begin
    W := cmNo;
  end
  else
  begin
    if (Act^.CopyMode = Inact^.CopyMode) and (TrueName(Act^.CopyRealPath) = TrueName(Inact^.CopyRealPath)) and
      (TrueName(Act^.CopyImageName) = TrueName(Inact^.CopyImageName)) then
    begin
      CantCopyToItself(False);
      W := cmCancel;
    end
    else
    begin
      S := '';
      if (Inact^.CopyMode = pmDisk) and (CopyDiskCopyMode <> dcFull) then S := ' '+ColorChar+'M'+ColorChar+'erge ';
      T := '';
      if C and 2 > 0 then T := ' '+ColorChar+'S'+ColorChar+'kip ';
      W := SureConfirm(stEmpty, 'The following file already exists',
        AddToPath(Inact^.CopyPath, Inact^.CopyImageName, chDirSep),
        'Do you wish to write over the old file?', stEmpty, stOverwrite, S, T, stEmpty, stEmpty,
        nil, CurHelpCtx, ayAllYes, False, AllOverwrite);
      AppendFile := (W = cmExtra);
    end;
  end;
  O := (W in [cmOK, cmExtra]);
end;

{Ask for confirmation about preformatting the destination disk if the source
  disk has more tracks}
procedure TooManySourceTracks;
var
  W             : Word;
begin
  W := SureConfirm(stEmpty, 'The source disk has more tracks than the destination,',
    'you may have problems with copying onto the extra tracks.',
    'Do you wish to format the destination disk?',
    stEmpty, stYes, stNo, stEmpty, stEmpty, stCancel, nil, CurHelpCtx, ayNone, False, DummyByte);
  case W of
    cmOK: C := C or 1;
    cmCancel: O := False;
  end;
  if W <> cmCancel then Inact^.CopyDiskType := Act^.CopyDiskType and dtTypeMask;
end;

function CallDiskEdit(BAMEditor, AllowEdit: Boolean): Boolean;
var
  F             : Boolean;
  M             : Byte;
begin
  M := Act^.CopyMode;
  F := ShowOwner;
  ShowOwner := False;
  CallDiskEdit := DiskEdit(cfSelected, True, BAMEditor, AllowEdit);
  ShowOwner := F;
  Act^.CopyMode := M;
end;

begin
  ChangeHelpCtx(hcCopyDisk);
  BoxTitle := 'Copy disk';
  FileProcessed := False;
  ExtDrive := False;
  AllDiskCopy := ayNone;
  AllOverwrite := ayNone;
  AllTruncateExtraTracks := ayNone;
  AllCreateErrorInfo := ayNone;
  AllChangeBAMIDToSectorHeaderID := ayNone;
  AllCopyFullDiskWhenBAMInvalid := ayNone;
  AllUseFCOPY := ayNone;
  CopiedSize := 0;
  H := False;
  D := nil;
  X := True;
  RetryCount := RetryNum + 1;
  InfoBuffer := New(PInfoBuffer);
  InfoBuffer2 := New(PInfoBuffer);
  CopyFileNum := 0;
  CopyFileMode := Mode;
  if (CopyFileMode <> cfSelected) or (Act^.Mode in [pmDOS, pmExt, pmDisk, pmGCRDisk, pmDiskZip, pmSixZip]) then
  begin
    case CopyFileMode of
      cfSelected:
      begin
        K := (Act^.Mode = pmExt);
        Z := '';
        case Inact^.Mode of
          pmExt: Z := DriveNumber(Inact, False);
          pmDisk, pmGCRDisk..pmSixZip: Z := MakeTypeStr(Inact^.Mode) + ':' +
            AddToPath(Inact^.CopyRealPath, Inact^.CopyImageName, chDirSep);
        end;
        if K then
        begin
          ExtDrive := True;
          Inact^.CopyPath := Inact^.Path;
          O := True;
        end
        else
        begin
          SaveSelection;
          if Act^.Mode = pmDOS then
          begin
            Act^.FirstFile := True;
            O := NextDisk;
          end
          else
          begin
            O := True;
            Act^.CopyName := Act^.ImageName;
          end;
          if (Z = '') and (Act^.CopyMode = pmDisk) then Z := DriveNumber(Inact, False) else
        end;
        if Z = '' then Z := MakeTypeStr(pmDisk) + ':' + AddToPath(Inact^.CopyRealPath, stEmpty, chDirSep);
      end;
      cfAutomatic:
      begin
        Act^.Prepare(SourceName, False, True, False);
        Inact^.Prepare(DestName, False, True, True);
        if Act^.CopyMode <> pmExt then Act^.CopyMode := pmDOS;
        K := (Act^.CopyMode = pmExt);
        if K then ExtDrive := True;
        Act^.NamePattern := Act^.CopyImageName;
        Act^.FirstFile := True;
        if Act^.CopyMode = pmDOS then O := NextDisk;
        Z := DestName;
      end;
    end;
    C := ShellBuffer^.DiskCopySettings;
    if O then
    begin
      OrigDiskCopyMode := DiskCopyMode;
      Inact^.CopyImageName := Inact^.ImageName;
      repeat
        AppendFile := False;
        Q := True;
        DiskCopySelection := False;
        ManualSelect := nil;
        Act^.CopyImageName := Act^.CopyName;
        A1 := nil;
        if K then
        begin
          O := True;
          Act^.CopyDiskType := CopyExtDiskType;
          CheckSideNumber;
        end
        else
        begin
          Act^.CopyImageProtoName := Act^.CopyImageName;
          Act^.CorrectImageName(False);
          N := Act^.CopyImageName;
        end;
        A1 := NewSItem('Copy multiple disks with index',
          NewSItem('Use disk side letters in index',
          NewSItem('Auto skip existing files',
          NewSItem('Check BAM ID against header ID',
          NewSItem('Show source dir before copy', nil)))));
        if K then
        begin
          CopyExtDisk := X;
          A1 := NewSItem('Extended 1541 disks', A1);
        end
        else
        begin
          A1 := NewSItem('Format destination disk', A1);
        end;
        if O then
        begin
          GetCheckData := C;
          GetRadioData := OrigDiskCopyMode;
          DestName := Z;
          if K and CopyExtDisk then GetCheckData := Byte(ExtendedDisk) or (GetCheckData and not 1);
          if H then InitDiskChange;
          if CopyFileMode = cfSelected then
          begin
            if K then Z := DriveNumber(Act, True) else Z := '"' + N + '"';
            ChangeHelpCtx(hcCopyDisk2);
            if (CopyFileMode = cfSelected) and (not H or K or (Inact^.CopyMode = pmExt)) then
              O := Act^.GetFileName(stEmpty, 'Copy ' + Z + ' to', '[ '+ColorChar+'C'+ColorChar+'opy ]', A1, nil, False,
                False, True, True, False, False, eeNone, aaDiskCopy);
            ChangeHelpCtx(hcCopyDisk);
            KeyBar^.Update;
          end;
          if H and SaverInUse then SaverOff;
          DoneDiskChange;
          ErrorDown := 0;
          if O then
          begin
            C := GetCheckData;
            OrigDiskCopyMode := GetRadioData;
            CopyDiskCopyMode := OrigDiskCopyMode;
            Z := DestName;
            if Act^.CopyMode = pmDOS then Act^.CopyMode := DetermineTypeName(SourceName);
            if not K then Act^.CopyImageName := N;
            M := DestName;
            if DetermineTypePrefix(M) <> pmDOS then M := Copy(M, LeftPos(':', M) + 1, MaxStrLen);
            U := (CutPath(M, chDirSep) = '');
            M := DestName;
            if U then
            begin
              if K then
              begin
                if C and 4 = 0 then N := 'disk0001' else N := 'disk001';
                C := C or 2;
                M := AddToPath(M, N, chDirSep);
                DestName := M;
              end
              else
              begin
                if Inact^.CopyMode <> pmExt then
                begin
                  case Act^.CopyMode of
                    pmDiskZip: N := Copy(N, 3, MaxStrLen);
                    pmSixZip: N := Copy(N, 4, MaxStrLen);
                  end;
                  if Inact^.CopyMode in [pmDiskZip, pmSixZip] then N := MakeFileExt(N, stEmpty);
                  M := AddToPath(M, N, chDirSep);
                end;
              end;
            end;
            if O then
            begin
              P := Act^.CopyMode;
              if P = pmExt then P := pmDisk;
              repeat
                Inact^.Prepare(AddToPath(M, stEmpty, chDirSep), True, False, (Act^.CopyMode <> pmExt));
                if Inact^.CopyMode = pmDOS then M := MakeTypeStr(P) + ':' + M;
              until Inact^.CopyMode <> pmDOS;
            end;
            if K and (Inact^.CopyMode = pmExt) then
            begin
              NoTwoDrives;
            end
            else
            begin
              ClockOff;
              if ExtDrive then MouseOff;
              if K then
              begin
                if CopyExtDisk then
                begin
                  CopyExtDisk := (C and 1 > 0);
                  C := C and not 1;
                end;
              end;
              if (C and 32 > 0) then O := ConfirmDiskCopy;
              BothBAMsOK := False;
              if O and (not K or (CopyDiskCopyMode <> dcFull)) then
              begin
                if Act^.CopyMode = pmSixZip then
                begin
                  Act^.CopyDiskType := dt1541;
                  O := (Act^.OpenZipFile(Act^.Image, 0, 0, fmReadOnly) = 0);
                  if O then
                  begin
                    ExtSeek(Act^.Image, 0);
                    ExtBlockRead(Act^.Image, TempBuffer, 3);
                    if TempBuffer[2] > Max1541Tracks then Act^.CopyDiskType := dt1541Ext;
                    ExtClose(Act^.Image);
                  end;
                  CopyDiskCopyMode := dcFull;
                end
                else
                begin
                  O := (Act^.OpenImage(False, True, True, True, True) = 0);
                  if O then
                  begin
                    Act^.CloseImage(False);
                    O := CheckBAM;
                  end
                  else
                  begin
                    if CopyDiskCopyMode = dcManualSelect then
                    begin
                      Act^.ClearBAM(True);
                      O := True;
                    end;
                  end;
                end;
              end;
              if O then
              begin
                if CopyDiskCopyMode <> dcFull then DuplicateBAM;
                if CopyDiskCopyMode = dcManualSelect then O := CallDiskEdit(True, True);
              end;
              if Act^.CopyDiskType = dtInvalid then
              begin
                ErrorWin(stEmpty, 'The following image file is invalid', Act^.CopyImageName, CurHelpCtx, sbNone);
                O := False;
              end;
              if O then
              begin
                if Act^.CopyMode = pmExt then
                begin
                  if (CopyDiskCopyMode = dcFull) and (C and 32 = 0) then
                    Act^.CopyDiskType := DetectDiskType(CopyExtDisk, True);
                  Inact^.CopyDiskType := Act^.CopyDiskType;
                  CheckDiskType;
                  Act^.CopyFullName := DriveNumber(Act, True);
                end
                else
                begin
                  ExtDrive := (Inact^.CopyMode = pmExt);
                  if O then
                  begin
                    case Inact^.CopyMode of
                      pmExt:
                      begin
                        if ExtDrive then MouseOff;
                        CBMDevNum := Ord(DestName[1]) - Ord('0');
                        if CBMDevNum < 2 then Inc(CBMDevNum, 10);
                        DestName := '';
                        S := NameOffset(Act^.CopyDiskType, Act^.ExtBAMMode);
                        for P := 0 to CBMNameLen - 1 do DestName := DestName + Chr(Act^.BAM[P + S]);
                        DestName := DestName + ',';
                        Inc(S, DiskIDRelPos);
                        for P := 0 to CBMBAMIDLen - 1 do DestName := DestName + Chr(Act^.BAM[P + S]);
                        O := True;
                        Inact^.CopyDiskType := DetectDiskType(ExtendedDisk, True);
                        P := C;
                        if (ExternalDrive = xd1571) and (Act^.CopyDiskType and dtTypeMask = dt1571) and
                          (Inact^.CopyDiskType and dtTypeMask = dt1541) then TooManySourceTracks;
                        if (C and 1 = 0) and (Act^.CopyDiskType and dtTypeMask = dt1541Ext) and
                          (Inact^.CopyDiskType and dtTypeMask in [dt1541, dt1571]) then
                        begin
                          if ExternalDrive in [xd1570, xd1571] then
                          begin
                            ErrorWin(stEmpty, 'Can''t copy an extended 1541 disk image',
                              'onto a disk in a 1570 or 1571 drive.', AppHelpCtx, sbNone);
                            O := False;
                          end
                          else
                          begin
                            if Inact^.CopyDiskType = dt1541 then TooManySourceTracks;
                          end;
                        end;
                        O := O and CheckLPTPorts(True) and ((C and 1 = 0) or (FormatDisk(Inact, False, False, False) = 0));
                        if C and 1 > 0 then Inact^.CopyDiskType := DetectDiskType(ExtendedDisk, True);
                        C := P;
                        Inact^.CopyFullName := DriveNumber(Inact, True);
                      end;
                      pmDisk, pmGCRDisk, pmDiskZip, pmSixZip: Inact^.CopyDiskType := Act^.CopyDiskType;
                    end;
                  end;
                  CheckDiskType;
                  Act^.CopyImageProtoName := Act^.CopyImageName;
                  Act^.CorrectImageName(False);
                  Act^.CopyFullName := Act^.CopyImageName;
                end;
                if O then
                begin
                  if Inact^.CopyMode <> pmExt then
                  begin
                    NextFileName(False);
                    P := Inact^.CopyDiskType;
                    if Inact^.OpenImage(False, True, True, True, False) = 0 then
                    begin
                      BothBAMsOK := BothBAMsOK and Inact^.IsBAMValid;
                      Inact^.CloseImage(False);
                    end;
                    Inact^.CopyDiskType := P;
                  end;
                  if not AppendFile then CheckDiskType;
                  if (Act^.CopyDiskType and dtTypeMask <> Inact^.CopyDiskType and dtTypeMask) and
                    not Both1541or1571Disks and ((Inact^.CopyMode = pmExt) or AppendFile) then
                  begin
                    ErrorWin(stEmpty, 'Can''t copy data between disks', 'of different types.', CurHelpCtx, sbNone);
                    O := False;
                  end;
                end;
              end;
              if O then
              begin
                CheckDiskType;
                if Inact^.CopyMode <> pmExt then
                begin
                  repeat
                    O := not FileExists(AddToPath(Inact^.CopyPath, Inact^.CopyImageName, chDirSep), False);
                    Q := True;
                    if not O then
                    begin
                      ConfirmOverwrite;
                      if W = cmNo then
                      begin
                        NextFileName(True);
                        Q := False;
                      end;
                    end;
                  until Q;
                end;
              end;
              if O then
              begin
                Q := False;
                SkipAuto := False;
                WriteFile.LongName := '';
                if ExtDrive then MouseOff;
                O := False;
                CopyStep := DriveInts[DriveIntIndex + Byte(not K)];
                if (not ExtDrive or CheckLPTPorts(True)) and OpenDiskRead(True) and OpenDiskWrite(True) then
                begin
                  FileProcessed := True;
                  MakeDisplayTS(D, stEmpty, Act^.CopyFullName, Inact^.CopyFullName);
                  if (Act^.CopyDiskType and dtTypeMask = dt1541Ext) and (Inact^.CopyDiskType and dtTypeMask = dt1541) then
                  begin
                    Inact^.CopyDiskType := (Inact^.CopyDiskType and dtErrorInfo) or dt1541Ext;
                    Inact^.CheckDiskType;
                    ClearExtraBAM;
                    Inact^.CopyMaxTrack := Max1541Tracks;
                  end;
                  O := not CancelTransfer(True);
                  P := NameOffset(Act^.CopyDiskType, Act^.ExtBAMMode) + DiskIDRelPos;
                  if O then
                  begin
                    FetchDiskID := True;
                    Act^.Track := Act^.DirTrack;
                    FillChar(TrackMap, TrackMapSize, 1);
                    TrackMap[0] := 0;
                    TrackMap[TrackMapSize] := 1;
                    O := ReadTrack(False);
                    ConvertTrack(Act^.Track, GetTrackType(Act), ttNormal, 1, False);
                    if CopyDiskCopyMode = dcFull then Move(TempBuffer[0], Act^.BAM, 256);
                    if not AppendFile then Move(TempBuffer[0], Inact^.BAM, 256);
                    case Act^.CopyMode of
                      pmExt: if CopyTransferMode <> tmWarp then Move(Act^.BAM[P], CopyHeaderID, CBMHeaderIDLen);
                      pmDisk: if not Act^.FCOPYBlock then Move(Act^.BAM[P], CopyHeaderID, CBMHeaderIDLen);
                      pmDiskZip:
                      begin
                        if Act^.OpenZipFile(TempFile, 0, 2, fmReadOnly) = 0 then
                        begin
                          ExtBlockRead(TempFile, CopyHeaderID, CBMHeaderIDLen);
                          ExtClose(TempFile);
                        end;
                      end;
                    end;
                  end;
                  FetchDiskID := False;
                  if O then
                  begin
                    Move(Act^.BAM[P], CopyBAMID, CBMHeaderIDLen);
                    if Inact^.CopyMode = pmDisk then
                    begin
                      if C and 16 > 0 then
                      begin
                        if CopyHeaderID <> CopyBAMID then
                        begin
                          M[0] := #5;
                          Move(Act^.BAM[P], M[1], Length(M));
                          A1 := NewSItem('Header ID: "' + MakeIDString(@CopyHeaderID, False) + '"',
                            NewSItem('   BAM ID: "' + MakeCBMName(M, False) + '"', nil));
                          R.Assign(0, 0, 19, 2);
                          V := New(PCBMTextList, Init(R, A1));
                          W := SureConfirm(stEmpty, 'The BAM ID does not match the sector header ID.',
                            'Do you wish to change the BAM ID accordingly?',
                            stEmpty, stEmpty, stYes, stEmpty, stEmpty, stNo, stEmpty, V, CurHelpCtx,
                            ayAllYes, True, AllChangeBAMIDToSectorHeaderID);
                          case W of
                            cmOK:
                            begin
                              M[0] := Chr(CBMHeaderIDLen + 1);
                              Move(CopyHeaderID, M[1], CBMHeaderIDLen);
                              M[CBMHeaderIDLen + 1] := chShiftSpace;
                              M := M + DiskTypeSignature(Act^.CopyDiskType, Act^.ExtBAMMode);
                              Move(M[1], Act^.BAM[P], Length(M));
                              Move(M[1], Inact^.BAM[P], Length(M));
                              BufferChanged := True;
                            end;
                            cmCancel: O := False;
                          end;
                        end;
                      end;
                    end
                    else
                    begin
                      if (Act^.CopyMode = pmDisk) and (Inact^.CopyMode <> pmExt) then
                      begin
                        M[0] := Chr(CBMHeaderIDLen);
                        Move(CopyBAMID, M[1], CBMHeaderIDLen);
                        UserTitle := ConvertCBMName(M, False, False, hxPercent);
                        R.Assign(0, 0, 19, 1);
                        A2 := New(PInputLine, Init(R, 6, 6, 'Header ID:', drLeft));
                        W := SureConfirm(stEmpty, 'The source disk contains no sector header ID.',
                          'Do you wish to enter the sector header ID manually?',
                          stEmpty, stEmpty, stYes, stEmpty, stEmpty, stNo, stEmpty, A2, CurHelpCtx,
                          ayNone, True, DummyByte);
                        case W of
                          cmOK:
                          begin
                            M := ReconvertCBMName(UserTitle, False, True, hxPercent);
                            Move(M[1], CopyHeaderID, CBMHeaderIDLen);
                          end;
                          cmCancel: O := False;
                        end;
                      end;
                    end;
                  end;
                  Act^.Track := 1;
                  while (Act^.Track < Act^.CopyMaxTrack) and O do
                  begin
                    O := ReadTrack(True);
                    if O then
                    begin
                      ConvertTrack(Act^.Track, GetTrackType(Act), GetTrackType(Inact), Act^.SectorNum(Act^.Track), True);
                      O := WriteTrack(True);
                    end;
                    if O then Inc(Act^.Track);
                  end;
                  Inact^.CheckDiskType;
                  CloseDiskRead;
                  if O then
                  begin
                    if Inact^.CopyMode = pmExt then
                    begin
                      CloseDiskWrite(True);
                    end
                    else
                    begin
                      if ExtDrive then OpenCBMChannel(saCommand, 'I0', True);
                      if Inact^.CopyMode <> pmExt then
                      begin
                        if Inact^.CopyDiskType and dtTypeMask = dt1541Ext then
                        begin
                          F := True;
                          W := Inact^.DiskPos(Max1541Tracks, 0);
                          while F and (W < Inact^.CopyDiskSize) do
                          begin
                            F := (InfoBuffer^[W] in [ds20READ, ds21READ, dsEmpty]);
                            Inc(W);
                          end;
                          if F and (SureConfirm(stEmpty, 'No data was copied from the extended tracks of the source disk.',
                            'Do you wish to truncate them off the destination disk?',
                            stEmpty, stEmpty, stYes, stEmpty, stEmpty, stEmpty, stNo, nil, CurHelpCtx,
                            ayAllYes, True, AllTruncateExtraTracks) = cmOK) then
                          begin
                            Inact^.CopyDiskType := (Inact^.CopyDiskType and dtErrorInfo) or dt1541;
                            Inact^.CheckDiskType;
                            case Inact^.CopyMode of
                              pmDisk:
                              begin
                                ExtSeek(WriteFile, Inact^.CopyDiskSize shl 8);
                                ExtTruncate(WriteFile);
                              end;
                              pmDiskZip:
                              begin
                                CloseDiskWrite(True);
                                LongErase(WriteFile.LongName);
                              end;
                              pmSixZip:
                              begin
                                ExtSeek(WriteFile, CopiedSize);
                                ExtTruncate(WriteFile);
                                CloseDiskWrite(True);
                                M := GetPath(WriteFile.LongName, chDirSep);
                                N := CutPath(WriteFile.LongName, chDirSep);
                                P := Max1541Tracks;
                                for N[1] := '1' to Chr(Ord('0') + SixZipFileNum) do
                                begin
                                  if LongOpenFile(AddToPath(M, N, chDirSep), WriteFile, fmReadWrite) = 0 then
                                  begin
                                    ExtGetFTime(WriteFile, L);
                                    ExtSeek(WriteFile, 2);
                                    ExtBlockWrite(WriteFile, P, 1);
                                    ExtSetFTime(WriteFile, L);
                                    ExtClose(WriteFile);
                                  end;
                                end;
                              end;
                            end;
                          end;
                        end;
                        if Inact^.CopyMode = pmDisk then
                        begin
                          for W := 0 to Inact^.CopyDiskSize - 1 do
                            if InfoBuffer^[W] = dsEmpty then InfoBuffer^[W] := dsOK;
                          F := False;
                          W := 0;
                          while not F and (W < Inact^.CopyDiskSize) do
                          begin
                            F := (InfoBuffer^[W] <> dsOK);
                            Inc(W);
                          end;
                          if AppendFile then ExtSeek(WriteFile, Inact^.CopyDiskSize shl 8);
                          if F then
                          begin
                            repeat
                              W := SureConfirm(stEmpty, 'There were errors during the disk copy.',
                                'Do you wish to create an error info block?', stEmpty, stEmpty, stYes, stEmpty, stEmpty,
                                ' '+ColorChar+'D'+ColorChar+'isplay ', stNo, nil, CurHelpCtx,
                                ayAllYes, True, AllCreateErrorInfo);
                              if W = cmSkip then CallDiskEdit(False, False);
                            until (W = cmOK) or (W = cmCancel);
                            F := (W = cmOK);
                          end;
                          if F then ExtBlockWrite(WriteFile, InfoBuffer^, Inact^.CopyDiskSize) else
                            ExtTruncate(WriteFile);
                          if AppendFile then
                          begin
                            BufferChanged := True;
                            F := False;
                            W := 0;
                            while not F and (W < FirstDirSec(Inact^.CopyDiskType)) do
                            begin
                              F := ManualSelect^.Contains(Act^.DiskPos(Act^.MainDirTrack, W));
                              Inc(W);
                            end;
                            if F then
                            begin
                              CopyBAMChunk($0000, $0004);
                              case Act^.CopyDiskType and dtTypeMask of
                                dt1541: CopyBAMChunk($00AC, (SizeOf(TBlock) - $00AC));
                                dt1541Ext:
                                begin
                                  case Act^.ExtBAMMode of
                                    xbSpeedDOS:
                                    begin
                                      CopyBAMChunk($00AC, ($00C0 - $00AC));
                                      CopyBAMChunk($00D4, (SizeOf(TBlock) - $00D4));
                                    end;
                                    xbDolphinDOS: CopyBAMChunk($00C0, (SizeOf(TBlock) - $00C0));
                                    xbPrologicDOS:
                                    begin
                                      F := False;
                                      CopyBAMChunk($00A4, (SizeOf(TBlock) - $00A4));
                                    end;
                                  end;
                                end;
                                dt1571: CopyBAMChunk($00AC, ($00DD - $00AC));
                                dt1581:
                                begin
                                  F := False;
                                  CopyBAMChunk($0004, ($0110 - $0004));
                                  CopyBAMChunk($0200, ($0210 - $0200));
                                end;
                              end;
                              if F then CopyBAMChunk($0090, ($00AC - $0090));
                            end;
                          end;
                          if BufferChanged then
                          begin
                            ExtSeek(WriteFile, Inact^.DiskPos(Inact^.DirTrack, 0) shl 8);
                            ExtBlockWrite(WriteFile, Inact^.BAM, FirstDirSec(Inact^.CopyDiskType) shl 8);
                          end;
                        end;
                      end;
                      CloseDiskWrite(True);
                    end;
                  end
                  else
                  begin
                    CloseDiskWrite(False);
                    if not AppendFile and (WriteFile.LongName <> '') then DeleteDiskWrite;
                  end;
                  Dispose(D, Done);
                end
                else
                begin
                  CloseDiskRead;
                  CloseDiskWrite(False);
                end;
              end
              else
              begin
                Q := False;
              end;
            end;
          end;
        end;
        H := True;
        if Act^.Mode = pmDOS then
        begin
          Act^.FirstFile := False;
          UnselectFile(Act, O);
          if O then UnselectAllVolumes;
        end;
        if O then NextFileName(True);
        if K then
        begin
          Q := Q or (C and 2 = 0);
        end
        else
        begin
          if CopyFileMode = cfAutomatic then Act^.CopyMode := pmDOS else Act^.CopyMode := Act^.Mode;
          Q := Q or (Act^.CopyMode in [pmDisk, pmGCRDisk, pmDiskZip, pmSixZip]) or not NextDisk;
          O := True;
        end;
        if ManualSelect <> nil then Dispose(ManualSelect);
      until Q;
      if not K then UnselectProcessed;
      MouseOn;
      ClockOn;
      if Act^.Cur = -1 then Act^.Cur := 0;
      Act^.DrawPanel;
      if FileProcessed then
      begin
        if Act^.CopyMode <> pmExt then Act^.Read;
        if (Inact^.Mode <> pmExt) or ExtDrive then Inact^.Read;
      end;
    end;
  end;
  CopyExtDisk := ExtendedDisk;
  ShellBuffer^.DiskCopySettings := C and not 1;
  Dispose(InfoBuffer);
  Dispose(InfoBuffer2);
  RestoreHelpCtx;
  NextBatchCommand;
end;

end.
