
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Turbo Vision Unit                               }
{                                                       }
{       Copyright (c) 1992 Borland International        }
{                                                       }
{*******************************************************}

unit Dialogs;

{$A+,B-,D+,E-,F+,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Drivers, Objects, Validate, Views;

const
{Maximum length of strings}
  MaxStrLen     = 254;
type
  TDegranulateClipboardProc = procedure (Data: Pointer; var Len: Longint);
  TGetClipboardSizeProc = function: Longint;
  TGetClipboardProc = function (Data: Pointer; MaxLen: Longint): Boolean;
  TPutClipboardProc = function (Data: Pointer; Len: Longint): Boolean;

const

{ Color palettes }

  CStaticText    = #1#2;
  CButton        = #1#3#2;
  CCluster       = #1;
  CInputLine     = #3#1#4#2;

type

{ TDialog object }

  PDialog = ^TDialog;
  TDialog = object(TWindow)
    VerticalWrap: Boolean;
    constructor Init(var Bounds: TRect; ATitle: TTitleStr; AFrameX, AFrameY: Integer; AVerticalWrap: Boolean);
    procedure HandleEvent(var Event: TEvent); virtual;
    function Valid(Command: Word): Boolean; virtual;
  end;

{ TSItem }

  PSItem = ^TSItem;
  TSItem = record
    Value: PString;
    Next: PSItem;
  end;

{ TInputLine object }

  PInputLine = ^TInputLine;
  TInputLine = object(TView)
    Data, Title: PString;
    TitleDir: Byte;
    Len, MaxLen: Integer;
    OldCurPos,
    CurPos: Integer;
    FirstPos: Integer;
    SelStart: Integer;
    SelEnd: Integer;
    FirstDataChar: Integer;
    Validator: PValidator;
    ChangedData, ChangedPos: Boolean;
    First, CursorStill, ConvertEOL: Boolean;
    CanQuote, Quote, InsMode: Boolean;
    constructor Init(var Bounds: TRect; ALen, AMaxLen: Integer; const ATitle: string; ATitleDir: Byte);
    destructor Done; virtual;
    procedure DeleteAll;
    procedure Draw; virtual;
    procedure GetData(var Rec); virtual;
    function GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    function MouseInView(Mouse: TPoint): Boolean;
    procedure SelectAll;
    procedure SetData(var Rec); virtual;
    procedure SetState(AState: Word; Enable: Boolean); virtual;
    procedure SetValidator(AValid: PValidator);
    function Valid(Command: Word): Boolean; virtual;
  end;

{ TButton object }

  PButton = ^TButton;
  TButton = object(TView)
    Title: PString;
    Command: Word;
    constructor Init(var Bounds: TRect; ATitle: TTitleStr; ACommand: Word);
    destructor Done; virtual;
    procedure Draw; virtual;
    function GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Press; virtual;
    procedure SetState(AState: Word; Enable: Boolean); virtual;
  end;

{ TCluster }

  PCluster = ^TCluster;
  TCluster = object(TView)
    Value: LongInt;
    Sel: Integer;
    GroupedTab: Boolean;
    Strings: TStringCollection;
    constructor Init(var Bounds: TRect; AStrings: PSItem);
    destructor Done; virtual;
    procedure DrawBox(const Icon: string; Marker: Char);
    procedure GetData(var Rec); virtual;
    function GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    function Mark(Item: Integer): Boolean; virtual;
    procedure Press(Item: Integer; Keyboard: Boolean); virtual;
    procedure SetData(var Rec); virtual;
    procedure SetState(AState: Word; Enable: Boolean); virtual;
  end;

{ TRadioButtons }

  PRadioButtons = ^TRadioButtons;
  TRadioButtons = object(TCluster)
    constructor Init(var Bounds: TRect; AStrings: PSItem);
    procedure Draw; virtual;
    function Mark(Item: Integer): Boolean; virtual;
    procedure Press(Item: Integer; Keyboard: Boolean); virtual;
    procedure SetData(var Rec); virtual;
  end;

{ TCheckBoxes }

  PCheckBoxes = ^TCheckBoxes;
  TCheckBoxes = object(TCluster)
    procedure Draw; virtual;
    function Mark(Item: Integer): Boolean; virtual;
    procedure Press(Item: Integer; Keyboard: Boolean); virtual;
  end;

{ TStaticText }

  PStaticText = ^TStaticText;
  TStaticText = object(TView)
    Text: PString;
    constructor Init(var Bounds: TRect; const AText: string);
    destructor Done; virtual;
    procedure Draw; virtual;
    function GetPalette: PPalette; virtual;
  end;

{ TColorText }

  PColorText = ^TColorText;
  TColorText = object(TStaticText)
    procedure Draw; virtual;
  end;

function IsSeparator(Ch: Char): Boolean;

{ SItem routines }

function NewSItem(const Str: string; ANext: PSItem): PSItem;

var
  GoSound: Boolean;
  DegranulateClipboardProc: TDegranulateClipboardProc;
  GetClipboardSizeProc: TGetClipboardSizeProc;
  GetClipboardProc: TGetClipboardProc;
  PutClipboardProc: TPutClipboardProc;

implementation

{ Utility functions }

function IsSeparator(Ch: Char): Boolean;
begin
  IsSeparator := (Ch in [chTab, ' ', '+', ':', '.', ',', '>', '<', '|', '\']);
end;

{ TDialog }

constructor TDialog.Init(var Bounds: TRect; ATitle: TTitleStr; AFrameX, AFrameY: Integer; AVerticalWrap: Boolean);
begin
  Inc(Bounds.B.X, Bounds.A.X);
  Inc(Bounds.B.Y, Bounds.A.Y);
  inherited Init(Bounds, ATitle, AFrameX, AFrameY);
  Palette := wpDialog;
  VerticalWrap := AVerticalWrap;
end;

procedure TDialog.HandleEvent(var Event: TEvent);

procedure SelectNeighbor(View: PView; Side: Byte; Force: Boolean);
var
  Vertical, Forwards, HorizontalWrap: Boolean;
  B, F: Boolean;
  X: Byte;
  I, J, K, M, Z, W: Integer;
  A, P, R: PView;
  C, D, O, Q, S, V: TPoint;

function RightDirection(Coord1, Coord2: Integer): Boolean;
begin
  RightDirection := ((Coord1 = Coord2) or ((Coord1 < Coord2) = Forwards));
end;

function PointPointDistance(Coord1, Coord2, OwnerSize: Integer): Integer;
var
  I: Integer;
begin
  I := Abs(Coord1 - Coord2);
  if (OwnerSize > 0) and (I > 0) and not RightDirection(Coord1, Coord2) then I := OwnerSize - I;
  PointPointDistance := I;
end;

function PointRectDistance(Coord1, Orig2, Size2, OwnerSize: Integer): Integer;
var
  I, J: Integer;
begin
  if (Coord1 >= Orig2) and (Coord1 < Orig2 + Size2) then
  begin
    I := 0;
  end
  else
  begin
    I := PointPointDistance(Coord1, Orig2, OwnerSize);
    J := PointPointDistance(Coord1, Orig2 + Size2 - 1, OwnerSize);
    if I > J then I := J;
  end;
  PointRectDistance := I;
end;

function RectRectDistance(Orig1, Size1, Orig2, Size2, OwnerSize: Integer): Integer;
var
  I, J: Integer;
begin
  if ((Orig1 >= Orig2) and (Orig1 < Orig2 + Size2)) or ((Orig2 >= Orig1) and (Orig2 < Orig1 + Size1)) then
  begin
    I := 0;
  end
  else
  begin
    I := PointPointDistance(Orig1, Orig2 + Size2 - 1, OwnerSize);
    J := PointPointDistance(Orig1 + Size1 - 1, Orig2, OwnerSize);
    if I > J then I := J;
  end;
  RectRectDistance := I;
end;

function IsCluster(View: PView): Boolean;
begin
  IsCluster := ((View <> nil) and (View^.ViewType = vtCluster));
end;

function IsButton(View: PView): Boolean;
begin
  IsButton := ((View <> nil) and (View^.ViewType = vtButton));
end;

function IsOKButton(View: PView): Boolean;
begin
  IsOKButton := (IsButton(View) and (PButton(View)^.Command = cmOK));
end;

begin
  Vertical := (Side in [drUp, drDown]);
  Forwards := (Side in [drDown, drRight]);
  HorizontalWrap := Force or IsButton(View);
  B := IsCluster(View);
  A := View;
  V.X := 0;
  V.Y := 0;
  if View <> nil then V := View^.Origin;
  if IsOKButton(View) then V.X := 5;
  C := V;
  if IsCluster(View) then
  begin
    Inc(C.X, View^.Cursor.X);
    Inc(C.Y, View^.Cursor.Y);
  end;
  if Vertical then if Forwards then Inc(C.Y) else Dec(C.Y) else
    if Forwards then Inc(C.X) else Dec(C.X);
  S.X := 1;
  S.Y := 1;
  if View <> nil then S := View^.Size;
  O := Size;
  Inc(O.X, 20);
  Inc(O.Y, 2);
  M := MaxInt;
  Z := MaxInt;
  W := MaxInt;
  X := 0;
  if View = nil then
  begin
    R := First;
    P := R;
  end
  else
  begin
    R := View;
    P := View^.Next;
  end;
  repeat
    if P^.Options and ofSelectable > 0 then
    begin
      Q := P^.Origin;
      if IsOKButton(P) then Inc(X);
      if X = 1 then Q.X := 5;
      if (Vertical and (VerticalWrap or RightDirection(C.Y, Q.Y))) or
        (not Vertical and (HorizontalWrap or RightDirection(C.X, Q.X))) then
      begin
        if Vertical then
        begin
          D.X := RectRectDistance(V.X, S.X, Q.X, P^.Size.X, 0);
          D.Y := PointPointDistance(C.Y, Q.Y, O.Y);
          if B then J := PointRectDistance(C.X, Q.X, P^.Size.X, 0) else
            J := RectRectDistance(V.X, S.X, Q.X, P^.Size.X, 0);
          F := True;
        end
        else
        begin
          D.X := PointPointDistance(C.X, Q.X, O.X);
          if B then D.Y := PointRectDistance(C.Y, Q.Y, P^.Size.Y, 0) else
            D.Y := RectRectDistance(V.Y, S.Y, Q.Y, P^.Size.Y, 0);
          F := Force or ((D.Y = 0) or ((D.Y = 1) and (D.X > 10)));
        end;
        K := PointPointDistance(C.X, Q.X, O.X) + PointPointDistance(C.Y, Q.Y, O.Y);
        if Force then I := D.X + D.Y else if Vertical then I := 20 * D.X + D.Y else I := D.X + 20 * D.Y;
        if F and ((M > I) or (Vertical and (M = I) and ((Z > J) or ((Z = J) and (W > K))))) then
        begin
          A := P;
          M := I;
          Z := J;
          W := K;
        end;
      end;
    end;
    if X <> 1 then P := P^.Next;
  until (X <> 1) and (P = R);
  if (A <> View) and IsCluster(A) then
  begin
    I := C.Y - A^.Origin.Y;
    if Vertical and not RightDirection(C.Y, A^.Origin.Y) then if Forwards then Dec(I, O.Y) else Inc(I, O.Y);
    if I < 0 then I := 0 else if I >= A^.Size.Y then I := A^.Size.Y - 1;
    PCluster(A)^.Sel := I;
  end;
  if (A <> nil) and (A^.Options and ofSelectable > 0) then A^.Select;
end;

begin
  TWindow.HandleEvent(Event);
  case Event.What of
    evKeyDown:
      case Event.KeyCode of
        kbEnter, kbEsc:
          begin
            Event.What := evCommand;
            if Event.KeyCode = kbEnter then Event.Command := cmOK else Event.Command := cmCancel;
            PutEvent(Event);
            ClearEvent(Event);
          end;
        kbUp, kbShiftTab:
          begin
            SelectNeighbor(Current, drUp, False);
            ClearEvent(Event);
          end;
        kbDown, kbTab:
          begin
            SelectNeighbor(Current, drDown, False);
            ClearEvent(Event);
          end;
        kbLeft:
          begin
            SelectNeighbor(Current, drLeft, False);
            ClearEvent(Event);
          end;
        kbRight:
          begin
            SelectNeighbor(Current, drRight, False);
            ClearEvent(Event);
          end;
        kbHome: if VerticalWrap then SelectNeighbor(nil, drDown, True) else SelectNeighbor(nil, drRight, True);
        kbPgUp: if VerticalWrap then SelectNeighbor(nil, drDown, True);
        kbEnd: if VerticalWrap then SelectNeighbor(nil, drUp, True) else SelectNeighbor(nil, drLeft, True);
        kbPgDn: if VerticalWrap then SelectNeighbor(nil, drUp, True);
      end;
    evCommand:
      case Event.Command of
        cmOK..cmCancel:
          if State and sfModal <> 0 then
          begin
            EndModal(Event.Command);
            ClearEvent(Event);
          end;
      end;
  end;
end;

function TDialog.Valid(Command: Word): Boolean;
begin
  if Command = cmCancel then Valid := True
  else Valid := TGroup.Valid(Command);
end;

function NewSItem(const Str: string; ANext: PSItem): PSItem;
var
  Item: PSItem;
begin
  New(Item);
  Item^.Value := NewStr(Str);
  Item^.Next := ANext;
  NewSItem := Item;
end;

function HotKey(const S: string): Char;
var
  P: Word;
begin
  P := LeftPos(ColorChar,S);
  if P <> 0 then HotKey := UpCase(S[P+1])
  else HotKey := #0;
end;

{ TInputLine }

constructor TInputLine.Init(var Bounds: TRect; ALen, AMaxLen: Integer; const ATitle: string; ATitleDir: Byte);
begin
  Inc(Bounds.B.X, Bounds.A.X);
  Inc(Bounds.B.Y, Bounds.A.Y);
  TView.Init(Bounds);
  State := State or sfCursorVis;
  Options := Options or (ofSelectable + ofFirstClick);
  GetMem(Data, AMaxLen + 1);
  GetMem(Title, Length(ATitle) + 1);
  Data^ := '';
  Title^ := ATitle;
  TitleDir := ATitleDir;
  CurPos := 0;
  FirstPos := 0;
  Len := ALen;
  MaxLen := AMaxLen;
  CanQuote := True;
  Quote := False;
  CursorStill := False;
  ConvertEOL := True;
  FirstDataChar := 0;
  First := True;
  if TitleDir in [drLeft, drLeftClose] then
  begin
    FirstDataChar := Length(Title^);
    if TitleDir = drLeft then Inc(FirstDataChar);
  end;
  ViewType := vtInputLine;
end;

destructor TInputLine.Done;
begin
  FreeMem(Data, MaxLen + 1);
  FreeMem(Title, Length(Title^) + 1);
  SetValidator(nil);
  TView.Done;
end;

procedure TInputLine.DeleteAll;
begin
  Data^ := '';
  CurPos := 0;
  SelStart := 0;
  SelEnd := 0;
  ChangedData := True;
  First := False;
end;

procedure TInputLine.Draw;
var
  Color: Word;
  X, Y, Z: Integer;
  B: TDrawBuffer;

function GetMark(Mark: Integer): Integer;
begin
  Dec(Mark, FirstPos);
  if Mark < 0 then Mark := 0;
  if Mark > Len then Mark := Len;
  GetMark := Mark;
end;

begin
  Color := GetColor($0402);
  X := 0;
  Y := 0;
  MoveChar(B, ' ', Color, Size.X);
  if TitleDir = drUp then
  begin
    MoveStr(B, Title^, Color);
    WriteBuf(0, 0, Size.X, 1, B);
    MoveChar(B, ' ', Color, Size.X);
    Inc(Y);
  end;
  case TitleDir of
    drLeft, drLeftClose:
    begin
      MoveCStr(B, Title^, Color);
      X := FirstDataChar;
    end;
    drRight: MoveCStr(B[Len + 1], Title^, Color);
  end;
  Color := GetColor(1);
  MoveChar(B[X], ' ', Color, Len);
  MoveStr(B[X], Copy(Data^, FirstPos + 1, Len), Color);
  if FirstPos > 0 then MoveChar(B[X], ArrowChars[1], Color, 1);
  if FirstPos + Len < Length(Data^) then MoveChar(B[X + Len - 1], ArrowChars[2], Color, 1);
  if SelStart < SelEnd then
  begin
    Z := GetMark(SelStart);
    MoveColor(B[X + Z], GetColor(3), GetMark(SelEnd) - Z);
  end;
  WriteBuf(0, Y, Size.X, 1, B);
  SetCursor(X + CurPos - FirstPos, Y);
end;

procedure TInputLine.GetData(var Rec);
begin
  FillChar(Rec, MaxLen + 1, #0);
  Move(Data^, Rec, Length(Data^) + 1);
end;

function TInputLine.GetPalette: PPalette;
const
  P: string[Length(CInputLine)] = CInputLine;
begin
  GetPalette := @P;
end;

procedure TInputLine.HandleEvent(var Event: TEvent);
var
  Ch: Char;
  Delta, I, J: Integer;
  OldData: string;
  OldFirstPos, OldSelStart, OldSelEnd: Integer;
  FirstMove, ClipInsOK, DelSel, SelMove, WasAppending: Boolean;
  L: Longint;
  S: array [0..2 * MaxStrLen - 1] of Byte;

function CanScroll(Delta: Integer): Boolean;
begin
  if Len <= MaxLen then CanScroll := False else
    if Delta < 0 then
      CanScroll := FirstPos > 0 else
    if Delta > 0 then
      CanScroll := Length(Data^) - FirstPos >= Len else
      CanScroll := False;
end;

function MouseDelta: Integer;
var
  Mouse: TPoint;
begin
  MakeLocal(Event.Where, Mouse);
  if Mouse.X <= 0 then MouseDelta := -1 else
  if Mouse.X >= Len - 1 then MouseDelta := 1 else
  MouseDelta := 0;
end;

function MousePos: Integer;
var
  Pos: Integer;
  Mouse: TPoint;
begin
  MakeLocal(Event.Where, Mouse);
  Dec(Mouse.X, FirstDataChar);
  Pos := Mouse.X + FirstPos;
  if Pos < 0 then Pos := 0;
  if Pos > Length(Data^) then Pos := Length(Data^);
  MousePos := Pos;
end;

procedure SaveState;
begin
  OldCurPos := CurPos;
  SelMove := (GetShiftState and (kbLeftShift + kbRightShift) > 0);
  if Validator <> nil then
  begin
    OldData := Data^;
    OldFirstPos := FirstPos;
    OldSelStart := SelStart;
    OldSelEnd := SelEnd;
    WasAppending := Length(Data^) = CurPos;
  end;
end;

procedure RestoreState;
begin
  if Validator <> nil then
  begin
    Data^ := OldData;
    CurPos := OldCurPos;
    FirstPos := OldFirstPos;
    SelStart := OldSelStart;
    SelEnd := OldSelEnd;
  end;
end;

function CheckValid(NoAutoFill, Movement: Boolean): Boolean;
var
  OldLen: Integer;
  NewData: string;
begin
  if Validator <> nil then
  begin
    CheckValid := False;
    OldLen := Length(Data^);
    NewData := Data^;
    if Validator^.IsValidInput(NewData, NoAutoFill) then
    begin
      if Length(NewData) > MaxLen then NewData[0] := Char(MaxLen);
      Data^ := NewData;
      if (CurPos >= OldLen) and (Length(Data^) > OldLen) then
        CurPos := Length(Data^);
      CheckValid := True;
    end
    else
      RestoreState;
  end
  else
    CheckValid := True;
  if Movement and SelMove and (CurPos <> OldCurPos) then
  begin
    if OldCurPos = SelStart then
    begin
      SelStart := CurPos;
    end
    else
    begin
      if OldCurPos <> SelEnd then SelStart := OldCurPos;
      SelEnd := CurPos;
    end;
    if SelStart > SelEnd then
    begin
      OldLen := SelStart;
      SelStart := SelEnd;
      SelEnd := OldLen;
    end;
  end;
end;

function FindSeparator(Back: Boolean): Byte;
var
  Index: Byte;
begin
  Index := CurPos;
  if Back then
  begin
    while (Index > 0) and IsSeparator(Data^[Index]) do Dec(Index);
    while (Index > 0) and not IsSeparator(Data^[Index]) do Dec(Index);
  end
  else
  begin
    while (Index < Length(Data^)) and IsSeparator(Data^[Index + 1]) do Inc(Index);
    while (Index < Length(Data^)) and not IsSeparator(Data^[Index + 1]) do Inc(Index);
  end;
  FindSeparator := Index;
end;

procedure MoveSelMarks(Change: Integer);

procedure MoveMark(var Mark: Integer);
var
  L             : Longint;
begin
  if Mark >= CurPos then
  begin
    if Mark < CurPos - Change then
    begin
      Mark := CurPos;
    end
    else
    begin
      L := Mark - CurPos;
      if L > Change then L := Change;
      Inc(Mark, L);
    end;
  end;
end;

begin
  MoveMark(SelStart);
  MoveMark(SelEnd);
end;

procedure DeleteSel;
begin
  Delete(Data^, SelStart + 1, SelEnd - SelStart);
  CurPos := SelStart;
  SelStart := 0;
  SelEnd := 0;
  ChangedData := True;
end;

procedure DelWordRight;
begin
  if CurPos < Length(Data^) then
  begin
    Delete(Data^, CurPos + 1, FindSeparator(False) - CurPos);
    MoveSelMarks(-1);
    ChangedData := True;
  end;
  CheckValid(True, False);
end;

procedure ScrollLine(Step: Integer);
begin
  if ChangedData then ChangedPos := True;
  if ChangedPos then
  begin
    I := Step;
    if Len < I then I := Len;
    if CurPos < FirstPos then if CurPos > I then FirstPos := CurPos - (I - 1) else FirstPos := 0;
    I := CurPos - Len;
    if MaxLen > Len then Inc(I);
    if FirstPos < I then FirstPos := I;
    First := False;
    DrawView;
    ClearEvent(Event);
  end;
end;

begin
  ChangedData := False;
  ChangedPos := False;
  TView.HandleEvent(Event);
  if State and sfSelected <> 0 then
  begin
    case Event.What of
      evMouseDown:
        begin
          if MouseButtons and mbMiddleButton > 0 then
          begin
            Event.What := evKeyboard;
            Event.KeyCode := kbEnter;
          end
          else
          begin
            FirstMove := True;
            Delta := MouseDelta;
            if CanScroll(Delta) then
            begin
              repeat
                if CanScroll(Delta) then
                begin
                  Inc(FirstPos, Delta);
                  DrawView;
                end;
              until not MouseEvent(Event, evMouseAuto);
            end else
            if not Event.Double then
            begin
              repeat
                CurPos := MousePos;
                ChangedPos := True;
                if Event.Buttons and mbRightButton > 0 then
                begin
                  if FirstMove then
                  begin
                    FirstMove := False;
                    OldCurPos := CurPos;
                    SelStart := CurPos;
                    SelEnd := SelStart;
                  end
                  else
                  begin
                    if CurPos < OldCurPos then
                    begin
                      SelStart := CurPos;
                      SelEnd := OldCurPos;
                    end
                    else
                    begin
                      SelStart := OldCurPos;
                      SelEnd := CurPos;
                    end;
                  end;
                end;
                CheckValid(True, False);
                ScrollLine(1);
                DrawView;
              until not MouseEvent(Event, evMouseMove + evMouseAuto);
            end;
            ClearEvent(Event);
          end;
        end;
      evKeyDown:
        begin
          SaveState;
          if not Quote then
          begin
            case Event.KeyCode of
              $0001..$001F: Quote := True;
            end;
          end;
          if not CursorStill and not Quote then
          begin
            case Event.KeyCode of
              kbLeft, kbCtrlS:
                if CurPos > 0 then
                begin
                  Dec(CurPos);
                  ChangedPos := True;
                  CheckValid(True, True);
                  ClearEvent(Event);
                end;
              kbRight, kbCtrlD:
                if CurPos < Length(Data^) then
                begin
                  Inc(CurPos);
                  ChangedPos := True;
                  CheckValid(True, True);
                  ClearEvent(Event);
                end;
              kbCtrlLeft, kbCtrlA:
                begin
                  CurPos := FindSeparator(True);
                  ChangedPos := True;
                  CheckValid(True, True);
                end;
              kbCtrlRight, kbCtrlF:
                begin
                  CurPos := FindSeparator(False);
                  ChangedPos := True;
                  CheckValid(True, True);
                end;
              kbHome, kbCtrlHome:
                begin
                  ChangedPos := True;
                  CurPos := 0;
                  CheckValid(True, True);
                end;
              kbEnd, kbCtrlEnd:
                begin
                  CurPos := Length(Data^);
                  ChangedPos := True;
                  CheckValid(True, True);
                end;
              kbBack, kbCtrlH:
                if CurPos > 0 then
                begin
                  Delete(Data^, CurPos, 1);
                  Dec(CurPos);
                  MoveSelMarks(-1);
                  ChangedData := True;
                  CheckValid(True, False);
                end;
              kbCtrlBack, kbCtrlW:
                begin
                  if CurPos > 0 then
                  begin
                    I := FindSeparator(True);
                    J := I - CurPos;
                    Delete(Data^, I + 1, CurPos - I);
                    CurPos := I;
                    MoveSelMarks(J);
                    ChangedData := True;
                  end;
                  Event.KeyCode := 0;
                  CheckValid(True, False);
                end;
              kbDel, kbCtrlG:
                begin
                  Delete(Data^, CurPos + 1, 1);
                  if CurPos < SelEnd then MoveSelMarks(-1);
                  ChangedData := True;
                  CheckValid(True, False);
                end;
              kbCtrlY:
                begin
                  DeleteAll;
                  ChangedData := True;
                  CheckValid(True, False);
                end;
              kbCtrlK:
                begin
                  Delete(Data^, CurPos + 1, MaxStrLen);
                  ChangedData := True;
                  CheckValid(True, False);
                end;
              kbCtrlV, kbIns:
                begin
                  InsMode := not InsMode;
                  SetState(sfCursorIns, InsMode);
                end;
              kbCtrlDel: if SelStart = SelEnd then DelWordRight else DeleteSel;
              kbCtrlT: DelWordRight;
              kbCtrlQ:
                if CanQuote then
                begin
                  Quote := True;
                  ClearEvent(Event);
                  Exit;
                end;
              kbCtrlIns, kbShiftDel:
                begin
                  ClipInsOK := True;
                  DelSel := (Event.KeyCode = kbShiftDel);
                  I := SelStart;
                  J := SelEnd;
                  if I >= J then
                  begin
                    if DelSel then ClipInsOK := False else
                    begin
                      I := 0;
                      J := Length(Data^);
                    end;
                  end;
                  if ClipInsOK then
                  begin
                    PutClipboardProc(@Data^[I + 1], J - I);
                    if DelSel then DeleteSel;
                  end;
                end;
              kbShiftIns:
                begin
                  L := GetClipboardSizeProc;
                  if (L <= 512) and GetClipboardProc(@S[1], L) then
                  begin
                    DegranulateClipboardProc(@S[1], L);
                    if ConvertEOL then
                    begin
                      I := 1;
                      J := 0;
                      while I <= L do
                      begin
                        Ch := Chr(S[I]);
                        if Ch in [chCR, chLF] then
                        begin
                          Inc(J);
                          S[J] := Ord(' ');
                          while (I <= L) and (Ch in [chCR, chLF]) do
                          begin
                            Inc(I);
                            Ch := Chr(S[I]);
                          end;
                        end
                        else
                        begin
                          Inc(I);
                          Inc(J);
                          S[J] := Ord(Ch);
                        end;
                      end;
                    end
                    else
                    begin
                      J := L;
                    end;
                    if (J > 0) and (J <= 255) and (J + Length(Data^) <= MaxLen) then
                    begin
                      S[0] := J;
                      if First then DeleteAll;
                      Insert(PString(@S)^, Data^, CurPos + 1);
                      SelStart := CurPos;
                      Inc(CurPos, S[0]);
                      SelEnd := CurPos;
                      ChangedData := True;
                    end;
                  end;
                end;
            end;
          end;
          if (Quote and (Event.CharCode > #0)) or (Event.CharCode in [' '..#255]) then
          begin
            if CheckValid(True, False) then
            begin
              if First then DeleteAll;
              if CurPos < MaxLen then
              begin
                if Length(Data^) = MaxLen then Data^ := Copy(Data^, 1, MaxLen - 1);
                if FirstPos > CurPos then FirstPos := CurPos;
                if InsMode then
                begin
                  if CurPos = Length(Data^) then Inc(Data^[0]);
                  Data^[CurPos + 1] := Event.CharCode;
                end
                else
                begin
                  Insert(Event.CharCode, Data^, CurPos + 1);
                  MoveSelMarks(1);
                end;
                if not CursorStill then Inc(CurPos);
                ChangedData := True;
              end
              else
              begin
                GoSound := True;
              end;
              CheckValid(False, False);
            end;
          end;
          Quote := False;
        end;
    end;
    ScrollLine(16);
  end;
end;

function TInputLine.MouseInView(Mouse: TPoint): Boolean;
var
  Extent: TRect;

function ExtContains(const Rect: TRect; P: TPoint): Boolean; assembler;
asm
        LES     DI,Rect
        MOV     AL,0
        MOV     DX,P.X
        CMP     DX,ES:[DI].TRect.A.X
        JL      @@1
        CMP     DX,ES:[DI].TRect.B.X
        JG      @@1
        MOV     DX,P.Y
        CMP     DX,ES:[DI].TRect.A.Y
        JL      @@1
        CMP     DX,ES:[DI].TRect.B.Y
        JGE     @@1
        INC     AX
@@1:
end;

begin
  MakeLocal(Mouse, Mouse);
  GetExtent(Extent);
  MouseInView := ExtContains(Extent, Mouse);
end;

procedure TInputLine.SelectAll;
begin
  First := True;
  DrawView;
end;

procedure TInputLine.SetData(var Rec);
begin
  Move(Rec, Data^[0], MaxLen + 1);
  DrawView;
end;

procedure TInputLine.SetState(AState: Word; Enable: Boolean);
begin
  TView.SetState(AState, Enable);
  if AState = sfFocused then DrawView;
end;

procedure TInputLine.SetValidator(AValid: PValidator);
begin
  if Validator <> nil then Validator^.Free;
  Validator := AValid;
end;

function TInputLine.Valid(Command: Word): Boolean;
begin
  Valid := inherited Valid(Command);
  if (Validator <> nil) and (State and sfDisabled = 0) and
    (Command <> cmCancel) and not Validator^.Valid(Data^) then
    begin
      Select;
      Valid := False;
    end;
end;

{ TButton }

constructor TButton.Init(var Bounds: TRect; ATitle: TTitleStr; ACommand: Word);
begin
  Inc(Bounds.B.X, Bounds.A.X);
  Inc(Bounds.B.Y, Bounds.A.Y);
  TView.Init(Bounds);
  Options := Options or (ofSelectable + ofFirstClick +
    ofPreProcess + ofPostProcess);
  EventMask := EventMask or evBroadcast;
  if not CommandEnabled(ACommand) then State := State or sfDisabled;
  Title := NewStr(ATitle);
  Command := ACommand;
  ViewType := vtButton;
end;

destructor TButton.Done;
begin
  DisposeStr(Title);
  TView.Done;
end;

procedure TButton.Draw;
var
  CButton: Word;
  B: TDrawBuffer;
begin
  CButton := GetColor($0101);
  if State and (sfActive + sfSelected) = (sfActive + sfSelected) then CButton := GetColor($0202);
  MoveChar(B, ' ', Byte(CButton), Size.X);
  MoveCStr(B, Title^, CButton);
  WriteLine(0, 0, Size.X, 1, B);
end;

function TButton.GetPalette: PPalette;
const
  P: string[Length(CButton)] = CButton;
begin
  GetPalette := @P;
end;

procedure TButton.HandleEvent(var Event: TEvent);
var
  Down: Boolean;
  C: Char;
  Mouse: TPoint;
  ClickRect: TRect;
begin
  if (Event.What and evMouse > 0) and (Event.Buttons and mbLeftButton = 0) then
    ClearEvent(Event);
  GetExtent(ClickRect);
  if Event.What = evMouseDown then
  begin
    MakeLocal(Event.Where, Mouse);
    if not ClickRect.Contains(Mouse) then ClearEvent(Event);
  end;
  case Event.What of
    evMouseDown:
      begin
        if State and sfDisabled = 0 then
        begin
          Down := False;
          repeat
            MakeLocal(Event.Where, Mouse);
            if Down <> ClickRect.Contains(Mouse) then
            begin
              Select;
              Down := not Down;
            end;
          until not MouseEvent(Event, evMouseMove);
          if Down then Press;
        end;
        ClearEvent(Event);
      end;
    evKeyDown:
      begin
        if GetState(sfSelected) then
        begin
          case Event.KeyCode of
            kbEnter:
              begin
                Press;
                ClearEvent(Event);
              end;
            kbBack: Event.KeyCode := kbLeft;
            kbSpace: Event.KeyCode := kbRight;
          else
            C := HotKey(Title^);
            if (C <> #0) and (UpCase(Event.CharCode) = C) then
            begin
              Press;
              ClearEvent(Event);
            end;
          end;
        end;
        C := HotKey(Title^);
        if (C <> #0) and ((C = GetAltChar(Event.KeyCode)) or ((Owner^.Phase = phPostProcess) and (C <> #0) and
          (UpCase(Event.CharCode) = C))) then
        begin
          Press;
          ClearEvent(Event);
        end;
      end;
  end;
end;

procedure TButton.Press;
var
  E: TEvent;
begin
  Select;
  E.What := evCommand;
  E.Command := Command;
  PutEvent(E);
end;

procedure TButton.SetState(AState: Word; Enable: Boolean);
begin
  TView.SetState(AState, Enable);
  if AState and (sfSelected + sfActive) <> 0 then DrawView;
end;

{ TCluster }

constructor TCluster.Init(var Bounds: TRect; AStrings: PSItem);
var
  I: Integer;
  P: PSItem;
begin
  Inc(Bounds.B.X, Bounds.A.X);
  Inc(Bounds.B.Y, Bounds.A.Y);
  TView.Init(Bounds);
  Options := Options or (ofSelectable + ofFirstClick + ofPreProcess +
    ofPostProcess);
  I := 0;
  P := AStrings;
  while P <> nil do
  begin
    Inc(I);
    P := P^.Next;
  end;
  Strings.Init(I);
  while AStrings <> nil do
  begin
    P := AStrings;
    Strings.Insert(AStrings^.Value);
    AStrings := AStrings^.Next;
    Dispose(P);
  end;
  Value := 0;
  Sel := 0;
  SetCursor(1,0);
  ShowCursor;
  GroupedTab := False;
  ViewType := vtCluster;
end;

destructor TCluster.Done;
begin
  Strings.Done;
  TView.Done;
end;

procedure TCluster.DrawBox(const Icon: string; Marker: Char);
var
  I: Integer;
  Color: Word;
  B: TDrawBuffer;
begin
  Color := GetColor($0101);
  for I := 0 to Size.Y do
  begin
    MoveChar(B, ' ', Byte(Color), Size.X);
    if I < Strings.Count then
    begin
      MoveChar(B, ' ', Byte(Color), Size.X);
      MoveStr(B, Icon, Byte(Color));
      if Mark(I) then MoveChar(B[1], Marker, Color, 1);
      MoveCStr(B[4], PString(Strings.At(I))^, Color);
    end;
    WriteBuf(0, I, Size.X, 1, B);
  end;
  SetCursor(1, Sel);
end;

procedure TCluster.GetData(var Rec);
begin
  Word(Rec) := Value;
end;

function TCluster.GetPalette: PPalette;
const
  P: string[Length(CCluster)] = CCluster;
begin
  GetPalette := @P;
end;

procedure TCluster.HandleEvent(var Event: TEvent);
var
  Mouse: TPoint;
  I, S: Integer;
  C: Char;
  K: Word;

procedure MoveSel;
begin
  if S <= Strings.Count then
  begin
    Sel := S;
    DrawView;
  end;
end;

function FindSel(P: TPoint): Integer;
var
  R: TRect;
begin
  GetExtent(R);
  if not R.Contains(P) then FindSel := -1
    else FindSel := P.Y;
end;

begin
  if Event.What and evMouse > 0 then if Event.Buttons and mbLeftButton = 0 then ClearEvent(Event);
  TView.HandleEvent(Event);
  if (Options and ofSelectable) = 0 then Exit;
  if Event.What = evMouseDown then
  begin
    MakeLocal(Event.Where, Mouse);
    S := FindSel(Mouse);
    if S <> -1 then Sel := S;
    DrawView;
    repeat
      MakeLocal(Event.Where, Mouse);
      if FindSel(Mouse) = Sel then
        ShowCursor else
        HideCursor;
    until not MouseEvent(Event,evMouseMove); {Wait for mouse up}
    ShowCursor;
    MakeLocal(Event.Where, Mouse);
    if FindSel(Mouse) = Sel then
    begin
      Press(Sel, False);
      DrawView;
    end;
    ClearEvent(Event);
  end else if Event.What = evKeyDown then
  begin
    S := Sel;
    K := Event.KeyCode;
    if not GroupedTab then
    begin
      case K of
        kbShiftTab: K := kbUp;
        kbTab: K := kbDown;
      end;
    end;
    case K of
      kbUp:
        if State and sfFocused <> 0 then
        begin
          Dec(S);
          if S >= 0 then
          begin
            MoveSel;
            ClearEvent(Event);
          end;
        end;
      kbDown:
        if State and sfFocused <> 0 then
        begin
          Inc(S);
          if S < Strings.Count then
          begin
            MoveSel;
            ClearEvent(Event);
          end;
        end;
    else
      begin
        for I := 0 to Strings.Count-1 do
        begin
          C := HotKey(PString(Strings.At(I))^);
          if (C <> #0) and ((C = GetAltChar(Event.KeyCode)) or
            ((Owner^.Phase = phPostProcess) or (State and sfFocused <> 0))
            and (UpCase(Event.CharCode) = C)) then
          begin
            if Focus then
            begin
              Sel := I;
              Press(Sel, False);
              DrawView;
            end;
            ClearEvent(Event);
            Exit;
          end;
        end;
        if (Event.CharCode = ' ') and (State and sfFocused <> 0) then
        begin
          Press(Sel, True);
          DrawView;
          ClearEvent(Event);
        end;
      end
    end
  end;
end;

procedure TCluster.SetData(var Rec);
begin
  Value := Word(Rec);
  DrawView;
end;

procedure TCluster.SetState(AState: Word; Enable: Boolean);
begin
  TView.SetState(AState, Enable);
  if AState = sfFocused then DrawView;
end;

function TCluster.Mark(Item: Integer): Boolean;
begin
  Mark := False;
end;

procedure TCluster.Press(Item: Integer; Keyboard: Boolean);
begin
end;

{ TRadioButtons }

constructor TRadioButtons.Init(var Bounds: TRect; AStrings: PSItem);
begin
  TCluster.Init(Bounds, AStrings);
  GroupedTab := True;
end;

procedure TRadioButtons.Draw;
begin
  DrawBox('( )', #7);
end;

function TRadioButtons.Mark(Item: Integer): Boolean;
begin
  Mark := Item = Value;
end;

procedure TRadioButtons.Press(Item: Integer; Keyboard: Boolean);
begin
  if Keyboard and (Value = Item) then
  begin
    Value := (Item + 1) mod Size.Y;
    Sel := Value;
  end else Value := Item;
end;

procedure TRadioButtons.SetData(var Rec);
begin
  TCluster.SetData(Rec);
  Sel := Integer(Value);
end;

{ TCheckBoxes }

procedure TCheckBoxes.Draw;
begin
  DrawBox('[ ]', 'x');
end;

function TCheckBoxes.Mark(Item: Integer): Boolean;
begin
  Mark := Value and (1 shl Item) <> 0;
end;

procedure TCheckBoxes.Press(Item: Integer; Keyboard: Boolean);
begin
  Value := Value xor (1 shl Item);
end;

{ TStaticText }

constructor TStaticText.Init(var Bounds: TRect; const AText: string);
begin
  Inc(Bounds.B.X, Bounds.A.X);
  Inc(Bounds.B.Y, Bounds.A.Y);
  TView.Init(Bounds);
  Text := NewStr(AText);
end;

destructor TStaticText.Done;
begin
  DisposeStr(Text);
  TView.Done;
end;

procedure TStaticText.Draw;
var
  B: TDrawBuffer;
begin
  MoveChar(B, ' ', GetColor(1), Size.X);
  if Text <> nil then MoveStr(B, Text^, GetColor($0201));
  WriteBuf(0, 0, Size.X, 1, B);
end;

function TStaticText.GetPalette: PPalette;
const
  P: string[Length(CStaticText)] = CStaticText;
begin
  GetPalette := @P;
end;

{ TColorText }

procedure TColorText.Draw;
var
  B: TDrawBuffer;
begin
  MoveChar(B, ' ', GetColor(1), Size.X);
  if Text <> nil then MoveCStr(B, Text^, GetColor($0201));
  WriteBuf(0, 0, Size.X, 1, B);
end;

end.
