
{*************************************************}
{                 Joe Forster/STA                 }
{                                                 }
{                    DISKED.PAS                   }
{                                                 }
{       The Star Commander Disk editor unit       }
{*************************************************}

unit Disked;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O+,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Dialogs, Drivers, Menus, Objects, Views,
  Base1, Base2, Constant, Panel2;

const
{Reread block modes after exiting the BAM or error info editor}
  rrNone        = 0;
  rrView        = 1;
  rrEdit        = 2;
{Switch status strings}
  SwitchStr     : array [False..True] of string[3] = ('off', 'on');
{Shifted numbers}
  ShiftedNums   = '!@#$%^&*';
{Indents for the decimal mode of the edit field of the disk editor}
  DecIndent     : array [0..16] of Byte = (0, 4, 8, 12, 16, 20, 24, 28, 32, 36, 40, 44, 48, 52, 56, 60, 64);

type
{Sector read/write procedure}
  TSectorProc   = function(T, S: Byte; Save, Ask: Boolean; P: PBlock; PostProcess: Boolean): Boolean;
{Sector map procedure}
  TSectorMapProc= procedure(T, S: Byte);
{Error info buffer}
  TInfoBuffer   = array [0..MaxLocations - 1] of Byte;
  PInfoBuffer   = ^TInfoBuffer;
{Disk editor edit field}
  TEditInput    = object(TView)
    constructor Init(Bounds: TRect);
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Draw; virtual;
  end;
  PEditInput    = ^TEditInput;
{Disk editor command field}
  TCmdInput     = object(TInputLine)
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure SetCmd(const Text, MakePos: string; MakeLast: Boolean);
  end;
  PCmdInput     = ^TCmdInput;
{Disk editor dialog box}
  TDiskEditor   = object(TDialog)
    constructor Init(Bounds: TRect);
    procedure HandleEvent(var Event: TEvent); virtual;
  end;
  PDiskEditor   = ^TDiskEditor;
{Disk editor temporary buffers}
  TEditBuffer   = array [0..9] of TBlock;
  PEditBuffer   = ^TEditBuffer;
{Disk editor directory buffer}
  TEditDirBuf   = array [0..DirBufferLen - 1] of Byte;
  PEditDirBuf   = ^TEditDirBuf;

var
  DiskCopySelection,
  AutoRead,
  StoreLoc,
  ShowOwner,
  CopyShowOwner,
  BAMOK,
  MapEdit,
  BAMEdit,
  EditAllowed,
  ViewDir,
  EditDirEnd    : Boolean;
  ErrorCode,
  EditViewMode,
  EditLStart,
  EditLEnd,
  EditStart,
  EditEnd,
  EditDirSector,
  EditDirNum,
  EditDirLine,
  EditTrack,
  EditSector,
  EditPos,
  RereadBlock,
  OrigTrack,
  OrigSector,
  LastTrack,
  LastSector    : Byte;
  EditClipLen   : Word;
  EditMapDelta,
  EditMapHeight,
  EditMapIndPosX,
  EditMapIndPosY,
  EditMapIndent,
  EditMapDensity: Integer;
  DiskEditor    : PDiskEditor;
  EditTitle     : PTitle;
  InpTrack,
  InpSector,
  InpPos        : PNumInput;
  InpStatus,
  InpOffset,
  InpCommand,
  InpHistory    : PCmdInput;
  InpEdit       : PEditInput;
  InfoBuffer,
  InfoBuffer2   : PInfoBuffer;
  Locations     : PLocBuffer;
  Locations2    : PLocBuffer;
  LocationSet   : PLocationSet;
  EditBuffer    : PEditBuffer;
  EditDirBuffer : PEditDirBuf;
  LastCmd,
  CmdText       : string[80];

function ScanOwners(Turbo: Boolean): Boolean;
function DiskEdit(Z: Byte; CallEditor, BAMEditor, AllowEdit: Boolean): Boolean;

implementation

uses
  App, DOS,
  ExtFiles, FWPrep, LowLevel, Panel1, Script;

{Determine whether a sector is allocated or free in the BAM
  Input : T, S: the location of the sector
  Output: when True, the sector is allocated; otherwise free}
function IsBlockUsed(T, S: Byte): Boolean;
begin
  if DiskCopySelection then IsBlockUsed := ManualSelect^.Contains(Act^.DiskPos(T, S)) else
    IsBlockUsed := Act^.IsBlockUsed(T, S);
end;

{Allocate or free a sector in the BAM
  Input : T, S: the location of the sector
  Output: when True, the sector is allocated; otherwise freed}
procedure AllocBlock(T, S: Byte; Alloc: Boolean);
begin
  if DiskCopySelection then
  begin
    if Alloc then ManualSelect^.Include(Act^.DiskPos(T, S)) else ManualSelect^.Exclude(Act^.DiskPos(T, S));
  end
  else
  begin
    Act^.AllocBlock(T, S, Alloc);
  end;
end;

{Save the current location into the location buffer}
procedure SaveLoc;
var
  I             : Integer;
begin
  if StoreLoc and ((LastLoc = 0) or (EditTrack <> Locations^[LastLoc - 1].Track)
    or (EditSector <> Locations^[LastLoc - 1].Sector)) then
  begin
    if LastLoc = MaxLocations - 1 then
    begin
      for I := 1 to LastLoc do Locations^[I - 1] := Locations^[I];
      if CurLoc > 0 then Dec(CurLoc);
    end;
    Locations^[LastLoc].Track := EditTrack;
    Locations^[LastLoc].Sector := EditSector;
    if LastLoc < MaxLocations - 1 then Inc(LastLoc);
    CurLoc := LastLoc - 1;
  end;
end;

{Save the current location as the previous location}
procedure SaveLast;
var
  I             : Integer;
begin
  LastTrack := EditTrack;
  LastSector := EditSector;
  SaveLoc;
end;

{Display the disk editor title
  Input : Name: the name of the file the current sector belongs to; not
                displayed, if empty}
procedure MakeTitle(const Name: string);
var
  B             : Byte;
  S             : string;
begin
  FillChar(S[1], ScreenWidth, ' ');
  B := ScreenWidth - 1;
  if Name <> '' then Dec(B, ScreenWidth shr 1);
  S := BoxTitle + ': ' + LimitNameLen(Act^.CopyImageName, B - Length(BoxTitle) - 2);
  S[0] := Chr(B);
  if Name <> '' then S := S + stSpace + Name;
  S[0] := Chr(ScreenWidth);
  Move(S, EditTitle^.Text^, Length(S) + 1);
end;

{Display the disk editor header: track, sector and position}
procedure DisplayHeader;
var
  O             : Boolean;
  A,
  S,
  T             : Byte;
  F,
  W,
  X,
  Z             : Word;
  Y             : PString;
  U,
  V             : string;
begin
  U := '';
  if CopyShowOwner then
  begin
    StoreLoc := False;
    LocationSet^.Init(False);
    CurLoc := Act^.DiskPos(EditTrack, EditSector);
    if (Locations2^[CurLoc].Track = 0) and (Locations^[CurLoc].Track = MaxByte) then
    begin
      T := MaxByte;
    end
    else
    begin
      T := EditTrack;
      S := EditSector;
      repeat
        CurLoc := Act^.DiskPos(T, S);
        T := Locations2^[CurLoc].Track;
        S := Locations2^[CurLoc].Sector;
        LocationSet^.Include(CurLoc);
      until (T = 0) or not Act^.ValidPos(T, S);
      X := 0;
      T := EditTrack;
      S := EditSector;
      repeat
        if T and $80 = 0 then
        begin
          Inc(X);
          CurLoc := Act^.DiskPos(T, S);
          T := Locations^[CurLoc].Track;
          S := Locations^[CurLoc].Sector;
          LocationSet^.Include(CurLoc);
        end;
      until T and $80 > 0;
    end;
    if T = MaxByte then
    begin
      U := stSpace;
    end
    else
    begin
      O := True;
      W := (T and $3F) shl 8 or S;
      if W = 0 then
      begin
        F := FirstDirSec(Act^.CopyDiskType);
        if X > F then
        begin
          Dec(X, F);
          U := 'the directory';
        end
        else
        begin
          if F > 1 then
          begin
            U := 'the BAM';
          end
          else
          begin
            U := 'BAM sector';
            O := False;
          end;
        end;
      end
      else
      begin
        if Act^.CopyMode = pmExt then Dec(W);
        if (Act^.Mode = pmDOS) or (Act^.GetSortMode <> psUnsorted) then
        begin
          U := '';
          Y := nil;
          O := (Act^.OpenImage(False, False, False, True, True) = 0);
          Z := W;
          if O then
          begin
            repeat
              Act^.ReadCBMEntry(Entry);
              if Entry.Attr > 0 then Dec(Z);
            until Z = 0;
            A := Entry.ExtAttr;
            Y := @Entry.Name;
            Act^.CloseImage(False);
          end;
        end
        else
        begin
          A := Act^.Dir[W].ExtAttr;
          Y := Act^.GetNamePtr(W);
        end;
        if Y <> nil then U := MakeCBMName('"' + Y^ + '"', Act^.CopyGEOSFormat);
      end;
      if T and $40 = 0 then
      begin
        V := 'Block ' + LeadingSpace(X, 4);
      end
      else
      begin
        if W = 0 then
        begin
          U := 'GEOS border sector';
          O := False;
        end
        else
        begin
          if A = 0 then V := 'Side sector ' + LeadingSpace(X, 3) else V := 'Info block';
        end;
      end;
      if O then U := V + ' of ' + U;
    end;
  end;
  MakeTitle(U);
  EditTitle^.DrawView;
  U := LeadingSpace(EditTrack, 0);
  InpTrack^.SetData(U);
  U := LeadingSpace(EditSector, 0);
  InpSector^.SetData(U);
  U := HexaPrefix + HexaStr(Act^.DiskPos(EditTrack, EditSector) shl 8, 8);
  InpOffset^.SetData(U);
  if HexaMode then U := HexaPrefix + HexaStr(EditPos, 2) else U := LeadingZero(EditPos, 3);
  InpPos^.SetData(U);
end;

{Display the allocation status of the current block}
procedure DisplayAlloc;
var
  S             : string[10];
begin
  if BAMOK then
  begin
    if IsBlockUsed(EditTrack, EditSector) then S := 'Used' else S := 'Free';
  end
  else
  begin
    S := '';
  end;
  InpStatus^.SetData(S);
end;

{Input the track and sector number
  Output: when False, the input was cancelled}
function GetTS: Boolean;
var
  O,
  Q             : Boolean;
  T,
  S,
  M             : Byte;
  C             : Word;
  L             : Longint;
  D             : string[10];
begin
  O := False;
  Q := False;
  while not Q do
  begin
    InpTrack^.CurPos := 0;
    InpTrack^.First := True;
    InpTrack^.Select;
    InpTrack^.ShowCursor;
    repeat
      C := Application^.ExecView(DiskEditor, True, False);
      InpTrack^.GetData(D);
      Val(D, T, NumOK);
    until (C <> cmOK) or (T in [1..Act^.CopyMaxTrack - 1]);
    if C = cmOK then
    begin
      InpSector^.CurPos := 0;
      InpSector^.First := True;
      InpSector^.Select;
      InpSector^.ShowCursor;
      M := Act^.SectorNum(T);
      repeat
        C := Application^.ExecView(DiskEditor, True, False);
        InpSector^.GetData(D);
        Val(D, S, NumOK);
      until (C <> cmOK) or (S in [0..M - 1]);
      if C = cmOK then
      begin
        O := True;
        Q := True;
        SaveLast;
        EditTrack := T;
        EditSector := S;
      end;
    end
    else
    begin
      Q := True;
    end;
  end;
  DisplayHeader;
  GetTS := O;
end;

{Turn the cursor of all input field off}
procedure CursorOff;
begin
  InpTrack^.HideCursor;
  InpSector^.HideCursor;
  InpPos^.HideCursor;
  InpCommand^.HideCursor;
end;

{Position the cursor in the edit field}
procedure DPosCursor;
var
  X,
  Y,
  Z             : Integer;
begin
  if MapEdit then
  begin
    Z := EditMapHeight - 1;
    if (EditSector >= Z) and (EditMapDelta < Z) then EditMapDelta := Z;
    if (EditSector < Z) and (EditMapDelta >= Z) then EditMapDelta := 0;
    X := EditTrack shl EditMapDensity + EditMapIndent;
    Y := EditSector + 2 - EditMapDelta;
  end
  else
  begin
    if TextMode then X := (EditPos and $0F) + 64 else if HexaMode then X := HexIndent[EditPos and $0F] + 10 + EditChar else
      X := DecIndent[EditPos and $0F] + EditChar;
    Y := (EditPos shr 4) + 6;
  end;
  InpEdit^.SetCursor(X, Y);
  if InsertMode then InpEdit^.BlockCursor else InpEdit^.NormalCursor;
  DisplayHeader;
  InpEdit^.DrawView;
end;

{Save the contents of the edit buffer into the undo buffer}
procedure SaveBlock;
begin
  Move(DataBuffer, UndoBuffer, 256);
  BufferChanged := False;
end;

{Compare the contents of the edit buffer with the undo buffer}
procedure CompareBlock;
var
  B             : Byte;
begin
  B := 0;
  BufferChanged := False;
  repeat
    BufferChanged := (UndoBuffer[B] <> DataBuffer[B]);
    Inc(B);
  until BufferChanged or (B = 0);
end;

{Swap the contents of the edit buffer with the specified temporary
  buffer
  Input : Z: buffer number to be displayed
          P: temporary buffer to be swapped with the edit buffer}
procedure SwapBuffer(Z: Byte; P: PBlock);
var
  B,
  D             : Byte;
begin
  if Z < 10 then InpCommand^.SetCmd('Swap block with buffer #' + Chr(Z + Ord('0')), stEmpty, False);
  for B := 0 to 255 do
  begin
    D := DataBuffer[B];
    DataBuffer[B] := P^[B];
    P^[B] := D;
  end;
  CompareBlock;
  DiskEditor^.DrawView;
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
end;

{Send the disk editor control program to the external CBM drive}
procedure SendTurbo;
begin
  SendConfigData;
  OpenCBMChannel(saCommand, 'I0', True);
  if SendDriveProg(deTurboDiskEdit, True) then ExecDriveProg(deTurboDiskEdit, stEmpty);
  ShutDownAsync;
  asm
    mov ax, 5000;
    call Delay;
  end;
end;

{Shut down the disk editor control program in the external CBM drive}
procedure ShutDownTurbo;
begin
  MouseOff;
  asm
    call ParallelOutput;
    mov al, 'Q';
    call TSend;
    mov ax, 50;
    call Delay;
    call ParallelInput;
  end;
  TurboOff;
end;

{Create an error message on basis of the status returned by the external CBM
  drive
  Input : E: status code
          S: string to contain the error message
  Output: when True, the status code tells that an error occured}
function MakeError(E: Byte; var S: string): Boolean;
var
  O             : Boolean;
begin
  O := True;
  if (E < 16) then
  begin
    if not MakeErrorStr(E, EditTrack, EditSector, S) then E := 1;
  end
  else
  begin
    E := 0;
    S := 'Timeout detected';
  end;
  MakeError := (E <> 1);
end;

{Change the help context while the function is being executed
  Input : P: procedure to be executed
          B: when True, the function key bar is updated to reflect the
             help context changes}
procedure DiskEditHelp(P: TProc; B: Boolean);
var
  O             : Boolean;
  H             : Word;
begin
  H := hcDiskEdit2;
  O := HelpCtxSet;
  HelpCtxSet := True;
  CurHelpCtx := hcKeyDiskEdit;
  AppHelpCtx := hcKeyDiskEdit;
  LastShiftState := MaxByte;
  if B then KeyBar^.Update;
  P;
  CurHelpCtx := H;
  AppHelpCtx := H;
  HelpCtxSet := O;
  LastShiftState := MaxByte;
  if B then KeyBar^.Update;
end;

{Read a block into the edit buffer
  Input : T, S: track and sector number
          Save: when True, the track and sector number is saved into the
                location buffer
          Ask: dummy parameter, for compatibility with the type SectorProc
          P: buffer to read the sector to
          PostProcess: dummy value
  Output: when False, an error occurred}
function ReadSector(T, S: Byte; Save, Ask: Boolean; P: PBlock; PostProcess: Boolean): Boolean; far;
var
  O,
  Q             : Boolean;
  E             : Byte;
  N             : string;
  X             : TBlock;
begin
  ClockOff;
  CursorOff;
  DisplayHeader;
  O := False;
  Q := False;
  case Act^.CopyMode of
    pmExt:
    begin
      MouseOff;
      case CopyTransferMode of
        tmNormal:
        begin
          ReadExtBlock(T, S, @X);
          O := ReadCBMError(N, False, False, True);
          if not O then ErrorWin(stError, N, stEmpty, CurHelpCtx, (sbSkip + sbSkipAll));
        end;
        tmTurbo, tmWarp:
        begin
          while not Q do
          begin
            asm
              push word ptr CopyPriorityMode;
              call InterruptOff;
              call ParallelOutput;
              mov al, 'R';
              call TSend;
              mov al, S;
              call TSend;
              mov al, T;
              call TSend;
              cmp Status, 0;
              je @1;
              mov E, 16;
              jmp @3;
          @1: mov ax, 50;
              call Delay;
              call ParallelInput;
              call TReceive;
              cmp Status, 0;
              jne @3;
              mov E, al;
              cmp al, 'O';
              jne @3;
              push ss;
              pop es;
              lea di, X;
              mov si, 256;
          @2: call TReceive;
              mov es:[di], al;
              inc di;
              dec si;
              jne @2;
          @3: call InterruptOn;
            end;
            ShutDownAsync;
            if (E = Ord('O')) or (E = 16) then
            begin
              O := True;
            end
            else
            begin
              MakeError(E, N);
              ErrorWin(stError, N, stEmpty, CurHelpCtx, (sbSkip + sbSkipAll));
            end;
            Q := (E <> 16);
            if not Q then SendTurbo;
          end;
        end;
      end;
      MouseOn;
    end;
    pmDisk, pmDiskZip:
    begin
      if (Act^.CopyMode = pmDiskZip) or (LongOpenFile(Act^.CopyImageName, Act^.Image, fmReadOnly) = 0) then
      begin
        Act^.ReadDiskBlock(T, S, @X, True);
        ExtClose(Act^.Image);
        O := True;
      end;
    end;
  end;
  if O then Move(X, P^, 256) else FillFormatPattern(P);
  EditPos := 0;
  EditChar := 0;
  EditMarking := False;
  EditMarked := False;
  if Save then
  begin
    BufferChanged := False;
    SaveBlock;
    SaveLoc;
  end;
  DisplayHeader;
  if BufferOK then DisplayAlloc;
  DiskEditor^.DrawView;
  ClockOn;
  ReadSector := O;
end;

{Process the BAM of the current disk or disk image
  Input : Proc: the procedure to process the BAM with}
procedure ProcessBAM(Proc: TSectorProc);
begin
  case Act^.CopyDiskType and dtTypeMask of
    dt1541, dt1541Ext: Proc(Act^.DirTrack, 0, False, False, @Act^.BAM, True);
    dt1571:
    begin
      Proc(Act^.DirTrack, 0, False, False, @Act^.BAM, False);
      Proc(Act^.DirTrack2, 0, False, False, @Act^.BAM2, True);
    end;
    dt1581:
    begin
      Proc(Act^.DirTrack, 0, False, False, @Act^.BAM, False);
      Proc(Act^.DirTrack, 1, False, False, @Act^.BAM2, False);
      Proc(Act^.DirTrack, 2, False, False, @Act^.BAM3, True);
    end;
  end;
end;

{Read the BAM into the BAM buffer}
procedure ReadBAM;
begin
  FillChar(Act^.BAM, 3 * 256, 0);
  ProcessBAM(ReadSector);
  BAMOK := Act^.IsBAMValid;
  if (EditTrack = Act^.DirTrack) and (EditSector = 0) then
  begin
    Move(Act^.BAM, DataBuffer, 256);
    SaveBlock;
    DiskEditor^.DrawView;
  end;
end;

{Create the database for the owners of blocks
  Input : Turbo: when True, the disk editor control program is shut down before
                 the disk scan and then executed again
  Output: when False, an error occured}
function ScanOwners(Turbo: Boolean): Boolean;
var
  OrigShowReadErrors,
  O,
  Q,
  Z             : Boolean;
  E,
  F,
  M,
  N,
  P,
  S,
  T             : Byte;
  I,
  J,
  K,
  X             : Word;
  B,
  C             : TBlock;

procedure Trace;
begin
  T := Hi(I) or $80;
  if O then T := T or $40;
  S := Lo(I);
  X := 0;
  if Z then J := Entry.Size;
  Q := False;
  O := not O and (Entry.ExtAttr and xaGEOSVLIR > 0);
  repeat
    CurLoc := Act^.DiskPos(Entry.Track, Entry.Sector);
    if Z then
    begin
      M := Entry.Track;
      N := Entry.Sector + 1;
      if N >= Act^.SectorNum(M) then
      begin
        Inc(M);
        N := 0;
      end;
      Dec(J);
      if J = 0 then M := 0;
    end
    else
    begin
      if Act^.CopyMode = pmExt then
      begin
        O := False;
        J := CurLoc shl 1;
        M := TempBuffer[J];
        N := TempBuffer[J + 1];
      end
      else
      begin
        Act^.ReadDiskBlock(Entry.Track, Entry.Sector, @B, True);
        M := B[0];
        N := B[1];
      end;
      if O and (X = 0) then Move(B, C, 256);
    end;
    Q := (M = 0);
    if not Q then
    begin
      K := Act^.DiskPos(M, N);
      Q := ((Entry.Track = M) and (Entry.Sector = N)) or (Locations^[K].Track <> MaxByte) or (Locations^[K].Sector <> MaxByte);
      if Q then M := 0;
    end;
    Locations^[CurLoc].Track := T;
    Locations^[CurLoc].Sector := S;
    T := Entry.Track;
    S := Entry.Sector;
    Entry.Track := M;
    Entry.Sector := N;
    if Z then
    begin
      Q := (M = 0);
    end
    else
    begin
      Q := not Act^.ValidPos(Entry.Track, Entry.Sector);
      if not Q then
      begin
        if (Entry.Track = 0) then
        begin
          if O then
          begin
            if X = 0 then P := 0;
            repeat
              Inc(P, 2);
              Q := (P = 0);
              if not Q then
              begin
                Entry.Track := C[P];
                Entry.Sector := C[P + 1];
              end;
            until (Entry.Track > 0) or Q;
          end
          else
          begin
            Q := True;
          end;
        end;
      end;
    end;
    if not Q then
    begin
      Locations2^[CurLoc].Track := Entry.Track;
      Locations2^[CurLoc].Sector := Entry.Sector;
      Escape;
      Inc(X);
    end;
  until EscPressed or Q;
end;

begin
  OrigShowReadErrors := ShowReadErrors;
  ShowReadErrors := False;
  FillChar(Locations^, MaxLocations shl 1, MaxByte);
  FillChar(Locations2^, MaxLocations shl 1, 0);
  O := True;
  if Act^.CopyMode = pmExt then
  begin
    if Turbo then ShutDownTurbo;
    O := SendDriveProg(deWarpDiskValidate, True) and ExecDriveProg(deWarpDiskValidate, stEmpty);
    if O then
    begin
      FillChar(TempBuffer, TempBufferSize, 0);
      for T := 1 to Act^.CopyMaxTrack - 1 do
      begin
        F := Act^.SectorNum(T);
        X := Act^.DiskPos(T, 0) shl 1;
        asm
          push bp;
          call ParallelOutput;
          mov al, T;
          call TSend;
          mov ax, 50;
          call Delay;
          mov al, F;
          xor ah, ah;
          mov si, ax;
          mov di, Offset(TempBuffer);
          add di, X;
          call ParallelInput;
      @1: call TReceive;
          cmp Status, 0;
          jne @2;
          xor ah, ah;
          mov bp, ax;
          shl bp, 1;
          call TReceive;
          mov ds:[di][bp], al;
          inc bp;
          call TReceive;
          mov ds:[di][bp], al;
          dec si;
          jne @1;
      @2: pop bp;
        end;
      end;
      TurboOff;
    end;
  end;
  O := O and (Act^.OpenImage(False, False, True, True, True) = 0);
  if O then
  begin
    Escape;
    I := 1;
    while not EscPressed and Act^.ReadCBMEntry(Entry) do
    begin
      if Entry.Attr > 0 then
      begin
        Z := ((Act^.CopyDiskType and dtTypeMask = dt1581) and (Entry.Attr and faTypeMask = faPartition));
        O := False;
        Trace;
        Entry.Track := Entry.SideTrack;
        Entry.Sector := Entry.SideSector;
        if not Z and (Entry.Track > 0) and Act^.ValidPos(Entry.Track, Entry.Sector) then
        begin
          O := True;
          Trace;
        end;
        Inc(I);
      end;
    end;
    if not EscPressed then
    begin
      E := Act^.DirTrack;
      F := FirstDirSec(Act^.CopyDiskType);
      Entry.Track := E;
      Entry.Sector := F;
      Entry.ExtAttr := 0;
      I := 0;
      O := False;
      Z := False;
      Trace;
      for S := 1 to F do
      begin
        CurLoc := Act^.DiskPos(E, S);
        Locations^[CurLoc].Track := E;
        Locations^[CurLoc].Sector := S - 1;
        Locations2^[CurLoc - 1].Track := E;
        Locations2^[CurLoc - 1].Sector := S;
      end;
      CurLoc := Act^.DiskPos(E, 0);
      Locations^[CurLoc].Track := $80;
      Locations^[CurLoc].Sector := 0;
      Locations2^[CurLoc].Track := E;
      Locations2^[CurLoc].Sector := 1;
      if Act^.CopyGEOSFormat then
      begin
        CurLoc := Act^.DiskPos(Act^.BAM[GEOSBorderPos], Act^.BAM[GEOSBorderPos + 1]);
        Locations^[CurLoc].Track := $C0;
        Locations^[CurLoc].Sector := 0;
      end;
    end;
    O := not EscPressed;
    Act^.CloseImage(False);
  end;
  if (Act^.CopyMode = pmExt) and Turbo then SendTurbo;
  MouseOn;
  ShowReadErrors := OrigShowReadErrors;
  ScanOwners := O;
end;

{Create the database for the owners of blocks
  Input : Turbo: when True, the disk editor control program is shut down before
                 the disk scan and then executed again}
procedure MakeOwner(Turbo: Boolean);
var
  O             : Boolean;
  D             : PDialog;
begin
  if CopyShowOwner then
  begin
    InpCommand^.HideCursor;
    D := InfoWin(stEmpty, 'Scanning the disk,', 'please wait...', stEmpty, 2);
    O := ScanOwners(Turbo);
    Dispose(D, Done);
    ErrorDown := 0;
    if Turbo then InpCommand^.ShowCursor;
  end;
  if O then LastLoc := MaxLocations else CopyShowOwner := False;
  DisplayHeader;
end;

{Write the contents of the edit buffer into a block
  Input : T, S: track and sector number
          Save: when True, the track and sector number is saved into the
                location buffer
          Ask: when True and disk editor confirmations are on, the user is
               asked for confirmation about writing the sector
          P: buffer to write the sector from
  Output: when False, an error occurred}
function WriteSector(T, S: Byte; Save, Ask: Boolean; P: PBlock; PostProcess: Boolean): Boolean; far;
var
  O,
  Q             : Boolean;
  E             : Byte;
  W             : Word;
  X             : PBlock;
  N             : string;
begin
  O := False;
  CursorOff;
  if (not DiskEditConfirm or not Ask) or (Confirm(stEmpty, 'Do you wish to write to', 'the block at ' + LeadingZero(T, 2) +
    ';' + LeadingZero(S, 2), stWrite, stEmpty, stEmpty, nil, CurHelpCtx, True, DummyByte) = cmOK) then
  begin
    ClockOff;
    DiskChanged := True;
    DisplayHeader;
    Q := False;
    case Act^.CopyMode of
      pmExt:
      begin
        MouseOff;
        case CopyTransferMode of
          tmNormal:
          begin
            WriteExtBlock(T, S, P);
            O := ReadCBMError(N, False, False, True);
            if not O then ErrorWin(stError, N, stEmpty, CurHelpCtx, (sbSkip + sbSkipAll));
          end;
          tmTurbo, tmWarp:
          begin
            while not Q do
            begin
              asm
                push word ptr CopyPriorityMode;
                call InterruptOff;
                call ParallelOutput;
                mov al, 'W';
                call TSend;
                mov al, S;
                call TSend;
                mov al, T;
                call TSend;
                cmp Status, 0;
                je @1;
                mov E, 16;
                jmp @3;
            @1: les si, P;
                mov di, 256;
            @2: mov al, es:[si];
                call TSend;
                inc si;
                dec di;
                jne @2;
                mov ax, 50;
                call Delay;
                call ParallelInput;
                call TReceive;
                mov E, al;
            @3: call ParallelInput;
                call InterruptOn;
              end;
              ShutDownAsync;
              if (E = Ord('O')) or (E = 16) then
              begin
                O := True;
              end
              else
              begin
                MakeError(E, N);
                ErrorWin(stError, N, stEmpty, CurHelpCtx, (sbSkip + sbSkipAll));
              end;
              Q := (E <> 16);
              if not Q then SendTurbo;
            end;
          end;
        end;
        MouseOn;
      end;
      pmDisk, pmDiskZip:
      begin
        LongGetFAttr(Act^.CopyImageName, W);
        if W and ReadOnly > 0 then
        begin
          ErrorWin(stEmpty, 'The following image file is marked read-only.', Act^.CopyImageName, CurHelpCtx, sbNone);
        end
        else
        begin
          if Act^.CopyMode = pmDisk then
          begin
            if LongOpenFile(Act^.CopyImageName, Act^.Image, fmReadWrite) = 0 then
            begin
              Act^.WriteDiskBlock(T, S, P);
              if KeepTime then ExtSetFTime(Act^.Image, ReadTime);
              ExtClose(Act^.Image);
              O := True;
            end;
          end
          else
          begin
            ErrorWin(stEmpty, 'Can''t write into a ' + MakeFullTypeStr(Act^.CopyMode) + stDot,
              Act^.CopyImageName, CurHelpCtx, sbNone);
          end;
        end;
      end;
    end;
    DiskEditor^.DrawView;
    if Save then SaveLoc;
    if O and PostProcess then
    begin
      SaveBlock;
      X := nil;
      E := Act^.CopyDiskType and dtTypeMask;
      if T = Act^.DirTrack then
      begin
        if S = 0 then
        begin
          X := @Act^.BAM;
        end
        else
        begin
          if E = dt1581 then
          begin
            case S of
              1: X := @Act^.BAM2;
              2: X := @Act^.BAM3;
            end;
          end;
        end;
      end;
      if (T = Act^.DirTrack2) and (E = dt1571) and (S = 0) then X := @Act^.BAM2;
      if X <> nil then
      begin
        Move(P^, X^, 256);
        BAMOK := Act^.IsBAMValid;
        DisplayAlloc;
      end;
      if CopyShowOwner then MakeOwner(True);
    end;
  end;
  ClockOn;
  WriteSector := O;
end;

{Save the BAM or error info back to the disk or disk image}
procedure DSaveMap;
var
  O             : Boolean;
begin
  if BAMEdit then
  begin
    if BAMOK then
    begin
      InpEdit^.HideCursor;
      ProcessBAM(WriteSector);
      InpEdit^.ShowCursor;
    end;
  end
  else
  begin
    InpEdit^.HideCursor;
    if not DiskEditConfirm or (Confirm(stEmpty, 'Do you wish to write the error info to', Act^.CopyImageName, stWrite,
      stEmpty, stEmpty, nil, CurHelpCtx, True, DummyByte) = cmOK) then
    begin
      if LongOpenFile(Act^.CopyImageName, ReadFile, fmReadWrite) = 0 then
      begin
        ExtSeek(ReadFile, Act^.CopyDiskSize shl 8);
        ExtBlockWrite(ReadFile, InfoBuffer^, Act^.CopyDiskSize);
        if KeepTime then ExtSetFTime(ReadFile, ReadTime);
        ExtClose(ReadFile);
        DiskChanged := True;
        if IOResult = 0 then
        begin
          Act^.CopyDiskType := Act^.CopyDiskType or dtErrorInfo;
          Act^.CheckDiskType;
          BufferChanged := False;
        end;
      end;
    end;
    InpEdit^.ShowCursor;
  end;
end;

{Confirm reading a new sector if a change has been made to the edit buffer}
function NewSector: Boolean;
var
  B             : Boolean;
begin
  InpCommand^.HideCursor;
  InpEdit^.HideCursor;
  B := DiskCopySelection or not BufferChanged or not DiskEditConfirm;
  if not B then
  begin
    case SureConfirm(stEmpty, 'You''ve made changes since the last save.', stEmpty, stEmpty, stEmpty,
      stSave, stEmpty, stEmpty, ' '+ColorChar+'D'+ColorChar+'on''t save ', stEmpty, nil, CurHelpCtx,
      ayNone, False, DummyByte) of
      cmOK:
      begin
        if MapEdit then DSaveMap else WriteSector(EditTrack, EditSector, True, False, @DataBuffer, True);
        B := True;
      end;
      cmSkip: B := True;
      cmCancel: B := False;
    end;
  end;
  InpCommand^.ShowCursor;
  InpEdit^.ShowCursor;
  NewSector := B;
end;

{Automatically read the block if the track or sector number has changed}
procedure AutoReadSector;
begin
  if AutoRead then ReadSector(EditTrack, EditSector, True, False, @DataBuffer, True) else DisplayHeader;
end;

{Process sectors of the same file
  Input : SectorMapProc: the procedure to call for each sector}
procedure ProcessCurrentFile(SectorMapProc: TSectorMapProc);
var
  S,
  T             : Byte;
begin
  if CopyShowOwner then
  begin
    for T := 1 to Act^.CopyMaxTrack - 1 do
      for S := 0 to Act^.SectorNum(T) - 1 do
        if LocationSet^.Contains(Act^.DiskPos(T, S)) then SectorMapProc(T, S);
    InpEdit^.DrawView;
  end;
end;

{Free the sectors of the current file in the BAM editor or delete the error
  info from the disk image in the error info editor}
procedure DDeleteMap;
var
  O             : Boolean;
begin
  if not BAMEdit then
  begin
    if Act^.CopyDiskType and dtErrorInfo > 0 then
    begin
      InpEdit^.HideCursor;
      if not DiskEditConfirm or (Confirm(stEmpty, 'Do you wish to delete the error info from', Act^.CopyImageName,
        ' '+ColorChar+'S'+ColorChar+'trip ', stEmpty, stEmpty, nil, CurHelpCtx, True, DummyByte) = cmOK) then
      begin
        if LongOpenFile(Act^.CopyImageName, ReadFile, fmReadWrite) = 0 then
        begin
          ExtSeek(ReadFile, Act^.CopyDiskSize shl 8);
          ExtTruncate(ReadFile);
          if KeepTime then ExtSetFTime(ReadFile, ReadTime);
          ExtClose(ReadFile);
          DiskChanged := True;
          if IOResult = 0 then
          begin
            Act^.CopyDiskType := Act^.CopyDiskType and dtTypeMask;
            Act^.CheckDiskType;
            BufferChanged := False;
          end;
        end;
      end;
      InpEdit^.ShowCursor;
    end;
  end;
end;

procedure InitDisplayPos;
begin
  InpTrack^.MoveTo(1, 1);
  InpSector^.MoveTo(12, 1);
end;

{Edit the BAM or error info
  Input : EditBAM: when True, the BAM is edited, otherwise the error info
                   block
  Output: the command with which the editor was left}
function EditMap(EditBAM: Boolean): Word;
var
  B,
  O             : Boolean;
  U             : Byte;
  H             : Word;
  E             : TEvent;
begin
  if not DiskCopySelection then
  begin
    InpCommand^.HideCursor;
    H := CurHelpCtx;
    O := HelpCtxSet;
    HelpCtxSet := True;
    CurHelpCtx := hcDiskEdit;
    AppHelpCtx := hcDiskEdit;
    LastShiftState := MaxByte;
    KeyBar^.Update;
  end;
  B := False;
  if not DiskCopySelection then
  begin
    FillChar(InfoBuffer^, Act^.CopyDiskSize, dsOK);
    if Act^.CopyDiskType and dtErrorInfo > 0 then
    begin
      if LongOpenFile(Act^.CopyImageName, ReadFile, fmReadOnly) = 0 then
      begin
        ExtSeek(ReadFile, Act^.CopyDiskSize shl 8);
        ExtBlockRead(ReadFile, InfoBuffer^, Act^.CopyDiskSize);
        ExtClose(ReadFile);
      end;
    end;
  end;
  if EditBAM then
  begin
    if (EditTrack <> Act^.DirTrack) or (EditSector <> 0) or NewSector then
      B := (BAMOK or (SureConfirm(stEmpty, 'The BAM of this disk is invalid.', 'Do you wish to view it?',
      stEmpty, stEmpty, stYes, stEmpty, stEmpty, stEmpty, stNo, nil, CurHelpCtx, ayNone, False, DummyByte) = cmOK));
  end
  else
  begin
    if Act^.CopyDiskType and dtErrorInfo > 0 then
    begin
      B := True;
    end
    else
    begin
      if not DiskCopySelection and (Act^.CopyMode <> pmDisk) then
      begin
        MakeSound := False;
        ErrorWin(stEmpty, 'Can''t make error info on', Act^.CopyImageName, CurHelpCtx, sbNone);
      end
      else
      begin
        BAMEdit := False;
        B := True;
      end;
    end;
  end;
  if B then
  begin
    if EditBAM and not BAMOk then MenuBar^.DisableCommands([cmDSaveMap]) else MenuBar^.EnableCommands([cmDSaveMap]);
    if EditBAM then
    begin
      MenuBar^.DisableCommands([cmDDeleteMap]);
      InpCommand^.SetCmd('Edit BAM', stEmpty, False);
    end
    else
    begin
      MenuBar^.EnableCommands([cmDDeleteMap]);
      InpCommand^.SetCmd('Edit error info', stEmpty, False);
    end;
    OrigTrack := EditTrack;
    OrigSector := EditSector;
    RereadBlock := rrNone;
    if EditBAM and not DiskCopySelection then
    begin
      BAMOK := False;
      ReadBAM;
    end;
    BufferOK := True;
    InpEdit^.HideCursor;
    InpTrack^.Hide;
    InpSector^.Hide;
    InpOffset^.Hide;
    InpStatus^.Hide;
    InpPos^.Hide;
    InpCommand^.Hide;
    InpHistory^.Hide;
    if not DiskCopySelection then
    begin
      CurHelpCtx := hcMapEdit;
      AppHelpCtx := hcMapEdit;
      LastShiftState := MaxByte;
      KeyBar^.Update;
    end;
    EditPos := 0;
    EditChar := 0;
    TextMode := False;
    InsertMode := False;
    B := BufferChanged;
    BufferChanged := False;
    MapEdit := True;
    BAMEdit := EditBAM;
    DPosCursor;
    InpEdit^.DrawView;
    InpTrack^.MoveTo(EditMapIndPosX, EditMapIndPosY);
    InpTrack^.Show;
    InpSector^.MoveTo(EditMapIndPosX + 11, EditMapIndPosY);
    InpSector^.Show;
    InpEdit^.Select;
    InpEdit^.ShowCursor;
    EditMap := Application^.ExecView(DiskEditor, True, False);
    InpEdit^.HideCursor;
    if EditBAM then BAMOK := False;
    BufferChanged := B;
    DisplayAlloc;
    EditTrack := OrigTrack;
    EditSector := OrigSector;
    BufferOK := False;
    InpTrack^.Hide;
    InpSector^.Hide;
    if not DiskCopySelection then
    begin
      BAMEdit := False;
      MapEdit := False;
      InpEdit^.DrawView;
      InitDisplayPos;
      InpTrack^.Show;
      InpSector^.Show;
      InpOffset^.Show;
      InpStatus^.Show;
      InpPos^.Show;
      InpCommand^.Show;
      InpHistory^.Show;
      if RereadBlock <> rrNone then AutoReadSector;
    end;
    BufferOK := True;
    DisplayHeader;
    InpEdit^.DrawView;
    if not DiskCopySelection then
    begin
      CurHelpCtx := hcKeyDiskEdit;
      AppHelpCtx := hcKeyDiskEdit;
      LastShiftState := MaxByte;
      KeyBar^.Update;
      if EditBAM then ReadBAM;
      DisplayAlloc;
      if RereadBlock = rrEdit then
      begin
        E.What := evCommand;
        E.Command := cmDEdit;
        Application^.PutEvent(E);
      end;
      InpCommand^.SetCmd(stEmpty, stEmpty, True);
    end;
  end;
  if not DiskCopySelection then
  begin
    CurHelpCtx := H;
    AppHelpCtx := H;
    HelpCtxSet := O;
    LastShiftState := MaxByte;
    KeyBar^.Update;
    InpCommand^.ShowCursor;
  end;
end;

{Toggle automatic block read on/off}
procedure AutoReadBlock; far;
begin
  AutoRead := not AutoRead;
  InpCommand^.SetCmd('Auto-read ' + SwitchStr[AutoRead], stEmpty, False);
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
end;

{Edit the BAM}
procedure EditBAM; far;
begin
  EditMap(True);
end;

{Display the directory}
procedure DisplayDir; far;
var
  B             : Byte;
begin
  InpCommand^.SetCmd('Display directory', stEmpty, False);
  EditDirSector := FirstDirSec(Act^.CopyDiskType);
  EditDirNum := 8;
  EditDirLine := 0;
  EditDirEnd := False;
  FillChar(EditDirBuffer^, DirBufferLen, 0);
  ViewDir := True;
  InpEdit^.Select;
  InpEdit^.DrawView;
  ReadSector(Act^.DirTrack, EditDirSector, False, False, @WorkBuffer, True);
  for B := 0 to 31 do EditDirBuffer^[B + (DirBufferLen - 32)] := WorkBuffer[B];
  Inc(EditDirLine);
  InpEdit^.DrawView;
  InpEdit^.ShowCursor;
  InpEdit^.SetCursor(60, 22);
  Application^.ExecView(DiskEditor, True, False);
  InpEdit^.HideCursor;
  ViewDir := False;
  InpEdit^.DrawView;
  DisplayHeader;
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
end;

{Edit the block}
procedure EditBlock; far;
var
  O             : Boolean;
  H             : Word;
begin
  H := CurHelpCtx;
  O := HelpCtxSet;
  HelpCtxSet := True;
  CurHelpCtx := hcBlockEdit;
  AppHelpCtx := hcBlockEdit;
  LastShiftState := MaxByte;
  KeyBar^.Update;
  InpCommand^.SetCmd('Edit block', stEmpty, False);
  InsertMode := False;
  DPosCursor;
  InpEdit^.ShowCursor;
  InpEdit^.Select;
  InpEdit^.SetState(sfFocused, True);
  Application^.ExecView(DiskEditor, True, False);
  InpEdit^.HideCursor;
  if EditMarking then
  begin
    EditMarking := False;
    EditMarked := True;
  end;
  InpEdit^.DrawView;
  DisplayHeader;
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
  CurHelpCtx := H;
  AppHelpCtx := H;
  HelpCtxSet := O;
  LastShiftState := MaxByte;
  KeyBar^.Update;
end;

{Go to a block
  Input : Track, Sector: the location of the block}
procedure GotoBlock(Track, Sector: Byte); far;
begin
  InpCommand^.SetCmd('Go', stEmpty, False);
  if (Track <> EditTrack) or (Sector <> EditSector) and NewSector then
  begin
    SaveLast;
    EditTrack := Track;
    EditSector := Sector;
    AutoReadSector;
    InpCommand^.SetCmd(stEmpty, 'to', True);
  end
  else
  begin
    InpCommand^.SetCmd(stEmpty, stEmpty, False);
  end;
end;

{Toggle hexadecimal display on/off}
procedure HexaDisplay; far;
begin
  HexaMode := not HexaMode;
  InpCommand^.SetCmd('Hexa display ' + SwitchStr[HexaMode], stEmpty, False);
  DisplayHeader;
  InpEdit^.DrawView;
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
end;

{Edit the error info}
procedure EditInfo; far;
begin
  EditMap(False);
end;

{Return to the last block read or written}
procedure LastBlock; far;
var
  S,
  T             : Byte;
begin
  if (LastTrack <> 0) and ((LastTrack <> EditTrack) or (LastSector <> EditSector)) and NewSector then
  begin
    InpCommand^.SetCmd('Last block', stEmpty, False);
    T := LastTrack;
    S := LastSector;
    SaveLast;
    EditTrack := T;
    EditSector := S;
    AutoReadSector;
    InpCommand^.SetCmd(stEmpty, 'at', True);
  end;
end;

{Move to the next block of the file on basis of the sector chaining}
procedure NextBlock; far;
var
  T,
  S             : Byte;
begin
  T := DataBuffer[0];
  S := DataBuffer[1];
  if (T > 0) and (T < Act^.CopyMaxTrack) and (S < Act^.SectorNum(T)) and NewSector then
  begin
    InpCommand^.SetCmd('Next block', stEmpty, False);
    SaveLast;
    EditTrack := T;
    EditSector := S;
    AutoReadSector;
    InpCommand^.SetCmd(stEmpty, 'at', True);
  end;
end;

{Toggle the display of the owner of the current block}
procedure ToggleOwner; far;
begin
  LastLoc := 0;
  ShowOwner := not ShowOwner;
  CopyShowOwner := ShowOwner;
  InpCommand^.SetCmd('Owner ' + SwitchStr[ShowOwner], stEmpty, False);
  if CopyShowOwner then MakeOwner(True) else DisplayHeader;
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
end;

{Read a block into the edit buffer}
procedure ReadBlock; far;
var
  O             : Boolean;
begin
  InpCommand^.SetCmd('Read block', stEmpty, False);
  O := GetTS;
  if O and NewSector then ReadSector(EditTrack, EditSector, True, False, @DataBuffer, True);
  InpCommand^.SetCmd(stEmpty, 'at', O);
end;

{Toggle storing locations on/off}
procedure StoreLocation; far;
begin
  StoreLoc := not StoreLoc;
  InpCommand^.SetCmd('Store locations ' + SwitchStr[StoreLoc], stEmpty, False);
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
end;

{Trace the file to its last block on basis of the sector chaining}
procedure TraceFile; far;
var
  B,
  O             : Boolean;
  T,
  S             : Byte;

{Determine whether to continue tracing the file or stop
  Output: when True, the trace may continue}
function ContTrace: Boolean;
begin
  ContTrace := (not Escape and (T > 0) and (T < Act^.CopyMaxTrack) and (S < Act^.SectorNum(T)));
end;

begin
  T := DataBuffer[0];
  S := DataBuffer[1];
  O := False;
  if ContTrace and NewSector then
  begin
    while ContTrace do
    begin
      if not O then
      begin
        InpCommand^.SetCmd('Trace file', stEmpty, False);
        O := True;
        SaveLast;
      end;
      SaveLoc;
      EditTrack := T;
      EditSector := S;
      ReadSector(EditTrack, EditSector, True, False, @DataBuffer, True);
      T := DataBuffer[0];
      S := DataBuffer[1];
    end;
  end;
  if O then InpCommand^.SetCmd(stEmpty, 'to', True);
end;

{Undo all changes made in the edit buffer}
procedure UndoBlock; far;
begin
  if BufferChanged then
  begin
    InpCommand^.SetCmd('Undo block', stEmpty, False);
    Move(UndoBuffer, DataBuffer, 256);
    CompareBlock;
    DiskEditor^.DrawView;
    InpCommand^.SetCmd(stEmpty, stEmpty, True);
  end;
end;

{Toggle the display of the sector contents between PETSCII and ASCII}
procedure ToggleViewMode; far;
var
  S             : string[7];
begin
  EditViewMode := EditViewMode xor 1;
  case EditViewMode of
    vmASCII: S := 'ASCII';
    vmPETSCII: S := 'PETSCII';
  end;
  InpCommand^.SetCmd('Set view mode to ' + S, stEmpty, False);
  DiskEditor^.DrawView;
  InpCommand^.SetCmd(stEmpty, stEmpty, True);
end;

{Write the contents of the edit buffer into a block}
procedure WriteBlock; far;
var
  O             : Boolean;
begin
  InpCommand^.SetCmd('Write block', stEmpty, False);
  O := GetTS;
  if O then O := WriteSector(EditTrack, EditSector, True, True, @DataBuffer, True);
  if O then SaveBlock;
  DiskEditor^.DrawView;
  InpCommand^.SetCmd(stEmpty, 'at', O);
end;

{Fetch a link from a directory sector
  Input : Entry: the entry number
          Info: when True, the link to the GEOS info block is fetched}
procedure FetchDirLink(Entry: Byte; Info: Boolean);
var
  B,
  S,
  T             : Byte;
begin
  B := Entry shl 5 + 3;
  if Info then Inc(B, $12);
  T := DataBuffer[B];
  S := DataBuffer[B + 1];
  if (T > 0) and (Act^.ValidPos(T, S)) then
  begin
    InpCommand^.SetCmd('Read block', stEmpty, False);
    if NewSector then
    begin
      SaveLast;
      EditTrack := T;
      EditSector := S;
      ReadSector(EditTrack, EditSector, True, False, @DataBuffer, True);
    end;
    InpCommand^.SetCmd(stEmpty, 'at', True);
  end;
end;

{Display the command menu
  Input : E: event record to be handled}
procedure DisplayMenu(var E: TEvent);
var
  O             : Boolean;
  C,
  H             : Word;
  X,
  Y             : Integer;
  D             : PDialog;
  A,
  P             : PHistoryItem;
  M             : PHistory;
  R             : TRect;
begin
  H := CurHelpCtx;
  O := HelpCtxSet;
  HelpCtxSet := True;
  CurHelpCtx := hcDiskEdit;
  AppHelpCtx := hcDiskEdit;
  LastShiftState := MaxByte;
  InpCommand^.HideCursor;
  P := NewHistoryItem('A', 'Auto-read on/off', 1, True, nil);
  A := P;
  P := NewHistoryItem('B', 'Edit BAM', 2, True, P);
  P := NewHistoryItem('D', 'Display directory', 3, True, P);
  P := NewHistoryItem('E', 'Edit block', 4, True, P);
  P := NewHistoryItem('H', 'Hexa display on/off', 5, True, P);
  P := NewHistoryItem('I', 'Edit error info', 6, True, P);
  P := NewHistoryItem('L', 'Last block', 7, True, P);
  P := NewHistoryItem('N', 'Next block', 8, True, P);
  P := NewHistoryItem('O', 'Owner on/off', 9, True, P);
  P := NewHistoryItem('R', 'Read block', 10, True, P);
  P := NewHistoryItem('S', 'Store locations on/off', 11, True, P);
  P := NewHistoryItem('T', 'Trace file', 12, True, P);
  P := NewHistoryItem('U', 'Undo block', 13, True, P);
  P := NewHistoryItem('V', 'View mode (PETSCII/ASCII)', 14, True, P);
  P := NewHistoryItem('W', 'Write block', 15, True, P);
  Y := 15;
  R.Assign(4, 2, 0, Y + 2);
  M := New(PHistory, Init(R, A, 0, True));
  MakeWinBounds(R, M^.Size.X, Y);
  D := New(PDialog, Init(R, 'Disk Edit', fxNormal, fyNormal, False));
  D^.Insert(M);
  D^.Palette := wpHistory;
  C := Application^.ExecView(D, True, True);
  if HistoryItem >= 0 then
  begin
    P := M^.Items;
    while (P <> nil) and (HistoryItem <> P^.Code) do P := P^.Next;
    if P <> nil then E.CharCode := P^.HotStr[1];
    E.What := evKeyboard;
  end;
  Dispose(D, Done);
  InpCommand^.ShowCursor;
  CurHelpCtx := H;
  AppHelpCtx := H;
  HelpCtxSet := O;
  LastShiftState := MaxByte;
end;

{Move to the physically next sector or track
  Input : Z: step mode (next sector, next track)}
procedure BlockPlus(Z: Byte);
begin
  if Z = MaxByte then Z := ShiftCode;
  if NewSector then
  begin
    case Z of
      skNone:
      begin
        InpCommand^.SetCmd('Sector +', stEmpty, False);
        SaveLast;
        Inc(EditSector);
        if EditSector >= Act^.SectorNum(EditTrack) then
        begin
          EditSector := 0;
          Inc(EditTrack);
          if EditTrack >= Act^.CopyMaxTrack then EditTrack := 1;
        end;
        AutoReadSector;
        InpCommand^.SetCmd(stEmpty, 'to', True);
      end;
      skShift:
      begin
        InpCommand^.SetCmd('Track +', stEmpty, False);
        SaveLast;
        Inc(EditTrack);
        if EditTrack >= Act^.CopyMaxTrack then EditTrack := 1;
        if EditSector >= Act^.SectorNum(EditTrack) then EditSector := Act^.SectorNum(EditTrack) - 1;
        AutoReadSector;
        InpCommand^.SetCmd(stEmpty, 'to', True);
      end;
    end;
  end;
end;

{Move to the physically previous sector or track
  Input : Z: step mode (previous sector, previous track)}
procedure BlockMinus(Z: Byte);
begin
  if Z = MaxByte then Z := ShiftCode;
  if NewSector then
  begin
    case Z of
      skNone:
      begin
        InpCommand^.SetCmd('Sector -', stEmpty, False);
        SaveLast;
        if EditSector = 0 then
        begin
          if EditTrack = 1 then EditTrack := Act^.CopyMaxTrack;
          Dec(EditTrack);
          EditSector := Act^.SectorNum(EditTrack);
        end;
        Dec(EditSector);
        AutoReadSector;
        InpCommand^.SetCmd(stEmpty, 'to', True);
      end;
      skShift:
      begin
        InpCommand^.SetCmd('Track -', stEmpty, False);
        SaveLast;
        if EditTrack = 1 then EditTrack := Act^.CopyMaxTrack;
        Dec(EditTrack);
        if EditSector >= Act^.SectorNum(EditTrack) then EditSector := Act^.SectorNum(EditTrack) - 1;
        AutoReadSector;
        InpCommand^.SetCmd(stEmpty, 'to', True);
      end;
    end;
  end;
end;

function PrevLink: Boolean;
var
  T,
  S             : Byte;
begin
  PrevLink := False;
  CurLoc := Act^.DiskPos(EditTrack, EditSector);
  T := Locations^[CurLoc].Track;
  S := Locations^[CurLoc].Sector;
  if (T and $80 = 0) and Act^.ValidPos(T, S) then
  begin
    PrevLink := True;
    EditTrack := T;
    EditSector := S;
  end;
end;

function NextLink: Boolean;
var
  T,
  S             : Byte;
begin
  NextLink := False;
  CurLoc := Act^.DiskPos(EditTrack, EditSector);
  T := Locations2^[CurLoc].Track;
  S := Locations2^[CurLoc].Sector;
  if (T <> 0) and Act^.ValidPos(T, S) then
  begin
    NextLink := True;
    EditTrack := T;
    EditSector := S;
  end;
end;

{Move 1 or 10 locations backward in the location buffer
  Input : Z: step mode (1 backward, 10 backward)}
procedure BackLocation(Z: Byte);
var
  O             : Boolean;
  C             : Byte;
  S             : string[20];
begin
  O := False;
  if LastLoc > 0 then
  begin
    case Z of
      skNone:
      begin
        S := 'Backward';
        C := 1;
        O := True;
      end;
      skShift:
      begin
        S := '10 backward';
        C := 10;
        O := True;
      end;
    end;
    if O then
    begin
      if CopyShowOwner then
      begin
        O := False;
        while C > 0 do
        begin
          if PrevLink then O := True;
          Dec(C);
        end;
      end
      else
      begin
        O := (CurLoc > 0);
        if O then
        begin
          if CurLoc > C then Dec(CurLoc, C) else CurLoc := 0;
          EditTrack := Locations^[CurLoc].Track;
          EditSector := Locations^[CurLoc].Sector;
        end;
      end;
      if O then
      begin
        InpCommand^.SetCmd(S, stEmpty, False);
        AutoReadSector;
        InpCommand^.SetCmd(stEmpty, 'to', True);
      end;
    end;
  end;
end;

{Move 1 or 10 locations forward in the location buffer
  Input : Z: step mode (1 forward, 10 forward)}
procedure ForeLocation(Z: Byte);
var
  O             : Boolean;
  C             : Byte;
  S             : string;
begin
  O := False;
  if LastLoc > 0 then
  begin
    case Z of
      skNone:
      begin
        S := 'Forward';
        C := 1;
        O := True;
      end;
      skShift:
      begin
        S := '10 forward';
        C := 10;
        O := True;
      end;
    end;
    if O then
    begin
      if CopyShowOwner then
      begin
        O := False;
        while C > 0 do
        begin
          if NextLink then O := True;
          Dec(C);
        end;
      end
      else
      begin
        O := (CurLoc < LastLoc - 1);
        if O then
        begin
          if CurLoc + C < LastLoc then Inc(CurLoc, C) else CurLoc := LastLoc - 1;
          EditTrack := Locations^[CurLoc].Track;
          EditSector := Locations^[CurLoc].Sector;
        end;
      end;
      if O then
      begin
        InpCommand^.SetCmd(S, stEmpty, False);
        AutoReadSector;
        InpCommand^.SetCmd(stEmpty, 'to', True);
      end;
    end;
  end;
end;

{Move to the first location in the location buffer or clear the location
  buffer}
procedure HomeLocation;
var
  O,
  Q             : Boolean;
  Z             : Byte;
begin
  Z := ShiftCode;
  if LastLoc > 0 then
  begin
    case Z of
      skNone:
      begin
        if CopyShowOwner then
        begin
          O := False;
          Q := False;
          repeat
            if PrevLink then O := True else Q := True;
          until Q;
        end
        else
        begin
          O := ((CurLoc > 0) or (EditTrack <> Locations^[0].Track) or
            (EditSector <> Locations^[0].Sector));
          if O then
          begin
            CurLoc := 0;
            EditTrack := Locations^[CurLoc].Track;
            EditSector := Locations^[CurLoc].Sector;
          end;
        end;
        if O then
        begin
          InpCommand^.SetCmd('Home', stEmpty, False);
          AutoReadSector;
          InpCommand^.SetCmd(stEmpty, 'at', True);
        end;
      end;
      skShift:
      begin
        InpCommand^.SetCmd('Clear locations', stEmpty, False);
        LastLoc := 0;
        CurLoc := 0;
        CopyShowOwner := False;
        DisplayHeader;
        InpCommand^.SetCmd(stEmpty, stEmpty, True);
      end;
    end;
  end;
end;

{Move to the last location in the location buffer}
procedure EndLocation;
var
  O,
  Q             : Boolean;
begin
  if (LastLoc > 0) then
  begin
    if CopyShowOwner then
    begin
      O := False;
      Q := False;
      repeat
        if NextLink then O := True else Q := True;
      until Q;
    end
    else
    begin
      O := ((CurLoc < LastLoc - 1) or (EditTrack <> Locations^[LastLoc - 1].Track) or
        (EditSector <> Locations^[LastLoc - 1].Sector));
      if O then
      begin
        CurLoc := LastLoc - 1;
        EditTrack := Locations^[CurLoc].Track;
        EditSector := Locations^[CurLoc].Sector;
      end;
    end;
    if O then
    begin
      InpCommand^.SetCmd('End', stEmpty, False);
      AutoReadSector;
      InpCommand^.SetCmd(stEmpty, 'at', True);
    end;
  end;
end;

{Move a digit left in the edit field}
procedure DLeft;
begin
  if MapEdit then
  begin
    repeat
      Dec(EditTrack);
      if EditTrack = 0 then EditTrack := Act^.CopyMaxTrack - 1;
    until EditSector < Act^.SectorNum(EditTrack);
  end
  else
  begin
    if TextMode then
    begin
      Dec(EditPos);
    end
    else
    begin
      if EditChar = 0 then
      begin
        if HexaMode then EditChar := 1 else EditChar := 2;
        Dec(EditPos);
      end
      else
      begin
        Dec(EditChar);
      end;
    end;
  end;
  DPosCursor;
end;

{Move a digit right in the edit field}
procedure DRight;
begin
  if MapEdit then
  begin
    repeat
      Inc(EditTrack);
      if EditTrack = Act^.CopyMaxTrack then EditTrack := 1;
    until EditSector < Act^.SectorNum(EditTrack);
  end
  else
  begin
    if TextMode then
    begin
      Inc(EditPos);
    end
    else
    begin
      if (EditChar = 0) or (not HexaMode and (EditChar = 1)) then
      begin
        Inc(EditChar);
      end
      else
      begin
        EditChar := 0;
        Inc(EditPos);
      end;
    end;
  end;
  DPosCursor;
end;

{Move a byte left in the edit field or to the previous block of the current
  file in the BAM or error info editor}
procedure DCtrlLeft;
var
  B             : Byte;
  W             : Word;
begin
  if MapEdit then
  begin
    if CopyShowOwner then
    begin
      W := Act^.DiskPos(EditTrack, EditSector);
      B := Locations^[W].Track;
      if B and $80 = 0 then
      begin
        EditTrack := B;
        EditSector := Locations^[W].Sector;
        DPosCursor;
      end;
    end;
  end
  else
  begin
    Dec(EditPos);
    DPosCursor;
  end;
end;

{Move a byte right in the edit field or to the next block of the current file
  in the BAM or error info editor}
procedure DCtrlRight;
var
  B             : Byte;
  W             : Word;
begin
  if MapEdit then
  begin
    if CopyShowOwner then
    begin
      W := Act^.DiskPos(EditTrack, EditSector);
      B := Locations2^[W].Track;
      if B > 0 then
      begin
        EditTrack := B;
        EditSector := Locations2^[W].Sector;
        DPosCursor;
      end;
    end;
  end
  else
  begin
    Inc(EditPos);
    DPosCursor;
  end;
end;

{Move a line up in the edit field}
procedure DUp;
begin
  if MapEdit then
  begin
    if EditSector > 0 then Dec(EditSector) else EditSector := Act^.SectorNum(EditTrack) - 1;
  end
  else
  begin
    Dec(EditPos, 16);
  end;
  DPosCursor;
end;

{Move a line down in the edit field}
procedure DDown;
begin
  if MapEdit then
  begin
    if EditSector < Act^.SectorNum(EditTrack) - 1 then Inc(EditSector) else EditSector := 0;
  end
  else
  begin
    Inc(EditPos, 16);
  end;
  DPosCursor;
end;

{Move to the first line in the edit field}
procedure DPgUp;
begin
  if MapEdit then EditSector := 0 else EditPos := EditPos and $0F;
  DPosCursor;
end;

{Move to the last line in the edit field}
procedure DPgDn;
begin
  if MapEdit then EditSector := Act^.SectorNum(EditTrack) - 1 else EditPos := EditPos and $0F + $F0;
  DPosCursor;
end;

{Move to the first byte in the edit field}
procedure DCtrlPgUp;
var
  B             : Byte;
  W             : Word;
begin
  if MapEdit then
  begin
    if CopyShowOwner then
    begin
      repeat
        W := Act^.DiskPos(EditTrack, EditSector);
        B := Locations^[W].Track;
        if B and $80 = 0 then
        begin
          EditTrack := B;
          EditSector := Locations^[W].Sector;
        end;
      until B and $80 > 0;
      DPosCursor;
    end;
  end
  else
  begin
    EditPos := 0;
    EditChar := 0;
    DPosCursor;
  end;
end;

{Move to the last byte in the edit field}
procedure DCtrlPgDn;
var
  B             : Byte;
  W             : Word;
begin
  if MapEdit then
  begin
    if CopyShowOwner then
    begin
      repeat
        W := Act^.DiskPos(EditTrack, EditSector);
        B := Locations2^[W].Track;
        if B > 0 then
        begin
          EditTrack := B;
          EditSector := Locations2^[W].Sector;
        end;
      until B = 0;
      DPosCursor;
    end;
  end
  else
  begin
    EditPos := MaxByte;
    EditChar := 0;
    DPosCursor;
  end;
end;

{Move to the first column in the edit field}
procedure DHome;
var
  Z             : Byte;
begin
  if MapEdit then
  begin
    EditTrack := 1;
  end
  else
  begin
    if ShiftCode = skShift then
    begin
      FillFormatPattern(@DataBuffer);
      CompareBlock;
      EditPos := 0;
      EditChar := 0;
      InpEdit^.DrawView;
    end
    else
    begin
      EditPos := EditPos and $F0;
      EditChar := 0;
    end;
  end;
  DPosCursor;
end;

{Move to the last column in the edit field}
procedure DEnd;
begin
  if MapEdit then
  begin
    EditTrack := Act^.CopyMaxTrack - 1;
    while (EditTrack > 0) and (EditSector >= Act^.SectorNum(EditTrack)) do Dec(EditTrack);
  end
  else
  begin
    EditPos := EditPos and $F0 + $0F;
    EditChar := 0;
  end;
  DPosCursor;
end;

{Toggle text mode on/off}
procedure DTab;
begin
  if not MapEdit then
  begin
    TextMode := not TextMode;
    EditChar := 0;
    DPosCursor;
  end;
end;

{Toggle insert mode on/off}
procedure DIns;
begin
  if not MapEdit then
  begin
    InsertMode := not InsertMode;
    DPosCursor;
  end;
end;

{Delete a byte in the edit field}
procedure DDel;
var
  B             : Byte;
begin
  if not MapEdit then
  begin
    if EditPos < MaxByte then for B := EditPos + 1 to 255 do DataBuffer[B - 1] := DataBuffer[B];
    DataBuffer[255] := 0;
    CompareBlock;
    InpEdit^.DrawView;
  end;
end;

{Delete a byte backward in the edit field}
procedure DBack;
var
  B             : Byte;
begin
  if not MapEdit then
  begin
    if EditPos = 0 then
    begin
      DDel;
    end
    else
    begin
      if EditPos < MaxByte then for B := EditPos to 255 do DataBuffer[B - 1] := DataBuffer[B];
      DataBuffer[255] := 0;
      Dec(EditPos);
      DPosCursor;
      InpEdit^.DrawView;
    end;
    CompareBlock;
  end;
end;

{Undo a byte in the edit field}
procedure DCtrlU;
begin
  if not ViewDir and not MapEdit then
  begin
    DataBuffer[EditPos] := UndoBuffer[EditPos];
    Inc(EditPos);
    EditChar := 0;
    DPosCursor;
    InpEdit^.DrawView;
    CompareBlock;
  end;
end;

{Place the start or end marker}
procedure DMark;
begin
  if not EditMarking then EditLStart := EditPos;
  EditLEnd := EditPos;
  EditMarking := not EditMarking;
  EditMarked := True;
  InpEdit^.DrawView;
end;

{Cancel the markers}
procedure DUnmark;
begin
  EditMarking := False;
  EditMarked := False;
  InpEdit^.DrawView;
end;

{Copy the marked block into the clipboard}
procedure DCopy;
begin
  if EditMarked then
  begin
    EditClipLen := EditEnd - EditStart + 1;
    FillChar(ClipBuffer, 256, 0);
    Move(DataBuffer[EditStart], ClipBuffer, EditClipLen);
    EditMarking := False;
    EditMarked := False;
    InpEdit^.DrawView;
  end;
end;

{Paste the clipboard into the edit buffer}
procedure DPaste(Swap: Boolean);
var
  I             : Word;
begin
  if not EditMarking and (EditClipLen > 0) then
  begin
    I := EditClipLen;
    if EditPos + EditClipLen > 256 then I := 256 - EditPos;
    if Swap then SwapMem(ClipBuffer, DataBuffer[EditPos], I) else Move(ClipBuffer, DataBuffer[EditPos], I);
    EditMarked := False;
    InpEdit^.DrawView;
    CompareBlock;
  end;
end;

{Delete the marked block}
procedure DDelete;
begin
  if EditMarked then
  begin
    EditClipLen := EditEnd - EditStart + 1;
    FillChar(DataBuffer[EditStart], EditClipLen, 0);
    EditMarking := False;
    EditMarked := False;
    InpEdit^.DrawView;
    CompareBlock;
  end;
end;

{Pick the two bytes under the cursor and goto that sector}
procedure DFollow;
var
  S,
  T             : Byte;
  E             : TEvent;
begin
  T := DataBuffer[EditPos];
  S := DataBuffer[EditPos + 1];
  if (T > 0) and Act^.ValidPos(T, S) then
  begin
    if NewSector then
    begin
      SaveLast;
      EditTrack := T;
      EditSector := S;
      ReadSector(EditTrack, EditSector, True, False, @DataBuffer, True);
      DPosCursor;
    end;
  end;
end;

{Allocate the current sector
  Input : T, S: track and sector number}
procedure AllocSector(T, S: Byte); far;
begin
  if not Act^.IsBlockUsed(T, S) then BufferChanged := True;
  Act^.AllocBlock(T, S, True);
end;

{Free the current sector
  Input : T, S: track and sector number}
procedure FreeSector(T, S: Byte); far;
begin
  if Act^.IsBlockUsed(T, S) then BufferChanged := True;
  Act^.AllocBlock(T, S, False);
end;

{Put an error code onto the current sector
  Input : T, S: track and sector number}
procedure PutErrorCode(T, S: Byte); far;
var
  W             : Word;
begin
  W := Act^.DiskPos(T, S);
  BufferChanged := BufferChanged or (InfoBuffer^[W] <> ErrorCode);
  InfoBuffer^[W] := ErrorCode;
end;

{Handle keypresses inside the edit buffer
  Input : W: scan code to be handled
  Output: when False, the scan code has to be handled by the caller}
function EWrite(W: Word): Boolean;
var
  A,
  O             : Boolean;
  B,
  D,
  N             : Byte;
  C,
  E             : Char;
  J             : Word;
  I             : Integer;

{Allocate, free or invert a complete track in the BAM
  Input : T: the track to process
          Mode: processing mode: 1 for inversion, 2 for free and 3 for allocate}
procedure ProcessTrack(T, Mode: Byte);
var
  S             : Byte;
begin
  for S := 0 to Act^.SectorNum(T) - 1 do
  begin
    A := IsBlockUsed(T, S);
    case Mode of
      0:
      begin
        BufferChanged := True;
        AllocBlock(T, S, not A);
      end;
      1:
      begin
        if A then BufferChanged := True;
        AllocBlock(T, S, False);
      end;
      2:
      begin
        if not A then BufferChanged := True;
        AllocBlock(T, S, True);
      end;
    end;
  end;
end;

begin
  O := False;
  if EditAllowed then
  begin
    C := Chr(W);
    if ViewDir then
    begin
      if W = kbEsc then
      begin
        O := False;
      end
      else
      begin
        if not EditDirEnd then
        begin
          Dec(EditDirNum);
          if EditDirNum > 0 then
          begin
            O := True;
          end
          else
          begin
            if WorkBuffer[0] > 0 then
            begin
              EditDirSector := WorkBuffer[1];
              InpEdit^.HideCursor;
              ReadSector(Act^.DirTrack, EditDirSector, False, False, @WorkBuffer, True);
              InpEdit^.ShowCursor;
              EditDirNum := 8;
              O := True;
            end
            else
            begin
              EditDirEnd := True;
            end;
          end;
          if O then
          begin
            if EditDirLine < 16 then Inc(EditDirLine);
            for I := 0 to DirBufferLen - 33 do EditDirBuffer^[I] := EditDirBuffer^[I + 32];
            for I := 0 to 31 do EditDirBuffer^[I + (DirBufferLen - 32)] := WorkBuffer[I + (8 - EditDirNum) shl 5];
          end;
          InpEdit^.DrawView;
        end;
      end;
    end
    else
    begin
      if MapEdit then
      begin
        N := 0;
        if BAMEdit then
        begin
          if BAMOK then
          begin
            case W of
              kbGrayMul: N := 4;
              kbGrayMinus: N := 5;
              kbGrayPlus: N := 6;
              kbCtrlGMul: N := 7;
              kbCtrlMinus, kbCtrlGMinus: N := 8;
              kbCtrlGPlus: N := 9;
              kbAltMinus, kbAltGMinus: N := 10;
              kbAltEqual, kbAltGPlus: N := 11;
            else
              case UpCase(C) of
                '-': N := 1;
                '=': N := 2;
                ' ': N := 3;
                '_': N := 5;
                '+': N := 6;
              end;
            end;
            if (N in [5, 6]) and (ShiftCode <> skShift) then Dec(N, 4);
            A := IsBlockUsed(EditTrack, EditSector);
            case N of
              1:
              begin
                if A then BufferChanged := True;
                AllocBlock(EditTrack, EditSector, False);
              end;
              2:
              begin
                if not A then BufferChanged := True;
                AllocBlock(EditTrack, EditSector, True);
              end;
              3:
              begin
                BufferChanged := True;
                AllocBlock(EditTrack, EditSector, not A);
              end;
              4..6:
              begin
                ProcessTrack(EditTrack, N - 4);
                if N in [5, 6] then DRight;
              end;
              7..9: for D := Act^.FirstTrack to Act^.LastTrack - 1 do ProcessTrack(D, N - 7);
              10: ProcessCurrentFile(AllocSector);
              11: ProcessCurrentFile(FreeSector);
            end;
          end;
        end
        else
        begin
          A := False;
          N := MaxByte;
          case W of
            kbGrayMinus, kbGrayPlus: N := 1;
          else
            E := GetAltChar(W);
            A := (E <> #0);
            if not A then E := GetChar(W);
            if E = #0 then E := UpCase(C);
            case E of
              ' ', '-', '=', '_', '+': N := 1;
              '0'..'9': N := Ord(E) - Ord('0');
              'A'..'F': N := Ord(E) - Ord('A') + 10;
            end;
          end;
          if N = MaxByte then
          begin
            N := 0;
          end
          else
          begin
            if A then
            begin
              ErrorCode := N;
              ProcessCurrentFile(PutErrorCode);
            end
            else
            begin
              if (C in ['A'..'F']) or (E <> UpCase(C)) then
              begin
                J := Act^.DiskPos(EditTrack, 0);
                for I := 0 to Act^.SectorNum(EditTrack) - 1 do
                begin
                  BufferChanged := BufferChanged or (InfoBuffer^[J] <> N);
                  InfoBuffer^[J] := N;
                  Inc(J);
                end;
                DRight;
              end
              else
              begin
                J := Act^.DiskPos(EditTrack, EditSector);
                BufferChanged := BufferChanged or (InfoBuffer^[J] <> N);
                InfoBuffer^[J] := N;
              end;
            end;
            N := 1;
          end;
        end;
        if N > 0 then
        begin
          InpEdit^.DrawView;
          DPosCursor;
          O := True;
        end;
      end
      else
      begin
        if TextMode then
        begin
          B := MaxByte;
          case W of
            kbEnter:
          else
            if ((C > #0) and (C < ' ') and (Hi(W) = $00)) or (GetCtrlChar(W) <> #0) then B := Ord(C);
            if C >= ' ' then
            begin
              B := Ord(C);
              if EditViewMode = vmPETSCII then B := ASCtoPET[B];
            end;
            if (B = Ord(' ')) and (C <> ' ') then B := MaxByte;
            if (B = Ord(' ')) and (ShiftCode = skShift) then B := 160;
          end;
          if B <> MaxByte then O := True;
        end
        else
        begin
          if HexaMode then N := LeftPos(UpCase(C), HexaNum) - 1 else N := Ord(C) - Ord('0');
          if (N < 10) or (HexaMode and (N < 16)) then
          begin
            O := True;
            B := DataBuffer[EditPos];
            if HexaMode then
            begin
              if EditChar = 0 then B := N shl 4 + B and $0F else B := B and $F0 + N and $0F;
            end
            else
            begin
              case EditChar of
                0: B := N * 100 + B mod 100;
                1: B := (B div 100) * 100 + N * 10 + B mod 10;
                2: B := (B div 10) * 10 + N;
              end;
            end;
          end;
        end;
        if O then
        begin
          if InsertMode and (TextMode or (EditChar = 0)) and (EditPos < MaxByte) then
          begin
            if not TextMode then B := B and $F0;
            for N := 254 downto EditPos do DataBuffer[N + 1] := DataBuffer[N];
          end;
          DataBuffer[EditPos] := B;
          DRight;
          CompareBlock;
          InpEdit^.DrawView;
        end;
      end;
    end;
  end;
  EWrite := O;
end;

{Confirm exiting the disk editor, if needed
  Input : Edit: when True, only exiting the edit field, otherwise the disk
                editor itself}
function EQuit(Edit: Boolean): Boolean;
var
  O             : Boolean;
  S             : string[10];
begin
  O := True;
  if Edit then
  begin
    if MapEdit then
    begin
      if BAMEdit then S := 'BAM' else S := 'error info';
      O := False;
    end;
  end
  else
  begin
    S := 'disk';
    O := False;
  end;
  if not O then O := NewSector;
  EQuit := O;
end;

{Initialize the edit field
  Input : Bounds: bounds of the edit field}
constructor TEditInput.Init(Bounds: TRect);
begin
  TView.Init(Bounds);
  SetState(sfAlwaysIns, False);
  Options := Options or ofSelectable;
  EventMask := EventMask or evBroadcast;
end;

{Handle edit field events
  Input : Event: event record to be handled}
procedure TEditInput.HandleEvent(var Event: TEvent);
var
  O             : Boolean;
  B,
  C             : Byte;
  X,
  Y             : Integer;
  I             : PIndent;
  P             : TPoint;
begin
  if Event.What and evKeyboard > 0 then
  begin
    O := True;
    if not EWrite(Event.KeyCode) and not ViewDir then
    begin
      case Event.KeyCode of
        kbEnter:
        begin
          O := False;
          Event.What := evCommand;
          if MapEdit and not DiskCopySelection then Event.Command := cmDEditMap else Event.Command := cmOK;
        end;
        kbLeft, kbCtrlS: DLeft;
        kbRight, kbCtrlD: DRight;
        kbCtrlLeft, kbCtrlA: DCtrlLeft;
        kbCtrlRight, kbCtrlF: DCtrlRight;
        kbUp, kbCtrlE: DUp;
        kbDown, kbCtrlX: DDown;
        kbPgUp, kbCtrlR: DPgUp;
        kbPgDn, kbCtrlC: DPgDn;
        kbCtrlPgUp, kbCtrlHome: DCtrlPgUp;
        kbCtrlPgDn, kbCtrlEnd: DCtrlPgDn;
        kbHome: DHome;
        kbEnd: DEnd;
        kbIns, kbCtrlV: DIns;
        kbDel, kbCtrlG: DDel;
        kbBack, kbCtrlH: DBack;
        kbTab: DTab;
        kbCtrlU: DCtrlU;
      else
        O := False;
      end;
    end;
    if O and (Event.KeyCode <> kbEsc) then ClearEvent(Event);
  end;
  if not ViewDir and (Event.What and evMouse > 0) then
  begin
    if State and sfFocused = 0 then
    begin
      ClearEvent(Event);
    end
    else
    begin
      repeat
        if MouseInView(Event.Where) then
        begin
          MakeLocal(Event.Where, P);
          X := P.X;
          Y := P.Y;
          if Y >= 7 then
          begin
            Dec(Y, 7);
            if Y > 15 then Y := 15;
            if X >= 64 then
            begin
              TextMode := True;
              EditPos := Y shl 4 + X - 64;
              DPosCursor;
            end
            else
            begin
              TextMode := False;
              if not HexaMode or (X >= 10) then
              begin
                if HexaMode then
                begin
                  Dec(X, 10);
                  I := @HexIndent;
                  C := 2;
                end
                else
                begin
                  I := @DecIndent;
                  C := 3;
                end;
                B := 0;
                O := False;
                while not O and (B < 16) do
                begin
                  if (X >= I^[B]) and (X < I^[B] + C) then
                  begin
                    O := True;
                    EditPos := Y shl 4 + B;
                    EditChar := X - I^[B];
                    DPosCursor;
                  end;
                  Inc(B);
                end;
              end;
            end;
          end;
        end;
      until not MouseEvent(Event, evMouseMove);
    end;
  end;
  if (Event.What and evCommand > 0) then
  begin
    case Event.Command of
      cmDMark: DMark;
      cmDUnmark: DUnmark;
      cmDCopy: DCopy;
      cmDPaste: DPaste(False);
      cmDSwap: DPaste(True);
      cmDDelete: DDelete;
      cmDFollow: DFollow;
      cmDSaveMap: DSaveMap;
      cmDViewMap, cmDEditMap:
      begin
        if NewSector then
        begin
          OrigTrack := EditTrack;
          OrigSector := EditSector;
          RereadBlock := rrView;
          if Event.Command = cmDEditMap then RereadBlock := rrEdit;
          Event.Command := cmCancel;
        end;
      end;
      cmDDeleteMap: DDeleteMap;
      cmCancel: if not EQuit(True) then ClearEvent(Event);
    end;
  end;
  TView.HandleEvent(Event);
end;

{Draw the edit field}
procedure TEditInput.Draw;
var
  F,
  O             : Boolean;
  A,
  E             : Word;
  I,
  X,
  Y             : Integer;
  B             : TDrawBuffer;

{Create a line of directory data
  Input : Y: number of current line}
procedure MakeDirLine(Y: Integer);
var
  X             : Byte;
  W             : Word;
  L             : Longint;
  S             : string;
begin
  if Y >= 16 - EditDirLine then
  begin
    W := Y shl 5;
    L := BytesToLongint(EditDirBuffer^[W + 30], EditDirBuffer^[W + 31], 0, 0);
    S := LeadingSpace(L, 0);
    MoveStr(B[20], S, A);
    S := '';
    for X := 0 to CBMNameLen - 1 do S := S + Chr(EditDirBuffer^[W + X + 5]);
    MoveCBMStr(@B[26], MakeCBMName(S, Act^.GEOSFormat), A, False);
    X := EditDirBuffer^[W + 2];
    if X and faClosed > 0 then S := stSpace else S := stAsterisk;
    S := S + ShortCBMExt[X and faTypeMask];
    if X and faWriteProt > 0 then S := S + '<';
    MoveStr(B[44], S, A);
    S := LeadingZero(EditDirBuffer^[W + 3], 3);
    MoveStr(B[51], S, A);
    MoveChar(B[55], ':', A, 1);
    S := LeadingZero(EditDirBuffer^[W + 4], 3);
    MoveStr(B[57], S, A);
  end;
end;

{Create a line of BAM or error info data
  Input : Y: number of current line}
procedure MakeMapLine(Y: Integer);
var
  O             : Boolean;
  D,
  F,
  X             : Byte;
  C             : Char;
  G,
  H,
  M,
  Z             : Word;
begin
  if BufferOK then
  begin
    Z := Y;
    if Act^.CopyDiskType and dtTypeMask in [dt1541, dt1571] then MoveStr(B[2], LeadingZero(Y, 2), A);
    for X := 1 to Act^.CopyMaxTrack - 1 do
    begin
      M := Act^.SectorNum(X);
      if Y < M then
      begin
        G := A;
        if CopyShowOwner and LocationSet^.Contains(Act^.DiskPos(X, Y)) then G := E;
        F := InfoBuffer^[Z] and $0F;
        O := IsBlockUsed(X, Y);
        if BAMEdit then
        begin
          C := DotChar;
          if O then C := '*';
          O := (Act^.CopyMode <> pmDisk) or (F = 1);
        end
        else
        begin
          if F = 1 then C := DotChar else C := HexaNum[F + 1];
        end;
        D := Hi(G);
        if O then D := Lo(G);
        MoveChar(B[X shl EditMapDensity + EditMapIndent], C, D, 1);
      end;
      Inc(Z, M);
    end;
  end
  else
  begin
    MoveChar(B, ' ', A, ScreenWidth);
  end;
end;

{Create a line of block data
  Input : Y: number of current line
          Start: start of the selected block
          _End: end of the selected block
          Marked: when True, a block is selected
          Comp: when True, the data is compared to that of the undo buffer
                and a mismatching byte is colored otherwise
          Buffer: pointer to the buffer to be displayed}
procedure MakeDataLine(Y: Integer; Start, _End: Byte; Marked, Comp: Boolean; Buffer: PBlock);
var
  O             : Boolean;
  C,
  Z             : Byte;
  I,
  J,
  X             : Integer;
  S             : string[5];
begin
  if BufferOK then
  begin
    X := Y shl 4;
    if HexaMode then
    begin
      S := '0000:';
      S[3] := HexaNum[Y + 1];
      MoveStr(B, S, A);
    end;
    J := X;
    O := ((X <= Start) or (X + 15 >= _End));
    if not O and Marked then
    begin
      Z := E;
      if HexaMode then MoveChar(B[10], ' ', Z, 50) else MoveChar(B, ' ', Z, 63);
    end;
    O := O or not Marked;
    for I := 0 to 15 do
    begin
      C := Ord(Buffer^[J]);
      if O then
      begin
        if Marked and (J in [Start.._End]) then Z := E else
          if Comp and (C <> UndoBuffer[J]) then Z := Hi(A) else Z := Lo(A);
      end;
      if HexaMode then
      begin
        MoveStr(B[HexIndent[I] + 10], HexaNum[C shr 4 + 1] + HexaNum[C and $0F + 1], Z);
        if O and (J <> _End) and (I < 15) then MoveChar(B[HexIndent[I] + 12], ' ', Z, 2);
      end
      else
      begin
        MoveStr(B[DecIndent[I]], LeadingZero(C, 3), Z);
        if O and (J <> _End) and (I < 15) then MoveChar(B[DecIndent[I] + 3], ' ', Z, 1);
      end;
      case EditViewMode of
        vmASCII:
        begin
          C := C and $7F;
          if not (C in [$20..$7E]) then C := Ord('*');
          MoveChar(B[I + 64], Chr(C), Z, 1);
        end;
        vmPETSCII:
        begin
          case CharSetMode of
            csIBMLower: MoveChar(B[I + 64], Chr(PETtoASCLower[C]), Z, 1);
            csIBMUpper: MoveChar(B[I + 64], Chr(PETtoASCUpper[C]), Z, 1);
            csCBMLower:
            begin
              if C and $7F < $20 then MoveChar(B[I + 64], Chr(PETtoExtLower[C]), (Z and $0F) shl 4 + Z shr 4, 1) else
                MoveChar(B[I + 64], Chr(PETtoExtLower[C]), Z, 1);
            end;
            csCBMUpper:
            begin
              if C and $7F < $20 then MoveChar(B[I + 64], Chr(PETtoExtUpper[C]), (Z and $0F) shl 4 + Z shr 4, 1) else
                MoveChar(B[I + 64], Chr(PETtoExtUpper[C]), Z, 1);
            end;
          end;
        end;
      end;
      Inc(J);
    end;
  end
  else
  begin
    MoveChar(B, ' ', A, ScreenWidth);
  end;
end;

begin
  if EditMarking then EditLEnd := EditPos;
  if EditMarked then
  begin
    if EditLStart <= EditLEnd then
    begin
      EditStart := EditLStart;
      EditEnd := EditLEnd;
    end
    else
    begin
      EditStart := EditLEnd;
      EditEnd := EditLStart;
    end;
  end;
  A := GetColor($0401);
  E := GetColor($0502);
  O := (BufferOK and not ViewDir and (Size.Y >= 42));
  F := (EditClipLen > 0);
  X := EditMapHeight + 1;
  for I := 0 to Size.Y - 1 do
  begin
    MoveChar(B, ' ', A, Size.X);
    if MapEdit then
    begin
      if BufferOK then
      begin
        case I of
          0: for Y := 1 to Act^.CopyMaxTrack - 1 do MoveChar(B[EditMapIndent + Y shl EditMapDensity],
            Chr((Y div 10) + Ord('0')), A, 1);
          1: for Y := 1 to Act^.CopyMaxTrack - 1 do MoveChar(B[EditMapIndent + Y shl EditMapDensity],
            Chr((Y mod 10) + Ord('0')), A, 1);
        else
          if (I <= X) then MakeMapLine(I + EditMapDelta - 2);
        end;
      end;
    end
    else
    begin
      case I of
        2, 5: MoveChar(B, FrameChars[2], A, Size.X);
        6..21: if ViewDir then MakeDirLine(I - 6) else
          MakeDataLine(I - 6, EditStart, EditEnd, EditMarked, True, @DataBuffer);
        23: if O then MoveStr(B[1], 'Clipboard:', A);
        25..40: if O then MakeDataLine(I - 25, 0, EditClipLen - 1, F, False, @ClipBuffer);
      end;
    end;
    WriteBuf(0, I, Size.X, 1, B);
  end;
end;

{Handle command field events}
procedure TCmdInput.HandleEvent(var Event: TEvent);
var
  O             : Boolean;
  B             : Byte;
  C             : Char;
begin
  Quote := False;
  O := False;
  if Event.What and evCommand > 0 then
  begin
    case Event.Command of
      cmDWrite: DiskEditHelp(WriteBlock, True);
      cmDRead: DiskEditHelp(ReadBlock, True);
      cmDEdit: DiskEditHelp(EditBlock, True);
      cmDGotoBAM: GotoBlock(Act^.DirTrack, 0);
      cmDGotoDir: GotoBlock(Act^.DirTrack, FirstDirSec(Act^.CopyDiskType));
      cmDGotoFirst: GotoBlock(Act^.FirstTrack, 0);
      cmDGotoLast: GotoBlock(Act^.LastTrack - 1, 0);
      cmDMenu: DisplayMenu(Event);
      cmDFile1..cmDFile8: FetchDirLink(Event.Command - cmDFile1, False);
      cmDInfo1..cmDInfo8: FetchDirLink(Event.Command - cmDInfo1, True);
    end;
  end;
  if Event.What and evKeyboard > 0 then
  begin
    O := True;
    case Event.KeyCode of
      kbUp: BackLocation(skNone);
      kbDown: ForeLocation(skNone);
      kbPgUp: BackLocation(skShift);
      kbPgDn: ForeLocation(skShift);
      kbHome: HomeLocation;
      kbEnd: EndLocation;
      kbEnter:
      begin
        DiskEditHelp(EditBlock, True);
        O := True;
      end;
      kbEqual: BlockPlus(skNone);
      kbMinus: BlockMinus(skNone);
      kbPlus: BlockPlus(skShift);
      kbUnderStrk: BlockMinus(skShift);
      kbGrayPlus: BlockPlus(skUnknown);
      kbGrayMinus: BlockMinus(skUnknown);
      kbEsc:
      begin
        O := not EQuit(False);
        if not O then
        begin
          Event.What := evCommand;
          Event.Command := cmCancel;
        end;
      end;
      kbLeft, kbRight, kbIns, kbDel, kbBack:
    else
      O := False;
    end;
    C := UpCase(Event.CharCode);
    case C of
      '0'..'9': SwapBuffer(Ord(C) - Ord('0'), @EditBuffer^[Ord(C) - Ord('0')]);
      'A': DiskEditHelp(AutoReadBlock, False);
      'B': DiskEditHelp(EditBAM, True);
      'D': DiskEditHelp(DisplayDir, True);
      'E': DiskEditHelp(EditBlock, True);
      'H': DiskEditHelp(HexaDisplay, False);
      'I': DiskEditHelp(EditInfo, True);
      'L': DiskEditHelp(LastBlock, False);
      'N': DiskEditHelp(NextBlock, False);
      'O': DiskEditHelp(ToggleOwner, False);
      'R': DiskEditHelp(ReadBlock, True);
      'S': DiskEditHelp(StoreLocation, False);
      'T': DiskEditHelp(TraceFile, True);
      'U': DiskEditHelp(UndoBlock, False);
      'V': DiskEditHelp(ToggleViewMode, False);
      'W': DiskEditHelp(WriteBlock, True);
    end;
    if C >= ' ' then O := True;
  end;
  if Event.What and evMouse > 0 then O := True;
  if O then ClearEvent(Event);
  TInputLine.HandleEvent(Event);
end;

{Set the command string and history string, if needed}
procedure TCmdInput.SetCmd(const Text, MakePos: string; MakeLast: Boolean);
begin
  if MakeLast then
  begin
    GetData(LastCmd);
    if MakePos <> '' then LastCmd := LastCmd + stSpace + MakePos + stSpace + LeadingZero(EditTrack, 2) +  ';' +
      LeadingZero(EditSector, 2);
    InpHistory^.SetData(LastCmd);
  end;
  if Text = '' then
  begin
    DiskEditor^.EndState := 0;
    Select;
  end;
  CmdText := Text;
  Select;
  SetData(CmdText);
  FirstPos := 0;
  CurPos := 0;
  ShowCursor;
  DrawView;
end;

{Initialize the disk editor}
constructor TDiskEditor.Init(Bounds: TRect);
var
  R             : TRect;
begin
  TDialog.Init(Bounds, stEmpty, fxNone, fyNone, False);
  Options := 0;
  SetState(sfShadow, False);
  Palette := wpViewEdit;
  R.Assign(0, 0, ScreenWidth, Size.Y);
  InpEdit := New(PEditInput, Init(R));
  InpEdit^.Options := InpEdit^.Options and not ofSelectable;
  R.Assign(0, 0, 9, 1);
  InpTrack := New(PNumInput, Init(R, 2, 2, 'Track:', drLeft));
  InpTrack^.Options := InpTrack^.Options and not ofSelectable;
  R.Assign(0, 0, 10, 1);
  InpSector := New(PNumInput, Init(R, 2, 2, 'Sector:', drLeft));
  InpSector^.Options := InpSector^.Options and not ofSelectable;
  R.Assign(27, 1, 17, 1);
  InpOffset := New(PCmdInput, Init(R, 9, 9, 'Offset:', drLeft));
  InpOffset^.Options := InpOffset^.Options and not ofSelectable;
  R.Assign(48, 1, 12, 1);
  InpStatus := New(PCmdInput, Init(R, 4, 4, 'Status:', drLeft));
  InpStatus^.Options := InpStatus^.Options and not ofSelectable;
  R.Assign(64, 1, 13, 1);
  InpPos := New(PNumInput, Init(R, 3, 3, 'Position:', drLeft));
  InpPos^.Options := InpPos^.Options and not ofSelectable;
  R.Assign(1, 3, 59, 1);
  InpHistory := New(PCmdInput, Init(R, 50, 50, 'History:', drLeft));
  InpHistory^.Options := InpHistory^.Options and not ofSelectable;
  R.Assign(1, 4, 59, 1);
  InpCommand := New(PCmdInput, Init(R, 50, 50, 'Command:', drLeft));
  InpCommand^.Options := InpCommand^.Options and not ofSelectable;
  InitDisplayPos;
  Insert(InpEdit);
  Insert(InpTrack);
  Insert(InpSector);
  Insert(InpOffset);
  Insert(InpPos);
  Insert(InpStatus);
  Insert(InpHistory);
  Insert(InpCommand);
end;

{Handle disk editor events}
procedure TDiskEditor.HandleEvent(var Event: TEvent);
begin
  if (Current = PView(InpCommand)) and (Event.What = evCommand) and (Event.Command = cmCancel) then
  begin
    Event.What := evKeyboard;
    Event.KeyCode := kbEsc;
  end;
  TDialog.HandleEvent(Event);
end;

{'Disk editor' item in the 'Commands' menu: edit the contents of a disk or
  a disk image
  Input : Z: file selection mode (file under the cursor bar, file name
             already given)
          CallEditor: when True, the disk editor starts up at once with
                      the BAM or error info editor
          BAMEditor: when True, it is the BAM editor that the disk editor
                     starts up at once with
          AllowEdit: when True, editing data is allowed, otherwise not
  Output: when False, the user left the disk editor with Escape}
function DiskEdit(Z: Byte; CallEditor, BAMEditor, AllowEdit: Boolean): Boolean;
var
  O             : Boolean;
  B,
  C,
  T             : Byte;
  W             : Word;
  S             : string;
  R             : TRect;
  E             : TEvent;
begin
  EditAllowed := AllowEdit;
  DiskCopySelection := CallEditor;
  if not DiskCopySelection then
  begin
    BoxTitle := 'Disk editor';
    ChangeHelpCtx(hcDiskEdit);
    CopyFileMode := Z;
    if BatchMode = bmNone then AllErrorSkip := ayNone;
  end;
  ClockOff;
  O := False;
  if (CopyFileMode = cfAutomatic) or Act^.OK or (Act^.Mode = pmExt) then
  begin
    if CopyFileMode = cfAutomatic then
    begin
      Act^.CopyMode := DetermineTypeName(SourceName);
      Act^.CopyPath := GetPath(SourceName, chDirSep);
      Act^.CopyImageName := CutPath(SourceName, chDirSep);
      Act^.CopyImageProtoName := Act^.CopyImageName;
      Act^.CopyImagePath := '';
      if Act^.OpenImage(False, True, False, True, True) = 0 then
      begin
        O := True;
        EditTrack := Act^.MainDirTrack;
        EditSector := 0;
        Act^.CloseImage(False);
      end;
    end
    else
    begin
      Act^.CopyMode := Act^.Mode;
      case Act^.CopyMode of
        pmDOS:
        begin
          if DiskCopySelection then S := Act^.CopyImageName else S := Act^.GetNamePtr(Act^.Cur)^;
          B := DetermineTypeName(S);
          if B in [pmDisk, pmDiskZip, pmSixZip] then
          begin
            if not DiskCopySelection then
            begin
              Act^.CopyDiskType := dtInvalid;
              case B of
                pmDisk: Act^.CopyDiskType := GetDiskType(LonglongintToLongint(Act^.Dir[Act^.Cur].Size));
                pmDiskZip:
                begin
                  Act^.CopyDiskType := dt1541;
                  S[1] := '5';
                  if FileExists(AddToPath(Act^.CopyRealPath, S, chDirSep), False) then Act^.CopyDiskType := dt1541Ext;
                end;
              end;
            end;
            O := (Act^.CopyDiskType <> dtInvalid);
            if O then
            begin
              Act^.CopyMode := B;
              Act^.DirTrack := 0;
              Act^.CheckDiskType;
              Act^.CopyImageProtoName := S;
              EditTrack := Act^.MainDirTrack;
              EditSector := 0;
            end;
          end;
        end;
        pmExt, pmDisk, pmDiskZip, pmSixZip:
        begin
          if (Act^.Max > 0) and ((Act^.Mode = pmExt) or (Act^.Cur > 0)) then
          begin
            EditTrack := Act^.Dir[Act^.Cur].Track;
            EditSector := Act^.Dir[Act^.Cur].Sector;
            O := (EditTrack > 0) and (EditTrack < Act^.CopyMaxTrack) and (EditSector < Act^.SectorNum(EditTrack));
          end;
          if not O then
          begin
            EditTrack := Act^.DirTrack;
            EditSector := 0;
          end;
          O := True;
          if (Act^.Mode = pmExt) and not DiskCopySelection then
          begin
            if CheckDevice then
            begin
              Act^.CopyDiskType := DetectDiskType(ExtendedDisk, True);
            end
            else
            begin
              O := False;
              if not ReadCBMError(S, True, False, True) then ErrorWin(stError, S, stEmpty, CurHelpCtx, sbNone);
            end;
          end;
          Act^.CopyImageProtoName := Act^.CopyImageName;
        end;
      end;
    end;
  end;
  if O then
  begin
    Act^.CheckDiskType;
    case Act^.CopyMode of
      pmExt: Act^.CopyImageName := 'Disk in drive ' + LeadingSpace(Act^.CBMDev, 1) + ':'
    else
      Act^.CorrectImageName(False);
    end;
    R.Assign(0, 1, ScreenWidth, WinSize);
    DiskEditor := New(PDiskEditor, Init(R));
    DiskEditor^.SetState(sfVisible, False);
    Application^.Insert(DiskEditor);
    EditTitle := New(PTitle, Init);
    EditTitle^.Hide;
    Application^.Insert(EditTitle);
    KeyBar^.Update;
    InpCommand^.SetCmd(stEmpty, stEmpty, True);
    InpCommand^.SetCmd(stEmpty, stEmpty, True);
    S := '';
    InpOffset^.SetData(S);
    InpStatus^.SetData(S);
    CopyShowOwner := False;
    DiskChanged := False;
    MapEdit := DiskCopySelection;
    BAMEdit := BAMEditor;
    ClockOff;
    CursorOff;
    EditMarking := False;
    EditMarked := False;
    TextMode := False;
    LastLoc := 0;
    MakeTitle(stEmpty);
    T := Act^.CopyDiskType and dtTypeMask;
    EditMapIndent := 3 - (Byte(T in [dt1541Ext, dt1581]) * 4);
    EditMapDensity := Byte(T in [dt1541, dt1541Ext]);
    EditPos := 0;
    EditChar := 0;
    EditMapDelta := 0;
    EditMapHeight := DiskEditor^.Size.Y - 2;
    EditMapIndPosY := EditMapHeight + 1;
    B := Act^.SectorNum(1) + 1;
    C := B;
    EditMapIndPosX := 53;
    if (T = dt1571) and (EditMapHeight <= B) then Inc(EditMapIndPosX, 5);
    case T of
      dt1571: Inc(C, 2);
      dt1581:
      begin
        if EditMapHeight < 42 then
        begin
          EditMapIndPosY := 23;
          if EditMapHeight > 22 then Inc(EditMapIndPosY);
          EditMapHeight := 21;
        end;
        Inc(C, 2);
      end;
    else
      Dec(EditMapIndPosY);
    end;
    if EditMapHeight > B then EditMapHeight := B;
    if EditMapIndPosY > C then EditMapIndPosY := C;
    DisplayHeader;
    SaveLast;
    FillChar(DataBuffer, 256, 0);
    BufferOK := False;
    DiskEditor^.SetState(sfActive, True);
    DiskEditor^.SetState(sfFocused, True);
    DiskEditor^.SetState(sfModal, True);
    EditBuffer := New(PEditBuffer);
    EditDirBuffer := New(PEditDirBuf);
    if not DiskCopySelection then InfoBuffer := New(PInfoBuffer);
    Locations := New(PLocBuffer);
    Locations2 := New(PLocBuffer);
    LocationSet := New(PLocationSet);
    FillChar(EditBuffer^, SizeOf(TEditBuffer), 0);
    LocationSet^.Init(False);
    if DiskCopySelection then
    begin
      InpCommand^.Hide;
      InpHistory^.Hide;
      InpTrack^.Hide;
      InpSector^.Hide;
      InpOffset^.Hide;
      InpStatus^.Hide;
      InpPos^.Hide;
      BAMOK := True;
    end;
    DiskEditor^.Show;
    EditTitle^.Show;
    CopyShowOwner := ShowOwner;
    if CopyShowOwner then MakeOwner(False);
    if not DiskCopySelection then
    begin
      if (CopyTransferMode <> tmNormal) and (Act^.CopyMode = pmExt) then SendTurbo;
      ReadBAM;
    end;
    Act^.CheckGEOSFormat(@Act^.BAM);
    EditViewMode := vmPETSCII;
    if Act^.CopyGEOSFormat then EditViewMode := vmASCII;
    if not DiskCopySelection and ((EditTrack <> Act^.DirTrack) or (EditSector <> 0)) then
      ReadSector(EditTrack, EditSector, True, False, @DataBuffer, True);
    SaveBlock;
    InpCommand^.SetCmd(stEmpty, stEmpty, True);
    if not DiskCopySelection then BufferOK := True;
    DisplayAlloc;
    DiskEditor^.DrawView;
    if not DiskCopySelection then
    begin
      CurHelpCtx := hcDiskEdit2;
      AppHelpCtx := hcDiskEdit2;
      LastShiftState := MaxByte;
    end;
    if DiskCopySelection then W := EditMap(BAMEditor) else W := Application^.ExecView(DiskEditor, True, True);
    DiskEdit := (W = cmOK);
    if not DiskCopySelection then InpCommand^.SetCmd('Quit disk editor', stEmpty, True);
    CursorOff;
    ClockOff;
    if not DiskCopySelection and (CopyTransferMode <> tmNormal) and (Act^.CopyMode = pmExt) then
    begin
      ShutDownTurbo;
      OpenCBMChannel(saCommand, 'I0', True);
      MouseOn;
    end;
    Dispose(Locations2);
    Dispose(Locations);
    Dispose(LocationSet);
    if not DiskCopySelection then Dispose(InfoBuffer);
    Dispose(EditDirBuffer);
    Dispose(EditBuffer);
    ClockOn;
    Dispose(EditTitle, Done);
    Dispose(DiskEditor, Done);
    if not DiskCopySelection and DiskChanged then RereadPanels;
  end
  else
  begin
    ClockOn;
  end;
  if not DiskCopySelection then
  begin
    RestoreHelpCtx;
    NextBatchCommand;
  end;
end;

end.
