
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Turbo Vision Unit                               }
{                                                       }
{       Copyright (c) 1992 Borland International        }
{                                                       }
{*******************************************************}

unit Menus;

{$A+,B-,D+,E-,F+,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses Objects, Drivers, Views;

const

{ Menu selection modes}

  msNormal       = 0;
  msPrevItem     = 1;
  msLast         = 2;

{ Color palettes }

  CMenuView   = #8#9#10#11#12#13#14#15#16;
  CKeyBar     = #6#7;

{ Help contexts }

  hcOnlyQuit  = 127;
  hcNormal    = 128;
  hcShift     = 129;
  hcControl   = 130;
  hcAlt       = 131;

type

{ TMenu types }

  TMenuStr = string[31];

  PMenu = ^TMenu;

  PMenuItem = ^TMenuItem;
  TMenuItem = record
    Next: PMenuItem;
    Name: PString;
    Command: Word;
    Disabled: Boolean;
    KeyCode: Word;
    HelpCtx: Word;
    case Integer of
      0: (Param: PString);
      1: (SubMenu: PMenu);
  end;

  TMenu = record
    Items: PMenuItem;
    Default: PMenuItem;
  end;

{ TMenuView object }

  PMenuView = ^TMenuView;
  TMenuView = object(TView)
    ParentMenu: PMenuView;
    Menu: PMenu;
    Current: PMenuItem;
    constructor Init(var Bounds: TRect);
    function Execute: Word; virtual;
    function FindItem(Ch: Char): PMenuItem;
    procedure GetItemRect(Item: PMenuItem; var R: TRect); virtual;
    function GetHelpCtx: Word; virtual;
    function GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    function HotKey(KeyCode: Word): PMenuItem;
    function NewSubView(var Bounds: TRect; AMenu: PMenu;
      AParentMenu: PMenuView): PMenuView; virtual;
  end;

{ TMenuBar object }

  PMenuBar = ^TMenuBar;
  TMenuBar = object(TMenuView)
    constructor Init(var Bounds: TRect; AMenu: PMenu);
    destructor Done; virtual;
    procedure Draw; virtual;
    procedure GetItemRect(Item: PMenuItem; var R: TRect); virtual;
  end;

{ TMenuBox object }

  PMenuBox = ^TMenuBox;
  TMenuBox = object(TMenuView)
    constructor Init(var Bounds: TRect; AMenu: PMenu;
      AParentMenu: PMenuView);
    procedure Draw; virtual;
    procedure GetItemRect(Item: PMenuItem; var R: TRect); virtual;
  end;

{ TStatusItem }

  PStatusItem = ^TStatusItem;
  TStatusItem = record
    Next: PStatusItem;
    Text: PString;
    KeyCode: Word;
    Command: Word;
  end;

{ TStatusDef }

  PStatusDef = ^TStatusDef;
  TStatusDef = record
    Next: PStatusDef;
    Min, Max: Word;
    Items: PStatusItem;
  end;

{ TKeyBar }

  PKeyBar = ^TKeyBar;
  TKeyBar = object(TView)
    Items: PStatusItem;
    Defs: PStatusDef;
    constructor Init(var Bounds: TRect; ADefs: PStatusDef);
    destructor Done; virtual;
    procedure Draw; virtual;
    function GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Update; virtual;
  private
    procedure DrawSelect(Selected: PStatusItem);
    procedure FindItems;
  end;

const
  MenuSelectMode: Byte = msNormal;

{ TMenuItem routines }

function NewItem(Name, Param: TMenuStr; KeyCode: Word; Command: Word;
  AHelpCtx: Word; Next: PMenuItem): PMenuItem;
function NewLine(Next: PMenuItem): PMenuItem;
function NewSubMenu(Name: TMenuStr; AHelpCtx: Word; SubMenu: PMenu;
  Next: PMenuItem): PMenuItem;

{ TMenu routines }

function NewMenu(Items: PMenuItem): PMenu;
procedure DisposeMenu(Menu: PMenu);

{ TKeyBar routines }

function NewStatusDef(AMin, AMax: Word; AItems: PStatusItem;
  ANext: PStatusDef): PStatusDef;
function NewStatusKey(const AText: string; AKeyCode: Word; ACommand: Word;
  ANext: PStatusItem): PStatusItem;

var
  MenuInUse, HelpInUse: Boolean;

implementation

{ TMenuItem routines }

function NewItem(Name, Param: TMenuStr; KeyCode: Word; Command: Word;
  AHelpCtx: Word; Next: PMenuItem): PMenuItem;
const
  T: PView = nil;
var
  P: PMenuItem;
begin
  if (Name <> '') and (Command <> 0) then
  begin
    New(P);
    P^.Next := Next;
    P^.Name := NewStr(Name);
    P^.Command := Command;
    P^.Disabled := not T^.CommandEnabled(Command);
    P^.KeyCode := KeyCode;
    P^.HelpCtx := AHelpCtx;
    P^.Param := NewStr(Param);
    NewItem := P;
  end else
  NewItem := Next;
end;

function NewLine(Next: PMenuItem): PMenuItem;
var
  P: PMenuItem;
begin
  New(P);
  P^.Next := Next;
  P^.Name := nil;
  P^.HelpCtx := hcNoContext;
  NewLine := P;
end;

function NewSubMenu(Name: TMenuStr; AHelpCtx: Word; SubMenu: PMenu;
  Next: PMenuItem): PMenuItem;
var
  P: PMenuItem;
begin
  if (Name <> '') and (SubMenu <> nil) then
  begin
    New(P);
    P^.Next := Next;
    P^.Name := NewStr(Name);
    P^.Command := 0;
    P^.Disabled := False;
    P^.HelpCtx := AHelpCtx;
    P^.SubMenu := SubMenu;
    NewSubMenu := P;
  end else
  NewSubMenu := Next;
end;

{ TMenu routines }

function NewMenu(Items: PMenuItem): PMenu;
var
  P: PMenu;
begin
  New(P);
  P^.Items := Items;
  P^.Default := nil;
  NewMenu := P;
end;

procedure DisposeMenu(Menu: PMenu);
var
  P, Q: PMenuItem;
begin
  if Menu <> nil then
  begin
    P := Menu^.Items;
    while P <> nil do
    begin
      if P^.Name <> nil then
      begin
        DisposeStr(P^.Name);
        if P^.Command <> 0 then
          DisposeStr(P^.Param) else
          DisposeMenu(P^.SubMenu);
      end;
      Q := P;
      P := P^.Next;
      Dispose(Q);
    end;
    Dispose(Menu);
  end;
end;

{ TMenuView }

constructor TMenuView.Init(var Bounds: TRect);
begin
  TView.Init(Bounds);
  EventMask := EventMask or evBroadcast;
end;

function TMenuView.Execute: Word;
type
  MenuAction = (DoNothing, DoSelect, DoReturn);
var
  AutoSelect: Boolean;
  Action: MenuAction;
  Ch: Char;
  Result: Word;
  ItemShown, P: PMenuItem;
  Target: PMenuView;
  R: TRect;
  E: TEvent;
  MouseActive: Boolean;

procedure TrackMouse;
var
  Mouse: TPoint;
  R: TRect;
begin
  MakeLocal(E.Where, Mouse);
  Current := Menu^.Items;
  while Current <> nil do
  begin
    if (Current^.Name <> nil) and not (Current^.Disabled) then
    begin
      GetItemRect(Current, R);
      if R.Contains(Mouse) then
      begin
        MouseActive := True;
        Exit;
      end;
    end;
    Current := Current^.Next;
  end;
end;

procedure TrackKey(FindNext: Boolean);

procedure NextItem;
begin
  Current := Current^.Next;
  if Current = nil then Current := Menu^.Items;
end;

procedure PrevItem;
var
  P: PMenuItem;
begin
  P := Current;
  if P = Menu^.Items then P := nil;
  repeat NextItem until Current^.Next = P;
end;

begin
  if Current <> nil then
    repeat
      if FindNext then NextItem else PrevItem;
    until (Current^.Name <> nil) and not (Current^.Disabled);
end;

function MouseInOwner: Boolean;
var
  Mouse: TPoint;
  R: TRect;
begin
  MouseInOwner := False;
  if (ParentMenu <> nil) and (ParentMenu^.Size.Y = 1) then
  begin
    ParentMenu^.MakeLocal(E.Where, Mouse);
    ParentMenu^.GetItemRect(ParentMenu^.Current, R);
    MouseInOwner := R.Contains(Mouse);
  end;
end;

function MouseInMenus: Boolean;
var
  P: PMenuView;
begin
  P := ParentMenu;
  while (P <> nil) and not P^.MouseInView(E.Where) do P := P^.ParentMenu;
  MouseInMenus := P <> nil;
end;

function TopMenu: PMenuView;
var
  P: PMenuView;
begin
  P := @Self;
  while P^.ParentMenu <> nil do P := P^.ParentMenu;
  TopMenu := P;
end;

begin
  MenuInUse := True;
  if not Exposed then Show;
  AutoSelect := (MenuSelectMode = msPrevItem);
  Result := 0;
  ItemShown := nil;
  Current := Menu^.Items;
  case MenuSelectMode of
    msPrevItem: Current := Menu^.Default;
    msLast:
    begin
      while Current^.Next <> nil do Current := Current^.Next;
      MenuSelectMode := msNormal;
    end;
  end;
  MouseActive := False;
  repeat
    Action := DoNothing;
    GetEvent(E);
    case E.What of
      evMouseDown:
        if MouseInView(E.Where) or MouseInOwner then
        begin
          TrackMouse;
          if Size.Y = 1 then AutoSelect := True;
        end else Action := DoReturn;
      evMouseUp:
        begin
          TrackMouse;
          if MouseInOwner then
            Current := Menu^.Items
          else
            if (Current <> nil) and (Current^.Name <> nil) then
              Action := DoSelect
            else
              if MouseActive or MouseInView(E.Where) then Action := DoReturn
              else
              begin
                Current := Menu^.Items;
                Action := DoReturn;
              end;
        end;
      evMouseMove:
        if E.Buttons <> 0 then
        begin
          TrackMouse;
          if not (MouseInView(E.Where) or MouseInOwner) and
            MouseInMenus then Action := DoReturn;
        end;
      evKeyDown:
        case E.KeyCode of
          kbUp, kbDown:
            if Size.Y <> 1 then
              TrackKey(E.KeyCode = kbDown) else
              if E.KeyCode = kbDown then AutoSelect := True;
          kbLeft, kbRight:
            if ParentMenu = nil then
              TrackKey(E.KeyCode = kbRight) else
              Action := DoReturn;
          kbHome, kbEnd, kbPgUp, kbPgDn:
            if Size.Y <> 1 then
            begin
              Current := Menu^.Items;
              if (E.KeyCode = kbEnd) or (E.KeyCode = kbPgDn) then TrackKey(False);
            end;
          kbEnter, kbCtrlEnter:
            begin
              if Size.Y = 1 then AutoSelect := True;
              Action := DoSelect;
            end;
          kbEsc:
            begin
              Action := DoReturn;
              if (ParentMenu = nil) or (ParentMenu^.Size.Y <> 1) then
                ClearEvent(E);
            end;
        else
          Target := @Self;
          Ch := E.CharCode;
          P := Target^.FindItem(Ch);
          if P = nil then
          begin
            P := HotKey(E.KeyCode);
            if (P <> nil) and CommandEnabled(P^.Command) then
            begin
              Result := P^.Command;
              Action := DoReturn;
            end
          end else
            if Target = @Self then
            begin
              if Size.Y = 1 then AutoSelect := True;
              Action := DoSelect;
              Current := P;
            end else
              if (ParentMenu <> Target) or (ParentMenu^.Current <> P) then
                Action := DoReturn;
        end;
      evCommand:
        if E.Command = cmMenu then
        begin
          if ParentMenu <> nil then Action := DoReturn;
        end else Action := DoReturn;
    end;
    if ItemShown <> Current then
    begin
      ItemShown := Current;
      DrawView;
    end;
    if (Action = DoSelect) or ((Action = DoNothing) and AutoSelect) then
      if Current <> nil then with Current^ do if Name <> nil then
        if Command = 0 then
        begin
          if E.What and (evMouseDown + evMouseMove) <> 0 then PutEvent(E);
          GetItemRect(Current, R);
          R.A.X := R.A.X + Origin.X;
          R.A.Y := R.B.Y + Origin.Y;
          R.B := Owner^.Size;
          Target := TopMenu^.NewSubView(R, SubMenu, @Self);
          Result := Owner^.ExecView(Target, True, True);
          Dispose(Target, Done);
        end else if Action = DoSelect then Result := Command else MenuSelectMode := msNormal;
    if (Result <> 0) and CommandEnabled(Result) then
    begin
      Action := DoReturn;
      ClearEvent(E);
    end
    else
      Result := 0;
  until Action = DoReturn;
  if E.What <> evNothing then
    if (ParentMenu <> nil) or (E.What = evCommand) then PutEvent(E);
  if Current <> nil then
  begin
    if Result <> 0 then Menu^.Default := Current;
    Current := nil;
    DrawView;
  end;
  Execute := Result;
  if not ShowMenu then Hide;
  LastShiftState := MaxByte;
  if ParentMenu = nil then MenuInUse := False;
end;

function TMenuView.FindItem(Ch: Char): PMenuItem;
var
  P: PMenuItem;
  I: Integer;
begin
  Ch := UpCase(Ch);
  P := Menu^.Items;
  while P <> nil do
  begin
    if (P^.Name <> nil) and not P^.Disabled then
    begin
      I := LeftPos(ColorChar, P^.Name^);
      if (I <> 0) and (Ch = UpCase(P^.Name^[I + 1])) then
      begin
        FindItem := P;
        Exit;
      end;
    end;
    P := P^.Next;
  end;
  FindItem := nil;
end;

procedure TMenuView.GetItemRect(Item: PMenuItem; var R: TRect);
begin
end;

function TMenuView.GetHelpCtx: Word;
var
  C: PMenuView;
  H: Word;
begin
  if GetShiftState > 0 then H := hcNoContext else
  begin
    C := @Self;
    while (C <> nil) and
       ((C^.Current = nil) or (C^.Current^.HelpCtx = hcNoContext) or
        (C^.Current^.Name = nil)) do
      C := C^.ParentMenu;
    if C <> nil then H := C^.Current^.HelpCtx
    else H := hcHelp;
  end;
  if not HelpInUse then AppHelpCtx := H;
  GetHelpCtx := H;
end;

function TMenuView.GetPalette: PPalette;
const
  P: string[Length(CMenuView)] = CMenuView;
begin
  GetPalette := @P;
end;

procedure TMenuView.HandleEvent(var Event: TEvent);
var
  CallDraw: Boolean;
  P: PMenuItem;

procedure UpdateMenu(Menu: PMenu);
var
  P: PMenuItem;
  CommandState: Boolean;
begin
  P := Menu^.Items;
  while P <> nil do
  begin
    if P^.Name <> nil then
      if P^.Command = 0 then UpdateMenu(P^.SubMenu)
      else
      begin
        CommandState := CommandEnabled(P^.Command);
        if P^.Disabled = CommandState then
        begin
          P^.Disabled := not CommandState;
          CallDraw := True;
        end;
      end;
    P := P^.Next;
  end;
end;

procedure DoSelect;
begin
  PutEvent(Event);
  Event.Command := Owner^.ExecView(@Self, True, True);
  if (Event.Command <> 0) and CommandEnabled(Event.Command) then
  begin
    Event.What := evCommand;
    PutEvent(Event);
  end;
  ClearEvent(Event);
end;

begin
  if Menu <> nil then
    case Event.What of
      evMouseDown:
        DoSelect;
      evKeyDown:
        if (FindItem(GetAltChar(Event.KeyCode)) <> nil) then
          DoSelect
        else
        begin
          P := HotKey(Event.KeyCode);
          if (P <> nil) and (CommandEnabled(P^.Command)) then
          begin
            Event.What := evCommand;
            Event.Command := P^.Command;
            PutEvent(Event);
            ClearEvent(Event);
          end;
        end;
      evCommand:
        if Event.Command = cmMenu then DoSelect;
      evBroadcast:
        if Event.Command = cmCommandSetChanged then
        begin
          CallDraw := False;
          UpdateMenu(Menu);
          if CallDraw then DrawView;
        end;
    end;
end;

function TMenuView.HotKey(KeyCode: Word): PMenuItem;

function FindHotKey(P: PMenuItem): PMenuItem;
var
  T: PMenuItem;
begin
   while P <> nil do
   begin
     if P^.Name <> nil then
       if (P^.Command <> 0) and not P^.Disabled and
         (P^.KeyCode <> kbNoKey) and (P^.KeyCode = KeyCode) then
       begin
         FindHotKey := P;
         Exit;
       end;
     P := P^.Next;
   end;
  FindHotKey := nil;
end;

begin
  HotKey := FindHotKey(Menu^.Items);
end;

function TMenuView.NewSubView(var Bounds: TRect; AMenu: PMenu;
  AParentMenu: PMenuView): PMenuView;
begin
  NewSubView := New(PMenuBox, Init(Bounds, AMenu, AParentMenu));
end;

{ TMenuBar }

constructor TMenuBar.Init(var Bounds: TRect; AMenu: PMenu);
begin
  TMenuView.Init(Bounds);
  GrowMode := gfGrowHiX;
  Menu := AMenu;
  Options := Options or ofPreProcess;
end;

destructor TMenuBar.Done;
begin
  TMenuView.Done;
  DisposeMenu(Menu);
end;

procedure TMenuBar.Draw;
var
  X, L: Integer;
  CNormal, CSelect, Color: Word;
  P: PMenuItem;
  B: TDrawBuffer;
begin
  CNormal := GetColor($0101);
  CSelect := GetColor($0202);
  MoveChar(B, ' ', Byte(CNormal), Size.X);
  if Menu <> nil then
  begin
    X := 1;
    P := Menu^.Items;
    while P <> nil do
    begin
      if P^.Name <> nil then
      begin
        L := CStrLen(P^.Name^);
        if X + L < Size.X then
        begin
          if P = Current then Color := CSelect else Color := CNormal;
          MoveCStr(B[X + 1], P^.Name^, Color);
        end;
        Inc(X, L);
      end;
      P := P^.Next;
    end;
  end;
  WriteBuf(0, 0, Size.X, 1, B);
end;

procedure TMenuBar.GetItemRect(Item: PMenuItem; var R: TRect);
var
  P: PMenuItem;
begin
  R.Assign(1, 0, 2, 1);
  P := Menu^.Items;
  while True do
  begin
    R.A.X := R.B.X;
    if P^.Name <> nil then Inc(R.B.X, CStrLen(P^.Name^));
    if P = Item then Exit;
    P := P^.Next;
  end;
end;

{ TMenuBox }

constructor TMenuBox.Init(var Bounds: TRect; AMenu: PMenu;
  AParentMenu: PMenuView);
var
  W, H, L: Integer;
  P: PMenuItem;
  R: TRect;
begin
  W := 10;
  H := 2;
  if AMenu <> nil then
  begin
    P := AMenu^.Items;
    while P <> nil do
    begin
      if P^.Name <> nil then
      begin
        L := CStrLen(P^.Name^) + 6;
        if P^.Command = 0 then Inc(L, 3) else
          if P^.Param <> nil then Inc(L, CStrLen(P^.Param^) + 1);
        if L > W then W := L;
      end;
      Inc(H);
      P := P^.Next;
    end;
  end;
  R.Copy(Bounds);
  if R.A.X + W < R.B.X then R.B.X := R.A.X + W else R.A.X := R.B.X - W;
  if R.A.Y + H < R.B.Y then R.B.Y := R.A.Y + H else R.A.Y := R.B.Y - H;
  TMenuView.Init(R);
  State := State or sfShadow;
  Options := Options or ofPreProcess;
  Menu := AMenu;
  ParentMenu := AParentMenu;
end;

procedure TMenuBox.Draw;
var
  CNormal, CSelect, CDisabled, Color: Word;
  CFrame, CMinus: Byte;
  Y: Integer;
  P: PMenuItem;
  B: TDrawBuffer;

procedure FrameLine(N, X: Integer);
begin
  MoveChar(B[0], ' ', Byte(Color), Size.X);
  MoveBuf(B[0], MenuFrameChars[N], CFrame, 1);
  MoveChar(B[X], MenuFrameChars[N + 1], Byte(Color), Size.X - (X shl 1));
  MoveBuf(B[Size.X - 1], MenuFrameChars[N + 2], CFrame, 1);
end;

procedure DrawLine;
begin
  WriteBuf(0, Y, Size.X, 1, B);
  Inc(Y);
end;

begin
  CNormal := GetColor($0504);
  CSelect := GetColor($0706);
  CDisabled := GetColor($0808);
  CFrame := GetColor(3);
  CMinus := GetColor(9);
  Y := 0;
  Color := CFrame;
  FrameLine(1, 1);
  DrawLine;
  if Menu <> nil then
  begin
    P := Menu^.Items;
    while P <> nil do
    begin
      Color := CNormal;
      if P^.Name = nil then
      begin
        Color := CFrame;
        FrameLine(7, 2);
      end
      else
      begin
        if P^.Disabled then Color := CDisabled else
          if P = Current then Color := CSelect;
        FrameLine(4, 2);
        MoveCStr(B[1], P^.Name^, Color);
        if P^.Disabled then MoveChar(B[1], '-', CMinus, 1);
        if P^.Param <> nil then
          MoveStr(B[Size.X - 2 - Length(P^.Param^)],
            P^.Param^, Byte(Color));
      end;
      DrawLine;
      P := P^.Next;
    end;
  end;
  Color := CFrame;
  FrameLine(10, 1);
  DrawLine;
end;

procedure TMenuBox.GetItemRect(Item: PMenuItem; var R: TRect);
var
  Y: Integer;
  P: PMenuItem;
begin
  Y := 1;
  P := Menu^.Items;
  while P <> Item do
  begin
    Inc(Y);
    P := P^.Next;
  end;
  R.Assign(1, Y, Size.X - 1, Y + 1);
end;

{ TKeyBar }

constructor TKeyBar.Init(var Bounds: TRect; ADefs: PStatusDef);
begin
  TView.Init(Bounds);
  Options := Options or ofPreProcess;
  EventMask := EventMask or evBroadcast;
  GrowMode := gfGrowLoY + gfGrowHiX + gfGrowHiY;
  Defs := ADefs;
  FindItems;
end;

destructor TKeyBar.Done;
var
  T: PStatusDef;

procedure DisposeItems(Item: PStatusItem);
var
  T: PStatusItem;
begin
  while Item <> nil do
  begin
    T := Item;
    Item := Item^.Next;
    DisposeStr(T^.Text);
    Dispose(T);
  end;
end;

begin
  while Defs <> nil do
  begin
    T := Defs;
    Defs := Defs^.Next;
    DisposeItems(T^.Items);
    Dispose(T);
  end;
  TView.Done;
end;

procedure TKeyBar.Draw;
begin
  DrawSelect(nil);
end;

procedure TKeyBar.DrawSelect(Selected: PStatusItem);
var
  B: TDrawBuffer;
  T: PStatusItem;
  C, I, W, X: Integer;
  Color: Word;
  S: string;
begin
  HideMouse;
  Color := GetColor($0201);
  MoveChar(B, ' ', Lo(Color), Size.X);
  T := Items;
  C := 0;
  while T <> nil do
  begin
    Inc(C);
    T := T^.Next;
  end;
  if C > 0 then
  begin
    I := (Size.X mod C) shr 1;
    W := (Size.X div C) - 2;
    T := Items;
    X := 1;
    while T <> nil do
    begin
      Str(X, S);
      if X > 1 then S := ' ' + S;
      MoveStr(B[I], S, Lo(Color));
      Inc(I, Length(S));
      S := '';
      if (T^.Text <> nil) and CommandEnabled(T^.Command) then S := T^.Text^;
      while Length(S) < W do
      begin
        Inc(S[0]);
        S[Length(S)] := ' ';
      end;
      MoveStr(B[I], S, Hi(Color));
      Inc(I, Length(S));
      T := T^.Next;
      Inc(X);
    end;
  end;
  WriteLine(0, 0, Size.X, 1, B);
  ShowMouse;
end;

procedure TKeyBar.FindItems;
var
  P: PStatusDef;
begin
  P := Defs;
  while (P <> nil) and ((HelpCtx < P^.Min) or (HelpCtx > P^.Max)) do
    P := P^.Next;
  if P = nil then Items := nil else Items := P^.Items;
end;

function TKeyBar.GetPalette: PPalette;
const
  P: string[Length(CKeyBar)] = CKeyBar;
begin
  GetPalette := @P;
end;

procedure TKeyBar.HandleEvent(var Event: TEvent);
var
  Mouse: TPoint;
  T: PStatusItem;
  D: PStatusDef;
  F: Boolean;
  I: Integer;
  H: Word;
  Shift, S: Byte;

function ItemNumMouseIsIn: Integer;
var
  C, I, W, X: Integer;
  Z: Integer;
  T: PStatusItem;
begin
  ItemNumMouseIsIn := -1;
  if Mouse.Y <> 0 then Exit;
  T := Items;
  C := 0;
  while T <> nil do
  begin
    Inc(C);
    T := T^.Next;
  end;
  if C > 0 then
  begin
    I := (Size.X mod C) shr 1;
    W := (Size.X div C);
    T := Items;
    Z := 0;
    while T <> nil do
    begin
      if Mouse.X in [I..I + W - 1] then
      begin
        ItemNumMouseIsIn := Z;
        Exit;
      end;
      Inc(I, W);
      Inc(Z);
      T := T^.Next;
    end;
  end;
end;

function GetItem(ItemNum: Integer): PStatusItem;
begin
  if ItemNum < 0 then
  begin
    GetItem := nil;
    Exit;
  end;
  T := Items;
  while (T <> nil) and (ItemNum > 0) do
  begin
    Dec(ItemNum);
    T := T^.Next;
  end;
  GetItem := T;
end;

function ItemMouseIsIn: PStatusItem;
begin
  ItemMouseIsIn := GetItem(ItemNumMouseIsIn);
end;

begin
  TView.HandleEvent(Event);
  case Event.What of
    evMouseDown:
      begin
        MakeLocal(Event.Where, Mouse);
        H := AppHelpCtx;
        Shift := GetShiftState;
        I := ItemNumMouseIsIn;
        repeat
          MakeLocal(Event.Where, Mouse);
          F := False;
          S := GetShiftState;
          if Shift <> S then
          begin
            F := True;
            Shift := S;
          end;
          if S = 0 then
          begin
            if (Event.Buttons and mbRightButton) > 0 then
            begin
              F := True;
              AppHelpCtx := AltHelpCtx;
            end
            else if (Event.Buttons and mbMiddleButton) > 0 then
            begin
              F := True;
              AppHelpCtx := ShiftHelpCtx;
            end;
          end;
          if F then
          begin
            Update;
          end;
          F := False;
          if I = ItemNumMouseIsIn then
          begin
            T := GetItem(I);
            if CommandEnabled(T^.Command) and (T^.Text <> nil) and (T^.Text^ <> '') then
            begin
              F := True;
              SetDefMouseCursorChar(CheckedChar);
            end;
          end;
          if not F then SetDefMouseCursorChar(EmptyMouseCursorChar);
        until not MouseEvent(Event, evMouseMove + evMouseAuto);
        T := nil;
        if I = ItemNumMouseIsIn then T := ItemMouseIsIn;
        AppHelpCtx := H;
        LastShiftState := MaxByte;
        SetDefMouseCursorChar(EmptyMouseCursorChar);
        Update;
        if (T <> nil) and CommandEnabled(T^.Command) then
        begin
          Event.What := evCommand;
          Event.Command := T^.Command;
          PutEvent(Event);
        end;
        ClearEvent(Event);
        DrawView;
      end;
    evKeyDown:
      begin
        F := (AppHelpCtx >= hcNormal);
        D := Defs;
        if F then while (D <> nil) and (D^.Min < hcNormal) do D := D^.Next;
        while D <> nil do
        begin
          if F then T := D^.Items else T := Items;
          while T <> nil do
          begin
            if (Event.KeyCode = T^.KeyCode) and
              CommandEnabled(T^.Command) then
            begin
              Event.What := evCommand;
              Event.Command := T^.Command;
              Exit;
            end;
            T := T^.Next;
          end;
          if F then D := D^.Next else D := nil;
        end;
      end;
    evBroadcast:
      if Event.Command = cmCommandSetChanged then DrawView;
  end;
end;

procedure TKeyBar.Update;
begin
  if TopView <> nil then TopView^.GetHelpCtx;
  if HelpCtx <> AppHelpCtx then
  begin
    HelpCtx := AppHelpCtx;
    FindItems;
    DrawView;
  end;
end;

function NewStatusDef(AMin, AMax: Word; AItems: PStatusItem;
  ANext:PStatusDef): PStatusDef;
var
  T: PStatusDef;
begin
  New(T);
  with T^ do
  begin
    Next := ANext;
    Min := AMin;
    Max := AMax;
    Items := AItems;
  end;
  NewStatusDef := T;
end;

function NewStatusKey(const AText: string; AKeyCode: Word; ACommand: Word;
  ANext: PStatusItem): PStatusItem;
var
  T: PStatusItem;
begin
  New(T);
  T^.Text := NewStr(AText);
  T^.KeyCode := AKeyCode;
  T^.Command := ACommand;
  T^.Next := ANext;
  NewStatusKey := T;
end;

end.
