
{*************************************************}
{                 Joe Forster/STA                 }
{                                                 }
{                    PANEL2.PAS                   }
{                                                 }
{         The Star Commander panel unit #2        }
{*************************************************}

unit Panel2;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Dialogs, Drivers, Objects, Validate, Views,
  Base2, Constant, ExtFiles, LowLevel;

const
{Selection modes}
  smSelect      = 0;
  smUnselect    = 1;
  smInvertSel   = 2;
{Directory allocation flags}
  daNone        = 0;
  daCheck       = 1;
  daAlloc       = 2;
{Allocation flags for validation}
  alFree        = 0;
  alAllocate    = 1;
  alBadSectors  = 2;
{Maximum number of entries in the list of selected files}
  MaxSelNum     = 255;
{Number of bytes necessary for a set of locations}
  LocSetSize    = ((MaxLocations + 7) shr 3);
{Palette of the panel}
  CPanel        = #1#2#3#4#5;
{Name of the temporary input file holding uncompressed archive data}
  ArcTempUncomp : string[12] = '~sc~arc1.tmp';
{Name of the temporary input file}
  ArcTempInput : string[12] = '~sc~arc2.tmp';
{Name of the temporary output file}
  ArcTempOutput : string[12] = '~sc~arc3.tmp';
{Name of the general temporary file}
  TempFileName  : string[12] = '~sc~tmp~.tmp';
{Signature of Convert files}
  ConvertSign   : string[28] = 'PRG formatted GEOS file V1.0';
{Check status characters}
  CheckChr      : array [False..True] of Char = (' ', ' ');
{CBM file types}
  LongCBMExt    : array [0..faTypeMask] of string[10] =
    ('Deleted', 'Sequential', 'Program', 'User', 'Relative', 'Unknown', 'Unknown', 'Frozen');
{GEOS file types}
  LongGEOSExt   : array [1..GEOSTypeNum] of string[14] =
    ('BASIC', 'Assembly', 'Data', 'System', 'Desk accessory', 'Application', 'Document', 'Font',
     'Printer driver', 'Input driver', 'Disk driver', 'System boot', 'Temporary', 'Auto execute');

type
  POldPanel     = ^TOldPanel;
{Speed-search dialog box}
  TFileSearch   = object(TInputLine)
    Panel       : POldPanel;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;
  PFileSearch   = ^TFileSearch;
{Speed-search file name checker}
  TFileValid    = object(TValidator)
    Panel       : POldPanel;
    function IsValidInput(var S: string; SuppressFill: Boolean): Boolean; virtual;
  end;
  PFileValid    = ^TFileValid;
{Radio buttons whose state can also be selected with function keys}
  TFuncButtons  = object(TRadioButtons)
    Min, Max    : Word;
    constructor Init(var Bounds: TRect; AStrings: PSItem; AMin, AMax: Word);
    procedure HandleEvent(var Event: TEvent); virtual;
  end;
  PFuncButtons  = ^TFuncButtons;
{Set of locations}
  TLocationSet  = object
    Bitmap      : array [0.. LocSetSize - 1] of Byte;
    procedure Init(Fill: Boolean);
    procedure Include(Loc: Word);
    procedure Exclude(Loc: Word);
    function Contains(Loc: Word): Boolean;
  end;
  PLocationSet  = ^TLocationSet;
{Panel}
  TOldPanel     = object(TView)
    GEOSFormat,
    CopyGEOSFormat,
    Vis,
    MustRead,
    MustReread,
    HiddenFiles,
    ImageReadOnly,
    MakeListFile,
    BackupDone,
    OK,
    Success,
    DirFull,
    Loading,
    Changing,
    Working,
    Sel,
    Error,
    DirAllocOK,
    FirstFile,
    FirstCopyFile,
    FileOpen,
    SameAsPanel,
    Wildcard,
    OnlyPath,
    LongNamesList,
    FirstBlock,
    _End,
    FCOPYBlock  : Boolean;
    ExtBAMMode,
    DiskType,
    CopyDiskType,
    MaxTrack,
    CopyMaxTrack,
    FirstTrack,
    LastTrack,
    Mode,
    OrigMode,
    NewMode,
    CopyMode,
    SortMode,
    FileFilter,
    QuickView,
    Track,
    Sector,
    SideTrack,
    SideSector,
    SideNumber,
    SideGroup,
    SidePos,
    SideListPos,
    MainDirTrack,
    DirTrack,
    DirTrack2,
    DirSector,
    DirAllocMode,
    CopyPartBlock,
    CopyPartByte,
    Number,
    EntryPos,
    CBMDev,
    CopyCBMDev,
    PanSize,
    ColumnMode,
    DOSNameColWidth,
    CopyDOSNameColWidth,
    CBMNameColWidth,
    CopyCBMNameColWidth,
    ColumnWidth,
    ColumnNum,
    MiniStatus,
    CopyAttr,
    CopyExtAttr,
    CopyRecordLen,
    StartLo,
    StartHi     : Byte;
    DirSep      : Char;
    NameEnd,
    ListEnd,
    DirLength,
    DirPos,
    CopyDirPos,
    ImageSize,
    CopyImageSize,
    Blocks,
    ClusterSize : Word;
    SortDirection,
    PanLen,
    Max,
    Cur,
    SelNum,
    ListNum,
    NewCur,
    DeltaY      : Integer;
    CopyDiskSize,
    LastLynxEnd,
    OrigSize,
    FilesCount,
    DirsCount,
    FileTime,
    Block,
    PrevSize,
    HeaderPos,
    ImagePos,
    CopyShortFree,
    CopyLongFree: Longint;
    _DiskSize,
    _DiskFree,
    _Free,
    CopyFree,
    FilesSize,
    FilesSizeReal: Longlongint;
    FSDialog    : PDialog;
    FileSearch  : PFileSearch;
    CopyID      : string[CBMBAMIDLen];
    _Label,
    CopyShortLabel,
    CopyLabel   : string[24];
    TapeName    : string[32];
    PanelTitle,
    BackupName,
    Top,
    Under,
    NamePattern,
    OrigNamePattern,
    ImageName,
    CopyName,
    CopyImageName,
    CopyImageProtoName,
    OrigImageName,
    Path,
    CopyPath,
    ImagePath,
    RealImagePath,
    CopyImagePath,
    CopyRealImagePath,
    OrigPath,
    OrigImagePath,
    CopyRealPath,
    CopyFullName: string;
    SortMenu    : TSortMenu;
    ModeMenu    : TModeMenu;
    BAM,
    BAM2,
    BAM3,
    DirBuffer,
    SideBuffer  : TBlock;
    Dir         : TDirBuffer;
    Names       : TNameBuffer;
    ListNames   : TListBuffer;
    CopyEntry   : ExtSearchRec;
    Image       : ExtFile;
    List        : array [0..MaxFiles - 1] of TSelFile;
    constructor Init(Bounds: TRect);
    function GetNamePtr(Pos: Integer): PString;
    function InsName(const Name: string): Integer;
    function GetListPtr(Pos: Integer): PString;
    function InsListName(const Name: string): Integer;
    function GetMiniStatus: Byte;
    function GetColumnMode: Byte;
    function GetSortMode: Byte;
    function GetReverseMode: Boolean;
    function GetPalette: PPalette; virtual;
    procedure SearchName;
    function SearchPattern(const Pattern: string; From: Integer; Jump: Boolean): Boolean;
    procedure SetPanelSize;
    procedure SetModeMenu;
    procedure SortDirectory;
    procedure MakeDirSep;
    procedure MakeRealPath;
    procedure CorrectName;
    procedure CorrectImageName(AnyDiskType: Boolean);
    procedure CheckPath;
    procedure MakeFullName;
    procedure MakeName;
    procedure ClearBAM(Alloc: Boolean);
    procedure MakeBAM;
    procedure SetBAMLabel(const Name: string);
    procedure ReadBAM(Errors: Boolean);
    procedure WriteBAM;
    function ValidPos(T, S: Byte): Boolean;
    function SectorNum(T: Byte): Byte;
    function DiskPos(T, S: Byte): Longint;
    function GetBAMOffset(T: Byte; Map: Boolean): Word;
    function ReadDiskBlock(T, S: Byte; Buffer: PBlock; Errors: Boolean): Byte;
    procedure WriteDiskBlock(T, S: Byte; Buffer: PBlock);
    procedure ReadTapeBlock(S: Word; Buffer: PBlock);
    procedure WriteTapeBlock(S: Word; Buffer: PBlock);
    procedure AllocBlock(T, S: Byte; Alloc: Boolean);
    function IsBlockUsed(T, S: Byte): Boolean;
    function IsTrackFree(T: Byte): Boolean;
    function IsTrackEmpty(T: Byte): Boolean;
    function IsBAMValid: Boolean;
    procedure AllocCopyBlock(Found: Boolean);
    function FirstCopyBlock: Boolean;
    function NextCopyBlock: Boolean;
    function ClearTrackMap(Alloc: Boolean): Boolean;
    function NextSector(Step: Boolean): Boolean;
    function MakeTempName: string;
    procedure MakeBackupFile(B: Boolean);
    procedure CheckDiskType;
    function ReadCBMEntry(var Entry: TDirEntry): Boolean;
    procedure CheckDirAlloc(T, S: Byte);
    procedure CheckGEOSFormat(P: PBlock);
    function InitArchive: Integer;
    function OpenImage(Write, OnlyBAM, GetBAM, DetectDisk, Errors: Boolean): Integer;
    procedure CloseImage(Write: Boolean);
    function Validate(Alloc: Byte; Part, VLIR, Force: Boolean; PartSize: Word): Boolean;
    function OpenZipFile(var ZipFile: ExtFile; Pos: Word; Start: Longint; FileMode: Byte): Integer;
    procedure ShutImageMode;
    procedure OutOfMem;
    procedure ShutQuickView;
    procedure DrawPanel;
  end;

var
  ContProcess,
  AppendFile    : Boolean;
  ManualSelect  : PLocationSet;

procedure DisplayProgress;

implementation

uses
  App, DOS, Menus,
  Base1;

{Update the file copy progress indicator}
procedure DisplayProgress;
var
  P             : Longint;
  S             : string[4];
begin
  if CopyInd <> nil then
  begin
    if ExtDrive then
    begin
      S := LeadingSpace(CopiedBlock, 4);
      Move(S[1], CopyInd^.Text^[7], Length(S));
    end
    else
    begin
      if IndicatorLen > 0 then
      begin
        P := CopySize div IndicatorLen;
        if P = 0 then
        begin
          if CopySize > 0 then P := (CopiedSize * IndicatorLen) div CopySize;
        end
        else
        begin
          P := CopiedSize div P;
        end;
        if P > IndicatorLen then P := IndicatorLen;
        FillChar(CopyInd^.Text^[1], P, SelectionChars[3]);
      end;
    end;
    CopyInd^.Draw;
  end;
end;

{Initialize the panel
  Input : Bounds: bounds of the panel}
constructor TOldPanel.Init(Bounds: TRect);
begin
  TView.Init(Bounds);
  OK := False;
  GrowMode := 0;
  Vis := False;
  LongNamesList := LongNames;
  DOSNameColWidth := DefDOSNameColWidth;
  CBMNameColWidth := DefCBMNameColWidth;
end;

{Get pointer to a given file name
  Input : Pos: the position of the file name}
function TOldPanel.GetNamePtr(Pos: Integer): PString;
begin
  GetNamePtr := PString(@Names[Dir[Pos].Name]);
end;

{Insert a file name into the file name table
  Input : the file name to insert
  Output: when nonzero, indicates the offset where the file name was stored;
          otherwise there was not enough room to insert the file name}
function TOldPanel.InsName(const Name: string): Integer;
begin
  if NameEnd + Length(Name) <= SizeOf(TNameBuffer) then
  begin
    InsName := NameEnd;
    Move(Name, Names[NameEnd], Length(Name) + 1);
    Inc(NameEnd, Length(Name) + 1);
  end
  else
  begin
    InsName := -1;
    NameEnd := SizeOf(TNameBuffer);
  end;
end;

{Get pointer to a given file name in the list of select files
  Input : Pos: the position of the file name}
function TOldPanel.GetListPtr(Pos: Integer): PString;
begin
  GetListPtr := PString(@ListNames[List[Pos].Name]);
end;

{Insert a file name into the table of selected file names
  Input : the file name to insert
  Output: when nonzero, indicates the offset where the file name was stored;
          otherwise there was not enough room to insert the file name}
function TOldPanel.InsListName(const Name: string): Integer;
begin
  if ListEnd + Length(Name) <= SizeOf(TListBuffer) then
  begin
    InsListName := ListEnd;
    Move(Name, ListNames[ListEnd], Length(Name) + 1);
    Inc(ListEnd, Length(Name) + 1);
  end
  else
  begin
    InsListName := -1;
    ListEnd := SizeOf(TListBuffer);
  end;
end;

{Get Brief/Full/Wide mode for the current DOS/CBM mode
  Output: Brief/Full/Wide mode}
function TOldPanel.GetColumnMode: Byte;
begin
  if Mode = pmDOS then GetColumnMode := ColumnMode and cmModeMask else GetColumnMode := (ColumnMode shr 2) and cmModeMask;
end;

{Get sort mode for the current DOS/CBM mode
  Output: sort mode}
function TOldPanel.GetSortMode: Byte;
begin
  if Mode = pmDOS then GetSortMode := SortMode and psModeMask else GetSortMode := (SortMode shr 3) and psModeMask;
end;

{Get reverse mode for the current DOS/CBM mode
  Output: when True, reverse mode is on}
function TOldPanel.GetReverseMode: Boolean;
begin
  if Mode = pmDOS then GetReverseMode := ((SortMode and psReverse) > 0)
    else GetReverseMode := (((SortMode shr 1) and psReverse) > 0);
end;

{Get the mini status mode for the current DOS/CBM mode
  Output: mini status mode}
function TOldPanel.GetMiniStatus: Byte;
begin
  if Mode = pmDOS then GetMiniStatus := MiniStatus and msModeMask else GetMiniStatus := (MiniStatus shr 2) and msModeMask;
end;

{Get the palette of the panel
  Output: string containing the palette}
function TOldPanel.GetPalette: PPalette;
const
  P: string[Length(CPanel)] = CPanel;
begin
  GetPalette := @P;
end;

{Move the cursor back to the file name it stood on before}
procedure TOldPanel.SearchName;
var
  P             : TCaseProc;
begin
  SetPanelSize;
  if Mode = pmDOS then P := UpperCase else P := CBMLowerCase;
  Cur := 0;
  while (Cur < Max) and (P(Top) <> P(GetNamePtr(Cur)^)) do Inc(Cur);
  if Cur < Max then DeltaY := Cur;
  if (NewCur < Max) and (P(Under) = P(GetNamePtr(NewCur)^)) then
  begin
    Cur := NewCur;
  end
  else
  begin
    Cur := 0;
    while (Cur < Max) and (P(Under) <> P(GetNamePtr(Cur)^)) do Inc(Cur);
  end;
  if Cur < Max then
  begin
    if Cur < DeltaY then DeltaY := Cur;
    if Cur > DeltaY + PanLen - 1 then DeltaY := Cur - PanLen + 1;
    if DeltaY + PanLen > Max then if Max > PanLen - 1 then DeltaY := Max - PanLen else DeltaY := 0;
  end
  else
  begin
    Cur := 0;
    DeltaY := 0;
  end;
end;

{Search for a file name that matches the specified wildcard pattern
  Input : Pattern: string containing the pattern to search for
          From: number of entry to start searching from
          Jump: when True, the cursor bar jumps to the matching entry
  Output: when True, the entry was found}
function TOldPanel.SearchPattern(const Pattern: string; From: Integer; Jump: Boolean): Boolean;
var
  B,
  F,
  O             : Boolean;
  Y             : Integer;
  N             : string;
begin
  F := False;
  B := (Mode = pmDOS);
  if Max > 0 then
  begin
    Y := From;
    O := True;
    N := Pattern;
    while ((Y <> Cur) or O) and not F do
    begin
      if B then F := CompareDOSEntry(N, GetNamePtr(Y)^, LongFileNames, True) else
        F := CompareCBMEntry(N, GetNamePtr(Y)^, Dir[Y].Attr, True);
      if not F then
      begin
        if Y = Cur then
        begin
          if O then Inc(Y);
          O := False;
        end
        else
        begin
          Inc(Y);
        end;
        if Y = Max then Y := 0;
      end;
    end;
    if F and Jump then
    begin
      Cur := Y;
      Y := PanSize shr 1;
      if Cur < DeltaY + Y then if Cur > Y then DeltaY := Cur - Y else DeltaY := 0;
      if Cur > DeltaY + PanLen - Y - 1 then DeltaY := Cur - PanLen + Y + 1;
      if DeltaY + PanLen > Max then if Max > PanLen - 1 then DeltaY := Max - PanLen else DeltaY := 0;
      DrawPanel;
    end;
  end;
  SearchPattern := F;
end;

{Set the panel size and the number of files displayed in the panel at a
  time}
procedure TOldPanel.SetPanelSize;
var
  B,
  C             : Byte;
  I,
  J,
  X             : Integer;
begin
  PanSize := PanWinSize - 3 - GetMiniStatus;
  if GetMiniStatus <> msOff then Dec(PanSize);
  C := 0;
  if Mode = pmDOS then
  begin
    B := CopyDOSNameColWidth;
    case GetColumnMode of
      cmBrief: I := B;
      cmFull, cmWide: I := B + 26;
    end;
  end
  else
  begin
    CopyCBMNameColWidth := CBMNameColWidth;
    X := Size.X - 10;
    if GetColumnMode = cmBrief then Inc(X, 8);
    if CopyCBMNameColWidth > X then CopyCBMNameColWidth := X;
    B := CopyCBMNameColWidth;
    case GetColumnMode of
      cmBrief: I := B;
      cmFull, cmWide:
      begin
        C := 7;
        I := B + 14;
        if StartInfo then Inc(I, 6);
      end;
    end;
  end;
  Inc(I);
  ColumnWidth := I;
  X := Size.X - 2;
  J := X mod I;
  I := X div I;
  if J >= B + C then Inc(I);
  if I < 1 then I := 1;
  ColumnNum := I;
  PanLen := PanSize * I;
end;

{Mark the menu item linked to the current Brief/Full/Wide/Info/Quick View
  mode}
procedure TOldPanel.SetModeMenu;
var
  I,
  J             : Byte;
  C             : Char;
begin
  if QuickView <> qvNone then J := 4 else if Mode = pmInfo then J := 3 else J := GetColumnMode;
  CheckChr[True] := CheckedChar;
  for I := 0 to 4 do ModeMenu[I]^.Name^[1] := CheckChr[I = J];
end;

{Sort the directory of the panel (using the Shell-Metzner algorithm)}
procedure TOldPanel.SortDirectory;
var
  B,
  F,
  O             : Boolean;
  M             : Byte;
  I,
  J,
  K,
  S             : Integer;
  Q             : PPanelEntry;
  P             : TPanelEntry;
  T,
  U             : PString;

{Compare two DOS directory entries for the sort routine}
function DOSCompare: Boolean;
var
  I             : Integer;
  E1,
  E2,
  N1,
  N2            : string;
begin
  SplitName(T^, N1, E1);
  SplitName(U^, N2, E2);
  case M of
    psName:
    begin
      I := CompareString(N1, N2);
      if I = 0 then I := CompareString(E1, E2);
    end;
    psExt:
    begin
      I := CompareString(E1, E2);
      if I = 0 then I := CompareString(N1, N2);
    end;
    psTime:
    begin
      if P.Time > Q^.Time then
      begin
        I := -1;
      end
      else
      begin
        if P.Time = Q^.Time then
        begin
          I := CompareString(N1, N2);
          if I = 0 then I := CompareString(E1, E2);
        end
        else
        begin
          I := 1;
        end;
      end;
    end;
    psSize:
    begin
      I := CompLonglongint(Q^.Size, P.Size);
      if I = 0 then
      begin
        I := CompareString(N1, N2);
        if I = 0 then I := CompareString(E1, E2);
      end;
    end;
  end;
  DOSCompare := (I <> SortDirection);
end;

{Compare two CBM directory entries for the sort routine}
function CBMCompare: Boolean;
var
  I             : Integer;
  T1,
  T2            : TFileExtStr;
  N1,
  N2            : string;
begin
  N1 := T^;
  N2 := U^;
  if GEOSFormat then T1 := ShortGEOSExt[P.ExtAttr and xaTypeMask] else T1 := ShortCBMExt[P.Attr and faTypeMask];
  if GEOSFormat then T2 := ShortGEOSExt[Q^.ExtAttr and xaTypeMask] else T2 := ShortCBMExt[Q^.Attr and faTypeMask];
  case M of
    psName, psTime:
    begin
      I := CompareString(N1, N2);
      if I = 0 then I := CompareString(T1, T2);
    end;
    psExt:
    begin
      I := CompareString(T1, T2);
      if I = 0 then I := CompareString(N1, N2);
    end;
    psSize:
    begin
      I := CompLonglongint(Q^.Size, P.Size);
      if I = 0 then
      begin
        I := CompareString(N1, N2);
        if I = 0 then I := CompareString(T1, T2);
      end;
    end;
  end;
  CBMCompare := (I <> SortDirection);
end;

begin
  M := GetSortMode;
  if GetReverseMode then SortDirection := 1 else SortDirection := -1;
  B := (M <> psUnsorted);
  F := (((Mode in [pmExt, pmDisk]) and (DiskType = dt1581)) or GetPanelModeAttrib(Mode, paDirectories));
  S := Max shr 1;
  while S > 0 do
  begin
    for I := 0 to S - 1 do
    begin
      J := I + S;
      while J < Max do
      begin
        P := Dir[J];
        T := GetNamePtr(J);
        K := J - S;
        O := False;
        while not O and (K >= 0) do
        begin
          if B then
          begin
            Q := @Dir[K];
            U := GetNamePtr(K);
            if Mode = pmDOS then
            begin
              O := ((T^ <> '..') and (U^ = '..'));
              if not O then
              begin
                O := (P.Attr and Directory < Q^.Attr and Directory);
                if not O and (P.Attr and Directory = Q^.Attr and Directory) then
                begin
                  if P.Attr and Directory > 0 then
                  begin
                    O := DOSCompare;
                  end
                  else
                  begin
                    O := ((P.Attr and (Hidden + SysFile) > 0) < (Q^.Attr and (Hidden + SysFile) > 0));
                    if not O and ((P.Attr and (Hidden + SysFile) > 0) = (Q^.Attr and (Hidden + SysFile) > 0)) then
                      O := DOSCompare;
                  end;
                end;
              end;
            end
            else
            begin
              O := ((Mode <> pmExt) and (J > 0) and (K = 0));
              if F then
              begin
                if not O then O := ((P.Attr and faTypeMask = faPartition) < (Q^.Attr and faTypeMask = faPartition));
                if not O and ((P.Attr and faTypeMask = faPartition) = (Q^.Attr and faTypeMask = faPartition)) then
                  O := CBMCompare;
              end
              else
              begin
                if not O then O := CBMCompare;
              end;
            end;
          end
          else
          begin
            O := Dir[K].Name <= Dir[J].Name;
          end;
          if not O then
          begin
            Dir[K + S] := Dir[K];
            Dec(K, S);
          end;
        end;
        if not O then K := I - S;
        Dir[K + S] := P;
        Inc(J, S);
      end;
    end;
    S := S shr 1;
  end;
end;

{Determine the directory separator character}
procedure TOldPanel.MakeDirSep;
begin
  DirSep := '/';
  if GetPanelModeAttrib(CopyMode, paBackslash) then DirSep := chDirSep;
end;

{Expand the path into a fully qualified path}
procedure TOldPanel.MakeRealPath;
var
  I             : Integer;
  S             : string;
begin
  if CopyMode = pmExt then
  begin
    S := LeadingSpace(CopyCBMDev, 0);
    CopyRealPath := Copy(S, Length(S), 1);
    S := CopyPath;
    if (S[1] in ['0'..'9']) and (S[2] = ':') then S := Copy(S, 3, MaxStrLen);
    CopyRealPath := CopyRealPath + ':' + S;
  end
  else
  begin
    I := IOResult;
    CopyRealPath := LongFExpand(AddToPath(CopyPath, '', chDirSep));
    InOutRes := I;
    if (Length(CopyRealPath) > 3) and (CopyRealPath[Length(CopyRealPath)] = chDirSep) then
      Dec(CopyRealPath[0]);
  end;
end;

{Correct the name of a DOS file}
procedure TOldPanel.CorrectName;
begin
  if (CopyMode = pmDOS) and not LongFileNames then CopyName := CorrectDOSName(CopyName, False);
end;

{Correct the name of an image or archive file
  Input : AnyDiskType: when True, the extension of a disk image is not needed,
                       the first disk image with that name will be used}
procedure TOldPanel.CorrectImageName(AnyDiskType: Boolean);
var
  O             : Boolean;
  B             : Byte;
  S             : string;
begin
  CopyImageName := CopyImageProtoName;
  case CopyMode of
    pmDisk, pmTape, pmLynx..pmZIP, pmGCRDisk:
    begin
      S := LowerCase(FileExt(CopyImageName));
      O := False;
      if CopyMode = pmDisk then
      begin
        if AnyDiskType then
        begin
          O := IsDiskExt(S);
          if not O and FileExists(CopyPath, True) then
          begin
            B := 0;
            while not O and (B < DiskTypeNum) do
            begin
              S := MakeFileExt(CopyImageName, GetDiskExt(B));
              if FileExists(S, False) then
              begin
                CopyImageName := S;
                CopyDiskType := B;
                O := True;
              end;
              Inc(B);
            end;
          end;
        end
        else
        begin
          CopyImageName := MakeFileExt(CopyImageName, GetDiskExt(CopyDiskType));
          O := True;
        end;
      end;
      if not O and (S <> DOSExt[CopyMode]) and not ((CopyMode = pmLHA) and ((S = 'lha') or (S = 'sfx'))) then
        CopyImageName := MakeFileExt(CopyImageName, DOSExt[CopyMode]);
    end;
    pmFileZip:
    begin
      if (Length(CopyImageName) > 2) and (LoCase(CopyImageName[1]) in ['a'..'x']) and (CopyImageName[2] = '!') then
        CopyImageName := Copy(CopyImageName, 3, MaxStrLen);
      CopyImageName := 'x!' + CopyImageName;
    end;
    pmDiskZip:
    begin
      if (Length(CopyImageName) > 2) and (CopyImageName[1] in ['0'..'9']) and (CopyImageName[2] = '!') then
        CopyImageName := Copy(CopyImageName, 3, MaxStrLen);
      CopyImageName := '1!' + CopyImageName;
    end;
    pmSixZip:
    begin
      if (Length(CopyImageName) > 3) and (CopyImageName[1] in ['0'..'9']) and (CopyImageName[2] = '!')
        and (CopyImageName[3] = '!') then
        CopyImageName := Copy(CopyImageName, 4, MaxStrLen);
      CopyImageName := '1!!' + CopyImageName;
    end;
  end;
  if (Length(CopyPath) >= 2) and (CopyPath[2] = ':') and (UpCase(CopyPath[1]) in ['A'..'Z']) then
    CopyPath[1] := UpCase(CopyPath[1]);
  MakeFullName;
end;

{Check the presence of the destination path and fix it, if needed}
procedure TOldPanel.CheckPath;
var
  E,
  N,
  P             : string;
begin
  if FirstFile and not OnlyPath then
  begin
    if FileExists(CopyPath, True) then
    begin
      if CopyMode = pmDOS then SameAsPanel := False;
      CopyPath := LongName(AddToPath(CopyPath, stEmpty, chDirSep), True);
      MakeRealPath;
    end
    else
    begin
      NamePattern := CutPath(CopyPath, chDirSep);
      if not GetPanelModeAttrib(CopyMode, paASCII) then
        NamePattern := ReconvertCBMName(NamePattern, CopyGEOSFormat, False, hxPercent);
      CopyPath := LongName(AddToPath(GetPath(CopyPath, chDirSep), stEmpty, chDirSep), True);
      MakeRealPath;
      MakeName;
    end;
  end;
  CorrectName;
  CopyFullName := AddToPath(CopyPath, CopyName, chDirSep);
end;

{Expand the file name into a fully qualified file name}
procedure TOldPanel.MakeFullName;
var
  S,
  T             : string;
begin
  if CopyMode = pmDOS then
  begin
    CorrectName;
    T := CopyName;
    if CopyAttr and Directory > 0 then T := AddToPath(T, stEmpty, chDirSep);
    S := LongName(T, False);
    if not LongFileNames or (UpperCase(S) <> UpperCase(CopyName)) then CopyName := S;
    CopyFullName := AddToPath(CopyPath, CopyName, chDirSep);
  end
  else
  begin
    CopyImageName := LongName(CopyImageName, False);
    if GetPanelModeAttrib(CopyMode, paASCII) then
    begin
      S := CopyName;
    end
    else
    begin
      if Length(CopyName) > CBMNameLen then CopyName[0] := Chr(CBMNameLen);
      S := MakeCBMName(CopyName, CopyGEOSFormat);
    end;
    if SameAsPanel then
    begin
      CopyFullName := S;
    end
    else
    begin
      if CopyMode = pmExt then
      begin
        if Length(CopyPath) > 3 then CopyPath := Copy(CopyPath, 1, 2) + chDirSep;
        CopyFullName := AddToPath(CopyPath, MakeCBMName(CopyName, CopyGEOSFormat), chDirSep);
      end
      else
      begin
        CopyFullName := AddToPath(ConvertCBMPath(CopyImagePath, CopyGEOSFormat, True, hxNone, chDirSep), stEmpty, chDirSep);
        CopyFullName := MakeTypeStr(CopyMode) + ':' + AddToPath(CopyPath, CopyImageName, chDirSep) +
          chDirSep + CopyFullName + S;
      end;
    end;
  end;
  MakeRealPath;
end;

{Clone the file name using the original file name and the cloning pattern}
procedure TOldPanel.MakeName;
begin
  if CopyMode = pmDOS then CopyName := CloneDOSName(CopyName, NamePattern) else
    CopyName := CloneName(CopyName, NamePattern, True, True);
  CorrectImageName(True);
end;

{Create an empty BAM in the BAM buffer
  Input : Alloc: when True, all sectors are allocated; otherwise freed}
procedure TOldPanel.ClearBAM(Alloc: Boolean);
var
  B             : Boolean;
  P,
  Q             : Byte;
  W             : Word;
  L             : Longint;
begin
  B := (CopyDiskType and dtTypeMask = dt1581);
  for P := 1 to CopyMaxTrack - 1 do
  begin
    W := 0;
    L := 0;
    if not Alloc and (P >= FirstTrack) and (P <> DirTrack2) and (P < LastTrack) then
    begin
      W := SectorNum(P);
      for Q := 1 to W do L := (L shl 1) or 1;
    end;
    BAM[GetBAMOffset(P, False)] := W;
    W := GetBAMOffset(P, True);
    BAM[W] := L;
    BAM[W + 1] := L shr 8;
    BAM[W + 2] := L shr 16;
    if B then
    begin
      BAM[W + 3] := L shr 24;
      BAM[W + 4] := L shr 32;
    end;
  end;
end;

{Create an empty BAM sector in the BAM buffer}
procedure TOldPanel.MakeBAM;
var
  B,
  C,
  D             : Byte;
begin
  FillChar(BAM, 3 * 256, 0);
  C := NameOffset(CopyDiskType, ExtBAMMode);
  D := $1A;
  B := CopyDiskType and dtTypeMask;
  BAM[0] := DirTrack;
  BAM[1] := FirstDirSec(CopyDiskType);
  if B = dt1581 then
  begin
    D := $18;
    BAM[2] := Ord('D');
    BAM2[0] := DirTrack;
    BAM2[1] := 2;
    BAM2[2] := Ord('D');
    BAM2[3] := Byte(not Ord('D'));
    BAM2[6] := $C0;
    BAM3[0] := 0;
    BAM3[1] := $FF;
    BAM3[2] := Ord('D');
    BAM3[3] := Byte(not Ord('D'));
    BAM3[6] := $C0;
  end
  else
  begin
    if (CopyDiskType and dtTypeMask = dt1541Ext) and (ExtBAMMode = xbPrologicDOS) then BAM[2] := Ord('P') else
      BAM[2] := Ord('A');
  end;
  if B = dt1571 then BAM[3] := $80;
  for B := 0 to D do BAM[B + C] := $A0;
  ClearBAM(False);
end;

{Set the disk label in the BAM buffer
  Input : Name: new disk label}
procedure TOldPanel.SetBAMLabel(const Name: string);
var
  O             : Boolean;
  P,
  Q             : Byte;
  I             : string[CBMBAMIDLen];
  L             : string[CBMNameLen];
  N             : string[25];
begin
  P := CopyDiskType and dtTypeMask;
  N := CorrectBAMLabel(Name, P, ExtBAMMode);
  L := Copy(N, 1, CBMNameLen);
  I := Copy(N, CBMNameLen + 2, CBMBAMIDLen);
  Q := NameOffset(P, ExtBAMMode);
  O := (P = dt1581);
  for P := 0 to CBMNameLen - 1 do BAM[P + Q] := Ord(L[P + 1]);
  Inc(Q, DiskIDRelPos);
  for P := 0 to CBMBAMIDLen - 1 do BAM[P + Q] := Ord(I[P + 1]);
  if O then
  begin
    BAM2[$04] := Ord(I[1]);
    BAM2[$05] := Ord(I[2]);
    BAM3[$04] := Ord(I[1]);
    BAM3[$05] := Ord(I[2]);
  end;
end;

{Read the BAM from the disk image
  Input : Errors: when True, errors are displayed}
procedure TOldPanel.ReadBAM(Errors: Boolean);
begin
  FillChar(BAM, 3 * 256, 0);
  case CopyDiskType and dtTypeMask of
    dt1541, dt1541Ext: ReadDiskBlock(DirTrack, 0, @BAM, Errors);
    dt1571:
    begin
      ReadDiskBlock(DirTrack, 0, @BAM, Errors);
      ReadDiskBlock(DirTrack2, 0, @BAM2, Errors);
    end;
    dt1581:
    begin
      ReadDiskBlock(DirTrack, 0, @BAM, Errors);
      ReadDiskBlock(DirTrack, 1, @BAM2, Errors);
      ReadDiskBlock(DirTrack, 2, @BAM3, Errors);
    end;
  end;
end;

{Write the BAM into the disk image}
procedure TOldPanel.WriteBAM;
begin
  case CopyDiskType and dtTypeMask of
    dt1541, dt1541Ext: WriteDiskBlock(DirTrack, 0, @BAM);
    dt1571:
    begin
      WriteDiskBlock(DirTrack, 0, @BAM);
      WriteDiskBlock(DirTrack2, 0, @BAM2);
    end;
    dt1581:
    begin
      WriteDiskBlock(DirTrack, 0, @BAM);
      WriteDiskBlock(DirTrack, 1, @BAM2);
      WriteDiskBlock(DirTrack, 2, @BAM3);
    end;
  end;
end;

{Determine if a disk position is valid
  Input : T, S; the track and sector number
  Output: when True, the position is valid}
function TOldPanel.ValidPos(T, S: Byte): Boolean;
begin
  ValidPos := ((T < LastTrack) and ((T = 0) or (S < SectorNum(T))));
end;

{Get the number of sectors on the specified track
  Input : T: the track number
  Output: the number of sectors on the track}
function TOldPanel.SectorNum(T: Byte): Byte;
var
  B             : Byte;
begin
  B := CopyDiskType and dtTypeMask;
  if B = dt1581 then
  begin
    SectorNum := 40;
  end
  else
  begin
    case T of
      1..17: SectorNum := 21;
      18..24: SectorNum := 19;
      25..30: SectorNum := 18;
      31..42: SectorNum := 17;
    end;
    if B = dt1571 then
    begin
      case T of
        Max1541Tracks..52: SectorNum := 21;
        53..59: SectorNum := 19;
        60..65: SectorNum := 18;
        66..70: SectorNum := 17;
      end;
    end;
  end;
end;

{Compute the offset of the specified block in a disk image
  Input : T, S: the track and sector number of the block
  Output: the offset of the block divided by 256}
function TOldPanel.DiskPos(T, S: Byte): Longint;
var
  I             : Integer;
  P             : Longint;
begin
  P := 0;
  for I := 1 to T - 1 do Inc(P, SectorNum(I));
  Inc(P, S);
  DiskPos := P;
end;

{Get the offset of the BAM entry
  Input : T: track number
          Map: when True, the offset of the bitmap of sectors on the track
               is requested, otherwise that of the free sector counter
  Output: offset of BAM entry}
function TOldPanel.GetBAMOffset(T: Byte; Map: Boolean): Word;
var
  W             : Word;
begin
  W := T shl 2;
  if Map then Inc(W);
  case CopyDiskType and dtTypeMask of
    dt1541Ext:
    begin
      if T >= Max1541Tracks then
      begin
        case ExtBAMMode of
          xbSpeedDOS: Inc(W, $30);
          xbDolphinDOS: Inc(W, $1C);
        end;
      end;
    end;
    dt1571: if T >= Max1541Tracks then if Map then W := (T - Max1541Tracks) * 3 + 256 else W := T + $B9;
    dt1581:
    begin
      W := T * 6;
      if Map then Inc(W);
      if T <= 40 then Inc(W, $010A) else Inc(W, $011A);
    end;
  end;
  GetBAMOffset := W;
end;

{Read the specified block from the disk image into the given buffer
  Input : T, S: track and sector number of the block
          Buffer: buffer to read the block into
          Errors: when True, errors are displayed
  Output: error code attached to sector}
function TOldPanel.ReadDiskBlock(T, S: Byte; Buffer: PBlock; Errors: Boolean): Byte;
var
  O             : Boolean;
  B,
  C,
  D             : Byte;
  W,
  X             : Word;
  E             : string;
begin
  B := dsOK;
  W := DiskPos(T, S);
  case CopyMode of
    pmDisk:
    begin
      if ShowReadErrors and (CopyDiskType and dtErrorInfo > 0) then
      begin
        ExtSeek(Image, (DiskPos(CopyMaxTrack, 0) shl 8) + W);
        ExtBlockRead(Image, B, 1);
      end;
      ExtSeek(Image, Longint(W) shl 8);
      ExtBlockRead(Image, Buffer^, 256);
    end;
    pmDiskZip:
    begin
      C := DiskZipFileNum - 1;
      while W < DiskZipBlocks[C] do Dec(C);
      O := False;
      if OpenZipFile(Image, W, 0, fmReadOnly) = 0 then
      begin
        O := True;
        repeat
          O := ReadZipCodeBlock(Image, Buffer, False, zrUncompress, C, D, X);
        until not O or ((C = T) and (D = S));
      end;
      if not O then
      begin
        B := ds20READ;
        FillChar(Buffer^, 256, 0);
      end;
    end;
  end;
  if Errors and (B <> dsOK) and MakeErrorStr(B, T, S, E) then ErrorWin(stError, E, stEmpty, hcOnlyQuit, sbSkip + sbSkipAll);
  ReadDiskBlock := B;
end;

{Write the contents of the given buffer into the specified block of the
  disk image
  Input : T, S: track and sector number of the block
          Buffer: buffer to write the block from}
procedure TOldPanel.WriteDiskBlock(T, S: Byte; Buffer: PBlock);
var
  B             : Byte;
  W             : Word;
begin
  W := DiskPos(T, S);
  case CopyMode of
    pmDisk:
    begin
      if CopyDiskType and dtErrorInfo > 0 then
      begin
        ExtSeek(Image, (DiskPos(CopyMaxTrack, 0) shl 8) + W);
        B := dsOK;
        ExtBlockWrite(Image, B, 1);
      end;
      ExtSeek(Image, Longint(W) shl 8);
      ExtBlockWrite(Image, Buffer^, 256);
    end;
  end;
end;

{Read the specified block from the tape image into the given buffer
  Input : S: number of the block
          Buffer: buffer to read the block into}
procedure TOldPanel.ReadTapeBlock(S: Word; Buffer: PBlock);
begin
  ExtSeek(Image, S shl 5);
  ExtBlockRead(Image, Buffer^, 32);
end;

{Write the contents of the given buffer into the specified block of the
  tape image
  Input : S: number of the block
          Buffer: buffer to write the block from}
procedure TOldPanel.WriteTapeBlock(S: Word; Buffer: PBlock);
begin
  ExtSeek(Image, S shl 5);
  ExtBlockWrite(Image, Buffer^, 32);
end;

{Allocate or free a block
  Input : T, S: track and sector number of the block
          Alloc: when True, the block is allocated; otherwise freed}
procedure TOldPanel.AllocBlock(T, S: Byte; Alloc: Boolean);
var
  P             : Byte;
  W,
  X             : Word;
begin
  P := 1 shl (S and 7);
  W := GetBAMOffset(T, True) + (S shr 3);
  if Alloc = (BAM[W] and P > 0) then
  begin
    X := GetBAMOffset(T, False);
    if Alloc then
    begin
      Dec(BAM[X]);
      BAM[W] := BAM[W] and not P;
    end
    else
    begin
      Inc(BAM[X]);
      BAM[W] := BAM[W] or P;
    end;
  end;
end;

{Check if the specified block is allocated
  Input : T, S: track and sector number of the block
  Output: when True, the block is allocated, otherwise free}
function TOldPanel.IsBlockUsed(T, S: Byte): Boolean;
begin
  IsBlockUsed := (BAM[GetBAMOffset(T, True) + (S shr 3)] and (1 shl (S and 7)) = 0);
end;

{Check if there is at least one block free on the specified track
  Input : T: track number
  Output: when True, there is at least one block free on the track}
function TOldPanel.IsTrackFree(T: Byte): Boolean;
begin
  IsTrackFree := (BAM[GetBAMOffset(T, False)] > 0);
end;

{Check if the specified track is completely empty
  Input : T: track number
  Output: when True, the track is empty}
function TOldPanel.IsTrackEmpty(T: Byte): Boolean;
begin
  IsTrackEmpty := (BAM[GetBAMOffset(T, False)] = SectorNum(T));
end;

{Check if the BAM is valid
  Output: when True, the BAM is valid}
function TOldPanel.IsBAMValid: Boolean;
var
  O             : Boolean;
  F,
  T,
  S             : Byte;
begin
  O := True;
  T := 1;
  while O and (T < CopyMaxTrack) do
  begin
    F := 0;
    for S := 0 to SectorNum(T) - 1 do if not IsBlockUsed(T, S) then Inc(F);
    if F <> BAM[GetBAMOffset(T, False)] then O := False;
    Inc(T);
  end;
  IsBAMValid := O;
end;

{Allocate the new block of the file
  Input : Found: when True, a free block was successfully found}
procedure TOldPanel.AllocCopyBlock(Found: Boolean);
begin
  if Found then
  begin
    AllocBlock(Track, Sector, True);
  end
  else
  begin
    Track := 0;
    Sector := 0;
  end;
end;

{Search for a free block on the whole disk for an output file
  Output: when True, a free block was found}
function TOldPanel.FirstCopyBlock: Boolean;
var
  F             : Boolean;
  M,
  P             : Byte;
begin
  F := False;
  if CopyGEOSFormat then
  begin
    Track := 1;
    Sector := 0;
    F := NextCopyBlock;
  end
  else
  begin
    P := 1;
    while not F and (P < 128) do
    begin
      Track := DirTrack - P;
      if (Track >= FirstTrack) and (Track < LastTrack) then F := IsTrackFree(Track);
      if not F then
      begin
        Track := DirTrack + P;
        if Track < LastTrack then F := IsTrackFree(Track);
      end;
      if not F then Inc(P);
    end;
    if not F and CopyToDirTrack then
    begin
      Track := DirTrack;
      F := IsTrackFree(Track);
    end;
    if F then
    begin
      M := SectorNum(Track);
      Sector := 0;
      repeat
        F := not IsBlockUsed(Track, Sector);
        if not F then Inc(Sector);
      until F or (Sector >= M);
    end;
    AllocCopyBlock(F);
  end;
  FirstCopyBlock := F;
end;

{Search for a next block for the output file
  Output: when True, a free block was found}
function TOldPanel.NextCopyBlock: Boolean;
var
  F             : Boolean;
  C,
  M,
  P,
  T             : Byte;
begin
  if (Track = 0) or (Track >= CopyMaxTrack) then
  begin
    NextCopyBlock := False;
  end
  else
  begin
    C := 3;
    F := False;
    T := Track;
    while not F and (C > 0) do
    begin
      M := SectorNum(Track);
      if IsTrackFree(Track) then
      begin
        if (Track = T) or not CopyGEOSFormat then
        begin
          Inc(Sector, SectorStep);
          if CopyGEOSFormat and (Track >= 25) then Dec(Sector);
        end
        else
        begin
          Sector := (Track - T) shl 1 + 4 + SectorStep;
        end;
        while Sector >= M do
        begin
          Dec(Sector, M);
          if (Sector > 0) and not CopyGEOSFormat then Dec(Sector);
        end;
        P := Sector;
        repeat
          F := not IsBlockUsed(Track, Sector);
          if not F then Inc(Sector);
          if Sector >= M then Sector := 0;
        until F or (Sector = P);
      end
      else
      begin
        if Track = DirTrack then
        begin
          C := 0;
        end
        else
        begin
          if CopyGEOSFormat then
          begin
            Inc(Track);
            if (Track = DirTrack) or (Track = DirTrack2) then Inc(Track);
            if Track = LastTrack then C := 0;
          end
          else
          begin
            if Track < DirTrack then
            begin
              Dec(Track);
              if Track < FirstTrack then
              begin
                Track := DirTrack + 1;
                Sector := 0;
                if Track < LastTrack then Dec(C) else C := 0;
              end;
            end
            else
            begin
              Inc(Track);
              if Track = DirTrack2 then Inc(Track);
              if Track = LastTrack then
              begin
                Track := DirTrack - 1;
                Sector := 0;
                if Track >= FirstTrack then Dec(C) else C := 0;
              end;
            end;
          end;
        end;
      end;
      if not F and (C = 0) and (Track <> DirTrack) and CopyToDirTrack then
      begin
        Track := DirTrack;
        Inc(C);
      end;
    end;
    AllocCopyBlock(F);
    NextCopyBlock := F;
  end;
end;

{Clear track allocation map and fill with BAM data, if needed
  Input : Alloc: when True, the first free block is selected
  Output: when True, there is at least one allocated block on the track}
function TOldPanel.ClearTrackMap(Alloc: Boolean): Boolean;
var
  M,
  I,
  S             : Byte;
begin
  M := SectorNum(Track) - 1;
  TrackMap[TrackMapSize] := M;
  if not Alloc then Inc(TrackMap[TrackMapSize]);
  if CopyDiskCopyMode = dcFull then
  begin
    ClearTrackMap := True;
    S := 0;
    FillChar(TrackMap, M + 1, 0);
  end
  else
  begin
    ClearTrackMap := False;
    S := MaxByte;
    for I := 0 to M do
    begin
      if ManualSelect^.Contains(DiskPos(Track, I)) then
      begin
        TrackMap[I] := 0;
        ClearTrackMap := True;
        if S = MaxByte then S := I;
      end
      else
      begin
        TrackMap[I] := 1;
        Dec(TrackMap[TrackMapSize]);
      end;
    end;
  end;
  Move(TrackMap, TrackMap2, TrackMapSize + 1);
  if Alloc and (S < MaxByte) then
  begin
    Sector := S;
    TrackMap[Sector] := 1;
  end;
end;

{Search for the next sector on track to be copied
  Input : Step: when True, interleave is taken into account, otherwise
                blocks are allocated in sequential order
  Output: when True, a free block was found}
function TOldPanel.NextSector(Step: Boolean): Boolean;
var
  F             : Boolean;
  M             : Byte;

{Search for the next block on the track to be copied
  Output: when True, a free block was found}
function FindCopyBlock: Boolean;
var
  F             : Boolean;
  M             : Byte;
begin
  M := SectorNum(Track);
  F := False;
  while not F and (Sector < M) do
  begin
    F := (TrackMap[Sector] = 0);
    if F then
    begin
      TrackMap[Sector] := 1;
      Dec(TrackMap[TrackMapSize]);
    end
    else
    begin
      Inc(Sector);
    end;
  end;
  FindCopyBlock := F;
end;

begin
  F := False;
  if Step then
  begin
    if TrackMap[TrackMapSize] > 0 then
    begin
      M := SectorNum(Track);
      while not F do
      begin
        Inc(Sector, CopyStep);
        if Sector >= M then Dec(Sector, M);
        F := FindCopyBlock;
        if not F then
        begin
          Sector := 0;
          F := FindCopyBlock;
        end;
      end;
    end;
  end
  else
  begin
    TrackMap[Sector] := 1;
    if TrackMap[TrackMapSize] > 0 then Dec(TrackMap[TrackMapSize]);
    F := (TrackMap[TrackMapSize] > 0);
  end;
  NextSector := F;
end;

{Create a temporary file name
  Output: the temporary file name}
function TOldPanel.MakeTempName: string;
begin
  MakeTempName := AddToPath(CopyPath, TempFileName, chDirSep);
end;

{Rename the backup file to the original name of the image file if an
  error occured
  Input : B: when True, the original file was updated successfully and the
             backup file must remain, otherwise the original file must be
             restored from the backup}
procedure TOldPanel.MakeBackupFile(B: Boolean);
begin
  if BackupDone and MakeBackup and (CopyMode in [pmDisk, pmTape]) and not B then
  begin
    BackupDone := False;
    LongErase(AddToPath(CopyPath, CopyImageName, chDirSep));
    LongRename(AddToPath(CopyPath, BackupName, chDirSep), AddToPath(CopyPath, CopyImageName, chDirSep));
  end;
end;

{Check disk type: drive mode, number of tracks, with or without error info}
procedure TOldPanel.CheckDiskType;
var
  T             : Byte;
begin
  FirstTrack := 1;
  case CopyDiskType and dtTypeMask of
    dt1541:
    begin
      T := 18;
      DirTrack2 := MaxByte;
      CopyMaxTrack := Max1541Tracks;
    end;
    dt1541Ext:
    begin
      T := 18;
      DirTrack2 := MaxByte;
      CopyMaxTrack := 41;
    end;
    dt1571:
    begin
      T := 18;
      DirTrack2 := 53;
      CopyMaxTrack := 71;
    end;
    dt1581:
    begin
      T := 40;
      DirTrack2 := MaxByte;
      CopyMaxTrack := 81;
    end;
  end;
  MainDirTrack := T;
  if DirTrack = 0 then DirTrack := T;
  LastTrack := CopyMaxTrack;
  CopyDiskSize := CBMDiskSize(CopyDiskType);
end;

{Read a directory entry from the disk, image file or archive file
  Input : Entry: entry record to contain the directory entry
  Output: when False, an empty directory entry was read}
function TOldPanel.ReadCBMEntry(var Entry: TDirEntry): Boolean;
var
  B,
  C,
  F,
  O             : Boolean;
  M,
  P,
  Q,
  R             : Byte;
  W,
  X,
  Y             : Word;
  K,
  L,
  T             : Longint;
  E             : string[4];
  N,
  S             : string;
  Z             : TZIPCDirSign;

{Determine whether the current entry is inside the current path inside the
  archive file (if not, skip it) and create fake directory entries if
  necessary}
procedure ProcessArcPath;
begin
  if Copy(GetPath(S, DirSep), 1, Length(RealImagePath)) = RealImagePath then
  begin
    if S[Length(S)] = DirSep then
    begin
      Q := faPartition;
      Dec(S[0]);
    end;
    M := Length(RealImagePath) + 1;
    if M > 1 then Inc(M);
    N := Copy(S, M, MaxStrLen);
    if N <> '' then
    begin
      if Pos(DirSep, N) > 0 then
      begin
        Q := faPartition;
        M := LeftPos(DirSep, N);
        if M = 0 then S := N else S := Copy(N, 1, M - 1);
        Entry.Time := 0;
      end;
      if Q = faPartition then L := 0;
      Entry.Attr := Q or faClosed;
      Entry.Size := L;
      Entry.Name := CutPath(S, DirSep);
    end;
  end
  else
  begin
    Entry.Attr := 0;
    Entry.Name := '';
  end;
  if CopyMode <> pmZIP then Inc(CopyImageSize);
  O := True;
end;

{Reset the directory entry}
procedure ResetEntry;
begin
  Entry.ExtAttr := 0;
  Entry.Attr := 0;
  Entry.Name := '';
  Entry.RecordLen := 0;
end;

begin
  B := True;
  O := False;
  ResetEntry;
  case CopyMode of
    pmGCRDisk, pmSixZip: B := False;
    pmExt, pmDisk, pmDiskZip:
    begin
      Inc(Number);
      Inc(DirPos);
      if Number = 8 then
      begin
        Number := 0;
        if (CopyTransferMode <> tmNormal) and (DirBuffer[0] <> 0) and (DirBuffer[1] >= SectorNum(DirTrack)) then
          DirBuffer[0] := 0;
        if (DirBuffer[0] = 0) then
        begin
          Number := 7;
          B := False;
        end
        else
        begin
          if CopyMode = pmExt then
          begin
            if CopyTransferMode = tmNormal then Inc(DirSector) else DirSector := DirBuffer[1];
            ReadDirBlock(@DirBuffer);
            if CopyTransferMode = tmNormal then
            begin
              if Status = 0 then
              begin
                DirBuffer[0] := DirTrack;
              end
              else
              begin
                DirBuffer[0] := 0;
                B := (Status and not ssEOF = 0);
              end;
            end;
          end
          else
          begin
            DirSector := DirBuffer[1];
            if (DirBuffer[0] = DirTrack) and ValidPos(DirTrack, DirSector) and (GCRMap[DirSector] = gmEmpty) then
            begin
              ReadDiskBlock(DirTrack, DirSector, @DirBuffer, True);
              GCRMap[DirSector] := 1;
              if DirAllocMode <> daNone then CheckDirAlloc(DirTrack, DirSector);
            end
            else
            begin
              DirBuffer[0] := 0;
              B := False;
            end;
          end;
          if GCRError > 0 then
          begin
            ErrorWin(stError, ReadErrorStr(GCRError, DirTrack, DirSector), stEmpty, CurHelpCtx, sbSkip);
            B := False;
            Error := True;
          end;
        end;
      end;
      if B then
      begin
        EntryPos := Number shl 5;
        Entry.Attr := DirBuffer[EntryPos + 2];
        Entry.RecordLen := DirBuffer[EntryPos + 23];
        Entry.ExtAttr := DirBuffer[EntryPos + 24];
        if not GEOSSupport or not (Entry.ExtAttr in [1..GEOSTypeNum]) then Entry.ExtAttr := 0;
        if (Entry.ExtAttr > 0) and (Entry.RecordLen > 0) then Entry.ExtAttr := Entry.ExtAttr or xaGEOSVLIR;
        Entry.Name := '';
        for P := 0 to CBMNameLen - 1 do Entry.Name := Entry.Name + Chr(DirBuffer[EntryPos + P + 5]);
        Entry.Name := CutChar(Entry.Name, chShiftSpace);
        Entry.Size := BytesToLongint(DirBuffer[EntryPos + 30], DirBuffer[EntryPos + 31], 0, 0);
        Track := DirBuffer[EntryPos + 3];
        Sector := DirBuffer[EntryPos + 4];
        SideTrack := DirBuffer[EntryPos + 21];
        SideSector := DirBuffer[EntryPos + 22];
        Entry.Track := Track;
        Entry.Sector := Sector;
        Entry.SideTrack := SideTrack;
        Entry.SideSector := SideSector;
        if (Entry.Attr and faTypeMask = faPartition) and (CopyDiskType and dtTypeMask = dt1581) then
        begin
          if (Entry.Size >= 120) and (Entry.Size mod 40 = 0) and (Entry.Sector = 0) then
          begin
            P := Entry.Track;
            Q := P + (Entry.Size div 40);
            if ((P < DirTrack) and (Q <= DirTrack)) or ((P > DirTrack) and (Q > DirTrack)) then
              Entry.ExtAttr := xaDirectory;
          end;
        end;
      end;
    end;
    pmTape:
    begin
      Inc(DirPos);
      if DirPos > CopyImageSize + 1 then
      begin
        B := False;
        ImagePos := ExtFileSize(Image);
        LongintToLonglongint(CopyFree, CopyImageSize + 2);
      end
      else
      begin
        ReadTapeBlock(DirPos, @DirBuffer);
        Entry.Attr := DirBuffer[0];
        if Entry.Attr in [faTapeNormal, faTapeFrozen] then
        begin
          if Entry.Attr = faTapeNormal then Entry.Attr := DirBuffer[1] else Entry.Attr := faFrozen;
          if not (Entry.Attr in [faDeleted..faRelative, faFrozen]) then Entry.Attr := faProgram;
          Entry.Attr := Entry.Attr or faClosed;
          ImagePos := BytesToLongint(DirBuffer[8], DirBuffer[9], DirBuffer[10], DirBuffer[11]);
          Entry.Name := '';
          for P := 0 to CBMNameLen - 1 do Entry.Name := Entry.Name + Chr(DirBuffer[P + 16]);
          Entry.Name := CutChar(Entry.Name, ' ');
          Entry.Start := BytesToLongint(DirBuffer[2], DirBuffer[3], 0, 0);
        end
        else
        begin
          B := False;
          ImagePos := ExtFileSize(Image);
          LongintToLonglongint(CopyFree, DirPos);
        end;
        Entry.Size := ImagePos;
      end;
    end;
    pmFile:
    begin
      Inc(DirPos);
      if DirPos = 0 then
      begin
        S := LowerCase(FileExt(CopyImageName));
        Entry.Attr := 0;
        for P := faDeleted to faRelative do if S[1] = ShortCBMExt[P][1] then Entry.Attr := P or faClosed;
        Entry.Name := CutChar(CopyLabel, #0);
        Entry.Start := BytesToLongint(DirBuffer[26], DirBuffer[27], 0, 0);
        Entry.Size := ImagePos - 26;
        Entry.RecordLen := DirBuffer[25];
      end
      else
      begin
        B := False;
      end;
    end;
    pmLynx, pmArkive:
    begin
      Inc(DirPos);
      if DirPos >= CopyImageSize then
      begin
        B := False;
        Inc(ImagePos, PrevSize);
        Inc(ImagePos, PadSize(ImagePos));
      end
      else
      begin
        ExtSeek(Image, HeaderPos);
        FillChar(TempBuffer, 40, 0);
        ExtBlockRead2(Image, TempBuffer, 40, X);
        B := (IOResult = 0);
        if B then
        begin
          M := CopyMode;
          R := 0;
          C := True;
(* ?ASM? *)
          asm
            cmp M, pmArkive;
            je @7;
            xor si, si;
        @2: mov al, byte ptr TempBuffer[si];
            cmp al, chReturn;
            je @3;
            mov byte ptr S[si][1], al;
            inc si;
            cmp si, CBMNameLen + 1;
            jb @2;
            xor si, si;
        @3: mov ax, si;
            mov byte ptr S[0], al;
            or si, si;
            je @1;
            inc si;
            mov cx, 10;
            call ReadNum;
            jc @1;
            or dx, dx;
            jne @1;
            cmp bl, chReturn;
            jne @1;
            mov X, ax;
            inc si;
            mov ah, byte ptr TempBuffer[si];
            xor al, al;
            cmp ah, 'D';
            je @5;
            inc al;
            cmp ah, 'S';
            je @5;
            inc al;
            cmp ah, 'P';
            je @5;
            inc al;
            cmp ah, 'U';
            je @5;
            inc al;
            cmp ah, 'R';
            jne @1;
        @5: mov Q, al;
            inc si;
            cmp byte ptr TempBuffer[si], chReturn;
            jne @1;
            inc si;
            cmp al, faRelative;
            jne @9;
            mov cx, 10;
            call ReadNum;
            jc @1;
            cmp bl, chReturn;
            jne @1;
            or ah, ah;
            jne @1;
            or dx, dx;
            jne @1;
            mov R, al;
            inc si;
        @9: mov cx, 10;
            call ReadNum;
            jnc @6;
            mov C, False;
            jmp @4;
        @6: cmp bl, chReturn;
            jne @1;
            or ah, ah;
            jne @1;
            or dx, dx;
            jne @1;
            mov Y, ax;
            inc si;
        @8: mov W, si;
            jmp @4;
        @7: mov si, Offset(TempBuffer);
            cld;
            lodsb;
            mov Q, al;
            lodsb;
            xor ah, ah;
            mov Y, ax;
            push ss;
            pop es;
            lea di, S;
            mov al, CBMNameLen;
            stosb;
            mov cl, al;
            xor ch, ch;
            rep movsb;
            lodsb;
            mov R, al;
            add si, 8;
            lodsw;
            mov X, ax;
            mov si, 29;
            jmp @8;
        @1: mov B, False;
        @4:
          end;
        end;
        K := ImagePos + PrevSize;
        Inc(K, PadSize(K));
        if not C then
        begin
          if DirPos = CopyImageSize - 1 then
          begin
            if X = 0 then
            begin
              Y := 0;
            end
            else
            begin
              L := OrigSize - K - (Longint(X) - 1) * 254;
              if L > 254 then Y := 255 else Y := L mod 254 + 1;
            end;
          end
          else
          begin
            B := False;
          end;
        end;
        L := 0;
        if X <> 0 then L := Longint(X) * 254 - 255 + Y;
        Error := not B;
        if B then
        begin
          Inc(HeaderPos, W);
          ImagePos := K;
          Inc(Blocks, X);
          Entry.Name := CutChar(S, chShiftSpace);
          if M = pmLynx then Q := Q or faClosed;
          Entry.Attr := Q;
          if PrevSize > 0 then Inc(LastLynxEnd, PadSize(LastLynxEnd) + PrevSize);
          PrevSize := L;
          Entry.Size := L;
          Entry.RecordLen := R;
        end;
      end;
    end;
    pmTAR:
    begin
      repeat
        Inc(DirPos);
        ResetEntry;
        ExtSeek(Image, HeaderPos);
        ExtBlockRead(Image, TempBuffer, 512);
        B := ((IOResult = 0) and (TempBuffer[0] <> 0));
        if B then
        begin
(* ?ASM? *)
          asm
            push False;
            call ComputeTARCheck;
            push ax;
            push dx;
            mov si, $0094;
            mov cx, 8;
            call ReadNum;
            pop cx;
            pop bx;
            jc @3;
            cmp ax, bx;
            jne @3;
            cmp dx, cx;
            jne @3;
            mov si, Offset(TempBuffer);
            push ss;
            pop es;
            lea di, S[1];
            cld;
            xor cl, cl;
        @6: lodsb;
            or al, al;
            je @5;
            stosb;
            inc cl;
            jmp @6;
        @5: mov byte ptr S[0], cl;
            mov si, $0080;
            mov cx, 8;
            call ReadNum;
            jc @3;
            mov word ptr L[0], ax;
            mov word ptr L[2], dx;
            mov si, $0088;
            mov cx, 8;
            call ReadNum;
            jc @3;
            mov word ptr T[0], ax;
            mov word ptr T[2], dx;
            jmp @4;
        @3: mov B, False;
        @4:
          end;
          Error := not B;
          if B then
          begin
            PrevSize := L and $FFFFFE00;
            if L and $000001FF > 0 then Inc(PrevSize, 512);
            ImagePos := HeaderPos + 512;
            HeaderPos := ImagePos + PrevSize;
            Entry.Time := UnixToDOS(T);
            Q := 2;
            if S <> '' then
            begin
              if (Length(S) > 4) and (S[Length(S) - 3] = '.') then
              begin
                E := LowerCase(Copy(S, Length(S) - 2, 3));
                F := False;
                M := 0;
                while not F and (M < 4) do
                begin
                  F := (E = ShortCBMExt[M]);
                  if not F then Inc(M);
                end;
                if F then
                begin
                  Dec(S[0], 4);
                  Q := M;
                end;
              end;
              ProcessArcPath;
            end;
          end;
        end;
      until not B or O;
    end;
    pmLHA:
    begin
      repeat
        Inc(DirPos);
        ResetEntry;
        ExtSeek(Image, HeaderPos);
        ExtBlockRead(Image, LHAHeaderLen, 1);
        B := (LHAHeaderLen in [1..79]) and (IOResult = 0);
        if B then
        begin
          ExtBlockRead(Image, LHAEntry, LHAHeaderLen + 1);
          B := (IOResult = 0);
          if B then
          begin
(* ?ASM? *)
            asm
              mov si, Offset(LHAEntry[1]);
              mov cl, LHAHeaderLen;
              xor ch, ch;
              xor ah, ah;
              cld;
          @5: lodsb;
              add ah, al;
              loop @5;
              mov al, LHAEntry.Checksum;
              cmp ah, LHAEntry.Checksum;
              jne @3;
              sub si, 2;
              lodsw;
              push ds;
              pop es;
              mov di, Offset(LHAEntry.Name);
              mov bx, di;
              mov cl, [di];
              xor ch, ch;
              xor al, al;
              jcxz @1;
              inc di;
              cld;
              repne scasb;
              jne @1;
              mov ax, di;
              sub ax, bx;
              sub ax, 2;
              mov [bx], al;
              mov al, [di];
          @1: or al, al;
              jne @6;
              mov al, 'P';
          @6: mov P, al;
              mov ah, P;
              xor al, al;
              cmp ah, 'D';
              je @2;
              inc al;
              cmp ah, 'S';
              je @2;
              inc al;
              cmp ah, 'P';
              je @2;
              inc al;
              cmp ah, 'U';
              je @2;
              mov al, 2;
          @2: mov Q, al;
              mov di, Offset(LHAEntry.Name);
              mov bx, di;
              mov cl, [di];
              xor ch, ch;
              add di, cx;
              mov ax, [di][1];
              mov LHAEntry.CRCCheck, ax;
              jmp @4;
          @3: mov B, False;
          @4:
            end;
            Error := not B;
            if B then
            begin
              Inc(HeaderPos, LHAHeaderLen + LHAEntry.PackSize + 2);
              Inc(ImagePos, PrevSize);
              S := LHAEntry.Name;
              L := LHAEntry.OrigSize;
              Entry.Time := LHAEntry.Time;
              ProcessArcPath;
              S := Replace(S, chDirSep, '/');
              PrevSize := LHAEntry.OrigSize;
            end;
          end;
        end;
      until not B or O;
    end;
    pmZIP:
    begin
      repeat
        Inc(DirPos);
        if DirPos >= CopyImageSize then
        begin
          B := False;
        end
        else
        begin
          ExtSeek(Image, HeaderPos);
          ExtBlockRead(Image, ZIPEntry, SizeOf(TZIPCDirEntry));
          B := (IOResult = 0);
          if B then
          begin
            W := ZIPEntry.Signature shr 16;
            Error := (Word(ZIPEntry.Signature) <> Word(ZIPDirSign)) or (ZIPEntry.NameLen > MaxStrLen);
            if not Error then
            begin
              case W of
                (ZIPCDirEndSign shr 16): B := False;
                (ZIPCDirSign shr 16):
                begin
                  S[0] := Chr(ZIPEntry.NameLen);
                  ExtBlockRead(Image, S[1], ZIPEntry.NameLen);
                  B := (IOResult = 0);
                  if B then
                  begin
                    Inc(HeaderPos, SizeOf(TZIPCDirEntry) + ZIPEntry.NameLen + ZIPEntry.ExtHeadSize + ZIPEntry.CommentSize);
                    Inc(ImagePos, PrevSize);
                    Q := faProgram;
                    L := ZIPEntry.OrigSize;
                    ProcessArcPath;
                    Entry.Time := ZIPEntry.Time;
                    PrevSize := ZIPEntry.OrigSize;
                  end;
                end;
              else
                Error := True;
              end;
            end;
          end;
        end;
        B := B and not Error;
      until not B or O;
    end;
    pmFileZip:
    begin
      Inc(DirPos);
      if DirPos >= CopyImageSize then
      begin
        B := False;
        Inc(ImagePos, PrevSize);
        PrevSize := 0;
      end
      else
      begin
        ExtSeek(Image, HeaderPos);
        ExtBlockRead(Image, TempBuffer, 21);
        B := (IOResult = 0);
        if B then
        begin
(* ?ASM? *)
          asm
            mov si, Offset(TempBuffer);
            push ss;
            pop es;
            lea di, S;
            mov cx, CBMNameLen;
            cld;
            mov al, cl;
            stosb;
            rep movsb;
            lodsb;
            mov P, al;
            lodsw;
            mov X, ax;
            mov ah, P;
            and ah, $7F;
            xor al, al;
            cmp ah, 'D';
            je @2;
            inc al;
            cmp ah, 'S';
            je @2;
            inc al;
            cmp ah, 'P';
            je @2;
            inc al;
            cmp ah, 'U';
            jne @3;
        @2: mov Q, al;
            jmp @4;
        @3: mov B, False;
        @4:
          end;
        end;
        Error := not B;
        if B then
        begin
          Inc(HeaderPos, 21);
          Inc(ImagePos, PrevSize);
          Entry.Name := CutChar(S, chShiftSpace);
          Entry.Attr := Q or faClosed;
          Entry.Size := Longint(X);
          PrevSize := Longint(X);
        end;
      end;
    end;
  end;
  Entry.Status := fsNormal;
  ReadCBMEntry := B;
end;

{Check the allocation of a directory block and allocate it, if needed
  Input : T, S: track and sector number of the block}
procedure TOldPanel.CheckDirAlloc(T, S: Byte);
begin
  case DirAllocMode of
    daCheck: DirAllocOK := DirAllocOK and IsBlockUsed(T, S);
    daAlloc: AllocBlock(T, S, True);
  end;
end;

{Check whether the current disk or disk image is in GEOS format
  Input : Block: the buffer containing the BAM sector}
procedure TOldPanel.CheckGEOSFormat(P: PBlock);
var
  S             : string[11];
begin
  CopyGEOSFormat := False;
  if GEOSSupport then
  begin
    S[0] := #11;
    Move(P^[GEOSSignPos], S[1], 11);
    CopyGEOSFormat := (S = Copy(GEOSSign, 1, 11));
  end;
end;

{Initialize archive file, moving pointer to first file entry
  Output: when not 0, an error occured}
function TOldPanel.InitArchive: Integer;
var
  B             : Byte;
  I             : Integer;
begin
  I := 0;
  case CopyMode of
    pmLHA:
    begin
      ArcBufPos := 0;
      ArcBufSize := 0;
      ArcEnd := False;
      ArcSize := ExtFileSize(Image);
      ArcPos := 0;
      if ArcSize = 1 then
      begin
        ExtBlockRead(Image, B, 1);
        if B = 0 then HeaderPos := 0 else I := 255;
      end
      else
      begin
        if not SearchArchive(Image, LHASign, False) then I := 255;
        if I = 0 then HeaderPos := ArcPos - 5;
      end;
      CopyImageSize := 0;
    end;
    pmZIP:
    begin
      ArcBufPos := 0;
      ArcBufSize := 0;
      ArcEnd := False;
      ArcSize := ExtFileSize(Image);
      ArcPos := ArcSize;
      if not SearchArchive(Image, ZIPSign, True) then I := 255;
      if I = 0 then
      begin
        ExtSeek(Image, ArcPos);
        ExtBlockRead(Image, ZipDirEnd, SizeOf(TZipCDirEnd));
        if CompMem(ZipDirEnd.Signature, ZIPSign[1], Length(ZIPSign)) and (ZIPDirEnd.VolumeNum = 0) then
        begin
          HeaderPos := ZipDirEnd.DirOffs;
          CopyImageSize := ZipDirEnd.FileNum;
        end
        else
        begin
          I := 255;
        end;
      end;
    end;
  end;
  InitArchive := I;
end;

{Open the disk, image file or archive file
  Input : Write: when True, the image file or archive file is opened for
                 read and write, otherwise only for read
          OnlyBAM: when True, only the BAM is read from disk images and
                   external drives, otherwise the first directory sector,
                   too
          GetBAM: when True, the BAM is read in
          DetectDisk: when True, extra tracks on 1541 disks and number of
                      sides on 1571 disks are detected
          Errors: when True, errors are displayed
  Output: when not 0, an error occured}
function TOldPanel.OpenImage(Write, OnlyBAM, GetBAM, DetectDisk, Errors: Boolean): Integer;
var
  B,
  F             : Boolean;
  P,
  Q             : Byte;
  W,
  X             : Word;
  I             : Integer;
  H,
  L             : Longint;
  S,
  T             : string;
  E             : TDirEntry;

{Reset directory-related variables}
procedure ResetDir;
begin
  DirBuffer[0] := DirTrack;
  DirBuffer[1] := 0;
  Number := 7;
end;

begin
  InOutRes := 0;
  DirAllocOK := True;
  GCRError := 0;
  LongintToLonglongint(CopyFree, 0);
  CopyShortFree := 0;
  CopyLongFree := 0;
  PrevSize := 0;
  DirPos := MaxWord;
  DirLength := MaxWord;
  ImagePos := 0;
  Blocks := 0;
  CopyLabel := '';
  CopyID := '';
  TapeName := '';
  I := 0;
  MakeDirSep;
  CopyImagePath := CopyRealImagePath;
  if Write then P := fmReadWrite else P := fmReadOnly;
  if CopyMode in [pmExt, pmDisk, pmGCRDisk..pmSixZip] then
  begin
    if CopyDiskType and dtTypeMask = dt1581 then DirLength := 296 else DirLength := 144;
  end;
  case CopyMode of
    pmDOS: I := 254;
    pmExt:
    begin
      I := 254;
      CBMDevNum := CopyCBMDev;
      ExtBAMMode := DiskExtBAMMode;
      CheckDevice;
      if Status = 0 then
      begin
        SendConfigData;
        CopyDiskType := DetectDiskType(CopyExtDisk, DetectDisk);
        if Status = 0 then
        begin
          case CopyTransferMode of
            tmNormal:
            begin
              OpenCBMChannel(saCommand, 'I0', True);
              if CopyTransferMode = tmNormal then
              begin
                case CopyDiskType and dtTypeMask of
                  dt1571: ReadSpecDirBlock(DirTrack2, 0, @BAM2, True, True);
                  dt1581:
                  begin
                    DirTrack := 40;
                    ReadSpecDirBlock(DirTrack, 1, @BAM2, True, False);
                    ReadSpecDirBlock(DirTrack, 2, @BAM3, False, True);
                  end;
                end;
              end;
              OpenCBMChannel(saData, '$0', True);
              asm
                mov al, saData;
                call Talk;
              end;
            end;
            tmTurbo: if SendDriveProg(deTurboDirLoad, True) then ExecDriveProg(deTurboDirLoad, stEmpty);
            tmWarp: if SendDriveProg(deWarpDirLoad, True) then ExecDriveProg(deWarpDirLoad, stEmpty);
          end;
        end;
      end;
      if Status = 0 then I := 0;
    end;
  else
    ExtBAMMode := ImageExtBAMMode;
    S := CopyImageName;
    case CopyMode of
      pmFileZip: S[1] := 'x';
      pmDiskZip, pmSixZip: S[1] := '1';
    end;
    S := AddToPath(CopyRealPath, S, chDirSep);
    LongGetFAttr(S, ImageAttr);
    ImageReadOnly := (ImageAttr and ReadOnly > 0);
    I := LongOpenFile(S, Image, P);
    if I = 0 then
    begin
      ExtGetFTime(Image, FileTime);
      OrigSize := ExtFileSize(Image);
    end;
  end;
  if I = 0 then
  begin
    case CopyMode of
      pmGCRDisk, pmSixZip:
      begin
        CopyDiskType := dt1541;
        I := 255;
      end;
      pmDisk, pmExt, pmDiskZip:
      begin
        B := True;
        case CopyMode of
          pmDisk:
          begin
            CopyDiskType := GetDiskType(OrigSize);
            B := (CopyDiskType <> dtInvalid);
          end;
          pmDiskZip:
          begin
            CopyDiskType := dt1541;
            S := CopyImageName;
            S[1] := '5';
            if FileExists(AddToPath(CopyRealPath, S, chDirSep), False) then CopyDiskType := dt1541Ext;
          end;
        end;
        if B then
        begin
          DirTrack := 0;
          CheckDiskType;
          FirstTrack := 1;
          LastTrack := CopyMaxTrack;
          if (CopyMode <> pmExt) and (CopyDiskType and dtTypeMask = dt1581) and (CopyRealImagePath <> '') then
          begin
            DirTrack := 40;
            S := CopyRealImagePath;
            CopyRealImagePath := '';
            repeat
              ResetDir;
              P := LeftPos(DirSep, S);
              if P = 0 then P := Length(S) + 1;
              T := Copy(S, 1, P - 1);
              S := Copy(S, P + 1, MaxStrLen);
              DirSector := FirstDirSec(CopyDiskType);
              FillChar(GCRMap, TrackMapSize, gmEmpty);
              F := False;
              while not F and (ReadCBMEntry(Entry)) do F := ((Entry.Name = T) and (Entry.Attr and faTypeMask = faPartition) and
                (Entry.ExtAttr and xaDirectory > 0));
              if F then
              begin
                CopyRealImagePath := AddToPath(CopyRealImagePath, T, DirSep);
                DirTrack := Entry.Track;
                FirstTrack := Entry.Track;
                LastTrack := Entry.Track + (Entry.Size div SectorNum(Entry.Track));
              end;
            until (S = '') or not F;
          end
          else
          begin
            CopyRealImagePath := '';
          end;
          ResetDir;
          DirSector := 0;
          DirPos := MaxWord;
          FillChar(GCRMap, TrackMapSize, gmEmpty);
          if GetBAM then
          begin
            if CopyMode = pmExt then
            begin
              ReadDirBlock(@BAM);
              case CopyDiskType and dtTypeMask of
                dt1571: if CopyTransferMode <> tmNormal then ReadDirBlock(@BAM2);
                dt1581:
                begin
                  if CopyTransferMode <> tmNormal then
                  begin
                    ReadDirBlock(@BAM2);
                    ReadDirBlock(@BAM3);
                  end;
                end;
              end;
            end
            else
            begin
              ReadBAM(Errors);
              if DirAllocMode <> daNone then
              begin
                for P := 0 to FirstDirSec(CopyDiskType) - 1 do CheckDirAlloc(DirTrack, P);
                if DirTrack2 < LastTrack then CheckDirAlloc(DirTrack2, 0);
              end;
            end;
          end;
          if (CopyMode <> pmExt) or ((Status and not ssEOF = 0) and (GCRError = 0)) then
          begin
            if GetBAM then
            begin
              CopyImageSize := 0;
              Q := NameOffset(CopyDiskType, ExtBAMMode);
              for P := 0 to CBMNameLen - 1 do CopyLabel := CopyLabel + Chr(BAM[P + Q]);
              CopyShortLabel := CutChar(CopyLabel, chShiftSpace);
              Inc(Q, DiskIDRelPos);
              for P := 0 to CBMBAMIDLen - 1 do CopyID := CopyID + Chr(BAM[P + Q]);
              CopyLabel := CopyLabel + ',' + CopyID;
              for P := FirstTrack to LastTrack - 1 do
              begin
                Q := BAM[GetBAMOffset(P, False)];
                Inc(CopyLongFree, Q);
                if (P = DirTrack) or (P = DirTrack2) then
                begin
                  if (CopyMode <> pmExt) and CopyToDirTrack then IncLonglongint(CopyFree, Q);
                end
                else
                begin
                  Inc(CopyShortFree, Q);
                  IncLonglongint(CopyFree, Q);
                end;
              end;
              CheckGEOSFormat(@BAM);
            end;
            if not OnlyBAM then
            begin
              FillChar(GCRMap, DirSector + 1, 1);
              DirSector := FirstDirSec(CopyDiskType);
              if CopyMode = pmExt then
              begin
                ReadDirBlock(@DirBuffer);
                if CopyTransferMode = tmNormal then if Status = 0 then DirBuffer[0] := DirTrack else DirBuffer[0] := 0;
              end
              else
              begin
                ReadDiskBlock(DirTrack, DirSector, @DirBuffer, Errors);
                GCRMap[DirSector] := 1;
                if DirAllocMode <> daNone then CheckDirAlloc(DirTrack, DirSector);
              end;
            end;
          end;
          if GCRError > 0 then
          begin
            ErrorWin(stError, ReadErrorStr(GCRError, DirTrack, DirSector), stEmpty, CurHelpCtx, sbSkip);
            Status := 0;
          end;
          Number := MaxByte;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmTape:
      begin
        if ExtFileSize(Image) > 95 then
        begin
          DirPos := 0;
          ReadTapeBlock(DirPos, @DirBuffer);
          for P := 0 to 31 do TapeName := TapeName + Chr(DirBuffer[P]);
          Inc(DirPos);
          ReadTapeBlock(DirPos, @DirBuffer);
          CopyImageSize := BytesToLongint(DirBuffer[2], DirBuffer[3], 0, 0);
          for P := 0 to 23 do CopyLabel := CopyLabel + Chr(DirBuffer[P + 8]);
          CopyLabel := CutChar(CopyLabel, ' ');
          CopyShortLabel := CopyLabel;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmFile:
      begin
        if ExtFileSize(Image) > 25 then
        begin
          ExtBlockRead(Image, DirBuffer, 28);
          for P := 0 to 7 do CopyLabel := CopyLabel + Chr(DirBuffer[P]);
        end;
        if CopyLabel = PC64Sign then
        begin
          ImagePos := ExtFileSize(Image);
          CopyLabel := '';
          for P := 0 to CBMNameLen - 1 do CopyLabel := CopyLabel + Chr(DirBuffer[P + 8]);
          CopyLabel := CutChar(CopyLabel, #0);
        end
        else
        begin
          I := 255;
        end;
      end;
      pmLynx:
      begin
        if ExtFileSize(Image) >= 254 then
        begin
          ExtBlockRead(Image, TempBuffer, 128);
(* ?ASM? *)
          asm
            xor si, si;
        @3: cmp word ptr TempBuffer[si], 0;
            jne @1;
            cmp word ptr TempBuffer[si][2], $0D00;
            jne @1;
            add si, 4;
            jmp @2;
        @1: inc si;
            cmp si, 92;
            jb @3;
            xor si, si;
        @2: mov word ptr L[0], si;
            mov word ptr L[2], 0;
          end;
          ExtSeek(Image, L);
          ExtBlockRead(Image, TempBuffer, 128);
(* ?ASM? *)
          asm
            xor si, si;
            mov cx, 10;
            call ReadNum;
            jc @1;
            or dx, dx;
            jne @1;
            or ax, ax;
            je @1;
            mov W, ax;
            mov dx, si;
            xor bl, bl;
        @5: cmp byte ptr TempBuffer[si], chReturn;
            je @2;
            cmp word ptr TempBuffer[si], 'YL';
            jne @4;
            cmp word ptr TempBuffer[si][2], 'XN';
            jne @4;
            inc bl;
        @4: inc si;
            cmp si, 30;
            jb @5;
            jmp @1;
        @2: or bl, bl;
            je @1;
            inc si;
            xchg si, dx;
            mov cx, dx;
            sub cx, si;
            add si, Offset(TempBuffer);
            lea di, S;
            push ss;
            pop es;
            cld;
            mov al, cl;
            stosb;
            rep movsb;
            mov si, dx;
            add dx, word ptr L[0];
            mov LynxCountPos, dx;
            mov cx, 10;
            call ReadNum;
            jc @1;
            or dx, dx;
            jne @1;
            cmp bl, chReturn;
            jne @1;
            mov X, ax;
            inc si;
            add word ptr L[0], si;
            jmp @3;
        @1: mov I, 255;
        @3:
          end;
          if I = 0 then
          begin
            CopyImageSize := X;
            ImagePos := W * 254;
            LastLynxEnd := ImagePos;
            HeaderPos := L;
          end;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmArkive:
      begin
        if ExtFileSize(Image) >= 254 then
        begin
          ExtBlockRead(Image, P, 1);
(* ?ASM? *)
          asm
            mov al, P;
            xor ah, ah;
            mov X, ax;
            mov cl, 29;
            mul cl;
            inc ax;
            mov cl, 254;
            div cl;
            or ah, ah;
            je @1;
            inc al;
        @1: xor ah, ah;
            mov W, ax;
          end;
          CopyImageSize := X;
          ImagePos := W * 254;
          HeaderPos := 1;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmTAR:
      begin
        if ExtFileSize(Image) >= 512 then
        begin
          CopyImageSize := 0;
          HeaderPos := 0;
          ImagePos := 0;
          PrevSize := 0;
        end
        else
        begin
          I := 255;
        end;
      end;
      pmLHA, pmZIP: I := InitArchive;
      pmFileZip:
      begin
        if ExtFileSize(Image) > 512 then
        begin
          ExtSeek(Image, 512);
          ExtBlockRead(Image, P, 1);
          CopyImageSize := P;
          HeaderPos := 513;
        end;
      end;
    end;
    if (I <> deOK) and not ExtIsClosed(Image) then ExtClose(Image);
  end;
  OpenImage := I;
end;

{Close the disk, image file or archive file
  Input : Write: when True, the image or archive has been written to}
procedure TOldPanel.CloseImage(Write: Boolean);
begin
  if CopyMode = pmExt then
  begin
    if CopyTransferMode <> tmNormal then TurboOff;
    Untalk;
    CloseCBMChannel(saData);
  end
  else
  begin
    if KeepTime and Write then ExtSetFTime(Image, FileTime);
    ExtClose(Image);
  end;
end;

{Validate/scratch a file (allocate/free all blocks) starting at the current
  track and sector or check whether there are bad sectors in it
  Input : Alloc: when 0, blocks are freed; when 1, allocated; when 2, a
                 check is done whether the file contains bad sectors
          Part: when True, the file is a partition and has to be handled
                in a special way
          VLIR: when True, the first block of the file contains links to
                the parts of the file
          Force: when True, the file is validated even if starts at or links
                 to the BAM sectors
          PartSize: size of the partition
  Output: when False, the file contained at least one bad sector}
function TOldPanel.Validate(Alloc: Byte; Part, VLIR, Force: Boolean; PartSize: Word): Boolean;
var
  E,
  O             : Boolean;
  F,
  S,
  T             : Byte;

{Trace through a file chain and allocate/free its blocks or check whether
  its blocks are bad
  Input : T, S: the start position of the file chain
  Output: when False, the file chain contained at least one bad sector}
function TraceFile(T, S: Byte): Boolean;
var
  E,
  O             : Boolean;
  P,
  Q             : Byte;
begin
  E := False;
  O := True;
  while O and not E and (T > 0) and ValidPos(T, S) do
  begin
    if not Force and (T = DirTrack) and (S <= F) then
    begin
      T := 0;
    end
    else
    begin
      case Alloc of
        alFree, alAllocate: AllocBlock(T, S, (Alloc <> alFree));
        alBadSectors: O := (TempBuffer[DiskPos(T, S)] = 1);
      end;
      ReadDiskBlock(T, S, @DataBuffer, True);
      P := DataBuffer[0];
      Q := DataBuffer[1];
      if WipeFiles and (Alloc = alFree) then
      begin
        FillFormatPattern(@DataBuffer);
        WriteDiskBlock(T, S, @DataBuffer);
      end;
      T := P;
      S := Q;
      E := Escape;
    end;
  end;
  TraceFile := O;
end;

begin
  E := False;
  O := True;
  F := FirstDirSec(CopyDiskType);
  if Part then
  begin
    T := Track;
    S := Sector;
    while O and (PartSize > 0) do
    begin
      Dec(PartSize);
      case Alloc of
        alFree, alAllocate: AllocBlock(T, S, (Alloc <> alFree));
        alBadSectors: O := (TempBuffer[DiskPos(T, S)] = 1);
      end;
      Inc(S);
      if S >= SectorNum(T) then
      begin
        Inc(T);
        if T > LastTrack then PartSize := 0;
        S := 0;
      end;
    end;
  end
  else
  begin
    if VLIR then
    begin
      ReadDiskBlock(Track, Sector, @SideBuffer, True);
      SidePos := 2;
      while SidePos > 0 do
      begin
        if SideBuffer[SidePos] > 0 then O := TraceFile(SideBuffer[SidePos], SideBuffer[SidePos + 1]);
        Inc(SidePos, 2);
      end;
    end;
    if O then O := TraceFile(Track, Sector) and TraceFile(SideTrack, SideSector);
  end;
  Validate := O;
end;

{Open the filepacked ZipCode archive for input
  Input : ZipFile: file record
          Pos: position into archive in blocks
          Start: starting offset to seek to; when 0, fixed to the offset
                 of the first data byte
          FileMode: when 0, the file is opened for read only; when 1, for write
                    only; when 2, for read/write
  Output: when not 0, an error occured}
function TOldPanel.OpenZipFile(var ZipFile: ExtFile; Pos: Word; Start: Longint; FileMode: Byte): Integer;
var
  O             : Boolean;
  C             : Char;
  W             : Word;
  I             : Integer;
  L             : Longint;
  S             : string;
  X             : array [0..3] of Byte;
begin
  O := True;
  case CopyMode of
    pmFileZip:
    begin
      W := Pos div FileZipBlocks;
      C := Chr(Ord('a') + W);
      if W >= (Ord('x') - Ord('a')) then
      begin
        if FileMode <> fmReadOnly then ErrorWin(stEmpty, 'There is no more space in', CopyImageName, CurHelpCtx, sbNone);
        ContProcess := False;
        I := 254;
        O := False;
      end;
      L := 3;
    end;
    pmDiskZip:
    begin
      W := DiskZipFileNum - 1;
      while Pos < DiskZipBlocks[W] do Dec(W);
      C := Chr(Ord('1') + W);
      L := 2;
      if W = 0 then L := 4;
    end;
    pmSixZip:
    begin
      W := SixZipFileNum - 1;
      while Pos < SixZipBlocks[W] do Dec(W);
      C := Chr(Ord('1') + W);
      L := 3;
    end;
  end;
  if O then
  begin
    O := False;
    I := 0;
    if not ExtIsClosed(ZipFile) then
    begin
      O := True;
      S := CutPath(ZipFile.LongName, chDirSep);
      if S[1] <> C then
      begin
        if KeepTime and (FileMode <> fmReadOnly) then ExtSetFTime(ZipFile, FileTime);
        ExtClose(ZipFile);
        I := IOResult;
        O := False;
      end;
    end;
    if not O then
    begin
      S := CopyImageName;
      S[1] := C;
      I := LongOpenFile(AddToPath(CopyRealPath, S, chDirSep), ZipFile, FileMode);
      if ((FileMode = fmReadWrite) and (I <> 0)) or ((FileMode = fmWriteOnly) and (I = 0)) then
      begin
        if FileMode = fmReadWrite then I := LongOpenFile(ZipFile.LongName, ZipFile, fmWriteOnly);
        if I = 0 then
        begin
          case CopyMode of
            pmFileZip:
            begin
              X[0] := $FF;
              X[1] := $03;
              X[2] := 0;
            end;
            pmDiskZip:
            begin
              if W = 0 then
              begin
                X[0] := $FE;
                X[1] := $03;
                Move(CopyHeaderID, X[2], CBMHeaderIDLen);
              end
              else
              begin
                X[0] := $00;
                X[1] := $04;
              end;
            end;
            pmSixZip:
            begin
              X[0] := $FF;
              X[1] := $03;
              X[2] := CopyMaxTrack;
            end;
          end;
          ExtBlockWrite(ZipFile, X, L);
          if FileMode = fmReadWrite then
          begin
            ExtClose(ZipFile);
            I := LongOpenFile(ZipFile.LongName, ZipFile, FileMode);
          end;
        end;
      end;
      if (I = 0) and (ReadTime > FileTime) then FileTime := ReadTime;
    end;
  end;
  if I = 0 then
  begin
    if Start > 0 then L := Start;
    ArcPos := L;
    ArcSize := ExtFileSize(ZipFile) - ArcPos;
    ArcBufPos := 0;
    ArcBufSize := 0;
    ArcEnd := False;
    ExtSeek(ZipFile, ArcPos);
  end;
  OpenZipFile := I;
end;

{Change the panel back to DOS mode on an error during opening the image
  file}
procedure TOldPanel.ShutImageMode;
begin
  Working := True;
  Mode := pmDOS;
  NewMode := pmDOS;
  Top := '';
  Under := LongName(ImageName, False);
  ImageName := '';
  ImagePath := '';
  RealImagePath := '';
end;

{Display an error message if the panel ran out of memory during reading
  the directory}
procedure TOldPanel.OutOfMem;
begin
  if DirFull then
  begin
    DirFull := False;
    ErrorWin(stError, 'Out of memory while reading the directory.',
      'Not all files are visible in the panel.', hcOnlyQuit, sbNone);
  end;
end;

{Empty the Quick view panel if no valid image or archive is under the cursor
  of the other panel}
procedure TOldPanel.ShutQuickView;
begin
  Mode := pmDOS;
  OK := True;
  Changing := True;
  QuickView := qvEmpty;
  ImageName := '';
  ImagePath := '';
  RealImagePath := '';
end;

{Draw the panel containing valid data}
procedure TOldPanel.DrawPanel;
begin
  Working := False;
  DrawView;
end;

{Handle speed-search dialog box events
  Input : Event: event record to be handled}
procedure TFileSearch.HandleEvent(var Event: TEvent);
var
  O             : Boolean;
  C             : Char;
  Y             : Integer;
  S             : string;
begin
  Quote := False;
  O := False;
  if Event.What and evKeyboard > 0 then
  begin
    C := Event.CharCode;
    if C < ' ' then
    begin
      case Event.KeyCode of
        kbCtrlJ, kbCtrlEnter:
        begin
          GetData(S);
          if not GetPanelModeAttrib(Panel^.Mode, paASCII) and not Panel^.GEOSFormat then
            S := ReconvertCBMName(S, False, True, hxNone);
          Y := Panel^.Cur + 1;
          if Y >= Panel^.Max then Y := 0;
          Panel^.SearchPattern(S, Y, True);
          ClearEvent(Event);
        end;
        kbCtrlS, kbCtrlD, kbCtrlA, kbCtrlF, kbBack, kbCtrlH, kbCtrlBack,
        kbCtrlW, kbDel, kbCtrlG, kbCtrlY, kbCtrlK, kbCtrlDel, kbCtrlT, kbCtrlQ:
      else
        begin
          C := GetAltChar(Event.KeyCode);
          if C = #0 then
          begin
            O := True;
            if Event.KeyCode = kbEsc then ClearEvent(Event);
          end
          else
          begin
            Event.KeyCode := Ord(LoCase(C));
          end;
        end;
      end;
    end;
  end;
  if not O then O := ((Event.What and evMouseDown > 0) or ((Event.What and evCommand > 0) and (Event.Command <> 0)));
  if O then
  begin
    SearchInUse := False;
    LastShiftState := MaxByte;
    Exit;
  end;
  TInputLine.HandleEvent(Event);
  if C <> #0 then ClearEvent(Event);
end;

{Check if the file name typed in the speed-search dialog box exists in the
  panel
  Input : S: file name in the input line
          SuppressFill: unused, only needed for the parameter syntax
  Output: when True, the file name is valid}
function TFileValid.IsValidInput(var S: string; SuppressFill: Boolean): Boolean;
var
  T             : string;
begin
  if GetPanelModeAttrib(Panel^.Mode, paASCII) or Panel^.GEOSFormat then T := S else
    T := ReconvertCBMName(S, False, True, hxNone);
  IsValidInput := Panel^.SearchPattern(T, Panel^.Cur, True);
end;

{Initialize function key controllable radio buttons
  Input : Bounds: the rectangle containing the radio buttons
          AStrings: the list of texts assigned to each radio button
          AMin, AMax: limits of command word accepted by the radio buttons}
constructor TFuncButtons.Init(var Bounds: TRect; AStrings: PSItem; AMin, AMax: Word);
begin
  TRadioButtons.Init(Bounds, AStrings);
  Min := AMin;
  Max := AMax;
end;

{Handle function key controllable radio button events
  Input : Event: event record to be handled}
procedure TFuncButtons.HandleEvent(var Event: TEvent);
begin
  if (Event.What = evCommand) and (Event.Command in [Min..Max]) then
  begin
    if Min = Max then Press(Value, True) else
      Press(Event.Command - Min, False);
    DrawView;
    ClearEvent(Event);
  end;
  TRadioButtons.HandleEvent(Event);
end;

{Initialize the location set
  Input : Fill: when True, the set is filled with ones; otherwise cleared
                with zeros}
procedure TLocationSet.Init(Fill: Boolean);
var
  B             : Byte;
begin
  B := 0;
  if Fill then B := $FF;
  FillChar(Bitmap, LocSetSize, B);
end;

{Include a location in the set
  Input : Loc: the location to include}
procedure TLocationSet.Include(Loc: Word);
begin
  Bitmap[Loc shr 3] := Bitmap[Loc shr 3] or (1 shl (Loc and 7));
end;

{Exclude a location from the set
  Input : Loc: the location to exclude}
procedure TLocationSet.Exclude(Loc: Word);
begin
  Bitmap[Loc shr 3] := Bitmap[Loc shr 3] and not (1 shl (Loc and 7));
end;

{Check whether a location is in the set
  Input : Loc: the location to check
  Output: when True, the location is in the set}
function TLocationSet.Contains(Loc: Word): Boolean;
begin
  Contains := ((Bitmap[Loc shr 3] and (1 shl (Loc and 7))) > 0);
end;

end.
