
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Turbo Vision Unit                               }
{                                                       }
{       Copyright (c) 1992 Borland International        }
{                                                       }
{*******************************************************}

unit Views;

{$A+,B-,D+,E-,F+,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  Drivers, Memory, Objects;

const
{Length of palette}
  PaletteLen    = 40;
{Undefined attribute}
  ErrorAttr     = $07;

const

{ TView State masks }

  sfVisible     = $0001;
  sfCursorVis   = $0002;
  sfCursorIns   = $0004;
  sfShadow      = $0008;
  sfActive      = $0010;
  sfSelected    = $0020;
  sfFocused     = $0040;
  sfDragging    = $0080;
  sfDisabled    = $0100;
  sfModal       = $0200;
  sfDefault     = $0400;
  sfExposed     = $0800;
  sfAlwaysIns   = $1000;

{ TView Option masks }

  ofSelectable  = $0001;
  ofTopSelect   = $0002;
  ofFirstClick  = $0004;
  ofFramed      = $0008;
  ofPreProcess  = $0010;
  ofPostProcess = $0020;
  ofBuffered    = $0040;
  ofValidate    = $0400;

{ TView GrowMode masks }

  gfGrowLoX = $01;
  gfGrowLoY = $02;
  gfGrowHiX = $04;
  gfGrowHiY = $08;
  gfGrowAll = $0F;

{ TView Help context codes }

  hcNoContext = 0;

{ TWindow palette entries }

  wpDialog     = 0;
  wpConfig     = 1;
  wpHistory    = 2;
  wpHelp       = 3;
  wpError      = 4;
  wpViewEdit   = 5;

{ Standard command codes }

  cmNone    = 0;
  cmQuit    = 1;
  cmError   = 2;
  cmMenu    = 3;
  cmClose   = 4;
  cmNext    = 5;
  cmPrev    = 6;

{ TDialog standard commands }

  cmOK      = 7;
  cmYes     = 8;
  cmNo      = 9;
  cmSkip    = 10;
  cmExtra   = 11;
  cmExtra2  = 12;
  cmCancel  = 13;

{ Standard messages }

  cmCommandSetChanged = 14;

{ View types }
  vtNone         = 0;
  vtButton       = 1;
  vtCluster      = 2;
  vtInputLine    = 3;
  vtItemFrame    = 4;
  vtHistory      = 5;

{ Color palettes }

  CFrame         = #1;
  CDialog        = #17#18#19#20;
  CConfig        = #21#22#23#24;
  CHistory       = #25#26#0#0;
  CHelp          = #27#28#29#30;
  CError         = #31#32#33#34;
  CViewEdit      = #1#2#23#3#4#36#8;

{ TDrawBuffer maximum view width }

  MaxViewWidth = MaxScrWidth;

{ Shadow modes }

  shNone = 0;
  shLowBrightness = 1;
  shGrayOnBlack = 2;

{ Default shadow attributes }

  DefShadowAttr = $07;
  DefShadowANDAttr = $77;
  DefShadowORAttr = 0;

const

{ Frame characters }

  DefFrameChars = '¿ Ŵ';
  DefDoubleFrameChars = 'ѻ Ŷϼ';
  AltDefDoubleFrameChars = 'ͻ źͼ';
  DefMenuFrameChars = 'Ŀ ĳ';

{ Other graphical characters }

  DefCheckedChar = #251;
  AltDefCheckedChar = '*';
  CancelledChar = 'x';

  ArrowChars: string[4] = #27#26#25#24;
  DoubleArrowChars: string[2] = #29#18;
  FullArrowChars: string[2] = #17#16;
  SelectionChars: string[3] = #176#222#219;
  VertBlockChars: string[3] = #223#220#219;
  DefDotChar: Char = #249;
  AltDefDotChar: Char = '.';
  CenterBlockChar: Char = #254;

type

{ Command sets }

  PCommandSet = ^TCommandSet;
  TCommandSet = set of Byte;

{ Color palette type }

  PPalette = ^TPalette;
  TPalette = string[PaletteLen];

{ TDrawBuffer, buffer used by draw methods }

  TDrawBuffer = array[0..MaxViewWidth - 1] of Word;
  PDrawBuffer = ^TDrawBuffer;

{ TView object Pointer }

  PView = ^TView;

{ TGroup object Pointer }

  PGroup = ^TGroup;

{ TView object }

  TView = object(TObject)
    Owner: PGroup;
    Next: PView;
    Origin: TPoint;
    Size: TPoint;
    Cursor: TPoint;
    GrowMode: Byte;
    HelpCtx: Word;
    State: Word;
    Options: Word;
    EventMask: Word;
    ViewType: Byte;
    constructor Init(var Bounds: TRect);
    destructor Done; virtual;
    procedure BlockCursor;
    procedure CalcBounds(var Bounds: TRect; Delta: TPoint); virtual;
    procedure ChangeBounds(var Bounds: TRect); virtual;
    procedure ClearEvent(var Event: TEvent);
    function CommandEnabled(Command: Word): Boolean;
    procedure DisableCommands(Commands: TCommandSet);
    procedure Draw; virtual;
    procedure DrawView;
    procedure EnableCommands(Commands: TCommandSet);
    procedure EndModal(Command: Word); virtual;
    function EventAvail: Boolean;
    function Execute: Word; virtual;
    function Exposed: Boolean;
    function Focus: Boolean;
    procedure GetBounds(var Bounds: TRect);
    procedure GetClipRect(var Clip: TRect);
    function GetColor(Color: Word): Word;
    procedure GetCommands(var Commands: TCommandSet);
    procedure GetData(var Rec); virtual;
    procedure GetEvent(var Event: TEvent); virtual;
    procedure GetExtent(var Extent: TRect);
    function GetHelpCtx: Word; virtual;
    function GetPalette: PPalette; virtual;
    function GetState(AState: Word): Boolean;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Hide;
    procedure HideCursor;
    procedure KeyEvent(var Event: TEvent);
    procedure MakeFirst;
    procedure MakeLocal(Source: TPoint; var Dest: TPoint);
    function MouseEvent(var Event: TEvent; Mask: Word): Boolean;
    function MouseInView(Mouse: TPoint): Boolean;
    procedure MoveTo(X, Y: Integer);
    function NextView: PView;
    procedure NormalCursor;
    function Prev: PView;
    function PrevView: PView;
    procedure PutEvent(var Event: TEvent); virtual;
    procedure PutInFrontOf(Target: PView);
    procedure Select;
    procedure SetBounds(var Bounds: TRect);
    procedure SetCommands(Commands: TCommandSet);
    procedure SetCmdState(Commands: TCommandSet; Enable: Boolean);
    procedure SetCursor(X, Y: Integer);
    procedure SetData(var Rec); virtual;
    procedure SetState(AState: Word; Enable: Boolean); virtual;
    procedure Show;
    procedure ShowCursor;
    function TopView: PView;
    function Valid(Command: Word): Boolean; virtual;
    procedure WriteBuf(X, Y, W, H: Integer; var Buf);
    procedure WriteChar(X, Y: Integer; C: Char; Color: Byte; Count: Integer);
    procedure WriteLine(X, Y, W, H: Integer; var Buf);
    procedure WriteStr(X, Y: Integer; const Str: string; Color: Byte);
  private
    procedure DrawCursor;
    procedure DrawHide(LastView: PView);
    procedure DrawShow(LastView: PView);
    procedure DrawUnderRect(var R: TRect; LastView: PView);
    procedure DrawUnderView(DoShadow: Boolean; LastView: PView);
    procedure ResetCursor; virtual;
  end;

{ TFrame types }

  TTitleStr = string[80];

{ TFrame object }

  PFrame = ^TFrame;
  TFrame = object(TView)
    FrameX, FrameY: Integer;
    constructor Init(var Bounds: TRect; AFrameX, AFrameY: Integer);
    procedure Draw; virtual;
    function GetPalette: PPalette; virtual;
    procedure SetState(AState: Word; Enable: Boolean); virtual;
  private
    FrameMode: Word;
    procedure FrameLine(var FrameBuf; Y, N: Integer; Color: Byte);
  end;

{ Video buffer }

  PVideoBuf = ^TVideoBuf;
  TVideoBuf = array[0..(MaxScrWidth * MaxScrHeight) - 1] of Word;

{ Selection modes }

  SelectMode = (NormalSelect, EnterSelect, LeaveSelect);

{ TGroup object }

  TGroup = object(TView)
    Last: PView;
    Current: PView;
    Phase: (phFocused, phPreProcess, phPostProcess);
    Buffer: PVideoBuf;
    EndState: Word;
    constructor Init(var Bounds: TRect);
    destructor Done; virtual;
    procedure ChangeBounds(var Bounds: TRect); virtual;
    procedure Delete(P: PView);
    procedure Draw; virtual;
    procedure EndModal(Command: Word); virtual;
    procedure EventError(var Event: TEvent); virtual;
    function ExecView(P: PView; MakeCur, DelView: Boolean): Word;
    function Execute: Word; virtual;
    function First: PView;
    function FirstThat(P: Pointer): PView;
    function FocusNext(Forwards: Boolean): Boolean;
    procedure ForEach(P: Pointer);
    function GetHelpCtx: Word; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Insert(P: PView);
    procedure InsertBefore(P, Target: PView);
    procedure InsertView(P, Target: PView);
    procedure Lock;
    procedure Redraw;
    procedure RemoveView(P: PView);
    procedure ResetCurrent;
    procedure SelectNext(Forwards: Boolean);
    procedure SetState(AState: Word; Enable: Boolean); virtual;
    procedure Unlock;
    function Valid(Command: Word): Boolean; virtual;
  private
    Clip: TRect;
    LockFlag: Byte;
    function At(Index: Integer): PView;
    procedure DrawSubViews(P, Bottom: PView);
    function FirstMatch(AState: Word; AOptions: Word): PView;
    function FindNext(Forwards: Boolean): PView;
    procedure FreeBuffer;
    procedure GetBuffer;
    function IndexOf(P: PView): Integer;
    procedure ResetCursor; virtual;
    procedure SetCurrent(P: PView; Mode: SelectMode);
  end;

{ TWindow object }

  PWindow = ^TWindow;
  TWindow = object(TGroup)
    Palette: Integer;
    Frame: PFrame;
    FrameX, FrameY: Integer;
    Title: PString;
    constructor Init(var Bounds: TRect; ATitle: TTitleStr; AFrameX, AFrameY: Integer);
    destructor Done; virtual;
    function GetPalette: PPalette; virtual;
    function GetTitle: TTitleStr; virtual;
    procedure InitFrame; virtual;
    procedure SetState(AState: Word; Enable: Boolean); virtual;
  end;

{ Message dispatch function }

procedure Message(Receiver: PView; What, Command: Word);

const

{ Redraw all views completely }

  RedrawAll: Boolean = False;

{ Event masks }

  PositionalEvents: Word = evMouse;
  FocusedEvents: Word = evKeyboard + evCommand;

{ Shadow definitions }

  ShadowSize: TPoint = (X: 2; Y: 1);
  ShadowAttr: Byte = DefShadowAttr;
  ShadowANDAttr: Byte = DefShadowANDAttr;
  ShadowORAttr: Byte = DefShadowORAttr;

{ True if the command set has changed since being set to false }

  CommandSetChanged: Boolean = False;

{ Direction flags }

  drNone        = 0;
  drLeft        = 1;
  drRight       = 2;
  drUp          = 3;
  drDown        = 4;
  drLeftClose   = 5;

{ Default help context }

  hcHelp        = 100;

var
  CanChangeChars, Laptop, ShowMenu: Boolean;
  LastShiftState: Byte;
  AppHelpCtx, ShiftHelpCtx, AltHelpCtx: Word;
  HistoryItem: Integer;

  FrameChars: string[16];
  DoubleFrameChars: string[16];
  MenuFrameChars: string[12];
  CheckedChar: Char;
  DotChar: Char;

implementation

uses
  App;

const
  OwnerGroup: PGroup = nil;
  TheTopView: PView = nil;

{ Current command set. All but window commands are active by default }

  CurCommandSet: TCommandSet = [0..255];

{ Convert color into attribute                          }
{ In    AL = Color                                      }
{ Out   AL = Attribute                                  }

procedure MapColor; near; assembler;
const
  Self = 6;
  TView_GetPalette = vmtHeaderSize + $24;
asm
        OR      AL,AL
        JE      @@3
        LES     DI,[BP].Self
@@1:    PUSH    ES
        PUSH    DI
        PUSH    AX
        PUSH    ES
        PUSH    DI
        MOV     DI,ES:[DI]
        CALL    DWORD PTR [DI].TView_GetPalette
        MOV     BX,AX
        MOV     ES,DX
        OR      AX,DX
        POP     AX
        POP     DI
        POP     DX
        JE      @@2
        CMP     AL,ES:[BX]
        JA      @@3
        SEGES   XLAT
        OR      AL,AL
        JE      @@3
@@2:    MOV     ES,DX
        LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES
        OR      SI,DI
        JNE     @@1
        JMP     @@4
@@3:    MOV     AL,ErrorAttr
@@4:
end;

{ Convert color pair into attribute pair                }
{ In    AX = Color pair                                 }
{ Out   AX = Attribute pair                             }

procedure MapCPair; near; assembler;
asm
        OR      AH,AH
        JE      @@1
        XCHG    AL,AH
        CALL    MapColor
        XCHG    AL,AH
@@1:    CALL    MapColor
end;

{ Write to view                                         }
{ In    AX    = Y coordinate                            }
{       BX    = X coordinate                            }
{       CX    = Count                                   }
{       ES:DI = Buffer Pointer                          }

procedure WriteView; near; assembler;
const
  Self   =   6;
  Target =  -4;
  Buffer =  -8;
  BufOfs = -10;
asm
        MOV     [BP].BufOfs,BX
        MOV     [BP].Buffer[0],DI
        MOV     [BP].Buffer[2],ES
        ADD     CX,BX
        XOR     DL,DL
        LES     DI,[BP].Self
        OR      AX,AX
        JL      @@3
        CMP     AX,ES:[DI].TView.Size.Y
        JGE     @@3
        OR      BX,BX
        JGE     @@1
        XOR     BX,BX
@@1:    CMP     CX,ES:[DI].TView.Size.X
        JLE     @@2
        MOV     CX,ES:[DI].TView.Size.X
@@2:    CMP     BX,CX
        JL      @@10
@@3:    RET
@@10:   TEST    ES:[DI].TView.State,sfVisible
        JE      @@3
        CMP     ES:[DI].TView.Owner.Word[2],0
        JE      @@3
        MOV     [BP].Target[0],DI
        MOV     [BP].Target[2],ES
        ADD     AX,ES:[DI].TView.Origin.Y
        MOV     SI,ES:[DI].TView.Origin.X
        ADD     BX,SI
        ADD     CX,SI
        ADD     [BP].BufOfs,SI
        LES     DI,ES:[DI].TView.Owner
        CMP     AX,ES:[DI].TGroup.Clip.A.Y
        JL      @@3
        CMP     AX,ES:[DI].TGroup.Clip.B.Y
        JGE     @@3
        CMP     BX,ES:[DI].TGroup.Clip.A.X
        JGE     @@11
        MOV     BX,ES:[DI].TGroup.Clip.A.X
@@11:   CMP     CX,ES:[DI].TGroup.Clip.B.X
        JLE     @@12
        MOV     CX,ES:[DI].TGroup.Clip.B.X
@@12:   CMP     BX,CX
        JGE     @@3
        LES     DI,ES:[DI].TGroup.Last
@@20:   LES     DI,ES:[DI].TView.Next
        CMP     DI,[BP].Target[0]
        JNE     @@21
        MOV     SI,ES
        CMP     SI,[BP].Target[2]
        JNE     @@21
        JMP     @@40
@@21:   TEST    ES:[DI].TView.State,sfVisible
        JE      @@20
        MOV     SI,ES:[DI].TView.Origin.Y
        CMP     AX,SI
        JL      @@20
        ADD     SI,ES:[DI].TView.Size.Y
        CMP     AX,SI
        JL      @@23
        TEST    ES:[DI].TView.State,sfShadow
        JE      @@20
        ADD     SI,ShadowSize.Y
        CMP     AX,SI
        JGE     @@20
        MOV     SI,ES:[DI].TView.Origin.X
        ADD     SI,ShadowSize.X
        CMP     BX,SI
        JGE     @@22
        CMP     CX,SI
        JLE     @@20
        CALL    @@30
@@22:   ADD     SI,ES:[DI].TView.Size.X
        JMP     @@26
@@23:   MOV     SI,ES:[DI].TView.Origin.X
        CMP     BX,SI
        JGE     @@24
        CMP     CX,SI
        JLE     @@20
        CALL    @@30
@@24:   ADD     SI,ES:[DI].TView.Size.X
        CMP     BX,SI
        JGE     @@25
        CMP     CX,SI
        JLE     @@31
        MOV     BX,SI
@@25:   TEST    ES:[DI].TView.State,sfShadow
        JE      @@20
        PUSH    SI
        MOV     SI,ES:[DI].TView.Origin.Y
        ADD     SI,ShadowSize.Y
        CMP     AX,SI
        POP     SI
        JL      @@27
        ADD     SI,ShadowSize.X
@@26:   CMP     BX,SI
        JGE     @@27
        INC     DL
        CMP     CX,SI
        JLE     @@27
        CALL    @@30
        DEC     DL
@@27:   JMP     @@20
@@30:   PUSH    [BP].Target.Word[2]
        PUSH    [BP].Target.Word[0]
        PUSH    [BP].BufOfs.Word[0]
        PUSH    ES
        PUSH    DI
        PUSH    SI
        PUSH    DX
        PUSH    CX
        PUSH    AX
        MOV     CX,SI
        CALL    @@20
        POP     AX
        POP     CX
        POP     DX
        POP     SI
        POP     DI
        POP     ES
        POP     [BP].BufOfs.Word[0]
        POP     [BP].Target.Word[0]
        POP     [BP].Target.Word[2]
        MOV     BX,SI
@@31:   RET
@@40:   LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES:[DI].TGroup.Buffer.Word[2]
        OR      SI,SI
        JE      @@44
        CMP     SI,ScreenBuffer.Word[2]
        JE      @@41
        CALL    @@50
        JMP     @@44
@@41:   CLI
        CMP     AX,MouseWhere.Y
        JNE     @@42
        CMP     BX,MouseWhere.X
        JA      @@42
        CMP     CX,MouseWhere.X
        JA      @@43
@@42:   MOV     MouseIntFlag,0
        STI
        CALL    @@50
        CMP     MouseIntFlag,0
        JE      @@44
@@43:   STI
        CALL    HideMouse
        CALL    @@50
        CALL    ShowMouse
@@44:   CMP     ES:[DI].TGroup.LockFlag,0
        JNE     @@31
        JMP     @@10
@@50:   PUSH    ES
        PUSH    DS
        PUSH    DI
        PUSH    CX
        PUSH    AX
        MUL     ES:[DI].TView.Size.X.Byte[0]
        ADD     AX,BX
        SHL     AX,1
        ADD     AX,ES:[DI].TGroup.Buffer.Word[0]
        MOV     DI,AX
        MOV     ES,SI
        XOR     AL,AL
        CMP     SI,ScreenBuffer.Word[2]
        JNE     @@51
        MOV     AL,CheckSnow
@@51:   PUSH    BX
        SUB     CX,BX
        MOV     SI,BX
        MOV     BL,ShadowANDAttr
        MOV     BH,ShadowORAttr
        SUB     SI,[BP].BufOfs
        SHL     SI,1
        ADD     SI,[BP].Buffer.Word[0]
        MOV     DS,[BP].Buffer.Word[2]
        CLD
        OR      AL,AL
        JNE     @@60
        OR      DL,DL
        JNE     @@52
        REP     MOVSW
        JMP     @@70
@@52:   LODSW
        TEST    AH,80H
        JNE     @@71
        AND     AH,0FH
@@71:   TEST    AH,08H
        JNE     @@72
        AND     AH,0F0H
@@72:   AND     AH,BL
        OR      AH,BH
        STOSW
        LOOP    @@52
        JMP     @@70
@@60:   PUSH    DX
        PUSH    BP
        OR      DL,DL
        MOV     DX,03DAH
        JNE     @@65
@@61:   LODSW
        MOV     BP,AX
@@62:   IN      AL,DX
        TEST    AL,1
        JNE     @@62
        CLI
@@63:   IN      AL,DX
        TEST    AL,1
        JE      @@63
        MOV     AX,BP
        STOSW
        STI
        LOOP    @@61
        JMP     @@68
@@65:   LODSW
        TEST    AH,80H
        JNE     @@73
        AND     AH,0FH
@@73:   TEST    AH,08H
        JNE     @@74
        AND     AH,0F0H
@@74:   AND     AH,BL
        OR      AH,BH
        MOV     BP,AX
@@66:   IN      AL,DX
        TEST    AL,1
        JNE     @@66
        CLI
@@67:   IN      AL,DX
        TEST    AL,1
        JE      @@67
        MOV     AX,BP
        STOSW
        STI
        LOOP    @@65
@@68:   POP     BP
        POP     DX
@@70:   POP     BX
        MOV     SI,ES
        POP     AX
        POP     CX
        POP     DI
        POP     DS
        POP     ES
        RET
end;

{ TView }

constructor TView.Init(var Bounds: TRect);
begin
  TObject.Init;
  Owner := nil;
  State := sfVisible + sfAlwaysIns;
  SetBounds(Bounds);
  HelpCtx := hcNoContext;
  EventMask := evMouseDown + evKeyDown + evCommand;
end;

destructor TView.Done;
begin
  Hide;
  if Owner <> nil then Owner^.Delete(@Self);
end;

procedure TView.BlockCursor;
begin
  SetState(sfCursorIns, True);
end;

procedure TView.CalcBounds(var Bounds: TRect; Delta: TPoint);
begin
  GetBounds(Bounds);
  if GrowMode and gfGrowLoX <> 0 then Inc(Bounds.A.X, Delta.X);
  if GrowMode and gfGrowHiX <> 0 then Inc(Bounds.B.X, Delta.X);
  if Bounds.B.X - Bounds.A.X > MaxViewWidth then
    Bounds.B.X := Bounds.A.X + MaxViewWidth;
  if GrowMode and gfGrowLoY <> 0 then Inc(Bounds.A.Y, Delta.Y);
  if GrowMode and gfGrowHiY <> 0 then Inc(Bounds.B.Y, Delta.Y);
end;

procedure TView.ChangeBounds(var Bounds: TRect);
begin
  SetBounds(Bounds);
  DrawView;
end;

procedure TView.ClearEvent(var Event: TEvent);
begin
  Event.What := evNothing;
end;

function TView.CommandEnabled(Command: Word): Boolean;
begin
  CommandEnabled := (Command > 255) or (Command in CurCommandSet);
end;

procedure TView.DisableCommands(Commands: TCommandSet);
begin
  CommandSetChanged := CommandSetChanged or (CurCommandSet * Commands <> []);
  CurCommandSet := CurCommandSet - Commands;
end;

procedure TView.Draw;
var
  B: TDrawBuffer;
begin
  MoveChar(B, ' ', GetColor(1), Size.X);
  WriteLine(0, 0, Size.X, Size.Y, B);
end;

procedure TView.DrawCursor;
begin
  if State and sfFocused <> 0 then ResetCursor;
end;

procedure TView.DrawHide(LastView: PView);
begin
  DrawCursor;
  DrawUnderView(State and sfShadow <> 0, LastView);
end;

procedure TView.DrawShow(LastView: PView);
begin
  DrawView;
  if State and sfShadow <> 0 then DrawUnderView(True, LastView);
end;

procedure TView.DrawUnderRect(var R: TRect; LastView: PView);
begin
  Owner^.Clip.Intersect(R);
  Owner^.DrawSubViews(NextView, LastView);
  Owner^.GetExtent(Owner^.Clip);
end;

procedure TView.DrawUnderView(DoShadow: Boolean; LastView: PView);
var
  R: TRect;
begin
  GetBounds(R);
  if DoShadow then
  begin
    Inc(R.B.X, ShadowSize.X);
    Inc(R.B.Y, ShadowSize.Y);
  end;
  DrawUnderRect(R, LastView);
end;

procedure TView.DrawView;
begin
  if Exposed then
  begin
    Draw;
    DrawCursor;
  end;
end;

procedure TView.EnableCommands(Commands: TCommandSet);
begin
  CommandSetChanged := CommandSetChanged or
    (CurCommandSet * Commands <> Commands);
  CurCommandSet := CurCommandSet + Commands;
end;

procedure TView.EndModal(Command: Word);
var
  P: PView;
begin
  P := TopView;
  if TopView <> nil then TopView^.EndModal(Command);
end;

function TView.EventAvail: Boolean;
var
  Event: TEvent;
begin
  GetEvent(Event);
  if Event.What <> evNothing then PutEvent(Event);
  EventAvail := Event.What <> evNothing;
end;

procedure TView.GetBounds(var Bounds: TRect); assembler;
asm
        PUSH    DS
        LDS     SI,Self
        ADD     SI,OFFSET TView.Origin
        LES     DI,Bounds
        CLD
        LODSW                           {Origin.X}
        MOV     CX,AX
        STOSW
        LODSW                           {Origin.Y}
        MOV     DX,AX
        STOSW
        LODSW                           {Size.X}
        ADD     AX,CX
        STOSW
        LODSW                           {Size.Y}
        ADD     AX,DX
        STOSW
        POP     DS
end;

function TView.Execute: Word;
begin
  Execute := cmCancel;
end;

function TView.Exposed: Boolean; assembler;
var
  Target: Pointer;
asm
        LES     DI,Self
        TEST    ES:[DI].TView.State,sfExposed
        JE      @@2
        XOR     AX,AX
        CMP     AX,ES:[DI].TView.Size.X
        JGE     @@2
        CMP     AX,ES:[DI].TView.Size.Y
        JGE     @@2
@@1:    XOR     BX,BX
        MOV     CX,ES:[DI].TView.Size.X
        PUSH    AX
        CALL    @@11
        POP     AX
        JNC     @@3
        LES     DI,Self
        INC     AX
        CMP     AX,ES:[DI].TView.Size.Y
        JL      @@1
@@2:    MOV     AL,0
        JMP     @@30
@@3:    MOV     AL,1
        JMP     @@30
@@8:    STC
@@9:    RETN
@@10:   LES     DI,ES:[DI].TView.Owner
        CMP     ES:[DI].TGroup.Buffer.Word[2],0
        JNE     @@9
@@11:   MOV     Target.Word[0],DI
        MOV     Target.Word[2],ES
        ADD     AX,ES:[DI].TView.Origin.Y
        MOV     SI,ES:[DI].TView.Origin.X
        ADD     BX,SI
        ADD     CX,SI
        LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES
        OR      SI,DI
        JE      @@9
        CMP     AX,ES:[DI].TGroup.Clip.A.Y
        JL      @@8
        CMP     AX,ES:[DI].TGroup.Clip.B.Y
        JGE     @@8
        CMP     BX,ES:[DI].TGroup.Clip.A.X
        JGE     @@12
        MOV     BX,ES:[DI].TGroup.Clip.A.X
@@12:   CMP     CX,ES:[DI].TGroup.Clip.B.X
        JLE     @@13
        MOV     CX,ES:[DI].TGroup.Clip.B.X
@@13:   CMP     BX,CX
        JGE     @@8
        LES     DI,ES:[DI].TGroup.Last
@@20:   LES     DI,ES:[DI].TView.Next
        CMP     DI,Target.Word[0]
        JNE     @@21
        MOV     SI,ES
        CMP     SI,Target.Word[2]
        JE      @@10
@@21:   TEST    ES:[DI].TView.State,sfVisible
        JE      @@20
        MOV     SI,ES:[DI].TView.Origin.Y
        CMP     AX,SI
        JL      @@20
        ADD     SI,ES:[DI].TView.Size.Y
        CMP     AX,SI
        JGE     @@20
        MOV     SI,ES:[DI].TView.Origin.X
        CMP     BX,SI
        JL      @@22
        ADD     SI,ES:[DI].TView.Size.X
        CMP     BX,SI
        JGE     @@20
        MOV     BX,SI
        CMP     BX,CX
        JL      @@20
        STC
        RETN
@@22:   CMP     CX,SI
        JLE     @@20
        ADD     SI,ES:[DI].TView.Size.X
        CMP     CX,SI
        JG      @@23
        MOV     CX,ES:[DI].TView.Origin.X
        JMP     @@20
@@23:   PUSH    Target.Word[2]
        PUSH    Target.Word[0]
        PUSH    ES
        PUSH    DI
        PUSH    SI
        PUSH    CX
        PUSH    AX
        MOV     CX,ES:[DI].TView.Origin.X
        CALL    @@20
        POP     AX
        POP     CX
        POP     BX
        POP     DI
        POP     ES
        POP     Target.Word[0]
        POP     Target.Word[2]
        JC      @@20
        RETN
@@30:
end;

function TView.Focus: Boolean;
var
  Result: Boolean;
begin
  Result := True;
  if State and (sfSelected + sfModal) = 0 then
  begin
    if Owner <> nil then
    begin
      Result := Owner^.Focus;
      if Result then
        if ((Owner^.Current = nil) or
          (Owner^.Current^.Options and ofValidate = 0)) then
          Select
        else
          Result := False;
    end;
  end;
  Focus := Result;
end;

procedure TView.GetClipRect(var Clip: TRect);
begin
  GetBounds(Clip);
  if Owner <> nil then Clip.Intersect(Owner^.Clip);
  Clip.Move(-Origin.X, -Origin.Y);
end;

function TView.GetColor(Color: Word): Word; assembler;
asm
        MOV     AX,Color
        CALL    MapCPair
end;

procedure TView.GetCommands(var Commands: TCommandSet);
begin
  Commands := CurCommandSet;
end;

procedure TView.GetData(var Rec);
begin
end;

procedure TView.GetEvent(var Event: TEvent);
begin
  if Owner <> nil then Owner^.GetEvent(Event);
end;

procedure TView.GetExtent(var Extent: TRect); assembler;
asm
        PUSH    DS
        LDS     SI,Self
        ADD     SI,OFFSET TView.Size
        LES     DI,Extent
        CLD
        XOR     AX,AX
        STOSW
        STOSW
        MOVSW
        MOVSW
        POP     DS
end;

function TView.GetHelpCtx: Word;
begin
  if GetShiftState = 0 then GetHelpCtx := HelpCtx else GetHelpCtx := hcNoContext;
end;

function TView.GetPalette: PPalette;
begin
  GetPalette := nil;
end;

function TView.GetState(AState: Word): Boolean;
begin
  GetState := State and AState = AState;
end;

procedure TView.HandleEvent(var Event: TEvent);
begin
  if Event.What = evMouseDown then
    if (State and (sfSelected + sfDisabled) = 0) and
       (Options and ofSelectable <> 0) then
      if not Focus or (Options and ofFirstClick = 0) then
        ClearEvent(Event);
end;

procedure TView.Hide;
begin
  if State and sfVisible <> 0 then SetState(sfVisible, False);
end;

procedure TView.HideCursor;
begin
  SetState(sfCursorVis, False);
end;

procedure TView.KeyEvent(var Event: TEvent);
begin
  repeat GetEvent(Event) until Event.What = evKeyDown;
end;

procedure TView.MakeFirst;
begin
  PutInFrontOf(Owner^.First);
end;

procedure TView.MakeLocal(Source: TPoint; var Dest: TPoint); assembler;
asm
        LES     DI,Self
        XOR     AX,AX
        MOV     DX,AX
@@1:    ADD     AX,ES:[DI].TView.Origin.X
        ADD     DX,ES:[DI].TView.Origin.Y
        LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES
        OR      SI,DI
        JNE     @@1
        NEG     AX
        NEG     DX
        ADD     AX,Source.X
        ADD     DX,Source.Y
        LES     DI,Dest
        CLD
        STOSW
        XCHG    AX,DX
        STOSW
end;

function TView.MouseEvent(var Event: TEvent; Mask: Word): Boolean;
begin
  repeat GetEvent(Event) until Event.What and (Mask or evMouseUp) <> 0;
  MouseEvent := Event.What <> evMouseUp;
end;

function TView.MouseInView(Mouse: TPoint): Boolean;
var
  Extent: TRect;
begin
  MakeLocal(Mouse, Mouse);
  GetExtent(Extent);
  MouseInView := Extent.Contains(Mouse);
end;

procedure TView.MoveTo(X, Y: Integer);
var
  R: TRect;
begin
  R.Assign(X, Y, X + Size.X, Y + Size.Y);
  ChangeBounds(R);
end;

function TView.NextView: PView;
begin
  if @Self = Owner^.Last then NextView := nil else NextView := Next;
end;

procedure TView.NormalCursor;
begin
  SetState(sfCursorIns, False);
end;

function TView.Prev: PView; assembler;
asm
        LES     DI,Self
        MOV     CX,DI
        MOV     BX,ES
@@1:    MOV     AX,DI
        MOV     DX,ES
        LES     DI,ES:[DI].TView.Next
        CMP     DI,CX
        JNE     @@1
        MOV     SI,ES
        CMP     SI,BX
        JNE     @@1
end;

function TView.PrevView: PView;
begin
  if @Self = Owner^.First then PrevView := nil else PrevView := Prev;
end;

procedure TView.PutEvent(var Event: TEvent);
begin
  if Owner <> nil then Owner^.PutEvent(Event);
end;

procedure TView.PutInFrontOf(Target: PView);
var
  P, LastView: PView;

procedure MoveView;
begin
  Owner^.RemoveView(@Self);
  Owner^.InsertView(@Self, Target);
end;

begin
  if (Owner <> nil) and (Target <> @Self) and (Target <> NextView) and
    ((Target = nil) or (Target^.Owner = Owner)) then
    if State and sfVisible = 0 then MoveView else
    begin
      LastView := NextView;
      if LastView <> nil then
      begin
        P := Target;
        while (P <> nil) and (P <> LastView) do P := P^.NextView;
        if P = nil then LastView := Target;
      end;
      State := State and not sfVisible;
      if LastView = Target then DrawHide(LastView);
      MoveView;
      State := State or sfVisible;
      if LastView <> Target then DrawShow(LastView);
      if Options and ofSelectable <> 0 then
      begin
        Owner^.ResetCurrent;
        Owner^.ResetCursor;
      end;
    end;
end;

procedure TView.ResetCursor; assembler;
asm
        LES     DI,Self
        MOV     AX,ES:[DI].TView.State
        NOT     AX
        TEST    AX,sfVisible+sfCursorVis+sfFocused
        JNE     @@4
        MOV     AX,ES:[DI].TView.Cursor.Y
        MOV     DX,ES:[DI].TView.Cursor.X
@@1:    OR      AX,AX
        JL      @@4
        CMP     AX,ES:[DI].TView.Size.Y
        JGE     @@4
        OR      DX,DX
        JL      @@4
        CMP     DX,ES:[DI].TView.Size.X
        JG      @@4
        ADD     AX,ES:[DI].TView.Origin.Y
        ADD     DX,ES:[DI].TView.Origin.X
        MOV     CX,DI
        MOV     BX,ES
        LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES
        OR      SI,DI
        JE      @@5
        TEST    ES:[DI].TView.State,sfVisible
        JE      @@4
        LES     DI,ES:[DI].TGroup.Last
@@2:    LES     DI,ES:[DI].TView.Next
        CMP     CX,DI
        JNE     @@3
        MOV     SI,ES
        CMP     BX,SI
        JNE     @@3
        LES     DI,ES:[DI].TView.Owner
        JMP     @@1
@@3:    TEST    ES:[DI].TView.State,sfVisible
        JE      @@2
        MOV     SI,ES:[DI].TView.Origin.Y
        CMP     AX,SI
        JL      @@2
        ADD     SI,ES:[DI].TView.Size.Y
        CMP     AX,SI
        JGE     @@2
        MOV     SI,ES:[DI].TView.Origin.X
        CMP     DX,SI
        JL      @@2
        ADD     SI,ES:[DI].TView.Size.X
        CMP     DX,SI
        JGE     @@2
@@4:    CALL    CursorOff
        JMP     @@10
@@5:    LES     SI,Self
@@9:    MOV     BX,ES
        LES     DI,ES:[SI].TView.Owner
        MOV     CX,ES
        OR      CX,DI
        JE      @@8
        LES     DI,ES:[DI].TGroup.Current
        MOV     CX,ES
        CMP     SI,DI
        JNE     @@4
        CMP     BX,CX
        JNE     @@4
        LES     SI,ES:[SI].TView.Owner
        JMP     @@9
@@8:    MOV     DH,AL
        XOR     BH,BH
        MOV     AH,2
        CALL    VideoInt
        MOV     CX,CursorLines
        LES     DI,Self
        TEST    ES:[DI].TView.State,sfCursorIns
        JNE     @@7
        CMP     Laptop,0
        JE      @@6
        TEST    ES:[DI].TView.State,sfAlwaysIns
        JE      @@6
@@7:    MOV     CH,0
        OR      CL,CL
        JNE     @@6
        MOV     CL,7
@@6:    MOV     AH,1
        CALL    VideoInt
@@10:
end;

procedure TView.Select;
begin
  if Options and ofTopSelect <> 0 then MakeFirst else
    if Owner <> nil then Owner^.SetCurrent(@Self, NormalSelect);
end;

procedure TView.SetBounds(var Bounds: TRect); assembler;
asm
        PUSH    DS
        LES     DI,Self
        LDS     SI,Bounds
        MOV     AX,[SI].TRect.A.X
        MOV     ES:[DI].Origin.X,AX
        MOV     AX,[SI].TRect.A.Y
        MOV     ES:[DI].Origin.Y,AX
        MOV     AX,[SI].TRect.B.X
        SUB     AX,[SI].TRect.A.X
        MOV     ES:[DI].Size.X,AX
        MOV     AX,[SI].TRect.B.Y
        SUB     AX,[SI].TRect.A.Y
        MOV     ES:[DI].Size.Y,AX
        POP     DS
end;

procedure TView.SetCmdState(Commands: TCommandSet; Enable: Boolean);
begin
  if Enable then EnableCommands(Commands)
    else DisableCommands(Commands);
end;

procedure TView.SetCommands(Commands: TCommandSet);
begin
  CommandSetChanged := CommandSetChanged or (CurCommandSet <> Commands);
  CurCommandSet := Commands;
end;

procedure TView.SetCursor(X, Y: Integer);
begin
  Cursor.X := X;
  Cursor.Y := Y;
  DrawCursor;
end;

procedure TView.SetData(var Rec);
begin
end;

procedure TView.SetState(AState: Word; Enable: Boolean);
var
  Command: Word;
begin
  if Enable then
    State := State or AState else
    State := State and not AState;
  if Owner <> nil then
    case AState of
      sfVisible:
        begin
          if Owner^.State and sfExposed <> 0 then
            SetState(sfExposed, Enable);
          if Enable then DrawShow(nil) else DrawHide(nil);
          if Options and ofSelectable <> 0 then Owner^.ResetCurrent;
        end;
      sfCursorVis, sfCursorIns:
        DrawCursor;
      sfShadow:
        DrawUnderView(True, nil);
      sfFocused: ResetCursor;
    end;
end;

procedure TView.Show;
begin
  if State and sfVisible = 0 then SetState(sfVisible, True);
end;

procedure TView.ShowCursor;
begin
  SetState(sfCursorVis, True);
end;

function TView.TopView: PView;
var
  P: PView;
begin
  if TheTopView = nil then
  begin
    P := @Self;
    while (P <> nil) and (P^.State and sfModal = 0) do P := P^.Owner;
    TopView := P;
  end
  else TopView := TheTopView;
end;

function TView.Valid(Command: Word): Boolean;
begin
  Valid := True;
end;

procedure TView.WriteBuf(X, Y, W, H: Integer; var Buf); assembler;
var
  Target: Pointer; {Variables used by WriteView}
  Buffer: Pointer;
  Offset: Word;
asm
        CMP     H,0
        JLE     @@2
@@1:    MOV     AX,Y
        MOV     BX,X
        MOV     CX,W
        LES     DI,Buf
        CALL    WriteView
        MOV     AX,W
        SHL     AX,1
        ADD     WORD PTR Buf[0],AX
        INC     Y
        DEC     H
        JNE     @@1
@@2:
end;

procedure TView.WriteChar(X, Y: Integer; C: Char; Color: Byte; Count: Integer); assembler;
var
  Target: Pointer; {Variables used by WriteView}
  Buffer: Pointer;
  Offset: Word;
asm
        MOV     AL,Color
        CALL    MapColor
        MOV     AH,AL
        MOV     AL,C
        MOV     CX,Count
        OR      CX,CX
        JLE     @@2
        CMP     CX,256
        JLE     @@1
        MOV     CX,256
@@1:    MOV     DI,CX
        SHL     DI,1
        SUB     SP,DI
        MOV     DI,SP
        PUSH    SS
        POP     ES
        MOV     DX,CX
        CLD
        REP     STOSW
        MOV     CX,DX
        MOV     DI,SP
        MOV     AX,Y
        MOV     BX,X
        CALL    WriteView
@@2:
end;

procedure TView.WriteLine(X, Y, W, H: Integer; var Buf); assembler;
var
  Target: Pointer; {Variables used by WriteView}
  Buffer: Pointer;
  Offset: Word;
asm
        CMP     H,0
        JLE     @@2
@@1:    MOV     AX,Y
        MOV     BX,X
        MOV     CX,W
        LES     DI,Buf
        CALL    WriteView
        INC     Y
        DEC     H
        JNE     @@1
@@2:
end;

procedure TView.WriteStr(X, Y: Integer; const Str: string; Color: Byte); assembler;
var
  Target: Pointer; {Variables used by WriteView}
  Buffer: Pointer;
  Offset: Word;
asm
        MOV     AL,Color
        CALL    MapColor
        MOV     AH,AL
        MOV     BX,DS
        LDS     SI,Str
        CLD
        LODSB
        MOV     CL,AL
        XOR     CH,CH
        JCXZ    @@3
        MOV     DI,CX
        SHL     DI,1
        SUB     SP,DI
        MOV     DI,SP
        PUSH    SS
        POP     ES
        MOV     DX,CX
@@1:    LODSB
        STOSW
        LOOP    @@1
        MOV     DS,BX
        MOV     CX,DX
        MOV     DI,SP
        MOV     AX,Y
        MOV     BX,X
        CALL    WriteView
        JMP     @@2
@@3:    MOV     DS,BX
@@2:
end;

{ TFrame }

constructor TFrame.Init(var Bounds: TRect; AFrameX, AFrameY: Integer);
begin
  TView.Init(Bounds);
  FrameX := AFrameX;
  FrameY := AFrameY;
  GrowMode := gfGrowHiX + gfGrowHiY;
  EventMask := EventMask or evBroadcast;
end;

procedure TFrame.FrameLine(var FrameBuf; Y, N: Integer; Color: Byte); assembler;
var
  FrameMask: array[0..MaxViewWidth-1] of Byte;
  AFrameX: Word;
asm
        LES     BX,Self
        MOV     DX,ES:[BX].TFrame.FrameX
        MOV     AFrameX,DX
        MOV     DX,ES:[BX].TFrame.Size.X
        SUB     DX,AFrameX
        SUB     DX,AFrameX
        MOV     CX,DX
        DEC     CX
        DEC     CX
        MOV     AX,N
        LEA     DI,FrameMask
        PUSH    SS
        POP     ES
        CLD
        STOSB
        INC     AL
        REP     STOSB
        ADD     AL,2
        STOSB
        LES     BX,Self
        LES     BX,ES:[BX].TFrame.Owner
        LES     BX,ES:[BX].TGroup.Last
        DEC     DX
@1:     LES     BX,ES:[BX].TView.Next
        CMP     BX,WORD PTR Self[0]
        JNE     @2
        MOV     AX,ES
        CMP     AX,WORD PTR Self[2]
        JE      @10
@2:     TEST    ES:[BX].TView.Options,ofFramed
        JE      @1
        TEST    ES:[BX].TView.State,sfVisible
        JE      @1
        MOV     AX,Y
        SUB     AX,ES:[BX].TView.Origin.Y
        JL      @3
        CMP     AX,ES:[BX].TView.Size.Y
        JG      @1
        MOV     AX,0005H
        JL      @4
        MOV     AX,0A03H
        JMP     @4
@3:     INC     AX
        JNE     @1
        MOV     AX,0A06H
@4:     MOV     SI,ES:[BX].TView.Origin.X
        MOV     DI,ES:[BX].TView.Size.X
        ADD     DI,SI
        CMP     SI,1
        JG      @5
        MOV     SI,1
@5:     CMP     DI,DX
        JL      @6
        MOV     DI,DX
@6:     CMP     SI,DI
        JGE     @1
        OR      BYTE PTR FrameMask[SI-1],AL
        XOR     AL,AH
        OR      BYTE PTR FrameMask[DI],AL
        OR      AH,AH
        JE      @1
        MOV     CX,DI
        SUB     CX,SI
@8:     OR      BYTE PTR FrameMask[SI],AH
        INC     SI
        LOOP    @8
        JMP     @1
@10:    INC     DX
        MOV     AH,Color
        MOV     BX,OFFSET DoubleFrameChars[1]
        MOV     CX,DX
        LEA     SI,FrameMask
        LES     DI,FrameBuf
        ADD     DI,AFrameX
        ADD     DI,AFrameX
@11:    SEGSS   LODSB
        XLAT
        STOSW
        LOOP    @11
end;

procedure TFrame.Draw;
var
  Color: Word;
  I, L, Width: Integer;
  B: TDrawBuffer;
  Title: TTitleStr;
  Min, Max: TPoint;
begin
  Color := GetColor(1);
  Width := Size.X;
  L := Width - 10;
  MoveChar(B, ' ', Color, Size.X);
  WriteLine(0, 0, Size.X, FrameY, B);
  FrameLine(B, 0, 0, Byte(Color));
  if Owner <> nil then Title := PWindow(Owner)^.GetTitle else Title := '';
  if Title <> '' then
  begin
    L := Length(Title);
    if L > Width - 10 then L := Width - 10;
    if L < 0 then L := 0;
    I := (Width - L + 1) shr 1;
    MoveChar(B[I - 1], ' ', Color, 1);
    MoveBuf(B[I], Title[1], Color, L);
    MoveChar(B[I + L], ' ', Color, 1);
  end;
  WriteLine(0, FrameY, Size.X, 1, B);
  for I := FrameY + 1 to Size.Y - FrameY - 2 do
  begin
    FrameLine(B, I, 4, Byte(Color));
    WriteLine(0, I, Size.X, 1, B);
  end;
  FrameLine(B, Size.Y - 2, 12, Byte(Color));
  WriteLine(0, Size.Y - FrameY - 1, Size.X, 1, B);
  MoveChar(B, ' ', Color, Size.X);
  WriteLine(0, Size.Y - FrameY, Size.X, FrameY, B);
end;

function TFrame.GetPalette: PPalette;
const
  P: string[Length(CFrame)] = CFrame;
begin
  GetPalette := @P;
end;

procedure TFrame.SetState(AState: Word; Enable: Boolean);
begin
  TView.SetState(AState, Enable);
  if AState and (sfActive + sfDragging) <> 0 then DrawView;
end;

{ TGroup }

constructor TGroup.Init(var Bounds: TRect);
begin
  TView.Init(Bounds);
  Options := Options or (ofSelectable + ofBuffered);
  GetExtent(Clip);
  EventMask := $FFFF;
end;

destructor TGroup.Done;
var
  P, T: PView;
begin
  Hide;
  P := Last;
  if P <> nil then
  begin
    repeat
      P^.Hide;
      P := P^.Prev;
    until P = Last;
    repeat
      T := P^.Prev;
      Dispose(P, Done);
      P := T;
    until Last = nil;
  end;
  FreeBuffer;
  TView.Done;
end;

function TGroup.At(Index: Integer): PView; assembler;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     CX,Index
@@1:    LES     DI,ES:[DI].TView.Next
        LOOP    @@1
        MOV     AX,DI
        MOV     DX,ES
end;

procedure TGroup.ChangeBounds(var Bounds: TRect);
var
  D: TPoint;

procedure DoCalcChange(P: PView); far;
var
  R: TRect;
begin
  P^.CalcBounds(R, D);
  P^.ChangeBounds(R);
end;

begin
  D.X := Bounds.B.X - Bounds.A.X - Size.X;
  D.Y := Bounds.B.Y - Bounds.A.Y - Size.Y;
  if Longint(D) = 0 then
  begin
    SetBounds(Bounds);
    DrawView;
  end else
  begin
    FreeBuffer;
    SetBounds(Bounds);
    GetExtent(Clip);
    GetBuffer;
    Lock;
    ForEach(@DoCalcChange);
    Unlock;
  end;
end;

procedure TGroup.Delete(P: PView);
var
  SaveState: Word;
begin
  SaveState := P^.State;
  P^.Hide;
  RemoveView(P);
  P^.Owner := nil;
  P^.Next := nil;
  if SaveState and sfVisible <> 0 then P^.Show;
end;

procedure TGroup.Draw;
var
  R: TRect;
begin
  if RedrawAll then FreeBuffer;
  if Buffer = nil then
  begin
    GetBuffer;
    if Buffer <> nil then
    begin
      Inc(LockFlag);
      Redraw;
      Dec(LockFlag);
    end;
  end;
  if Buffer <> nil then WriteBuf(0, 0, Size.X, Size.Y, Buffer^) else
  begin
    GetClipRect(Clip);
    Redraw;
    GetExtent(Clip);
  end;
end;

procedure TGroup.DrawSubViews(P, Bottom: PView);
begin
  if P <> nil then
    while P <> Bottom do
    begin
      P^.DrawView;
      P := P^.NextView;
    end;
end;

procedure TGroup.EndModal(Command: Word);
begin
  if State and sfModal <> 0 then EndState := Command
  else TView.EndModal(Command);
end;

procedure TGroup.EventError(var Event: TEvent);
begin
  if Owner <> nil then Owner^.EventError(Event);
end;

function TGroup.Execute: Word;
var
  E: TEvent;
begin
  repeat
    EndState := 0;
    repeat
      GetEvent(E);
      HandleEvent(E);
      if E.What <> evNothing then EventError(E);
    until EndState <> 0;
  until Valid(EndState);
  Execute := EndState;
end;

function TGroup.ExecView(P: PView; MakeCur, DelView: Boolean): Word;
var
  SaveOptions: Word;
  SaveOwner: PGroup;
  SaveTopView: PView;
  SaveCommands: TCommandSet;
begin
  if P <> nil then
  begin
    SaveOptions := P^.Options;
    SaveOwner := P^.Owner;
    SaveTopView := TheTopView;
    GetCommands(SaveCommands);
    TheTopView := P;
    P^.Options := P^.Options or ofSelectable;
    P^.SetState(sfModal, True);
    if MakeCur then SetCurrent(P, EnterSelect);
    if SaveOwner = nil then Insert(P);
    ExecView := P^.Execute;
    if DelView and (SaveOwner = nil) then Delete(P);
    if TheTopView <> P then P^.SetState(sfModal, False);
    P^.Options := SaveOptions;
    TheTopView := SaveTopView;
    SetCommands(SaveCommands);
    ResetCurrent;
    Redraw;
  end else ExecView := cmCancel;
end;

function TGroup.First: PView;
begin
  if Last = nil then First := nil else First := Last^.Next;
end;

function TGroup.FirstMatch(AState: Word; AOptions: Word): PView;

function Matches(P: PView): Boolean; far;
begin
  Matches := (P^.State and AState = AState) and
    (P^.Options and AOptions = AOptions);
end;

begin
  FirstMatch := FirstThat(@Matches);
end;

function TGroup.FirstThat(P: Pointer): PView; assembler;
var
  ALast: Pointer;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     AX,ES
        OR      AX,DI
        JE      @@3
        MOV     WORD PTR ALast[2],ES
        MOV     WORD PTR ALast[0],DI
@@1:    LES     DI,ES:[DI].TView.Next
        PUSH    ES
        PUSH    DI
        PUSH    ES
        PUSH    DI
        PUSH    WORD PTR [BP]
        CALL    P
        POP     DI
        POP     ES
        OR      AL,AL
        JNE     @@2
        CMP     DI,WORD PTR ALast[0]
        JNE     @@1
        MOV     AX,ES
        CMP     AX,WORD PTR ALast[2]
        JNE     @@1
        XOR     DI,DI
        MOV     ES,DI
@@2:    MOV     SP,BP
@@3:    MOV     AX,DI
        MOV     DX,ES
end;

function TGroup.FindNext(Forwards: Boolean): PView;
var
  P: PView;
begin
  FindNext := nil;
  if Current <> nil then
  begin
    P := Current;
    repeat
      if Forwards then P := P^.Next else P := P^.Prev;
    until ((P^.State and (sfVisible + sfDisabled) = sfVisible) and
      (P^.Options and ofSelectable <> 0)) or (P = Current);
    if P <> Current then FindNext := P;
  end;
end;

function TGroup.FocusNext(Forwards: Boolean): Boolean;
var
  P: PView;
begin
  P := FindNext(Forwards);
  FocusNext := True;
  if P <> nil then FocusNext := P^.Focus;
end;

procedure TGroup.ForEach(P: Pointer); assembler;
var
  ALast: Pointer;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     AX,ES
        OR      AX,DI
        JE      @@4
        MOV     WORD PTR ALast[2],ES
        MOV     WORD PTR ALast[0],DI
        LES     DI,ES:[DI].TView.Next
@@1:    CMP     DI,WORD PTR ALast[0]
        JNE     @@2
        MOV     AX,ES
        CMP     AX,WORD PTR ALast[2]
        JE      @@3
@@2:    PUSH    WORD PTR ES:[DI].TView.Next[2]
        PUSH    WORD PTR ES:[DI].TView.Next[0]
        PUSH    ES
        PUSH    DI
        PUSH    WORD PTR [BP]
        CALL    P
        POP     DI
        POP     ES
        JMP     @@1
@@3:    PUSH    WORD PTR [BP]
        CALL    P
@@4:
end;

procedure TGroup.FreeBuffer;
begin
  if (Options and ofBuffered <> 0) and (Buffer <> nil) then
    DisposeCache(Pointer(Buffer));
end;

{ Allocate a group buffer if the group is exposed, buffered, and
  its area is less than 32768 bytes }

procedure TGroup.GetBuffer; assembler;
asm
	LES	DI,Self
        TEST	ES:[DI].State,sfExposed
        JZ	@@1
        TEST	ES:[DI].Options,ofBuffered
        JZ	@@1
        MOV	AX,ES:[DI].Buffer.Word[0]
        OR	AX,ES:[DI].Buffer.Word[2]
        JNZ	@@1
        MOV	AX,ES:[DI].TView.Size.X
	MUL	ES:[DI].TView.Size.Y
        JO	@@1
        SHL	AX,1
        JC	@@1
        JS	@@1
        LEA	DI,[DI].TView.Buffer
        PUSH	ES
        PUSH	DI
        PUSH	AX
        CALL	NewCache
@@1:
end;

function TGroup.GetHelpCtx: Word;
var
  H: Word;
begin
  H:= hcNoContext;
  if Current <> nil then H := Current^.GetHelpCtx;
  if H = hcNoContext then H := TView.GetHelpCtx;
  GetHelpCtx := H;
end;

procedure TGroup.HandleEvent(var Event: TEvent);

procedure DoHandleEvent(P: PView); far;
begin
  if (P = nil) or ((P^.State and sfDisabled <> 0)
    and (Event.What and (PositionalEvents or FocusedEvents) <> 0)) then Exit;
  case Phase of
    phPreProcess: if P^.Options and ofPreProcess = 0 then Exit;
    phPostProcess: if P^.Options and ofPostProcess = 0 then Exit;
  end;
  if Event.What and P^.EventMask <> 0 then P^.HandleEvent(Event);
end;

function ContainsMouse(P: PView): Boolean; far;
begin
  ContainsMouse := (P^.State and sfVisible <> 0) and
    P^.MouseInView(Event.Where);
end;

begin
  TView.HandleEvent(Event);
  if Event.What and FocusedEvents <> 0 then
  begin
    Phase := phPreProcess;
    ForEach(@DoHandleEvent);
    Phase := phFocused;
    DoHandleEvent(Current);
    Phase := phPostProcess;
    ForEach(@DoHandleEvent);
  end else
  begin
    Phase := phFocused;
    if (Event.What and PositionalEvents <> 0) then
      DoHandleEvent(FirstThat(@ContainsMouse)) else
      ForEach(@DoHandleEvent);
  end;
end;

function TGroup.IndexOf(P: PView): Integer; assembler;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     AX,ES
        OR      AX,DI
        JE      @@3
        MOV     CX,DI
        MOV     BX,ES
        XOR     AX,AX
@@1:    INC     AX
        LES     DI,ES:[DI].TView.Next
        MOV     DX,ES
        CMP     DI,P.Word[0]
        JNE     @@2
        CMP     DX,P.Word[2]
        JE      @@3
@@2:    CMP     DI,CX
        JNE     @@1
        CMP     DX,BX
        JNE     @@1
        XOR     AX,AX
@@3:
end;

procedure TGroup.Insert(P: PView);
begin
  InsertBefore(P, First);
end;

procedure TGroup.InsertBefore(P, Target: PView);
var
  SaveState: Word;
begin
  if (P <> nil) and (P^.Owner = nil) and
    ((Target = nil) or (Target^.Owner = @Self)) then
  begin
    SaveState := P^.State;
    P^.Hide;
    InsertView(P, Target);
    if SaveState and sfVisible <> 0 then P^.Show;
    if State and sfActive <> 0 then
      P^.SetState(sfActive, True);
  end;
end;

procedure TGroup.InsertView(P, Target: PView);
begin
  P^.Owner := @Self;
  if Target <> nil then
  begin
    Target := Target^.Prev;
    P^.Next := Target^.Next;
    Target^.Next := P;
  end else
  begin
    if Last = nil then P^.Next := P else
    begin
      P^.Next := Last^.Next;
      Last^.Next := P;
    end;
    Last := P;
  end;
end;

procedure TGroup.Lock;
begin
  if (Buffer <> nil) or (LockFlag <> 0) then Inc(LockFlag);
end;

procedure TGroup.Redraw;
begin
  if RedrawAll then FreeBuffer;
  DrawSubViews(First, nil);
end;

procedure TGroup.RemoveView(P: PView); assembler;
asm
        PUSH    DS
        LDS     SI,Self
        LES     DI,P
        LDS     SI,DS:[SI].TGroup.Last
        PUSH    BP
        MOV     AX,DS
        OR      AX,SI
        JE      @@7
        MOV     AX,SI
        MOV     DX,DS
        MOV     BP,ES
@@1:    MOV     BX,WORD PTR DS:[SI].TView.Next[0]
        MOV     CX,WORD PTR DS:[SI].TView.Next[2]
        CMP     CX,BP
        JE      @@5
@@2:    CMP     CX,DX
        JE      @@4
@@3:    MOV     SI,BX
        MOV     DS,CX
        JMP     @@1
@@4:    CMP     BX,AX
        JNE     @@3
        JMP     @@7
@@5:    CMP     BX,DI
        JNE     @@2
        MOV     BX,WORD PTR ES:[DI].TView.Next[0]
        MOV     CX,WORD PTR ES:[DI].TView.Next[2]
        MOV     DS:WORD PTR [SI].TView.Next[0],BX
        MOV     DS:WORD PTR [SI].TView.Next[2],CX
        CMP     DX,BP
        JNE     @@7
        CMP     AX,DI
        JNE     @@7
        CMP     CX,BP
        JNE     @@6
        CMP     BX,DI
        JNE     @@6
        XOR     SI,SI
        MOV     DS,SI
@@6:    POP     BP
        PUSH    BP
        LES     DI,Self
        MOV     WORD PTR ES:[DI].TView.Last[0],SI
        MOV     WORD PTR ES:[DI].TView.Last[2],DS
@@7:    POP     BP
        POP     DS
end;

procedure TGroup.ResetCurrent;
begin
  SetCurrent(FirstMatch(sfVisible, ofSelectable), NormalSelect);
end;

procedure TGroup.ResetCursor;
begin
  if Current <> nil then Current^.ResetCursor;
end;

procedure TGroup.SelectNext(Forwards: Boolean);
var
  P: PView;
begin
  P := FindNext(Forwards);
  if P <> nil then P^.Select;
end;

procedure TGroup.SetCurrent(P: PView; Mode: SelectMode);

procedure SelectView(P: PView; Enable: Boolean);
begin
  if P <> nil then P^.SetState(sfSelected, Enable);
end;

procedure FocusView(P: PView; Enable: Boolean);
begin
  if (State and sfFocused <> 0) and (P <> nil) then
    P^.SetState(sfFocused, Enable);
end;

begin
  if Current <> P then
  begin
    Lock;
    FocusView(Current, False);
    if Mode <> EnterSelect then SelectView(Current, False);
    if Mode <> LeaveSelect then SelectView(P, True);
    FocusView(P, True);
    Current := P;
    Unlock;
  end;
end;

procedure TGroup.SetState(AState: Word; Enable: Boolean);

procedure DoSetState(P: PView); far;
begin
  P^.SetState(AState, Enable);
end;

procedure DoExpose(P: PView); far;
begin
  if P^.State and sfVisible <> 0 then P^.SetState(sfExposed, Enable);
end;

begin
  TView.SetState(AState, Enable);
  case AState of
    sfActive, sfDragging:
      begin
        Lock;
        ForEach(@DoSetState);
        Unlock;
      end;
    sfFocused:
      if Current <> nil then Current^.SetState(sfFocused, Enable);
    sfExposed:
      begin
        ForEach(@DoExpose);
        if not Enable then FreeBuffer;
      end;
  end;
end;

procedure TGroup.Unlock;
begin
  if LockFlag <> 0 then
  begin
    Dec(LockFlag);
    if LockFlag = 0 then DrawView;
  end;
end;

function TGroup.Valid(Command: Word): Boolean;

function IsInvalid(P: PView): Boolean; far;
begin
  IsInvalid := not P^.Valid(Command);
end;

begin
  Valid := FirstThat(@IsInvalid) = nil;
end;

{ TWindow }

constructor TWindow.Init(var Bounds: TRect; ATitle: TTitleStr; AFrameX, AFrameY: Integer);
begin
  TGroup.Init(Bounds);
  State := State or sfShadow;
  Options := Options or ofSelectable;
  GrowMode := 0;
  Title := NewStr(ATitle);
  Palette := wpDialog;
  FrameX := AFrameX;
  FrameY := AFrameY;
  if FrameX + FrameY >= 0 then InitFrame else Frame := nil;
  if Frame <> nil then Insert(Frame);
end;

destructor TWindow.Done;
begin
  TGroup.Done;
  DisposeStr(Title);
end;

function TWindow.GetPalette: PPalette;
const
  P: array[wpDialog..wpViewEdit] of string[Length(CViewEdit)] =
    (CDialog, CConfig, CHistory, CHelp, CError, CViewEdit);
begin
  GetPalette := @P[Palette];
end;

function TWindow.GetTitle: TTitleStr;
begin
  if Title <> nil then GetTitle := Title^ else GetTitle := '';
end;

procedure TWindow.InitFrame;
var
  R: TRect;
begin
  GetExtent(R);
  Frame := New(PFrame, Init(R, FrameX, FrameY));
end;

procedure TWindow.SetState(AState: Word; Enable: Boolean);
var
  WindowCommands: TCommandSet;
begin
  TGroup.SetState(AState, Enable);
  if AState = sfSelected then
    SetState(sfActive, Enable);
  if (AState = sfSelected) or ((AState = sfExposed) and
    (State and sfSelected <> 0)) then
  begin
    WindowCommands := [cmNext, cmPrev];
    if Enable then EnableCommands(WindowCommands)
    else DisableCommands(WindowCommands);
  end;
end;

{ Message dispatch function }

procedure Message(Receiver: PView; What, Command: Word);
var
  Event: TEvent;
begin
  if Receiver <> nil then
  begin
    Event.What := What;
    Event.Command := Command;
    Receiver^.HandleEvent(Event);
  end;
end;

end.
