
-- ListObj 0.12 beta
-- List objects in a MS-SQL 7.0/2000 database
-- by Joe Forster/STA (2004-07-28)

set nocount on
GO

-- Drop temporary objects, in case a previous run resulted in an error and
-- left the objects existing
if exists (select 1 from sysobjects where name = 'tbl_temp_Dependency' and xtype = 'U')
  drop table tbl_temp_Dependency
if exists (select 1 from sysobjects where name = 'tbl_temp_ObjectTypes' and xtype = 'U')
  drop table tbl_temp_ObjectTypes
if exists (select 1 from sysobjects where name = 'tbl_temp_Permissions' and xtype = 'U')
  drop table tbl_temp_Permissions
if exists (select 1 from sysobjects where name = 'sp_temp_decrypt_7' and xtype = 'P')
  drop procedure sp_temp_decrypt_7
if exists (select 1 from sysobjects where name = 'sp_temp_Print' and xtype = 'P')
  drop procedure sp_temp_Print
if exists (select 1 from sysobjects where name = 'sp_temp_FixInt' and xtype = 'P')
  drop procedure sp_temp_FixInt
if exists (select 1 from sysobjects where name = 'sp_temp_FixString' and xtype = 'P')
  drop procedure sp_temp_FixString
if exists (select 1 from sysobjects where name = 'sp_temp_FixDate' and xtype = 'P')
  drop procedure sp_temp_FixDate
if exists (select 1 from sysobjects where name = 'sp_temp_ObjFullName' and xtype = 'P')
  drop procedure sp_temp_ObjFullName
if exists (select 1 from sysobjects where name = 'sp_temp_ListTable' and xtype = 'P')
  drop procedure sp_temp_ListTable
GO

-- Valid object types
create table tbl_temp_ObjectTypes (
  xtype varchar(2) not null primary key,
  usrtype varchar(2) not null,
  typename varchar(100) not null,
  prop varchar(100) not null,
  ord int not null)
insert into tbl_temp_ObjectTypes values ('U', 'T', 'TABLE', 'IsUserTable', 1)
insert into tbl_temp_ObjectTypes values ('V', 'V', 'VIEW', 'IsView', 2)
insert into tbl_temp_ObjectTypes values ('P', 'P', 'PROCEDURE', 'IsProcedure', 3)
GO

-- Valid permission bits
create table tbl_temp_Permissions (
  val int not null primary key,
  name varchar(100) not null,
  ord int not null)
insert into tbl_temp_Permissions values (1, 'SELECT', 1)
insert into tbl_temp_Permissions values (2, 'UPDATE', 3)
insert into tbl_temp_Permissions values (4, 'DRI', 6)
insert into tbl_temp_Permissions values (8, 'INSERT', 2)
insert into tbl_temp_Permissions values (16, 'DELETE', 4)
insert into tbl_temp_Permissions values (32, 'EXECUTE', 5)
GO

-- Print text
-- Input : @string: text to print
create procedure sp_temp_Print
  @string varchar(8000) = null
as
-- Comment first PRINT and uncomment second PRINT to have all (non-wrapped)
--   lines start with Char(1) and all lines end with Char(1)
/* * UnWrAp1 */
  print coalesce(@string, rtrim(char(13) + char(10)))
/* UnWrAp1 * */
/* UnWrAp1
  print char(1) + coalesce(@string, '') + char(1)
UnWrAp1 */
GO

-- Convert data of string (char/varchar/text) type to data format
-- Input : @column_name: name of column
--         @value: value of column
--         @last_column: when not 0, this is the last column in the table
--         @format: listing format
-- Output: @result: variable to append text to
create procedure sp_temp_FixString
  @column_name varchar(8000) = null,
  @value varchar(8000) = null,
  @last_column int = null,
  @result varchar(8000) output,
  @format int = 0
as
  if @column_name is null
  begin
    set @result = @result + @value + rtrim(case @format
        when 1 then ''
        when 2 then char(9)
        when 3 then ','
      end)
  end
  else
  begin
    if @format = 2
    begin
      set @value = rtrim(replace(replace(replace(@value, char(13), ' '), char(10), ' '), char(9), ' '))
      if @value = ''
        set @value = ' '
    end
    set @result = @result + case @format
        when 1 then coalesce('''' + rtrim(replace(@value, '''', '''''')) + '''', 'NULL')
        when 2 then coalesce(@value, '')
        when 3 then coalesce('"' + rtrim(replace(@value, '"', '""')) + '"', '')
      end
    if @last_column = 0 set @result = @result + rtrim(case @format
        when 1 then ','
        when 2 then char(9)
        when 3 then ','
      end)
  end
GO

-- Convert data of integer (tinyint/smallint/int) type to data format
-- Input : @column_name: name of column
--         @value: value of column
--         @last_column: when not 0, this is the last column in the table
--         @format: listing format
-- Output: @result: variable to append text to
create procedure sp_temp_FixInt
  @column_name varchar(8000) = null,
  @value int = null,
  @last_column int = null,
  @result varchar(8000) output,
  @format int = 0
as
  set @result = @result + case @format
      when 1 then coalesce(convert(varchar, @value), 'NULL')
      when 2 then coalesce(convert(varchar, @value), '')
      when 3 then coalesce(convert(varchar, @value), '')
    end
  if @last_column = 0 set @result = @result + rtrim(case @format
      when 1 then ', '
      when 2 then char(9)
      when 3 then ','
    end)
GO

-- Convert data of datetime type to date format
-- Input : @column_name: name of column
--         @value: value of column
--         @last_column: when not 0, this is the last column in the table
--         @format: listing format
-- Output: @result: variable to append text to
create procedure sp_temp_FixDate
  @column_name varchar(8000) = null,
  @value datetime = null,
  @last_column int = null,
  @result varchar(8000) output,
  @format int = 0
as
  set @result = @result + case @format
      when 1 then coalesce('''' + convert(varchar, @value, 120) + '''', 'NULL')
      when 2 then coalesce(convert(varchar, @value, 120), '')
      when 3 then coalesce(convert(varchar, @value, 120), '')
    end
  if @last_column = 0 set @result = @result + rtrim(case @format
      when 1 then ', '
      when 2 then char(9)
      when 3 then ','
    end)
GO

-- Create full qualified object name
-- Input : @objname: object name
--         @ownname: owner name
--         @format: listing format
--         @identquote*: identifier quotation marks
-- Output: @ownfullname: fully qualified owner name
create procedure sp_temp_ObjFullName
  @objname sysname = null,
  @ownname sysname = null,
  @objfullname sysname output,
  @format int = 0,
  @identquoteopen varchar(8000) = null,
  @identquoteclose varchar(8000) = null
as
  set @objfullname = case @format % 100
      when 1 then @ownname + '.'
      when 11 then @identquoteopen + @ownname + @identquoteclose + '.'
      else ''
    end + case
      when (@format % 100) / 10 = 0 then @objname
      else @identquoteopen + @objname + @identquoteclose
    end
GO

-- List data in a table
-- Input : @objname: name of table
--         @ownname: name of table owner
--         @batchlength: when not NULL, this many INSERT statements are
--           created in one batch
--         @batchsep: the string separating batches
--         @idcol: when not 0, there is an identity column in the table
--         @format: listing format
--         @identformat: identifier format
--         @identquote*: identifier quotation marks
--         @cond: filtering condition; ignored, if NULL or empty
--         @sort: sorting condition; ignored, if NULL or empty
create procedure sp_temp_ListTable
  @objname varchar(8000) = null,
  @ownname varchar(8000) = null,
  @batchlength int = null,
  @batchsep varchar(8000) = null,
  @idcol int = 0,
  @format int = 0,
  @identformat int = 0,
  @identquoteopen varchar(8000) = null,
  @identquoteclose varchar(8000) = null,
  @cond varchar(8000) = null,
  @sort varchar(8000) = null
as
  declare @@cmd varchar(8000)
  declare @@fetch varchar(8000)
  declare @@print varchar(8000)
  declare @@colname varchar(8000)
  declare @@collist varchar(8000)
  declare @@colnum int
  declare @@desc cursor
  declare @@column_name sysname
  declare @@ordinal_position int
  declare @@column_characteristics_flags int
  declare @@column_size int
  declare @@data_type_sql smallint
  declare @@column_precision tinyint
  declare @@column_scale tinyint
  declare @@order_position int
  declare @@order_direction varchar(1)
  declare @@hidden_column smallint
  declare @@columnid int
  declare @@objectid int
  declare @@dbid int
  declare @@last_column int
  declare @@dbname sysname
  declare @@dummyint int
  declare @@objfullname sysname
  set ansi_warnings off
  exec sp_temp_ObjFullName @objname, @ownname, @@objfullname output, @identformat, @identquoteopen, @identquoteclose
  exec('declare datcursor cursor for select 1 from ' + @@objfullname)
  open datcursor
  fetch next from datcursor into @@dummyint
  close datcursor
  deallocate datcursor
  if @@fetch_status <> 0
    return 2
  set @@cmd = 'declare @@line varchar(8000)' + char(13) + char(10) +
    'declare @@insertbatchlength int' + char(13) + char(10) +
    'declare @@insertcount int' + char(13) + char(10)
  set @@fetch = 'fetch next from selcursor into '
  set @@collist = ''
  if @format % 10 in (2, 3) and (@format % 1000) / 100 <> 0
    set @@collist = @objname
  set @@colnum = 0
  exec('declare datcursor cursor for select * from ' + @@objfullname)
  open datcursor
  exec sp_describe_cursor_columns @@desc output, 'global', 'datcursor'
  close datcursor
  deallocate datcursor
  fetch next from @@desc into @@column_name, @@ordinal_position,
    @@column_characteristics_flags, @@column_size, @@data_type_sql,
    @@column_precision, @@column_scale, @@order_position,
    @@order_direction, @@hidden_column, @@columnid, @@objectid,
    @@dbid, @@dbname
  if @format % 10 = 1
  begin
    exec sp_temp_Print
    set @@print = 'set nocount on'
    exec sp_temp_Print @@print
    if coalesce(@idcol, 0) <> 0
    begin
      set @@print = 'SET IDENTITY_INSERT ' + @@objfullname + ' ON'
      exec sp_temp_Print @@print
    end
  end
  set @@print = ''
  while @@fetch_status <> -1
  begin
    set @@colname = '@@col' + convert(varchar, @@colnum)
    set @@cmd = @@cmd + 'declare ' + @@colname + ' '
    set @@print = @@print + '  exec '
    if @@collist <> ''
      set @@collist = @@collist + rtrim(case @format % 10
          when 1 then ', '
          when 2 then char(9)
          when 3 then ','
      end)
    set @@collist = @@collist + @@column_name
    if @@data_type_sql in (48, 52, 56)
    begin
      set @@cmd = @@cmd + 'int' + char(13) + char(10)
      set @@print = @@print + 'sp_temp_FixInt'
    end
    else
    begin
      if @@data_type_sql in (35, 167, 175)
      begin
        if @@data_type_sql = 35
          set @@column_size = 8000
        set @@cmd = @@cmd + 'varchar(' + convert(varchar, @@column_size) + ')' + char(13) + char(10)
        set @@print = @@print + 'sp_temp_FixString'
      end
      else
      begin
        if @@data_type_sql in (61)
        begin
          set @@cmd = @@cmd + 'datetime' + char(13) + char(10)
          set @@print = @@print + 'sp_temp_FixDate'
        end
        else
        begin
          raiserror ('Unknown column type: "%s" = %d.', 16, 1, @@column_name, @@data_type_sql)
          close @@desc
          deallocate @@desc
          return 1
        end
      end
    end
    if @@colnum > 0 set @@fetch = @@fetch + ', '
    set @@fetch = @@fetch + @@colname
    set @@print = @@print + ' ''' + @@column_name + ''', ' + @@colname + ', #@?@#, @@line output, ' + convert(varchar, @format % 10) + char(13) + char(10)
    fetch next from @@desc into @@column_name, @@ordinal_position,
      @@column_characteristics_flags, @@column_size, @@data_type_sql,
      @@column_precision, @@column_scale, @@order_position,
      @@order_direction, @@hidden_column, @@columnid, @@objectid,
      @@dbid, @@dbname
    set @@last_column = 0
    if @@fetch_status = -1
      set @@last_column = 1
    set @@print = replace(@@print, '#@?@#', convert(varchar, @@last_column))
    set @@colnum = @@colnum + 1
  end
  if @format % 10 = 1
    set @@cmd = @@cmd + 'set @@insertbatchlength = ' + coalesce(convert(varchar, @batchlength), 'null') + char(13) + char(10) +
      'set @@insertcount = 0' + char(13) + char(10)
  if (@format % 100) / 10 <> 0
    set @@cmd = @@cmd + 'exec sp_temp_Print ''' + @@collist + '''' + char(13) + char(10)
  set @@cmd = @@cmd + 'declare selcursor cursor for select * from ' + @@objfullname
  if coalesce(@cond, '') <> '' set @@cmd = @@cmd + ' where ' + @cond
  if coalesce(@sort, '') <> '' set @@cmd = @@cmd + ' order by ' + @sort
  set @@cmd = @@cmd + char(13) + char(10) +
    'open selcursor' + char(13) + char(10) +
    @@fetch + char(13) + char(10) +
    'while @@fetch_status <> -1' + char(13) + char(10) +
    'begin' + char(13) + char(10) +
    '  set @@line = ''''' + char(13) + char(10)
  if @format % 10 = 1 or (@format % 1000) / 100 <> 0
    set @@cmd = @@cmd + '  exec sp_temp_FixString null, ''' + case when @format % 10 = 1 then @@objfullname else @objname end + ''', 0, @@line output, ' + convert(varchar, @format % 10) + char(13) + char(10)
  if @format % 10 = 1
    set @@cmd = @@cmd + '  set @@line = ''INSERT INTO '' + @@line + '' (' + @@collist + ') VALUES (''' + char(13) + char(10)
  set @@cmd = @@cmd + @@print
  if @format % 10 = 1
    set @@cmd = @@cmd + '  set @@line = @@line + '')''' + char(13) + char(10)
  set @@cmd = @@cmd + '  exec sp_temp_Print @@line' + char(13) + char(10)
  if @format % 10 = 1
  begin
  set @@cmd = @@cmd + '  set @@insertcount = @@insertcount + 1' + char(13) + char(10) +
    '  if @@insertcount >= @@insertbatchlength' + char(13) + char(10) +
    '  begin' + char(13) + char(10) +
    '    set @@line = ''' + @batchsep + '''' + char(13) + char(10) +
    '    exec sp_temp_Print @@line' + char(13) + char(10) +
    '    set @@insertcount = 0' + char(13) + char(10) +
    '  end' + char(13) + char(10)
  end
  set @@cmd = @@cmd + '  ' + @@fetch + char(13) + char(10) +
    'end' + char(13) + char(10)
  exec(@@cmd)
  if @format % 10 = 1 and coalesce(@idcol, 0) <> 0
  begin
    set @@print = 'SET IDENTITY_INSERT ' + @@objfullname + ' OFF'
    exec sp_temp_Print @@print
  end
  close selcursor
  deallocate selcursor
  close @@desc
  deallocate @@desc
GO

-- Decrypts SQL Server 7 stored procedures created with "with encryption"
-- option
-- Input : @encstring: encrypted text
-- Output: @value: decrypted text
create procedure sp_temp_decrypt_7
  @encstring varbinary(8000),
  @value varchar(8000) output
as
  set nocount on

-- Reversed by jimmers (Martin Rakhmanoff)
-- 12:58 PM 6/26/2002
-- jimmers@yandex.ru
-- http://jimmers.narod.ru

-- Rewritten for MS-SQL 7.0, with int data types instead of bigint and output
-- parameter instead of select by Joe Forster/STA, sta@c64.org (2004-01-28)
-- Note that the encryption in MS-SQL 2000 is different; you will not be able
-- to decrypt its encrypted stored procedures properly with this routine!

-- This is "magic" data used for crypting
-- May be found in sqlservr.exe 7/2000 image

  create table #magic (
    id int identity(0,1) primary key not null,
    val int)
  create table #output (
    id int identity(0,1) primary key not null,
    val int)

  insert into #magic values (0x00000000)
  insert into #magic values (0x77073096)
  insert into #magic values (0xEE0E612C)
  insert into #magic values (0x990951BA)
  insert into #magic values (0x076DC419)
  insert into #magic values (0x706AF48F)
  insert into #magic values (0xE963A535)
  insert into #magic values (0x9E6495A3)
  insert into #magic values (0x0EDB8832)
  insert into #magic values (0x79DCB8A4)
  insert into #magic values (0xE0D5E91E)
  insert into #magic values (0x97D2D988)
  insert into #magic values (0x09B64C2B)
  insert into #magic values (0x7EB17CBD)
  insert into #magic values (0xE7B82D07)
  insert into #magic values (0x90BF1D91)
  insert into #magic values (0x1DB71064)
  insert into #magic values (0x6AB020F2)
  insert into #magic values (0xF3B97148)
  insert into #magic values (0x84BE41DE)
  insert into #magic values (0x1ADAD47D)
  insert into #magic values (0x6DDDE4EB)
  insert into #magic values (0xF4D4B551)
  insert into #magic values (0x83D385C7)
  insert into #magic values (0x136C9856)
  insert into #magic values (0x646BA8C0)
  insert into #magic values (0xFD62F97A)
  insert into #magic values (0x8A65C9EC)
  insert into #magic values (0x14015C4F)
  insert into #magic values (0x63066CD9)
  insert into #magic values (0xFA0F3D63)
  insert into #magic values (0x8D080DF5)
  insert into #magic values (0x3B6E20C8)
  insert into #magic values (0x4C69105E)
  insert into #magic values (0xD56041E4)
  insert into #magic values (0xA2677172)
  insert into #magic values (0x3C03E4D1)
  insert into #magic values (0x4B04D447)
  insert into #magic values (0xD20D85FD)
  insert into #magic values (0xA50AB56B)
  insert into #magic values (0x35B5A8FA)
  insert into #magic values (0x42B2986C)
  insert into #magic values (0xDBBBC9D6)
  insert into #magic values (0xACBCF940)
  insert into #magic values (0x32D86CE3)
  insert into #magic values (0x45DF5C75)
  insert into #magic values (0xDCD60DCF)
  insert into #magic values (0xABD13D59)
  insert into #magic values (0x26D930AC)
  insert into #magic values (0x51DE003A)
  insert into #magic values (0xC8D75180)
  insert into #magic values (0xBFD06116)
  insert into #magic values (0x21B4F4B5)
  insert into #magic values (0x56B3C423)
  insert into #magic values (0xCFBA9599)
  insert into #magic values (0xB8BDA50F)
  insert into #magic values (0x2802B89E)
  insert into #magic values (0x5F058808)
  insert into #magic values (0xC60CD9B2)
  insert into #magic values (0xB10BE924)
  insert into #magic values (0x2F6F7C87)
  insert into #magic values (0x58684C11)
  insert into #magic values (0xC1611DAB)
  insert into #magic values (0xB6662D3D)
  insert into #magic values (0x76DC4190)
  insert into #magic values (0x01DB7106)
  insert into #magic values (0x98D220BC)
  insert into #magic values (0xEFD5102A)
  insert into #magic values (0x71B18589)
  insert into #magic values (0x06B6B51F)
  insert into #magic values (0x9FBFE4A5)
  insert into #magic values (0xE8B8D433)
  insert into #magic values (0x7807C9A2)
  insert into #magic values (0x0F00F934)
  insert into #magic values (0x9609A88E)
  insert into #magic values (0xE10E9818)
  insert into #magic values (0x7F6A0DBB)
  insert into #magic values (0x086D3D2D)
  insert into #magic values (0x91646C97)
  insert into #magic values (0xE6635C01)
  insert into #magic values (0x6B6B51F4)
  insert into #magic values (0x1C6C6162)
  insert into #magic values (0x856530D8)
  insert into #magic values (0xF262004E)
  insert into #magic values (0x6C0695ED)
  insert into #magic values (0x1B01A57B)
  insert into #magic values (0x8208F4C1)
  insert into #magic values (0xF50FC457)
  insert into #magic values (0x65B0D9C6)
  insert into #magic values (0x12B7E950)
  insert into #magic values (0x8BBEB8EA)
  insert into #magic values (0xFCB9887C)
  insert into #magic values (0x62DD1DDF)
  insert into #magic values (0x15DA2D49)
  insert into #magic values (0x8CD37CF3)
  insert into #magic values (0xFBD44C65)
  insert into #magic values (0x4DB26158)
  insert into #magic values (0x3AB551CE)
  insert into #magic values (0xA3BC0074)
  insert into #magic values (0xD4BB30E2)
  insert into #magic values (0x4ADFA541)
  insert into #magic values (0x3DD895D7)
  insert into #magic values (0xA4D1C46D)
  insert into #magic values (0xD3D6F4FB)
  insert into #magic values (0x4369E96A)
  insert into #magic values (0x346ED9FC)
  insert into #magic values (0xAD678846)
  insert into #magic values (0xDA60B8D0)
  insert into #magic values (0x44042D73)
  insert into #magic values (0x33031DE5)
  insert into #magic values (0xAA0A4C5F)
  insert into #magic values (0xDD0D7CC9)
  insert into #magic values (0x5005713C)
  insert into #magic values (0x270241AA)
  insert into #magic values (0xBE0B1010)
  insert into #magic values (0xC90C2086)
  insert into #magic values (0x5768B525)
  insert into #magic values (0x206F85B3)
  insert into #magic values (0xB966D409)
  insert into #magic values (0xCE61E49F)
  insert into #magic values (0x5EDEF90E)
  insert into #magic values (0x29D9C998)
  insert into #magic values (0xB0D09822)
  insert into #magic values (0xC7D7A8B4)
  insert into #magic values (0x59B33D17)
  insert into #magic values (0x2EB40D81)
  insert into #magic values (0xB7BD5C3B)
  insert into #magic values (0xC0BA6CAD)
  insert into #magic values (0xEDB88320)
  insert into #magic values (0x9ABFB3B6)
  insert into #magic values (0x03B6E20C)
  insert into #magic values (0x74B1D29A)
  insert into #magic values (0xEAD54739)
  insert into #magic values (0x9DD277AF)
  insert into #magic values (0x04DB2615)
  insert into #magic values (0x73DC1683)
  insert into #magic values (0xE3630B12)
  insert into #magic values (0x94643B84)
  insert into #magic values (0x0D6D6A3E)
  insert into #magic values (0x7A6A5AA8)
  insert into #magic values (0xE40ECF0B)
  insert into #magic values (0x9309FF9D)
  insert into #magic values (0x0A00AE27)
  insert into #magic values (0x7D079EB1)
  insert into #magic values (0xF00F9344)
  insert into #magic values (0x8708A3D2)
  insert into #magic values (0x1E01F268)
  insert into #magic values (0x6906C2FE)
  insert into #magic values (0xF762575D)
  insert into #magic values (0x806567CB)
  insert into #magic values (0x196C3671)
  insert into #magic values (0x6E6B06E7)
  insert into #magic values (0xFED41B76)
  insert into #magic values (0x89D32BE0)
  insert into #magic values (0x10DA7A5A)
  insert into #magic values (0x67DD4ACC)
  insert into #magic values (0xF9B9DF6F)
  insert into #magic values (0x8EBEEFF9)
  insert into #magic values (0x17B7BE43)
  insert into #magic values (0x60B08ED5)
  insert into #magic values (0xD6D6A3E8)
  insert into #magic values (0xA1D1937E)
  insert into #magic values (0x38D8C2C4)
  insert into #magic values (0x4FDFF252)
  insert into #magic values (0xD1BB67F1)
  insert into #magic values (0xA6BC5767)
  insert into #magic values (0x3FB506DD)
  insert into #magic values (0x48B2364B)
  insert into #magic values (0xD80D2BDA)
  insert into #magic values (0xAF0A1B4C)
  insert into #magic values (0x36034AF6)
  insert into #magic values (0x41047A60)
  insert into #magic values (0xDF60EFC3)
  insert into #magic values (0xA867DF55)
  insert into #magic values (0x316E8EEF)
  insert into #magic values (0x4669BE79)
  insert into #magic values (0xCB61B38C)
  insert into #magic values (0xBC66831A)
  insert into #magic values (0x256FD2A0)
  insert into #magic values (0x5268E236)
  insert into #magic values (0xCC0C7795)
  insert into #magic values (0xBB0B4703)
  insert into #magic values (0x220216B9)
  insert into #magic values (0x5505262F)
  insert into #magic values (0xC5BA3BBE)
  insert into #magic values (0xB2BD0B28)
  insert into #magic values (0x2BB45A92)
  insert into #magic values (0x5CB36A04)
  insert into #magic values (0xC2D7FFA7)
  insert into #magic values (0xB5D0CF31)
  insert into #magic values (0x2CD99E8B)
  insert into #magic values (0x5BDEAE1D)
  insert into #magic values (0x9B64C2B0)
  insert into #magic values (0xEC63F226)
  insert into #magic values (0x756AA39C)
  insert into #magic values (0x026D930A)
  insert into #magic values (0x9C0906A9)
  insert into #magic values (0xEB0E363F)
  insert into #magic values (0x72076785)
  insert into #magic values (0x05005713)
  insert into #magic values (0x95BF4A82)
  insert into #magic values (0xE2B87A14)
  insert into #magic values (0x7BB12BAE)
  insert into #magic values (0x0CB61B38)
  insert into #magic values (0x92D28E9B)
  insert into #magic values (0xE5D5BE0D)
  insert into #magic values (0x7CDCEFB7)
  insert into #magic values (0x0BDBDF21)
  insert into #magic values (0x86D3D2D4)
  insert into #magic values (0xF1D4E242)
  insert into #magic values (0x68DDB3F8)
  insert into #magic values (0x1FDA836E)
  insert into #magic values (0x81BE16CD)
  insert into #magic values (0xF6B9265B)
  insert into #magic values (0x6FB077E1)
  insert into #magic values (0x18B74777)
  insert into #magic values (0x88085AE6)
  insert into #magic values (0xFF0F6A70)
  insert into #magic values (0x66063BCA)
  insert into #magic values (0x11010B5C)
  insert into #magic values (0x8F659EFF)
  insert into #magic values (0xF862AE69)
  insert into #magic values (0x616BFFD3)
  insert into #magic values (0x166CCF45)
  insert into #magic values (0xA00AE278)
  insert into #magic values (0xD70DD2EE)
  insert into #magic values (0x4E048354)
  insert into #magic values (0x3903B3C2)
  insert into #magic values (0xA7672661)
  insert into #magic values (0xD06016F7)
  insert into #magic values (0x4969474D)
  insert into #magic values (0x3E6E77DB)
  insert into #magic values (0xAED16A4A)
  insert into #magic values (0xD9D65ADC)
  insert into #magic values (0x40DF0B66)
  insert into #magic values (0x37D83BF0)
  insert into #magic values (0xA9BCAE53)
  insert into #magic values (0xDEBB9EC5)
  insert into #magic values (0x47B2CF7F)
  insert into #magic values (0x30B5FFE9)
  insert into #magic values (0xBDBDF21C)
  insert into #magic values (0xCABAC28A)
  insert into #magic values (0x53B39330)
  insert into #magic values (0x24B4A3A6)
  insert into #magic values (0xBAD03605)
  insert into #magic values (0xCDD70693)
  insert into #magic values (0x54DE5729)
  insert into #magic values (0x23D967BF)
  insert into #magic values (0xB3667A2E)
  insert into #magic values (0xC4614AB8)
  insert into #magic values (0x5D681B02)
  insert into #magic values (0x2A6F2B94)
  insert into #magic values (0xB40BBE37)
  insert into #magic values (0xC30C8EA1)
  insert into #magic values (0x5A05DF1B)
  insert into #magic values (0x2D02EF8D)

-- Temporary storage for the simulation of bigint multiplication and division
  declare @temp01 int
  declare @temp02 int
  declare @temp03 int
  declare @temp04 int
  declare @temp05 int
  declare @temp06 int
  declare @temp07 int
  declare @temp08 int
  declare @temp09 int
  declare @temp10 int
  declare @temp11 int

-- This is "worker set" - changed every cycle
-- Was: declare @worker1 bigint
  declare @worker1lo int
  declare @worker1hi int
-- Was: declare @worker2 bigint
  declare @worker2lo int
  declare @worker2hi int
-- Was: declare @worker3 bigint
  declare @worker3lo int
  declare @worker3hi int

-- Initialize it...
-- Was: set @worker1 = 0x37C5D22A
  set @worker1lo = 0x0000D22A
  set @worker1hi = 0x000037C5
-- Was: set @worker2 = 0x77BE9B9C
  set @worker2lo = 0x00009B9C
  set @worker2hi = 0x000077BE
-- Was: set @worker3 = 0x36456178
  set @worker3lo = 0x00006178
  set @worker3hi = 0x00003645

  declare @decstring varchar(8000)
  declare @length int
  declare @i int
-- Was: declare @val bigint
  declare @val int
-- Was: declare @result bigint
  declare @result int

  set @decstring = ''
  set @length = len(@encstring)

  if @length <= 2
    goto errorhandler

  set @i = 1

  while (@i <= @length)
  begin

    set @val = cast(substring(@encstring, @i, 1) as int)

-- Was: set @result = (@worker3 & 0xFFFF) | 0x2
    set @result = (@worker3lo & 0xFFFF) | 0x2
-- Was: set @result = ((@result * (@result ^ 1)) ^ @val) & 0xFF
    set @temp01 = (@result & 0xFF) * (@result ^ 1)
    set @temp02 = (@result / 0x100) * (@result ^ 1)
    set @result = (((@temp01 / 0x100) + @temp02) ^ @val) & 0xFF

    insert #output values (@result)

-- Was: select @val = val from #magic where id = (@worker1 ^ @result) & 0xFF
    select @val = val from #magic where id = (@worker1lo ^ @result) & 0xFF
-- Was: set @worker1 = ((@worker1 / 0x100) ^ @val) & 0xFFFFFFFF
    set @worker1lo = (@worker1lo / 0x100) + ((@worker1hi & 0xFF) * 0x100) ^ (@val & 0xFFFF)
    set @worker1hi = @worker1hi / 0x100 ^ (((@val - case when @val < 0 then 0xFFFF else 0 end) / 0x10000) & 0xFFFF)

-- Was: set @worker2 = @worker2 + (@worker1 & 0xFF)
    set @worker2hi = @worker2hi + case when @worker2lo + (@worker1lo & 0xFF) >= 0x10000 then 1 else 0 end
    set @worker2lo = (@worker2lo + (@worker1lo & 0xFF)) & 0xFFFF
-- Was: set @worker2 = ((@worker2 * 0x08088405) & 0xFFFFFFFF) + 1
    set @temp01 = (@worker2lo & 0xFF) * 0x05
    set @temp02 = (@worker2lo & 0xFF) * 0x84
    set @temp03 = (@worker2lo & 0xFF) * 0x08
    set @temp04 = (@worker2lo & 0xFF) * 0x08
    set @temp05 = (@worker2lo / 0x100) * 0x05
    set @temp06 = (@worker2lo / 0x100) * 0x84
    set @temp07 = (@worker2lo / 0x100) * 0x08
    set @temp08 = (@worker2lo / 0x100) * 0x08
    set @temp09 = (@worker2hi & 0xFF) * 0x05
    set @temp10 = (@worker2hi & 0xFF) * 0x84
    set @temp11 = (@worker2hi / 0x100) * 0x05
    set @worker2lo = @temp01 + (@temp02 & 0xFF) * 0x100 + (@temp05 & 0xFF) * 0x100
    set @worker2hi = (@temp02 / 0x100 + @temp05 / 0x100 + @temp03 + @temp06 + @temp09 + (@temp04 + @temp07 + @temp10 + @temp11) * 0x100) & 0xFFFF + case when @worker2lo >= 0x10000 then 1 else 0 end
    set @worker2lo = @worker2lo & 0xFFFF
    set @worker2hi = @worker2hi + case when @worker2lo + 1 >= 0x10000 then 1 else 0 end
    set @worker2lo = (@worker2lo + 1) & 0xFFFF

-- Was: select @val = val from #magic where id = (@worker3 ^ (@worker2 / 0x1000000)) & 0xFF
    select @val = val from #magic where id = (@worker3lo ^ (@worker2hi / 0x100)) & 0xFF
-- Was: set @worker3 = ((@worker3 / 0x100) ^ @val) & 0xffffffff
    set @worker3lo = (@worker3lo / 0x100) + ((@worker3hi & 0xFF) * 0x100) ^ (@val & 0xFFFF)
    set @worker3hi = @worker3hi / 0x100 ^ (((@val - case when @val < 0 then 0xFFFF else 0 end) / 0x10000) & 0xFFFF)

    set @i = @i + 1
  end

  select @decstring = @decstring + nchar(o2.val * 0x100 + o1.val)
    from #output o1
   inner join #output o2
      on o1.id = o2.id - 1
   where o1.id % 2 = 0

  set @value = @decstring

  return 0

  errorhandler:
    raiserror ('Empty input for decryption.', 16, 1)
    return 1
GO

-- Variables
declare @@_insertbatchlength int
declare @@_batchsep varchar(8000)
declare @@_wordsep varchar(8000)
declare @@_indentstr varchar(8000)
declare @@_identquoteopen varchar(8000)
declare @@_identquoteclose varchar(8000)
declare @@_listsep char
declare @@_listsep2 char
declare @@_listnot char

declare @@_identformat int
declare @@_sepfiles int
declare @@_roles int
declare @@_users int
declare @@_tables int
declare @@_views int
declare @@_procedures int
declare @@_drop int
declare @@_create int
declare @@_primarykey int
declare @@_index int
declare @@_default int
declare @@_foreignkey int
declare @@_permission int
declare @@_data int
declare @@_objnames varchar(8000)

declare @@basephase int
declare @@phase int
declare @@phasebegin int
declare @@phaseend int
declare @@phasenextobj int
declare @@encrypted bit
declare @@colstatus tinyint
declare @@typestatus tinyint
declare @@actadd smallint
declare @@actmod smallint
declare @@collength smallint
declare @@objid int
declare @@objxtypeindex int
declare @@index int
declare @@index2 int
declare @@linelen int
declare @@linenum int
declare @@headerfixstatus int
declare @@idxstatus int
declare @@idxindid int
declare @@deplevel int
declare @@permbit int
declare @@incomment int
declare @@lineoffs int
declare @@createfound int
declare @@identitycolumn int
declare @@objxtype varchar(100)
declare @@objxtypestr varchar(100)
declare @@objpropstr varchar(100)
declare @@permstr varchar(100)
declare @@objxtypenames varchar(8000)
declare @@text varchar(8000)
declare @@line varchar(8000)
declare @@line2 varchar(8000)
declare @@line3 varchar(8000)
declare @@textbin varbinary(8000)
declare @@usrname sysname
declare @@logname sysname
declare @@grpname sysname
declare @@objname sysname
declare @@ownname sysname
declare @@colname sysname
declare @@colxtype sysname
declare @@idxname sysname
declare @@objname2 sysname
declare @@ownname2 sysname
declare @@colname2 sysname
declare @@defname sysname
declare @@objfullname sysname
declare @@objfullname2 sysname
declare @@filename varchar(8000)

-- Constants
-- Format for identifiers; 0: identifier names; 1: owner name prefixed; 10:
-- quoted identifier names; 11: quoted owner and identifier names
set @@_identformat = 1
-- Number of INSERT statements in each batch
set @@_insertbatchlength = 100
-- Batch separator text
set @@_batchsep = 'GO'
-- Word separator characters
set @@_wordsep = ' (),' + char(9) + char(10) + char(13)
-- Indenting text
set @@_indentstr = char(9)
--set @@_indentstr = '  '
-- Identifier quotation marks
set @@_identquoteopen = '[' set @@_identquoteclose = ']'
-- List control characters; DO NOT CHANGE!
set @@_listsep = '|'
set @@_listsep2 = '#'
set @@_listnot = '~'

-- - @@_sepfiles: Separate files (script blocks) for each object; see the
--   documentation for more details
-- - @@_roles: Process database roles
-- - @@_users: Process database users
-- - @@_tables: @@_tables: Process user-defined tables
-- - @@_views: Process user-defined views
-- - @@_procedures: Process user-defined stored procedures
-- - @@_drop: Drop processed objects (and, for tables, foreign keys, as well)
-- - @@_create: Create processed objects
-- - @@_primarykey: Create primary keys
-- - @@_index: Create indexes
-- - @@_default: Create default constraints
-- - @@_foreignkey: Create foreign keys; 1 = all foreign keys; 2 = only those
--   foreign keys that refer to tables also processed
-- - @@_permission: Grant or deny user and role permissions to processed
--   objects
-- - @@_data: List data; 1 = INSERT statements; 2 = Tab separated; 3 =  Comma
--   separated (CSV); add 10 for column headers; add 100 for prepending table
--   name
-- - @@_objnames: Filter names of objects to be processed; "*" and "?"
--   wildcards allowed; separate object names with "|"; prepend type character
--   ("T": table; "V": view; "P": stored procedure) plus "#" to filter for
--   object type; prepend "~" to exclude objects from processing
-- !!!PARAMETERS!!!
set @@_sepfiles = 0
set @@_roles = 0
set @@_users = 0
set @@_tables = 1
set @@_views = 1
set @@_procedures = 1
set @@_drop = 1
set @@_create = 1
set @@_primarykey = 1
set @@_index = 1
set @@_default = 1
set @@_foreignkey = 1
set @@_permission = 1
set @@_data = 0
set @@_objnames = ''

-- Collect internal types of globally accepted objects
set @@objxtypenames = @@_listsep
if coalesce(@@_tables, 0) <> 0
  select @@objxtypenames = @@objxtypenames + xtype + @@_listsep
    from tbl_temp_ObjectTypes
    where typename = 'TABLE'
if coalesce(@@_views, 0) <> 0
  select @@objxtypenames = @@objxtypenames + xtype + @@_listsep
    from tbl_temp_ObjectTypes
    where typename = 'VIEW'
if coalesce(@@_procedures, 0) <> 0
  select @@objxtypenames = @@objxtypenames + xtype + @@_listsep
    from tbl_temp_ObjectTypes
    where typename = 'PROCEDURE'
set @@objxtypeindex = 2
-- Delete identifier quotation marks when identifier quotation is disabled
if (@@_identformat % 100) / 10 = 0
begin
  set @@_identquoteopen = ''
  set @@_identquoteclose = ''
end
-- Do not script table modifiers and data when only dropping objects
if coalesce(@@_drop, 0) <> 0 and coalesce(@@_create, 0) = 0
begin
  set @@_primarykey = 0
  set @@_index = 0
  set @@_default = 0
  set @@_foreignkey = 0
  set @@_permission = 0
  set @@_data = 0
end
-- Do not display table name or column headers for INSERT statements
if coalesce(@@_data, 0) % 10 = 1
  set @@_data = @@_data % 10
-- Do not allow unknown data formats
if coalesce(@@_data, 0) % 10 > 3
  set @@_data = 0
-- Do not script anything else when scripting non-INSERT-statement data
if coalesce(@@_data, 0) > 1
begin
  set @@_roles = 0
  set @@_users = 0
  set @@_procedures = 0
  set @@_drop = 0
  set @@_create = 0
  set @@_primarykey = 0
  set @@_index = 0
  set @@_default = 0
  set @@_foreignkey = 0
  set @@_permission = 0
end
-- Dependency table
create table tbl_temp_Dependency (
  name sysname not null,
  id int not null,
  xtype varchar(2) not null,
  deplevel int null)
-- Collect specific object names and types
set @@line = null
set @@line2 = null
while coalesce(@@_objnames, '') <> ''
begin
  set @@index = charindex(@@_listsep, @@_objnames)
  if @@index = 0
    set @@index = len(@@_objnames)
  else
    set @@index = @@index - 1
  set @@text = substring(@@_objnames, 1, @@index)
  set @@_objnames = substring(@@_objnames, @@index + 2, 8000)
  set @@linenum = 0
  if substring(@@text, 1, 1) = @@_listnot
  begin
    set @@linenum = @@linenum | 0x01
    set @@text = substring(@@text, 2, 8000)
  end
  set @@objxtype = null
  set @@index = charindex(@@_listsep2, @@text)
  if @@index <> 0
  begin
    set @@objxtype = substring(@@text, 1, @@index - 1)
    set @@text = substring(@@text, @@index + 1, 8000)
  end
  if charindex('*', @@text) <> 0 or charindex('?', @@text) <> 0
  begin
    set @@linenum = @@linenum | 0x02
    set @@text = replace(replace(replace(@@text, '_', '[_]'), '*', '%'), '?', '_')
  end
  set @@text = '(name ' + case when @@linenum & 0x02 = 0 then '=' else 'like' end + ' ''' + @@text + '''' + coalesce(' and usrtype = ''' + @@objxtype + '''', '') + ')'
  if @@linenum & 0x01 = 0
    set @@line = coalesce(@@line, '') + case when @@line is not null then ' or ' else '' end + @@text
  else
    set @@line2 = coalesce(@@line2, '') + case when @@line2 is not null then ' or ' else '' end  + @@text
end
-- Collect globally accepted object types
set @@line3 = ''
select @@line3 = @@line3 + ' or objectproperty(id, ''' + prop + ''') <> 0'
  from tbl_temp_ObjectTypes
  where charindex(@@_listsep + xtype + @@_listsep, @@objxtypenames) <> 0
  order by ord
-- Put objects, to be processed, into dependency table, excluding temporary
-- objects of the script
set @@text = 'insert into tbl_temp_Dependency' + char(13) + char(10) +
  'select name, id, rtrim(sysobjects.xtype), null' + char(13) + char(10) +
  'from sysobjects' + char(13) + char(10) +
  'join tbl_temp_ObjectTypes on tbl_temp_ObjectTypes.xtype = sysobjects.xtype' + char(13) + char(10) +
  'where charindex(rtrim(sysobjects.xtype), ''' + @@objxtypenames + ''') > 0' + char(13) + char(10) +
  'and objectproperty(id, ''IsMSShipped'') = 0 and name not like ''tbl_temp_%'' and name not like ''sp_temp_%''' + char(13) + char(10) +
  coalesce('and (' + @@line + ')' + char(13) + char(10), '') +
  coalesce('and not (' + @@line2 + ')' + char(13) + char(10), '') +
  'and (0 = 1' + @@line3 + ')'
exec(@@text)
-- Set objects, that do not depend on other objects of the same kind, to
-- dependency level zero
update tbl_temp_Dependency
  set deplevel = 0
  where (xtype = (select xtype
        from tbl_temp_ObjectTypes
        where typename = 'TABLE')
      and not exists (select 1
        from sysforeignkeys
          join sysobjects on sysforeignkeys.rkeyid = sysobjects.id
        where sysforeignkeys.fkeyid = tbl_temp_Dependency.id
          and tbl_temp_Dependency.xtype = sysobjects.xtype
          and exists (select 1
            from tbl_temp_Dependency
            where tbl_temp_Dependency.xtype = sysobjects.xtype
              and tbl_temp_Dependency.name = sysobjects.name)))
    or (xtype <> (select xtype
        from tbl_temp_ObjectTypes
        where typename = 'TABLE')
      and not exists (select 1
        from sysdepends
          join sysobjects on sysdepends.depid = sysobjects.id
        where sysdepends.id = tbl_temp_Dependency.id
          and tbl_temp_Dependency.xtype = sysobjects.xtype))
-- Table dependencies; for dependent objects, set the dependency level to
-- the level of the tables, they depend on, plus one; based on foreign key
-- structure stored in the sysforeignkeys table
set @@deplevel = 1
while exists (select 1
  from tbl_temp_Dependency
    where xtype = (select xtype
        from tbl_temp_ObjectTypes
        where typename = 'TABLE')
      and exists (select 1
        from sysforeignkeys
          join tbl_temp_Dependency temp2 on sysforeignkeys.rkeyid = temp2.id
        where sysforeignkeys.fkeyid = tbl_temp_Dependency.id
          and temp2.xtype = tbl_temp_Dependency.xtype
          and temp2.deplevel = @@deplevel - 1))
begin
  update tbl_temp_Dependency
    set deplevel = @@deplevel
    where xtype = (select xtype
        from tbl_temp_ObjectTypes
        where typename = 'TABLE')
      and exists (select 1
        from sysforeignkeys
          join tbl_temp_Dependency temp2 on sysforeignkeys.rkeyid = temp2.id
        where sysforeignkeys.fkeyid = tbl_temp_Dependency.id
          and temp2.xtype = tbl_temp_Dependency.xtype
          and temp2.deplevel = @@deplevel - 1)
  set @@deplevel = @@deplevel + 1
end
-- View and stored procedure dependencies; for dependent objects, set the
-- dependency level to the level of objets of the same type, they depend on,
-- plus one; based on dependency structure stored in the sysdepends table
set @@deplevel = 1
while exists (select 1
  from tbl_temp_Dependency
    where xtype <> (select xtype
        from tbl_temp_ObjectTypes
        where typename = 'TABLE')
      and exists (select 1
        from sysdepends
          join tbl_temp_Dependency temp2 on sysdepends.depid = temp2.id
        where sysdepends.id = tbl_temp_Dependency.id
          and temp2.xtype = tbl_temp_Dependency.xtype
          and temp2.deplevel = @@deplevel - 1))
begin
  update tbl_temp_Dependency
    set deplevel = @@deplevel
    where xtype <> (select xtype
        from tbl_temp_ObjectTypes
        where typename = 'TABLE')
      and exists (select 1
        from sysdepends
          join tbl_temp_Dependency temp2 on sysdepends.depid = temp2.id
        where sysdepends.id = tbl_temp_Dependency.id
          and temp2.xtype = tbl_temp_Dependency.xtype
          and temp2.deplevel = @@deplevel - 1)
  set @@deplevel = @@deplevel + 1
end
-- Reset phase counter and set end of phase
set @@basephase = 0
set @@phase = 0
set @@phasebegin = 0
set @@phaseend = 1
-- Loop until end of all phase
while @@phase >= 0
begin
  if @@phaseend = 0
  begin
    if @@phase = 1
    begin
-- Drop objects
      exec sp_temp_Print
      exec sp_temp_ObjFullName @@objname, @@ownname, @@objfullname output, @@_identformat, @@_identquoteopen, @@_identquoteclose
      set @@line = 'IF EXISTS (SELECT * FROM sysobjects WHERE id = OBJECT_ID(''' + @@objfullname + ''') AND OBJECTPROPERTY(id, ''' + @@objpropstr + ''') <> 0)'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'DROP ' + @@objxtypestr + ' ' + @@objfullname
      exec sp_temp_Print @@line
      exec sp_temp_Print @@_batchsep
      set @@line = ''
    end
    else if @@phase = 2
    begin
-- Drop users
      exec sp_temp_Print
      set @@line = 'IF EXISTS (SELECT * FROM sysusers WHERE name = ''' + @@usrname + ''' AND uid < 16382)'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'EXEC sp_dropuser ''' + @@usrname + ''''
      exec sp_temp_Print @@line
      exec sp_temp_Print @@_batchsep
      set @@line = ''
    end
    else if @@phase = 3
    begin
-- Drop roles
      exec sp_temp_Print
      set @@line = 'IF EXISTS (SELECT * FROM sysusers WHERE name = ''' + @@grpname + ''' AND uid > 16399)'
      exec sp_temp_Print @@line
      set @@line = 'BEGIN'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'DECLARE @@usrname sysname'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'DECLARE usrcursor CURSOR FOR SELECT sysusr1.name'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + @@_indentstr + 'FROM sysusers sysusr1 JOIN sysmembers ON memberuid = sysusr1.uid JOIN SYSUSERS sysusr2 ON groupuid = sysusr2.uid'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + @@_indentstr + 'WHERE sysusr2.name = ''' + @@grpname + ''' AND sysusr1.uid > 1 AND sysusr1.uid NOT BETWEEN 16382 AND 16399 AND sysusr1.issqlrole = 0 AND sysusr1.hasdbaccess <> 0 AND sysusr1.sid IS NOT NULL'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'OPEN usrcursor'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'FETCH NEXT FROM usrcursor INTO @@usrname'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'WHILE @@fetch_status <> -1'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'BEGIN'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + @@_indentstr + 'EXEC sp_droprolemember ''' + @@grpname + ''', @@usrname'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + @@_indentstr + 'FETCH NEXT FROM usrcursor INTO @@usrname'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'END'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'CLOSE usrcursor'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'DEALLOCATE usrcursor'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'EXEC sp_droprole ''' + @@grpname + ''''
      exec sp_temp_Print @@line
      set @@line = 'end'
      exec sp_temp_Print @@line
      exec sp_temp_Print @@_batchsep
      set @@line = ''
    end
    else if @@phase = 4
    begin
-- Create roles
      exec sp_temp_Print
      set @@line = 'IF NOT EXISTS (SELECT * FROM sysusers WHERE name = ''' + @@grpname + ''' AND uid > 16399)'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'EXEC sp_addrole ''' + @@grpname + ''''
      exec sp_temp_Print @@line
      exec sp_temp_Print @@_batchsep
      set @@line = ''
    end
    else if @@phase = 5
    begin
-- Create users
      exec sp_temp_Print
      set @@line = 'IF NOT EXISTS (SELECT * FROM sysusers WHERE name = ''' + @@usrname + ''' AND uid < 16382)'
      exec sp_temp_Print @@line
      set @@line = @@_indentstr + 'EXEC sp_grantdbaccess ''' + @@logname + ''', ''' + @@usrname + ''''
      exec sp_temp_Print @@line
      if @@_sepfiles <> 0
      begin
        declare grpcursor cursor for select sysusr2.name
          from sysusers sysusr1 join sysmembers on memberuid = sysusr1.uid join sysusers sysusr2 on groupuid = sysusr2.uid
          where sysusr1.name = @@usrname and sysusr1.uid > 1 and sysusr1.uid not between 16382 and 16399 and sysusr1.issqlrole = 0 and sysusr1.hasdbaccess <> 0 and sysusr1.sid is not null
        open grpcursor
        fetch next from grpcursor into @@grpname
        while @@fetch_status <> -1
        begin
          set @@line = @@_indentstr + 'EXEC sp_addrolemember ''' + @@grpname + ''', ''' + @@usrname + ''''
          exec sp_temp_Print @@line
         fetch next from grpcursor into @@grpname
        end
        close grpcursor
        deallocate grpcursor
      end
      exec sp_temp_Print @@_batchsep
      set @@line = ''
    end
    else if @@phase = 6
    begin
-- Add users to roles as members
      exec sp_temp_Print
      set @@line = 'EXEC sp_addrolemember ''' + @@grpname + ''', ''' + @@usrname + ''''
      exec sp_temp_Print @@line
      exec sp_temp_Print @@_batchsep
      set @@line = ''
    end
    else if @@phase = 7
    begin
-- Process objects
      exec sp_temp_ObjFullName @@objname, @@ownname, @@objfullname output, @@_identformat, @@_identquoteopen, @@_identquoteclose
      if @@objxtype = (select xtype
          from tbl_temp_ObjectTypes
          where typename = 'TABLE')
      begin 
-- Tables
        if coalesce(@@_create, 0) <> 0
        begin
-- Create table
          exec sp_temp_Print
          set @@line = 'CREATE TABLE ' + @@objfullname + ' ('
          exec sp_temp_Print @@line
          set @@line = ''
          set @@identitycolumn = 0
          declare colcursor cursor for
            select syscolumns.name,
                systypes.name,
                syscolumns.length,
                syscolumns.status,
                systypes.status,
                syscolumns.autoval
              from syscolumns
                join systypes on syscolumns.xtype = systypes.xtype
              where id = @@objid
              order by colorder
          open colcursor
          fetch next from colcursor into @@colname,
            @@colxtype,
            @@collength,
            @@colstatus,
            @@typestatus,
            @@textbin
          while @@fetch_status <> -1
          begin
            if @@line <> ''
            begin
              set @@line = @@line + ','
              exec sp_temp_Print @@line
            end
            set @@line = @@_indentstr + @@_identquoteopen + @@colname + @@_identquoteclose + ' ' + @@_identquoteopen + @@colxtype + @@_identquoteclose +
              case
                when @@typestatus & 2 > 0 then '(' + rtrim(convert(char, @@collength)) + ')'
                else ''
              end + ' ' +
              case
                when @@colstatus & 0x08 = 0 then 'NOT NULL'
                else 'NULL'
              end +
              case
                when @@colstatus & 0x80 <> 0 then ' IDENTITY (' +
                  case
                    when @@colxtype = 'int' then convert(varchar, cast(cast(reverse(cast(substring(@@textbin, 9, 4) as varchar)) as varbinary) as int)) + ', ' + convert(varchar, cast(cast(reverse(cast(substring(@@textbin, 5, 4) as varchar)) as varbinary) as int))
                    when @@colxtype = 'smallint' then convert(varchar, cast(cast(reverse(cast(substring(@@textbin, 5, 2) as varchar)) as varbinary) as int)) + ', ' + convert(varchar, cast(cast(reverse(cast(substring(@@textbin, 3, 2) as varchar)) as varbinary) as int))
                    when @@colxtype = 'tinyint' then convert(varchar, cast(substring(@@textbin, 3, 1) as int)) + ', ' + convert(varchar, cast(substring(@@textbin, 3, 2) as int))
                    when @@colxtype in ('decimal', 'numeric') then convert(varchar, cast(cast(reverse(cast(substring(@@textbin, 45, 4) as varchar)) as varbinary) as int)) + ', ' + convert(varchar, cast(cast(reverse(cast(substring(@@textbin, 25, 4) as varchar)) as varbinary) as int))
                  end + ')'
                else ''
              end
            if @@colstatus & 0x80 <> 0
              set @@identitycolumn = 1
            fetch next from colcursor into @@colname,
              @@colxtype,
              @@collength,
              @@colstatus,
              @@typestatus,
              @@textbin
          end
          if @@line <> ''
            exec sp_temp_Print @@line
          exec sp_temp_Print ')'
          exec sp_temp_Print @@_batchsep
          close colcursor
          deallocate colcursor
        end
        if coalesce(@@_primarykey, 0) <> 0
        begin
-- Primary keys
          set @@linenum = 0
          declare idxcursor cursor for
            select sysindexes.status,
                sysindexes.indid,
                sysindexes.name,
                syscolumns.name
              from sysindexkeys
                join sysindexes on sysindexkeys.id = sysindexes.id
                  and sysindexkeys.indid = sysindexes.indid
                join syscolumns on sysindexkeys.id = syscolumns.id
                  and sysindexkeys.colid = syscolumns.colid
              where sysindexes.id = @@objid
                and sysindexes.status & 0x800000 = 0
                and sysindexes.status & 0x0800 <> 0
            order by sysindexes.indid
          open idxcursor
          fetch next from idxcursor into @@idxstatus,
            @@idxindid,
            @@idxname,
            @@colname
          while @@fetch_status <> -1
          begin
            if @@linenum = 0
            begin
              exec sp_temp_Print
              set @@line = 'ALTER TABLE ' + @@objfullname + ' WITH NOCHECK ADD'
              exec sp_temp_Print @@line
              set @@line = @@_indentstr + 'CONSTRAINT ' + @@_identquoteopen + @@idxname + @@_identquoteclose + ' PRIMARY KEY ' +
                case
                  when @@idxindid = 1 then 'CLUSTERED'
                  else 'NONCLUSTERED'
                end + ' ('
              exec sp_temp_Print @@line
              set @@line = ''
            end
            if @@line <> ''
            begin
              set @@line = @@line + ','
              exec sp_temp_Print @@line
            end
            set @@line = @@_indentstr + @@_indentstr + @@_identquoteopen + @@colname + @@_identquoteclose
            set @@linenum = @@linenum + 1
            fetch next from idxcursor into @@idxstatus,
              @@idxindid,
              @@idxname,
              @@colname
          end
          if @@linenum <> 0
          begin
            if @@line <> ''
              exec sp_temp_Print @@line
            set @@line = @@_indentstr + ')'
            exec sp_temp_Print @@line
            exec sp_temp_Print @@_batchsep
          end
          close idxcursor
          deallocate idxcursor
        end
        if coalesce(@@_index, 0) <> 0
        begin
-- Indexes
          set @@linenum = 0
          declare idxcursor cursor for
            select sysindexes.status,
                sysindexes.indid,
                sysindexes.name,
                syscolumns.name
              from sysindexkeys
                join sysindexes on sysindexkeys.id = sysindexes.id
                  and sysindexkeys.indid = sysindexes.indid
                join syscolumns on sysindexkeys.id = syscolumns.id
                  and sysindexkeys.colid = syscolumns.colid
              where sysindexes.id = @@objid
                and sysindexes.status & 0x800800 = 0
            order by sysindexes.indid
          open idxcursor
          fetch next from idxcursor into @@idxstatus,
            @@idxindid,
            @@idxname,
            @@colname
          while @@fetch_status <> -1
          begin
            if @@linenum = 0
              exec sp_temp_Print
            set @@line = 'CREATE INDEX ' + @@_identquoteopen + @@idxname + @@_identquoteclose + ' ON ' + @@objfullname + '(' + @@_identquoteopen + @@colname + @@_identquoteclose + ')'
            exec sp_temp_Print @@line
            set @@linenum = @@linenum + 1
            fetch next from idxcursor into @@idxstatus,
              @@idxindid,
              @@idxname,
              @@colname
          end
          if @@linenum <> 0
            exec sp_temp_Print @@_batchsep
          close idxcursor
          deallocate idxcursor
        end
        if coalesce(@@_default, 0) <> 0
        begin
-- Defaults
          set @@linenum = 0
          declare defcursor cursor for
            select sysobjects.name,
                syscolumns.name,
                syscomments.text
              from sysconstraints
                join sysobjects on sysconstraints.constid = sysobjects.id
                join syscomments on syscomments.id = sysconstraints.constid
                join syscolumns on syscolumns.id = sysconstraints.id
                  and syscolumns.colid = sysconstraints.colid
              where sysconstraints.id = @@objid
                and sysconstraints.status & 0x0F = 5
                and sysconstraints.status & 0x10 <> 0
              order by sysconstraints.colid
          open defcursor
          fetch next from defcursor into @@defname,
            @@colname,
            @@text
          while @@fetch_status <> -1
          begin
            if @@linenum = 0
            begin
              exec sp_temp_Print
              set @@line = 'ALTER TABLE ' + @@objfullname + ' WITH NOCHECK ADD'
              exec sp_temp_Print @@line
              set @@line = ''
            end
            if @@line <> ''
            begin
              set @@line = @@line + ','
              exec sp_temp_Print @@line
            end
            set @@line = @@_indentstr + 'CONSTRAINT ' + @@_identquoteopen + @@defname + @@_identquoteclose + ' DEFAULT ' + @@text + ' FOR ' + @@_identquoteopen + @@colname + @@_identquoteclose
            set @@linenum = @@linenum + 1
            fetch next from defcursor into @@defname,
              @@colname,
              @@text
          end
          if @@linenum <> 0
          begin
            if @@line <> ''
              exec sp_temp_Print @@line
            exec sp_temp_Print @@_batchsep
          end
          close defcursor
          deallocate defcursor
        end
        if coalesce(@@_foreignkey, 0) <> 0
        begin
-- Foreign keys
          set @@linenum = 0
          declare fkeycursor cursor for
            select sysobjects.name,
                syscol1.name,
                sysobj2.name,
                syscol2.name,
                sysusers.name
              from sysforeignkeys
                join sysobjects on sysforeignkeys.constid = sysobjects.id
                join sysconstraints on sysforeignkeys.constid = sysconstraints.constid
                join syscolumns syscol1 on sysforeignkeys.fkeyid = syscol1.id
                  and sysforeignkeys.fkey = syscol1.colid
                join syscolumns syscol2 on sysforeignkeys.rkeyid = syscol2.id
                  and sysforeignkeys.rkey = syscol2.colid
                join sysobjects sysobj2 on syscol2.id = sysobj2.id
                join sysusers on sysobjects.uid = sysusers.uid
              where sysforeignkeys.fkeyid = @@objid
                and sysconstraints.status & 0x0F = 3
                and (coalesce(@@_foreignkey, 0) <> 2 or exists (select 1
                  from tbl_temp_Dependency
                  where tbl_temp_Dependency.xtype = sysobj2.xtype
                    and tbl_temp_Dependency.name = sysobj2.name))
              order by syscol1.colid
          open fkeycursor
          fetch next from fkeycursor into @@idxname,
            @@colname,
            @@objname2,
            @@colname2,
            @@ownname2
          while @@fetch_status <> -1
          begin
            if @@linenum = 0
            begin
              exec sp_temp_Print
              set @@line = 'ALTER TABLE ' + @@objfullname + ' WITH NOCHECK ADD'
              exec sp_temp_Print @@line
              set @@line = ''
            end
            if @@line <> ''
            begin
              set @@line = @@line + ','
              exec sp_temp_Print @@line
            end
            exec sp_temp_ObjFullName @@objname2, @@ownname2, @@objfullname2 output, @@_identformat, @@_identquoteopen, @@_identquoteclose
            set @@line = @@_indentstr + 'CONSTRAINT ' + @@_identquoteopen + @@idxname + @@_identquoteclose + ' FOREIGN KEY ('
            exec sp_temp_Print @@line
            set @@line = @@_indentstr + @@_indentstr + @@_identquoteopen + @@colname + @@_identquoteclose
            exec sp_temp_Print @@line
            set @@line = @@_indentstr + ') REFERENCES ' + @@objfullname2 + ' ('
            exec sp_temp_Print @@line
            set @@line = @@_indentstr + @@_indentstr + @@_identquoteopen + @@colname2 + @@_identquoteclose
            exec sp_temp_Print @@line
            set @@line = @@_indentstr + ')'
            set @@linenum = @@linenum + 1
            fetch next from fkeycursor into @@idxname,
              @@colname,
              @@objname2,
              @@colname2,
              @@ownname2
          end
          if @@linenum <> 0
          begin
            if @@line <> ''
              exec sp_temp_Print @@line
            exec sp_temp_Print @@_batchsep
          end
          close fkeycursor
          deallocate fkeycursor
        end
      end
      else
      begin
-- Views and stored procedures 
        if coalesce(@@_create, 0) <> 0
        begin
          exec sp_temp_Print
          set @@headerfixstatus = 0
          set @@incomment = 0
          set @@createfound = 0
          set @@lineoffs = 1
          declare textcursor cursor for
            select colid,
                encrypted,
                text,
                ctext
              from syscomments
              where id = @@objid
            order by colid
          open textcursor
          fetch next from textcursor into @@index,
            @@encrypted,
            @@text,
            @@textbin
          while @@fetch_status <> -1
          begin
-- Decrypt definition, if encrypted
            if @@encrypted <> 0
              exec sp_temp_decrypt_7 @@textbin, @@text output
            set @@linelen = charindex(char(10), @@text)
            while @@linelen > 0
            begin
              set @@line = @@line + substring(@@text, 1, @@linelen)
-- Delete "with encryption" option in the CREATE statement
              set @@index = charindex('WITH ENCRYPTION', @@text)
              if @@index > 0
              begin
                set @@text = substring(@@text, 1, @@index - 1) + substring(@@text, @@index + (len('WITH ENCRYPTION') + 1), 8000)
                set @@linelen = charindex(char(10), @@text)
              end
-- Replace object and owner name in the CREATE statement with standard syntax
              if @@headerfixstatus = 0
              begin
-- Skip comments in front of the CREATE statement
                while @@createfound = 0 and @@lineoffs < len(@@text)
                begin
                  if substring(@@text, @@lineoffs, 1) = '-'
                  begin
                    if @@incomment & 0x02 = 0
                    begin
                      set @@incomment = @@incomment & ~(convert(int, 0x60))
                      if @@incomment & 0x10 <> 0
                        set @@incomment = @@incomment & ~(convert(int, 0x70)) | 0x01
                      else
                        set @@incomment = @@incomment | 0x10
                    end
                  end
                  else if substring(@@text, @@lineoffs, 1) = '/'
                  begin
                    set @@incomment = @@incomment & ~(convert(int, 0x10))
                    if @@incomment & 0x01 = 0
                    begin
                      if @@incomment & 0x40 <> 0
                        set @@incomment = @@incomment & ~(convert(int, 0x62))
                      else
                        set @@incomment = @@incomment | 0x20
                    end
                  end
                  else if substring(@@text, @@lineoffs, 1) = '*'
                  begin
                    set @@incomment = @@incomment & ~(convert(int, 0x10))
                    if @@incomment & 0x01 = 0
                    begin
                      if @@incomment & 0x20 <> 0
                        set @@incomment = @@incomment & ~(convert(int, 0x60)) | 0x02
                      else
                        set @@incomment = @@incomment | 0x40
                    end
                  end
                  else
                  begin
                    if substring(@@text, @@lineoffs, 1) in (char(10), char(13))
                      set @@incomment = @@incomment & ~(convert(int, 0x01))
                    set @@incomment = @@incomment & ~(convert(int, 0xF0))
                  end
                  if @@incomment = 0 and substring(@@text, @@lineoffs, len('CREATE')) = 'CREATE'
                    set @@createfound = 1
                  set @@lineoffs = @@lineoffs + 1
                end
                if @@createfound = 1
                begin
                  set @@index = charindex(@@objname, @@text, @@lineoffs)
                  if @@index > 0
                  begin
-- Find object name in text and replace it with full qualified object name
                    while @@index > 2 and charindex(substring(@@text, @@index - 1, 1), @@_wordsep) = 0
                      set @@index = @@index - 1
                    set @@line2 = substring(@@text, @@index, 8000)
                    set @@index2 = 1
                    while @@index2 < len(@@line2) and charindex(substring(@@line2, @@index2 + 1, 1), @@_wordsep) = 0
                      set @@index2 = @@index2 + 1
                    set @@text = substring(@@text, 1, @@index - 1) + @@objfullname + substring(@@text, @@index + @@index2, 8000)
                    set @@headerfixstatus = 1
                  end
                end
              end
              set @@line = nullif(replace(replace(@@line, char(13), ''), char(10), ''), '')
              exec sp_temp_Print @@line
              set @@line = ''
              set @@text = substring(@@text, @@linelen + 1, 8000)
              set @@linelen = charindex(char(10), @@text)
            end
            set @@line = @@text
            fetch next from textcursor into @@index,
              @@encrypted,
              @@text,
              @@textbin
          end
          if @@line <> ''
          begin
            exec sp_temp_Print @@line
            set @@line = ''
          end
          exec sp_temp_Print @@_batchsep
          close textcursor
          deallocate textcursor
        end
      end
      if coalesce(@@_permission, 0) <> 0
      begin
-- Permissions
        set @@linenum = 0
        declare permcursor cursor for
          select name,
              actadd,
              actmod
            from syspermissions
              join sysusers on syspermissions.grantee = sysusers.uid
            where id = @@objid
              and ((issqlrole <> 0
                  and uid = gid)
                or (issqlrole = 0
                  and hasdbaccess <> 0
                  and sid is not null))
          order by name
        declare permcursor2 cursor for
          select val,
              name
            from tbl_temp_Permissions
            order by ord
        open permcursor
        fetch next from permcursor into @@usrname,
          @@actadd,
          @@actmod
        while @@fetch_status <> -1
        begin
          if @@actadd <> 0 or @@actmod <> 0
          begin
            set @@line = ''
            set @@line2 = ''
-- Process all available permission bits
            open permcursor2
            fetch next from permcursor2 into @@permbit, @@permstr
            while @@fetch_status <> -1
            begin
              if @@actmod & @@permbit > 0
              begin
-- Process "deny" permissions; these have priority over "grant" permissions
                if @@line = ''
                  set @@line = 'DENY'
                else
                  set @@line = @@line + ','
                set @@line = @@line + ' ' + @@permstr
                set @@linelen = @@linelen + 1
              end
              else if @@actadd & @@permbit > 0
              begin
-- Process "grant" permissions
                if @@line2 = ''
                  set @@line2 = 'GRANT'
                else
                  set @@line2 = @@line2 + ','
                set @@line2 = @@line2 + ' ' + @@permstr
              end
              fetch next from permcursor2 into @@permbit, @@permstr
            end
            close permcursor2
          end
          if @@line <> '' or @@line2 <> ''
          begin
            if @@linenum = 0
              exec sp_temp_Print
            set @@line3 = ' ON ' + @@objfullname + ' TO ' + @@_identquoteopen + @@usrname + @@_identquoteclose
            if @@line <> ''
            begin
              set @@line = @@line + @@line3
              exec sp_temp_Print @@line
            end
            if @@line2 <> ''
            begin
              set @@line2 = @@line2 + @@line3
              exec sp_temp_Print @@line2
            end
            set @@linenum = @@linenum + 1
          end
          fetch next from permcursor into @@usrname,
            @@actadd,
            @@actmod
        end
        if @@linenum <> 0
          exec sp_temp_Print @@_batchsep
        set @@line = ''
        close permcursor
        deallocate permcursor
        deallocate permcursor2
      end
      set @@line = ''
      if (@@objxtype = (select xtype
            from tbl_temp_ObjectTypes
            where typename = 'TABLE')
          and coalesce(@@_data, 0) <> 0)
        or (@@objxtype in (select xtype
            from tbl_temp_ObjectTypes
            where typename = 'VIEW')
          and coalesce(@@_data, 0) <> 0 and coalesce(@@_data, 0) % 10 <> 1)
      begin
-- Data
        exec @@index = sp_temp_ListTable @@objname, @@ownname, @@_insertbatchlength, @@_batchsep, @@identitycolumn, @@_data, @@_identformat, @@_identquoteopen, @@_identquoteclose
        if @@index = 0 and @@_data % 10 = 1
          exec sp_temp_Print @@_batchsep
      end
      set @@line = ''
    end
  end
  if @@phaseend <> 0
  begin
-- Unitialize phase upon end of phase
    if @@phasebegin <> 0
    begin
-- Close phase when at least one object has been processed
      if @@phase = 1
      begin
-- Drop objects
        close objcursor
        deallocate objcursor
      end
      else if @@phase = 2
      begin
-- Drop users
        close usrcursor
        deallocate usrcursor
      end
      else if @@phase = 3
      begin
-- Drop roles
        close usrcursor
        deallocate usrcursor
      end
      else if @@phase = 4
      begin
-- Create roles
        close usrcursor
        deallocate usrcursor
      end
      else if @@phase = 5
      begin
-- Create users
        close usrcursor
        deallocate usrcursor
      end
      else if @@phase = 6
      begin
-- Add users to roles as members
        close usrcursor
        deallocate usrcursor
      end
      else if @@phase = 7
      begin
-- Process objects
        close objcursor
        deallocate objcursor
      end
      else if @@phase <> 0
        set @@phase = -1
    end
-- Initialize next phase
    if @@_sepfiles = 0
    begin
      set @@phase = @@phase + 1
      set @@basephase = @@phase
    end
    else
    begin
-- Step to next base phase
      set @@basephase = case
-- Start with dropping and/or creating roles
          when @@basephase = 0 then case when coalesce(@@_drop, 0) = 0 then 4 else 3 end
-- Step to dropping and/or creating users
          when @@basephase in (3, 4) then case when coalesce(@@_drop, 0) = 0 then 5 else 2 end
-- Step to dropping and/or creating objects
          when @@basephase in (2, 5) then case when coalesce(@@_drop, 0) = 0 then 7 else 1 end
          else -1
        end
      set @@phase = @@basephase
    end
    set @@phasebegin = 0
    set @@phaseend = 0
    set @@phasenextobj = 0
    set @@linenum = -1
    set @@line = ''
-- Open phase
    if @@phase = 1
    begin
-- Drop objects
      if (coalesce(@@_tables, 0) <> 0 or coalesce(@@_views, 0) <> 0 or coalesce(@@_procedures, 0) <> 0)
        and coalesce(@@_drop, 0) <> 0
      begin
        set @@phasebegin = 1
-- Process objects in reverse order of object type and dependency level (for
--   single script) and forward order of object type and dependency level (for
--   one script for each object)
        set @@text = 'declare objcursor cursor for' + char(13) + char(10) +
          'select sysobjects.id,' + char(13) + char(10) +
          '    sysobjects.name,' + char(13) + char(10) +
          '    sysobjects.xtype,' + char(13) + char(10) +
          '    sysusers.name,' + char(13) + char(10) +
          '    typename,' + char(13) + char(10) +
          '    prop' + char(13) + char(10) +
          '  from tbl_temp_Dependency' + char(13) + char(10) +
          '    join sysobjects on tbl_temp_Dependency.id = sysobjects.id' + char(13) + char(10) +
          '    join sysusers on sysobjects.uid = sysusers.uid' + char(13) + char(10) +
          '    join tbl_temp_ObjectTypes on tbl_temp_Dependency.xtype = tbl_temp_ObjectTypes.xtype' + char(13) + char(10) +
          '  order by charindex(''' + @@_listsep + ''' + tbl_temp_Dependency.xtype + ''' + @@_listsep + ''', ''' + @@objxtypenames + ''')' + case when @@_sepfiles = 0 then ' desc' else '' end + ',' + char(13) + char(10) +
          '    tbl_temp_Dependency.deplevel' + case when @@_sepfiles = 0 then ' desc' else '' end + ',' + char(13) + char(10) +
          '    sysobjects.name'
        exec(@@text)
        open objcursor
      end
    end
    else if @@phase = 2
    begin
-- Drop users
      if coalesce(@@_drop, 0) <> 0 and coalesce(@@_users, 0) <> 0
      begin
        set @@phasebegin = 1
        declare usrcursor cursor for
          select sysusers.name,
              syslogins.name
            from sysusers
              join master..syslogins syslogins on sysusers.sid = syslogins.sid
            where uid > 1
              and uid not between 16382 and 16399
              and issqlrole = 0
              and hasdbaccess <> 0
              and sysusers.sid is not null
            order by sysusers.name
        open usrcursor
      end
    end
    else if @@phase = 3
    begin
-- Drop roles
      if coalesce(@@_drop, 0) <> 0 and coalesce(@@_users, 0) <> 0
      begin
        set @@phasebegin = 1
        declare usrcursor cursor for
          select name
            from sysusers
            where uid > 1
              and uid not between 16382 and 16399
              and issqlrole <> 0
              and uid = gid
            order by name
        open usrcursor
      end
    end
    else if @@phase = 4
    begin
-- Create roles
      if coalesce(@@_create, 0) <> 0 and coalesce(@@_roles, 0) <> 0
      begin
        set @@phasebegin = 1
        declare usrcursor cursor for
          select name
            from sysusers
            where uid > 1
              and uid not between 16382 and 16399
              and issqlrole <> 0
              and uid = gid
            order by name
        open usrcursor
      end
    end
    else if @@phase = 5
    begin
-- Create users
      if coalesce(@@_create, 0) <> 0 and coalesce(@@_users, 0) <> 0
      begin
        set @@phasebegin = 1
        declare usrcursor cursor for
          select sysusers.name,
              syslogins.name
            from sysusers
              join master..syslogins syslogins on sysusers.sid = syslogins.sid
            where uid > 1
              and uid not between 16382 and 16399
              and issqlrole = 0
              and hasdbaccess <> 0
              and sysusers.sid is not null
            order by sysusers.name
        open usrcursor
      end
    end
    else if @@phase = 6
    begin
-- Add users to roles as members
      if coalesce(@@_create, 0) <> 0 and coalesce(@@_users, 0) <> 0
      begin
        set @@phasebegin = 1
        declare usrcursor cursor for
          select sysusr1.name,
              sysusr2.name
            from sysusers sysusr1
              join sysmembers on memberuid = sysusr1.uid
              join sysusers sysusr2 on groupuid = sysusr2.uid
            where sysusr1.uid > 1
              and sysusr1.uid not between 16382 and 16399
              and sysusr1.issqlrole = 0
              and sysusr1.hasdbaccess <> 0
              and sysusr1.sid is not null
            order by sysusr1.name
        open usrcursor
      end
    end
    else if @@phase = 7
    begin
-- Process objects
      if (coalesce(@@_tables, 0) <> 0 or coalesce(@@_views, 0) <> 0 or coalesce(@@_procedures, 0) <> 0)
        and (coalesce(@@_create, 0) <> 0 or coalesce(@@_primarykey, 0) <> 0 or coalesce(@@_index, 0) <> 0
          or coalesce(@@_default, 0) <> 0 or coalesce(@@_foreignkey, 0) <> 0 or coalesce(@@_permission, 0) <> 0
          or coalesce(@@_data, 0) <> 0)
      begin
        set @@phasebegin = 1
-- Process objects in object type and dependency level order
        declare objcursor cursor for
          select sysobjects.id,
              sysobjects.name,
              sysobjects.xtype,
              sysusers.name,
              typename,
              prop
            from tbl_temp_Dependency
              join sysobjects on tbl_temp_Dependency.id = sysobjects.id
              join sysusers on sysobjects.uid = sysusers.uid
              join tbl_temp_ObjectTypes on tbl_temp_Dependency.xtype = tbl_temp_ObjectTypes.xtype
          order by charindex(@@_listsep + tbl_temp_Dependency.xtype + @@_listsep, @@objxtypenames),
            tbl_temp_Dependency.deplevel,
            sysobjects.name
        open objcursor
      end
    end
    else
      set @@phase = -1
-- End phase if it has not even begun
    if @@phasebegin = 0
      set @@phaseend = 1
    else
      set @@phasenextobj = 1
  end
-- Step to next phase if not at the end of phase but generating separate files
--   for each object
  else if @@_sepfiles <> 0
  begin
-- Do not continue with next object
    set @@phasenextobj = 0
    set @@linenum = 0
    if @@basephase in (1, 7)
    begin
-- Processing objects
      if @@phase = 1
-- From dropping object, step to creating object
        set @@phase = 7
      else
-- After creating object, continue with next object
        set @@phasenextobj = 1
    end
    else if @@basephase in (2, 5)
    begin
-- Processing users
      if @@phase = 2
-- From dropping user, step to creating user
        set @@phase = 5
      else
-- After creating user, continue with next user
        set @@phasenextobj = 1
    end
    else if @@basephase in (3, 4)
    begin
-- Processing roles
      if @@phase = 3
-- From dropping role, step to creating role
        set @@phase = 4
      else
-- After creating role, continue with next role
        set @@phasenextobj = 1
    end
    if @@phasenextobj <> 0
      set @@linenum = -1
  end
-- Continue phase with next object
  if @@phasenextobj <> 0
  begin
    if @@phase = 1
    begin
-- Drop objects
      fetch next from objcursor into @@objid,
        @@objname,
        @@objxtype,
        @@ownname,
        @@objxtypestr,
        @@objpropstr
    end
    else if @@phase = 2
    begin
-- Drop users
      fetch next from usrcursor into @@usrname,
        @@logname
    end
    else if @@phase = 3
    begin
-- Drop roles
      fetch next from usrcursor into @@grpname
    end
    else if @@phase = 4
    begin
-- Create roles
      fetch next from usrcursor into @@grpname
    end
    else if @@phase = 5
    begin
-- Create users
      fetch next from usrcursor into @@usrname,
        @@logname
    end
    else if @@phase = 6
    begin
-- Add users to roles as members
      fetch next from usrcursor into @@usrname,
        @@grpname
    end
    else if @@phase = 7
    begin
-- Process objects
      fetch next from objcursor into @@objid,
        @@objname,
        @@objxtype,
        @@ownname,
        @@objxtypestr,
        @@objpropstr
    end
    else
      set @@phase = -1
    set @@linenum = @@linenum + 1
-- End phase when out of objects
    if @@fetch_status = -1
      set @@phaseend = 1
-- Restore original phase when generating separate files for each object
    else if @@_sepfiles <> 0
    begin
      set @@phase = @@basephase
      if @@phaseend = 0
      begin
        if @@phase in (1, 7)
        begin
          set @@index = @@_identformat % 10
          exec sp_temp_ObjFullName @@objname, @@ownname, @@line output, @@index, @@_identquoteopen, @@_identquoteclose
          set @@line2 = @@objxtypestr
        end
        else if @@phase in (2, 5)
        begin
          set @@line = @@usrname
          set @@line2 = 'USER'
        end
        else if @@phase in (3, 4)
        begin
          set @@line = @@grpname
          set @@line2 = 'ROLE'
        end
        else
          set @@line = ''
        if @@line <> ''
        begin
          set @@line = '-- !NeWoBjEcT! Object: "' + @@line + '" Type: "' + lower(@@line2) + '"' 
          exec sp_temp_Print @@line
          set @@line = ''
        end
      end
    end
  end
end

-- Drop temporary objects
drop table tbl_temp_Dependency
drop table tbl_temp_ObjectTypes
drop table tbl_temp_Permissions
drop procedure sp_temp_decrypt_7
drop procedure sp_temp_Print
drop procedure sp_temp_FixInt
drop procedure sp_temp_FixString
drop procedure sp_temp_FixDate
drop procedure sp_temp_ListTable
drop procedure sp_temp_ObjFullName
GO
