;Ŀ
;                 Joe Forster/STA                 
;                                                 
;                   STARLFN.ASM                   
;                                                 
;         Star LFN - LFN Emulator for DOS         
;

		jumps				;Autodetect branch distance

CSeg		segment
		assume	cs:CSeg, ds:CSeg, ss:CSeg

;Compilation flags: comment out to disable, uncomment to enable
Redirect	equ	Yes			;Redirect some ordinary DOS
						; functions to their LFN
						; equivalents
KeepFullLoCase	equ	Yes			;Keep fully lowercase long
						; file names; along with
						; "Redirect", it may cause
						; DOS programs to create long
						; file names if they open
						; files using lowercase names
HideDataBase	equ	Yes			;Hide the data base file from
						; the directory
HiddenDBFile	equ	Yes			;Create the data base as a
						; hidden system file

;Compilation constants: change values at will
SearchRecNum	equ	16			;Maximum number of search
						; records available at a time,
						; increase if not enough for
						; some programs; not that
						; increasing this value raises
						; memory requirement rapidly

;Boolean constants
False		equ	0			;False
True		equ	1			;True

;Special characters
chTab		equ	9			;Tab
chLF		equ	10			;Line feed
chCR		equ	13			;Carriage return

;Critical error return codes
ceFail		equ	3			;Fail critical errors

;LFN directory entry flags (for byte at offset 0)
leDeleted	equ	80h			;LFN entry deleted flag
leLast		equ	40h			;Last LFN entry flag
leNumMask	equ	3Fh			;LFN entry number mask

;SFN directory entry values (for byte at offset 0)
seDeleted	equ	0E5h			;SFN entry deleted flag
seReplaceE5h	equ	05h			;SFN replacement for E5h char

;Directory entry attributes
faReadOnly	equ	01h			;Attribute of read-only files
faHidden	equ	02h			;Attribute of hidden files
faSystem	equ	04h			;Attribute of system files
faVolLabel	equ	08h			;Attribute of volume labels
faLFN		equ	0Fh			;Attribute of LFN entries
faDirectory	equ	10h			;Attribute of directories
faArchive	equ	20h			;Attribute of files to archive

;File search allowed attributes
faNormalFile	equ	37h			;Attribute of normal files
faAnyFile	equ	3Fh			;Attribute of any files

;File open modes
fmRead		equ	0			;Open file for read only
fmWrite		equ	1			;Open file for write only
fmReadWrite	equ	2			;Open file for read and write
fmErrorIfEx	equ	0			;Make error if file exists
fmOpenIfEx	equ	1			;Open file if exists
fmTruncateIfEx	equ	2			;Truncate file if exists
fmErrorIfNEx	equ	0			;Make error if file not exists
fmCreateIfNEx	equ	16			;Create file if not exists

;Installation command
icNone		equ	0			;Do nothing
icInstall	equ	1			;Install program
icUninstall	equ	2			;Uninstall program

;File attributes for data base files
	ifdef HiddenDBFile
DataBaseAttrib	equ	faArchive + faHidden + faSystem
	else
DataBaseAttrib	equ	faArchive
	endif

;Fake operating system version numbers for problem programs
DOSVerNum	equ	7			;Fake DOS version number
WinVerNum	equ	4			;Fake Windows version number

;Miscellaneous constants
MCBNameOffs	equ	8			;Offset of program name in MCB
EntryShift	equ	5			;Shift count for entry length
LFNCharPerEntry	equ	13			;LFN chars per directory entry

;Data structure sizes
SFNNameLen	equ	8			;Short name length
SFNExtLen	equ	3			;Short extension length
EntrySize	equ	32			;Length of directory entry
ShortNameLen	equ	128			;Length of short file names
LongNameLen	equ	255			;Length of long file names
MaxSectorSize	equ	512			;Maximum sector size
StackSize	equ	1024			;Stack size of resident module
DataBufferSize	equ	8192			;Size of data buffer
ProgramSize	equ	(8192 + (ResidentEnd - DataEnd))

;Internal search record
SearchRec	struc
srUsed		db	?			;Search record used flag
srPattern	db	LongNameLen + 1 dup (?)	;File name pattern
srAllowedAttr	db	?			;Allowed attributes
srMustAttr	db	?			;Required attributes
srProcess	dw	?			;Process ID of owner
srDirFound	db	?			;Directory found flag
srReserved	db	21 dup (?)		;Data reserved by DOS
srAttr		db	?			;File attributes
srDate		dd	?			;File date stamp
srSize		dd	?			;File size
srName		db	13 dup (?)		;File name
SearchRec	ends
SearchRecSize	equ	(type SearchRec)	;Internal search record size

;Windows long file name search record
LongSearchRec	struc
lsAttr		dd	?			;File attributes
lsCreateDate	dd	?, ?			;File creation date stamp
lsAccessDate	dd	?, ?			;File last access date stamp
lsModifyDate	dd	?, ?			;File modification date stamp
lsSizeHi	dd	?			;File size high double word
lsSize		dd	?			;File size
lsReserved	db	8 dup (?)		;Reserved data
lsName		db	260 dup (?)		;Long file name
lsShortName	db	14 dup (?)		;Short file name
LongSearchRec	ends

;DOS short file name search record
ShortSearchRec	struc
ssSearchDrive	db	?			;Search drive (bits 0-6)
						;Search file name pattern
ssSearchName	db	SFNNameLen + SFNExtLen dup (?)
ssSearchAttr	db	?			;Search allowed attributes
ssSearchEntry	dw	?			;Current offset into cluster
ssSearchCluster	dw	?			;Current directory cluster
ssReserved	db	4 dup (?)
ssAttr		db	?			;Current file attributes
ssDate		dd	?			;Current file date stamp
ssSize		dd	?			;Current file size
						;Current file name
ssName		db	SFNNameLen + SFNExtLen + 2 dup (?)
ShortSearchRec	ends

;Directory entry on the disk
DirEntry	struc
deName		db	SFNNameLen dup (?)	;File name
deExt		db	SFNExtLen dup (?)	;File extension
deAttr		db	?			;File attributes
deReserved1	db	?
deChecksum	db	?			;LFN checksum for short alias
deReserved2	db	8 dup (?)
deDate		dd	?			;File date stamp
deFirstCluster	dw	?			;First data cluster of file
deSize		dd	?			;File size
DirEntry	ends

;Device parameter block
DeviceParBlock	struc
dbSpecialFunc	db	?
dbDeviceType	db	?
dbDeviceAttr	dw	?
dbCylinderNum	dw	?
dbMediaType	db	?
;BIOS parameter block, part of the device parameter block
bbBIOSParBlock	label
bbBytePerSector	dw	?			;Number of bytes per sector
bbSectorPerClus	db	?			;Number of sectors per cluster
bbResSectorNum	dw	?			;Number of sectors before FAT
bbFATNum	db	?			;Number of FAT copies
bbRootDirLen	dw	?			;Number of root dir entries
bbSectorNum	dw	?			;Total number of sectors
bbMediaID	db	?			;Media descriptor byte
bbSectorPerFAT	db	?			;Number of sectors per FAT
bbSecPerTrack	dw	?			;Number of sectors per track
bbHeadNum	dw	?			;Number of heads
bbHiddenSecNum	dw	?			;Number of hidden sectors
bbExtSectorNum	dd	?			;Total number of sectors
bbReserved	db	6 dup (?)
bbCylinderNum	dw	?			;Number of cylinders
bbDeviceType	db	?			;Device type
bbDeviceAttr	db	?			;Device attributes
DeviceParBlock	ends

;Drive parameter block
DriveParBlock	struc
dpDrive		db	?			;Drive (0 for A: etc.)
dpUnit		db	?			;Unit into drive
dpBytePerSector	dw	?			;Number of bytes per sector
dpSectorPerClus	db	?			;Number of sectors per cluster
dpClusterShift	db	?			;Sectors per cluster shift
dpResSectorNum	dw	?			;Number of sectors before FAT
dpFATNum	db	?			;Number of FAT copies
dpRootDirLen	dw	?			;Number of root dir entries
dpFirstDataSec	dw	?			;First data sector
dpClusterNum	dw	?			;Number of clusters + 1
dpSectorPerFAT	dw	?			;Number of sectors per FAT
dpFirstDirSec	dw	?			;First root dir sector
dpDeviceHeader	dd	?			;Disk device header
dpMediaID	db	?			;Media descriptor byte
dpDiskAccessed	db	?			;Disk accessed flag
dpNextBlock	dd	?			;Next drive parameter block
dpFirstFreeClus	dw	?			;Start of free cluster search
dpFreeClusNum	dw	?			;Number of free clusters
DriveParBlock	ends
DrvParBlockSize	equ	(type DriveParBlock)

;Disk transfer block
DiskXferBlock	struc
dbFirstSector	dd	?			;First sector to transfer
dbSectorNum	dw	?			;Number of sectors to transfer
dbBuffer	dd	?			;Data buffer
DiskXferBlock	ends

;Current directory structure
CurDirStruc	struc
cdPath		db	67 dup (?)		;Real abs path of current dir
cdAttr		dw	?			;Drive attributes
cdDriveParBlock	dd	?			;Drive parameter block
cdCurDirCluster	dw	?			;First cluster of current dir
cdReserved1	db	4 dup (?)
cdRootDirLen	dw	?			;Length of root dir in path
cdDeviceType	db	?			;Device type for remote drives
cdRedirBlock	dd	?			;IFS/network redirector block
cdIFSData	dw	?			;Data area for IFS
CurDirStruc	ends
CurDirStrucSize	equ	(type CurDirStruc)

;Disk sector buffer
SectorBuffer	struc
		db	MaxSectorSize dup (?)
SectorBuffer	ends

;Command line buffer
		org	0080h
CommandLineLen	db	?			;Length of command line
CommandLine	db	127 dup (?)		;Command line data

;Entry point
		org	0100h
Main:		jmp	Start			;Skip resident module

;Get the address of a search record
;  Input : CX: search record number
;  Output: BX: offset of the search record
GetSearchAddr:	push	ax
		mov	ax, type SearchRec	;Multiply handle of search
		mul	cx			; record by the size of a
		add	ax, Offset SearchRecs	; search record and add the
		mov	bx, ax			; offset of search records
		pop	ax
		retn

;Free all search records belonging to a process
;  Input : AX: process ID; zero, if all search records have to be freed
DropSearch:	xor	cx, cx			;Begin with first record
@03_DropSearch:	call	GetSearchAddr		;Get offset of search record
		or	ax, ax			;Check process ID and do not
		je	@01_DropSearch		; free if does not match
		cmp	ax, [bx].SearchRec.srProcess
		jne	@02_DropSearch		;Free search record
@01_DropSearch:	mov	byte ptr [bx].SearchRec.srUsed, False
@02_DropSearch:	inc	cx			;Go to next search record
		cmp	cx, SearchRecNum	;Quit if no more records
		jb	@03_DropSearch
		retn

;Save original DOS swap area
SaveSwap:	mov	ah, 59h			;Get extended error info
		xor	bx, bx
		int	21h
		mov	word ptr cs:OrigDOSPars[12], ds
		push	cs			;Save registers into buffer
		pop	ds
		mov	OrigDOSPars[0], ax
		mov	OrigDOSPars[2], bx
		mov	OrigDOSPars[4], cx
		mov	OrigDOSPars[6], dx
		mov	OrigDOSPars[8], si
		mov	OrigDOSPars[10], di
		mov	OrigDOSPars[14], es
		retn

;Search for a string in a table of strings, ordered by string length
;  Input : DS:SI: the string table to search in
;          ES:DI: the string to search for
;          CX: the length of the string
;  Output: CF: if zero, the string has been found in the table
FindStr:	xor	ax, ax
@02_FindStr:	add	si, ax			;Go to next string
		cld
		lodsb				;Load length of string
		or	al, al			;End if no more strings
		stc
		jz	@01_FindStr
		cmp	cx, ax			;If the string searched for is
		jb	@01_FindStr		; shorter then not found; if
		ja	@02_FindStr		; longer, go to next string
		push	cx
		push	si
		push	di
		repe	cmpsb			;Compare the two strings
		pop	di
		pop	si
		pop	cx
		jne	@02_FindStr		;Go to next string if mismatch
		clc
@01_FindStr:	retn

;Check the name of the caller program
;  Output: CF: if zero, the name of the caller program was found in the list
SetVer:		push	ds			;Save original registers and
		push	cs			; setup own data segment
		pop	ds
		push	es
		push	si
		push	di
		push	ax
		push	bx
		push	cx
		mov	ah, 62h			;Get current process ID
		int	21h
		dec	bx			;Decrease process ID by one to
		mov	es, bx			; get segment of program's MCB
		mov	di, MCBNameOffs		;Program name is at offset 8
		push	di
		mov	cx, SFNNameLen		;Maximum name length is 8
		xor	al, al			;Compute the length of the
		cld				; program's name in the MCB
		repne	scasb
		jne	@01_SetVer
		inc	cx
@01_SetVer:	sub	cx, MCBNameOffs
		neg	cx
		mov	si, Offset SetverNames	;Search for the program's name
		pop	di			; in the table of known
		call	FindStr			; programs
		pop	cx			;Restore original registers
		pop	bx
		pop	ax
		pop	di
		pop	si
		pop	es
		pop	ds
		retn

;Setup environment for the resident core
InitCore:	mov	cs:OrigDS, ds		;Save original registers and
		push	cs			; setup own segment registers
		pop	ds
		mov	OrigES, es
		push	cs
		pop	es
		mov	OrigSI, si
		mov	OrigDI, di
		mov	OrigBP, bp
		mov	OrigAX, ax
		mov	OrigBX, bx
		mov	OrigCX, cx
		mov	OrigDX, dx
		inc	Active			;Resident module activated
		mov	ah, 62h			;Save original process ID
		int	21h
		mov	OrigProcessID, bx
		mov	ah, 50h			;Setup own process ID
		mov	bx, cs
		int	21h
		mov	ah, 2Fh			;Save original DTA
		int	21h
		mov	word ptr OrigDTA[0], bx
		mov	word ptr OrigDTA[2], es
		call	SaveSwap		;Save original DOS swap area
		mov	ax, 351Bh		;Save original Ctrl-Break
		int	21h			; interrupt
		mov	word ptr OldInt1B[0], bx
		mov	word ptr OldInt1B[2], es
		mov	ax, 3523h		;Save original Ctrl-Break exit
		int	21h			; interrupt
		mov	word ptr OldInt23[0], bx
		mov	word ptr OldInt23[2], es
		mov	ax, 3524h		;Save original critical error
		int	21h			; interrupt
		mov	word ptr OldInt24[0], bx
		mov	word ptr OldInt24[2], es
		mov	dx, Offset NewInt1B	;Setup own Ctrl-Break
		mov	ax, 251Bh		; interrupt
		int	21h
		mov	dx, Offset NewInt23	;Setup own Ctrl-Break exit
		mov	ax, 2523h		; interrupt
		int	21h
		mov	dx, Offset NewInt24	;Setup own critical error
		mov	ax, 2524h		; interrupt
		int	21h
		push	cs
		pop	es
		xor	al, al
		mov	RequestOK, al		;Clear request valid flag
		mov	CtrlBreak, al		;Clear Ctrl-Break pressed flag
		retn

;Restore the original environment
DoneCore:	lds	dx, cs:OldInt1B		;Restore original Ctrl-Break
		mov	ax, 251Bh		; interrupt
		int	21h
		lds	dx, cs:OldInt23		;Restore original Ctrl-Break
		mov	ax, 2523h		; exit interrupt
		int	21h
		lds	dx, cs:OldInt24		;Restore original critical
		mov	ax, 2524h		; error interrupt
		int	21h
		push	cs
		pop	ds
		mov	ax, 5D0Ah		;Restore original DOS swap
		xor	bx, bx			; area
		mov	dx, Offset OrigDOSPars
		int	21h
		mov	ah, 1Ah			;Restore original DTA
		lds	dx, cs:OrigDTA
		int	21h
		push	cs
		pop	ds
		mov	ah, 50h			;Restore original process ID
		mov	bx, OrigProcessID
		int	21h
		cmp	CtrlBreak, False	;If Ctrl-Break was pressed
		je	@01_DoneCore		; while the core was active,
		int	1Bh			; signal the original handler
@01_DoneCore:	dec	Active			;Resident module deactivated
		mov	ax, OrigAX		;Restore original registers
		mov	bx, OrigBX
		mov	cx, OrigCX
		mov	dx, OrigDX
		mov	si, OrigSI
		mov	di, OrigDI
		mov	bp, OrigBP
		mov	es, OrigES
		mov	ds, OrigDS
		cmp	cs:RequestOK, True	;If request invalid, set CF
		retn

;New INT21 (DOS dispatcher) interrupt
NewInt21:	cmp	cs:Active, False	;If resident module already
		jne	@04_NewInt21		; active then skip everything
		mov	cs:AbsOrigAX, ax	; Store absolute original AX
		cmp	ah, 30h			;Check for caller if DOS
		jne	@05_NewInt21		; version number requested
		call	SetVer			;Determine if current program
		jc	@05_NewInt21		; needs fake version numbers
		mov	ax, DOSVerNum		;Return a fake DOS version
		iret				; number of 7.0
@05_NewInt21:	mov	cs:LFNFuncCall, False	;Clear LFN function call flag
		cmp	ah, 71h			;If the function code belongs
		jne	@06_NewInt21		; the LFN function group then
		inc	cs:LFNFuncCall		; set LFN function call flag
@06_NewInt21:	cmp	ah, 3Ah			;Convert an ordinary file or
		je	@01_NewInt21		; directory delete or file
		cmp	ah, 41h			; rename into its long file
		je	@01_NewInt21		; name equivalent
		cmp	ah, 56h
	ifdef Redirect
		je	@01_NewInt21
		cmp	ah, 39h			;Redirect create and change
		jb	@02_NewInt21		; directory, create and open
		cmp	ah, 3Dh			; file, get/set file
		jbe	@01_NewInt21		; attributes and extended
		cmp	ah, 43h			; open file
		je	@01_NewInt21
		cmp	ah, 5Bh
		je	@01_NewInt21
		cmp	ah, 6Ch
	endif
		jne	@02_NewInt21
@01_NewInt21:	mov	al, ah			;Convert ordinary DOS function
		mov	ah, 71h			; call to its LFN equivalent
@02_NewInt21:	cmp	ah, 4Ch			;If a process exits then close
		jne	@03_NewInt21		; its search records
@07_NewInt21:	push	ds			;Save original registers and
		push	cs			; setup own data segment
		pop	ds
		push	ax
		push	bx
		push	cx
		mov	ah, 62h			;Get current process ID
		int	21h
		mov	ax, bx			;Free all search records that
		call	DropSearch		; belong to current process
		pop	cx			;Restore original registers
		pop	bx
		pop	ax
		pop	ds
@03_NewInt21:	cmp	ah, 71h			;Skip main module if ordinary
		jne	@04_NewInt21		; DOS function was called
		mov	cs:OrigSS, ss		;Save original stack
		mov	cs:OrigSP, sp
		cli
		push	cs			;Setup own stack
		pop	ss
		mov	sp, Offset TSRStackEnd
		sti
		call	InitCore		;Setup own environment
		call	Core			;Call own function dispatcher
		call	DoneCore		;Restore original environment
		cli
		mov	ss, cs:OrigSS		;Restore original stack
		mov	sp, cs:OrigSP
		sti
		retf	2
@04_NewInt21:	jmp	cs:[OldInt21]		;Call original DOS dispatcher

;New INT1B (Ctrl-Break) interrupt
NewInt1B:	mov	cs:CtrlBreak, True	;Set Ctrl-Break pressed flag
		iret

;New INT23 (Ctrl-Break exit) interrupt
NewInt23:	iret

;New INT24 (critical error) interrupt
;  Output : 3, fail all critical errors
NewInt24:	mov	ax, ceFail
		iret

;New INT2F (multiplex) interrupt
NewInt2F:	cmp	ax, 0
		jne	@01_NewInt2F
		cmp	bx, 'ST'		;Check for signature
		jne	@01_NewInt2F
		cmp	cx, 'AL'
		jne	@01_NewInt2F
		cmp	dx, 'FN'
		jne	@01_NewInt2F
		mov	al, 0FFh		;Acknowledge call and return
		mov	bx, cs			; code segment of resident
		mov	cx, 'OK'		; module
		iret
@01_NewInt2F:	cmp	ax, 1600h		;Check for caller if Windows
		jne	@02_NewInt2F		; version number requested
		call	SetVer			;Determine if current program
		jc	@02_NewInt2F		; needs fake version numbers
		mov	ax, WinVerNum		;Return a fake Windows version
		iret				; number of 4.0
@02_NewInt2F:	jmp	cs:[OldInt2F]

;Convert a character into uppercase
;  Input : AL: the original character
;  Output: AL: the uppercase character
UpCase:		cmp	al, 'a'			;If the character is between
		jb	@01_UpCase		; lowercase 'a' and 'z' then
		cmp	al, 'z'			; turn it into uppercase
		ja	@01_UpCase
		sub	al, 'a' - 'A'
@01_UpCase:	retn

;Convert a character into lowercase
;  Input : AL: the original character
;  Output: AL: the lowercase character
LoCase:		cmp	al, 'A'			;If the character is between
		jb	@01_LoCase		; uppercase 'a' and 'z' then
		cmp	al, 'Z'			; turn it into lowercase
		ja	@01_LoCase
		add	al, 'a' - 'A'
@01_LoCase:	retn

;Convert a string into uppercase
;  Input : DS:SI: the original string
;  Output: ES:DI: the destination string
UpperCase:	cld
@01_UpperCase:	lodsb				;Fetch a character from source
		call	UpCase			; string, turn into uppercase,
		stosb				; put into destination string,
		or	al, al			; go to next character if the
		jne	@01_UpperCase		; end of source not reached
		retn

;Compute length of string
;  Input : DS:SI: the string
;  Output: AX: length of string
StrLen:		push	si
		xor	ah, ah
		cld
@01_StrLen:	inc	ah			;Increase length counter
		lodsb				;Fetch a character and go to
		or	al, al			; next if the end of the
		jne	@01_StrLen		; string was not reached
		dec	ah			;Decrease length counter and
		xchg	al, ah			; put to low byte
		pop	si
		retn

;Copy a string into another
;  Input : DS:SI: the source string
;          ES:DI: the destination string
StrCopy:	cld
@01_StrCopy:	lodsb				;Copy a character and go to
		stosb				; the next character if the
		or	al, al			; end of the source string was
		jne	@01_StrCopy		; not reached
		retn

;Copy part of a string into another
;  Input : DS:SI: the source string
;          ES:DI: the destination string
;          CX: number of characters to copy
StrNCopy:	cld
		jcxz	@03_StrNCopy		;End if no characters to copy
@02_StrNCopy:	lodsb				;Copy a character
		stosb
		or	al, al			;End if the end of the source
		je	@01_StrNCopy		; string was reached
		loop	@02_StrNCopy		;End if no more characters
@03_StrNCopy:	xor	al, al			;End destination string
		stosb
@01_StrNCopy:	retn

;Compare two strings in case insensitive mode
;  Input : DS:SI: the first string
;          DS:DI: the second string
;  Output: ZF: when not zero, the two strings match
StrComp:	mov	al, [di]		;Fetch character from second
		call	UpCase			; string and turn to uppercase
		mov	ah, al
		mov	al, [si]		;Fetch character from first
		call	UpCase			; string and turn to uppercase
		cmp	al, ah			;End if no match
		jne	@01_StrComp
		or	al, al			;End if the end of both
		je	@01_StrComp		; strings was reached
		inc	si			;Go to next characters
		inc	di
		jmp	StrComp
@01_StrComp:	retn

;Search for the leftmost instance of a character in a string
;  Input : DS:SI: the string
;          AL: the character to search for
;  Output: AX: position of rightmost instance
;          ZF: when not zero, an instance was found
StrPos:		push	si
		push	di
		push	cx
		push	ax
		call	StrLen			;Fetch length of string
		mov	cx, ax
		mov	di, si
		pop	ax
		cld				;Search for the character
		repne	scasb			; forwards
		pushf
		jne	@01_StrPos
		mov	cx, di			;Compute position if found
		sub	cx, si
		dec	cx
@01_StrPos:	mov	ax, cx
		popf
		pop	cx
		pop	di
		pop	si
		retn

;Search for the rightmost instance of a character in a string
;  Input : DS:SI: the string
;          AL: character to search for
;  Output: AX: position of rightmost instance
;          ZF: when not zero, an instance was found
StrRPos:	push	si
		push	di
		push	cx
		push	ax
		call	StrLen			;Fetch length of string and go
		mov	cx, ax			; to its end
		mov	di, si
		add	di, ax
		dec	di
		pop	ax
		std				;Search for the character
		repne	scasb			; backwards
		pushf
		je	@01_StrRPos
		dec	cx			;Correct position if found
@01_StrRPos:	mov	ax, cx
		popf
		pop	cx
		pop	di
		pop	si
		retn

;Determine if a character is a slash or a backslash
;  Input : AL: the character
;  Output: ZF: when not zero, the character is a slash or a backslash
IsSlash:	cmp	al, '\'
		je	@01_IsSlash
		cmp	al, '/'
@01_IsSlash:	retn

;Append a file name to a path
;  Input : DS:SI: the original path
;          DS:BX: the original file name
;          ES:DI: the destination full file name
AddToPath:	cmp	byte ptr [si], 0
		je	@01_AddToPath
		call	StrCopy			;If path is not empty then
		dec	di			; copy it into destination
		mov	al, [di][-1]		;If the last character of the
		call	IsSlash			; path is no slash, backslash
		je	@01_AddToPath		; or colon then append a
		cmp	al, ':'			; backslash to the path
		je	@01_AddToPath
		mov	al, '\'
		stosb
@01_AddToPath:	mov	si, bx			;Append the original file name
		jmp	StrCopy			; to the path

;Get the path of a file name
;  Input : DS:SI: the original file name
;          ES:DI: the destination path
GetPath:	push	bx
		mov	al, '\'			;Search for the rightmost
		call	StrRPos			; slash or backslash
		je	@03_GetPath
		mov	al, '/'
		call	StrRPos
@03_GetPath:	mov	bx, ax			;Skip slash or backslash, if
		pushf				; found
		inc	bx
		popf				;Skip check for drive letter
		je	@01_GetPath		; if either slash was found
		xor	bx, bx
		call	StrLen
		cmp	ax, 1			;If the file name is at least
		jb	@01_GetPath		; two characters long and the
		cmp	byte ptr [si][1], ':'	; second character is a colon
		jne	@01_GetPath		; then it is a drive letter
		mov	bx, 2
@01_GetPath:	cmp	bx, 2
		jbe	@02_GetPath
		mov	al, [si][bx][-1]	;Skip last slash or backslash
		call	IsSlash			; if it does not resemble the
		jne	@02_GetPath		; root directory
		cmp	byte ptr [si][bx][-2], ':'
		je	@02_GetPath
		dec	bx
@02_GetPath:	mov	cx, bx			;Copy first part of file name,
		pop	bx			; without the last component
		jmp	StrNCopy

;Cut the path off a file name
;  Input : DS:SI: the original file name
;          ES:DI: the destination file name
CutPath:	push	bx
		mov	al, '\'			;Search for the rightmost
		call	StrRPos			; slash or backslash
		je	@02_CutPath
		mov	al, '/'
		call	StrRPos
@02_CutPath:	mov	bx, ax			;Skip slash or backslash, if
		pushf				; found
		inc	bx
		popf				;Skip check for drive letter
		je	@01_CutPath		; if either slash was found
		xor	bx, bx
		call	StrLen
		cmp	ax, 1			;If the file name is at least
		jb	@01_CutPath		; two characters long and the
		cmp	byte ptr [si][1], ':'	; second character is a colon
		jne	@01_CutPath		; then it is a drive letter
		mov	bx, 2
@01_CutPath:	add	si, bx			;Skip path and copy the file
		pop	bx			; name itself only
		jmp	StrCopy

;Convert Unicode character to ASCII
;  Input : AX: the Unicode character
;  Output: ZF: when not zero, the end of the string has been reached
;          AL: the ASCII character
Unicode:	or	ah, ah			;If upper byte of Unicode
		je	@01_Unicode		; character is not zero then
		mov	al, '_'			; convert to an underscore
@01_Unicode:	or	al, al			;Determine if end of string
		retn

;Go to the next LFN character in the directory entry
;  Input : CL: the current offset into the directory entry
;          DS:SI: the current character
;  Output: CF: when not zero, the end of the directory entry has been reached
;          CL: the new offset into the directory entry
;          DS:SI: the next character
NextLFNChr:	add	si, 2
		add	cl, 2			;Increase offset into entry
		cmp	cl, 11			;Skip three bytes at the
		jne	@01_NextLFNChr		; offset eleven
		add	si, 3
		add	cl, 3
		jmp	@02_NextLFNChr
@01_NextLFNChr:	cmp	cl, 26			;Skip two bytes at offset 26
		jne	@02_NextLFNChr
		add	si, 2
		add	cl, 2
@02_NextLFNChr:	cmp	cl, EntrySize		;Determine if last character
		cmc
		retn

;Get the physical parameters of a drive
;  Input : AL: the drive to scan (0 for A: etc.)
;  Output: CF: if zero, the drive was found
DrivePars:	cmp	al, LastDrive		;Exit if drive is invalid
		stc
		jae	@01_DrivePars
		mov	ah, CurDirStrucSize
		mul	ah
		lds	si, CurDirStrucs	;Go to the current directory
		add	si, ax			; structure of the drive
		lds	si, [si].CurDirStruc.cdDriveParBlock
		mov	di, Offset DriveParams	;Copy drive parameter block
		mov	cx, DrvParBlockSize	; into own data segment
		cld
		rep	movsb
		push	cs			;Reload data segment
		pop	ds
                clc
@01_DrivePars:	retn
;!!! DEBUG !!!
;DrivePars2:	mov	dl, al			;Get drive parameter block
;		inc	dl
;		mov	ah, 32h
;		int	21h
;		mov	si, bx			;Copy drive parameter block
;		mov	di, Offset DriveParams	; into own data segment
;		mov	cx, DrvParBlockSize
;		cld
;		rep	movsb
;		push	cs			;Reload data segment
;		pop	ds
;		clc
;		retn
;DrivePars3:	mov	bl, al
;		inc	bl
;		mov	ax, 440Dh		;Ask the device driver for
;		mov	cx, 0860h		; the device parameter block
;		mov	dx, Offset DeviceParams
;		int	21h
;		jc	@01_DrivePars
;		push	bp
;		mov	si, dx
;		add	si, DeviceParBlock.bbBIOSParBlock
;		mov	bp, Offset DriveParams	;Convert BIOS parameter block
;		mov	ah, 53h			; into drive parameter block
;		int	21h
;		pop	bp
;		clc
;@01_DrivePars:	retn
;!!! DEBUG !!!

;Calculate the absolute sector number of the first sector of a cluster
;  Input : AX: the cluster number
;  Output: DX:AX: the absolute sector number of the first sector
AbsSector:	xor	dx, dx			;Zero high word of number
		cmp	ax, 2			;If cluster number is below
		jae	@01_AbsSector		; two then root directory
		mov	ax, DriveParams.dpFirstDirSec
		jmp	@02_AbsSector		;Fetch first root dir sector
@01_AbsSector:	sub	ax, 2			;Subtract two from cluster
		mov	cl, DriveParams.dpClusterShift
		xor	ch, ch
		jcxz	@03_AbsSector
@04_AbsSector:	shl	ax, 1			;Shift the cluster number to
		rcl	dx, 1			; get number of sectors
		loop	@04_AbsSector
@03_AbsSector:	add	ax, DriveParams.dpFirstDataSec
		adc	dx, 0			;Add first data sector
@02_AbsSector:	retn

;Read a FAT entry from the disk
;  Input : AX: the FAT entry number
;  Output: CF: when not zero, an error occured or the FAT chain ended
;          AX: the FAT entry
ReadFAT:	xor	bl, bl			;Clear 12-bit FAT flag
		xor	dx, dx			;Clear upper word of number
		cmp	DriveParams.dpClusterNum, 0FF6h
		ja	@01_ReadFAT		;Skip if 16-bit FAT
		inc	bl			;Set 12-bit FAT flag
		mov	cx, 3			;Compute offset of 12-bit
		mul	cx			; entry into FAT sector
		xor	bh, bh			;Multiply FAT entry number by
		shr	dx, 1			; three, then divide it by two
		rcr	ax, 1			; set second 12-bit flag
		rcl	bh, 1			; according to remainder
		jmp	@02_ReadFAT
@01_ReadFAT:	xor	dx, dx			;Compute offset of 16-bit
		shl	ax, 1			; entry into FAT sector
		rcl	dx, 1
		xor	bh, bh			;Clear second 12-bit flag
@02_ReadFAT:	div	DriveParams.dpBytePerSector
		push	bx
		push	dx			;Compute absolute sector
		xor	dx, dx			; number of FAT sector
		add	ax, DriveParams.dpResSectorNum
		adc	dx, 0
		mov	word ptr FATAbsSector[0], ax
		mov	word ptr FATAbsSector[2], dx
		call	ReadFATSector
		pop	si
		pop	bx
		jc	@03_ReadFAT
		mov	ax, word ptr FATBuffer[si]
		or	bl, bl			;Skip 12-bit FAT conversion
		je	@04_ReadFAT		; if 16-bit FAT
		or	bh, bh			;If the FAT entry is in the
		jne	@05_ReadFAT		; first 12 bit then strip off
		and	ax, 0FFFh		; the high four bits
		jmp	@06_ReadFAT
@05_ReadFAT:	inc	si			;Go to second half of entry
		cmp	si, DriveParams.dpBytePerSector
		jb	@07_ReadFAT		;If beyond sector boundary
		push	ax			; then read next FAT sector
		inc	word ptr FATAbsSector[0]
		jne	@08_ReadFAT
		inc	word ptr FATAbsSector[2]
@08_ReadFAT:	call	ReadFATSector
		pop	ax
		jc	@03_ReadFAT
		mov	ah, byte ptr FATBuffer[0]
@07_ReadFAT:	shr	ax, 4			;Shift entry into place
@06_ReadFAT:	cmp	ax, 0FF0h		;End if the the FAT chain
		jb	@04_ReadFAT		; ended or a bad cluster was
		or	ax, 0F000h		; encountered
@04_ReadFAT:	or	ax, ax			;Determine if the FAT chain
		stc				; ended or a bad cluster was
		je	@03_ReadFAT		; encountered
		cmp	ax, 0FFF0h
		cmc
@03_ReadFAT:	retn

;Save the cluster number of the directory cluster and compute the absolute
;  sector number of its first sector
;  Input : AX: the cluster number of the directory
SaveDirCluster:	mov	DirCluster, ax
		call	AbsSector
		mov	word ptr DirAbsSector[0], ax
		mov	word ptr DirAbsSector[2], dx
		retn

;Initialize transfer block
;  Input : CX:BX: number of sector to transfer
;          DS:DX: the sector buffer
;  Output: DS:BX: the initialized transfer block
;          CX: -1 for extended disk transfer
InitXferBlock:	mov	word ptr TransferBlock.dbFirstSector[0], bx
		mov	word ptr TransferBlock.dbFirstSector[2], cx
		mov	TransferBlock.dbSectorNum, 1
		mov	word ptr TransferBlock.dbBuffer[0], dx
		mov	word ptr TransferBlock.dbBuffer[2], cs
		mov	cx, -1
		mov	bx, Offset TransferBlock
		retn

;Read a sector from the disk
;  Input : AL: the drive to read from (0 for A: etc.)
;          CX:BX: number of sector to read
;          DS:DX: the sector buffer
;  Output: CF: when not zero, an error occured
ReadSector:	call	InitXferBlock		;Initialize transfer block
		int	25h
		add	sp, 2
		retn

;Read a FAT sector from the disk
;  Output: CF: when not zero, an error occured
ReadFATSector:	mov	al, LFNDrive
		mov	bx, word ptr FATAbsSector[0]
		mov	cx, word ptr FATAbsSector[2]
		mov	dx, Offset FATBuffer
		jmp	ReadSector

;Read a directory sector from the disk
;  Output: CF: when not zero, an error occured
ReadDirSector:	mov	al, LFNDrive
		mov	bx, word ptr DirAbsSector[0]
		mov	cx, word ptr DirAbsSector[2]
		mov	dx, Offset DirBuffer
		jmp	ReadSector

;Read the next directory entry from the disk
;  Output: CF: when not zero, an error occured or the directory ended
;          DS:BX: the entry
NextDir:	mov	ax, DirOffset		;Fetch current offset into
		add	ax, EntrySize		; directory sector and add
		mov	DirOffset, ax		; size of directory entry
		cmp	ax, DriveParams.dpBytePerSector
		jb	@02_NextDir		;Skip if still the same sector
@01_NextDir:	mov	DirOffset, 0		;Zero out offset into sector
		inc	word ptr DirAbsSector[0]
		jne	@05_NextDir		;Increase sector number
		inc	word ptr DirAbsSector[2]
@05_NextDir:	mov	ax, DirSector		;Increase number of sectors
		inc	ax			; into cluster
		mov	DirSector, ax
		je	@03_NextDir
		cmp	DirCluster, 2		;Check root directory in a
		jae	@07_NextDir		; different way
		add	ax, DriveParams.dpFirstDirSec
		cmp	ax, DriveParams.dpFirstDataSec
		jb	@04_NextDir
		stc				;End if last root directory
		jmp	@06_NextDir		; sector was left
@07_NextDir:	cmp	al, DriveParams.dpSectorPerClus
		jbe	@04_NextDir		;Read sector if same cluster
@03_NextDir:	mov	DirSector, 0		;Zero out sectors into cluster
		mov	ax, DirCluster		;Search for the next cluster
		call	ReadFAT			; of the directory in the FAT
		jc	@06_NextDir
		call	SaveDirCluster		;Save directory cluster
@04_NextDir:	call	ReadDirSector		;Read the directory sector and
		jc	@06_NextDir		; end if an error occured
@02_NextDir:	mov	bx, Offset DirBuffer	;Return the offset of the
		add	bx, DirOffset		; directory entry
		cmp	byte ptr [bx], 1	;Determine if last entry
@06_NextDir:	retn

;Copy part of the short file name
;  Input : AH: when not zero, the extension is being copied
;          CX: the maximum length of the file name part
;          DS:SI: the source file name part
;          DS:DI: the destination file name
CopySFN:	cld
		cmp	byte ptr [si], ' '	;If the file name part is not
		je	@01_CopySFN		; empty and it is the extension
		or	ah, ah			; then prepend a dot
		je	@01_CopySFN
		mov	al, '.'
		stosb
@01_CopySFN:	lodsb
		cmp	al, ' '
		je	@02_CopySFN
		stosb
		loop	@01_CopySFN
@02_CopySFN:	or	ah, ah			;If the extension was copied
		je	@03_CopySFN		; then end the string
		xor	al, al
		stosb
@03_CopySFN:	retn

;Read an entry from the disk into the current short and long file name
;  Output: CF: when not zero, an error occured or the directory ended
ReadDisk:	xor	al, al			;Invalidate LFN entries and
		mov	LFNEntriesOK, al	; long file name
		mov	LongFName[0], al
@09_ReadDisk:	call	NextDir			;Read next directory entry
		jnc	@14_ReadDisk
		inc	EndOfDir
		jmp	@04_ReadDisk
@14_ReadDisk:	cmp	byte ptr [bx].DirEntry.deAttr, faLFN
		jne	@01_ReadDisk		;Skip if normal entry
		mov	al, byte ptr [bx].DirEntry.deChecksum
		mov	cl, byte ptr [bx].DirEntry.deName[0]
		test	cl, leDeleted		;Skip LFN entry if deleted bit
		jne	@09_ReadDisk		; is set
		and	cl, not leDeleted
		xor	ch, ch			;Clear last LFN entry flag
		test	cl, leLast		;Skip if not last (physically
		je	@02_ReadDisk		; first) LFN entry
		and	cl, leNumMask		;Strip flags
		mov	LFNEntryNum, cl		;Store number of LFN entries
		mov	LFNChecksum, al		;Store checksum
		mov	LFNEntriesOK, True	;Set LFN entries correct flag
		inc	ch
		jmp	@03_ReadDisk
@02_ReadDisk:	cmp	LFNEntriesOK, False	;Skip all subsequent LFN
		je	@09_ReadDisk		; entries if an error occured
		cmp	LFNChecksum, al		;Compare checksum against the
		je	@03_ReadDisk		; previous one and invalidate
@11_ReadDisk:	mov	LFNEntriesOK, False	; entries if mismatch
		jmp	@09_ReadDisk
@03_ReadDisk:	mov	al, LFNEntryNum		;Fetch number of LFN entries
		or	al, al			;If there should be no more
		je	@11_ReadDisk		; LFN entries then invalidate
		cmp	al, cl			;If LFN entries in incorrect
		jne	@11_ReadDisk		; order then invalidate
		dec	LFNEntryNum		;Decrease num of LFN entries
		dec	cl			;Fetch number of LFN part
		mov	al, LFNCharPerEntry	;Convert number of LFN part
		mul	cl			; into offset into long file
		mov	di, Offset LongFName	; name
		add	di, ax
		mov	si, bx			;Go to first character
		inc	si
		mov	cl, 1			;Initialize offset into entry
		cld
@06_ReadDisk:	mov	ax, [si]		;Read Unicode character
		call	Unicode			;Convert character to ASCII
		je	@05_ReadDisk		;End if end of string
		stosb				;Store ASCII character
		call	NextLFNChr		;Go to the next character
		jnc	@06_ReadDisk
		or	ch, ch			;If this is the last entry
		je	@09_ReadDisk		; then end long file name
		jmp	@13_ReadDisk
@05_ReadDisk:	or	ch, ch			;If this is not the last entry
		je	@11_ReadDisk		; then invalidate LFN entries,
@13_ReadDisk:	xor	al, al			;End long file name and go to
		stosb				; the next entry
		jmp	@09_ReadDisk
@01_ReadDisk:	cmp	LFNEntriesOK, False	;Skip if there were no
		je	@07_ReadDisk		; preceding LFN entries
		mov	si, bx
		mov	cx, SFNNameLen + SFNExtLen
		xor	ah, ah			;Compute checksum for the
		cld				; short file name
@10_ReadDisk:	ror	ah, 1			;Rotate checksum one bit to
		lodsb				; the right and then add the
		add	ah, al			; current character of the
		loop	@10_ReadDisk		; short file name
		cmp	ah, LFNChecksum		;Invalidate LFN entries if the
		je	@07_ReadDisk		; checksum does not match
		mov	LFNEntriesOK, False
@07_ReadDisk:	mov	si, bx			;Save first cluster of file
		mov	ax, [si].DirEntry.deFirstCluster
		mov	FileCluster, ax
		mov	cx, SFNNameLen		;Copy the name part of the
		mov	di, Offset ShortFName	; short file name, maximum 8
		xor	ah, ah			; chars
		call	CopySFN
		mov	cx, SFNExtLen		;Copy the extension part of
		lea	si, [bx].DirEntry.deExt	; the short file name, maximum
		mov	ah, True		; 3 chars
		call	CopySFN
		cmp	LFNEntriesOK, False	;If the LFN entries are
		jne	@12_ReadDisk		; invalid then duplicate the
		mov	si, Offset ShortFName	; short file name
		mov	di, Offset LongFName
		call	StrCopy
@12_ReadDisk:	mov	al, byte ptr ShortFName[0]
		cmp	al, seDeleted		;Skip deleted entries
		je	ReadDisk
		cmp	al, seReplaceE5h	;Fix valid entries with a
		jne	@08_ReadDisk		; first character of E5h
		mov	byte ptr ShortFName[0], seDeleted
@08_ReadDisk:	mov	al, [bx].DirEntry.deAttr
		test	al, faVolLabel		;Skip volume labels
		jne	ReadDisk
		clc
@04_ReadDisk:	retn

;Read a data byte from the data base
;  Input : BX: data base file handle
;  Output: CF: when not zero, an error occured
;          AL: the current data byte
ReadData:	mov	si, DataPos		;Fetch position in data buffer
		cmp	si, DataLen
		jb	@01_ReadData
		push	cx			;If the end of the buffer was
		push	dx			; reached then read in next
		mov	dx, Offset DataBuffer	; chunk of data
		mov	cx, DataBufferSize
		mov	ah, 3Fh
		int	21h
		pop	dx
		pop	cx
		jc	@02_ReadData		;End if an error occured or
		or	ax, ax			; the number of bytes read is
		jne	@03_ReadData		; zero
		inc	EndOfDir
		stc
		jmp	@02_ReadData
@03_ReadData:	mov	DataLen, ax		;Store number of bytes
		xor	si, si			;Go back to the beginning of
		mov	DataPos, si		; buffer
@01_ReadData:	mov	al, DataBuffer[si]	;Read one character
		inc	DataPos			;Go to next character
		inc	word ptr LineEnd[0]	;Increase position in data
		jne	@04_ReadData		; base file
		inc	word ptr LineEnd[2]
@04_ReadData:	clc
@02_ReadData:	retn

;Go back to the previous data byte
PrevData:	dec	DataPos			;Go back to previous character
		cmp	word ptr LineEnd[0], 0	;Decrease position in data
		jne	@01_PrevData		; base file
		dec	word ptr LineEnd[2]
@01_PrevData:	dec	word ptr LineEnd[0]
		retn

;Determine if a character is a white space
;  Input : AL: the character
;  Output: ZF: when not zero, the character is a white space
IsSpace:	cmp	al, ' '			;If the character is a space
		je	@01_IsSpace		; or a Tab then it is a white
		cmp	al, chTab		; space
@01_IsSpace:	retn

;Determine if a character is an end-of-line mark
;  Input : AL: the character
;  Output: ZF: when not zero, the character is an end-of-line mark
IsEOL:		cmp	al, chCR		;If the character is a CR or a
		je	@01_IsEOL		; LF then it is and end of
		cmp	al, chLF		; line
@01_IsEOL:	retn

;Read an entry from the data base into the current short and long file name
ReadDB:		mov	ax, word ptr LineEnd[0]	;The end of previous line is
		mov	cx, word ptr LineEnd[2]	; the start of current one
		mov	word ptr LineStart[0], ax
		mov	word ptr LineStart[2], cx
@06_ReadDB:	call	ReadData		;Read one character
		jc	@01_ReadDB		;If error then abort
		call	IsEOL			;If end of line, read again
		je	@06_ReadDB
		call	PrevData		;Go back to previous character
		mov	cx, ShortNameLen	;Begin with short file name
		mov	di, Offset ShortFName
@03_ReadDB:	call	ReadData		;Read one character
		jc	@01_ReadDB		;If error then abort
		call	IsEOL			;If end of line then abort
		je	@01_ReadDB
		call	IsSpace			;If white space then go to
		je	@02_ReadDB		; long file name
		mov	[di], al		;Store character
		inc	di			;Read next character
		loop	@03_ReadDB		;End if file name too long
@02_ReadDB:	call	ReadData		;Read one character
		jc	@01_ReadDB		;If error then abort
		call	IsSpace			;If white space, read again
		je	@02_ReadDB
		call	PrevData		;Go back to previous character
		xor	al, al			;End short file name
		mov	[di], al
		mov	cx, LongNameLen		;Begin with long file name
		mov	di, Offset LongFName
@05_ReadDB:	call	ReadData		;Read one character
		jc	@04_ReadDB		;If error then end file name
		call	IsEOL			;If end of line then end file
		je	@04_ReadDB		; name
		cmp	al, '"'			;Skip quotation marks
		je	@05_ReadDB
		mov	[di], al		;Store character
		inc	di			;Read next character
		loop	@05_ReadDB		;End if file name too long
@04_ReadDB:	call	ReadData		;Read one character
		jc	@07_ReadDB		;If error then end file name
		call	IsEOL			;If end of line then end file
		je	@04_ReadDB		; name
		call	PrevData		;Go back to previous character
@07_ReadDB:	xor	al, al			;End long file name
		mov	[di], al
@01_ReadDB:	retn

;Read a directory entry into the current short and long file name
ReadEntry:	cmp	VFATSupport, False
		je	@01_ReadEntry
		jmp	ReadDisk
@01_ReadEntry:	jmp	ReadDB

;Initialize the disk directory
;  Input : DS:SI: the full file name
;  Output: CF: when not zero, an error occured
InitDisk:	mov	di, Offset DataFullName	;Get the path of the parent
		push	di			; directory of the file
		call	GetPath
		pop	si
;!!! DEBUG !!!
;		xor	bx, bx			;Path starts at first char
;		mov	ax, word ptr [si][0]	;Get first two characters
;		call	UpCase
;		cmp	al, 'A'			;If first two characters
;		jb	@02_InitDisk		; resemble a drive then skip
;		cmp	al, 'Z'			; it
;		ja	@02_InitDisk
;		cmp	ah, ':'
;		jne	@02_InitDisk
;		add	bx, 2			;Path starts at third char
;@02_InitDisk:	mov	al, byte ptr [si][bx]	;Get first path character, if
;		call	IsSlash			; it is a slash then an
;		jne	@03_InitDisk		; absolute path was specified,
;		inc	bx			; keep the slash, too
;@03_InitDisk:	mov	byte ptr [si][bx], 0	; of the base directory
;!!! DEBUG !!!
		mov	ah, 2Fh			;Save original DTA
		int	21h
		mov	word ptr OrigDTA2[0], bx
		mov	word ptr OrigDTA2[2], es
		push	ds			;Restore extra segment
		pop	es
		mov	dx, Offset SFNSearch	;Switch to own DTA
		mov	ah, 1Ah
		int	21h
		mov	di, si			;Append '*.*' to path
		mov	bx, Offset AllFiles
		push	di
		call	AddToPath
		pop	dx
		mov	ah, 4Eh			;Search into the directory
		mov	cx, faNormalFile	; to get directory information
		int	21h
		pushf
		push	ds
		lds	dx, OrigDTA2		;Restore original DTA
		mov	ah, 1Ah
		int	21h
		pop	ds
		popf
		jc	@01_InitDisk		;Exit if an error occurred
		mov	al, SFNSearch.ssSearchDrive
		and	al, 7Fh			;Save physical search drive
		dec	al
		cmp	al, LFNDrive		;If still on the same physical
		je	@02_InitDisk		; drive then skip parameters
		mov	LFNDrive, al		;Get parameters of drive and
		call	DrivePars		; exit if an error occurred
		jc	@01_InitDisk
@02_InitDisk:	mov	ax, SFNSearch.ssSearchCluster
		call	SaveDirCluster		;Save directory cluster
		clc
@01_InitDisk:	retn

;Initialize the data base, by creating the name of the data base in the
;  directory of a file
;  Input : DS:SI: the full file name
InitDB:		mov	di, Offset DataFullName	;Get the path part of the file
		push	di			; name
		call	GetPath
		pop	si
		mov	bx, Offset DataBaseName	;Add data base file name to
		mov	di, si			; the path
		call	AddToPath
		clc
		retn

;Initialize directory-related information inside the directory of a file
;  Input : DS:SI: the full file name
;  Output: CF: when not zero, an error occured
InitCompDir:	cmp	VFATSupport, False
		je	@01_InitCompDir
		jmp	InitDisk
@01_InitCompDir:jmp	InitDB

;Initialize directory-related information inside the base directory (current
;  directory, for a relative path, or root directory, for an absolute path)
;  of a file
;  Input : DS:SI: the full file name
;  Output: CF: when not zero, an error occured
InitBaseDir:	cmp	VFATSupport, False
		je	@01_InitBaseDir
		mov	LFNDrive, -1		;Invalidate physical drive
		jmp	InitDisk		;Initialize disk directory
@01_InitBaseDir:retn

;Open data base
;  Input : BX: access mode and sharing flags
;          CX: file attributes
;          DX: open mode
;  Output: CF: when not zero, an error occured
;          BX: file handle (if CF zero) or error code (if CF not zero)
OpenDB:		mov	si, Offset DataFullName
		mov	ax, 6C00h		;Extended open file
		int	21h
		mov	bx, ax
		retn

;Open disk directory
OpenDisk:	call	ReadDirSector		;Read the first sector of the
		jc	@01_OpenDisk		; directory
		xor	ax, ax			;Start at sector zero in the
		mov	DirSector, ax		; cluster
		sub	ax, EntrySize		;Start at -1'th entry so that
		mov	DirOffset, ax		; "next entry" gets the first
		clc
@01_OpenDisk:	retn

;Open directory
;  Input : BX: access mode and sharing flags (only for data base)
;          CX: file attributes (only for data base)
;          DX: open mode (only for data base)
;  Output: CF: when not zero, an error occured
;          BX: file handle (if CF zero, only for data base) or error code (if
;              CF not zero)
OpenDir:	cmp	VFATSupport, False
		je	@01_OpenDir
		jmp	OpenDisk
@01_OpenDir:	jmp	OpenDB

;Close the data base
;  Input : BX: the data base file handle
CloseDB:
		xor	cx, cx			;Fetch the length of data base
		xor	dx, dx			; file
		mov	ax, 4202h
		int	21h
		mov	si, ax
		or	si, dx
		mov	ah, 3Eh			;Close data base file
		int	21h
		or	si, si			;If file length is zero then
		jne	@01_CloseDB		; the file has to be deleted
		mov	dx, Offset DataFullName
		mov	ah, 41h
		int	21h
@01_CloseDB:	retn

;Close directory
;  Input : BX: the data base file handle (only for data base)
CloseDir:	cmp	VFATSupport, False
		je	@01_CloseDir
		retn
@01_CloseDir:	jmp	CloseDB

;Duplicate original file name, if no corresponding file name was found
DupOrigName:	mov	si, Offset FindPattern
		mov	di, Offset LongFName
		push	di
		call	CutPath
		pop	si
		mov	di, Offset ShortFName
		push	si
		push	di
		call	UpperCase
		pop	si
		pop	di
		jmp	StrCopy

;Find a file name in the conversion database and convert it to short form or
;  long form
ConvName:	push	si
		push	di
		push	ax
		push	bx
		push	cx
		push	dx
		mov	si, Offset FindPattern	;Initialize directory of find
		push	si			; pattern
		call	InitCompDir
		pop	si
		jc	@06_ConvName
		mov	di, si			;Get file name of find pattern
		call	CutPath
		xor	ax, ax			;Initialize data buffer and
		mov	EndOfDir, al		; file positions
		mov	DataPos, ax
		mov	DataLen, ax
		mov	word ptr LineStart[0], ax
		mov	word ptr LineStart[2], ax
		mov	word ptr LineEnd[0], ax
		mov	word ptr LineEnd[2], ax
		mov	NameFound, al		;File names not found yet
		mov	DirFound, al		;Directory not found yet
		xor	bx, bx			;Open directory for read only
		xor	cx, cx
		mov	dx, fmOpenIfEx + fmErrorIfNEx
		call	OpenDir
		jnc	@05_ConvName
@06_ConvName:	call	DupOrigName		;Duplicate original file name
		jmp	@02_ConvName		; if an error occured
@05_ConvName:	inc	DirFound		;Set directory found flag
@01_ConvName:	call	ReadEntry		;Read a line from data base
		mov	si, Offset FindPattern	;Compare find pattern with
		mov	di, Offset ShortFName	; short file name and end if
		call	StrComp			; found
		je	@04_ConvName
		mov	si, Offset FindPattern	;Compare find pattern with
		mov	di, Offset LongFName	; long file name and end if
		call	StrComp			; found
		je	@04_ConvName
		cmp	EndOfDir, False		;End if the end of the data
		je	@01_ConvName		; base was reached
		call	DupOrigName		;Duplicate original file name
		jmp	@03_ConvName		; if no file name corresponds
@04_ConvName:	inc	NameFound		;File names were found
@03_ConvName:	call	CloseDir
@02_ConvName:	pop	dx
		pop	cx
		pop	bx
		pop	ax
		pop	di
		pop	si
		retn

;Clear a search record
;  Input : CX: handle of the search record
;          DS:BX: the search record
ClearSearch:	push	cx			;Save search record number
		cld				;If search record is free then
		mov	cx, SearchRecSize	; fill it up with zeros
		mov	di, bx
		xor	al, al
		rep	stosb
		pop	cx			;Restore search record number
		retn

;Find a free search record
;  Output: CF: when zero, a free search record was found
;          CX: search record number
FindSearch:	xor	cx, cx			;Begin with first record
@03_FindSearch:	call	GetSearchAddr		;Get offset of search record
		cmp	byte ptr [bx].SearchRec.srUsed, False
		clc
		jne	@01_FindSearch
		call	ClearSearch
		jmp	@02_FindSearch
@01_FindSearch:	inc	cx			;Try next search record
		cmp	cx, SearchRecNum	;Quit if no more records
		jb	@03_FindSearch
		stc
@02_FindSearch:	retn

;Fetch search handle and check if it is inside the allowed interval
;  Output : CF: when not zero, the search handle is invalid
;           DS:BX: the search record
;           CX: search handle
GetSearch:	mov	cx, OrigBX		;Fetch search handle and
		dec	cx			; decrease it by one
		cmp	cx, SearchRecNum
		jae	@01_GetSearch
		call	GetSearchAddr		;If below number of handles
		clc				; then get offset of search
		jmp	@02_GetSearch		; record
@01_GetSearch:	mov	OrigAL, 18		;If above number of handles
		stc				; then return error code
@02_GetSearch:	retn

;Initialize file name conversion
;  Input: AL: zero for converting the file name into short form, not zero for
;             conversion into long form
InitConv:	push	si
		push	di
		mov	ConvToLong, al		;Store conversion direction
		mov	si, Offset LongFName	;Converting from long file
		mov	di, Offset ShortFName	; names into short ones
		or	al, al
		je	@01_InitConv
		xchg	si, di			;Swap source and destination
@01_InitConv:	mov	SourceOfs, si		;Store string offsets
		mov	DestOfs, di
		pop	di
		pop	si
		retn

;Process a file name and convert it, component by component, into full short
;  or full long form
;  Input: AL: zero for converting the file name into short form, not zero for
;             conversion into long form
;         DS:SI: the original file name
;         ES:DI: the destination file name
ProcName:	call	InitConv		;Init conversion
		mov	bp, di			;Save source file name
		mov	di, Offset ProcFullName
		call	StrCopy
		xor	al, al			;Clear destination file name
		mov	di, bp			; and short path
		mov	byte ptr [di][0], al
		mov	byte ptr ShortPath[0], al
		mov	FirstPathChar, al	;There was no drive letter
		xor	bx, bx			;Path starts at first char
		mov	ax, word ptr ProcFullName[0]
		call	UpCase
		cmp	al, 'A'			;If first two characters
		jb	@13_ProcName		; resemble a drive then copy
		cmp	al, 'Z'			; it into the destination file
		ja	@13_ProcName		; name
		cmp	ah, ':'
		jne	@13_ProcName
		add	bx, 2
@13_ProcName:	mov	FirstPathChar, bl
		mov	al, ProcFullName[bx]
		call	IsSlash			;If third character is a slash
		jne	@02_ProcName		; or a backslash then also
		inc	bl			; copy that character
@02_ProcName:	mov	cx, bx
		push	cx			;Copy drive letter into
		mov	si, Offset ProcFullName	; destination file name
		mov	di, bp
		push	si
		call	StrNCopy
		pop	si
		pop	cx
		push	cx			;Copy drive letter into short
		push	si			; path
		mov	di, Offset ShortPath
		call	StrNCopy
		mov	si, bp		;Initialize base directory of
		call	InitBaseDir		; file and exit if an error
		pop	si			; occurred
		pop	cx
		jc	@11_ProcName
		jcxz	@12_ProcName
		mov	di, si			;Delete drive letter from
		add	si, cx			; source file name
		push	di
		call	StrCopy
		pop	si
@12_ProcName:	cmp	byte ptr [si], 0	;If the file name was only a
		jne	@01_ProcName		; drive letter then it was
@11_ProcName:	inc	NameFound		; converted successfully
		jmp	@05_ProcName
@01_ProcName:	mov	si, Offset ProcFullName	;If the end of source file
		cmp	byte ptr [si], 0	; has been reached then
		jne	@04_ProcName		; terminate conversion
		jmp	@05_ProcName
@04_ProcName:	mov	di, Offset ProcFName	;Find first slash or backslash
		mov	al, '\'
		call	StrPos
		je	@07_ProcName
		mov	al, '/'
		call	StrPos
		je	@07_ProcName
@06_ProcName:	push	si			;No slashes or backslashes
		push	di			; were found, cut the whole
		call	StrCopy			; source file name into buffer
		pop	di
		pop	si
		mov	byte ptr [si], 0
		jmp	@08_ProcName
@07_ProcName:	push	di			;Copy first directory name
		push	si			; from source file name into
		push	ax			; buffer
		mov	cx, ax
		call	StrNCopy
		pop	ax
		pop	si
		mov	di, si			;Delete first directory from
		add	si, ax			; source file name
		inc	si
		call	StrCopy
		pop	di
		cmp	byte ptr [di], 0	;If buffer is empty then put
		jne	@08_ProcName		; in a backslash
		mov	word ptr [di], '\'
@08_ProcName:	mov	bx, di
		mov	ax, [di]		;Fetch first two characters
		or	ah, ah
		jne	@03_ProcName
		call	IsSlash			;If the buffer is a slash, a
		je	@09_ProcName		; backslash or a dot then do
		cmp	al, '.'			; not convert
		je	@09_ProcName
@03_ProcName:	cmp	ax, '..'		;If the buffer does not start
		jne	@10_ProcName		; with two dots then convert
		mov	ax, [di][2]		;Fetch second two characters
		or	al, al			;If the buffer is two dots
		je	@10_ProcName		; then do not convert
		cmp	ax, '.'			;If the buffer is three dots
		jne	@10_ProcName		; then strip all directory
		mov	cl, FirstPathChar	; names, keep the drive
		xor	ch, ch			; designator, if present, and
		mov	di, bp			; append a backslash
		add	di, cx
		mov	al, '\'
		stosw
		mov	di, Offset ShortPath	;Keep the drive designator of
		add	di, cx			; the short path and append a
		stosw				; backslash
		jmp	@01_ProcName
@10_ProcName:	mov	si, Offset ShortPath	;Add buffer to the short path
		mov	bx, di			; collected during the
		mov	di, Offset FindPattern	; conversion
		call	AddToPath
		mov	si, Offset ShortFName
		mov	di, Offset LongFName
		call	ConvName		;Convert the file name
		mov	di, DestOfs
		mov	bx, Offset ShortFName
@09_ProcName:	push	bx
		mov	si, bp			;Add converted file name to
		mov	bx, di			; the collected path
		mov	di, si
		call	AddToPath
		pop	bx
		mov	si, Offset ShortPath	;Add short file name to the
		mov	di, si			; collected short path
		call	AddToPath
		jmp	@01_ProcName
@05_ProcName:	retn

;Compare a file name against a pattern
;  Input : DS:SI: the pattern
;          DS:DI: the file name
;  Output: ZF: when not zero, the file name matches the pattern
CompName:	xor	bp, bp			;No need to truncate file name
		call	StrLen			;Go to the end of the pattern
		mov	bx, si
		add	bx, ax
		or	ax, ax
		je	@20_CompName
		cmp	byte ptr [bx][-1], '.'	;If the pattern ends with a
		je	@21_CompName		; dot or a dot plus asterisk,
		cmp	word ptr [bx][-2], '*.'	; check file name for dots
		jne	@20_CompName
@21_CompName:	xchg	si, di			;If there is no dot in the
		mov	al, '.'			; file name, append one to it
		call	StrPos
		xchg	si, di
		je	@20_CompName
		xchg	si, di
		call	StrLen
		mov	bx, si
		add	bx, ax
		mov	bp, bx
		mov	word ptr [bx], '.'
		xchg	si, di
@20_CompName:	mov	bl, True		;Set names match flag
@07_CompName:	or	bl, bl			;End if mismatch was found
		je	@01_CompName
		mov	al, [si]		;Fetch pattern character
		mov	ah, [di]		;Fetch file name character
		or	al, al			;End if both strings reached
		jne	@02_CompName		; their ends
		or	ah, ah
		je	@01_CompName
@02_CompName:	cmp	al, '*'			;Handle asterisk in pattern
		jne	@03_CompName
		mov	bh, True		;Set continue search flag
		cmp	byte ptr [si][1], 0	;End search if the asterisk is
		je	@22_CompName		; the last pattern character
		xor	bh, bh			;Clear continue search flag
		mov	cx, si			;Save current position in both
		mov	dx, di			; strings
		inc	bh			;Set continue search flag
		inc	si			;Go to next pattern character
@16_CompName:	or	bh, bh			;End if mismatch was found
		je	@17_CompName
		mov	al, [si]		;Fetch pattern character
		mov	ah, [di]		;Fetch file name character
		or	al, al			;End if both strings reached
		jne	@12_CompName		; their ends
		or	ah, ah
		je	@17_CompName
@12_CompName:	cmp	al, '*'			;Handle asterisk in pattern
		jne	@13_CompName
		xor	bh, bh			;Clear continue search flag
		inc	cx			;Go to next pattern character
		jmp	@09_CompName
@13_CompName:	cmp	al, '?'			;If the current pattern
		je	@14_CompName		; character is a question mark
		call	UpCase			; or the uppercase form of the
		xchg	al, ah			; current pattern and file
		call	UpCase			; name characters match then
		cmp	al, ah			; go to the next character,
		jne	@15_CompName		; otherwise end search
@14_CompName:	inc	si
		inc	di
		jmp	@16_CompName
@15_CompName:	xor	bh, bh			;Clear continue search flag
		inc	dx			;Go to next file name char
		push	di
		mov	di, dx
		cmp	byte ptr [di], 0	;If file name reached its end
		pop	di			; then a mismatch was found
		jne	@16_CompName
		xor	bl, bl
@17_CompName:	or	bh, bh
		je	@09_CompName
		cmp	byte ptr [si], 0	;If one of the strings has not
		jne	@18_CompName		; reached its end then the
		cmp	byte ptr [di], 0	; asterisk covers only a part
		je	@09_CompName		; of the file name
@18_CompName:	xor	bh, bh
@09_CompName:	mov	si, cx			;Restore position in both
		mov	di, dx			; strings
@22_CompName:	or	bh, bh			;If the asterisk covers the
		je	@07_CompName		; complete remainder of the
@10_CompName:	inc	si			; file name then go to the end
		cmp	byte ptr [si], 0	; of both strings
		jne	@10_CompName
@11_CompName:	cmp	byte ptr [di], 0
		je	@07_CompName
		inc	di
		jmp	@11_CompName
@03_CompName:	or	al, al			;End if either string has
		je	@04_CompName		; reached its end
		or	ah, ah
		je	@04_CompName
		cmp	al, '?'			;If the current pattern
		je	@05_CompName		; character is a question mark
		call	UpCase			; or the uppercase form of the
		xchg	al, ah			; current pattern and file
		call	UpCase			; name chars match then go to
		cmp	al, ah			; the next char, otherwise
		jne	@06_CompName		; strings do not match
@05_CompName:	inc	si
		inc	di
		jmp	@07_CompName
@06_CompName:	xor	bl, bl
@01_CompName:	or	bl, bl
		je	@08_CompName
		cmp	byte ptr [si], 0	;If one of the strings has not
		jne	@04_CompName		; reached its end then no
		cmp	byte ptr [di], 0	; match was found
		je	@08_CompName
@04_CompName:	xor	bl, bl
@08_CompName:	or	bp, bp			;If a dot was appended to the
		je	@23_CompName		; file name then truncate it
		mov	di, bp
		mov	byte ptr [di], 0
@23_CompName:	cmp	bl, True
		retn

;Extract short file data from search record into long search record
;  Input : DS:BX: the search record
;  Output: CF: when not zero, the file does not match the search criteria
ExtractDTA:	mov	ax, word ptr [bx].SearchRec.srAllowedAttr
		mov	dl, [bx].SearchRec.srAttr
		not	al			;Check file attribute against
		and	al, dl			; allowed attributes and skip
		je	@01_ExtractDTA		; if they do not match
@03_ExtractDTA:	stc
		jmp	@02_ExtractDTA
@01_ExtractDTA:	and	dl, ah			;Check file attributes against
		cmp	dl, ah			; required attributes and skip
		jne	@03_ExtractDTA		; if they do not match
		mov	di, Offset FindPattern	;Get path of find pattern
		push	di
		push	bx
		lea	si, [bx].SearchRec.srPattern
		call	GetPath
		pop	bx
		pop	si			;Add file name to path of find
		push	bx			; pattern
		lea	bx, [bx].SearchRec.srName
		mov	di, si
		call	AddToPath
		pop	bx
		cmp	byte ptr [bx].SearchRec.srDirFound, False
		jne	@06_ExtractDTA
		push	di
		push	bx
		mov	si, Offset FindPattern	;Get file name of find pattern
		mov	di, si			; and copy it into the short
		call	CutPath			; and long file name if could
		call	DupOrigName		; not find directory
		pop	bx
		pop	di
		jmp	@07_ExtractDTA
@06_ExtractDTA:	mov	al, True		;Convert file name into long
		call	InitConv		; form
		call	ConvName
		mov	al, DirFound		;Save directory found flag
		mov	[bx].SearchRec.srDirFound, al
@07_ExtractDTA:	mov	di, Offset FindPattern	;Get file name of find pattern
		push	di
		push	bx
		lea	si, [bx].SearchRec.srPattern
		call	CutPath
		pop	bx
		pop	si
		mov	di, Offset ShortFName
	ifdef HideDataBase
		cmp	VFATSupport, False	;Skip hiding the data base
		jne	@08_ExtractDTA		; file in VFAT mode
		push	si
		mov	si, Offset DataBaseName	;If the short file name is the
		push	di			; name of the data base then
		push	bx			; skip the current entry
		call	CompName
		pop	bx
		pop	di
		pop	si
		je	@03_ExtractDTA
	endif
@08_ExtractDTA:	push	si			;If the short or long file
		push	bx			; name does not match the file
		call	CompName		; name of the find pattern
		pop	bx			; then skip the current entry
		pop	si
		je	@04_ExtractDTA
		cmp	byte ptr [bx].SearchRec.srDirFound, False
		je	@03_ExtractDTA		;No compare if no directory
		mov	di, Offset LongFName	;Compare long file name with
		push	bx			; the search pattern
		call	CompName
		pop	bx
		jne	@03_ExtractDTA
@04_ExtractDTA:	cmp	bx, Offset IntSearchRec	;If using extra search record
		je	@05_ExtractDTA		; then no need to store data
		mov	ax, OrigProcessID	;Store the owner of record
		mov	[bx].SearchRec.srProcess, ax
		mov	di, OrigDI		;Fetch address of long search
		mov	es, OrigES		; record
						;Store file attributes
		mov	al, [bx].SearchRec.srAttr
		mov	byte ptr es:[di].LongSearchRec.lsAttr, al
						;Store file size
		mov	ax, word ptr [bx].SearchRec.srSize[0]
		mov	dx, word ptr [bx].SearchRec.srSize[2]
		mov	word ptr es:[di].LongSearchRec.lsSize[0], ax
		mov	word ptr es:[di].LongSearchRec.lsSize[2], dx
						;Store file date stamp
		mov	ax, word ptr [bx].SearchRec.srDate[0]
		mov	dx, word ptr [bx].SearchRec.srDate[2]
		mov	word ptr es:[di].LongSearchRec.lsCreateDate[0], ax
		mov	word ptr es:[di].LongSearchRec.lsCreateDate[2], dx
		mov	word ptr es:[di].LongSearchRec.lsAccessDate[0], ax
		mov	word ptr es:[di].LongSearchRec.lsAccessDate[2], dx
		mov	word ptr es:[di].LongSearchRec.lsModifyDate[0], ax
		mov	word ptr es:[di].LongSearchRec.lsModifyDate[2], dx
		mov	bp, di			;Store short file name
		lea	si, [bx].SearchRec.srName
		lea	di, [bp].LongSearchRec.lsShortName
		call	StrCopy
		mov	si, Offset LongFName	;Store long file name
		lea	di, [bp].LongSearchRec.lsName
		call	StrCopy
		push	ds
		pop	es
@05_ExtractDTA:	clc
@02_ExtractDTA:	retn

;Execute a DOS function and return the result
;  Input : all registers set for the DOS function
;  Output: CF: when not zero, an error occured
;          AX: error code
ExecDOS:	int	21h			;Call DOS
		mov	OrigAX, ax		;Store result or error code
		pushf				;Save original registers
		push	ds
		push	es
		push	si
		push	di
		push	bp
		push	ax
		push	bx
		push	cx
		push	dx
		call	SaveSwap		;Save original DOS swap area
		pop	dx			;Restore original registers
		pop	cx
		pop	bx
		pop	ax
		pop	bp
		pop	di
		pop	si
		pop	es
		pop	ds
		popf
		retn

;Set DTA for file searches
;  Input : DS:BX: the search record
SetDTA:		mov	ah, 1Ah
		lea	dx, [bx].SearchRec.srReserved
		int	21h
		retn

;Flush buffers for a given drive
;  Input : AX: function code (710Dh)
;          CX: subfunction code (0 for flushing buffers, 1 for flushing
;              buffers and cache, other codes not supported)
;  Output: CF: when not zero, an error occured
;          AX: error code
Flush:		cmp	OrigCX, 2		;Fetch subfunction code and
		jae	@01_Flush		; end if invalid
		mov	ah, 0Dh			;Flush buffers
		int	21h
		cmp	OrigCX, 0
		je	@02_Flush
		mov	ax, 4A10h		;Flush SmartDrive or
		mov	bx, 1			; compatible cache
		int	2Fh
@02_Flush:	inc	RequestOK		;Set request valid flag
@01_Flush:	retn

;Write a file name into the data base and append another string to it
;  Input : BX: data base file handle
;          DS:SI: the file name
;          DS:DX: the string to append
WriteName:	push	bx
		mov	di, Offset ProcFullName
		push	di
		push	dx			;Copy the file name
		call	StrCopy
		dec	di
		pop	si			;Copy the other string
		call	StrCopy
		pop	si
		call	StrLen			;Fetch length of the string
		pop	bx
		mov	dx, si
		mov	cx, ax
		mov	ah, 40h
		int	21h
		retn

;Determine if a file name belongs to an installed device
;  Input : DS:SI: the uppercase file name
;  Output: CF: if zero, the file name has been found among the devices
IsDevice:	call	StrLen			;If the file name is longer
		cmp	ax, SFNNameLen		; than 8 characters then it
		stc				; cannot be a device name
		ja	@06_IsDevice
		push	es
		les	di, DeviceChain		;Load the first device driver
@02_IsDevice:	test	byte ptr es:[di][5], 80h
		jz	@01_IsDevice		;Skip if block device
		mov	bx, di
		add	bx, 0Ah			;Go to the name of the device
		mov	cx, SFNNameLen		;Eight characters to compare
		cld
		push	si
@05_IsDevice:	lodsb				;Load character from file name
		mov	ah, al
		mov	al, es:[bx]		;Load character from device
		inc	bx			; name
		cmp	al, ' '			;Turn space characters into a
		jne	@03_IsDevice		; zero byte because device
		xor	al, al			; names are space padded
@03_IsDevice:	call	UpCase
		cmp	al, ah			;Go to the next device if
		stc				; characters do not match
		jne	@04_IsDevice
		or	al, al			;If the characters match and
		clc				; the end of the names were
		je	@04_IsDevice		; reached, file name found;
		loop	@05_IsDevice		; otherwise go to next char
@04_IsDevice:	pop	si
		jnc	@06_IsDevice
@01_IsDevice:	les	di, es:[di]		;Load the next device driver,
		cmp	di, 0FFFFh		; end if the offset is FFFFh
		jne	@02_IsDevice
		stc
@06_IsDevice:	pop	es
		retn

;Add the short and long file name to the data base, if the long file name does
;  not belong to an installed device
AddDB:	mov	si, Offset LongFName
		mov	di, Offset ProcFName
		push	di
		call	UpperCase
		pop	si			;Search for the long file name
		call	IsDevice		; among installed devices and
		jnc	@01_AddDB		; drop file name if found
		mov	si, Offset ShortPattern
		call	InitCompDir
		mov	bx, fmReadWrite		;Open directory for read/write
		mov	cx, DataBaseAttrib	; and create if does not exist
		mov	dx, fmOpenIfEx + fmCreateIfNEx
		call	OpenDir
		jc	@01_AddDB
		xor	cx, cx			;Seek to the end of data base
		xor	dx, dx
		mov	ax, 4202h
		int	21h
		jc	@02_AddDB
		mov	si, Offset ShortFName	;Turn short file name into
		mov	di, si			; full uppercase
		push	si
		call	UpperCase
		pop	si
		cmp	byte ptr LongFName[0], ' '
		mov	dx, Offset WhiteSpace	;Put short file name and white
		jne	@03_AddDB		; space into data base and add
		mov	dx, Offset WhiteSpace2	; a quotation mark if long the
@03_AddDB:	call	WriteName		; file name starts with space
		mov	si, Offset LongFName	;Put long file name and end of
		mov	dx, Offset EndOfLine	; line mark into data base
		call	WriteName
@02_AddDB:	call	CloseDir		;Close directory
@01_AddDB:	retn

;Store the long file name, if it does not belong to an installed device
AddData:	cmp	VFATSupport, False
		je	@01_AddData
		retn
@01_AddData:	jmp	AddDB

;Convert long file name part into short form, correcting invalid characters
;  Input : CX: maximum length of file name part
;          DS:SI: the long file name
;          ES:DI: the destination short file name
Correct:	cld
		xor	bl, bl			;Initialize conversion flags
		and	bh, 0FBh
@04_Correct:	lodsb				;Fetch character
		or	al, al			;End if the end of the long
		je	@01_Correct		; file name was reached
		mov	ah, al			;Save character
		call	UpCase			;If character differs from its
		cmp	ah, al			; uppercase form, lowercase
	ifdef KeepFullLoCase
		je	@09_Correct
	else
		je	@08_Correct
		test	bh, 2			;If there was a uppercase
		jne	@10_Correct		; character then long name
		or	bh, 1			;Set lowercase flag
@08_Correct:	call	LoCase			;If character differs from its
		cmp	ah, al			; lowercase form, uppercase
		je	@09_Correct
		test	bh, 1			;If there was an lowercase
		jne	@10_Correct		; character then long name
		or	bh, 2			;Set uppercase flag
		jmp	@09_Correct
	endif
@10_Correct:	or	bl, 2			;Mixed chars in long name
@09_Correct:	cmp	al, ' '			;If the character is a space
		jne	@11_Correct		; then skip it and set flag
		or	bl, 1			; for storing new file names
		jmp	@04_Correct
@11_Correct:	cmp	al, '.'			;If the character is a dot
		jne	@03_Correct		; then skip it and set flag
@02_Correct:	or	bl, 1			; for storing new file names
		jmp	@04_Correct
@03_Correct:	cmp	al, '+'			;If the character is a plus
		je	@05_Correct		; sign, a comma, a semi-colon,
		cmp	al, ','			; an equal sign or a bracket
		je	@05_Correct		; then convert it into an
		cmp	al, ';'			; underscore
		je	@05_Correct
		cmp	al, '='
		je	@05_Correct
		cmp	al, '['
		je	@05_Correct
		cmp	al, ']'
		jne	@06_Correct
@05_Correct:	mov	al, '_'			;Conversion took place, new
		or	bl, 1			; file names have to be stored
@06_Correct:	stosb				;Store character
		loop	@04_Correct		;End if long name too long
		or	bh, 4			;Long file may be too long
@01_Correct:	test	bh, 4			;If the end of the long file
		je	@07_Correct		; name was not reached then
		cmp	byte ptr [si], 0	; the new file names have to
		je	@07_Correct		; be stored
		or	bl, 1
@07_Correct:	xor	al, al			;End short file name
		stosb
		or	NewFileName, bl
		retn

;Shorten and correct long file name into a short form
;  Input : DS:SI: the original long file name
CorrName:	mov	di, Offset ShortFExt	;Clear file extension
		mov	byte ptr [di], 0
		mov	al, '.'			;Search for the rightmost dot
		call	StrRPos
		mov	bh, 0			;Clear case flags
		jne	@01_CorrName
		push	si
		add	si, ax
		mov	bp, si
@03_CorrName:	or	ax, ax			;If the dots are at the
		je	@04_CorrName		; beginning of the file name
		dec	ax			; then there is no extension
		dec	si
		cmp	byte ptr [si], '.'
		je	@04_CorrName
		mov	si, bp
		mov	byte ptr [si], 0	;If a dot was found then cut
		inc	si			; the extension off the file
		mov	cx, SFNExtLen		; name and convert the
		call	Correct			; extension, maximum 3 chars
@04_CorrName:	pop	si
@01_CorrName:	mov	di, Offset ShortFName	;Convert file name, maximum 8
		push	di			; chars
		mov	cx, SFNNameLen
		call	Correct
		pop	di
		cmp	byte ptr [di], 0	;If file name is empty then
		jne	@02_CorrName		; put in an underscore
		mov	ax, '_'
		stosw
		mov	NewFileName, True
@02_CorrName:	retn

;Generate a short alias
;  Input : BP: the number to append to the file name; if 0, neither the tilde
;              or a number is appended
GenAlias:	mov	cx, SFNNameLen		;File name length is eigth
		xor	si, si			;Number length is zero
		or	bp, bp			;Skip numbering if current
		je	@01_GenAlias		; number is zero
		dec	cx			;Decrease file name length
		mov	bx, 10
		mov	ax, bp
@02_GenAlias:	dec	cx			;Decrease file name length
		inc	si			;Increase number length
		xor	dx, dx			;Divide number by ten
		div	bx
		or	ax, ax			;If quotient not zero then
		jne	@02_GenAlias		; divide again
@01_GenAlias:	push	si
		mov	si, Offset ShortFName	;Copy the first characters of
		mov	di, Offset ProcFName	; the file name
		call	StrNCopy
		pop	si
		or	bp, bp			;If number is zero then do not
		je	@03_GenAlias		; append number to file name
		dec	di
		mov	al, '~'			;Append a tilde to the short
		stosb				; file name
		add	di, si
		push	di
		std
		xor	al, al			;End the short file name
		stosb
		mov	bx, 10
		mov	ax, bp
@04_GenAlias:	xor	dx, dx			;Divide number by ten
		div	bx
		push	ax
		mov	al, dl			;Convert remainder to a
		add	al, '0'			; character and add to short
		stosb				; file name backwards
		pop	ax
		or	ax, ax			;If quotient not zero then
		jne	@04_GenAlias		; divide again
		pop	di
@03_GenAlias:	mov	si, Offset ShortFExt	;If there is an extension then
		cmp	byte ptr [si], 0	; append at most 3 characters
		je	@05_GenAlias		; of it, with a dot between
		mov	al, '.'			; the name and the extension
		mov	cx, SFNExtLen
		cld
		stosb
		call	StrNCopy
@05_GenAlias:	retn

;Search for a short file name corresponding to the long file name and create
;  a new one if needed
Shorten:	mov	NewFileName, False	;No new file name was created
		mov	si, Offset LongPattern	;Convert file name into short
		mov	di, Offset ShortPattern	; form
		push	si
		xor	al, al
		call	ProcName
		pop	si
		cmp	NameFound, False	;If file name was found then
		je	@07_Shorten		; no need to create a new one
		retn
@07_Shorten:	mov	si, Offset LongPattern	;Get the file name part of the
		mov	di, Offset LongFName	; full long file name
		call	CutPath
		mov	si, Offset LongFName	;Save the original long file
		mov	di, Offset ProcFullName	; name
		push	di
		call	StrCopy
		pop	si
		mov	di, Offset ShortFName
		mov	ax, [si]		;Fetch first two characters
		or	al, al			;Skip if empty file name
		je	@01_Shorten
		cmp	ax, '.'			;If the name is a dot then do
		je	@01_Shorten		; not convert
		cmp	ax, '..'		;If the name does not start
		jne	@02_Shorten		; with two dots then convert
		mov	ax, [si][2]		;Fetch second two characters
		or	al, al			;If the name is two dots then
		je	@01_Shorten		; do not convert
		cmp	ax, '.'			;If the buffer is three dots
		jne	@02_Shorten		; then do not convert
@01_Shorten:	push	di			;Copy file name
		call	StrCopy
		pop	di
		jmp	@03_Shorten
@02_Shorten:	call	CorrName		;Shorten and correct long name
@03_Shorten:	mov	si, Offset ShortPattern	;Get the path part of the
		mov	di, Offset ShortPath	; short file name
		call	GetPath
		test	NewFileName, 1		;If conversion took place
		jne	@05_Shorten		; then append tilde and
		mov	si, Offset ShortPath	; number, otherwise append
		mov	bx, Offset LongFName	; long file name to short path
		mov	di, si
		call	AddToPath
		jmp	@04_Shorten
@05_Shorten:	mov	bp, 1			;Start with a number of one
		cmp	AliasHintReq, 0		;If an alias hint was
		je	@06_Shorten		; requested then try the
		mov	bp, AliasHint		; specified number first
@06_Shorten:	call	GenAlias		;Generate alias
		mov	si, Offset ShortPath	;Add the converted short file
		mov	bx, Offset ProcFName	; name to the path
		mov	di, Offset ShortPattern
		push	di
		call	AddToPath
		pop	dx			;Check for the existence of
		mov	ax, 4300h		; the file
		int	21h
		jc	@04_Shorten		;If the file exists then try
		inc	bp			; the next number
		cmp	AliasHintReq, 0		;If an alias hint requested,
		je	@06_Shorten		; clear the flag and start
		mov	AliasHintReq, 0		; again with a number of one
		jmp	@05_Shorten
@04_Shorten:	mov	si, Offset ShortPattern	;Get the file name part of the
		mov	di, Offset ShortFName	; full short file name
		jmp	CutPath

;Make directory
;  Input : AX: function code (7139h)
;          DS:DX: the directory name
;  Output: CF: when not zero, an error occured
;          AX: error code
MkDir:		push	ds			;Fetch directory name
		mov	si, OrigDX
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		mov	dx, Offset ShortPattern
		mov	ah, 39h			;Make directory and end if an
		call	ExecDOS			; error occured
		jc	@01_MkDir
		cmp	NewFileName, False	;If new directory names were
		je	@02_MkDir		; created then add them to the
		call	AddData			; data base
@02_MkDir:	inc	RequestOK		;Set request valid flag
@01_MkDir:	retn

;Delete the short and long file name from the data base
DelDB:	mov	si, Offset ShortPattern
		call	InitCompDir
		mov	bx, fmReadWrite		;Open directory for read/write
		xor	cx, cx
		mov	dx, fmOpenIfEx + fmErrorIfNEx
		call	OpenDir
		jc	@01_DelDB
		xor	si, si			;Clear delete file flag
@03_DelDB:	mov	dx, word ptr LineEnd[0]	;Seek to the line following
		mov	cx, word ptr LineEnd[2]	; the current one
		mov	ax, 4200h
		int	21h
		jc	@02_DelDB
		mov	dx, Offset DataBuffer	;Read a chunk of data from the
		mov	cx, DataBufferSize	; data base file
		mov	ah, 3Fh
		int	21h
		jc	@02_DelDB
		mov	di, ax			;Save number of bytes read
		mov	dx, word ptr LineStart[0]
		mov	cx, word ptr LineStart[2]
		mov	ax, 4200h		;Seek to the current line
		int	21h
		jc	@02_DelDB
		mov	dx, Offset DataBuffer	;Write a chunk of data to the
		mov	cx, di			; data base file
		mov	ah, 40h
		int	21h
		jc	@02_DelDB
		add	word ptr LineEnd[0], di	;Increase positions with the
		adc	word ptr LineEnd[2], 0	; number of bytes read
		add	word ptr LineStart[0], di
		adc	word ptr LineStart[2], 0
		or	di, di			;Read next chunk if the end of
		jne	@03_DelDB		; data base was not reached
@02_DelDB:	call	CloseDir		;Close directory
@01_DelDB:	retn

;Delete the long file name
DelData:	cmp	VFATSupport, False
		je	@01_DelData
		retn
@01_DelData:	jmp	DelDB

;Remove directory
;  Input : AX: function code (713Ah)
;          DS:DX: the directory name
;  Output: CF: when not zero, an error occured
;          AX: error code
RmDir:		push	ds			;Fetch directory name
		mov	si, OrigDX
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		mov	dx, Offset ShortPattern	;Remove directory
		mov	ah, 3Ah
		call	ExecDOS
		jc	@01_RmDir
		cmp	NameFound, False	;If directory names were found
		je	@02_RmDir		; in the data base then delete
		call	DelData			; them
@02_RmDir:	inc	RequestOK		;Set request valid flag
@01_RmDir:	retn

;Change directory
;  Input : AX: function code (713Bh)
;          DS:DX: the directory name
;  Output: CF: when not zero, an error occured
;          AX: error code
ChDir:		push	ds			;Fetch directory name
		mov	si, OrigDX
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		mov	dx, Offset ShortPattern	;Change directory
		mov	ah, 3Bh
		call	ExecDOS
		jc	@01_ChDir
		inc	RequestOK		;Set request valid flag
@01_ChDir:	retn

;Extended open file
;  Input : BX: file open mode
;          CX: file attributes when creating a file
;          DX: open mode (bit 0 for open file if exists, bit 1 for truncate
;              file if exists, bit 4 for create file in does not exist)
;  Output: CF: when not zero, an error occured
;          AX: file handle (if CF zero) or error code (if CF not zero)
ExtOpen:	mov	NewFileDevice, False
		push	bx
		mov	ah, 50h			;Setup original process ID
		mov	bx, OrigProcessID
		int	21h
		pop	bx
		and	bx, 1110000011110011b	;Delete unsupported flags
		mov	si, Offset ShortPattern
		mov	ax, 6C00h		;Open file
		call	ExecDOS
		jc	@01_ExtOpen
		cmp	cx, 2			;If a new file was created
		jne	@02_ExtOpen		; then check if it is a device
		push	ax
		mov	ax, 4400h
		mov	bx, OrigAX
		call	ExecDOS
		jc	@03_ExtOpen
		test	dx, 0080h
		je	@03_ExtOpen
		inc	NewFileDevice		;Set new file is a device flag
@03_ExtOpen:	pop	ax
		mov	OrigAX, ax
@02_ExtOpen:	clc
@01_ExtOpen:	pushf
		mov	ah, 50h			;Restore own process ID
		mov	bx, cs
		int	21h
		popf
		retn

;Create file
;  Input : AH: function code (3Ch or 5Bh)
;          CX: file attributes
;          DS:DX: the file name
;  Output: CF: when not zero, an error occured
;          AX: file handle (if CF zero) or error code (if CF not zero)
CreateFile:	push	ds			;Fetch long file name
		mov	si, OrigDX
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		mov	bx, 2			;Create file in read and write
		mov	cx, OrigCX		; mode with given attributes
		mov	dx, 12h			; and truncate it if it exists
		call	ExtOpen			;Call DOS to create file and
		jc	@01_CreateFile		; end if an error occured
		cmp	NewFileName, False	;If new file names were
		je	@02_CreateFile		; created and the new file is
		cmp	NewFileDevice, False	; not a device then add the
		jne	@02_CreateFile		; file names to the data base
		call	AddData
@02_CreateFile:	inc	RequestOK		;Set request valid flag
@01_CreateFile:	retn

;Open file (only an already existing one)
;  Input : AH: function code (3Dh)
;          AL: open mode (0 for read only, 1 for write only and 2 for read and
;              write)
;          DS:DX: the file name
;  Output: CF: when not zero, an error occured
;          AX: file handle (if CF zero) or error code (if CF not zero)
OpenFile2:	push	ds			;Fetch long file name
		mov	si, OrigDX
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		mov	bx, AbsOrigAX		;Open the file in the given
		xor	bh, bh			; mode and fail if it does not
		mov	dx, 01h			; exist
		call	ExtOpen			;Call DOS to open file and end
		jc	@01_OpenFile2		; if an error occured
		inc	RequestOK		;Set request valid flag
@01_OpenFile2:	retn

;Find the next file matching the search criteria
;  Input : AH: DOS function code (4Eh for the first file, 4Fh for others)
;          DS:BX: the search record
FindMatch:	call	ExecDOS			;Call DOS
		jnc	@01_FindMatch		;End if an error occured
		cmp	NewSearch, False	;Check if a new search is
		jne	@04_FindMatch		; being processed
		mov	cx, SearchHandle	;If an old search has ended
		call	GetSearchAddr		; then free up search record
		mov	byte ptr [bx].SearchRec.srUsed, False
		jmp	@02_FindMatch
@04_FindMatch:	cmp	OrigAX, 0012h		;If no more files match the
		jne	@02_FindMatch		; find pattern and this was a
		mov	OrigAX, 2		; new search, correct error
		jmp	@02_FindMatch		; code to "File not found"
@01_FindMatch:	push	cx			;Check DTA
		call	ExtractDTA
		pop	cx
		mov	ah, 4Fh			;Go to next file if current
		jc	FindMatch		; file does not match criteria
		mov	ax, SearchHandle	;If a new search is being done
		cmp	NewSearch, False	; then store search handle
		je	@03_FindMatch
		inc	ax			;Increase search handle so
		mov	OrigAX, ax		; that it is never zero
@03_FindMatch:	mov	byte ptr [bx].SearchRec.srUsed, True
		inc	RequestOK		;Set request valid flag
@02_FindMatch:	retn

;The real "find first" routine
;  Input : CX: handle of the search record
;          DS:BX: the search record
FindFirst2:	mov	SearchHandle, cx	;Store search handle
		mov	NewSearch, True		;A new search is processed
		mov	ax, OrigCX		;Fetch attributes, set bits 0
		or	al, 0E1h		; and 5-7 in allowed attribs
		mov	[bx].SearchRec.srAllowedAttr, al
		mov	[bx].SearchRec.srMustAttr, ah
		mov	byte ptr [bx].SearchRec.srDirFound, True
		call	SetDTA			;Move DTA onto search record
		push	bx
		push	ds
		mov	si, OrigDX		;Fetch search pattern
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		mov	si, Offset LongPattern	;Get file name of search
		push	si			; pattern
		mov	di, Offset SrcLongName
		call	CutPath
		pop	si
		push	si			;Get path of search pattern
		mov	di, si
		call	GetPath
		pop	si
		mov	di, Offset ShortPattern	;Convert path of search
		push	di			; pattern into short form
		xor	al, al
		call	ProcName
		pop	si
		pop	bx
		push	bx			;Add file pattern to path
		lea	di, [bx].SearchRec.srPattern
		mov	bx, Offset SrcLongName
		call	AddToPath
		pop	bx
		push	bx
		mov	si, Offset SrcLongName
		mov	di, Offset ProcFName
		push	di
		call	UpperCase		;Search for the long file name
		pop	si			; among the installed devices
		call	IsDevice		; and, if found, append the
		mov	bx, Offset ProcFName	; device name to the path and
		mov	cx, faNormalFile	; search for normal files
		jnc	@01_FindFirst2		; only; otherwise append '*.*'
		mov	bx, Offset AllFiles	; to the path and also search
		mov	cx, faAnyFile		; for volume labels
@01_FindFirst2:	mov	si, Offset ShortPattern
		mov	di, si
		mov	dx, si
		call	AddToPath
		pop	bx
		mov	ah, 4Eh			;Search for the first file
		jmp	FindMatch

;The real "find next" routine
;  Input : CX: handle of the search record
;          DS:BX: the search record
RealFindNext:	call	SetDTA			;Move DTA to search record
		mov	NewSearch, False	;An old search is processed
		mov	ah, 4Fh			;Search for the next file
		jmp	FindMatch

;Delete file
;  Input : AX: function code (7141h)
;          CL: mask of allowed attributes (bits 0 and 5 are ignored)
;          CH: mask of required attributes
;          SI: wildcard and attributes flag (0 for none, 1 for using search
;              attributes and accepting wildcards in the file name)
;          DS:DX: the file name
Delete:		cmp	LFNFuncCall, False	;If a DOS function was called
		je	@01_Delete		; then no wildcards allowed
		cmp	OrigSI, 0		;Determine whether wildcards
		jne	@02_Delete		; and attributes are allowed
@01_Delete:	push	ds			;Fetch long file name
		mov	si, OrigDX
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		mov	dx, Offset ShortPattern	;Delete file
		mov	ah, 41h
		call	ExecDOS
		jc	@03_Delete
		cmp	NameFound, False	;If file names were found in
		je	@04_Delete		; the data base then delete
		call	DelData			; them
@04_Delete:	inc	RequestOK		;Set request valid flag
		jmp	@03_Delete
@02_Delete:	mov	cx, SearchRecNum	;Use the extra search record
		mov	bx, Offset IntSearchRec
		call	ClearSearch
		mov	FileFound, False	;Clear match found flag
		call	FindFirst2		;Search for the first file
@06_Delete:	cmp	RequestOK, False	;End if no matching file was
		je	@03_Delete		; found
		mov	FileFound, True		;Set match found flag
		mov	si, Offset LongFName	;Copy long file name from
		mov	di, Offset LongPattern	; search record
		call	StrCopy
		call	Shorten			;Shorten long file name
		mov	dx, Offset ShortPattern	;Delete file
		mov	ah, 41h
		call	ExecDOS
		jc	@03_Delete
		cmp	NameFound, False	;If file names were found in
		je	@05_Delete		; the data base then delete
		call	DelData			; them
@05_Delete:	mov	RequestOK, False	;Clear request valid flag
		mov	cx, SearchRecNum	;Use the extra search record
		mov	bx, Offset IntSearchRec
		call	RealFindNext		;Search for the next file and
		jmp	@06_Delete		; continue processing
@03_Delete:	cmp	FileFound, False	;If a matching file was found
		je	@07_Delete		; then set request valid flag
		inc	RequestOK
@07_Delete:	retn

;Get or set file attributes
;  Input : AX: function code (7143h)
;          BL: subfunction code (0 for getting, 1 for setting file attributes,
;              other codes not supported)
;          DS:DX: the file name
;  Output: CF: when not zero, an error occured
;          AX: error code
Attrib:		mov	bx, OrigBX		;Fetch subfunction code
		cmp	LFNFuncCall, False	;If ordinary DOS function was
		jne	@01_Attrib		; called, fetch subfunction
		mov	bx, AbsOrigAX		; code from original AX
@01_Attrib:	cmp	bl, 2			;End if code is invalid
		jae	@02_Attrib
		push	bx
		push	ds			;Fetch file name
		mov	si, OrigDX
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		pop	ax			;Get or set file attributes
		push	ax
		mov	cx, OrigCX
		mov	dx, Offset ShortPattern
		mov	ah, 43h
		call	ExecDOS
		pop	bx
		jc	@02_Attrib
		or	bl, bl			;Store file attributes if they
		jne	@03_Attrib		; were asked for
		mov	OrigCX, cx
@03_Attrib:	inc	RequestOK		;Set request valid flag
@02_Attrib:	retn

;Put the drive letter to the beginning of a string
;  Input : DL: drive number, 0 for current, 1-26 for drive A: to Z:
;          ES:DI: the destination string
;  Output: ES:DI: the end of the destination string
GetDrive:	mov	al, dl
		or	al, al
		jne	@01_GetDrive
		mov	ah, 19h			;Get current drive
		int	21h
		inc	al
@01_GetDrive:	add	al, 'A' - 1		;Convert number to letter
		mov	ah, ':'
		cld
		stosw				;Store the drive letter, a
		mov	ax, '\'			; colon, backslash and end the
		stosw				; string
		dec	di
		retn

;Get the current directory
;  Input : AX: function code (7147h)
;          DS:SI: the destination directory name
;  Output: CF: when not zero, an error occured
;          AX: error code
CurDir:		mov	dl, OrigDL
		mov	di, Offset ShortPattern
		call	GetDrive		;Get drive letter
		mov	si, di
		mov	ah, 47h			;Get current directory
		call	ExecDOS
		jc	@01_CurDir
		mov	si, Offset ShortPattern	;Convert directory name into
		mov	di, Offset LongPattern	; long form
		push	di
		mov	al, True
		call	ProcName
		pop	si
		add	si, 3
		mov	di, OrigSI		;Store long directory name
		mov	es, OrigDS
		call	StrCopy
		inc	RequestOK		;Set request valid flag
@01_CurDir:	retn

;Find the first file matching the search criteria
;  Input : AX: function code (714Eh)
;          CL: mask of allowed attributes (bits 0 and 5 are ignored)
;          CH: mask of required attributes
;          SI: date and time format (not supported, always assumed 1)
;          DS:DX: search file name pattern
;          ES:DI: the destination long search record
;  Output: CF: when not zero, an error occured
;          AX: search handle (if CF zero) or error code (if CF not zero)
;          CX: Unicode conversion flags (not supported, always 3)
FindFirst:	call	FindSearch		;Search for a free search
		jnc	@01_FindFirst		; record and return error code
		mov	OrigAL, 4		; if no more available
		retn
@01_FindFirst:	call	FindFirst2		;Call real find first routine
		cmp	RequestOK, False	;If a matching file was found
		je	@02_FindFirst		; then return Unicode flags
		mov	OrigCX, 3
@02_FindFirst:	retn

;Find the next file matching the search criteria
;  Input : AX: function code (714Fh)
;          BX: search handle received from FindFirst
;          SI: date and time format (not supported, always assumed 1)
;          ES:DI: the destination long search record
;  Output: CF: when not zero, an error occured
;          AX: error code
;          CX: Unicode conversion flags (not supported, always 3)
FindNext:	call	GetSearch		;Fetch search handle and end
		jc	@01_FindNext		; if invalid
		call	RealFindNext		;Call real "find next" routine
		cmp	RequestOK, False	;If a matching file was found
		je	@01_FindNext		; then return Unicode flags
		mov	OrigCX, 3
@01_FindNext:	retn

;Determine if a file name is in short form, that is, it conforms to the 8.3
;  format and contains either only lowercase or only uppercase characters
;  Input : DS:SI: the file name
;  Output: ZF: zero for long form, not zero for short form
IsShort:	xor	bx, bx			;Initialize flags
		mov	cx, SFNNameLen		;Set name length
		cld
@06_IsShort:	inc	cx			;Increase string length by one
@03_IsShort:	lodsb				;Fetch character
		or	al, al			;If the end of file name was
		je	@01_IsShort		; reached then short name
		cmp	al, '.'			;If a dot was encountered then
		jne	@02_IsShort		; here comes the extension
		or	bh, bh			;If there was already a dot
		jne	@01_IsShort		; then long name
		inc	bh			;Set extension flag
		mov	cx, SFNExtLen		;Set extension length
		jmp	@06_IsShort		;Go to next character
@02_IsShort:	mov	ah, al			;Save character
		call	UpCase			;If character different from
		cmp	ah, al			; uppercase form, lowercase
	ifdef KeepFullLoCase
		jne	@01_IsShort
	else
		je	@04_IsShort
		test	bl, 2			;If there was a uppercase
		jne	@01_IsShort		; character then long name
		or	bl, 1			;Set lowercase flag
@04_IsShort:	call	LoCase			;If character different from
		cmp	ah, al			; lowercase form, uppercase
		je	@05_IsShort
		test	bl, 1			;If there was an lowercase
		jne	@01_IsShort		; character then long name
		or	bl, 2			;Set uppercase flag
	endif
@05_IsShort:	loop	@03_IsShort		;If the file name is too long
		inc	cl			; then long name
@01_IsShort:	retn

;Shorten the destination file name of the rename function
ShortDest:	call	Shorten
		mov	si, Offset LongPattern	;Recover original destination
		mov	di, Offset LongFName	; long file name
		push	di
		call	CutPath
		pop	si
		call	IsShort			;Check whether destination
		mov	al, 0			; file name is in long form
		je	@01_ShortDest		; and set flag accordingly
		inc	al
@01_ShortDest:	mov	NewFileName, al
		retn

;Rename file
;  Input : AX: function code (7156h)
;          DS:DX: the original file name
;          ES:DI: the destination file name
;  Output: CF: not zero, an error occured
;          AX: error code (7100h, function not implemented)
Rename:		push	ds
		mov	si, OrigDX		;Fetch original file name
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		call	Shorten			;Shorten long file name
		mov	si, Offset ShortPattern	;Save full short file name
		mov	di, Offset SrcPattern
		call	StrCopy
		mov	si, Offset ShortFName	;Save short file name
		mov	di, Offset SrcShortName
		call	StrCopy
		mov	si, Offset LongFName	;Save long file name
		mov	di, Offset SrcLongName
		call	StrCopy
		mov	al, NameFound		;Save name found flag and file
		mov	SrcNameFound, al	; positions
		mov	ax, word ptr LineStart[0]
		mov	dx, word ptr LineStart[2]
		mov	word ptr SrcLineStart[0], ax
		mov	word ptr SrcLineStart[2], dx
		mov	ax, word ptr LineEnd[0]
		mov	dx, word ptr LineEnd[2]
		mov	word ptr SrcLineEnd[0], ax
		mov	word ptr SrcLineEnd[2], dx
		push	ds
		mov	si, OrigDI		;Fetch destination file name
		mov	di, Offset LongPattern
		mov	ds, OrigES
		call	StrCopy
		pop	ds
		call	ShortDest		;Shorten dest long file name
		mov	si, Offset SrcPattern	;Compare the short source and
		mov	di, Offset ShortPattern	; destination file names and
		call	StrComp			; skip changing the short file
		je	@04_Rename		; name if they match
		mov	dx, Offset SrcPattern	;Rename file
		mov	di, Offset ShortPattern
		mov	ah, 56h
		call	ExecDOS
		jc	@01_Rename
@04_Rename:	cmp	NewFileName, False	;If new file names were
		je	@02_Rename		; created then add them to the
		call	AddData			; data base
@02_Rename:	cmp	SrcNameFound, False	;If source file names were
		je	@03_Rename		; found then delete them
		mov	si, Offset SrcPattern	;Restore full short file name
		mov	di, Offset ShortPattern
		call	StrCopy
		mov	si, Offset SrcShortName	;Restore short file name
		mov	di, Offset ShortFName
		call	StrCopy
		mov	si, Offset SrcLongName	;Restore long file name
		mov	di, Offset LongFName
		call	StrCopy
						;Restore file positions
		mov	ax, word ptr SrcLineStart[0]
		mov	dx, word ptr SrcLineStart[2]
		mov	word ptr LineStart[0], ax
		mov	word ptr LineStart[2], dx
		mov	ax, word ptr SrcLineEnd[0]
		mov	dx, word ptr SrcLineEnd[2]
		mov	word ptr LineEnd[0], ax
		mov	word ptr LineEnd[2], dx
		call	DelData			;Delete file names
@03_Rename:	inc	RequestOK		;Set request valid flag
@01_Rename:	retn

;Expand a relative file name into an absolute one, resolving references to
;  '.', '..' and '...'
;  Input : DS:SI: the relative file name
;          ES:DI: the destination absolute file name
Expand:		cld
		push	di			;Save address of destination
		lodsw
		call	UpCase
		cmp	al, 'A'			;If first two characters
		jb	@01_Expand		; resemble a drive, copy it
		cmp	al, 'Z'			; into destination file name
		ja	@01_Expand
		cmp	ah, ':'
		je	@02_Expand
@01_Expand:	sub	si, 2			;Go back to the beginning
		mov	ah, 19h			;Get current drive
		int	21h
		add	al, 'A'			;Convert number to letter and
		mov	ah, ':'			; append a colon
@02_Expand:	stosw				;Store drive letter and colon
		mov	ah, al			;If the relative file name
		mov	al, [si]		; starts with a slash or a
		call	IsSlash			; backslash then no need to
		je	@03_Expand		; get current directory
		sub	ah, 'A' - 1		;Convert letter to number
		mov	dl, ah
		mov	al, '\'			;Store backslash for the root
		stosb				; directory
		push	si
		mov	si, di
		mov	ah, 47h			;Get current directory, skip
		call	ExecDOS			; result if an error occurred
		pop	si
		jc	@03_Expand
		xor	al, al			;Search for the end of the
		mov	cx, -1			; current directory
		repne	scasb
		dec	di
		cmp	byte ptr [di][-1], '\'	;Do not append a backslash
		je	@03_Expand		; when in the root directory
		mov	al, '\'			;Append a backslash to the
		stosb				; current directory
@03_Expand:	call	StrCopy			;Append the relative file name
		pop	si			; to the absolute file name
		mov	di, si			; and go back to the beginning
@05_Expand:	lodsb				;Fetch character
		mov	bl, al			;Save character
		or	al, al			;If the complete file name or
		je	@04_Expand		; a component of it ended then
		call	IsSlash			; resolve references
		je	@04_Expand
@11_Expand:	stosb				;Store character
		jmp	@05_Expand		;Go to the next character
@04_Expand:	cmp	word ptr [di][-1], '.'	;If the last character is no
		jne	@06_Expand		; dot then no references
		mov	al, [di][-2]		;If last but one character is
		call	IsSlash			; a slash or a backslash then
		jne	@07_Expand		; resolve reference to '.'
		sub	di, 2
		jmp	@06_Expand
@07_Expand:	cmp	al, '.'			;If second last character is
		jne	@06_Expand		; no dot then no references
		mov	al, [di][-3]		;If third last character is a
		call	IsSlash			; slash or a backslash then
		jne	@08_Expand		; resolve reference to '..'
		sub	di, 3
		mov	ah, 1			;Strip off only one component
@10_Expand:	cmp	byte ptr [di][-1], ':'	;Skip stripping off the last
		je	@06_Expand		; component if there is none
@09_Expand:	dec	di
		mov	al, [di]		;If last character is neither
		call	IsSlash			; a slash or a backslash then
		jne	@09_Expand		; continue backwards
		dec	ah			;Go to next component
		jne	@10_Expand
		jmp	@06_Expand
@08_Expand:	cmp	al, '.'			;If third last character is no
		jne	@06_Expand		; dot then no references
		mov	al, [di][-4]		;If fourth last character is a
		call	IsSlash			; slash or a backslash then
		jne	@08_Expand		; resolve reference to '...'
		sub	di, 4
		mov	ah, -1			;Strip off all components
		jmp	@10_Expand
@06_Expand:	mov	al, bl			;Restore character
		or	al, al			;If the end of file name was
		jne	@11_Expand		; not reached then continue
		cmp	byte ptr [di][-1], ':'	;Append a backslash for the
		jne	@12_Expand		; root directory if no
		mov	al, '\'			; components at all
		stosb
@12_Expand:	xor	al, al			;End destination file name
		stosb
		retn

;Convert file name
;  Input : AX: function code (7160h)
;          CL: subfunction code (0 for fetching true name, 1 for conversion to
;              short name, 2 for conversion to long name
;          CH: SUBST expansion flag (00h for replacing paths on SUBST'ed
;              drives with the real drive and path, 80h for keeping SUBST'ed
;              drives)
;          DS:SI: the original file name
;          ES:DI: the destination file name
Convert:	cmp	OrigCL, 3		;Fetch subfunction code and
		jae	@01_Convert		; end if invalid
		push	ds			;Fetch original file name
		mov	si, OrigSI
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		mov	si, Offset LongPattern	;Convert original file name
		mov	di, Offset ShortPattern	; into short form
		push	di
		xor	al, al
		call	ProcName
		pop	si
		mov	di, Offset ProcFullName	;Expand relative file name
		push	di			; into an absolute one
		call	Expand
		pop	si
		test	OrigCH, 80h		;Skip SUBST expansion if not
		jne	@03_Convert		; needed
		mov	al, byte ptr si[0]	;Fetch the drive letter of the
		call	UpCase			; absolute path and skip SUBST
		sub	al, 'A'			; expansion if drive letter
		cmp	al, LastDrive		; invalid
		jae	@03_Convert
		mov	bx, si			;Cut the drive letter off the
		add	bx, 3			; path
		mov	ah, CurDirStrucSize
		mul	ah
		les	si, CurDirStrucs	;Go to the current directory
		add	si, ax			; structure of the drive
		push	ds
		push	es
		pop	ds
		pop	es
		mov	cx, [si].CurDirStruc.cdRootDirLen
		inc	cx			;Get the length of the path
		mov	di, Offset ShortPattern	; representing the root
		push	di			; directory of SUBST'ed drive
		call	StrNCopy
		push	es
		pop	ds
		pop	si
		mov	di, si			;Append the path into the
		push	si			; SUBST'ed drive to the real
		call	AddToPath		; path of the root directory
		pop	si
@03_Convert:	cmp	OrigCL, 1
		je	@02_Convert
		mov	di, Offset LongPattern	;Convert true short file name
		push	di			; into long form, if needed
		mov	al, True
		call	ProcName
		pop	si
@02_Convert:	mov	di, OrigDI		;Store long file name
		mov	es, OrigES
		call	StrCopy
		inc	RequestOK		;Set request valid flag
@01_Convert:	retn

;Open file (all purpose)
;  Input : AX: function code (716Ch)
;          BX: access mode and sharing flags (bits 8-9 not supported)
;          CX: file attributes
;          DX: open mode (bit 0 for open file if exists, bit 1 for truncate
;              file if exists, bit 4 for create file in does not exist)
;          DI: alias hint, requested number at the end of short file name
;              (taken into account only if bit 10 in BX is set)
;          DS:SI: the file name
;  Output: CF: when not zero, an error occured
;          AX: file handle (if CF zero) or error code (if CF not zero)
;          CX: action taken (1 for opened file, 2 for created file, 3 for
;              replaced file)
OpenFile:	push	ds			;Fetch long file name
		mov	si, OrigSI
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		xor	al, al
		test	OrigBX, 0400h		;If an alias hint was
		je	@03_OpenFile		; requested by the client then
		inc	al			; set flag and store alias
		mov	bx, OrigDI		; hint
		mov	AliasHint, bx
@03_OpenFile:	mov	AliasHintReq, al
		call	Shorten			;Shorten long file name
		mov	bx, OrigBX
		mov	cx, OrigCX
		mov	dx, OrigDX
		call	ExtOpen			;Call DOS to open file and end
		jc	@01_OpenFile		; if an error occured
		mov	OrigCX, cx		;Store action taken
		cmp	cx, 2			;If a new file was created and
		jne	@02_OpenFile		; it is not a device then add
		cmp	NewFileName, False	; the new file names to the
		je	@02_OpenFile		; data base
		cmp	NewFileDevice, False
		jne	@02_OpenFile
		call	AddData
@02_OpenFile:	inc	RequestOK		;Set request valid flag
@01_OpenFile:	retn

;Get volume information
;  Input : AX: function code (71A0h)
;          DS:DX: the root directory designator
;          ES:DI: the destination file system string (always set to "FAT")
;          CX: the length of the file system string buffer
;  Output: CF: when not zero, an error occured
;          AX: error code, if CF not zero
;          BX: file system flags (always set to 4006h)
;          CX: maximum length of file names (always set to 255)
;          DX: maximum length of path names (always set to 260)
VolumeInfo:	mov	cx, OrigCX		;Fetch buffer length
		mov	OrigAX, 0		;Set return code
		mov	OrigBX, 4006h		;Set file system flags
		mov	OrigCX, 255		;Set maximum file name length
		mov	OrigDX, 260		;Set maximum path name length
		jcxz	@01_VolumeInfo
		mov	es, OrigES		;Copy file system designator
		mov	si, Offset FileSystem	; into buffer
		mov	di, OrigDI
		call	StrNCopy
@01_VolumeInfo:	inc	RequestOK		;Set request valid flag
		retn

;Close file search
;  Input : AX: function code (71A1h)
;          BX: search handle received from FindFirst
;  Output: CF: when not zero, an error occured
;          AX: error code
FindClose:	call	GetSearch		;Fetch search handle and if
		jc	@01_FindClose		; valid, free search record
		mov	byte ptr [bx].SearchRec.srUsed, False
		inc	RequestOK		;Set request valid flag
@01_FindClose:	retn

;Convert file date stamps (a quick hack, does no conversion at all)
;  Input : AX: function code (71A7h)
;          BL: subfunction code (0 for conversion from UTC to DOS, 1 for DOS
;              to UTC
;          BH: the hundredths of seconds, if BL=1
;          CX: the original DOS file time, if BL=1
;          DX: the original DOS file date, if BL=1
;          DS:SI: the original UTC, if BL=0
;          ES:DI: the destination UTC, if BL=1
;  Output: CF: when not zero, an error occured
;          AX: error code
;          BH: the hundredths of seconds, if BL=0
;          CX: the original DOS file time, if BL=0
;          DX: the original DOS file date, if BL=0
ConvTime:	mov	al, OrigBL		;Fetch subfunction code and
		cmp	al, 2			; end if invalid
		jae	@01_ConvTime
		cmp	al, 1
		je	@02_ConvTime
		mov	es, OrigES		;Load original UTC and return
		mov	si, OrigSI		; it unmodified
		mov	ax, es:[si]
		mov	OrigCX, ax
		mov	ax, es:[si][2]
		mov	OrigDX, ax
		mov	OrigBH, 0		;Zero hundredths of seconds
		jmp	@03_Convtime
@02_ConvTime:	mov	es, OrigES		;Load DOS date stamp and
		mov	si, OrigDI		; return it unmodified
		mov	ax, OrigCX
		mov	es:[si], ax
		mov	ax, OrigDX
		mov	es:[si][2], ax
		xor	ax, ax			;Zero higher half of UTC
		mov	es:[si][4], ax
		mov	es:[si][6], ax
@03_ConvTime:	inc	RequestOK		;Set request valid flag
@01_ConvTime:	retn

;Copy part of a file name into a directory entry or FCB
;  Input : CX: the number of characters to copy at most
;          DS:SI: the file name part
;          ES:DI: the current byte of the FCB file name
FCBStrCopy:	cld
@02_FCBStrCopy:	lodsb				;Fetch a character
		or	al, al			;If the file name part has
		jne	@01_FCBStrCopy		; ended then pad the remainder
		mov	al, ' '			; with spaces
		dec	si
@01_FCBStrCopy:	stosb
		loop	@02_FCBStrCopy
		retn

;Generate a short alias for a long file name
;  Input : AX: function code (71A8h)
;          DL: character set of original file name and alias (not supported,
;              assumed to be 11h)
;          DH: format of the alias (0 for directory entry/FCB form, 1 for 8.3
;              DOS file name form)
;          DS:SI: the original file name
;          ES:DI: the generated alias
;  Output: CF: when not zero, an error occured
;          AX: error code
GenShort:	cmp	OrigDH, 2		;Fetch alias format and end if
		jae	@01_GenShort		; invalid
		push	ds			;Fetch long file name
		mov	si, OrigSI
		mov	di, Offset LongPattern
		mov	ds, OrigDS
		call	StrCopy
		pop	ds
		mov	si, Offset LongPattern	;Shorten and correct long name
		call	CorrName
		cmp	OrigDH, 0
		jne	@02_GenShort
		mov	di, OrigDI
		mov	es, OrigES
		mov	si, Offset ShortFName	;Store the name part of the
		mov	cx, SFNNameLen		; alias
		call	FCBStrCopy
		mov	si, Offset ShortFExt	;Store the extension part of
		mov	cx, SFNExtLen		; the alias
		call	FCBStrCopy
		jmp	@03_GenShort
@02_GenShort:	xor	bp, bp			;If DOS file name format
		call	GenAlias		; specified then generate
		mov	si, Offset ProcFName	; alias file name and store it
		mov	di, OrigDI
		mov	es, OrigES
		call	StrCopy
@03_GenShort:	inc	RequestOK		;Set request valid flag
@01_GenShort:	retn

;Long file name function dispatcher routine
;  Input : AH: long file name function code (71h)
;          AL: subfunction code
;  Output: CF: when not zero, an error occured
;          AX: error code
Core:		cmp	ReqDropSearch, False	;Drop search records if flag
		je	@01_Core		; set
		mov	ReqDropSearch, False	;Clear drop records flag
		xor	ax, ax			;Free all search records
		call	DropSearch
@01_Core:	mov	di, Offset FunctionCodes
		mov	si, di
		mov	cx, FunctionNum
		mov	al, OrigAL
		cld				;Search for function code
		repne	scasb
		je	@02_Core
		mov	OrigAL, 0		;Return error code
		retn
@02_Core:	dec	di			;Fetch address of routine
		sub	di, si
		shl	di, 1
		jmp	Functions[di]		;Execute routine

WhiteSpace	db	' ', 0			;White space
WhiteSpace2	db	' "', 0			;White space with a quote mark
EndOfLine	db	chCR, chLF, 0		;End of line
AllFiles	db	'*.*', 0		;All the files
DataBaseName	db	'LONGNAME.DAT', 0	;Name of data base file
FileSystem	db	'FAT', 0		;File system designator
SetverNames:	db	3, 'ARJ'		;Programs needing a false DOS
		db	4, '4DOS'		; and Windows version number
		db	0			;End of problem program list
						;Function codes
FunctionCodes	db	00Dh, 039h, 03Ah, 03Bh, 03Ch, 03Dh, 041h, 043h
		db	047h, 04Eh, 04Fh, 056h, 05Bh, 060h, 06Ch, 0A0h
		db	0A1h, 0A7h, 0A8h
FunctionNum	equ	($ - FunctionCodes)	;Number of functions
						;Address of routines
Functions	dw	Flush, MkDir, RmDir, ChDir
		dw	CreateFile, OpenFile2, Delete, Attrib
		dw	CurDir, FindFirst, FindNext, Rename
		dw	CreateFile, Convert, OpenFile, VolumeInfo
		dw	FindClose, ConvTime, GenShort

DataStart:

;Original address of hooked interrupts
OldInt1B	dd	?			;Original address of INT 1B
OldInt21	dd	?			;Original address of INT 21
OldInt23	dd	?			;Original address of INT 23
OldInt24	dd	?			;Original address of INT 24
OldInt2F	dd	?			;Original address of INT 25

;Variable area
VFATSupport	db	?			;VFAT support enabled flag
Active		db	?			;Resident module active flag
RequestOK	db	?			;Request valid flag
ReqDropSearch	db	?			;Drop records flag
EndOfDir	db	?			;Flag for the end of data base
FileFound	db	?			;Match found flag
NameFound	db	?			;Name found in data base flag
DirFound	db	?			;Directory found flag
SrcNameFound	db	?			;Source name found flag
AliasHintReq	db	?			;Alias hint requested flag
NewFileName	db	?			;New file name created flag
NewFileDevice	db	?			;New file is a device flag
NewSearch	db	?			;New search in process flag
CtrlBreak	db	?			;Ctrl-Break pressed flag
FirstPathChar	db	?			;Position of first path char
ConvToLong	db	?			;Direction of file name conv
LFNFuncCall	db	?			;LFN function call flag
LastDrive	db	?			;Number of drive letters
LFNEntryNum	db	?			;Number of LFN entries
LFNEntriesOK	db	?			;Previous LFN entries OK flag
LFNDrive	db	?			;Current drive number
LFNCheckSum	db	?			;Checksum of short file name
SearchHandle	dw	?			;Current search handle
DataPos		dw	?			;Position in data buffer
DataLen		dw	?			;Length of data buffer
SourceOfs	dw	?			;Source file name offset
DestOfs		dw	?			;Destination file name offset
AliasHint	dw	?			;Alias hint passed by client
AbsOrigAX	dw	?			;Absolute original AX
DirCluster	dw	?			;Directory cluster number
DirSector	dw	?			;Sectors into dir cluster
DirOffset	dw	?			;Offset into directory sector
FileCluster	dw	?			;First cluster of current file
LineStart	dd	?			;Offset of first character
LineEnd		dd	?			;Offset of last character
SrcLineStart	dd	?			;Offset of first source char
SrcLineEnd	dd	?			;Offset of last source char
CurDirStrucs	dd	?			;Current directory structures
DeviceChain	dd	?			;Device driver chain
DirAbsSector	dd	?			;Absolute dir sector number
FATAbsSector	dd	?			;Absolute FAT sector number
OrigDTA2	dd	?			;Original DTA for init disk

;Storage area for original values of registers upon entering resident module
OrigProcessID	dw	?			;Original process ID
OrigDS		dw	?			;Original registers
OrigES		dw	?
OrigSI		dw	?
OrigDI		dw	?
OrigBP		dw	?
OrigAX		label	word
OrigAL		db	?
OrigAH		db	?
OrigBX		label	word
OrigBL		db	?
OrigBH		db	?
OrigCX		label	word
OrigCL		db	?
OrigCH		db	?
OrigDX		label	word
OrigDL		db	?
OrigDH		db	?
OrigSS		dw	?
OrigSP		dw	?
OrigDTA		dd	?			;Original address of DTA
OrigDOSPars	dw	11 dup (?)		;Original DOS swap area

DataEnd:

;String variable area
FindPattern	db	ShortNameLen + 1 dup (?)
ShortPattern	db	ShortNameLen + 1 dup (?)
ShortPath	db	ShortNameLen + 1 dup (?)
ShortFName	db	ShortNameLen + 1 dup (?)
ShortFExt	db	ShortNameLen + 1 dup (?)
DataFullName	db	ShortNameLen + 1 dup (?)
SrcPattern	db	ShortNameLen + 1 dup (?)
SrcShortName	db	ShortNameLen + 1 dup (?)
ProcFullName	db	LongNameLen + 1 dup (?)
ProcFName	db	LongNameLen + 1 dup (?)
LongPattern	db	LongNameLen + 1 dup (?)
LongPath	db	LongNameLen + 1 dup (?)
LongFName	db	LongNameLen + 1 dup (?)
SrcLongName	db	LongNameLen + 1 dup (?)

;Stack for resident module
TSRStack	db	StackSize dup (?)
TSRStackEnd:

						;Internal search records
SearchRecs	db	(SearchRecNum * SearchRecSize) dup (?)

BufferArea:
;!!! DEBUG !!!
;DeviceParams		DeviceParBlock <?>	;Device parameter block
;!!! DEBUG !!!
DriveParams		DriveParBlock <?>	;Drive parameter block
TransferBlock		DiskXferBlock <?>	;Disk transfer block
SFNSearch		ShortSearchRec <?>	;SFN search record
FATBuffer		SectorBuffer <?>	;Buffer for FAT sector
DirBuffer		SectorBuffer <?>	;Buffer for directory sector
IntSearchRec		SearchRec <?>		;Extra internal search record

		org	BufferArea
DataBuffer	db	DataBufferSize dup (?)	;Buffer for data base file

ResidentEnd:

		org	DataEnd

Hello		db	'Star LFN 0.31 beta by Joe Forster/STA', chCR, chLF, chCR, chLF, '$'
Usage		db	'This program emulates Windows'' long file name functions under plain DOS.', chCR, chLF, chCR, chLF
		db	'Install:   STARLFN [-|/]I[D|F]', chCR, chLF
		db	'Uninstall: STARLFN [-|/]U', chCR, chLF, '$'
OldDOS		db	'This program needs DOS 4.0 or later', chCR, chLF, '$'
NoMemory	db	'Out of memory', chCR, chLF, '$'
InvalidOption	db	'Invalid option', chCR, chLF, '$'
BeginSentence	db	'Star LFN $'
CannotUninst	db	'cannot be uninstalled - you''ve installed some programs after it.', chCR, chLF, '$'
Installed	db	'has been installed.', chCR, chLF, '$'
AlreadyInst	db	'has already been installed.', chCR, chLF, '$'
NotYetInst	db	'has not been installed yet.', chCR, chLF, '$'
Uninstalled	db	'has been uninstalled.', chCR, chLF, '$'

;Write the first and second half of a sentence
;  Input : DS:DX: second half of the sentence
WriteSentence:	push	dx
		mov	dx, Offset BeginSentence
		call	WriteStr
		pop	dx

;Write a string onto the screen
;  Input : DS:DX: second half of the sentence
WriteStr:	mov	ah, 9
		int	21h
		retn

;Check if an interrupt points to the resident module
;  Input : AL: interrupt number
;          ES: segment of resident module
;  Output: ZF: when zero, the interrupt points to the resident module
CheckInt:	push	es
		mov	ah, 35h			;Get interrupt vector
		int	21h
		mov	bx, es
		pop	es
		mov	dx, es			;Compare segment of interrupt
		cmp	dx, bx			; with that of resident module
		retn

;Check if the program is already installed
;  Output: ZF: when zero, the program is not installed yet
IsInstalled:	xor	ax, ax			;Check for resident module
		mov	es, ax
		mov	bx, 'ST'
		mov	cx, 'AL'
		mov	dx, 'FN'
		int	2Fh
		push	cs			;Restore data segment
		pop	ds
		cmp	cx, 'OK'
		retn

;Entry point
Start:		mov	bx, (ProgramSize + 15) / 16
		mov	ah, 4Ah			;Expand allocated area and
		int	21h			; setup stack pointer
		mov	dx, Offset NoMemory	; accordingly or display an
		jc	@07			; error message if out of
		mov	sp, ProgramSize		; memory
		xor	ax, ax
		push	ax
		mov	di, Offset DataStart	;Clear data area
		mov	cx, DataEnd - DataStart
		xor	al, al
		cld
		rep	stosb
		mov	dx, Offset Hello	;Display startup message
		mov	ah, 9
		int	21h
		xor	bh, bh			;Command is to do nothing
		mov	cl, CommandLineLen	;Process the command line and
		xor	ch, ch			; search for options
		jcxz	@15			;Skip if command line is empty
		mov	si, Offset CommandLine
		cld
@10:		lodsb				;Fetch next character and
		call	UpCase			; convert it to uppercase
		cmp	al, chCR		;End search if the end of the
		je	@02			; command line reached
		cmp	al, ' '			;Skip spaces, Tabs, hyphens
		je	@08			; and slashes
		cmp	al, chTab
		je	@08
		cmp	al, '-'
		je	@08
		cmp	al, '/'
		je	@08
		cmp	al, 'D'			;Option 'D': disk access
		jne	@16
		mov	VFATSupport, True
		jmp	@08
@16:		cmp	al, 'F'			;Option 'F': file access
		jne	@17
		mov	VFATSupport, False
		jmp	@08
@17:		cmp	al, 'I'			;Option 'I': install program
		jne	@18
		mov	bl, icInstall
		jmp	@08
@18:		cmp	al, 'U'			;Option 'U': uninstall program
		jne	@14
		mov	bl, icUninstall
		jmp	@08
@08:		loop	@10
		jmp	@15
@14:		mov	dx, Offset InvalidOption
		call	WriteStr
		jmp	@06
@15:		cmp	bl, icInstall
		je	@02
		cmp	bl, icUninstall
		je	@12
		mov	dx, Offset Usage	;Display usage information
		call	WriteStr		; and exit
		jmp	@06
@12:		call	IsInstalled		;Check if program is already
		je	@11			; installed and display an
		mov	dx, Offset NotYetInst	; error message if it is not
		jmp	@01
@11:		mov	es, bx			;Check for interrupts if
		mov	al, 21h			; resident module present
		call	CheckInt
		jne	@03
		mov	al, 2Fh
		call	CheckInt
		je	@04
@03:		mov	dx, Offset CannotUninst	;Print error message if cannot
		jmp	@01			; uninstall resident module
@04:		push	ds
		lds	dx, es:OldInt21		;Restore original DOS
		mov	ax, 2521h		; dispatcher interrupt
		int	21h
		lds	dx, es:OldInt2F		;Restore original multiplex
		mov	ax, 252Fh		; interrupt
		int	21h
		pop	ds
		mov	ah, 49h			;Free memory of resident
		int	21h			; module
		push	cs
		pop	ds
		mov	dx, Offset Uninstalled	;Print uninstallation message
		jmp	@01
@02:		call	IsInstalled		;Check if program is already
		jne	@13			; installed and display an
		mov	dx, Offset AlreadyInst	; error message if it is
		jmp	@01
@13:		mov	ah, 30h			;Get DOS version number
		int	21h
		xchg	al, ah
		cmp	ax, 0400h		;Check for DOS version number,
		jae	@05			; it has to be 4.0 or above
		mov	dx, Offset OldDOS	;Print error message if below
@07:		stc
		call	WriteStr
		jmp	@06
@05:		mov	es, word ptr cs:[002Ch]	;Free environment segment
		mov	ah, 49h
		int	21h
		mov	ax, 3521h		;Save original DOS dispatcher
		int	21h			; interrupt
		mov	word ptr OldInt21[0], bx
		mov	word ptr OldInt21[2], es
		mov	ax, 352Fh		;Save original multiplex
		int	21h			; interrupt
		mov	word ptr OldInt2F[0], bx
		mov	word ptr OldInt2F[2], es
		mov	dx, Offset NewInt21	;Setup own DOS dispatcher
		mov	ax, 2521h		; interrupt
		int	21h
		mov	dx, Offset NewInt2F	;Setup own multiplex interrupt
		mov	ax, 252Fh
		int	21h
		mov	ah, 52h			;Fetch the number of drives,
		int	21h			; the address of the device
		mov	al, es:[bx][21h]	; driver chain and the address
		mov	si, bx			; of the current directory
		add	si, 22h			; structures
		mov	word ptr DeviceChain[0], si
		mov	word ptr DeviceChain[2], es
		les	bx, es:[bx][16h]
		mov	LastDrive, al
		mov	word ptr CurDirStrucs[0], bx
		mov	word ptr CurDirStrucs[2], es
		mov	ReqDropSearch, True	;Set drop search records flag
		mov	dx, Offset Installed	;Print installation message
		call	WriteSentence
		mov	dx, Offset ResidentEnd[1]
		int	27h			;Go resident
@01:		call	WriteSentence		;Display a sentence and end
@06:		mov	ax, 4C00h		; execution
		int	21h

MainEnd:

CSeg		ends

		end	Main
