{
Ŀ
                 Joe Forster/STA                 
                                                 
                   STARBASE.PAS                  
                                                 
             Star Utilities base unit            

}

unit StarBase;

{$A+,B-,D+,E-,F-,G+,I-,L+,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y+}

interface

uses
  DOS;

type
{Option processor procedure type}
  TOptProc      = procedure(const Option: string);
{Table for conversion between different character sets}
  TConvTable    = array [0..255] of Byte;
  PConvTable    = ^TConvTable;

const
{Version string}
  VersionStr    = ' 0.83 (2010-01-11)';
{Copyright string}
  CopyrightStr  = ' by Joe Forster/STA';
{File open modes}
  fmReadOnly    = 0;
  fmWriteOnly   = 1;
  fmReadWrite   = 2;
{Panel modes}
  pmDOS         = 0;
  pmExt         = 1;
  pmDisk        = 2;
  pmTape        = 3;
  pmFile        = 4;
  pmInfo        = 5;
  pmLynx        = 6;
  pmArkive      = 7;
  pmLibrary     = 8;
  pmTAR         = 9;
  pmLHA         = 10;
  pmLHASFX      = 11;
  pmARC         = 12;
  pmARCSDA      = 13;
  pmFileZip     = 14;
{Question prompt modes}
  aaAsk         = 0;
  aaAllYes      = 1;
  aaAllNo       = 2;
  aaRename      = 3;
{Commodore file attributes}
  faDeleted     = 0;
  faSequential  = 1;
  faProgram     = 2;
  faUser        = 3;
  faRelative    = 4;
  faPartition   = 5;
{Disk types}
  dt1541        = $00;
  dt1541Ext     = $01;
  dt1571        = $02;
  dt1581        = $03;
  dt1541Error   = $80;
  dt1541ExtError= $81;
  dt1571Error   = $82;
  dt1581Error   = $83;
{Extended BAM modes}
  xbSpeedDOS    = 0;
  xbDolphinDOS  = 1;
  xbPrologicDOS = 2;
{Number of disk types}
  DiskTypeNum   = 3;
{Maximum number of tracks plus 1 on a normal 1541 disk}
  Max1541Tracks = 36;
{Position of GEOS format signature in the BAM}
  GEOSSignPos   = $AD;
{Offset of directory block count in the Lynx archive header}
  LynxBlockPos  = 95;
{Length of the Lynx archive header}
  LynxHeaderLen = 124;
{Length of the filepacked ZipCode archive directory lister}
  ZipHeaderLen  = 410;
{Size of the LHA self-extractor program}
  LHAExtractLen = 3721;
{Size of disk image buffer}
  DiskBufferSize= 8192;
{Size of copy buffer}
  BufferSize    = 65024;
{Default ID code for disk images}
  DefIDCode     : string[2] = '00';
{Base file name for temporary files}
  TempBaseName  = '~su~tmp~';
{Blank string to clear a line}
  BlankLine     = '                                                                               ';
{Signature of GEOS-formatted disks}
  GEOSSign      : string[16] = 'GEOS format V1.0';
{CBM extensions}
  ShortCBMExt   : array [0..7] of string[3] = ('del', 'seq', 'prg', 'usr', 'rel', 'cbm', '???', '???');
{Disk image extensions}
  DiskExt       : array [0..DiskTypeNum - 1] of string[3] = ('d64', 'd71', 'd81');
{PETSCII lowercase/uppercase to ASCII converter table}
  PETtoASCLower : TConvTable = (
    $40, $61, $62, $63, $64, $65, $66, $67, $68, $69, $6A, $6B, $6C, $6D, $6E, $6F,
    $70, $71, $72, $73, $74, $75, $76, $77, $78, $79, $7A, $5B, $9C, $5D, $18, $1B,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $40, $61, $62, $63, $64, $65, $66, $67, $68, $69, $6A, $6B, $6C, $6D, $6E, $6F,
    $70, $71, $72, $73, $74, $75, $76, $77, $78, $79, $7A, $5B, $9C, $5D, $18, $1B,
    $C4, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $C5, $B0, $B3, $B0, $B1,
    $C4, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $C5, $B0, $B3, $B0, $B1,
    $5F, $DD, $DC, $C4, $C4, $B3, $B0, $B3, $B0, $B0, $B3, $C3, $DA, $C0, $BF, $C4,
    $DA, $C1, $C2, $B4, $B3, $DD, $DE, $C4, $DF, $DC, $FB, $BF, $C0, $D9, $D9, $C5,
    $C4, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $C5, $B0, $B3, $B0, $B1,
    $5F, $DD, $DC, $C4, $C4, $B3, $B0, $B3, $B0, $B0, $B3, $C3, $DA, $C0, $BF, $C4,
    $DA, $C1, $C2, $B4, $B3, $DD, $DE, $C4, $DF, $DC, $FB, $BF, $C0, $D9, $D9, $B0);
{PETSCII uppercase/graphics to ASCII converter table}
  PETtoASCUpper : TConvTable = (
    $40, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $5B, $9C, $5D, $18, $1B,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $40, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $5B, $9C, $5D, $18, $1B,
    $C4, $06, $B3, $C4, $C4, $C4, $C4, $B3, $B3, $BF, $C0, $D9, $C0, $5C, $2F, $DA,
    $BF, $6F, $C4, $03, $B3, $DA, $58, $6F, $05, $B3, $04, $C5, $B0, $B3, $E3, $5C,
    $C4, $06, $B3, $C4, $C4, $C4, $C4, $B3, $B3, $BF, $C0, $D9, $C0, $5C, $2F, $DA,
    $BF, $6F, $C4, $03, $B3, $DA, $58, $6F, $05, $B3, $04, $C5, $B0, $B3, $E3, $5C,
    $5F, $DD, $DC, $C4, $C4, $B3, $B0, $B3, $B0, $2F, $B3, $C3, $DA, $C0, $BF, $C4,
    $DA, $C1, $C2, $B4, $B3, $DD, $DE, $C4, $DF, $DC, $D9, $BF, $C0, $D9, $D9, $C5,
    $C4, $06, $B3, $C4, $C4, $C4, $C4, $B3, $B3, $BF, $C0, $D9, $C0, $5C, $2F, $DA,
    $BF, $6F, $C4, $03, $B3, $DA, $58, $6F, $05, $B3, $04, $C5, $B0, $B3, $E3, $5C,
    $5F, $DD, $DC, $C4, $C4, $B3, $B0, $B3, $B0, $2F, $B3, $C3, $DA, $C0, $BF, $C4,
    $DA, $C1, $C2, $B4, $B3, $DD, $DE, $C4, $DF, $DC, $D9, $BF, $C0, $D9, $D9, $E3);
{ASCII to PETSCII converter table}
  ASCtoPET      : TConvTable = (
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $5E, $20, $20, $5F, $20, $20, $20, $20,
    $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2C, $2D, $2E, $2F,
    $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $3A, $3B, $3C, $3D, $3E, $3F,
    $40, $C1, $C2, $C3, $C4, $C5, $C6, $C7, $C8, $C9, $CA, $CB, $CC, $CD, $CE, $CF,
    $D0, $D1, $D2, $D3, $D4, $D5, $D6, $D7, $D8, $D9, $DA, $5B, $CD, $5D, $5E, $A0,
    $20, $41, $42, $43, $44, $45, $46, $47, $48, $49, $4A, $4B, $4C, $4D, $4E, $4F,
    $50, $51, $52, $53, $54, $55, $56, $57, $58, $59, $5A, $5B, $DD, $5D, $2D, $20,
    $C3, $55, $45, $41, $41, $41, $41, $43, $45, $45, $45, $49, $49, $49, $C1, $C1,
    $C5, $41, $C1, $4F, $4F, $4F, $55, $55, $59, $CF, $D5, $43, $5C, $D9, $20, $20,
    $41, $45, $4F, $55, $4E, $CE, $20, $20, $3F, $20, $20, $20, $20, $21, $20, $20,
    $A6, $A6, $A6, $DD, $B3, $B3, $B3, $AE, $AE, $B3, $A6, $AE, $BD, $BD, $BD, $AE,
    $AD, $B1, $B2, $AB, $C0, $DB, $AB, $AB, $AD, $B0, $B1, $B2, $AB, $C0, $DB, $B1,
    $B1, $B2, $B2, $AD, $AD, $B0, $B0, $DB, $DB, $BD, $B0, $20, $A2, $A1, $B6, $B8,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20,
    $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20, $BA, $20, $20, $20, $20);

type
  PString       = ^string;
{CBM-style block}
  TBlock        = array [0..255] of Byte;
  PBlock        = ^TBlock;
{Copy buffer (63.5 KBytes)}
  TBuffer       = array [0..BufferSize - 1] of Byte;
  PBuffer       = ^TBuffer;
{Disk image buffer (8 KBytes)}
  TDiskBuffer   = array [0..DiskBufferSize - 1] of Byte;
  PDiskBuffer   = ^TDiskBuffer;
{Directory entry}
  TDirEntry     = record
    Name        : string[16];
    Attr,
    ExtAttr     : Byte;
    Size        : Longint;
    case Union: Integer of
      pmDOS:    (Time: Longint);
      pmDisk:   (Track: Byte;
                 Sector: Byte;
                 SideTrack: Byte;
                 SideSector: Byte);
      pmTape,
      pmFile,
      pmLynx,
      pmArkive,
      pmTAR:     (Start: Word);
  end;
{ARC directory entry}
  TARCEntry     = record
    Version     : Byte;
    Method      : Byte;
    CheckSum    : Word;
    OrigSize    : Longint;
    Blocks      : Word;
    FileType    : Char;
    Name        : string;
    RecordLen   : Byte;
    Date        : Word;
  end;
{LHA directory entry}
  TLHAEntry     = record
    CheckSum    : Byte;
    Method      : array [0..4] of Char;
    PackSize    : Longint;
    OrigSize    : Longint;
    Date        : Longint;
    Attr        : Word;
    Name        : string;
  end;
{Extended file record}
  ExtFile       = record
    Orig        : file;
    LongName    : string;
  end;
  PExtFile      = ^ExtFile;
{Extended search record}
  ExtSearchRec  = record
    Orig        : SearchRec;
    HandleUsed  : Boolean;
    LongHandle  : Word;
    LongName    : string;
  end;
{Windows'95-style search record}
  LongSearchRec = record
    Attr        : Longint;
    Dummy1      : array [1..16] of Byte;
    Time        : Longint;
    Dummy2      : array [1..8] of Byte;
    Size        : Longint;
    Dummy3      : array [1..8] of Byte;
    LongName    : array [1..260] of Byte;
    ShortName   : array [1..14] of Byte;
  end;

var
  ErrorInfo,
  GEOSFormat,
  EscPressed,
  Verbose,
  List,
  Over,
  LineFeed,
  AlwaysYes,
  _End,
  Error,
  LongFileNames : Boolean;
  Confirm,
  Delete,
  Overwrite,
  DirTrack,
  DirTrack2,
  FirstDirSector,
  DiskType,
  MaxTrack,
  ExtBAMMode,
  SectorStep,
  EntryPos,
  Track,
  Sector,
  SideTrack,
  SideSector,
  Number,
  DirPos,
  DirSector,
  Attr,
  Data          : Byte;
  Protected,
  Closed,
  Command,
  FileType      : Char;
  AllBlocks,
  Count,
  BlockNum,
  ByteNum,
  FileNum,
  FileCount,
  DiskSize,
  Block         : Word;
  IOError       : Integer;
  AllSize,
  CopySize,
  FileDate      : Longint;
  Buffer        : PBuffer;
  ConvTable     : PConvTable;
  Ext1,
  Ext2,
  Name1,
  Name2,
  Dir,
  Dir1,
  Dir2,
  SearchPar,
  ArcName,
  DiskName,
  FileName,
  PCName,
  ASCIIName     : string;
  Image,
  TempFile,
  ArcFile       : ExtFile;
  Entry         : ExtSearchRec;
  CBMEntry      : TDirEntry;
  BAM,
  BAM2,
  BAM3,
  DataBuffer,
  DirBuffer     : TBlock;

procedure CommonInit;
function LeadingZero(W: Word; N: Byte): string;
function LeadingSpace(L: Longint; N: Byte): string;
function FileExt(const Name: string): string;
function GetDiskType(L: Longint): Byte;
procedure CheckDiskType;
function GetDiskExt(Disk: Byte): string;
function IsDiskExt(const Ext: string): Boolean;
procedure FixDiskExt(var Ext: string);
function CloneName(Name1, Name2: string): string;
procedure SplitName(const Path: string; var Dir, Name, Ext: string);
procedure FCBMSplit(const Path: string; var Dir, Name, Ext: string);
function ReadKey: Word;
procedure Escape;
function AddToPath(const P, S: string; C: Char): string;
function GetPath(const S: string; C: Char): string;
function CutPath(const S: string; C: Char): string;
function LeftPos(C: Char; const S: string): Byte;
function RightPos(C: Char; const S: string): Byte;
function ShrinkName(const S: string; C: Char): string;
function LoCase(Ch: Char): Char;
function UpperCase(const S: string): string;
function LowerCase(const S: string): string;
function CutChar(const Name: string; C: Char): string;
procedure MakeASCIIName;
procedure MakeName;
function BytesToLongint(B1, B2, B3, B4: Byte): Longint;
function ByteToBlock(B: Longint): Longint;
function SectorNum(Track: Byte): Byte;
function ValidPos(T, S: Byte): Boolean;
function DiskPos(Track, Sector: Byte): Longint;
procedure ReadDiskBlock(T, S: Byte; Buffer: PBlock);
procedure WriteDiskBlock(T, S: Byte; Buffer: PBlock);
function GetBAMOffset(T: Byte; Map: Boolean): Word;
procedure ClearBAM;
procedure MakeBAM;
procedure AllocBlock(T, S: Byte; Alloc: Boolean);
function IsBlockUsed(T, S: Byte): Boolean;
function IsTrackFree(T: Byte): Boolean;
function IsBAMValid: Boolean;
function SearchFreeBlock: Boolean;
function FindNextBlock: Boolean;
function SearchNextBlock: Boolean;
function ReadCBMEntry(var Entry: TDirEntry): Boolean;
procedure ReadBAM;
procedure CheckGEOSFormat(P: PBlock);
function OpenImage(Write: Boolean): Integer;
procedure CloseImage;
function CreateDisk: Integer;
procedure OpenRead;
function OpenWrite(Name: string; AAttr: Byte; Len: Longint; Force: Boolean): Integer;
procedure CloseWrite;
procedure ReadPart(Buffer: PBuffer; var Len: Word);
procedure WritePart(Buffer: PBuffer; Len: Word; FileEnd: Boolean);
procedure ConvertCBMName(var CBMName, PCName: string);
procedure CorrectPCName(var OrigName, NewName: string);
procedure ClrLine;
procedure AskStr(const Text: string; var Str: string; Len: Integer; Min, Max: Char; Upper: Boolean);
function Question(const Text, All, Never, Rename: string; var Answer: Byte): Boolean;
procedure CharSetOptions(const Option: string);
procedure AddOptions(const Option: string);
procedure ExtractOptions(const Option: string);
procedure NoOptions(const Option: string);
procedure ParseCmdLine(Process: TOptProc);
function InitLongNames: Boolean;
function ShortName(const Name: string; Path: Boolean): string;
function LongName(const Name: string; Path: Boolean): string;
function LongParamStr(Index: Byte): string;
procedure LongFSplit(const Path: string; var Dir, Name, Ext: string);
procedure LongErase(Name: string);
procedure LongGetFAttr(const Name: string; var Attr: Word);
procedure LongSetFAttr(const Name: string; Attr: Word);
procedure LongFindFirst(Path: string; Attr: Byte; var F: ExtSearchRec);
procedure LongFindNext(var F: ExtSearchRec);
procedure LongRename(OrigName, NewName: string);
function LongOpenFile(Name: string; var F: ExtFile; Mode: Byte): Integer;
procedure LongFindClose(var F: ExtSearchRec);
function LongFExpand(Path: string): string;
procedure ExtBlockRead(var F: ExtFile; var Buf; Count: Word);
procedure ExtBlockRead2(var F: ExtFile; var Buf; Count: Word; var Result: Word);
procedure ExtBlockWrite(var F: ExtFile; var Buf; Count: Word);
procedure ExtBlockWrite2(var F: ExtFile; var Buf; Count: Word; var Result: Word);
procedure ExtClose(var F: ExtFile);
function ExtEOF(var F: ExtFile): Boolean;
procedure ExtSeek(var F: ExtFile; Pos: Longint);
function ExtFileSize(var F: ExtFile): Longint;
function ExtFilePos(var F: ExtFile): Longint;
procedure ExtTruncate(var F: ExtFile);
procedure ExtGetFTime(var F: ExtFile; var Time: Longint);
procedure ExtSetFTime(var F: ExtFile; Time: Longint);

implementation

{Extended DOS functions}
function ExtFExpand(Path: string): string; far; external;
procedure ExtGetDir(D: Byte; var S: string); far; external;
{$L EXTDOS.OBJ}

{Initialize common variables and fetch command}
procedure CommonInit;
begin
  DiskType := dt1541;
  ExtBAMMode := xbSpeedDOS;
  AlwaysYes := False;
  Error := False;
  Verbose := False;
  LineFeed := False;
  EscPressed := False;
  FileMode := fmReadOnly;
  Confirm := aaAllYes;
  Delete := aaAllNo;
  Overwrite := aaAsk;
  Number := 3;
  Command := #0;
  ConvTable := @PETtoASCLower;
  SearchPar := UpperCase(LongParamStr(1));
  if Length(SearchPar) > 0 then
  begin
    if SearchPar[1] = '-' then SearchPar := Copy(SearchPar, 2, 255);
    if Length(SearchPar) > 0 then Command := SearchPar[1];
  end;
  List := (Command = 'L');
  InitLongNames;
end;

{Convert a word into a decimal string with leading zeros
  Input : W: word to be converted
          N: number of digits to put into the string
  Output: the decimal string}
function LeadingZero(W: Word; N: Byte): string;
var
  S             : string[80];
begin
  Str(W, S);
  while Length(S) < N do S := '0' + S;
  LeadingZero := S;
end;

{Convert a word into a decimal string with leading blanks
  Input : L: long integer to be converted
          N: number of digits to put into the string
  Output: the decimal string}
function LeadingSpace(L: Longint; N: Byte): string;
var
  S             : string;
begin
  Str(L:N, S);
  LeadingSpace := S;
end;

{Get the extension part of the file name
  Input : Name: the file name
  Output: the file extension}
function FileExt(const Name: string): string;
var
  P             : Byte;
begin
  P := RightPos('.', Name);
  if P = 0 then FileExt := '' else FileExt := Copy(Name, P + 1, 255);
end;

{Determine the type of disk image (35 or 40 tracks, with or without error
  info) on basis of the file size
  Input : L: the size of the disk image
  Output: the type of the disk image}
function GetDiskType(L: Longint): Byte;
var
  B             : Byte;
begin
  B := 255;
  if L = 174848 then B := dt1541;
  if L = 174848 + 683 then B := dt1541Error;
  if L = 196608 then B := dt1541Ext;
  if L = 196608 + 768 then B := dt1541ExtError;
  if L = 349696 then B := dt1571;
  if L = 349696 + 1366 then B := dt1571Error;
  if L = 819200 then B := dt1581;
  if L = 819200 + 3200 then B := dt1581Error;
  GetDiskType := B;
end;

{Return the disk size in blocks
  Output: disk size}
function GetDiskSize: Longint;
begin
  case DiskType and $7F of
    dt1541: GetDiskSize := 683;
    dt1541Ext: GetDiskSize := 768;
    dt1571: GetDiskSize := 1366;
    dt1581: GetDiskSize := 3200;
  end;
end;

{Check disk type: 35 or 40 tracks, with or without error info}
procedure CheckDiskType;
begin
  ErrorInfo := (DiskType and $80 > 0);
  FirstDirSector := 1;
  case DiskType and $7F of
    dt1541:
    begin
      DirTrack := 18;
      DirTrack2 := 255;
      MaxTrack := Max1541Tracks;
    end;
    dt1541Ext:
    begin
      DirTrack := 18;
      DirTrack2 := 255;
      MaxTrack := 41;
    end;
    dt1571:
    begin
      DirTrack := 18;
      DirTrack2 := 53;
      MaxTrack := 71;
    end;
    dt1581:
    begin
      DirTrack := 40;
      DirTrack2 := 255;
      MaxTrack := 81;
      FirstDirSector := 3;
    end;
  end;
end;

{Get the extension for a disk image type
  Input : Disk: disk type
  Output: the extension, including the leading dot}
function GetDiskExt(Disk: Byte): string;
var
  B             : Byte;
begin
  B := 255;
  case Disk and $7F of
    dt1541, dt1541Ext: B := 0;
    dt1571: B := 1;
    dt1581: B := 2;
  end;
  if B = 255 then GetDiskExt := '' else GetDiskExt := '.' + DiskExt[B];
end;

{Determine if an extension belongs to a disk image
  Input : Ext: the extension, including the leading dot
  Output: when True, the extension is that of a disk image}
function IsDiskExt(const Ext: string): Boolean;
var
  B             : Byte;
  S             : string;
begin
  B := 0;
  S := Copy(LowerCase(Ext), 2, 255);
  while (B < DiskTypeNum) and (DiskExt[B] <> S) do Inc(B);
  IsDiskExt := (B < DiskTypeNum);
end;

{Check the extension of the disk image and fix it, if needed}
procedure FixDiskExt(var Ext: string);
begin
  if not IsDiskExt(Ext) then Ext := GetDiskExt(dt1541);
end;

{Clone a file name on basis of the original file name and a cloning pattern
  Input : Name1: the original file name
          Name2: the cloning pattern
  Output: the cloned file name}
function CloneName(Name1, Name2: string): string; assembler;
asm
    les bx, Name1;
    mov cl, byte ptr es:[bx];
    xor ch, ch;
    les bx, Name2;
    mov dl, byte ptr es:[bx];
    xor dh, dh;
    mov si, 1;
    mov di, si;
    les bx, @Result;
    mov byte ptr es:[bx], 0;
    inc bx;
@5: cmp di, dx;
    ja @1;
    les bx, Name1;
    mov al, byte ptr es:[bx][si];
    les bx, Name2;
    mov ah, byte ptr es:[bx][di];
    cmp ah, '?';
    jne @2;
    cmp si, cx;
    ja @1;
    inc si;
    inc di;
    jmp @3;
@2: cmp ah, '*';
    jne @4;
    cmp si, cx;
    ja @1;
    inc si;
    jmp @3;
@4: mov al, ah;
    inc si;
    inc di;
@3: les bx, @Result;
    inc byte ptr es:[bx];
    add bl, byte ptr es:[bx];
    adc bh, 0;
    mov byte ptr es:[bx], al;
    jmp @5;
@1:
end;

{Split the full PC file name into dir, name and extension
  Input : Path: the full PC file name
          Dir: the string to contain the path
          Name: the string to contain the name
          Ext: the string to contain the extension}
procedure SplitName(const Path: string; var Dir, Name, Ext: string);
begin
  LongFSplit(Path, Dir, Name, Ext);
  if Name = '' then Name := '*';
  if Ext = '' then Ext := '.*';
  if Ext = '.' then Ext := '';
end;

{Split a file name that contains the CBM device number, the name of the
  file and a file type
  Input : Path: the full file name
          Dir: the string to contain the CBM device number
          Name: the string to contain the file name
          Ext: the string to contain the file type}
procedure FCBMSplit(const Path: string; var Dir, Name, Ext: string); assembler;
asm
    push ds;
    cld;
    lds si, Path;
    lodsb;
    mov dl, al;
    xor dh, dh;
    mov bx, dx;
    or bx, bx;
    je @2;
@1: cmp byte ptr [si][bx][-1], '\';
    je @2;
    cmp byte ptr [si][bx][-1], ':';
    je @2;
    dec bx;
    jne @1;
@2: mov ax, 3;
    les di, Dir;
    call @7;
    xor bx, bx;
    jmp @4;
@3: cmp byte ptr [si][bx], ',';
    je @5;
    inc bx;
@4: cmp bx, dx;
    jne @3;
@5: mov ax, 16;
    les di, Name;
    call @7;
    mov bx, dx;
    mov ax, 2;
    les di, Ext;
    call @7;
    pop ds;
    jmp @8;
@7: sub dx, bx;
    cmp ax, bx;
    jb @6;
    mov ax, bx;
@6: stosb;
    mov cx, ax;
    add bx, si;
    rep movsb;
    mov si, bx;
    retn;
@8:
end;

{Read a keypress from the keyboard}
function ReadKey: Word; assembler;
asm
    xor ah, ah;
    int $16;
end;

{Check if the user wants to stop the current function by pressing Escape}
procedure Escape; assembler;
asm
    xor ah, ah;
    mov EscPressed, ah;
    inc ah;
    int $16;
    je @1;
    cmp ax, $011B;
    jne @1;
    xor ah, ah;
    int $16;
    inc EscPressed;
@1:
end;

{Append a file name to a path
  Input : P: the path
          S: the file name
          C: the character dividing directories
  Output: the fully qualified file name}
function AddToPath(const P, S: string; C: Char): string;
var
  T             : string;
begin
  if P = '' then
  begin
    AddToPath := S;
  end
  else
  begin
    T := P;
    if not (T[Length(T)] in [C, ':']) then T := T + C;
    AddToPath := T + S;
  end;
end;

{Get the path part of a file name
  Input : S: the fully qualified file name
          C: the character dividing directories
  Output: the path part of the file name}
function GetPath(const S: string; C: Char): string;
var
  B             : Byte;
  T             : string;
begin
  B := RightPos(C, S);
  if (B = 0) and (Length(S) >= 2) and (S[2] = ':') then B := 2;
  if (B >= 2) and (S[B] = C) and (S[B - 1] <> ':') then Dec(B);
  T := Copy(S, 1, B);
  GetPath := T;
end;

{Cut the path off of a file name
  Input : S: the fully qualified file name
          C: the character dividing directories
  Output: the file name itself without the path}
function CutPath(const S: string; C: Char): string;
var
  B             : Byte;
begin
  B := RightPos(C, S);
  if (B = 0) and (Length(S) >= 2) and (S[2] = ':') then B := 2;
  CutPath := Copy(S, B + 1, 255);
end;

{Searches for the first instance from the left of a character in a string
  Input : C: the character to search for
          S: the string to search in
  Output: zero, if not found; otherwise, the position of the character}
function LeftPos(C: Char; const S: string): Byte; assembler;
asm
    les di, S;
    mov al, es:[di];
    xor ah, ah;
    mov cx, ax;
    jcxz @1;
    inc di;
    push cx;
    mov al, C;
    cld;
    repne scasb;
    pop ax;
    jne @1;
    sub ax, cx;
    jmp @2;
@1: mov ax, cx;
@2:
end;

{Searches for the last instance from the right of a character in a string
  Input : C: the character to search for
          S: the string to search in
  Output: zero, if not found; otherwise, the position of the character}
function RightPos(C: Char; const S: string): Byte; assembler;
asm
    les di, S;
    mov al, es:[di];
    xor ah, ah;
    mov cx, ax;
    jcxz @1;
    add di, cx;
    mov al, C;
    std;
    repne scasb;
    jne @1;
    inc cx;
@1: mov ax, cx;
end;

{Shrink the file name by stripping the path and keeping only the left part
  and the extension part (at most 3 characters to the right of the
  rightmost dot) of the file name so that it fits into 16 characters
  Input : S: the file name to shrink
          C: the character dividing directories
  Output: the shrinked file name}
function ShrinkName(const S: string; C: Char): string;
var
  P             : Byte;
  E,
  T             : string;
begin
  T := CutPath(S, C);
  E := '';
  if Length(T) > 16 then
  begin
    P := RightPos('.', T);
    if P = 0 then
    begin
      T[0] := #16;
    end
    else
    begin
      E := Copy(T, P, 255);
      T[0] := Chr(P - 1);
      if (Length(T) > 12) and (Length(E) > 4) then E[0] := #4;
      P := 16 - Length(E);
      if Length(T) > P then T[0] := Chr(P);
      T := T + E;
    end;
  end;
  ShrinkName := T;
end;

{Convert a character to lowercase
  Input : Ch: the original character
  Output: the character in lowercase}
function LoCase(Ch: Char): Char; assembler;
asm
    mov al, &Ch;
    cmp al, 'A';
    jb @1;
    cmp al, 'Z';
    ja @1;
    add al, 'a' - 'A';
@1:
end;

{Convert a string to uppercase
  Input : S: the original string
  Output: the string in uppercase}
function UpperCase(const S: string): string;
var
  I             : Integer;
  T             : string;
begin
  T[0] := Chr(Length(S));
  for I := 1 to Length(S) do T[I] := UpCase(S[I]);
  UpperCase := T;
end;

{Convert a string to lowercase
  Input : S: the original string
  Output: the string in lowercase}
function LowerCase(const S: string): string;
var
  I             : Integer;
  T             : string;
begin
  T[0] := Chr(Length(S));
  for I := 1 to Length(S) do T[I] := LoCase(S[I]);
  LowerCase := T;
end;

{Cut the specified character off of the end of the file name
  Input : Name: the original file name
          C: the character to be cut off
  Output: the corrected file name}
function CutChar(const Name: string; C: Char): string;
var
  S             : string;
begin
  S := Name;
  while (Length(S) > 0) and (S[Length(S)] = C) do Dec(S[0]);
  CutChar := S;
end;

{Convert the PETSCII file name into its ASCII representation}
procedure MakeASCIIName; assembler;
asm
    mov si, Offset(FileName);
    mov di, Offset(ASCIIName);
    mov bx, word ptr ConvTable[0];
    mov cl, [si];
    xor ch, ch;
    add si, cx;
    push ds;
    pop es;
    std;
@1: lodsb;
    cmp al, 160;
    jne @2;
    loop @1;
@2: mov byte ptr ASCIIName, cl;
    jcxz @5;
    add di, cx;
@3: xlat;
    stosb;
    lodsb;
    loop @3;
    clc;
    jmp @4;
@5: stc;
@4:
end;

{Correct the PETSCII file name and convert it to ASCII for displaying it on the
  screen}
procedure MakeName;
begin
  MakeASCIIName;
  FileType := UpCase(ShortCBMExt[Attr and 7][1]);
  FileName := CutChar(FileName, #160);
  PCName := '"' + ASCIIName + '"';
end;

{Convert four bytes into a long integer
  Input : B1, B2, B3, B4: the four bytes of the long integer
  Output: the long integer}
function BytesToLongint(B1, B2, B3, B4: Byte): Longint; assembler;
asm
    mov al, B1;
    mov ah, B2;
    mov dl, B3;
    mov dh, B4;
end;

{Convert a file size in bytes to a file size in blocks
  Input : B: file size in bytes
  Output: file size in blocks}
function ByteToBlock(B: Longint): Longint;
var
  L             : Longint;
begin
  L := B div 254;
  if B mod 254 > 0 then Inc(L);
  ByteToBlock := L;
end;

{Get the number of sectors on the specified track
  Input : Track: the number of the track
  Output: the number of sectors on the track}
function SectorNum(Track: Byte): Byte; assembler;
asm
    mov bl, DiskType;
    and bl, $7F;
    mov al, 40;
    cmp bl, dt1581;
    je @1;
    mov ah, Track;
    mov al, 21;
    cmp ah, 18;
    jb @1;
    mov al, 19;
    cmp ah, 25;
    jb @1;
    mov al, 18;
    cmp ah, 31;
    jb @1;
    mov al, 17;
    cmp bl, dt1571;
    jb @1;
    cmp ah, 36;
    jb @1;
    mov al, 21;
    cmp ah, 53;
    jb @1;
    mov al, 19;
    cmp ah, 60;
    jb @1;
    mov al, 18;
    cmp ah, 66;
    jb @1;
    mov al, 17;
@1:
end;

{Determine if a disk position is valid
  Input : T, S; the track and sector number
  Output: when True, the position is valid}
function ValidPos(T, S: Byte): Boolean;
begin
  ValidPos := ((T < MaxTrack) and ((T = 0) or (S < SectorNum(T))));
end;

{Compute the offset of the specified block in a disk image
  Input : Track, Sector: the track and sector number of the block
  Output: the offset of the block divided by 256}
function DiskPos(Track, Sector: Byte): Longint;
var
  I             : Integer;
  P             : Longint;
begin
  P := 0;
  for I := 1 to Track - 1 do Inc(P, SectorNum(I));
  Inc(P, Sector);
  DiskPos := P;
end;

{Read the specified block from the disk image into the given buffer
  Input : T, S: track and sector number of the block
          Buffer: buffer to read the block into}
procedure ReadDiskBlock(T, S: Byte; Buffer: PBlock);
begin
  ExtSeek(Image, DiskPos(T, S) shl 8);
  ExtBlockRead(Image, Buffer^, 256);
end;

{Write the contents of the given buffer into the specified block of the
  disk image
  Input : T, S: track and sector number of the block
          Buffer: buffer to write the block from}
procedure WriteDiskBlock(T, S: Byte; Buffer: PBlock);
begin
  ExtSeek(Image, DiskPos(T, S) shl 8);
  ExtBlockWrite(Image, Buffer^, 256);
end;

{Get the offset of the BAM entry
  Input : T: track number
          Map: when True, the offset of the bitmap of sectors on the track
               is requested, otherwise that of the free sector counter
  Output: offset of BAM entry}
function GetBAMOffset(T: Byte; Map: Boolean): Word;
var
  W             : Word;
begin
  W := T shl 2;
  if Map then Inc(W);
  case DiskType and $7F of
    dt1541Ext:
    begin
      if T >= Max1541Tracks then
      begin
        case ExtBAMMode of
          xbSpeedDOS: Inc(W, $30);
          xbDolphinDOS: Inc(W, $1C);
        end;
      end;
    end;
    dt1571: if T >= Max1541Tracks then if Map then W := (T - Max1541Tracks) * 3 + 256 else W := T + $B9;
    dt1581:
    begin
      W := T * 6;
      if Map then Inc(W);
      if T <= 40 then Inc(W, $010A) else Inc(W, $011A);
    end;
  end;
  GetBAMOffset := W;
end;

{Create an empty BAM in the BAM buffer}
procedure ClearBAM;
var
  B             : Boolean;
  P,
  Q             : Byte;
  W             : Word;
  L             : Longint;
begin
  BAM[0] := DirTrack;
  B := (DiskType and $7F = dt1581);
  if B then
  begin
    BAM[1] := 3;
    BAM2[0] := DirTrack;
    BAM2[1] := 2;
    BAM3[0] := 0;
    BAM3[1] := $FF;
  end
  else
  begin
    BAM[1] := 1;
  end;
  for P := 1 to MaxTrack - 1 do
  begin
    if P = DirTrack2 then
    begin
      W := 0;
      L := 0;
    end
    else
    begin
      L := 0;
      W := SectorNum(P);
      for Q := 1 to W do L := (L shl 1) or 1;
    end;
    BAM[GetBAMOffset(P, False)] := W;
    W := GetBAMOffset(P, True);
    BAM[W] := L;
    BAM[W + 1] := L shr 8;
    BAM[W + 2] := L shr 16;
    if B then
    begin
      BAM[W + 3] := L shr 24;
      BAM[W + 4] := L shr 32;
    end;
  end;
end;

{Return the offset of the name in the BAM
  Input : Disk: disk type
  Output: offset of the name}
function NameOffset(Disk: Byte): Byte;
begin
  case Disk and $7F of
    dt1541, dt1571: NameOffset := $90;
    dt1541Ext: if ExtBAMMode = xbPrologicDOS then NameOffset := $A4 else NameOffset := $90;
    dt1581: NameOffset := $04;
  end
end;

{Create an empty BAM sector in the BAM buffer}
procedure MakeBAM;
var
  B,
  C,
  D             : Byte;
begin
  FillChar(BAM, 3 * 256, 0);
  C := NameOffset(DiskType);
  D := $1A;
  if DiskType and $7F = dt1581 then
  begin
    D := $18;
    BAM[2] := Ord('D');
    BAM2[2] := Ord('D');
    BAM2[3] := Byte(not Ord('D'));
    BAM2[6] := $C0;
    BAM3[2] := Ord('D');
    BAM3[3] := Byte(not Ord('D'));
    BAM3[6] := $C0;
  end
  else
  begin
    if ExtBAMMode = xbPrologicDOS then BAM[2] := Ord('P') else BAM[2] := Ord('A');
  end;
  if DiskType and $7F = dt1571 then BAM[3] := $80;
  for B := 0 to D do BAM[C + B] := $A0;
  BAM[C + 18] := Ord(DefIDCode[1]);
  BAM[C + 19] := Ord(DefIDCode[2]);
  B := Ord('2');
  if DiskType and $7F = dt1581 then
  begin
    B := Ord('3');
    BAM2[4] := Ord(DefIDCode[1]);
    BAM2[5] := Ord(DefIDCode[2]);
    BAM3[4] := Ord(DefIDCode[1]);
    BAM3[5] := Ord(DefIDCode[2]);
  end;
  BAM[C + 21] := B;
  BAM[C + 22] := BAM[2];
  ClearBAM;
end;

{Initialize the BAM by making it empty and allocating the system sectors}
procedure InitBAM;
begin
  MakeBAM;
  AllocBlock(DirTrack, 0, True);
  AllocBlock(DirTrack, 1, True);
  if DiskType and $7F = dt1581 then
  begin
    AllocBlock(DirTrack, 2, True);
    AllocBlock(DirTrack, 3, True);
  end;
end;

{Allocate or free a block
  Input : T, S: track and sector number of the block
          Alloc: when True, the block is allocated; otherwise freed}
procedure AllocBlock(T, S: Byte; Alloc: Boolean);
var
  P             : Byte;
  W,
  X             : Word;
begin
  P := 1 shl (S and 7);
  W := GetBAMOffset(T, True) + (S shr 3);
  if Alloc = (BAM[W] and P > 0) then
  begin
    X := GetBAMOffset(T, False);
    asm
      mov di, Offset(BAM);
      mov si, di;
      add di, W;
      add si, X;
      mov al, P;
      cmp Alloc, False;
      jne @1;
      inc byte ptr [si];
      or byte ptr [di], al;
      jmp @2;
  @1: dec byte ptr [si];
      not al;
      and byte ptr [di], al;
  @2:
    end;
  end;
end;

{Check if the specified block is allocated
  Input : T, S: track and sector number of the block
  Output: when True, the block is allocated, otherwise free}
function IsBlockUsed(T, S: Byte): Boolean;
begin
  IsBlockUsed := (BAM[GetBAMOffset(T, True) + (S shr 3)] and (1 shl (S and 7)) = 0);
end;

{Check if there is at least one block free on the specified track
  Input : T: track number
  Output: when True, there is at least one block free on the track}
function IsTrackFree(T: Byte): Boolean;
begin
  IsTrackFree := (BAM[GetBAMOffset(T, False)] > 0);
end;

{Check if the BAM is valid
  Output: when True, the BAM is valid}
function IsBAMValid: Boolean;
var
  O             : Boolean;
  F,
  T,
  S             : Byte;
begin
  O := True;
  T := 1;
  while O and (T < MaxTrack) do
  begin
    F := 0;
    for S := 0 to SectorNum(T) - 1 do if not IsBlockUsed(T, S) then Inc(F);
    if F <> BAM[GetBAMOffset(T, False)] then O := False;
    Inc(T);
  end;
  IsBAMValid := O;
end;

{Search for a free block on the whole disk for an output file
  Output: when True, a free block was found}
function SearchFreeBlock: Boolean;
var
  F             : Boolean;
  M,
  P             : Byte;
begin
  F := False;
  P := 1;
  while not F and (P < 128) do
  begin
    Track := DirTrack - P;
    if (Track >= 1) and (Track < MaxTrack) then F := IsTrackFree(Track);
    if not F then
    begin
      Track := DirTrack + P;
      if Track < MaxTrack then F := IsTrackFree(Track);
    end;
    if not F then Inc(P);
  end;
  if F then
  begin
    M := SectorNum(Track);
    Sector := 0;
    repeat
      F := not IsBlockUsed(Track, Sector);
      if not F then Inc(Sector);
    until F or (Sector >= M);
  end;
  if F then AllocBlock(Track, Sector, True);
  SearchFreeBlock := F;
end;

{Search for the next free block on the current track
  Output: when True, a free block was found}
function FindNextBlock: Boolean;
var
  F             : Boolean;
  M             : Byte;
begin
  M := SectorNum(Track);
  repeat
    F := not IsBlockUsed(Track, Sector);
    if not F then Inc(Sector);
  until F or (Sector >= M);
  FindNextBlock := F;
end;

{Search for a next block for the output file
  Output: when True, a free block was found}
function SearchNextBlock: Boolean;
var
  F             : Boolean;
  C,
  M             : Byte;
begin
  C := 3;
  F := False;
  while not F and (C > 0) do
  begin
    M := SectorNum(Track);
    if IsTrackFree(Track) then
    begin
      Inc(Sector, SectorStep);
      if Sector >= M then
      begin
        Dec(Sector, M);
        if Sector > 0 then Dec(Sector);
      end;
      F := FindNextBlock;
      if not F then
      begin
        Sector := 0;
        F := FindNextBlock;
      end;
    end
    else
    begin
      if Track < DirTrack then
      begin
        Dec(Track);
        if Track = 0 then
        begin
          Track := DirTrack + 1;
          Sector := 0;
          Dec(C);
        end;
      end
      else
      begin
        if Track > DirTrack then
        begin
          Inc(Track);
          if Track = MaxTrack then
          begin
            Track := DirTrack - 1;
            Sector := 0;
            Dec(C);
          end;
        end;
      end;
    end;
  end;
  if F then AllocBlock(Track, Sector, True);
  SearchNextBlock := F;
end;

{Read a directory entry from the disk image
  Input : Entry: entry record to contain the directory entry
  Output: when False, an empty directory entry was read}
function ReadCBMEntry(var Entry: TDirEntry): Boolean;
var
  B             : Boolean;
  P             : Byte;
begin
  B := True;
  Inc(Number);
  Inc(DirPos);
  if Number = 8 then
  begin
    Number := 0;
    if (DirBuffer[0] = 0) then
    begin
      B := False;
    end
    else
    begin
      DirSector := DirBuffer[1];
      ReadDiskBlock(DirTrack, DirSector, @DirBuffer);
    end;
  end;
  if B then
  begin
    EntryPos := Number shl 5;
    Entry.Attr := DirBuffer[EntryPos + 2];
    Entry.ExtAttr := DirBuffer[EntryPos + 24];
    if not (Entry.ExtAttr in [1..14]) then Entry.ExtAttr := 0;
    if (Entry.ExtAttr > 0) and (DirBuffer[EntryPos + 23] > 0) then Entry.ExtAttr := Entry.ExtAttr or $80;
    Track := DirBuffer[EntryPos + 3];
    Sector := DirBuffer[EntryPos + 4];
    SideTrack := DirBuffer[EntryPos + 21];
    SideSector := DirBuffer[EntryPos + 22];
    Entry.Track := Track;
    Entry.Sector := Sector;
    Entry.SideTrack := SideTrack;
    Entry.SideSector := SideSector;
    Entry.Name := '';
    for P := 0 to 15 do Entry.Name := Entry.Name + Chr(DirBuffer[EntryPos + P + 5]);
    Entry.Name := CutChar(Entry.Name, #160);
    Entry.Size := BytesToLongint(DirBuffer[EntryPos + 30], DirBuffer[EntryPos + 31], 0, 0);
  end;
  ReadCBMEntry := B;
end;

{Read the BAM from the disk image}
procedure ReadBAM;
begin
  FillChar(BAM, 3 * 256, 0);
  case DiskType and $7F of
    dt1541, dt1541Ext: ReadDiskBlock(DirTrack, 0, @BAM);
    dt1571:
    begin
      ReadDiskBlock(DirTrack, 0, @BAM);
      ReadDiskBlock(DirTrack2, 0, @BAM2);
    end;
    dt1581:
    begin
      ReadDiskBlock(DirTrack, 0, @BAM);
      ReadDiskBlock(DirTrack, 1, @BAM2);
      ReadDiskBlock(DirTrack, 2, @BAM3);
    end;
  end;
end;

{Write the BAM into the disk image}
procedure WriteBAM;
begin
  case DiskType and $7F of
    dt1541, dt1541Ext: WriteDiskBlock(DirTrack, 0, @BAM);
    dt1571:
    begin
      WriteDiskBlock(DirTrack, 0, @BAM);
      WriteDiskBlock(DirTrack2, 0, @BAM2);
    end;
    dt1581:
    begin
      WriteDiskBlock(DirTrack, 0, @BAM);
      WriteDiskBlock(DirTrack, 1, @BAM2);
      WriteDiskBlock(DirTrack, 2, @BAM3);
    end;
  end;
end;

{Check whether the current disk or disk image is in GEOS format
  Input : Block: the buffer containing the BAM sector}
procedure CheckGEOSFormat(P: PBlock);
var
  S             : string[11];
begin
  GEOSFormat := False;
  S[0] := #11;
  Move(P^[GEOSSignPos], S[1], 11);
  GEOSFormat := (S = Copy(GEOSSign, 1, 11));
end;

{Open the disk image file
  Input : Write: when True, the disk image is opened for read and write,
                 otherwise only for read
  Output: when not 0, an error occured}
function OpenImage(Write: Boolean): Integer;
var
  P             : Byte;
  I             : Integer;
begin
  GEOSFormat := False;
  I := IOResult;
  if Write then FileMode := fmReadWrite else FileMode := fmReadOnly;
  I := LongOpenFile(DiskName, Image, FileMode);
  if I = 0 then
  begin
    ExtGetFTime(Image, FileDate);
    DiskType := GetDiskType(ExtFileSize(Image));
    if DiskType = 255 then
    begin
      I := 255;
    end
    else
    begin
      CheckDiskType;
      DirSector := 0;
      DirPos := 255;
      ReadBAM;
      if DiskType and $7F = dt1581 then DirSector := 3 else Inc(DirSector);
      ReadDiskBlock(DirTrack, DirSector, @DirBuffer);
      Number := 255;
      CheckGEOSFormat(@BAM);
    end;
  end;
  OpenImage := I;
  FileMode := fmReadOnly;
end;

{Close the disk image}
procedure CloseImage;
begin
  ExtSetFTime(Image, FileDate);
  ExtClose(Image);
end;

{Create the destination disk image}
function CreateDisk: Integer;
var
  F             : Byte;
  I             : Integer;
  L             : Longint;
begin
  CheckDiskType;
  I := LongOpenFile(Image.LongName, Image, fmWriteOnly);
  if I = 0 then
  begin
    Buffer := New(PBuffer);
    FillChar(Buffer^, BufferSize, 0);
    L := GetDiskSize shl 8;
    while L > 0 do
    begin
      if L > BufferSize then DiskSize := BufferSize else DiskSize := L;
      Dec(L, DiskSize);
      ExtBlockWrite(Image, Buffer^, DiskSize);
    end;
    Dispose(Buffer);
    InitBAM;
    WriteBAM;
    FillChar(DataBuffer, 256, 0);
    DataBuffer[1] := 255;
    WriteDiskBlock(DirTrack, FirstDirSector, @DataBuffer);
    ExtSetFTime(Image, FileDate);
    ExtClose(Image);
  end;
  CreateDisk := I;
end;

{Open the file for input
  Output: when not 0, an error occured}
procedure OpenRead;
begin
  Block := 0;
  Error := False;
  _End := Error;
end;

{Open the file for output
  Input : Name: name of the file to be opened
          AAttr: attribute of the file to be opened
          Len: length of data to be written into the file
          Force: force the creation of file even if another one with the
                 same name exists
  Output: when not 0, an error occured}
function OpenWrite(Name: string; AAttr: Byte; Len: Longint; Force: Boolean): Integer;
var
  F,
  O             : Boolean;
  M,
  Q             : Byte;
  I             : Integer;
begin
  Block := 0;
  I := 255;
  Attr := AAttr;
  if OpenImage(False) = 0 then
  begin
    F := False;
    if not Force then while not F and ReadCBMEntry(CBMEntry) do F := ((CBMEntry.Attr > 0) and (CBMEntry.Name = Name));
    if F then
    begin
      CloseImage;
      Attr := CBMEntry.Attr;
      I := 254;
    end
    else
    begin
      if not IsBAMValid then
      begin
        CloseImage;
        WriteLn(DiskName, ' has an invalid BAM');
        O := False;
      end
      else
      begin
        CloseImage;
        if OpenImage(True) = 0 then
        begin
          while Length(Name) < 16 do Name := Name + #160;
          F := False;
          while not F and ReadCBMEntry(CBMEntry) do F := (CBMEntry.Attr = 0);
          if not F then
          begin
            if IsTrackFree(DirTrack) then
            begin
              if DiskType and $7F = dt1581 then SectorStep := 1 else SectorStep := 3;
              Track := DirTrack;
              Sector := DirSector;
              F := SearchNextBlock;
              WriteDiskBlock(DirTrack, 0, @BAM);
              DirBuffer[0] := DirTrack;
              DirBuffer[1] := Sector;
              WriteDiskBlock(DirTrack, DirSector, @DirBuffer);
              DirSector := Sector;
              case DiskType and $7F of
                dt1541, dt1541Ext: SectorStep := 10;
                dt1571: SectorStep := 6;
                dt1581: SectorStep := 1;
              end;
              FillChar(DirBuffer, 256, 0);
              DirBuffer[1] := 255;
              EntryPos := 0;
              F := True;
            end
            else
            begin
              WriteLn('Directory is full in ', DiskName);
              O := False;
            end;
          end;
          if F then
          begin
            if Len = 0 then
            begin
              Track := DirTrack;
              Sector := 0;
            end
            else
            begin
              F := SearchFreeBlock;
            end;
            if F then
            begin
              for Q := 2 to 31 do DirBuffer[EntryPos + Q] := 0;
              DirBuffer[EntryPos + 2] := Attr;
              DirBuffer[EntryPos + 3] := Track;
              DirBuffer[EntryPos + 4] := Sector;
              for Q := 0 to 15 do DirBuffer[EntryPos + Q + 5] := Ord(Name[Q + 1]);
              WriteDiskBlock(DirTrack, DirSector, @DirBuffer);
              I := 0;
            end
            else
            begin
              WriteLn(DiskName, ' is full');
              O := False;
            end;
          end;
        end;
      end;
    end;
  end
  else
  begin
    WriteLn('Can''t open ', DiskName);
  end;
  Error := (I <> 0);
  _End := Error;
  OpenWrite := I;
end;

{Close the output file}
procedure CloseWrite;
begin
  DirBuffer[EntryPos + 2] := DirBuffer[EntryPos + 2] or 128;
  DirBuffer[EntryPos + 30] := Block;
  DirBuffer[EntryPos + 31] := Block shr 8;
  WriteDiskBlock(DirTrack, DirSector, @DirBuffer);
  WriteBAM;
  ExtSetFTime(Image, FileDate);
  ExtClose(Image);
end;

{Read a buffer of data from the input file
  Input : Buffer: buffer to read the data into
          Len: length of the data read}
procedure ReadPart(Buffer: PBuffer; var Len: Word);
var
  M,
  Q             : Byte;
begin
  Len := 0;
  while not Error and (Len < BufferSize) and (Track > 0) do
  begin
    Error := ((Track > MaxTrack) or (Sector >= SectorNum(Track)));
    if not Error then
    begin
      ReadDiskBlock(Track, Sector, @DataBuffer);
      Track := DataBuffer[0];
      Sector := DataBuffer[1];
      if Track > 0 then
      begin
        M := 254;
      end
      else
      begin
        M := Sector - 1;
        _End := True;
      end;
      for Q := 0 to M - 1 do Buffer^[Len + Q] := DataBuffer[Q + 2];
      Inc(Len, M);
    end;
  end;
  Error := Error or (IOResult <> 0);
  _End := Error or _End;
  if Error then Len := 0;
end;

{Write a buffer of data into the output file
  Input : Buffer: buffer containing the data to write
          Len: length of the data read
          FileEnd: when True, the input file has ended}
procedure WritePart(Buffer: PBuffer; Len: Word; FileEnd: Boolean);
var
  F,
  O             : Boolean;
  Q             : Byte;
  X,
  Z             : Longint;
begin
  X := 0;
  F := True;
  while F and (X < Len) do
  begin
    if X + 254 < Len then Z := 254 else Z := Len - X;
    for Q := Z to 255 do DataBuffer[Q] := 0;
    for Q := 0 to Z - 1 do DataBuffer[Q + 2] := Buffer^[X + Q];
    ExtSeek(Image, DiskPos(Track, Sector) shl 8);
    if (X + 254 < Len) or not FileEnd then
    begin
      F := SearchNextBlock;
    end
    else
    begin
      F := True;
      Track := 0;
      Sector := Z + 1;
    end;
    DataBuffer[0] := Track;
    DataBuffer[1] := Sector;
    ExtBlockWrite(Image, DataBuffer, 256);
    Inc(X, Z);
    Inc(Block);
    if not F then
    begin
      WriteLn(DiskName, ' is full');
      O := False;
      Error := True;
    end;
  end;
  Error := Error or (IOResult <> 0);
end;

{Convert the ASCII string into a CBM file name
  Input : CBMName: the CBM file name
          PCName: the ASCII string}
procedure ConvertCBMName(var CBMName, PCName: string); assembler;
asm
    push ds;
    push es;
    lds si, CBMName;
    les di, PCName;
    mov bx, Offset(ASCToPET);
    cld;
    lodsb;
    stosb;
    mov cl, al;
    xor ch, ch;
    jcxz @2;
@1: lodsb;
    xlat;
    stosb;
    loop @1;
@2: pop es;
    pop ds;
end;

{Change invalid ASCII characters to underscores in PC file names
  Input : OrigName: original file name
          NewName: corrected file name}
procedure CorrectPCName(var OrigName, NewName: string); assembler;
asm
    push ds;
    push es;
    lds si, OrigName;
    les di, NewName;
    cld;
    lodsb;
    stosb;
    mov cl, al;
    xor ch, ch;
@1: lodsb;
    cmp al, ' ';
    jbe @3;
    cmp al, '"';
    je @3;
    cmp al, '*';
    jb @4;
    cmp al, ',';
    jbe @3;
@4: cmp al, '.';
    jne @5;
    or bl, bl;
    jne @3;
    inc bl;
@5: cmp al, '/';
    je @3;
    cmp al, ':';
    jb @6;
    cmp al, '?';
    jbe @3;
@6: cmp al, '[';
    jb @7;
    cmp al, '^';
    jbe @3;
@7: cmp al, $FA;
    jne @2;
@3: mov al, '_';
@2: stosb;
    loop @1;
    pop es;
    pop ds;
end;

{Clear the current screen line without line feed}
procedure ClrLine;
begin
  Write(#13, BlankLine, #13);
end;

{Ask the user to specify a new file name
  Input : Text: text to write onto the screen
          Str: string containing file name
          Len: maximum length of file name
          Min, Max: characters specifying the interval of allowed characters}
procedure AskStr(const Text: string; var Str: string; Len: Integer; Min, Max: Char; Upper: Boolean);
var
  O             : Boolean;
  B             : Byte;
  C             : Word;
  S,
  T             : string[80];
begin
  ClrLine;
  S := Str;
  B := Length(S);
  FillChar(T, Len + 2, #8);
  O := True;
  repeat
    if O then
    begin
      O := False;
      T[0] := Chr(Length(S) - B + 1);
      Write(#13, Text, S, ' ', T);
    end;
    C := ReadKey;
    if Upper then C := Hi(C) shl 8 + Ord(UpCase(Chr(Lo(C))));
    if (Chr(C) in [Min..Max]) and (Length(S) < Len) then
    begin
      S := S + Chr(C);
      Inc(B);
      O := True;
    end
    else
    begin
      case C of
        $0E08:
        begin
          if B > 0 then
          begin
            S := Copy(S, 1, B - 1) + Copy(S, B + 1, 255);
            Dec(B);
            O := True;
          end;
        end;
        $5300:
        begin
          if B < Length(S) then
          begin
            S := Copy(S, 1, B) + Copy(S, B + 2, 255);
            O := True;
          end;
        end;
        $0E7F:
        begin
          if S <> '' then
          begin
            S[0] := Chr(Len);
            FillChar(S[1], Len, ' ');
            Write(#13, Text, S);
            S := '';
            B := 0;
            O := True;
          end;
        end;
        $4B00:
        begin
          if B > 0 then
          begin
            Dec(B);
            O := True;
          end;
        end;
        $4D00:
        begin
          if B < Length(S) then
          begin
            Inc(B);
            O := True;
          end;
        end;
        $4700:
        begin
          B := 0;
          O := True;
        end;
        $4F00:
        begin
          B := Length(S);
          O := True;
        end;
        $011B:
        begin
          S := '';
          C := $1C0D;
        end;
      end;
    end;
  until C = $1C0D;
  ClrLine;
  if S <> '' then Str := S;
end;

{Prompt the user and ask for confirmation etc.
  Input : Text: text to write onto the screen
          All: when True, the user is allowed to press 'A' to say Yes to all
          Never: when True, the user is allowed to press 'E' to say No to all
          Rename: when True, the user is allowed to rename the current file
          Answer: stores the answer code}
function Question(const Text, All, Never, Rename: string; var Answer: Byte): Boolean;
var
  C             : Char;
  I             : Byte;
begin
  if AlwaysYes and (Answer <> aaAllNo) then
  begin
    Question := True;
  end
  else
  begin
    case Answer of
      aaAllYes: Question := True;
      aaAllNo: Question := False;
    else
      ClrLine;
      Write(Text, ' (Yes/No');
      if All <> '' then Write('/', All);
      if Never <> '' then Write('/', Never);
      if Rename <> '' then Write('/', Rename);
      Write(') ? ');
      repeat
        C := UpCase(Chr(ReadKey));
        case C of
          #13: C := 'Y';
          #27: C := 'N';
        end;
      until ((All <> '') and (C = 'A')) or ((Never <> '') and (C = 'E')) or
        ((Rename <> '') and (C = 'R')) or (C = 'N') or (C = 'Y');
      Answer := aaAsk;
      case C of
        'A':
        begin
          Answer := aaAllYes;
          C := 'Y';
        end;
        'E':
        begin
          Answer := aaAllNo;
          C := 'N';
        end;
        'R':
        begin
          AskStr('New file name: ', ASCIIName, 18, ' ', #126, False);
          I := LeftPos(',', ASCIIName);
          if I > 0 then
          begin
            FileType := LoCase(ASCIIName[I + 1]);
            if not (FileType in ['d', 'p', 's', 'u']) then FileType := 'p';
            ASCIIName[0] := Chr(I - 1);
          end;
          ConvertCBMName(ASCIIName, FileName);
          Answer := aaRename;
          C := 'N';
        end;
      end;
      Write(C);
      Question := (C = 'Y');
      ClrLine;
    end;
  end;
end;

procedure CharSetOptions(const Option: string);
begin
  case Option[1] of
    'L': ConvTable := @PETtoASCLower;
    'U': ConvTable := @PETtoASCUpper;
  else
    Error := True;
  end;
end;

{Common 'Add' option processor}
procedure AddOptions(const Option: string);
begin
  case Option[1] of
    'C': Confirm := aaAsk;
    'D': Delete := aaAsk;
    'Y': AlwaysYes := True;
  else
    CharSetOptions(Option);
  end;
end;

{Common 'eXtract' option processor}
procedure ExtractOptions(const Option: string);
begin
  case Option[1] of
    '4': DiskType := dt1541;
    '7': DiskType := dt1571;
    '8': DiskType := dt1581;
    'C': Confirm := aaAsk;
    'D': Delete := aaAsk;
    'X':
    begin
      if DiskType in [dt1541, dt1541Ext] then
      begin
        DiskType := dt1541Ext;
        ExtBAMMode := xbSpeedDOS;
        if Length(Option) > 1 then
        begin
          case Option[2] of
            'S': ExtBAMMode := xbSpeedDOS;
            'D': ExtBAMMode := xbDolphinDOS;
            'P': ExtBAMMode := xbPrologicDOS;
          else
            Error := True;
          end;
        end;
      end
      else
      begin
        Error := True;
      end;
    end;
    'Y': AlwaysYes := True;
  else
    CharSetOptions(Option);
  end;
end;

{Common no options processor}
procedure NoOptions(const Option: string);
begin
  Error := True;
end;

{Process command line options
  Input : Process: the prodecure processing separate options}
procedure ParseCmdLine(Process: TOptProc);
var
  S             : string;
begin
  S := LongParamStr(Number);
  while not Error and (Number <= ParamCount) and (Length(S) > 1) and (S[1] in ['-', '/']) do
  begin
    Process(UpperCase(Copy(S, 2, 255)));
    Inc(Number);
    S := LongParamStr(Number);
  end;
  if Error then WriteLn('Invalid option');
end;

{Cut the parts of the DOS file name into the standard form of a 8 character
  name, a dot and a 3 character extension
  Input : Name: the original DOS file name
  Output: the corrected file name}
function CorrectDOSName(Name: string): string;
var
  P             : Byte;
  S             : string;
begin
  Name := CutChar(Name, ' ');
  if Name = '..' then
  begin
    S := Name;
  end
  else
  begin
    P := RightPos('.', Name);
    if P = 0 then
    begin
       S := CutChar(Copy(Name, 1, 8), ' ');
    end
    else
    begin
      if P > 8 then S := Copy(Name, 1, 8) else S := Copy(Name, 1, P - 1);
      S := CutChar(S, ' ');
      if P < Length(Name) then S := S + '.' + Copy(Name, P + 1, 3);
    end;
  end;
  CorrectDOSName := LowerCase(S);
end;

{Execute a Windows'95-style long file name function and return error if
  the function is not supported}
procedure ExecLFN; assembler;
asm
    mov ah, $71;
    stc;
    int $21;
    jc @1;
    cmp ax, $7100;
    stc;
    je @1;
    clc;
@1:
end;

{Determine if long file name support is installed
  Output: when True, long file names are supported}
function InitLongNames: Boolean; assembler;
var
  S,
  T             : string;
asm
    push ds;
    push ss;
    pop ds;
    push ss;
    pop es;
    lea si, S;
    mov byte ptr [si], 0;
    lea di, T;
    xor cl, cl;
    mov ch, $80;
    mov al, $60;
    call ExecLFN;
    mov bl, True;
    jnc @1;
    cmp ax, $7100;
    jne @1;
    mov bl, False;
@1: pop ds;
    mov LongFileNames, bl;
end;

{Execute a Windows'95-style long file name function with a file name as an
  argument and return the resulting file name
  Input : Name: the original file name
          Code: file name conversion code (1 to convert to short, 2 to
                convert to long file name)
          Path: when True, the path part of the result is also kept
  Output: the resulting file name}
function OtherName(Name: string; Code: Byte; Path: Boolean): string;
var
  W             : Word;
  S             : string;
begin
  asm
    push ds;
    push ss;
    pop ds;
    push ss;
    pop es;
    lea si, Name;
    mov bl, [si];
    xor bh, bh;
    inc si;
    mov byte ptr [si][bx], 0;
    lea di, S;
    inc di;
    mov cl, Code;
    mov ch, $80;
    mov al, $60;
    call ExecLFN;
    mov W, ax;
    mov bl, 0;
    jc @1;
    xor al, al;
    mov bx, di;
    mov cx, 255;
    cld;
    repne scasb;
    sub bx, di;
    neg bx;
    dec bx;
@1: mov byte ptr S[0], bl;
    pop ds;
  end;
  if S = '' then if W = $7100 then S := AddToPath(GetPath(Name, '\'), CorrectDOSName(CutPath(Name, '\')), '\') else S := Name;
  if not Path then S := CutPath(CutPath(S, '\'), '/');
  OtherName := S;
end;

function ProcessName(const Name: string; Code: Byte; Path: Boolean): string;
var
  B             : Byte;
  P,
  S,
  T             : string;
begin
  S := Name;
  P := '';
  if (Length(S) >= 2) and (S[2] = ':') and (UpCase(S[1]) in ['A'..'Z']) then
  begin
    B := 2;
    if (Length(S) >= 3) and (S[3] = '\') then Inc(B);
    P := UpperCase(Copy(S, 1, B));
    S := Copy(S, B + 1, 255);
  end;
  while S <> '' do
  begin
    B := LeftPos('\', S);
    if B = 0 then
    begin
      T := S;
      S := '';
    end
    else
    begin
      T := Copy(S, 1, B - 1);
      if T = '' then T := '\';
      S := Copy(S, B + 1, 255);
    end;
    if (T <> '\') and (T <> '.') and (T <> '..') then T := OtherName(AddToPath(P, T, '\'), Code, False);
    if (T <> '') and (not LongFileNames or (T = UpperCase(T)) or (T = LowerCase(T))) then
      if B = 0 then T := LowerCase(T) else T := UpperCase(T);
    P := AddToPath(P, T, '\');
  end;
  ProcessName := P;
end;

{Get the short file name for a Windows'95-style long file name; if Windows'95
  is not present, the original file name is returned
  Input : Name: the long file name
          Path: when True, the path part of the result is kept
  Output: the short file name}
function ShortName(const Name: string; Path: Boolean): string;
begin
  ShortName := ProcessName(Name, 1, Path);
end;

{Get the Windows'95-style long file name for a short file name; if Windows'95
  is not present, the original file name is returned
  Input : Name: the short file name
          Path: when True, the path part of the result is kept
  Output: the long file name}
function LongName(const Name: string; Path: Boolean): string;
var
  S             : string;
begin
  S := ProcessName(Name, 2, Path);
  if (S = '') or ((S = UpperCase(S)) and (S = UpperCase(Name))) then
    S := AddToPath(UpperCase(GetPath(Name, '\')), LowerCase(CutPath(Name, '\')), '\');
  LongName := S;
end;

{Get a long file name from the command line
  Input : Index: the number of the command line parameter; 0 not supported;
          when 255, the complete command line is returned
  Output: the command line parameter}
function LongParamStr(Index: Byte): string; assembler;
asm
    push ds;
    mov dl, Index;
    or dl, dl;
    je @12;
    xor dh, dh;
    mov ax, PrefixSeg;
    mov ds, ax;
    mov di, $0080;
    mov cl, [di];
    xor ch, ch;
    inc di;
    xor bx, bx;
    xor ah, ah;
@1: jcxz @3;
@2: mov al, [di];
    or al, al;
    jne @13;
    xor cx, cx;
    jmp @3;
@13:cmp al, '"';
    jne @7;
    xor ah, 1;
    inc di;
    dec cx;
    jmp @3;
@7: cmp al, ' ';
    ja @3;
    inc di;
    loop @2;
@3: mov si, di;
    jcxz @5;
@4: mov al, [di];
    or al, al;
    jne @14;
    xor cx, cx;
    jmp @5;
@14:cmp al, '"';
    jne @8;
    xor ah, 1;
@8: cmp dl, 255;
    je @9;
    cmp al, ' ';
    ja @9;
    or ah, ah;
    je @5;
@9: inc di;
    loop @4;
@5: mov ax, di;
    sub ax, si;
    je @6;
    cmp dl, 255;
    je @6;
    dec dx;
    jnz @1;
@6: les di, @Result;
    mov bx, di;
    inc di;
    xor dl, dl;
    mov cx, ax;
    jcxz @12;
@11:lodsb;
    cmp al, '"';
    je @10;
    stosb;
    inc dl;
@10:loop @11;
@12:mov es:[bx], dl;
    pop ds;
end;

{Split a long file name that contains the path, name and extension of the
  file
  Input : Path: the full file name
          Dir: the string to contain the path
          Name: the string to contain the file name
          Ext: the string to contain the extension}
procedure LongFSplit(const Path: string; var Dir, Name, Ext: string); assembler;
asm
    push ds;
    cld;
    lds si, Path;
    lodsb;
    mov dl, al;
    xor dh, dh;
    mov bx, dx;
    or bx, bx;
    je @2;
@1: cmp byte ptr [si][bx][-1], '\';
    je @2;
    cmp byte ptr [si][bx][-1], ':';
    je @2;
    dec bx;
    jne @1;
@2: mov ax, 255;
    les di, Dir;
    call @7;
    mov bx, dx;
    or bx, bx;
    je @4;
@3: dec bx;
    or bx, bx;
    jne @5;
    mov bx, dx;
    jmp @4;
@5: cmp byte ptr [si][bx], '.';
    jne @3;
@4: mov ax, 255;
    les di, Name;
    call @7;
    mov bx, dx;
    mov ax, 255;
    les di, Ext;
    call @7;
    pop ds;
    jmp @8;
@7: sub dx, bx;
    cmp ax, bx;
    jb @6;
    mov ax, bx;
@6: stosb;
    mov cx, ax;
    add bx, si;
    rep movsb;
    mov si, bx;
    retn;
@8:
end;

{Convert a full lowercase file name to full uppercase so that no LFN entry
  is created}
procedure DropLongName(var Name: string);
var
  S             : string;
begin
  S := CutPath(Name, '\');
  if not LongFileNames then S := CorrectDOSName(S);
  if S = LowerCase(S) then Name := AddToPath(GetPath(Name, '\'), UpperCase(S), '\');
end;

{Erase a file
  Input : Name: the name of the file to erase}
procedure LongErase(Name: string);
begin
  asm
    push ds;
    push ss;
    pop ds;
    lea si, Name;
    mov bl, [si];
    xor bh, bh;
    inc si;
    mov byte ptr [si][bx], 0;
    mov dx, si;
    xor si, si;
    mov al, $41;
    call ExecLFN;
    jc @1;
    xor ax, ax;
@1: pop ds;
    mov InOutRes, ax;
  end;
  if InOutRes = $7100 then
  begin
    InOutRes := 0;
    Assign(TempFile.Orig, Name);
    Erase(TempFile.Orig);
  end;
end;

{Get or set different attributes of a file
  Input : Name: the name of the file to process
          Code: when 0, attributes are read, otherwise written
          Attr: attribute to set or get}
procedure LongAttrib(Name: string; Code: Byte; var Attr: Longint);
var
  W             : Word;
begin
  asm
    push ds;
    push ss;
    pop ds;
    lds si, Attr;
    mov cx, word ptr ds:[si][0];
    mov di, word ptr ds:[si][2];
    lea si, Name;
    mov bl, [si];
    xor bh, bh;
    inc si;
    mov byte ptr [si][bx], 0;
    mov dx, si;
    mov bl, Code;
    mov al, $43;
    call ExecLFN;
    jc @1;
    xor ax, ax;
    jmp @2;
@1: mov cl, Code;
    or cl, cl;
    jne @3;
    xor cx, cx;
    mov di, cx;
@2: lds si, Attr;
    mov word ptr ds:[si][0], cx;
    mov word ptr ds:[si][2], di;
@3: pop ds;
    mov InOutRes, ax;
  end;
  if InOutRes = $7100 then
  begin
    InOutRes := 0;
    Assign(TempFile.Orig, Name);
    case Code of
      0:
      begin
        W := 0;
        GetFAttr(TempFile.Orig, W);
        if DOSError = 0 then Attr := W;
      end;
      1:
      begin
        W := Attr;
        SetFAttr(TempFile.Orig, W);
      end;
    else
      DOSError := 1;
    end;
    InOutRes := DOSError;
  end;
end;

{Get the attributes of a file
  Input : F: the file record
          Attr: the word to store the attributes into}
procedure LongGetFAttr(const Name: string; var Attr: Word);
var
  L             : Longint;
begin
  LongAttrib(Name, 0, L);
  Attr := L;
end;

{Set the attributes of a file
  Input : F: the file record
          Attr: the word containing the attributes}
procedure LongSetFAttr(const Name: string; Attr: Word);
var
  L             : Longint;
begin
  L := Attr;
  LongAttrib(Name, 1, L);
end;

{If the file name is a short one then convert it to full lowercase or
  full uppercase, depending on whether it is a file or a directory
  Input : F: the extended search record to contain the entry
          E: the long file name search record containing the entry
          B: length of the long file name
          C: error code returned by DOS}
procedure CorrectName(var F: ExtSearchRec; var E: LongSearchRec; B: Byte; C: Word);
begin
  if DOSError = 0 then
  begin
    case C of
      0:
      begin
        F.Orig.Attr := E.Attr;
        F.Orig.Time := E.Time;
        F.Orig.Size := E.Size;
        F.LongName[0] := Chr(B);
        Move(E.LongName, F.LongName[1], B);
      end;
      $7100: F.LongName := F.Orig.Name;
    end;
    if (F.LongName = '') or (F.LongName = UpperCase(F.LongName)) then if F.Orig.Attr and Directory = 0 then
      F.LongName := LowerCase(F.LongName) else F.LongName := UpperCase(F.LongName);
  end;
end;

{Find first instance of a file in a directory
  Input : Path: full wildcarded name of files to find
          Attr: allowed attributes
          F: record to store result into}
procedure LongFindFirst(Path: string; Attr: Byte; var F: ExtSearchRec);
var
  B             : Byte;
  C,
  W             : Word;
  E             : LongSearchRec;
begin
  asm
    xor ax, ax;
    mov DOSError, ax;
    mov W, ax;
    push ds;
    push ss;
    pop ds;
    push ss;
    pop es;
    lea si, Path;
    mov bl, [si];
    xor bh, bh;
    inc si;
    mov byte ptr [si][bx], 0;
    lea di, E;
    mov dx, si;
    mov cl, Attr;
    xor ch, ch;
    mov si, 1;
    mov al, $4E;
    call ExecLFN;
    pop ds;
    jc @1;
    mov W, ax;
    xor al, al;
    add di, LongSearchRec.LongName;
    mov bx, di;
    mov cx, 255;
    cld;
    repne scasb;
    sub bx, di;
    neg bx;
    dec bx;
    mov byte ptr B, bl;
    jmp @2;
@1: mov DOSError, ax;
@2:
  end;
  C := DOSError;
  case DOSError of
    0:
    begin
      F.HandleUsed := True;
      F.LongHandle := W;
    end;
    $7100: FindFirst(Path, Attr, F.Orig);
  end;
  CorrectName(F, E, B, C);
end;

{Find next instance of a file in a directory
  Input : F: record to read data from and store result into}
procedure LongFindNext(var F: ExtSearchRec);
var
  B             : Byte;
  C             : Word;
  E             : LongSearchRec;
begin
  asm
    mov DOSError, 0;
    les di, F;
    mov bx, es:[di].ExtSearchRec.LongHandle;
    push ss;
    pop es;
    lea di, E;
    mov si, 1;
    mov al, $4F;
    call ExecLFN;
    jc @1;
    xor al, al;
    add di, LongSearchRec.LongName;
    mov bx, di;
    mov cx, 255;
    cld;
    repne scasb;
    sub bx, di;
    neg bx;
    dec bx;
    mov byte ptr B, bl;
    jmp @2;
@1: mov DOSError, ax;
@2:
  end;
  C := DOSError;
  if DOSError = $7100 then FindNext(F.Orig);
  CorrectName(F, E, B, C);
end;

{Rename a file to another name
  Input : OrigName: the original name of the file
          NewName: the name to rename the file to}
procedure LongRename(OrigName, NewName: string);
begin
  DropLongName(NewName);
  asm
    push ds;
    push ss;
    pop ds;
    push ss;
    pop es;
    lea si, OrigName;
    mov bl, [si];
    xor bh, bh;
    inc si;
    mov byte ptr [si][bx], 0;
    lea di, NewName;
    mov bl, es:[di];
    xor bh, bh;
    inc di;
    mov byte ptr es:[di][bx], 0;
    mov dx, si;
    mov al, $56;
    call ExecLFN;
    jc @1;
    xor ax, ax;
@1: pop ds;
    mov InOutRes, ax;
  end;
  if InOutRes = $7100 then
  begin
    InOutRes := 0;
    Assign(TempFile.Orig, OrigName);
    Rename(TempFile.Orig, NewName);
  end;
end;

{Open a file with long file name
  Input : Name: the name of the file
          F: the file record
          Mode: access mode to open the file with
  Output: when not 0, an error occured}
function LongOpenFile(Name: string; var F: ExtFile; Mode: Byte): Integer;
var
  W             : Word;
  I             : Integer;
begin
  F.LongName := Name;
  FileMode := Mode;
  DropLongName(Name);
  asm
    mov InOutRes, 0;
    mov al, FileMode;
    push ds;
    push ss;
    pop ds;
    lea si, Name;
    mov bl, [si];
    xor bh, bh;
    inc si;
    mov byte ptr [si][bx], 0;
    mov bl, al;
    xor cx, cx;
    xor di, di;
    mov dx, 1;
    cmp bl, fmWriteOnly;
    jne @1;
    mov dx, $12;
@1: mov al, $6C;
    call ExecLFN;
    pop ds;
    jnc @2;
    mov InOutRes, ax;
    jmp @3;
@2: les di, F;
    mov es:[di].ExtFile.Orig.FileRec.Mode, fmInOut;
    mov es:[di].ExtFile.Orig.FileRec.Handle, ax;
    mov es:[di].ExtFile.Orig.FileRec.RecSize, 1;
@3:
  end;
  if InOutRes = $7100 then
  begin
    InOutRes := 0;
    Assign(F.Orig, Name);
    case FileMode of
      fmReadOnly, fmReadWrite: Reset(F.Orig, 1);
      fmWriteOnly: Rewrite(F.Orig, 1);
    end;
    I := IOResult;
    F.LongName := FileRec(F.Orig).Name;
  end
  else
  begin
    I := IOResult;
  end;
  LongOpenFile := I;
end;

{Close directory search
  Input : F: record to read data from}
procedure LongFindClose(var F: ExtSearchRec); assembler;
asm
    les di, F;
    mov byte ptr es:[di].ExtSearchRec.HandleUsed, 0;
    mov bx, es:[di].ExtSearchRec.LongHandle;
    mov al, $A1;
    call ExecLFN;
end;

function LongFExpand(Path: string): string;
begin
  LongFExpand := LongName(ExtFExpand(Path), True);
end;

{Read a block of data from a file
  Input : F: the file to read the data from
          Buf: the buffer to put the data into
          Count: the number of bytes to read}
procedure ExtBlockRead(var F: ExtFile; var Buf; Count: Word);
begin
  BlockRead(F.Orig, Buf, Count);
end;

{Read a block of data from a file
  Input : F: the file to read the data from
          Buf: the buffer to put the data into
          Count: the number of bytes to read
          Result: the integer to contain the number of bytes that have
                  actually been read}
procedure ExtBlockRead2(var F: ExtFile; var Buf; Count: Word; var Result: Word);
begin
  BlockRead(F.Orig, Buf, Count, Result);
end;

{Write a block of data into a file
  Input : F: the file to write the data into
          Buf: the buffer to get the data from
          Count: the number of bytes to write}
procedure ExtBlockWrite(var F: ExtFile; var Buf; Count: Word);
var
  W             : Word;
begin
  BlockWrite(F.Orig, Buf, Count, W);
end;

{Write a block of data into a file
  Input : F: the file to write the data into
          Buf: the buffer to get the data from
          Count: the number of bytes to write
          Result: the integer to contain the number of bytes that have
                  actually been written}
procedure ExtBlockWrite2(var F: ExtFile; var Buf; Count: Word; var Result: Word);
begin
  BlockWrite(F.Orig, Buf, Count, Result);
end;

{Truncate a file at the current position
  Input : F: the file to truncate}
procedure ExtTruncate(var F: ExtFile);
begin
  Truncate(F.Orig);
end;

{Close a file
  Input : F: the file to close}
procedure ExtClose(var F: ExtFile);
begin
  Close(F.Orig);
end;

{Determine if a file has reached its end
  Input : F: the file to check
  Output: when True, the end of the file has been reached; otherwise not}
function ExtEOF(var F: ExtFile): Boolean;
begin
  ExtEOF := EOF(F.Orig);
end;

{Seek in a file
  Input : F: the file to seek in
          Pos: the position to seek to}
procedure ExtSeek(var F: ExtFile; Pos: Longint);
begin
  Seek(F.Orig, Pos);
end;

{Determine the size of a file
  Input : F: the file to check
  Output: the size of the file in bytes}
function ExtFileSize(var F: ExtFile): Longint;
begin
  ExtFileSize := FileSize(F.Orig);
end;

{Determine the current position into a file
  Input : F: the file to check
  Output: the current position into the file}
function ExtFilePos(var F: ExtFile): Longint;
begin
  ExtFilePos := FilePos(F.Orig);
end;

{Get the date stamp of a file
  Input : F: the file to get the date stamp of
          Time: the longint to contain the date stamp of the file}
procedure ExtGetFTime(var F: ExtFile; var Time: Longint);
begin
  GetFTime(F.Orig, Time);
end;

{Set the date stamp of a file
  Input : F: the file to set the date stamp of
          Time: the new date stamp of the file}
procedure ExtSetFTime(var F: ExtFile; Time: Longint);
begin
  SetFTime(F.Orig, Time);
end;

end.
