/*****************************************************************************
 * avc2avi.c: raw h264 -> AVI
 *****************************************************************************
 * Copyright (C) 2004 Laurent Aimar
 * $Id: avc2avi.c,v 1.1 2006/12/10 14:10:30 crypto1 Exp $
 *
 * Authors: Laurent Aimar <fenrir@via.ecp.fr>
 * Modified by:
 *   DVBPortal
 *   Joe Forster/STA <sta@c64.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/* References:
   - H.264 Analyzer http://h264bitstream.sourceforge.net
   - H.264 specifications: http://www.itu.int/rec/T-REC-H.264/e */

/* Enable using standard input and output. Problems with standard input and output under Windows:
   - no file size for standard input;
   - no seeking in standard input and output. */
//#define STD_IN_OUT 1

/* Enable long options. Long options are available only when compiling under GCC or re-implemented. */
#define LONG_OPTIONS 1

#include "stdafx.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "posix/stdint.h"
#include <sys/types.h>
#include <signal.h>
#include <fcntl.h>
#define _GNU_SOURCE

/* Command line options must be re-implemented and long options are not available when not compiling with GCC. */
#if defined(__MINGW32__) || defined(__MINGW64__)
# define SYSTEM_GETOPT 1
#endif
#ifdef SYSTEM_GETOPT
# include <getopt.h>
#else
# include "getopt.h"
# ifdef LONG_OPTIONS
#  undef LONG_OPTIONS
# endif
#endif

#ifdef _MSC_VER
# include <io.h>     /* _setmode() */
# include <fcntl.h>  /* _O_BINARY */
#endif

#include "common/bs.h"

#define __unused_variable__(var) (void)a

#define DATA_MAX 3000000
uint8_t data[DATA_MAX];

#define FPS_DEFAULT 25.0

/* added: variables for splitting */
uint64_t i_written = 0;
uint64_t i_split = 0;
uint16_t i_part = 0;
int b_hiprofile = 0;

/* Ctrl-C handler */
static int     i_ctrl_c = 0;
static void    SigIntHandler( int a )
{
    __unused_variable__(a);
    i_ctrl_c = 1;
}

typedef struct
{
    char *psz_fin;
    char *psz_fout;

    float f_fps;
    char  fcc[4];
} cfg_t;

typedef struct
{
    int i_data;
    int i_data_max;
    uint8_t *p_data;
} vbuf_t;

void vbuf_init( vbuf_t * );
void vbuf_add( vbuf_t *, int i_data, void *p_data );
void vbuf_reset( vbuf_t * );

typedef struct
{
    FILE *f;

    float f_fps;
    char  fcc[4];

    int   i_width;
    int   i_height;

    int64_t i_movi;
    int64_t i_movi_end;
    int64_t i_riff;

    int      i_frame;
    int      i_idx_max;
    uint32_t *idx;

    int      buflen;
    char     *buf;
} avi_t;

enum nal_unit_type_e
{
    NAL_UNKNOWN = 0,
    NAL_SLICE   = 1,
    NAL_SLICE_DPA   = 2,
    NAL_SLICE_DPB   = 3,
    NAL_SLICE_DPC   = 4,
    NAL_SLICE_IDR   = 5,    /* ref_idc != 0 */
    NAL_SEI         = 6,    /* ref_idc == 0 */
    NAL_SPS         = 7,
    NAL_PPS         = 8
    /* ref_idc == 0 for 6,9,10,11,12 */
};
enum nal_priority_e
{
    NAL_PRIORITY_DISPOSABLE = 0,
    NAL_PRIORITY_LOW        = 1,
    NAL_PRIORITY_HIGH       = 2,
    NAL_PRIORITY_HIGHEST    = 3,
};

typedef struct
{
    int i_ref_idc;  /* nal_priority_e */
    int i_type;     /* nal_unit_type_e */

    /* This data are raw payload */
    int     i_payload;
    uint8_t *p_payload;
} nal_t;

typedef struct
{
    int i_width;
    int i_height;

    int i_nal_type;
    int i_ref_idc;
    int i_idr_pic_id;
    int i_frame_num;
    int i_poc;

    int b_key;
    int i_log2_max_frame_num;
    int i_poc_type;
    int i_log2_max_poc_lsb;

    float f_fps;
} h264_t;

void avi_init( avi_t *, FILE *, float, char fcc[4] );
void avi_write( avi_t *, vbuf_t *, int  );
void avi_end( avi_t *, h264_t * );

void h264_parser_init( h264_t * );
void h264_parser_parse( h264_t *h, nal_t *n, int *pb_nal_start );


static int nal_decode( nal_t *nal, void *p_data, int i_data );

static void Help( void );
static int  Parse( int argc, char **argv, cfg_t * );
static int  ParseNAL( nal_t *nal, avi_t *a, h264_t *h, int *pb_slice );

uint8_t *find_nal_header( void *inbuf, int inbufrpos, int inbufwpos )
{
    uint8_t *absend = &((uint8_t*)inbuf)[DATA_MAX];
    uint8_t *end = &((uint8_t*)inbuf)[inbufwpos];
    /* wrap around */
    if( end >= absend )
        end -= DATA_MAX;
    uint8_t *p = &((uint8_t*)inbuf)[inbufrpos];
    if( p >= absend )
        p -= DATA_MAX;
    uint8_t *p_orig;
    int pos = 0;
    int firstbyte = 1;
    while( p != end || firstbyte )
    {
        if( p >= absend )
            p -= DATA_MAX;
        /* remember the beginning of the sequence */
        if( pos == 0 )
            p_orig = p;
        /* keep counting elements of the 0x00 0x00 0x01 byte sequence */
        if( pos == 2 )
        {
            if( *p == 0x01 )
            {
                /* exit if found entirely: seek back to the beginning of the byte sequence, signal successful search by finishing counting */
                p = p_orig;
                pos++;
                break;
            }
            else
            {
                /* if the sequence breaks at the third byte then a 0x00 0x00 sequence was already read, restart counting at the second byte instead */
                p_orig++;
                if( p_orig >= absend )
                    p_orig -= DATA_MAX;
                pos--;
                continue;
            }
        }
        else if( *p == 0x00 )
            pos++;

        /* restart counting if the sequence was broken */
        else
            pos = 0;
        p++;
        firstbyte = 0;
    }
    return ( pos == 3 ) ? p : NULL;
}

/****************************************************************************
 * main:
 ****************************************************************************/
int _tmain(int argc, _TCHAR* argv[])
{
	cfg_t cfg;

    FILE    *fout;
    FILE    *fin;

    vbuf_t  vb;
    avi_t   avi;
    h264_t  h264;

    nal_t nal;
    int i_data;
    int b_eof;
    int b_key;
    int b_slice;

#if defined(_MSC_VER) || defined(__MINGW32__) || defined(__MINGW64__)
    setmode(_fileno(stdin), _O_BINARY);    /* thanks to Marcos Morais <morais at dee.ufcg.edu.br> */
    setmode(_fileno(stdout), _O_BINARY);
#endif

    /* Parse command line */
    if( Parse( argc, argv, &cfg ) < 0 )
    {
        return -1;
    }

    /* Open input */
#ifdef STD_IN_OUT
    if( cfg.psz_fin == NULL || *cfg.psz_fin == '\0' || !strcmp( cfg.psz_fin, "-" ) )
        fin = stdin;
    else
#endif
        fin = fopen( cfg.psz_fin, "rb" );
    if( fin == NULL )
    {
        fprintf( stderr, "cannot open input file\n" );
        return -1;
    }

    /* Open output */
#ifdef STD_IN_OUT
    if( cfg.psz_fout == NULL || *cfg.psz_fout == '\0' || !strcmp( cfg.psz_fout, "-" ) )
        fout = stdout;
    else
	{
#endif
		if (i_split > 0)
		{
			char* buf = (char*)malloc(2048);
			int pos = (int)(strrchr(cfg.psz_fout, '.') - cfg.psz_fout);
			if (pos > 0)
			{
				strncpy(buf, cfg.psz_fout, pos); 
				sprintf(buf + pos, ".%02d", i_part);
				strcat(buf, cfg.psz_fout + pos);
				cfg.psz_fout = strdup(buf);
				free(buf);
			}
		}
		fout = fopen( cfg.psz_fout, "wb" );
#ifdef STD_IN_OUT
	}
#endif
    if( fout == NULL )
    {
        fprintf( stderr, "cannot open output file\n" );
        return -1;
    }

    /* Init avi */
    avi_init( &avi, fout, cfg.f_fps, cfg.fcc );

    /* Init parser */
    h264_parser_init( &h264 );

    /* Control-C handler */
    signal( SIGINT, SigIntHandler );

    /* Init data */
    b_eof = 0;
    b_key = 0;
    b_slice = 0;
    i_data  = 0;

    /* Alloc space for a nal, used for decoding pps/sps/slice header */
    nal.p_payload = (uint8_t*)malloc( DATA_MAX );

    vbuf_init( &vb );

    /* create circular buffer to speed up reading */
    int inbufeof = 0;
    int inbufempty = 1;
    int inbufrpos = 0;
    int inbufwpos = 0;
    uint8_t* inbuf = (uint8_t*)malloc( DATA_MAX );

    /* create processing buffer */
    uint8_t* databuf = (uint8_t*)malloc( DATA_MAX );

    /* split frame */
    while( !i_ctrl_c)
    {
        uint8_t *p, *p_next;
        int i_size;

        /* if there is empty room in the buffer then fill it */
        i_data = 0;
        if( ( inbufempty || inbufrpos != inbufwpos ) && !b_eof )
        {
            /* fill the end of the buffer, after the write position */
            int intoreadlen1 = ( inbufempty || inbufrpos < inbufwpos ) ? ( DATA_MAX - inbufwpos ) : 0;
            int inreadlen1 = 0;
            if( intoreadlen1 > 0 && !inbufeof ) {
                inreadlen1 = fread( &inbuf[inbufwpos], 1, intoreadlen1, fin );
                inbufeof = (inreadlen1 < intoreadlen1);
                inbufwpos += inreadlen1;
                if( inbufwpos >= DATA_MAX )
                    inbufwpos -= DATA_MAX;
                /* if something was read, the buffer is not empty */
                if( inreadlen1 > 0 )
                    inbufempty = 0;
            }
            /* fill the middle of the buffer, between the write and read position */
            int intoreadlen2;
            if( inbufempty )
            {
                inbufrpos = inbufwpos = 0;
                intoreadlen2 = DATA_MAX;
            }
            else
                intoreadlen2 = inbufrpos - inbufwpos;
            int inreadlen2 = 0;
            if( intoreadlen2 > 0 && !inbufeof ) {
                inreadlen2 = fread( &inbuf[inbufwpos], 1, intoreadlen2, fin );
                inbufeof = (inreadlen2 < intoreadlen2);
                inbufwpos += inreadlen2;
                if( inbufwpos >= DATA_MAX )
                    inbufwpos -= DATA_MAX;
                if( inreadlen2 > 0 )
                    inbufempty = 0;
            }
            if( inbufempty && inbufeof )
                b_eof = 1;

            /* the buffer length is the difference between the write and read positions */
            if( !inbufempty )
                i_data = ( inbufrpos < inbufwpos ) ? ( inbufwpos - inbufrpos ) : ( DATA_MAX - inbufrpos + inbufwpos );
        }


        if( i_data < 3 )
            break;

        /* Search begin of a NAL */
        p = find_nal_header( inbuf, inbufrpos, inbufwpos );

        if( !p )
        {
            fprintf( stderr, "garbage (i_data = %d)\n", i_data );
            i_data = 0;
            continue;
        }
        inbufrpos = p - inbuf;
        if( inbufrpos >= DATA_MAX)
            inbufrpos -= DATA_MAX;

        /* Search end of NAL */
        p_next = find_nal_header( inbuf, inbufrpos + 3, inbufwpos );

        if( !p_next && i_data < DATA_MAX )
            p_next = &inbuf[inbufwpos];
        int inbufrnextpos = p_next - inbuf;

        /* Compute NAL size */
        i_size = ( ( inbufrnextpos > inbufrpos ) ? ( inbufrnextpos - inbufrpos ) : ( DATA_MAX - inbufrpos + inbufrnextpos ) ) - 3;
        if( i_size <= 0 )
        {
            if( b_eof )
                break;

            fprintf( stderr, "nal too large (FIXME) ?\n" );
            i_data = 0;
            continue;
        }
        /* copy the end of the circular buffer to the processing buffer */
        int datacopylen = 0;
        int databufpos = 0;
        if( inbufrnextpos < inbufrpos )
        {
            datacopylen = DATA_MAX - inbufrpos;
            memcpy( databuf, &inbuf[inbufrpos], datacopylen );
            databufpos += datacopylen;
            inbufrpos = 0;
        }
        /* copy the beginning of the circular buffer to the processing buffer */
        if( inbufrpos < inbufrnextpos )
        {
            datacopylen = inbufrnextpos - inbufrpos;
            memcpy( &databuf[databufpos], &inbuf[inbufrpos], datacopylen );
            inbufrpos += datacopylen;
        }

        /* Nal start at the beginning of the processing buffer with i_size length */
        nal_decode( &nal, databuf + 3, i_size < 2048 ? i_size : 2048 );

        b_key = h264.b_key;
		// split at key frame
		if (i_split > 0 && b_key && i_written >= i_split * (i_part + 1) - DATA_MAX)
		{
			fprintf( stderr, "Pos: %" __PRI64_PREFIX "d, part: %d\n", i_written, i_part );
			// end this part
			avi.i_width  = h264.i_width;
			avi.i_height = h264.i_height;

			avi_end( &avi, &h264 );
		    fclose( fout );

			// open next
			char* q = strrchr(cfg.psz_fout, '.');
			*q = 0;
			char* p = strrchr(cfg.psz_fout, '.');
			if (p != NULL)
			{
				sprintf(p, ".%02d", ++i_part);
				*q ='.';
			}
			fout = fopen( cfg.psz_fout, "wb" );
			if( fout == NULL )
			{
				fprintf( stderr, "cannot open output file %s\n", cfg.psz_fout );
				return -1;
			}
			/* Init avi */
			avi_init( &avi, fout, cfg.f_fps, cfg.fcc );
		}

        if( b_slice && vb.i_data && ( nal.i_type == NAL_SPS || nal.i_type == NAL_PPS ) )
        {
            avi_write( &avi, &vb, b_key );
            vbuf_reset( &vb );
            b_slice = 0;
        }

        /* Parse SPS/PPS/Slice */
        if( ParseNAL( &nal, &avi, &h264, &b_slice ) && vb.i_data > 0 )
        {
            avi_write( &avi, &vb, b_key );
            vbuf_reset( &vb );
        }

        /* fprintf( stderr, "nal:%d ref:%d\n", nal.i_type, nal.i_ref_idc ); */

        /* Append NAL to buffer */
        vbuf_add( &vb, i_size + 3, databuf );

        /* Remove this nal */
        inbufrpos = inbufrnextpos;
        /* if the read position caught up with the write position then the buffer is empty */
        if( inbufrpos == inbufwpos)
            inbufempty = 1;
    }

    if( vb.i_data > 0 )
    {
        avi_write( &avi, &vb, h264.b_key );
    }

    avi.i_width  = h264.i_width;
    avi.i_height = h264.i_height;

    avi_end( &avi, &h264 );

    /* free mem */
    free( nal.p_payload );

    fclose( fin );
    fclose( fout );

    return 0;
}


/*****************************************************************************
 * Help:
 *****************************************************************************/
static void Help( void )
{
#ifdef STD_IN_OUT
#define OPEN_OPT_PAR " ["
#define CLOSE_OPT_PAR " ]"
#define SEP_OPT_PAR " "
#else
#define OPEN_OPT_PAR ""
#define CLOSE_OPT_PAR ""
#define SEP_OPT_PAR ""
#endif
    fprintf( stderr,
			 "avc2avi mod (build: 2013-01-27)\n"
             "Syntax: avc2avi [options]" OPEN_OPT_PAR " -i input.h264" CLOSE_OPT_PAR SEP_OPT_PAR OPEN_OPT_PAR " -o output.avi" CLOSE_OPT_PAR "\n"
             "\n"
             "  -h, --help                  Print this help\n"
             "\n"
             "  -i, --input                 Specify input file"
#ifdef STD_IN_OUT
             " (default: stdin)"
#endif
             "\n"
             "  -o, --output                Specify output file"
#ifdef STD_IN_OUT
             " (default: stdout)"
#endif
             "\n"
             "\n"
             "  -f, --fps <float>           Set FPS (default: from input or %.3f)\n"
             "  -c, --codec <string>        Set the codec fourcc (default: 'h264')\n"
             "\n"
             "  -s, --split <pos> MB        Specify splitpoint (default: 2000 MB)\n"
             "\n",
             FPS_DEFAULT );
}

/*****************************************************************************
 * Parse:
 *****************************************************************************/
static int  Parse( int argc, char **argv, cfg_t *cfg )
{
    /* Set default values */
    cfg->psz_fin = NULL;
    cfg->psz_fout = NULL;
    cfg->f_fps = FPS_DEFAULT;
    memcpy( cfg->fcc, "h264", 4 );

    /* Parse command line options */
    for( ;; )
    {
        int c;

#ifdef LONG_OPTIONS
        //int long_options_index;
        static struct option long_options[] =
        {
            { "help",   no_argument,       NULL, 'h' },
            { "input",  required_argument, NULL, 'i' },
            { "output", required_argument, NULL, 'o' },
            { "fps",    required_argument, NULL, 'f' },
            { "codec",  required_argument, NULL, 'c' },
            { "split",  required_argument, NULL, 's' },
            {0, 0, 0, 0}
        };

		c = getopt_long( argc, argv, "hi:o:f:c:s:", long_options, NULL );
#else
		c = getopt( argc, argv, "hi:o:f:c:s:");
#endif


        if( c == -1 )
        {
            break;
        }

        switch( c )
        {
            case 'h':
                Help();
                return -1;

            case 0:
                break;
            case 'i':
                cfg->psz_fin = strdup( optarg );
                break;
            case 'o':
                cfg->psz_fout = strdup( optarg );
                break;
            case 'f':
                cfg->f_fps = (float)atof( optarg );
                break;
            case 'c':
                memset( cfg->fcc, ' ', 4 );
                memcpy( cfg->fcc, optarg, strlen( optarg ) < 4 ? strlen( optarg ) : 4 );
                break;
			case 's':
				i_split = atoi(optarg);
				if (i_split == 0)
					i_split = 2000;

				i_split *= 1024 * 1024;
				break;

            default:
                return -1;
        }
    }

#ifndef STD_IN_OUT
    /* quit if either the input or output file is unspecified */
    if( cfg->psz_fin == NULL || *cfg->psz_fin == '\0' || !strcmp( cfg->psz_fin, "-" )  ||
        cfg->psz_fout == NULL || *cfg->psz_fout == '\0' || !strcmp( cfg->psz_fout, "-" ) )
    {
        Help();
        return -1;
    }
#endif
    
    return 0;
}

/*****************************************************************************
 * h264_parser_*:
 *****************************************************************************/
void h264_parser_init( h264_t *h )
{
    h->i_width = 0;
    h->i_height = 0;
    h->b_key = 0;
    h->i_nal_type = -1;
    h->i_ref_idc = -1;
    h->i_idr_pic_id = -1;
    h->i_frame_num = -1;
    h->i_log2_max_frame_num = 0;
    h->i_poc = -1;
    h->i_poc_type = -1;
    h->f_fps = -1;
}
void h264_parser_parse( h264_t *h, nal_t *nal, int *pb_nal_start )
{
    bs_t s;
    *pb_nal_start = 0;

    if( nal->i_type == NAL_SPS || nal->i_type == NAL_PPS )
        *pb_nal_start = 1;

    bs_init( &s, nal->p_payload, nal->i_payload );
    if( nal->i_type == NAL_SPS )
    {
        int i_tmp;

        i_tmp = bs_read( &s, 8 );
        bs_skip( &s, 1+1+1 + 5 + 8 );
        /* sps id */
        bs_read_ue( &s );

        if( i_tmp >= 100 )
        {
            b_hiprofile = 1;
			bs_read_ue( &s ); // chroma_format_idc
            bs_read_ue( &s ); // bit_depth_luma_minus8
            bs_read_ue( &s ); // bit_depth_chroma_minus8
            bs_skip( &s, 1 ); // qpprime_y_zero_transform_bypass_flag
            if( bs_read( &s, 1 ) ) // seq_scaling_matrix_present_flag
            {
                int i, j;
                for( i = 0; i < 8; i++ )
                {
                    if( bs_read( &s, 1 ) ) // seq_scaling_list_present_flag[i]
                    {
                        uint8_t i_tmp = 8;
                        for( j = 0; j < (i<6?16:64); j++ )
                        {
                            i_tmp += bs_read_se( &s );
                            if( i_tmp == 0 )
                                break;
                        }
                    }
                }
            }
        }

        /* Skip i_log2_max_frame_num */
        h->i_log2_max_frame_num = bs_read_ue( &s ) + 4;
        /* Read poc_type */
        h->i_poc_type = bs_read_ue( &s );
        if( h->i_poc_type == 0 )
        {
            h->i_log2_max_poc_lsb = bs_read_ue( &s ) + 4;
        }
        else if( h->i_poc_type == 1 )
        {
            int i_cycle;
            /* skip b_delta_pic_order_always_zero */
            bs_skip( &s, 1 );
            /* skip i_offset_for_non_ref_pic */
            bs_read_se( &s );
            /* skip i_offset_for_top_to_bottom_field */
            bs_read_se( &s );
            /* read i_num_ref_frames_in_poc_cycle */
            i_cycle = bs_read_ue( &s ); 
            if( i_cycle > 256 ) i_cycle = 256;
            while( i_cycle > 0 )
            {
                /* skip i_offset_for_ref_frame */
                bs_read_se(&s );
            }
        }
        /* i_num_ref_frames */
        bs_read_ue( &s );
        /* b_gaps_in_frame_num_value_allowed */
        bs_skip( &s, 1 );

        /* Read size */
        h->i_width  = 16 * ( bs_read_ue( &s ) + 1 );
        h->i_height = 32 * ( bs_read_ue( &s ) + 1 );

        /* b_frame_mbs_only */
        if( bs_read1( &s ) )
        {
			h->i_height /= 2;
		}
		else
		{
			// mb_adaptive_frame_field_flag
            bs_skip( &s, 1 );
        }
        /* b_direct8x8_inference */
        bs_skip( &s, 1 );

        /* crop ? */
        if( bs_read1( &s ) )
        {
            /* left */
            h->i_width -= 2 * bs_read_ue( &s );
            /* right */
            h->i_width -= 2 * bs_read_ue( &s );
            /* top */
            h->i_height -= 2 * bs_read_ue( &s );
            /* bottom */
            h->i_height -= 2 * bs_read_ue( &s );
        }

        /* vui */
        if( bs_read1( &s ) )
        {
            /* aspect ratio info */
            if( bs_read1( &s ) )
            {
                /* aspect ratio idc = SAR_reserved? */
                if( bs_read( &s, 8 ) == 255 )
                    bs_skip( &s, 32);
            }
            /* overscan info */
            if( bs_read1( &s ) )
                bs_skip( &s, 1);
            /* video signal type */
            if( bs_read1( &s ) )
            {
                bs_skip( &s, 4);
                /* color description */
                if( bs_read1( &s ) )
                    bs_skip( &s, 24);
            }
            /* chroma loc info */
            if( bs_read1( &s ) )
            {
                i_tmp = bs_read_ue( &s );
                i_tmp = bs_read_ue( &s );
            }
            /* timing info */
            if( bs_read1( &s ) )
            {
                uint32_t num_units_in_tick = bs_read( &s, 32 );
                uint32_t time_scale = bs_read( &s, 32 );
                /* fixed frame rate */
                if( bs_read1( &s ) )
                    h->f_fps = ( (float) time_scale ) / ( num_units_in_tick * 2 );
            }
        }
    }
    else if( nal->i_type >= NAL_SLICE && nal->i_type <= NAL_SLICE_IDR )
    {
        int i_tmp;

        /* i_first_mb */
        bs_read_ue( &s );
        /* picture type */
        switch( bs_read_ue( &s ) )
        {
            case 0: case 5: /* P */
            case 1: case 6: /* B */
            case 3: case 8: /* SP */
                h->b_key = 0;
                break;
            case 2: case 7: /* I */
            case 4: case 9: /* SI */
                h->b_key = (nal->i_type == NAL_SLICE_IDR || (b_hiprofile && nal->i_type == NAL_SLICE));
                break;
        }
        /* pps id */
        bs_read_ue( &s );

        /* frame num */
        i_tmp = bs_read( &s, h->i_log2_max_frame_num );

        if( i_tmp != h->i_frame_num )
            *pb_nal_start = 1;

        h->i_frame_num = i_tmp;

        if( nal->i_type == NAL_SLICE_IDR )
        {
            i_tmp = bs_read_ue( &s );
            if( h->i_nal_type == NAL_SLICE_IDR && h->i_idr_pic_id != i_tmp )
                *pb_nal_start = 1;

            h->i_idr_pic_id = i_tmp;
        }

        if( h->i_poc_type == 0 )
        {
            i_tmp = bs_read( &s, h->i_log2_max_poc_lsb );
            if( i_tmp != h->i_poc )
                *pb_nal_start = 1;
            h->i_poc = i_tmp;
        }
    }
    h->i_nal_type = nal->i_type;
    h->i_ref_idc = nal->i_ref_idc;
}


static int  ParseNAL( nal_t *nal, avi_t *a, h264_t *h, int *pb_slice )
{
    __unused_variable__(a);
    int b_flush = 0;
    int b_start;

    h264_parser_parse( h, nal, &b_start );

    if( b_start && *pb_slice )
    {
        b_flush = 1;
        *pb_slice = 0;
    }

    if( nal->i_type >= NAL_SLICE && nal->i_type <= NAL_SLICE_IDR )
        *pb_slice = 1;

    return b_flush;
}

/*****************************************************************************
 * vbuf: variable buffer
 *****************************************************************************/
void vbuf_init( vbuf_t *v )
{
    v->i_data = 0;
    v->i_data_max = 10000;
    v->p_data = (uint8_t*)malloc( v->i_data_max );
}
void vbuf_add( vbuf_t *v, int i_data, void *p_data )
{
    if( i_data + v->i_data >= v->i_data_max )
    {
        v->i_data_max += i_data;
        v->p_data = (uint8_t*)realloc( v->p_data, v->i_data_max );
    }
    memcpy( &v->p_data[v->i_data], p_data, i_data );

    v->i_data += i_data;
}
void vbuf_reset( vbuf_t *v )
{
    v->i_data = 0;
}

/*****************************************************************************
 * avi:
 *****************************************************************************/
/* the current file position is the actual value plus the length of the unwritten part of the buffer */
int64_t avi_ftell( avi_t *a )
{
    return ftell( a->f ) + a->buflen;
}

/* flush the buffer if appending would cause overflow */
void avi_write_flush( avi_t *a, int writelen )
{
    if( a->buflen + writelen >= DATA_MAX )
    {
        fwrite( a->buf, 1, a->buflen, a->f );
        a->buflen = 0;
    }
}

/* redirect all writes to the buffer */
void avi_write_uint8( avi_t *a, uint8_t b )
{
    avi_write_flush( a, 1 );
    a->buf[a->buflen++] = b;
}

int avi_write_buf( avi_t *a, void *buf, int buflen )
{
    avi_write_flush( a, buflen );
    /* write too large buffer directly */
    if( buflen >= DATA_MAX )
    {
        avi_write_flush( a, DATA_MAX );
        fwrite( buf, 1, buflen, a->f );
    }
    else
    {
        memcpy( &a->buf[a->buflen], buf, buflen );
        a->buflen += buflen;
    }
    return buflen;
}

void avi_write_uint16( avi_t *a, uint16_t w )
{
    avi_write_uint8( a, ( w      ) & 0xff );
    avi_write_uint8( a, ( w >> 8 ) & 0xff );

	i_written += 2;
}

void avi_write_uint32( avi_t *a, uint32_t dw )
{
    avi_write_uint8( a, ( dw      ) & 0xff );
    avi_write_uint8( a, ( dw >> 8 ) & 0xff );
    avi_write_uint8( a, ( dw >> 16) & 0xff );
    avi_write_uint8( a, ( dw >> 24) & 0xff );

	i_written += 4;
}

void avi_write_fourcc( avi_t *a, char fcc[4] )
{
    avi_write_uint8( a, fcc[0] );
    avi_write_uint8( a, fcc[1] );
    avi_write_uint8( a, fcc[2] );
    avi_write_uint8( a, fcc[3] );

	i_written += 4;
}

/* Flags in avih */
#define AVIF_HASINDEX       0x00000010  // Index at end of file?
#define AVIF_ISINTERLEAVED  0x00000100
#define AVIF_TRUSTCKTYPE    0x00000800  // Use CKType to find key frames?

#define AVIIF_KEYFRAME      0x00000010L /* this frame is a key frame.*/

void avi_write_header( avi_t *a )
{
    avi_write_fourcc( a, "RIFF" );
    avi_write_uint32( a, (uint32_t)(a->i_riff > 0 ? a->i_riff - 8 : 0xFFFFFFFF) );
    avi_write_fourcc( a, "AVI " );

    avi_write_fourcc( a, "LIST" );
    avi_write_uint32( a,  4 + 4*16 + 12 + 4*16 + 4*12 );
    avi_write_fourcc( a, "hdrl" );

    avi_write_fourcc( a, "avih" );
    avi_write_uint32( a, 4*16 - 8 );
    /* round FPS numerator properly */
    avi_write_uint32( a, (uint32_t)(1000000 / a->f_fps + 0.5) );
    avi_write_uint32( a, 0xffffffff );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, AVIF_HASINDEX|AVIF_ISINTERLEAVED|AVIF_TRUSTCKTYPE);
    avi_write_uint32( a, a->i_frame );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, 1 );
    avi_write_uint32( a, 1000000 );
    avi_write_uint32( a, a->i_width );
    avi_write_uint32( a, a->i_height );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, 0 );

    avi_write_fourcc( a, "LIST" );
    avi_write_uint32( a,  4 + 4*16 + 4*12 );
    avi_write_fourcc( a, "strl" );

    avi_write_fourcc( a, "strh" );
    avi_write_uint32( a,  4*16 - 8 );
    avi_write_fourcc( a, "vids" );
    avi_write_fourcc( a, a->fcc );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, 1000 );
    avi_write_uint32( a, (uint32_t)(a->f_fps * 1000 + 0.5) );
    avi_write_uint32( a, 0 );
    avi_write_uint32( a, a->i_frame );
    avi_write_uint32( a, 1024*1024 );
    avi_write_uint32( a, -1 );
    avi_write_uint32( a, a->i_width * a->i_height );
    avi_write_uint32( a, 0 );
    avi_write_uint16( a, a->i_width );
    avi_write_uint16( a, a->i_height );

    avi_write_fourcc( a, "strf" );
    avi_write_uint32( a,  4*12 - 8 );
    avi_write_uint32( a,  4*12 - 8 );
    avi_write_uint32( a,  a->i_width );
    avi_write_uint32( a,  a->i_height );
    avi_write_uint16( a,  1 );
    avi_write_uint16( a,  24 );
    avi_write_fourcc( a,  a->fcc );
    avi_write_uint32( a, a->i_width * a->i_height );
    avi_write_uint32( a,  0 );
    avi_write_uint32( a,  0 );
    avi_write_uint32( a,  0 );
    avi_write_uint32( a,  0 );

    avi_write_fourcc( a, "LIST" );
    avi_write_uint32( a,  (uint32_t)(a->i_movi_end > 0 ? a->i_movi_end - a->i_movi + 4: 0xFFFFFFFF) );
    avi_write_fourcc( a, "movi" );
}

void avi_write_idx( avi_t *a )
{
    avi_write_fourcc( a, "idx1" );
    avi_write_uint32( a,  a->i_frame * 16 );
    i_written += avi_write_buf( a, a->idx, a->i_frame * 16 );
}

void avi_init( avi_t *a, FILE *f, float f_fps, char fcc[4] )
{
    a->f = f;
    a->f_fps = f_fps;
    memcpy( a->fcc, fcc, 4 );
    a->i_width = 0;
    a->i_height = 0;
    a->i_frame = 0;
    a->i_movi = 0;
    a->i_riff = 0;
    a->i_movi_end = 0;
    a->i_idx_max = 0;
    a->idx = NULL;

    a->buflen = 0;
    a->buf = (char*)malloc(DATA_MAX);
    
    avi_write_header( a );

    a->i_movi = avi_ftell( a );
}

static void avi_set_dw( void *_p, uint32_t dw )
{
    uint8_t *p = (uint8_t*)_p;

    p[0] = ( dw      )&0xff;
    p[1] = ( dw >> 8 )&0xff;
    p[2] = ( dw >> 16)&0xff;
    p[3] = ( dw >> 24)&0xff;
}

void avi_write( avi_t *a, vbuf_t *v, int b_key )
{
    int64_t i_pos = avi_ftell( a );

    /* chunk header */
    avi_write_fourcc( a, "00dc" );
    avi_write_uint32( a, v->i_data );

    i_written += avi_write_buf( a, v->p_data, v->i_data );

    if( v->i_data&0x01 )
    {
        /* pad */
        avi_write_uint8( a, 0 );
		i_written++;
    }

    /* Append idx chunk */
    if( a->i_idx_max <= a->i_frame )
    {
        a->i_idx_max += 10000;
        a->idx = (uint32_t*)realloc( a->idx, a->i_idx_max * 16 );
    }

    memcpy( &a->idx[4*a->i_frame+0], "00dc", 4 );
    avi_set_dw( &a->idx[4*a->i_frame+1], b_key ? AVIIF_KEYFRAME : 0 );
    avi_set_dw( &a->idx[4*a->i_frame+2], (uint32_t)i_pos );
    avi_set_dw( &a->idx[4*a->i_frame+3], v->i_data );

    a->i_frame++;
}

void avi_end( avi_t *a, h264_t *h )
{
    /* fetch FPS, if present, from input */
    if( h->f_fps > 0 )
        a->f_fps = h->f_fps;
    
    avi_write_flush( a, DATA_MAX );
    a->i_movi_end = avi_ftell( a );

    /* write index */
    avi_write_idx( a );
    avi_write_flush( a, DATA_MAX );

    a->i_riff = avi_ftell( a );
   
    /* write header */
    fseek( a->f, 0, SEEK_SET );
    avi_write_header( a );
    avi_write_flush( a, DATA_MAX );

    fprintf( stderr, "avi file written\n" );
    fprintf( stderr, "  - codec: %4.4s\n", a->fcc );
    fprintf( stderr, "  - size: %dx%d\n", a->i_width, a->i_height );
    fprintf( stderr, "  - fps: %.3f\n", a->f_fps );
    fprintf( stderr, "  - frames: %d\n", a->i_frame );
}

/*****************************************************************************
 * nal:
 *****************************************************************************/
int nal_decode( nal_t *nal, void *p_data, int i_data )
{
    uint8_t *src = (uint8_t*)p_data;
    uint8_t *end = &src[i_data];
    uint8_t *dst = nal->p_payload;

    nal->i_type    = src[0]&0x1f;
    nal->i_ref_idc = (src[0] >> 5)&0x03;

    src++;

    while( src < end )
    {
        if( src < end - 3 && src[0] == 0x00 && src[1] == 0x00  && src[2] == 0x03 )
        {
            *dst++ = 0x00;
            *dst++ = 0x00;

            src += 3;
            continue;
        }
        *dst++ = *src++;
    }

    nal->i_payload = (int)(dst - (uint8_t*)nal->p_payload);
    return 0;
}
