/***************************************************************************//**
 * @file DSAsmPP.c
 * @author Joe Forster/STA
 * @brief Implementation of the assembler source preprocessor.
 ******************************************************************************/

/*
 * DSAsmPP
 *
 * Created by Joe Forster/STA; part of DeadStrip, by Dorian Weber.
 *
 * No warranties given, not even implied ones, use this tool at your own risk.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <io.h>

#include "common.h"

#define SO_PROCESS_ONLY     1         /**<@brief Does not run assembler, only preprocesses the input. */

#define SO_ASSEMBLER        "as"      /**<@brief The default assembler. */

#define ASM_EXT_1           ".s"              /**<@brief Known extensions of assembler source files. */
#define ASM_EXT_2           ".asm"

#define SO_FILEDIRECTIVE    ".file"
#define SO_GLOBALDIRECTIVE  ".globl"
#define SO_SECTIONDIRECTIVE ".section"
#define SO_RDATASECTION     ".rdata"
#define SO_ALIGNDIRECTIVE   ".align"
#define SO_RDATASUFFIX      "$$"

#define SO_FOUNDGLOBAL      1
#define SO_FOUNDSECTION     2
#define SO_FOUNDLABEL       4

void appendToSectName(char* sectnameend, char* src, int srclen) {
  memmove(sectnameend + (sizeof(SO_RDATASUFFIX) - 1) + srclen, sectnameend, strlen(sectnameend) + 1);
  strncpy(sectnameend, SO_RDATASUFFIX, (sizeof(SO_RDATASUFFIX) - 1));
  sectnameend += (sizeof(SO_RDATASUFFIX) - 1);
  strncpy(sectnameend, src, srclen);
}

/**@brief Preprocesses an assembler source file.
 * @param[in] inname input filename
 * @param[in] outname output filename
 * @return error code; if negative, warning code
 */
int processAsmFile(char* inname, char* outname) {
  FILE* infile = inname ? fopen(inname, "r") : stdin;
  if (infile == NULL) {
    fprintf(stderr, "ERROR: could not open assembler source.\n");
    return 1;
  }
  FILE* outfile = outname ? fopen(outname, "w") : stdout;
  if (outfile == NULL) {
    fclose(infile);
    fprintf(stderr, "ERROR: could not open output file.\n");
    return 2;
  }
  char infname[MAX_PATH + 1];
  *infname = '\0';
  char buffer[SO_LINE_MAX + 1],
    gsymname[SO_LINE_MAX + 1],
    sectline[SO_LINE_MAX + 1],
    sectname[SO_LINE_MAX + 1],
    sectflags[SO_LINE_MAX + 1],
    alignline[SO_LINE_MAX + 1];
  char* ptr;
  char* symstart = NULL;
  char* sectnameend = NULL;
  int progress = 0,
    modified = 0,
    prevlinegsym = 0;
  *gsymname = *sectline = *sectname = *sectflags = *alignline = '\0';
  while (!feof(infile)) {
    progress &= ~SO_FOUNDLABEL;
    if (!prevlinegsym) {
      progress &= ~SO_FOUNDGLOBAL;
    }
    prevlinegsym = 0;
    *buffer = '\0';
    fgets(buffer, sizeof(buffer), infile);
    ptr = trim(buffer);
    int found = 0,
      write = 1;

    /* recognize ".file" directive, extract original file name */
    if (strncmp(ptr, SO_FILEDIRECTIVE, sizeof(SO_FILEDIRECTIVE) - 1) == 0
        && isspace(ptr[sizeof(SO_FILEDIRECTIVE) - 1])) {
      char* curinname;
      for (curinname = &ptr[sizeof(SO_FILEDIRECTIVE)]; isspace(*curinname); curinname++);
      if (*curinname == '\"') curinname++;
      char* curinfname;
      /* allow only alphanumerical characters and a few symbols in the object filename, replace everything else with underscores */
      for (curinfname = infname; *curinname != '\0' && *curinname != '\"'; curinname++, curinfname++) {
        char curinnamechar = tolower(*curinname);
        if ((curinnamechar >= 'a' && curinnamechar <= 'z')
            || (curinnamechar >= '0' && curinnamechar <= '9')
            || strchr(".$", curinnamechar) != NULL) {
          *curinfname = *curinname;
        } else {
          *curinfname = '_';
        }
      }
      *curinfname = '\0';

    /* recognize ".globl" directive, extract global symbol name */
    } else if (strncmp(ptr, SO_GLOBALDIRECTIVE, sizeof(SO_GLOBALDIRECTIVE) - 1) == 0
        && isspace(ptr[sizeof(SO_GLOBALDIRECTIVE) - 1])) {
      found = 1;
      progress = (progress | SO_FOUNDGLOBAL) & ~SO_FOUNDSECTION;
      char* buf = trim(&ptr[sizeof(SO_GLOBALDIRECTIVE) - 1]);
      char* bufend;
      for (bufend = buf; *bufend && !isspace(*bufend); bufend++);
      strncpy(gsymname, buf, bufend - buf);
      gsymname[bufend - buf] = '\0';
      prevlinegsym = 1;

    /* recognize ".section" directive, extract section name */
    } else if (strncmp(ptr, SO_SECTIONDIRECTIVE, sizeof(SO_SECTIONDIRECTIVE) - 1) == 0
        && isspace(ptr[sizeof(SO_SECTIONDIRECTIVE) - 1])) {
      char* buf = trim(&ptr[sizeof(SO_SECTIONDIRECTIVE) - 1]);
      for (sectnameend = buf; *sectnameend && *sectnameend != ',' && !isspace(*sectnameend); sectnameend++);
      strncpy(sectname, buf, sectnameend - buf);
      sectname[sectnameend - buf] = '\0';

      /* act upon ".rdata" section name only */
      if (strcmp(sectname, SO_RDATASECTION) == 0) {
        found = 1;
        progress |= SO_FOUNDSECTION;

        /* if a section name appears NOT after a global symbol name, defer the line */
        if (!(progress & SO_FOUNDGLOBAL)) {
          strcpy(sectline, buffer);
          write = 0;
        }
      } else {
        progress &= ~SO_FOUNDSECTION;
      }

    /* recognize label */
    } else if (ptr[0] == 'L') {
      symstart = ptr;
      if (*(++ptr) == 'C') ptr++;
      while (isdigit(*ptr)) ptr++;
      if (*ptr == ':') {
        found = 1;
        progress |= SO_FOUNDLABEL;
      }

    /* recognize ".align" directive, defer the line */
    } else if (strncmp(ptr, SO_ALIGNDIRECTIVE, sizeof(SO_ALIGNDIRECTIVE) - 1) == 0
        && isspace(ptr[sizeof(SO_ALIGNDIRECTIVE) - 1])) {
      strcpy(alignline, buffer);
      write = 0;
    }
    if (found) {
      switch (progress) {

        /* if a section name appears after a global symbol name, append the global symbol name to the section name */
        case SO_FOUNDGLOBAL | SO_FOUNDSECTION: {
          progress = 0;
          modified = 1;
          appendToSectName(sectnameend, gsymname, strlen(gsymname));
          break;
        }

        /* if a label appears after a section name, append the original file name and the label to the section name and write all */
        case SO_FOUNDSECTION | SO_FOUNDLABEL: {
          if (*infname == '\0') {
            fprintf(stderr, "ERROR: no original file name in assembler source.\n");
            return 3;
          }
          modified = 1;
          char tmpbuf[SO_LINE_MAX + 1];
          strcpy(tmpbuf, sectline);
          char* insertpoint = &tmpbuf[sectnameend - buffer];
          appendToSectName(insertpoint, symstart, ptr - symstart);
          if (*infname != '\0') {
            appendToSectName(insertpoint, infname, strlen(infname));
          }
          fputs(tmpbuf, outfile);
          break;
        }
      }
    }
    if (write) {

      /* write deferred ".align" directive, if any */
      if (*alignline != '\0') {
        fputs(alignline, outfile);
        *alignline = '\0';
      }
      fputs(buffer, outfile);
    }
  }
  fclose(infile);
  fclose(outfile);
  if (!modified) {
    remove(outname);
    return -1;
  }
  return 0;
}

int main(int argc, char* argv[]) {
  unsigned long assemblerflags = getDefaultToolFlags(SO_ASSEMBLER, argv[0]);
  char assembler[SO_LINE_MAX];
  *assembler = '\0';

  int inputisfile = 0;
  unsigned long flags = 0;
  char** aargs = (char**)calloc(sizeof(char*), argc);
  char** afiles = (char**)calloc(sizeof(char*), argc);
  int argi,
    alen = 0;
  for (argi = 1; argi < argc; argi++) {
    char* arg = argv[argi];

    /* process command line options */
    if (arg[0] == '-' && arg[1] == '-') {
      arg += 2;
      if (strcmp(arg, "proc") == 0) {
        flags |= SO_PROCESS_ONLY;
      } else if (strcmp(arg, "assembler") == 0) {
        if (++argi < argc) {
          strcpy(assembler, argv[argi]);
        }
      } else if (strcmp(arg, "x64") == 0) {
        assemblerflags |= SO_TOOL_X86TOX64;
      }
    } else {
      int argl = strlen(arg);
      aargs[argi] = arg;

      /* preprocess file if it is an assembler source */
      if (argl >= 3 &&
        (stricmp(&arg[argl - (sizeof(ASM_EXT_1) - 1)], ASM_EXT_1) == 0
          || stricmp(&arg[argl - (sizeof(ASM_EXT_2) - 1)], ASM_EXT_2) == 0)) {
        inputisfile = 1;
        char* argnamepos = findNamePos(arg);
        char* outname = (char*)malloc(strlen(arg) + sizeof(SO_DFILE));
        int argpathlen = argnamepos - arg;
        strncpy(outname, arg, argpathlen);
        strcpy(&outname[argpathlen], SO_DFILE);
        strcpy(&outname[argpathlen + (sizeof(SO_DFILE) - 1)], argnamepos);
        int ret = processAsmFile(arg, outname);
        if (ret > 0) {
          break;
        } else if (ret == 0) {
          afiles[argi] = aargs[argi] = outname;
        }
      }
      alen += strlen(aargs[argi]) + 1;
    }
  }
  if (*assembler == '\0') {
    getDefaultToolName(assembler, SO_ASSEMBLER, assemblerflags);
  }
  if (flags & SO_PROCESS_ONLY) {
    processAsmFile(NULL, NULL);
  } else {
    char* cmdLn = (char*)malloc(strlen(assembler) + 1 + alen + 1);
    char* tmpcmdLn = cmdLn;
    SO_SC(tmpcmdLn, assembler);
    for (argi = 1; argi < argc; argi++) {
      SO_SC(tmpcmdLn, " ");
      SO_SC(tmpcmdLn, aargs[argi]);
    }

    /* if there were no assembler sources then the source comes on standard input and must go through to standard output */
    if (inputisfile) {
      system(cmdLn);
    } else {
      dup2(fileno(popen(cmdLn, "w")), fileno(stdout));
      processAsmFile(NULL, NULL);
      pclose(stdout);
    }
    free(cmdLn);
    for (argi = 1; argi < argc; argi++) {
      if (afiles[argi] != NULL) {
        remove(afiles[argi]);
      }
    }
  }
  return 0;
}
