/***************************************************************************//**
 * @file main.c
 * @author Dorian Weber
 * @brief Contains the programs entry point and flow control.
 ******************************************************************************/

/*
 * DeadStrip
 *
 * Created by Dorian Weber.
 *
 * No warranties given, not even implied ones, use this tool at your own risk.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common.h"
#include "list.h"
#include "objectFile.h"

/* modify those, if you're migrating onto a new system */
#define SO_LINKER  "ld"               /**<@brief The default linker. */
#define SO_DUMPER  "objdump"          /**<@brief The object file dumper. */
#define SO_DPARAM  "-rh"              /**<@brief Parameters for the dumper. */
#define SO_REMOVER "objcopy"          /**<@brief Object copy tool. */
#define SO_RPARAM  "--strip-unneeded" /**<@brief Parameters for the copier. */
#define SO_RRMV    "-R"               /**<@brief Parameter to remove sections. */
#define SO_PIPE    ">"                /**<@brief Pipe symbol. */
#define SO_LINKOUT "a.exe"            /**<@brief Default filename for linker output. */

#define OBJ_EXT_1           ".o"              /**<@brief Known extensions of assembler source files. */
#define OBJ_EXT_2           ".obj"

static const char* hlp = "Usage: deadstrip [options] file...\n"
	"Options:\n"
	"  --help                display this HELP\n"
	"  --dcmd                Dumps the ComManD line\n"
	"  --ddis                Dumps DIScarded sections\n"
	"  --duse                Dumps USEd sections\n"
	"  --dmap                Dump the dependency MAP\n"
	"  --linker <filename>   use alternative LINKER (default: "SO_LINKER")\n"
	"  --dnrm                Do Not ReMove any sections\n"
	"  --rems <section>      REMoVe a Section\n"
	"  --save <item>         SAVE an item and its dependencies\n"
	"    > just pass the decorated variable/function name, not the section\n"
	"    > the main function gets saved by default\n"
	"\n"
	"Version 1.21 mod\n"
	"Last compiled on "__DATE__".\n";

#define SO_DUMP_CMDLN      1
#define SO_DUMP_DISCARDED  2
#define SO_DUMP_USED       4
#define SO_OBJECTS         8
#define SO_HELP           16
#define SO_DUMP_MAP       32
#define SO_DNRM           64
#define SO_COLLECT       128

int main(int argc, const char* argv[])
{
        unsigned long linkerflags = getDefaultToolFlags(SO_LINKER, argv[0]);
	char linker[SO_LINE_MAX];
        *linker = 0;

        unsigned long dumperflags = getDefaultToolFlags(SO_DUMPER, argv[0]);
	char dumper[SO_LINE_MAX];
	getDefaultToolName(dumper, SO_DUMPER, dumperflags);
	strcat(dumper, " " SO_DPARAM " ");

        unsigned long removerflags = getDefaultToolFlags(SO_REMOVER, argv[0]);
	char remover[SO_LINE_MAX];
	getDefaultToolName(remover, SO_REMOVER, removerflags);
	strcat(remover, " " SO_RPARAM " ");

	char** largs = (char**) malloc(sizeof(char*) * argc);
	objectFile** lobjs = (objectFile**) calloc(sizeof(objectFile*), argc);
	int i = argc, li = 0, llen = strlen(linker) + 2, olen = strlen(dumper)
	    + sizeof(SO_PIPE SO_DFILE), len;
	unsigned long flags = 0;
	list *lObject = newList(), *lSeed = newList();
	lSkipSection = newList();

	/* extract linker output name, default to "a.exe" */
	char* loutput = (char*) malloc(sizeof(SO_LINKOUT) + 1);
	strcpy(loutput, SO_LINKOUT);
	{
		int i;
		char** v;
		for (i = argc, v = (char**)argv; --i > 0;)
		{
			++v;
			if (!strcmp(*v, "-o") && i > 1)
                                loutput = (char*)*(v + 1);
		}
	}

	/* add console program/GUI program/DLL main procedure as seed for the graph coloring algorithm */
	listAdd(lSeed, "main");
	listAdd(lSeed, "WinMain");
	listAdd(lSeed, "DllMain");

	/* extract infos */
	while (--i > 0)
	{
		++argv;
		flags &= ~SO_COLLECT;
		if (**argv == '-')
		{
			if (argv[0][1] == '-')
			{
				if (!strcmp(*argv, "--save"))
				{
					++argv;

					if (--i)
						listAdd(lSeed, *argv);

					continue;
				}
				else if (!strcmp(*argv, "--rems"))
				{
					++argv;

					if (--i)
						listAdd(lSkipSection, *argv);

					continue;
				}
				else if (!strcmp(*argv, "--help"))
				{
					flags |= SO_HELP;
					continue;
				}
				else if (!strcmp(*argv, "--dcmd"))
				{
					flags |= SO_DUMP_CMDLN;
					continue;
				}
				else if (!strcmp(*argv, "--ddis"))
				{
					flags |= SO_DUMP_DISCARDED;
					continue;
				}
				else if (!strcmp(*argv, "--duse"))
				{
					flags |= SO_DUMP_USED;
					continue;
				}
				else if (!strcmp(*argv, "--dmap"))
				{
					flags |= SO_DUMP_MAP;
					continue;
				}
				else if (!strcmp(*argv, "--linker"))
				{
					++argv;
					if (--i)
					{
						llen += strlen(*argv) - strlen(linker);
						strcpy(linker, *argv);
					}
					continue;
				}
				else if (!strcmp(*argv, "--x64"))
				{
					linkerflags |= SO_TOOL_X86TOX64;
					continue;
				}
				else if (!strcmp(*argv, "--dnrm"))
				{
					flags |= SO_DNRM;
					continue;
				}
			}
			else if (!strncmp(*argv, "-o", sizeof("-o") - 1))
			{

				/* command line looks like -o Executable */
				if (!argv[0][sizeof("-o") - 1])
				{
					llen += sizeof("-o");
					largs[li++] = (char*) *argv;
					continue;
				}
			}
		}

       		/* check for known extensions of object files */
       		else if (strlen(*argv) >= 3 &&
       			(stricmp(&(*argv)[strlen(*argv) - (sizeof(OBJ_EXT_1)-1)], OBJ_EXT_1) == 0
       				|| stricmp(&(*argv)[strlen(*argv) - (sizeof(OBJ_EXT_2)-1)], OBJ_EXT_2) == 0
       			))
       		{
       			flags |= SO_COLLECT | SO_OBJECTS;
       		}

		len = strlen(*argv) + 1;
		char* argvtmp = (char*) *argv;


		/* collect objectfiles, leave their respective argument empty */
		if (flags & SO_COLLECT)
		{
			objectFile* tempobj = objectFileCreate(*argv, loutput);
			listAdd(lObject, tempobj);
			olen += len;
			argvtmp = (char*)objectFileGetTempName(tempobj);
			len = strlen((char*)objectFileGetTempName(tempobj)) + 1;
			lobjs[li] = tempobj;
		}

		/* collect linker arguments */
		llen += len;
		largs[li++] = argvtmp;
	}

	if (*linker == 0)
	{
		getDefaultToolName(linker, SO_LINKER, linkerflags);
	}

	argv -= argc - 1;
	largs[li] = 0;

	if (flags & SO_HELP)
		printf(hlp);


	/* perform analysis */
	if (flags & SO_OBJECTS)
	{

		/* command is: link an executable using no object files */
		if (listIsEmpty(lObject))
		{
			fprintf(stderr, "ERROR: You can't link an executable without "
				"providing object files.\n");
			return -1;
		}

		/* collect interesting sections */

		/* generate objdump */
		int loutputlen = strlen(loutput);
		olen += loutputlen;
		char* dumpName = (char*) malloc(loutputlen + sizeof(SO_DFILE) + 1);
		SO_SC(dumpName, loutput);
		SO_SC(dumpName, SO_DFILE);
		dumpName -= loutputlen + sizeof(SO_DFILE) - 1;
		
		{
			char* cmdLn = (char*) malloc(sizeof(char) * olen);

			SO_SC(cmdLn, dumper);

			listStart(lObject);
			while (listNext(lObject))
			{
				SO_SC(cmdLn, objectFileGetName((objectFile*)listGet(lObject)));
				SO_SC(cmdLn, " ");
			}

			SO_SC(cmdLn, SO_PIPE);
			SO_SC(cmdLn, dumpName);
			cmdLn -= olen - 1;

			system(cmdLn);
			free(cmdLn);
		}

		/* process */
		{
			FILE* objDump = fopen(dumpName, "r");

			if (objDump)
			{

				/* read all the sections from the file */
				listStart(lObject);
				while (listNext(lObject))
				{
					objectFile* obj = (objectFile*) listGet(lObject);
					objectFileCollect(obj, objDump);
				}

				/* compute the dependency graph */
				rewind(objDump);
				objectFileCompute(lObject, objDump);

				/* colorize all seeds */
				listStart(lSeed);
				while (listNext(lSeed))
					objectFileColorize((const char*) listGet(lSeed), 1);

				fclose(objDump);
				remove(dumpName);
			}
			else
				fprintf(stderr, "ERROR: Couldn't compute dependency graph, because "
					"dumpfile could not be opened.\n");
		}

		/* now remove unused sections */
		if (!(flags & SO_DNRM))
		{
			char* cmdLn = 0;

			listStart(lObject);
			while (listNext(lObject))
			{
				objectFile* obj = (objectFile*) listGet(lObject);

				/* if there's nothing to remove, skip object file */
				list* nonDepends = (list*) objectFileGetUnused(obj);
				if (listIsEmpty(nonDepends) && !sectskipped)
					continue;

				/* if there's nothing to keep, delete object file entirely */
				list* Depends = (list*) objectFileGetUsed(obj);
				if (listIsEmpty(Depends))
				{
					objectFileSetModified(obj, 2);
					continue;
				}
				objectFileSetModified(obj, 1);

				const char* file = objectFileGetName(obj);
				const char* tempfile = objectFileGetTempName(obj);
				unsigned long size = strlen(remover) + strlen(file) + 1 + strlen(tempfile);

				/* it's safer to calculate the size first */
				listStart(nonDepends);
				while (listNext(nonDepends))
					size += sizeof(SO_RRMV " ") + strlen((char*) listGet(nonDepends));
				if (lSkipSection) {
					listStart(lSkipSection);
					while (listNext(lSkipSection))
						size += sizeof(SO_RRMV " ") + strlen((const char*) listGet(lSkipSection));
				}

				cmdLn = (char*) realloc(cmdLn, size + 1);
				SO_SC(cmdLn, remover);

				listStart(nonDepends);
				while (listNext(nonDepends))
				{
					SO_SC(cmdLn, SO_RRMV " ");
					SO_SC(cmdLn, (char*)listGet(nonDepends));
					SO_SC(cmdLn, " ");
				}
				if (lSkipSection) {
					listStart(lSkipSection);
					while (listNext(lSkipSection)) {
						SO_SC(cmdLn, SO_RRMV " ");
						SO_SC(cmdLn, (const char*) listGet(lSkipSection));
						SO_SC(cmdLn, " ");
					}
				}
				SO_SC(cmdLn, file);
				SO_SC(cmdLn, " ");
				SO_SC(cmdLn, tempfile);
				cmdLn -= size;

				system(cmdLn);
				deleteList(nonDepends);
				deleteList(Depends);
			}

			free(cmdLn);
		}


		/* call linker */
		{
			char* cmdLn = (char*) malloc(sizeof(char) * llen);
			char* cmdLnStart = cmdLn;

			SO_SC(cmdLn, linker);
			SO_SC(cmdLn, " ");

			i = 0;
			while (i < li)
			{
				/* if the argument is an object file name, fetch it from the object file struct instead */
				char* FileName = lobjs[i] == 0 ? largs[i] :
					objectFileGetModified(lobjs[i]) == 1 ? (char*)objectFileGetTempName(lobjs[i])
					: objectFileGetModified(lobjs[i]) == 2 ? NULL
					: (char*)objectFileGetName(lobjs[i]);
				if (FileName)
				{
					SO_SC(cmdLn, FileName);
					SO_SC(cmdLn, " ");
				}
				++i;
			}

			system(cmdLnStart);
			free(cmdLnStart);
		}

		/* delete temporary, stripped object files */
		listStart(lObject);
		while (listNext(lObject))
		{
			objectFile* obj = (objectFile*) listGet(lObject);
			if (objectFileGetModified(obj) == 1)
			{
				remove(objectFileGetTempName(obj));
			}
		}
	}
	else if (!(flags & SO_HELP))
		printf(hlp);


	/* dump command line */
	if (flags & SO_DUMP_CMDLN)
	{
		printf("\nCOMMAND LINE:\n%s ", *argv++);
		while (--argc)
			printf("%s ", *argv++);
		printf("\n");
	}

	if (flags & SO_OBJECTS)
	{
		/* dump generated dependency graph */
		if (flags & SO_DUMP_MAP)
			objectFileDumpMap(lObject);


		/* dump used sections */
		if (flags & SO_DUMP_USED)
			objectFileDumpUsed(lObject);


		/* dump unused sections */
		if (flags & SO_DUMP_DISCARDED)
			objectFileDumpUnused(lObject);
	}

	/* just to be clean, although not really necessary */
	free(largs);
	deleteList(lObject);
	deleteList(lSeed);

	return 0;
}
