/* ev_feat_i586.c */


#include <stdio.h>
#include <stdlib.h>  		/* atoi() */

#include <fcntl.h>              /* O_CREAT */
#include <unistd.h>             /* getopt(), ioctl() */
#include <sys/ioctl.h>          /* ioctl */


#include "../../../include/general.h"   /* ad_t, OK, ERROR */


#include "evcnt_ix86.h"
#include "evcnt_feat_ix86.h"


struct ev_s {
  int fd;
};


static ErrCode ev_close(struct ev_s *evs) {
  if (close(evs->fd) < 0) {
    perror("close()");
    evs->fd = 0;
    return ERROR;
  }
  evs->fd = 0;
  return OK;
}


static ErrCode ev_open(struct ev_s *evs) {
  /* open monitor device */
  const int open_mode = O_RDONLY;
  if ((evs->fd = open(EVCNT_DEVICE, open_mode)) < 0) {
    evs->fd = 0;
    perror(EVCNT_DEVICE);
    return ERROR;
  }
  return OK;
}



/* test ... */


/* DIRECT_MSR */
#include <limits.h>             /* ULONG_MAX */

static ErrCode ev_read_msr(struct ev_s *evs, const int msr, hint *h) {
  hint h_tmp[2];
  h_tmp[0].lo = msr;
  h_tmp[0].hi = 0;
  if (ioctl(evs->fd, EV_GETMSR, h_tmp) < 0) {	/* get msr */
    perror("ioctl(EV_GETMSR)");
    return ERROR;
  }
  h->lo = h_tmp[1].lo;
  h->hi = h_tmp[1].hi;
  return OK;
}

static ErrCode ev_write_msr(struct ev_s *evs, const int msr, const hint *h) {
  hint h_tmp[2];
  h_tmp[0].lo = msr;
  h_tmp[0].hi = 0;
  h_tmp[1].lo = h->lo;
  h_tmp[1].hi = h->hi;
  if (ioctl(evs->fd, EV_SETMSR, h_tmp) < 0) {	/* set msr */
    perror("ioctl(EV_SETMSR)");
    return ERROR;
  }
  return OK;
}

static ErrCode test_msr(struct ev_s *evs) {
  const int msr = 0x0e;
  hint h;
  if (ev_read_msr(evs, msr, &h) != OK)  { return ERROR; }
  printf("read_msr (%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
    h.lo + h.hi * (double)ULONG_MAX);
  {
    char ch;
    hint h_mod;
    h_mod.lo = h.lo;
    h_mod.hi = h.hi;
    h_mod.lo |= 0x08;  /* disable internal L1 cache */
    /* h_mod.lo |= 0x04; */  /* deactivate V pipe */
    /* h_mod.lo |= 0x01; */	/* deactivate BTB */
   /* h.lo &= ~0x08; */
    printf("write_msr(%02x): %08lx%08lx (%.0f)\n", msr, h_mod.hi, h_mod.lo,
      h_mod.lo + h_mod.hi * (double)ULONG_MAX);
    if (ev_write_msr(evs, msr, &h_mod) != OK)  return ERROR;
    if (ev_close(evs) != OK)  return ERROR;
    fprintf(stderr,"Press a character to restore old value...\n");
    scanf("%c", &ch);
    if (ev_open(evs) != OK)  return ERROR;
    printf("write_msr(%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
      h.lo + h.hi * (double)ULONG_MAX);
    if (ev_write_msr(evs, msr, &h) != OK)  return ERROR;
  }
  return OK;
}
/* DIRECT_MSR */





#if defined(I586_EVCNT) || defined(I586MMX_EVCNT)
static ErrCode ev_get_feature(struct ev_s *evs, hint *h) {
  if (ioctl(evs->fd, EV_GETFEATURE, h) < 0) {
    perror("ioctl(EV_GETFEATURE)");
    return ERROR;
  }
  return OK;
}

static ErrCode ev_set_feature(struct ev_s *evs, const hint *h) {
  if (ioctl(evs->fd, EV_SETFEATURE, h) < 0) {
    perror("ioctl(EV_SETFEATURE)");
    return ERROR;
  }
  return OK;
}


static ErrCode do_set_feature(struct ev_s *evs, const int feature) {
  hint h;
  h.lo = feature;
  h.hi = 0;
  {
    hint h_tmp;
    if (ev_get_feature(evs, &h_tmp) < 0)  { return ERROR; }
    printf("old feature: %08lx%08lx (%s%s%s)\n", h_tmp.hi, h_tmp.lo,
      (h_tmp.lo & EV_FEA_BTB_DI) ? "BTB_DI " : "",
      (h_tmp.lo & EV_FEA_VPIPE_DI) ? "VPIPE_DI " : "",
      (h_tmp.lo & EV_FEA_L1_DI) ? "L1_DI " : "");
  }
  if (ev_set_feature(evs, &h) < 0)  { return ERROR; }
  {
    hint h_tmp;
    if (ev_get_feature(evs, &h_tmp) < 0)  { return ERROR; }
    printf("new feature: %08lx%08lx (%s%s%s)\n", h_tmp.hi, h_tmp.lo,
      (h_tmp.lo & EV_FEA_BTB_DI) ? "BTB_DI " : "",
      (h_tmp.lo & EV_FEA_VPIPE_DI) ? "VPIPE_DI " : "",
      (h_tmp.lo & EV_FEA_L1_DI) ? "L1_DI " : "");
  }
  return OK;
}
#endif /* I586_EVCNT */

#if defined(I686_EVCNT) || defined(I686MMX_EVCNT)
static ErrCode do_set_feature(struct ev_s *evs, const int feature) {
  fprintf(stderr, "Error: set_feature not supported on i686\n");
  return ERROR;
}
#endif /* I686_EVCNT */

#if defined(ATHLON_EVCNT)
static ErrCode do_set_feature(struct ev_s *evs, const int feature) {
  fprintf(stderr, "Error: set_feature not supported on Athlon\n");
  return ERROR;
}
#endif /* ATHLON_EVCNT */


/* ****************************
 *  read options, usage
 * **************************** */

#define DEF_FEATURE_MASK 0

static void usage(const char *pname) {
  fprintf(stderr, "ev_feat_i586 - Change features for i586\n");
  fprintf(stderr, "Marco Vieth, 17.7.1997\n");
  fprintf(stderr, "Usage: %s [-f n]\n", pname);
  fprintf(stderr, "\t-f num\t set feature mask to num (def. %d)\n", DEF_FEATURE_MASK);
  fprintf(stderr, "\t-t\t test MSR directly\n");
  fprintf(stderr, "\t-h\t help\n");
  fprintf(stderr, "\t-d\t debug\n");
  fprintf(stderr, "\n");
#if defined(I586_EVCNT) || defined(I586MMX_EVCNT)
    fprintf(stderr, "Feature codes:\n");
    fprintf(stderr, "%d : Branch Target Buffer disable (BTB_DI)\n", EV_FEA_BTB_DI);
    fprintf(stderr, "%d : V pipeline disable (VPIPE_DI)\n", EV_FEA_VPIPE_DI);
    fprintf(stderr, "%d : L1 cache disable (L1_DI)\n", EV_FEA_L1_DI);
#endif /* I586_EVCNT */
  exit(1);  
}


struct option_s {
  int feature_mask;
  int test_f;
  int debug_f;
};


static int read_opt(int argc, char **argv, struct option_s *opts) {
  int n;
  while ((n = getopt(argc, argv, "df:ht")) != EOF) {
    switch (n) {
      case 'd':         /* debug output */
        opts->debug_f = 1;
      break;

      case 'f':         /* feature mask */
        opts->feature_mask = atoi(optarg);
      break;

      case 't':         /* test flag */
        opts->test_f = 1;
      break;

      case 'h':  
      case '?':
        usage(argv[0]);
      break;
    }
  }
  return OK;
}


/* ****************************
 *  main
 * **************************** */

int main(const int argc, char **argv) {
  struct option_s opts = { DEF_FEATURE_MASK, 0, 0};
  ErrCode rc = OK;
  struct ev_s evs;

  if (read_opt(argc, argv, &opts) != OK)  { }
 
  if (ev_open(&evs) != OK)  return ERROR;
  if (opts.test_f > 0) {
    if (test_msr(&evs) != OK)  return ERROR;
  } else {
    if (do_set_feature(&evs, opts.feature_mask) != OK)  return ERROR;
  }
  if (ev_close(&evs) != OK)  return ERROR;
  return (rc);
}

/* end */
