;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;             CP/M for the PCW16 - FID file loader                            ;
;                                                                             ;
;   Copyright (C) 1998-1999, John Elliott <jce@seasip.demon.co.uk>            ;
;                                                                             ;
;    This program is free software; you can redistribute it and/or modify     ;
;    it under the terms of the GNU General Public License as published by     ;
;    the Free Software Foundation; either version 2 of the License, or        ;
;    (at your option) any later version.                                      ;
;                                                                             ;
;    This program is distributed in the hope that it will be useful,          ;
;    but WITHOUT ANY WARRANTY; without even the implied warranty of           ;
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            ;
;    GNU General Public License for more details.                             ;
;                                                                             ;
;    You should have received a copy of the GNU General Public License        ;
;    along with this program; if not, write to the Free Software              ;
;    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                ;
;                                                                             ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; /*  This file is the common source for FIDLDR.SYS (CP/M 2) and
;     FIDLDR3.SYS (CP/M 3) - the C preprocessor is used for any different 
;     bits  
;
; */

;
;The memory organisation of PCW16 CP/M is:
;
;* Bank 0: 80, X0, X1, X7
;* Bank 1: X4, X5, X6, X7  
;* Bank 2: 80, X2, X3, X7
;* Bank 3: 80, X8, X1, X7
;
;When PCW16 CP/M is loaded, the Loader (ANNELDR.PRG) contains a minimal
;code set: 
;
; * A CP/M 2.2 BDOS and CCP, or clones thereof; 
; * A CP/M 2.2 BIOS, supporting two devices:
;    - A single, read-only, 720k floppy drive;
;    - A 29x80 "glass TTY" display.
;
; If CPM3.SYS and CCP.COM are present in the *.EML file, they replace the 
;BIOS, BDOS and CCP to give a minimal CP/M 3 system.
;
; On cold boot, FIDLDR is loaded from the *.EML file into the TPA at 0100h.
; FIDLDR must perform the following functions:
;
; * Load and relocate FID support code;
; * Enable FID support in the BIOS;
; * Search for *.FID in the EML file;
; * Search for *.FID;
; * Load and relocate *.FID;
; * Allocate memory for disc buffers (CP/M 3);
; * Check for SUBMIT.COM and PROFILE.SUB (CP/M 2).
;
;Addresses in Bank 2
;
TOPFID	equ	401Ch	;Points to top FID
FSEMS	equ	402Fh	;Addresses in FIDSUPP
FSCALL	equ	402Ch	;Call an address in bank 2
FSLOWSP	equ	4032h	;SP to use for Rosanne calls
DHK1	equ	4034h	;SVC_D_HOOK
CHK1	equ	4037h	;SVC_C_HOOK
NOSVC	equ	403Ah	;Dummy SVC
ILIV1   equ     403Bh   ;Debug breakpoint
ENUM	equ	403Eh	;Called after each FID is loaded successfully.
SBRK	equ	4041h	;Base of allocatable memory in BIOS
DIRBUF	equ	4043h	;Directory data buffer in BIOS
SBRK0	equ	4049h	;Base of allocatable memory in bank 0 (CP/M 3 only)
;
;Addresses in Bank 0
;
FONTBF	equ	40F4h	;Font used by the dumb terminal driver. This is 
			;used for disc buffers by CP/M 3 after an alternative
			;terminal driver has loaded
BCBTOP	equ	40F6h	;Start of area in ANNELDR in which buffer control 
			;blocks can be allocated dynamically.
DIRBCB	equ	40F8h	;Pointers to DIRBCB and DTABCB linked lists
DTABCB	equ	40FAh
;
	include	annecpm.inc

FDOS	equ	5
FCB	equ	5Ch
;
; We start by telling any 80x86 CPU where to get off.
;
	org	0100h
	defb	0ebh,04h        ;DOS protection... JMPS LABE
	ex	de,hl
	jp	begin
	defb	0b4h,09h	;DOS protection... MOV AH,9
	defb	0bah
	defw	bvmes		;DOS protection... MOV DX,OFFSET BVMES
	defb	0cdh,021h	;DOS protection... INT 21h.
	defb	0cdh,020h	;DOS protection... INT 20h.

	defb	13,'CP/M FID loader for Anne',13,10
	defb	'Copyright 1998 by John Elliott',13,10,26
;
;/*
;The above area isn't completely for vanity.  The stack gets put here. It needs
;to be in the same quarter of memory as this program, but not below 100h 
;where it would get overwritten.
;*/
begin:	
;
;Check that this is a Z80 CPU running the correct version of CP/M.
;
	sub	a
	jp	pe,badver	;8080

	ld	c,0Ch
	call	FDOS		;Get CP/M version into A.

#ifdef CPM2			/* CP/M 2 version - check for CP/M 2 */

	cp	20h
	jp	c,badver	;2.0-2.F
	cp	30h
	jp	nc,badver

#else				/* CP/M 3 version - check for CP/M 3 */

	cp	30h
	jp	c,badver	;3.0-3.F
	cp	40h
	jp	nc,badver

	ld	c,2Dh
	ld	e,0FFh		;Don't crash if there's a disc error
	call	FDOS
	ld	de,scbpb	;Get SCB address
	ld	c,31h	
	call	FDOS
	ld	(svcscb),hl

#endif
;
; Check that this is a PCW16 BIOS (which has the signature "ANNE1",26 at a
;known place)
;
	ld	de,(1)	;DE -> base of BIOS
	ld	hl,66h	;->signature
	add	hl,de
	ld	de,anne1
	ld	b,6
cplp:	ld	a,(de)
	cp	(hl)
	jp	nz,badver	
	djnz	cplp
;
;We have got it. Now copy in and initialise the "FID support" module.
;
	ld	sp,begin	;Stack in low quarter, but not too low
;
;Get various BIOS calls.
;
	ld	de,(1)
	ld	hl,60h
	add	hl,de		;BIOS control (PCW16 specific)
	ld	(bctl+1),hl
	ld	hl,4eh
	add	hl,de		;SELMEM
	ld	(selmem+1),hl
	ld	a,2
	call	biosctl		;HL -> low memory stack pointer
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	ld	(lowsp),de	;Stack pointer to use when calling Rosanne.
	ld	a,1
	call	biosctl
	ld	de,memlist	;Get list of memory blocks used by CP/M 
	ld	bc,10		;environment.
	ldir
;
;memlist now exists, so we can use the subroutines "hbank2" and "hbank1".
;"hbank2" switches bank 2 in at 4000h-BFFFh; "hbank1" restores the TPA.
;
	call	hbank2
;
; Copy various variables into FIDSUPP (see fidsupp.zsm / fidsup3.zsm)
;
	ld	hl,fidscode	;Copy FID support code to bank 2:4000h
	ld	de,4000h
	ld	bc,fidsend-fidscode
	ldir			;End with DE->end of FIDSUPP
	ld	hl,0FFh
	add	hl,de
	ld	l,0		;HL -> 1st page available for FIDs.
	ld	(fidpage),hl
	ld	hl,memlist
	ld	de,4020h
	ld	bc,10		;Copy the list of memory blocks to FIDpage.
	ldir
	ld	de,(lowsp)
	ld	(FSLOWSP),de	;Tell it the Rosanne SP.
	ld	a,3
	call	biosctl		;Tell it about BIOS memory buffers.
	ld	(DIRBUF),de
	ld	(SBRK),hl
	ld	hl,FSEMS
	call	FSCALL		;Initialise Bank 2 code.
	jr	c,suppok
	call	hbank1		;Initialisation failed. Say why and depart.
	call	opmess		
	rst	0
;
; FID support has initialised OK. Now search for *.EML and *.FID
;
suppok:	call	hbank1
	call	opmess	;Output sign-on message
	ld	de,80h
	call	setdma
	ld	hl,endprg
	push	hl	;HL -> list of files that will be generated
	ld	hl,eqfcb
	ld	de,FCB
	ld	bc,24h
	ldir		;Search for *.EML
	ld	c,11h
	call	fcbdos
	inc	a
	jp	z,lddsk	;No .EML file. Search for *.FID.
	dec	a
	ld	l,a
	ld	h,0
	add	hl,hl	;*2
	add	hl,hl	;*4
	add	hl,hl	;*8
	add	hl,hl	;*16
	add	hl,hl	;*32
	ld	de,81h
	add	hl,de	;HL -> directory FCB for EML file
	ld	de,lufcb+1
	ld	bc,11
	ldir
	pop	hl
	call	luinit	; - searches the LBR for *.FID, and makes a name list.
	push	hl
;
;Search disc for *.FID
;
lddsk:	ld	de,80h
	call	setdma

	ld	hl,qfcb	;"*.FID"
	ld	de,FCB
	ld	bc,24h
	ldir
	ld	c,11h		;Search for first
lfdlp:	call	fcbdos		;Generate a list of *.FID files at ENDPRG
	cp	0ffh
	jp	z,fidend

	and	3		;A=0-3
	rrca			;A=00 80 01 or 81
	rrca			;A=00 40 80 or C0
	rrca			;A=00 20 40 or 60
	or	80h		;A=80 A0 C0 or E0
	ld	l,a
	ld	h,0

	pop	de
	ld	bc,32		;Copy directory FCB to the list
	ldir
	push	de
	ld	c,12h
	jr	lfdlp
;
; All *.FID files (if any) have been listed.
;
fidend:	pop	de		;(DE - endprg) / 32 = no. of FIDs
	ld	hl,endprg

fiilp:	push	hl		;Load and verify each one in turn.
	and	a
	sbc	hl,de
	pop	hl
	jp	z,exitapp	;If (DE = ENDPRG) then all FIDs have been done.
	push	de
	push	hl
	call	instfid		;Install a FID.
	pop	hl
	pop	de
	ld	bc,20h		;Go to next entry.
	add	hl,bc
	jr	fiilp
;
exitapp:
#ifdef CPM3

;
; Diagnostic code
;
;	call	ilprint
;	defb	'FIDs loaded up to $'
;	ld	hl,(fidpage)
;	call	hexhl
;	call	ilprint
;	defb	13,10,'$'

	push	ix
	call	make_buffers	;Allocate memory to disc buffers
;
;Set the cold start flag
;
	ld	ix,(svcscb)
	res	1,(ix+19h)	;Set the Cold Boot flag.	

	call	make_scrrun	;Set up SCR_RUN_ROUTINE.

	pop	ix
#else
;
;/*
;CP/M 2 doesn't normally have a start-of-day Submit file. FIDSUPP
;makes it try to do a SUBMIT PROFILE.
;*/
;
	ld	de,prosub	;Search for SUBMIT.COM and PROFILE.SUB
	ld	c,11h
	call	FDOS
	inc	a
	jr	z,endall
	ld	de,subcom	;SUBMIT.COM
	ld	c,11h
	call	fdos
	inc	a
	jr	z,endall
	ld	de,sssfcb	;Both are present. Create a $$$.SUB that says
	ld	c,16h		;"SUBMIT PROFILE".
	call	fdos
	ld	de,sssrec
	call	setdma	
	ld	de,sssfcb
	ld	c,15h		;Write out the command
	call	fdos
	ld	de,sssfcb
	ld	c,10h		;Close the file.
	call	fdos
#endif
endall:	rst	0
;
scbpb:	defw	003Ah		;Get address of the SCB itself.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Load, relocate and run a FID file. HL -> dir entry. If first byte is 0FFh,
;then it is a member of the EML file.
;
instfid:
	ld	bc,(fidpage)	;First free page currently available
	ld	(ldadd),bc	;Load address
	ld	a,(hl)
	cp	0ffh
	jp	nz,fileget
;
;The file is an LBR member.
;
	push	hl	;Copy its name to 5Dh so that FIDs get named correctly
	inc	hl
	ld	de,5Dh
	ld	bc,11
	ldir
	pop	hl

	call	luopen	;Open the LBR file. It stays closed while a FID is
	ret	nc	;being initialised in case the FID is a disc drive FID
	push	ix	;which overwrites the A: driver.
	push	hl
	pop	ix	;IX -> LBR member
	ld	l,(ix+12)
	ld	h,(ix+13)	;HL = 1st record
	ld	c,(ix+14)	;
	ld	b,(ix+15)	;BC = no. of records
lbrlp:	ld	a,b
	or	c
	jr	z,lbrle
	push	bc
	ld	bc,1		;Read 1 record at 80h
	ld	de,80h
	call	luread
	call	copyup		;Copy into bank 2.
	pop	bc
	ret	nc
	dec	bc		;Next record.
	jr	lbrlp
;
lbrle:	pop	ix		;FID file loaded from LBR.
	call	luclose
	jp	eofid
;
;File to be loaded from disc.
;
fileget:
	ld	de,5dh	;Copy its name to the FCB.
	inc	hl
	ld	bc,11
	ldir		;FCB holds filename

	ld	h,d
	ld	l,e
	inc	de
	ld	(hl),0	;Zero out the rest of the FCB.
	ld	bc,23
	ldir
	ld	c,0fh
	call	fcbdos	;Open the file. If that failed, return silently.
	inc	a
	ret	z
lfidlp:	ld	de,80h	;Load the entire FID, including its PRL header
	call	setdma
	ld	c,14h	;Read 1 record
	call	fcbdos
	or	a
	jr	nz,eofid
	call	copyup	
	ret	nc	;/* This FID would overflow bank 2 and end up in the */
	jr	lfidlp	;/* TPA. I'm allowing one page because the header is */
			;/* loaded but will subsequently disappear. */
;
;The FID file has loaded...
;
eofid:	call	sigfid	;Check its FID signature ("ANNE.FID")
	ret	nz
	call	sumfid	;Check its checksum
	ret	nz
	call	namefid	;Copy filename into the FID.
	call	relfid	;Relocate its memory image.
	ld	hl,(fidpage)
	call	hbank2
	call	FSCALL	;Run its FID_EMS
	call	hbank1
	push	af
	call	opmess	;Print what it had to say.	
	pop	af
	ret	nc

        call    hbank2          ;If the FID was loaded, activate immediately.
        ld      hl,ENUM		;This code replaces disc minidrivers with
        call    FSCALL		;full drivers instantly the full drivers
        call    hbank1		;are loaded. The minidriver MUST NOT be
        ld      c,0Dh		;used after a successful FID_EMS by the
        call    FDOS		;full driver, because its data structures 
				;will have been overwritten.

	call	chainfid	;Add the FID to the linked list of FIDs.
	ld	hl,(fidlen)
	ld	de,0FFh
	add	hl,de
	ld	l,0		;HL = length of FID, rounded up to nearest page
	ld	de,(fidpage)
	add	hl,de		;Set load address for next FID.
	ld	(fidpage),hl
	ret
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Copy a record from 80h to ldadd in bank 2
;
copyup: push	hl
	push	de
	call    hbank2
        ld      de,(ldadd)
        ld      hl,80h  ;Copy to bank 2.
        ld      bc,80h
        ldir
        ld      (ldadd),de
        call    hbank1
        ld      a,d
        cp      0C1h
      	pop	de
	pop	hl 
	ret     	;Return NC if FID load address is too high
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Append a FID to the linked list of memory images.
;
chainfid:
	call	hbank2
	ld	de,(TOPFID)	;Previous head of list
	ld	hl,(fidpage)
	ld	l,1Eh		;Offset of pointer to prev FID.
	ld	(hl),e	
	inc	hl		;Set new FID to point to prev FID.
	ld	(hl),d
	ld	hl,(fidpage)	
	ld	(TOPFID),hl	;Set pointer to point to new head of list.
	jp	hbank1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Check the signature of a FID (it must read "ANNE.FID")
;
sigfid:	call	hbank2
	ld	hl,(fidpage)	;HL -> FID
	inc	h
	ld	l,3		;Signature
	ld	de,annefid	;Our copy of signature
	ld	b,11
sigck:	ld	a,(de)		;Bytewise compare
	cp	(hl)	

	jp	nz,hbank1
	inc	hl
	inc	de
	djnz	sigck
	xor	a
	jp	hbank1	
	
annefid:
	defb	'ANNE    FID'
;
; Tell the FID its true name - a related routine, but one which must be done
;after checksumming 
;
namefid:
	call    hbank2
        ld      hl,(fidpage)
        inc     h
        ld      l,3     ;HL -> FID Signature
        ld      de,5dh
        ld      b,11	;DE -> name in FCB
pname:	ld      a,(de)
	ld	(hl),a
        inc     hl
        inc     de
        djnz	pname
        xor     a
        jp      hbank1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Check the checksum of a FID. This is the sum of all bytes in its binary
;image except the checksum field itself.
;
sumfid: call	hbank2
	ld	iy,(fidpage)
	ld	l,(iy+1)
	ld	h,(iy+2)	;HL = no. of bytes in image proper
	push	hl
	ld	de,7
	add	hl,de		;(X+7)/8 = no. of bytes in relocation map
	srl	h
	rr	l
	srl	h
	rr	l
	srl	h
	rr	l
	pop	de
	add	hl,de		;No. of bytes in image + map
	inc	h		;+header
	ld	b,h
	ld	c,l		;BC = no. of bytes to sum
	ld	hl,(fidpage)
	inc	h
	ld	l,10h
	ld	e,(hl)		;Get the current checksum into DE; then
	ld	(hl),0		;set those two bytes to zero.
	inc	hl
	ld	d,(hl)		;DE = old sum
	ld	(hl),0		;Zero those 2 bytes out

	push	de		;Calculate checksum of FID
	push	hl
	ld	hl,(fidpage)
	ld	d,0
	ld	iy,0
sumlp:	ld	e,(hl)
	add	iy,de		;Keep the total in IY.
	inc	hl	
	dec	bc
	ld	a,b
	or	c
	jr	nz,sumlp	;IY = sum
	pop	hl
	pop	de

	ld	(hl),d		;Restore the checksum bytes in the file.
	dec	hl
	ld	(hl),e
	push	iy
	pop	hl		;HL = calculated checksum
	and	a		;DE = stored checksum
	sbc	hl,de		;Returns Z if sum OK
	call	hbank1
	ret
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;/* Relocate a FID. This is where it gets moved down 1 page so that the
;header is lopped off. Only the binary image is moved, not the PRL
;map.
;
;This differs from normal PRL relocation in that if a byte is relocatable
;and 0FFh, it's a SuperVisor Call - an address resolved at load time.
; */
;
relfid:	call	hbank2
	ld	iy,(fidpage)
	ld	l,(iy+1)
	ld	h,(iy+2)	;HL = length of FID image	
	ld	c,l
	ld	b,h		;BC = length of FID image
	ld	(fidlen),bc
	ld	hl,(fidpage)
	inc	h
	add	hl,bc		;HL -> relocation map
	push	hl
	pop	iy		;IY -> relocation map
	ld	hl,(fidpage)	;HL -> destination
relf1:	ld	e,(iy+0)
	ld	d,8		;8 bits / byte
	inc	iy		;E = relocation bitmap
relf2:	inc	h
	ld	a,(hl)
	dec	h	
	ld	(hl),a		;Copy the byte
	bit	7,e		;Relocatable?
	jr	z,relf3
	cp	0FFh		;If byte is 0FFh and relocatable, its an SVC.
	jr	nz,nrel
;
;This word is an SVC. The previous byte gives its ID.
;
	dec	hl		;SVCs are a word; replace this and prev byte.
	ld	a,(hl)
	call	putsvc
	inc	hl
	jr	relf3

nrel:	push	hl
	ld	hl,(fidpage)
	add	a,h		;Normal relocatable byte
	pop	hl
	dec	a		;0100h => base address
	ld	(hl),a
relf3:	inc	hl
	rlc	e		;Next bit in relocation map
	dec	bc
	ld	a,b
	or	c
	jr	z,relend
	dec	d
	jr	nz,relf2
	jr	relf1		;Next byte in relocation map
;
relend:	call	hbank1
	ret
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Length of binary image (no PRL map or header)
;
fidlen:	defw	0
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Convert SVC to real address
;
putsvc:	push	de
	push	hl
	ld	e,a		;A = SVC number
	ld	d,0	
	ld	hl,svctbl	;HL -> -> SVC list
	add	hl,de
	add	hl,de
	ld	e,(hl)
	inc	hl
	ld	d,(hl)		;DE = real SVC address
	pop	hl
	ld	(hl),e
	inc	hl
	ld	(hl),d
	dec	hl
	pop	de
	ret
;
;/*
;The SVC table has gaps because I'm using the same SVC numbers as the
;Spectrum +3 version, which has a whole raft of SVCs including dynamic
;memory allocation.
;*/
;
svctbl:	defw	4020h	;0: Memlist
	defw	4074h	;1: SVC_CPM_ADDR, convert CHL to Rosanne address
	defw	ILIV1	;2: SVC_DBG_BRK, debug breakpoint
svcscb:	defw	0	;3: No SCB in CP/M 2
	defw	CHK1	;4: SVC_C_HOOK
	defw	DHK1	;5: SVC_D_HOOK
	defw	4077h	;6: SVC_D_CHANGED
	defw	NOSVC
	defw	NOSVC
	defw	NOSVC
	defw	407Ah	;10: SVC_C_FIND
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; This code is relocated to high memory.
;
fidscode:
#ifdef CPM3
	include fidsup3.dbi
#else
	include	fidsupp.dbi
#endif
fidsend:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	
;BIOS jumpblock entries and data
;
bctl:	jp	0	;BIOS control
selmem:	jp	0	;Select memory bank
;
memlist:
	defs	10	;10 bytes giving memory block numbers.
;
lowsp:	defw	0	;Stack pointer used during Rosanne calls.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Search patterns for *.FID and *.EML
;
qfcb:	defb	0,'????????FID'
	defw	0,0,0,0,0,0,0,0,0,0,0,0
;
eqfcb:	defb	0,'????????EML'
	defw	0,0,0,0,0,0,0,0,0,0,0,0
;
;FCB for loading the LBR file
;
lufcb:	defb	0,'***********'
	defw	0,0,0,0,0,0,0,0,0,0,0,0,0
;
#ifdef CPM2
subcom:	defb	0,'SUBMIT  COM'
	defw	0,0,0,0,0,0,0,0,0,0,0,0,0
prosub:	defb	0,'PROFILE SUB'
	defw	0,0,0,0,0,0,0,0,0,0,0,0,0
sssfcb:	defb	1,'$$$     SUB'
	defw	0,0,0,0,0,0,0,0,0,0,0,0,0

sssrec:	defb	14,'SUBMIT PROFILE',0

#endif
;
;Magic number for LBR file
;
ludir:	defb	0,'           ',0,0
ludire:
;
;Load address for next FID
;
fidpage:
	defw	0	;Its base
ldadd:	defw	0	;Address of current record
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Incorrect version of CP/M
;
badver:	ld	de,bvmes
	ld	c,9	;Print message and go.
	call	FDOS
	rst	0
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Output a 0FFh-terminated message in bank 2
;
opmess:	call	hbank2
	ld	a,(hl)
	cp	0FFh	;Get the byte
	call	hbank1
	ret	z
	ld	e,a
	inc	hl
	ld	c,2	;And output it.
	push	hl
	call	FDOS
	pop	hl
	jr	opmess
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Select an environment which is TPA below 4000h, Bank 2 above that.
;
hbank2:	push	af
	ld	a,(memlist+3)
	out	(0F1h),a
	ld	a,(memlist+4)
	out	(0F2h),a
	pop	af
	ret
;
;TPA below 4000h, Bank 0 above that
;
hbank0:	push	af	
	ld	a,(memlist+1)
	out	(0F1h),a
	ld	a,(memlist+2)
	out	(0F2h),a
	pop	af
	ret
;
;Return to the TPA.
;
hbank1: push	af
	push	bc
	push	de
	push	hl
	ld	a,1
	call	SELMEM
	pop	hl
	pop	de
	pop	bc
	pop	af
	ret
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; For debugging, it is useful to have a "print" routine which prints
;without altering registers (it takes its parameter inline)
;
ilprint:
	ex	(sp),hl
	push	de
	push	bc
	push	af	
ilp1:	ld	a,(hl)
	inc	hl
	cp	'$'
	jr	z,ilend
	ld	e,a
	ld	c,2
	push	hl
	call	FDOS
	pop	hl
	jr	ilp1	
;
ilend:	pop	af
	pop	bc
	pop	de
	ex	(sp),hl
	ret
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;And this terribly familiar set of functions deals with numeric output.
;
hexhl:	ld	a,h
	call	hexa
	ld	a,l
hexa:	push	af
	rrca
	rrca
	rrca
	rrca
	call	hexnib
	pop	af
hexnib:	push	af
	and	0Fh
	cp	10
	jr	c,hexn1
	add	a,7
hexn1:	add	a,'0'
	push	hl
	push	bc
	push	de
	ld	e,a
	ld	c,2
	call	FDOS
	pop	de
	pop	bc
	pop	hl
	pop	af
	ret
;
dmem:	push	af
	push	bc
	push	de
	push	hl
	call	ilprint
	defb	'{$'
	ld	b,8
dmlp:	ld	a,(de)
	call	hexa
	inc	de
	djnz	dmlp
	call	ilprint
	defb	'}',13,10,'$'
	jp	popa
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;make_buffers() allocates all free memory after FIDS have loaded as
;               disc buffers.
;
#ifdef CPM3

make_buffers:
;
;1. Directory buffers
;
	ld	a,1
	call	bctl	;D = 2 if a screen driver FID is loaded
	ld	a,d	;      (hence font available)
	cp	2
	jr	nz,nofbf

	call	hbank0
	ld	ix,(DIRBCB)	;IX -> the only directory BCB
	call	bcbwalk
	ld	hl,(FONTBF)	;HL -> buffer space
	call	hbank1
	ld	b,8		;8 buffers
	ld	c,0		;C = bank
	ld	de,200h
fromfnt:
	call	newbcb
	ret	nc		;All BCBs used up
	add	hl,de
	djnz	fromfnt	
;
;8 buffers now allocated from the font.
;
nofbf:	call	hbank0
	ld	ix,(DIRBCB)
	call	bcbwalk		;Let IX->last BCB in the directory chain.
	call	hbank2
	ld	hl,(SBRK0)
	call	hbank1	
	ld	c,0
	ld	de,200h
from0:	ld	a,h		;Allocate some more directory buffers
	cp	90h		;in bank 0
	jr	nc,enda0
	call	newbcb
	ret	nc		;No more space for BCBs
	add	hl,de
	jr	from0
;
enda0:	call	hbank0		;Allocate some data buffers in bank 2
	ld	ix,(DTABCB)
	call	bcbwalk		;IX -> last data BCB
	call	hbank1
	ld	hl,(fidpage)	;Top of FIDs
	ld	c,2		;Bank 2
	ld	de,200h
from2:	ld	a,h
	cp	0BEh		;Until we reach the top of Bank 2.
	ret	nc
	call	newbcb
	ret	nc
	add	hl,de
	jr	from2
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Allocate a new BCB, using buffer at HL, bank C. Append it to chain at IX.
;
;Return NC if there is no room for more BCBs
;
newbcb:	push	hl
	push	de
	push	bc
	call	hbank0
	ld	de,(BCBTOP)
	ld	hl,15		;A BCB is 15 bytes
	add	hl,de
	ld	a,h
	cp	80h
	jr	nc,nbcend	;BCBs cannot go past 8000h
	ld	(BCBTOP),hl
	ld	h,d
	ld	l,e
	ld	bc,0F00h	;Zero the BCB
zapbcb:	ld	(hl),c
	inc	hl
	djnz	zapbcb
	ld	(ix+13),e
	ld	(ix+14),d	;Link to next BCB
	push	de
	pop	ix		;IX -> new BCB
	pop	bc
	pop	de
	pop	hl
	ld	(ix+0),0FFh	;Empty
	ld	(ix+10),l	;Buffer
	ld	(ix+11),h
	ld	(ix+12),c	;Bank
	call	hbank0
	scf
	ret
;
nbcend:	call	hbank1
	pop	bc
	pop	de
	pop	hl
	ret	
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;If IX -> BCB, make IX -> last BCB in chain
;
bcbwalk:
	ld	e,(ix+13)
	ld	d,(ix+14)	;Link to next BCB
	ld	a,d
	or	e		;0 => end of chain
	ret	z
	push	de
	pop	ix
	jr	bcbwalk
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Set up SCR_RUN_ROUTINE
;
make_scrrun:
	ld	a,3
	call	bctl	;DE -> SCR_RUN_ROUTINE code in the BIOS	
	ld	h,d
	ld	l,e	;DE -> base, HL -> buffer

	ld	ix,scrrun
	ld	b,scrre-scrrun
makelp:	ld	a,(ix+1)
	cp	0F7h		;Relocatable?
	jr	nz,normal
	push	hl
	ld	l,(ix+0)
	ld	h,0
	add	hl,de		;HL = Buffer+X
	ld	a,l
	ld	c,h		;CA = buffer+X
	pop	hl
	ld	(hl),a
	inc	hl
	ld	a,c
	ld	(hl),c		;Replace the two bytes with computed address
	inc	hl
	inc	ix
	inc	ix
	dec	b
	jr	z,mkend
	djnz	makelp
	ret
;
normal:	ld	a,(ix+0)
	ld	(hl),a	
	inc	ix
	inc	hl
	djnz	makelp
mkend:	ret
;
scrrun:
	include	scrrun.dbi
scrre:
	defb	0	;/*So it doesn't try to relocate the last byte*/

;
#endif	/* CPM3 */
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Search LBR file for .FIDs. HL -> directory buffer.
;
;Firstly, check that this file is in LBR format.
;
luinit:	
	ld	(luptr),hl
	call	luopen		;Open the file.
	jp	nc,luend2
	ld	de,8000h
	call	setdma		;Load directory at 8000h
	ld	c,14h
	ld	de,lufcb
	call	FDOS		;Read directory, 1st record
	ld	de,8000h
	ld	hl,ludir
	ld	b,ludire-ludir	
lui1:	ld	a,(de)		;Check for directory signature
	cp	(hl)
	jp	nz,luend1
	inc	de
	inc	hl
	djnz	lui1
	ld	bc,(800Eh)	;No. of records in LBR directory
	ld	de,8000h
	ld	hl,0		;Load them all
	call	luread
	jr	nc,luend1	;LBR directory loaded at 8000h

	ld	hl,(800Eh)	;No. of directory records
	add	hl,hl
	add	hl,hl		;No. of directory entries
	push	ix
	push	iy
	ld	ix,8000h	;Search for *.FID
findlp:	ld	a,(ix+0)

;;;	push	af		;DEBUG
;;;	add	a,'5'
;;;	ld	(debch),a
;;;	call	ilprint
;;;debch:	defb	'*$'
;;;	pop	af		;DEBUG ends


	or	a
	jr	nz,findnxt	;Active entry?
	ld	a,(ix+9)
	cp	'F'
	jr	nz,findnxt
	ld	a,(ix+10)	;Filetype of .FID?
	cp	'I'
	jr	nz,findnxt
	ld	a,(ix+11)
	cp	'D'
	jr	nz,findnxt
;
;IX -> valid entry. Append it to the list.
;
	push	hl
	push	ix
	pop	hl
	inc	hl		;HL -> entry + 1
	ld	bc,1Fh
	ld	de,(luptr)
	ld	a,0FFh		;Drive byte of 0FFh means: This file is in the
	ld	(de),a		;LBR
	inc	de
	ldir
	ld	(luptr),de
	pop	hl
findnxt:
	ld	bc,20h		;Next LBR entry
	add	ix,bc
	dec	hl
	ld	a,h		;Done all entries?
	or	l
	jr	nz,findlp
	pop	iy
	pop	ix
luend1:	call	luclose		;Close the LBR file
luend2:	ld	hl,(luptr)
	ret
;
luptr:	defw	0
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
luread:	
;
;Read from LBR member, and auto-increment load address. 
;BC = no. of records, HL = 1st record, DE = load address. Returns DE = new load
;address, BC=0, HL = next record to read
;
	ld	(lufcb+21h),hl
	xor	a
	ld	(lufcb+23h),a
lurd1:	ld	a,b
	or	c
	scf
	ret	z	;Carry set if read succeeded
	call	setdma
	push	de
	push	bc

	ld	c,21h	;Read random record
	ld	de,lufcb
	call	FDOS
	pop	bc
	pop	de
	or	a
	ret	nz	;Carry clear if read failed	

	ld	hl,80h
	add	hl,de
	ex	de,hl	;Advance the record by 1
	ld	hl,(lufcb+21h)
	inc	hl
	ld	(lufcb+21h),hl
	dec	bc
	jr	lurd1	
;
; Open the LBR file
;
luopen:	push	hl
	push	de	
	push	bc
	ld	hl,0		;Reset FCB internal variables
	ld	(lufcb+14),hl
	ld	(lufcb+12),hl
	ld	(lufcb+20),hl
	ld	c,0fh
	ld	de,lufcb	;Open file
	call	FDOS
	pop	bc
	pop	de
	pop	hl
	cp	4
	ret		;Carry set if OK	
;
luclose:
	push	af	;Close LBR file
	push	bc
	push	de
	push	hl
	ld	c,10h
	ld	de,lufcb
	jr	fdpopa
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Set the CP/M DMA address without disturbing registers
;
setdma:	push	af
	push	bc
	push	de
	push	hl
	ld	c,1ah
fdpopa:	call	FDOS
popa:	pop	hl
	pop	de
	pop	bc	
	pop	af
	ret
;
;Call FDOS with DE=FCB
;
fcbdos:	ld	de,FCB
	jp	FDOS
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;Messages
;
#ifdef CPM3
bvmes:	defb	'This program requires PCW16 CP/M 3.x',13,10,'$'
#else
bvmes:	defb	'This program requires PCW16 CP/M 2.x',13,10,'$'
#endif
ANNE1:	defb	'ANNE',1,26	;BIOS signature
;
endprg:

	end

