/*  AO3Scraper 2.0: Scrapes AO3 to extract work statistics
    Copyright (C) 2013  John Elliott  <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
package info.seasip.ao3scraper;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.WindowEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.JProgressBar;
import javax.swing.JTable;
import javax.swing.JScrollPane;

/** The user interface for AO3Scraper. 
 *
 * @author John Elliott  jce@seasip.demon.co.uk
 */
public class MainFrame extends JFrame implements DownloaderEventListener {
    
    // User interface objects: 
    JPanel ctls = new JPanel();         // Panel containing the active controls
    JPanel status = new JPanel();       // Panel containing status displays
    JButton download = new JButton("Download");    // Download button
    JButton about = new JButton("About...");    // About-box button
    JButton stop  = new JButton("Stop");        // Stop button
    JTextField ao3user = new JTextField("", 50);// AO3 username input
    JLabel result = new JLabel("Ready.");       // Status display
    JProgressBar progress = new JProgressBar(); // Progress bar
    JTable table = new JTable();                // Table for display of results.

    Downloader activeDownloader = null;         // The downloader that clicking 'Stop' will stop.
    
    /** The default constructor. */
    MainFrame()
    {
        setTitle("AO3 Stats Downloader");
                
        // Construct the window.
        // Upper ('Controls') panel
        ctls.setLayout(new FlowLayout());
        ctls.add(new JLabel("AO3 username:"));
        ctls.add(ao3user);
        ctls.add(download);
        ctls.add(stop);
        ctls.add(about);

        // Lower('Status') panel
        status.setLayout(new BorderLayout());
        status.add(result, BorderLayout.CENTER);
        status.add(progress, BorderLayout.EAST);

        // Put all the parts together.
        add(ctls, BorderLayout.NORTH);
        add(new JScrollPane(table), BorderLayout.CENTER);
        add(status, BorderLayout.SOUTH);
        pack();

        stop.setEnabled(false);
        
        // Set up event handlers.
        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                dispose();
            }
            
            public void windowClosed(WindowEvent e) {
                System.exit(0);
            }
        });
        ao3user.addKeyListener(new KeyAdapter() {

            public void keyReleased(KeyEvent ke) {
                checkUser();
            }
        });
        checkUser();
        // Pressing RETURN in the username field, or 
        // clicking "Download", triggers a download.
        ao3user.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent ae) {
                onDownload();
            }
        });
        download.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent ae) {
                onDownload();
            }
        });
    
        stop.addActionListener(new ActionListener() {
            
            public void actionPerformed(ActionEvent ae) {
                onStop();
            }
        });
        
        // Clicking "About" triggers the about box.
        about.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent ae) {
                onAbout();
            }
        });
    
    
    }
    
   /** Entry point for the application.
     * @param args the command line arguments -- ignored.
     */
    public static void main(String[] args) {
        MainFrame mf = new MainFrame();
        mf.setVisible(true);
    }

    ////////////////// DownloaderEventListener methods /////////////////////

    /** Handle a Download Complete event.
     * 
     * @param d The Downloader that has completed.
     * @param message The message it gave us.
     */
    public void downloadComplete(Downloader d, String message) {
        stop.setEnabled(false);
        activeDownloader = null;
        download.setEnabled(true);
        result.setText(message);
        progress.setMaximum(1);
        progress.setValue(1);
        
        table.removeAll();
        d.fillTable(table);
        
        // Prompt to save the result.
        JFileChooser dlg = new JFileChooser();
         
        dlg.setName(d.getUsername() + ".tab");
        int val = dlg.showSaveDialog(this);
        
        // If the user agreed, then save what we got.
        if (val == JFileChooser.APPROVE_OPTION) {
            try {
                d.saveStats(dlg.getSelectedFile());
            }
            catch (Exception e) {
                result.setText(e.getClass().getName() + ": " + e.getMessage());
            }
        }
    }


    /** Handle a Download Failed event.
     * 
     * @param d The Downloader that has failed to download.
     * @param why The message it gave us.
     */
    public void downloadFailed(Downloader d, String why)
    {
        stop.setEnabled(false);
        activeDownloader = null;
        progress.setValue(0);
        download.setEnabled(true);
        result.setText(why);
    }

    /** Handle a Download Progress event.
     * 
     * @param d The Downloader giving us the details.
     * @param msg The message to display.
     * @param tot Total number of works to download.
     * @param cur Number downloaded so far.
     */
    public void downloadProgress(Downloader d, String msg, int tot, int cur)
    {
        result.setText(msg);   
        progress.setIndeterminate(false);
        progress.setMaximum(tot);
        progress.setValue(cur);
    }

    
    /** Check if a username has been typed, and enable / disable the 
     *  'download' button accordingly.
     */
    private void checkUser()
    {
        download.setEnabled(!ao3user.getText().equals(""));
    }

    // Event handlers 
    /** Display the About box. */
    private void onAbout() {
        AboutBox box = new AboutBox(this);
        
        box.setVisible(true);
    }

    private void onStop() {
        if (activeDownloader != null) {
            activeDownloader.onStop();
        }
    }
    
    /** Download stats for the specified username. */
    private void onDownload() {
        String user = ao3user.getText();
        activeDownloader = new Downloader(user);

        stop.setEnabled(true);
        
        progress.setIndeterminate(true);
        activeDownloader.addDownloadListener(this);
        result.setText("Processing");
        download.setEnabled(false);
        activeDownloader.start();
    }
    
    
}
